/* DTMLIB.C: Date manipulation

   Title   : DTMLIB
   Version : 2.0
   Date    : Nov 23,1996
   Author  : J R Ferguson
   Language: Turbo C 2.0, Turbo C++ 3.1 for Windows
   Usage   : function library
*/

#include <dos.h>
#include <mem.h>
#include "dtmlib.h"


/* --- local definitions --- */


#define IDFMIN       694325
#define IDFMAX       767008
#define JULOFSIDF    694325
#define JULOFSYEAR   1901
#define JULOFSDAY    001
#define YMDOFSIDF    694325
#define YMDOFSYEAR   1901
#define YMDOFSMONTH  01
#define YMDOFSDAY    01
#define CALOFSIDF    691769
#define CALOFSYEAR   1894
#define CALOFSWEEK   01
#define CALOFSDAY    01

static const int _MonthLen[2][12] /* [leapyear][month-1] */
               = {{  0, 31, 59, 90,120,151,181,212,243,273,304,334},
                  {  0, 31, 60, 91,121,152,182,213,244,274,305,335}};

static const unsigned int _WeekTable[28]
               = {   0,  52, 104, 156, 209, 261, 313, 365, 417, 469,
                   522, 574, 626, 678, 730, 783, 835, 887, 939, 991,
                  1043,1096,1148,1200,1252,1304,1356,1409};
#define WEEKTBLEN      28
#define WEEKMULTIPLE 1461

static struct date _CurDate;

#define  leapyear(year)   (((year) % 4) == 0 ? 1 : 0)
/* static int leapyear(unsigned int year) */


static void _JulToIdf(Dtm_DateRec *dt)
{
  dt->idf= JULOFSIDF
	 + (dt->jul.year - JULOFSYEAR) * 365.25
         + dt->jul.day - JULOFSDAY;
  dt->rc = Dtm_RCOK;
}


static void _YmdToIdf(Dtm_DateRec *dt)
{
  while (dt->ymd.month < 01) { dt->ymd.month += 12; --(dt->ymd.year); }
  while (dt->ymd.month > 12) { dt->ymd.month -= 12; ++(dt->ymd.year); }
  dt->idf= YMDOFSIDF
        + (dt->ymd.year - YMDOFSYEAR) * 365.25
        + _MonthLen[leapyear(dt->ymd.year)][dt->ymd.month - 1]
        + dt->ymd.day - YMDOFSDAY;
  dt->rc= Dtm_RCOK;
}


static void _CalToIdf(Dtm_DateRec *dt)
{ long int years, weeks, days;

  years= dt->cal.year - CALOFSYEAR;
  weeks= dt->cal.week - CALOFSWEEK;
  days = dt->cal.day  - CALOFSDAY;
  dt->idf = CALOFSIDF + ( (years / WEEKTBLEN) * WEEKMULTIPLE
			   + _WeekTable [years % WEEKTBLEN] + weeks
                          ) * 7
		      + days;
  dt->rc= Dtm_RCOK;
}


static void _CurToIdf(Dtm_DateRec *dt)
{
  getdate(&_CurDate);
  dt->ymd.year  = _CurDate.da_year;
  dt->ymd.month = _CurDate.da_mon;
  dt->ymd.day   = _CurDate.da_day;
  _YmdToIdf(dt);
}


static void _IdfToJul(Dtm_DateRec *dt)
{ long int years, days, quotient;

  dt->rc  = Dtm_RCOK;
  days    = dt->idf - JULOFSIDF;
  quotient= days / 1461; days %= 1461; years= 4 * quotient;
  quotient= days /  365; days %=  365;
  if (quotient == 4) { --quotient; days += 365; }
  years += quotient + JULOFSYEAR;
  days  += JULOFSDAY;
  if (dt->fn == Dtm_FNJUL)
    if ((years != dt->jul.year) || (days != dt->jul.day))
      dt->rc= Dtm_RCWRN;
  dt->jul.year= years; dt->jul.day= days;
}


static void _IdfToYmd(Dtm_DateRec *dt)
{ long int years, months, days, quotient;
  int      leap;

  dt->rc  = Dtm_RCOK;
  days    = dt->idf - YMDOFSIDF;
  quotient= days / 1461; days %= 1461;
  years   = YMDOFSYEAR + 4 * quotient;
  quotient= days / 365;  days %= 365;
  if (quotient==4) { --quotient; days += 365; }
  years += quotient;
  months = 12; leap= leapyear(years);
  while (_MonthLen[leap][months-1] > days) --months;
  days += YMDOFSDAY - _MonthLen[leap][months-1];
  if (dt->fn == Dtm_FNYMD)
     if ( (years  != dt->ymd.year ) ||
	  (months != dt->ymd.month) ||
	  (days   != dt->ymd.day  )
	)
	dt->rc= Dtm_RCWRN;
  dt->ymd.year = years;
  dt->ymd.month= months;
  dt->ymd.day  = days;
}


static void _IdfToCal(Dtm_DateRec *dt)
{ long int years, weeks, days, quotient;
  int      leap;
  int      i;   /* 0..WEEKTBLEN */

  dt->rc = Dtm_RCOK;
  days  = dt->idf - CALOFSIDF;
  weeks  = days / 7; days %= 7;
  years  = 0;
  while (weeks < 0) {
    weeks += WEEKMULTIPLE;
    years -= WEEKTBLEN;
  }
  while (weeks >= WEEKMULTIPLE) {
    weeks -= WEEKMULTIPLE;
    years += WEEKTBLEN;
  }
  i= WEEKTBLEN - 1; years += WEEKTBLEN - 1;
  while (_WeekTable[i] > weeks) { --i; --years; }
  weeks -= _WeekTable[i];
  years += CALOFSYEAR;
  weeks += CALOFSWEEK;
  days  += CALOFSDAY ; if (days > 7) days -= 7;
  if (dt->fn == Dtm_FNCAL)
     if ( (years != dt->cal.year) ||
          (weeks != dt->cal.week) ||
	  (days  != dt->cal.day)
        )
        dt->rc= Dtm_RCWRN;
  dt->cal.year= years;
  dt->cal.week= weeks;
  dt->cal.day = days;
}



/*
========================= global definitions =============================
*/


int Dtm_Convert(Dtm_DateRec *date)
{ Dtm_DateRec dt;
  Dtm_RcTyp   maxrc;

  memcpy(&dt, date, sizeof(Dtm_DateRec));
  maxrc= Dtm_RCOK;
  switch (dt.fn) {
    case Dtm_FNCUR : _CurToIdf(&dt) ;  break;
    case Dtm_FNIDF : dt.rc= Dtm_RCOK;  break;
    case Dtm_FNJUL : _JulToIdf(&dt) ;  break;
    case Dtm_FNYMD : _YmdToIdf(&dt) ;  break;
    case Dtm_FNCAL : _CalToIdf(&dt) ;  break;
    default        : dt.rc= Dtm_RCFUN;
  }
  if (dt.rc > maxrc) maxrc= dt.rc;
  if (maxrc <= Dtm_RCWRN) {
    if ((dt.idf < IDFMIN) || (dt.idf > IDFMAX)) maxrc= Dtm_RCRNG;
    else {
      _IdfToJul(&dt); if (dt.rc > maxrc) maxrc= dt.rc;
      if (maxrc <= Dtm_RCWRN) {
        _IdfToYmd(&dt); if (dt.rc > maxrc) maxrc= dt.rc;
        if (maxrc <= Dtm_RCWRN) {
          _IdfToCal(&dt); if (dt.rc > maxrc) maxrc= dt.rc;
        }
      }
    }
  }
  dt.rc= maxrc;
  if (dt.rc <= Dtm_RCWRN)
    memcpy(date, &dt, sizeof(Dtm_DateRec));
  else
    date->rc= dt.rc;
  return dt.rc == Dtm_RCOK;
}


int Dtm_Add(Dtm_DateRec *date_beg,
	    long int     days,
	    Dtm_DateRec *date_end)
{ int ok1, ok2;

  ok1= Dtm_Convert(date_beg);
  if (date_beg->rc > Dtm_RCWRN) {
    date_end->rc= date_beg->rc;
    return 0;
  }
  else {
    date_end->idf= date_beg->idf + days;
    date_end->fn = Dtm_FNIDF;
    ok2= Dtm_Convert(date_end);
    return ok1 && ok2;
  }
}


int Dtm_Subtract(Dtm_DateRec *date_end,
		 Dtm_DateRec *date_beg,
		 long int    *days    )
{ int ok1, ok2;

  ok1= Dtm_Convert(date_end);
  if (date_end->rc > Dtm_RCWRN) {
    date_beg->rc= date_end->rc;
    return 0;
  }
  else {
    ok2= Dtm_Convert(date_beg);
    if (date_beg->rc > Dtm_RCWRN) return 0;
    else {
      *days= date_end->idf - date_beg->idf;
      return ok1 && ok2;
    }
  }
}
