/*
** Joystick routines.
**
** This file is part of:
**
** Joystick Library for Dos.
** Version: 1.2
**
** Last modified: September 23rd, 1997
**
** (C) 1997 Simone Zanella Productions.
**
** E-mail: simone@zanella-hifi.com
**         zanella@prometeo.it
**         szanella@dsi.unive.it
**
** Web:    http://www.dsi.unive.it/~szanella/index.htm
**         http://members.tripod.com/~szanella/index.htm
**
** Changes from version 1.1:
** - fixed the lock up that happened when joysticks were disconnected
**   during port reading;
** - fixed a problem with automatic recalibration;
** - added support for 8-button joysticks.
**
** Changes from version 1.0:
**
** - changed calibration to manual, to avoid incompatibilities and
**   to get a faster reading;
** - added support for 4 and 6 buttons joysticks;
** - added parameter mode to function InstallJoystick, to set
**   the number of buttons for joysticks;
** - added parameter skipjoy to function InstallJoystick, to skip
**   installation/calibration of one joystick;
** - added parameter calibfile to function InstallJoystick, to
**   allow calibration saving/reloading;
** - changed to integer parameters 'first' and 'second' in function
**   ReadJoystick, to keep flags for fire buttons 5 and 6;
** - removed tolerances from function InstallJoystick.
**
** --------------------------------------------------------
**
** This is freeware: use it as you like, but ALWAYS include
** the original author's name in the docs of your program.
**
** --------------------------------------------------------
**
** Usage:
**
** - include "joystick.h" at the beginning of your program and
**   compile and link the module "joystick.c"
**
** - at program startup, use:
**
**   unsigned char j, mode;
**
**   // Set 'mode' to the kind of joystick requested:
**   // JOY_2BUTTONS    standard 2-button joysticks
**   // JOY_4BUTTONS    4-button joysticks (fire 3 and 4 are other joystick's
**   //                 fire buttons)
**   // JOY_6BUTTONS    6-button joystick (fire 3 & 4 as above, fire 5 and 6
**   //                 connected to left and up of joystick #2)
**   // JOY_8BUTTONS    8-button joystick (fire 3..6 as above, fire 7 and 8
**   //                 connected to down and right of joystick #2)
**   //
**   // Set 'calibfile' to the name of a file to be used for saving/reloading
**   // calibration (NULL to force re-calibration at each run).
**   //
**   // Set 'skipjoy' to:
**   // JOY_NONE    if all joysticks connected should be detected and calibrated;
**   // JOY_SECOND  if only first joystick is used and thus should be calibrated.
**
**   j = InstallJoystick(mode, skipjoy, calibfile);
**
**   Remember that in case of 8-button joysticks the fire buttons 5/7 and
**   6/8 are mutually exclusive, i.e. they can't be pressed together at
**   any time.
**
** - use the value j to determine number of joysticks connected:
**
**   j & JOY_FIRST    first joystick is present
**   j & JOY_SECOND   second joystick is present
**   j & JOY_BOTH     both joysticks present (or one 6/8-button joystick)
**
** - when you want to read the joystick(s), use:
**
**   unsigned int fj, sj;
**
**   ReadJoystick(&fj, &sj);
**
**   Then, you can determine if a direction is activated by:
**
**   fj & JOY_xx
**
**   where "xx" can be: UP, DN (down), LT (left), RT (right)
**
**   You can test the fire buttons in a similar way:
**
**   fj & JOY_AF    1st button
**   fj & JOY_BF    2nd button
**   fj & JOY_CF    3rd button (only when mode >= JOY_4BUTTONS)
**   fj & JOY_DF    4th button (only when mode >= JOY_4BUTTONS)
**   fj & JOY_EF    5th button (only joystick 1, when mode >= JOY_6BUTTONS)
**   fj & JOY_FF    6th button (only joystick 1, when mode >= JOY_6BUTTONS)
**   fj & JOY_GF    7th button (only joystick 1, when mode = JOY_8BUTTONS)
**   fj & JOY_HF    8th button (only joystick 1, when mode = JOY_8BUTTONS)
**
**   Of course, replace "fj" above with "sj" to test second joystick.
**
** If you want to poll joystick at regular intervals, you can chain the
** system timer (int 8).
*/

#include "joystick.h"

#include <stdlib.h>
#include <dos.h>
#include <limits.h>
#include <conio.h>
#include <io.h>
#include <fcntl.h>
#include <sys\stat.h>
#include <stdio.h>

#ifndef __BORLANDC__
#include <pc.h>
#endif

#define JOYPORT              0x201

/* Number of divisions, for determining thresholds */

#define SEGMENTS             2

/* Tolerance, to determine direction on 6/8-button joysticks */

#define JOY_TOL              100

/* Constant for timed joy reading */

#define JOYCOUNTER           8000

/* Pseudo-functions */

#define GetBit(c, n)         ( c & ( 1 << (n) ) )

/* Static variables used for storing:
** - x and y coordinates;
** - threshold values;
** - status (ready/not ready);
** - mode (2, 4, 6 or 8 buttons);
** - flags.
*/

static int x[2], y[2], calib[2][4];
static unsigned char jready[2], joymode = JOY_2BUTTONS;
static unsigned int jstatus[2];

/* Maps for fire buttons (internal) */

static unsigned char fire1[2] = {16, 64};
static unsigned char fire2[2] = {32, 128};

/* Structures used */

typedef struct calibrecord {
   unsigned char njoy;
   unsigned char joymode;
   int calib[2][4];
} CalibRecord;

/****************************************************************************/
/************** Internal functions: should NOT be called by user ************/
/****************************************************************************/

static unsigned char _readjport(void)
{
  /*
  ** Read joystick port, setting x and y coordinates for each joystick;
  ** the status of the 4 buttons is returned.
  */

  register int count;
  register unsigned char st, mask;

  mask = 0;
  if (jready[0])
    mask |= 3;

  if (jready[1])
    mask |= 12;

  count = x[0] = y[0] = x[1] = y[1] = 0;

  disable();

  /* Read port for JOYCOUNTER cycles */
  count = 0;
  outportb(JOYPORT, 0xFF);
  do
  {
    st = inportb(JOYPORT);
    if (st & 1)
      x[0]++;
    if (st & 2)
      y[0]++;
    if (st & 4)
      x[1]++;
    if (st & 8)
      y[1]++;

    count++;
  } while ((st & mask) && (count < JOYCOUNTER));

  enable();

  if (count == JOYCOUNTER)
  {
    if (jready[0] && (st & 3)) /* Joystick 1 was desconnected */
    {
      x[0] = calib[0][1];
      y[0] = calib[0][3];
      jready[0] = 0;
    }

    if (jready[1] && (st & 12)) /* Joystick 2 was desconnected */
    {
      x[1] = calib[1][1];
      y[1] = calib[1][3];
      jready[1] = 0;
    }
  }
  return st;
}

static unsigned char _readjbutton(const unsigned char j)
{
  /*
  ** Read joystick, returning 0 if no fire button is pressed.
  */

  register unsigned char st, mask, pressed;
  register int count;

  if (j)
    mask = 12;
  else
    mask = 3;
  outportb(JOYPORT, 0xFF);
  st = inportb(JOYPORT);
  pressed = (j) ? (!(st & 64) || !(st & 128)) : (!(st & 16) || !(st & 32));
  count = 0;
  while ((st & mask) && (count++ < JOYCOUNTER))
    st = inportb(JOYPORT);
  return pressed;
}

static unsigned char _readjpos(const unsigned char j)
{
  /*
  ** Wait for the user to press a joystick button, then read joystick
  ** port; return 0 if user aborted calibration.
  */

  delay(1000);

  /* Wait for button pressed */
  while (!_readjbutton(j))
    if (kbhit())
      if (getch() == 0x1B)
        return 0;

  _readjport();

  /* Wait for button released */
  while (_readjbutton(j));

  return 1;
}

static unsigned int _manualdir(const unsigned char j)
{
  /*
  ** Direction check for manual calibration.
  */

  register unsigned int jstat = 0;

  if (x[j] < calib[j][0])
    jstat |= JOY_LT; /* Left */
  else
    jstat |= (x[j] > calib[j][1]) ? JOY_RT : 0; /* Right or center */

  if (y[j] < calib[j][2])
    jstat |= JOY_UP; /* Up */
  else
    jstat |= (y[j] > calib[j][3]) ? JOY_DN : 0; /* Down or center */

  return jstat;
}

unsigned char _usercalib(const unsigned char j)
{
  unsigned char i, l;
  int centx[2], centy[2], minx[2], maxx[2], miny[2], maxy[2];
  int ocentx, ocenty;
  char *msg[]={"Center stick and press a button (Esc abort)...",
               "Move stick to upper left and press a button (Esc abort)...",
               "Move stick to lower right and press a button (Esc abort)...",
               "While pressing button 5, press button 1 or 2 (Esc abort)...",
               "While pressing button 6, press button 1 or 2 (Esc abort)...",
               "While pressing button 7, press button 1 or 2 (Esc abort)...",
               "While pressing button 8, press button 1 or 2 (Esc abort)..."};

  printf("Calibration of joystick %d..\n", j + 1);
  for (i = 0; i < 3; i++)
  {
    printf(msg[i]);
#ifndef __BORLANDC__
    fflush(stdout);
#endif
    l = _readjpos(j);
    printf("\n");
    if (!l)
      return 0;

    /* Insert values for center, maximum, minimum variables */
    switch (i)
    {
      case 0:
        centx[j] = x[j];
        centy[j] = y[j];
        ocentx = x[1];
        ocenty = y[1];
        break;
      case 1:
        minx[j] = x[j];
        miny[j] = y[j];
        break;
      default:
        maxx[j] = x[j];
        maxy[j] = y[j];
        break;
    }
  }

  if (joymode == JOY_6BUTTONS || joymode == JOY_8BUTTONS)
  {
    unsigned int jstat;
    int limit, is8button;

    /*
    ** We must calibrate pseudo-directions corresponding to buttons 5/6
    ** (and 7/8, if JOY_8BUTTONS).
    */

    limit = (joymode == JOY_6BUTTONS) ? 2 : 4;
    is8button = (joymode == JOY_8BUTTONS);
    for (i = 0; i < limit; i++)
    {
      printf(msg[3 + i]);
#ifndef __BORLANDC__
      fflush(stdout);
#endif
      delay(1000);

retry:

      /* Wait for button pressed */
      while (!_readjbutton(j))
        if (kbhit())
          if (getch() == 0x1B)
            return 0;

      _readjport();

      jstat = 0;

      if (x[1] < ocentx - JOY_TOL)
        jstat |= JOY_LT; /* Left */
      else
        jstat |= (x[1] > ocentx + JOY_TOL) ? JOY_RT : 0; /* Right or center */

      if (y[1] < ocenty - JOY_TOL)
        jstat |= JOY_UP; /* Up */
      else
        jstat |= (y[1] > ocenty + JOY_TOL) ? JOY_DN : 0; /* Down or center */

      /* Wait for button released */
      while (_readjbutton(j));


      if ((jstat & ((i % 2) ? (JOY_UP | JOY_DN) : (JOY_LT | JOY_RT)))
          && ((i == 2) ? (jstat & JOY_RT) : 1) && ((i == 3) ? (jstat & JOY_DN) : 1))
      {
        switch (i)
        {
          case 0: /* Button 5: LEFT (8 buttons) or HORIZONTAL (6 buttons) */
            if (is8button)
              minx[1] = x[1];
            else
              if (x[1] > ocentx) /* Button corresponds to RIGHT */
              {
                maxx[1] = x[1];
                minx[1] = 2 * ocentx - x[1];
              }
              else /* Button corresponds to LEFT */
              {
                maxx[1] = 2 * ocentx - x[1];
                minx[1] = x[1];
              }
            break;
          case 1: /* Button 6: UP (8 buttons) or VERTICAL (6 buttons) */
            if (is8button)
              miny[1] = y[1];
            else
              if (y[1] > ocenty) /* Button corresponds to DOWN */
              {
                maxy[1] = y[1];
                miny[1] = 2 * ocenty - y[1];
              }
              else /* Button corresponds to UP */
              {
                maxy[1] = 2 * ocenty - y[1];
                miny[1] = y[1];
              }
            break;
          case 2: /* Button 7: RIGHT (only 8 buttons) */
            maxx[1] = x[1];
            break;
          case 3: /* Button 8: DOWN (only 8 buttons) */
            maxy[1] = y[1];
            break;
        }
      }
      else
        goto retry;
      printf("\n");
    }
    /* Calculate virtual lines for fire buttons 5..8 */

    centx[1] = ocentx;
    centy[1] = ocenty;

    /* line BEFORE WHICH left is stuffed */
    calib[1][0] = centx[1] - ((centx[1] - minx[1]) / SEGMENTS);

    /* line AFTER WHICH right is stuffed */
    calib[1][1] = centx[1] + ((maxx[1] - centx[1]) / SEGMENTS);

    /* line BEFORE WHICH up is stuffed */
    calib[1][2] = centy[1] - ((centy[1] - miny[1]) / SEGMENTS);

    /* line AFTER WHICH down is stuffed */
    calib[1][3] = centy[1] + ((maxy[1] - centy[1]) / SEGMENTS);
  }

  /* Calculate virtual lines for directions */

  printf("\n");

  /* line BEFORE WHICH left is stuffed */
  calib[j][0] = centx[j] - ((centx[j] - minx[j]) / SEGMENTS);

  /* line AFTER WHICH right is stuffed */
  calib[j][1] = centx[j] + ((maxx[j] - centx[j]) / SEGMENTS);

  /* line BEFORE WHICH up is stuffed */
  calib[j][2] = centy[j] - ((centy[j] - miny[j]) / SEGMENTS);

  /* line AFTER WHICH down is stuffed */
  calib[j][3] = centy[j] + ((maxy[j] - centy[j]) / SEGMENTS);

  return 1;
}

static void _readjoy(void)
{
  /* Set flags for each joystick (directions and fire buttons) */

  register int  v;
  register unsigned char st, j;

  st = _readjport();

  for (j = 0; j < 2; j++)
    if (jready[j])
    {
      jstatus[j] = 0;

      if (!(st & fire1[j]))
        jstatus[j] |= JOY_AF;

      if (!(st & fire2[j]))
        jstatus[j] |= JOY_BF;

      jstatus[j] |= _manualdir(j);

      if (joymode) /* 4, 6 or 8 buttons */
      {
        v = j ? 0 : 1;

        if (!(st & fire1[v]))
          jstatus[j] |= JOY_CF;

        if (!(st & fire2[v]))
          jstatus[j] |= JOY_DF;

        /*
        ** Buttons 5..8 are connected to directions on second joystick.
        ** Only one 6/8-button joystick can be connected, so we use jstatus[0]
        ** instead of jstatus[j].
        */

        if (joymode == JOY_6BUTTONS)
        {
          v = _manualdir(1);

          if (v & (JOY_LT | JOY_RT))
            jstatus[0] |= JOY_EF;

          if (v & (JOY_UP | JOY_DN))
            jstatus[0] |= JOY_FF;

          j = 2;
        }
        else
          if (joymode == JOY_8BUTTONS)
          {
            v = _manualdir(1);

            if (v & JOY_LT)
              jstatus[0] |= JOY_EF;

            if (v & JOY_UP)
              jstatus[0] |= JOY_FF;

            if (v & JOY_RT)
              jstatus[0] |= JOY_GF;

            if (v & JOY_DN)
              jstatus[0] |= JOY_HF;

            j = 2;
          }
      }
    }
}

static unsigned char _joyready(const unsigned char j)
{
  /*
  ** Probe joystick, returning 1 if it is present, 0 otherwise.
  ** Presence is determined in this way:
  ** - if position (x and y) and both fire buttons read through bios are 0,
  **   joystick is absent;
  ** - otherwise, it is present.
  **
  ** If 'j' is 0, probes the first joystick; if 'j' is 1, probes the second.
  */

  union REGS regs;
  register int x, y, fa, fb;

  /* Read position */
  regs.h.ah = 0x84;
  regs.x.dx = 1;
  int86(0x15, &regs, &regs);
  if (regs.x.cflag)
    return 0; /* Error: joystick not found */

  if (j)
  {
    x = regs.x.cx;
    y = regs.x.dx;
  }
  else
  {
    x = regs.x.ax;
    y = regs.x.bx;
  }

  /* Read fire buttons */
  regs.h.ah = 0x84;
  regs.x.dx = 0;
  int86(0x15, &regs, &regs);
  if (regs.x.cflag)
    return 0; /* Error: joystick not found */

  if (j)
  {
    fa = !GetBit(regs.x.ax, 6);
    fb = !GetBit(regs.x.ax, 7);
  }
  else
  {
    fa = !GetBit(regs.x.ax, 4);
    fb = !GetBit(regs.x.ax, 5);
  }

  if (x + y + fa + fb == 0)
    return 0;
  else
    return 1;
}

static void _savecalib(const char *filename, const unsigned char njoy)
{
  /*
  ** Store number of joysticks connected, joystick mode and
  ** calibration informations into filename (which is created
  ** or overwritten). No errors are reported; if filename is NULL,
  ** calibration is not saved.
  */

  if (filename != NULL)
  {
    register int i, l;
    int h;
    CalibRecord cr;

    cr.njoy = njoy;
    cr.joymode = joymode;
    for (i = 0; i < 2; i++)
      for (l = 0; l < 4; l++)
        cr.calib[i][l] = calib[i][l];

    h = open(filename, O_WRONLY | O_BINARY | O_CREAT | O_TRUNC, S_IWRITE);

    if (h != -1)
    {
      write(h, (void *) &cr, sizeof(cr));
      close(h);
    }
  }
}

static unsigned char _checkcalib(const unsigned char j)
{
  /*
  ** Check if joystick j is centered, according to current calibration;
  ** returns JOY_FIRST or JOY_SECOND (according to j) if it needs to
  ** be re-calibrated, JOY_NONE otherwise.
  ** This routine does two checks, at 1/3-second interval each:
  ** if both return joystick off-centered, the program assumes that
  ** calibration is needed.
  */

  int try = 0;
  unsigned int jstat;

retry:

  _readjport();
  jstat = _manualdir(j);

  if (jstat)
    if (try)
      return j + 1;
    else
    {
      delay(300);
      try++;
      goto retry;
    }

  return JOY_NONE;
}

static unsigned char _loadcalib(const char *filename, const unsigned char njoy)
{
  /*
  ** Load calibration and returns:
  **
  ** JOY_NONE       if mode and joy # are correct and joys appear centered
  ** JOY_FIRST      if joy #1 must be re-calibrated
  ** JOY_SECOND     if joy #2 must be re-calibrated
  ** JOY_BOTH       if both joys need to be re-calibrated
  */

  if (filename != NULL)
  {
    int h, i, l;
    unsigned char oldnjoy;
    CalibRecord cr;

    h = open(filename, O_RDONLY | O_BINARY, S_IREAD);

    /* File not found: re-calibration */
    if (h == -1)
      return njoy;

    /* Wrong size: remove file and force re-calibration */
    if (filelength(h) != sizeof(cr))
    {
      close(h);
      remove(filename);
      return njoy;
    }

    l = read(h, (void *) &cr, sizeof(cr)) != sizeof(cr);
    close(h);

    /* Error reading: re-calibration */
    if (l)
      return njoy;

    /* Mode incompatibility: re-calibration */

    if ((joymode >= JOY_6BUTTONS && cr.joymode < JOY_6BUTTONS) ||
        (cr.joymode >= JOY_6BUTTONS && joymode < JOY_6BUTTONS))
      return njoy;

    oldnjoy = njoy;
    for (i = 0; i < 2; i++)
      for (l = 0; l < 4; l++)
        calib[i][l] = cr.calib[i][l];

    /* Different number of joysticks: re-calibration limited to the new joystick */
    if (oldnjoy != cr.njoy)
    {
      unsigned char t = 0;

      for (i = 1; i < 3; i++)
        if ((oldnjoy & i) && (cr.njoy & i))
          t |= _checkcalib(i - 1);
        else
          if (oldnjoy & i)
            t |= i;
      return t;
    }

    return JOY_NONE;
  }
  else
    return njoy;   /* NULL file specified: re-calibration */
}

/****************************************************************************/
/******************* External functions: joystick interface *****************/
/****************************************************************************/

unsigned char InstallJoystick(const unsigned char mode, const unsigned char skipjoy, const char *calibfile)
{
  /*
  ** This function checks and returns the number of installed joysticks.
  **
  ** 'mode' can be:
  **
  ** JOY_2BUTTONS   2-button joysticks (one or two)
  ** JOY_4BUTTONS   4-button joysticks (one or two)
  ** JOY_6BUTTONS   6-button joystick (only one)
  ** JOY_8BUTTONS   8-button joystick (only one)
  **
  ** 'skipjoy' can be:
  **
  ** JOY_NONE    all joysticks present are installed and calibrated
  ** JOY_FIRST   skip installation/calibration for first joystick
  ** JOY_SECOND  skip installation/calibration for second joystick
  **
  ** Most of the times, skipjoy will be JOY_NONE (all joysticks installed
  ** and calibrated) or JOY_SECOND (only first joystick is installed and
  ** calibrated, e.g. when your program uses just one joystick).
  **
  ** 'calibfile' is the name of the file from which calibration must be
  ** loaded (if exists) or to which it will be saved (if does NOT exist).
  ** If 'calibfile' is NULL, calibration won't be saved, i.e. the user will
  ** have to calibrate joystick(s) at each run.
  ** If 'mode' is JOY_2BUTTONS or JOY_4BUTTONS and mode determined from
  ** configuration file is JOY_6BUTTONS/JOY_8BUTTONS (or viceversa),
  ** calibration will be forced; the same will happen if number of joysticks
  ** connected is different or if the calibration appears incorrect.
  ** If calibration is aborted for one joystick, that joystick will be
  ** reported as not connected (it will be re-calibrated the next time you
  ** run the program).
  **
  ** Values returned
  ** JOY_NONE    if no joystick is installed
  ** JOY_FIRST   if only joystick 1 is installed
  ** JOY_SECOND  if only joystick 2 is installed
  ** JOY_BOTH    if both joysticks (or one 6/8-button joystick) are installed
  */

  register unsigned char retval = 0, j;
  unsigned char j2calib;

  joymode = mode;

  /* Determine joysticks attached */
  for (j = 0; j < 2; j++)
    if ((jready[j] = _joyready(j)) != 0)
      retval += j + 1;

  /* Revert to 4-button mode if 6/8-button joystick not found */
  if ((joymode == JOY_6BUTTONS || joymode == JOY_8BUTTONS) && retval < JOY_BOTH)
    joymode = JOY_4BUTTONS;

  /* Prompt the user to calibrate joystick(s) */
  j2calib = _loadcalib(calibfile, retval);
  for (j = 0; j < 2; j++)
  {
    if ((skipjoy & (j + 1)) || (jready[j] && (j2calib & (j + 1)) && !_usercalib(j)))
    {
      jready[j] = 0;
      retval -= j + 1;
    }

    /* No need to calibrate second (virtual) joystick when in 6/8-button mode */
    if (joymode == JOY_6BUTTONS || joymode == JOY_8BUTTONS)
      j = 2;
  }

  if (retval)
    _savecalib(calibfile, retval);

  return retval;
}

void ReadJoystick(unsigned int *first, unsigned int *second)
{
  /*
  ** Read both joysticks, returning into "first" and "second" an
  ** unsigned integer with the following flags set:
  **
  ** JOY_UP    Joystick moved up
  ** JOY_DN    Joystick moved down
  ** JOY_LT    Joystick moved to left
  ** JOY_RT    Joystick moved to right
  ** JOY_AF    Button 1 pressed
  ** JOY_BF    Button 2 pressed
  ** JOY_CF    Button 3 pressed (joymode >= JOY_4BUTTONS)
  ** JOY_DF    Button 4 pressed (joymode >= JOY_4BUTTONS)
  ** JOY_EF    Button 5 pressed (joymode >= JOY_6BUTTONS)
  ** JOY_FF    Button 6 pressed (joymode >= JOY_6BUTTONS)
  ** JOY_GF    Button 7 pressed (joymode = JOY_8BUTTONS)
  ** JOY_HF    Button 8 pressed (joymode = JOY_8BUTTONS)
  **
  ** If a NULL pointer is passed, no data is stored (useful if data
  ** for only one joystick are needed).
  */

  _readjoy();
  if (first)
    *first = jstatus[0];
  if (second)
    *second = jstatus[1];
}
