unit ESBMaths;

{
	ESBMaths - contains useful Mathematical routines for Delphi 3 & 4

	 1999 ESB Consultancy

	These routines are used by ESB Consultancy within the
	development of their Customised Applications, and have been
	under Development since the early Turbo Pascal days.
	Many of the routines were developed for specific needs

	ESB Consultancy retains full copyright.

	ESB Consultancy grants users of this code royalty free rights
	to do with this code as they wish.

	ESB Consultancy makes no guarantees nor excepts any liabilities
	due to the use of these routines

	We does ask that if this code helps you in you development
	that you send as an email mailto:esb@gold.net.au or even
	a local postcard. It would also be nice if you gave us a
	mention in your About Box or Help File.

	ESB Consultancy Home Page: http://www.gold.net.au/~esb

	Mail Address: PO Box 2259, Boulder, WA 6432 AUSTRALIA

	v1.1 22 Jan 1999 - Improved Delphi 4 support
		- Added: SumLWArray, SumSqEArray, SumSqDiffEArray, SumXYEArray
		- Added: IsPositiveEArray
		- Added: Geometric and Harmonic Means
		- Added ESBMean, Median and Mode
		- Added Routines for Variances and Means for Populations & Samples
		- Added Quartile Calculations
		- Fixed a couple of minor bugs
		- Fixed bug in XY2Polar reported by Wolfgang Werner
		- Added Help File

	v1.01 17 July 1998 - Added some improvements suggested by Rory Daulton
		- Added Factorials using Extendeds
		- Added Permutations using Extendeds
		- Added Combinations using Extendeds
		- Added SwapI32 to Swap Integers
		- Added SwapC to Swap Cardinals
		- Added Factorials computed using Extendeds
		- Added Permutations computed using Extendeds
		- Added Binomial Coefficients (Combinations) computed using Extendeds

	v1.00 17 April 1998 - first public/freeware release

	Also check out our Shareware application: ESBPDF Analysis which provides
		User-Friendly Probability Analysis - available for download on above site.

}

interface

{$DEFINE UseMath} 	{Comment out this line if you don't want to use the Delphi Math Library}
				{ Desktop versions of Delphi do not include the Math Library }

{$IFNDEF Ver120}
type
	LongWord = Cardinal;
{$ENDIF}

type
	{: Used for a Bit List of 16 bits from 15 -> 0 }
	TBitList = Word;

var
	{: Tolerance used to decide when float close enough to zero }
	ESBTolerance: Double = 0.0000001;

{--- Integer Operations ---}

{: Returns True if X1 and X2 are within ESBTolerance of each other }
function SameFloat (const X1, X2: Extended): Boolean;

{: Returns True if X is within ESBTolerance of 0 }
function FloatIsZero (const X: Extended): Boolean;

{: Increments a Byte up to Limit. If B >= Limit no increment occurs. }
procedure IncLim (var B: Byte; const Limit: Byte);

{: Increments a ShortInt up to Limit. If B >= Limit no increment occurs. }
procedure IncLimSI (var B: ShortInt; const Limit: ShortInt);

{: Increments a Word up to Limit. If B >= Limit no increment occurs. }
procedure IncLimW (var B: Word; const Limit: Word);

{: Increments an Integer up to Limit. If B >= Limit no increment occurs. }
procedure IncLimI (var B: Integer; const Limit: Integer);

{: Increments a LongInt up to Limit. If B >= Limit no increment occurs. }
procedure IncLimL (var B: LongInt; const Limit: LongInt);

{: Decrements a Byte down to Limit. If B <= Limit no increment occurs. BASM }
procedure DecLim (var B: Byte; const Limit: Byte);

{: Decrements a ShortInt down to Limit. If B <= Limit no increment occurs. }
procedure DecLimSI (var B: ShortInt; const Limit: ShortInt);

{: Decrements a Word down to Limit. If B <= Limit no increment occurs. }
procedure DecLimW (var B: Word; const Limit: Word);

{: Decrements an Integer down to Limit. If B <= Limit no increment occurs. }
procedure DecLimI (var B: Integer; const Limit: Integer);

{: Decrements a LongInt down to Limit. If B <= Limit no increment occurs. }
procedure DecLimL (var B: LongInt; const Limit: LongInt);

{: Returns the maximum value between two Bytes. BASM }
function MaxB (const B1, B2: Byte): Byte;

{: Returns the minimum value between two Bytes. BASM }
function MinB (const B1, B2: Byte): Byte;

{: Returns the maximum value between two ShortInts. }
function MaxSI (const B1, B2: ShortInt): ShortInt;

{: Returns the minimum value between two ShortInts. }
function MinSI (const B1, B2: ShortInt): ShortInt;

{: Returns the maximum value between two Words. BASM }
function MaxW (const B1, B2: Word): Word;

{: Returns the minimum value between two Words. BASM }
function MinW (const B1, B2: Word): Word;

{: Returns the maximum value between two Integers. }
function MaxI (const B1, B2: Integer): Integer;

{: Returns the minimum value between two Integers. }
function MinI (const B1, B2: Integer): Integer;

{: Returns the maximum value between two LongInts. }
function MaxL (const B1, B2: LongInt): LongInt;

{: Returns the minimum value between two LongInts. }
function MinL (const B1, B2: LongInt): LongInt;

{: Swap Two Bytes. BASM - using Registers }
procedure SwapB (var B1, B2: Byte); register;

{: Swap Two ShortInts. BASM - using Registers }
procedure SwapSI (var B1, B2: ShortInt); register;

{: Swap Two Words. BASM - using Registers }
procedure SwapW (var B1, B2: Word); register;

{: Swap Two Integers. BASM - using Registers }
procedure SwapI (var B1, B2: SmallInt); register;

{: Swap Two LongInts. BASM - using Registers }
procedure SwapL (var B1, B2: LongInt); register;

{: Swap Two Integers (32-bit). BASM - using Registers }
procedure SwapI32 (var B1, B2: Integer); register;

{: Swap Two Cardinals. BASM - using Registers }
procedure SwapC (var B1, B2: Cardinal); register;

{: Returns: <p>
	-1  if B < 0 <p>
	 0  if B = 0 <p>
	 1  if B > 0  BASM }
function Sign (const B: LongInt): ShortInt;

{: Returns the Maximum of 4 Words - BASM }
function Max4Word (const X1, X2, X3, X4: Word): Word;

{: Returns the Minimum of 4 Words - BASM }
function Min4Word (const X1, X2, X3, X4: Word): Word;

{: Returns the Maximum of 3 Words - BASM }
function Max3Word (const X1, X2, X3: Word): Word;

{: Returns the Minimum of 3 Words - BASM }
function Min3Word (const X1, X2, X3: Word): Word;

{: Returns the Maximum of an array of Bytes }
function MaxBArray (const B: array of Byte): Byte;

{: Returns the Maximum of an array of Words }
function MaxWArray (const B: array of Word): Word;

{: Returns the Maximum of an array of ShortInts }
function MaxSIArray (const B: array of ShortInt): ShortInt;

{: Returns the Maximum of an array of Integers }
function MaxIArray (const B: array of Integer): Integer;

{: Returns the Maximum of an array of LongInts }
function MaxLArray (const B: array of LongInt): LongInt;

{: Returns the Minimum of an array of Bytes }
function MinBArray (const B: array of Byte): Byte;

{: Returns the Minimum of an array of Words }
function MinWArray (const B: array of Word): Word;

{: Returns the Minimum of an array of ShortInts }
function MinSIArray (const B: array of ShortInt): ShortInt;

{: Returns the Minimum of an array of Integers }
function MinIArray (const B: array of Integer): Integer;

{: Returns the Minimum of an array of LongInts }
function MinLArray (const B: array of LongInt): LongInt;

{: Returns the Sum of an array of Bytes. All Operation in Bytes }
function SumBArray (const B: array of Byte): Byte;

{: Returns the Sum of an array of Bytes. All Operation in Words }
function SumBArray2 (const B: array of Byte): Word;

{: Returns the Sum of an array of ShortInts. All Operation in ShortInts }
function SumSIArray (const B: array of ShortInt): ShortInt;

{: Returns the Sum of an array of ShortInts. All Operation in Integers }
function SumSIArray2 (const B: array of ShortInt): Integer;

{: Returns the Sum of an array of Words. All Operation in Words }
function SumWArray (const B: array of Word): Word;

{: Returns the Sum of an array of Words. All Operation in Longints }
function SumWArray2 (const B: array of Word): LongInt;

{: Returns the Sum of an array of Integers. All Operation in Integers }
function SumIArray (const B: array of Integer): Integer;

{: Returns the Sum of an array of Longints. All Operation in Longints }
function SumLArray (const B: array of LongInt): LongInt;

{: Returns the Sum of an array of LongWord. All Operation in LongWords }
function SumLWArray (const B: array of LongWord): LongWord;

{: ISqrt (I) computes INT (SQRT (I)), that is, the integral part of the
  square root of integer I. It does not check for negative arguments.
  For all arguments 0..32767 the correct result is returned.
  Routine by Norbert Joffa. BASM }
function ISqrt (const I: SmallInt): SmallInt; register;

{--- Floating Point Operations ---}

{: Returns the 80x87 Control Word <p>
  15-12 Reserved  <p>
	On 8087/80287 12 was Infinity Control <p>
	 0 Projective <p>
	 1 Affine <p>
  11-10 Rounding Control <p>
    00 Round to nearest even <p>
    01 Round Down <p>
    10 Round Up <p>
    11 Chop - Truncate towards Zero <p>
  9-8  Precision Control <p>
    00 24 bits Single Precision <p>
    01 Reserved <p>
    10 53 bits Double Precision <p>
    11 64 bits Extended Precision (Default) <p>
  7-6  Reserved <p>
	On 8087 7 was Interrupt Enable Mask <p>
  5  Precesion Exception Mask <p>
  4  Underflow Exception Mask <p>
  3  Overflow Exception Mask <p>
  2  Zero Divide Exception Mask <p>
  1  Denormalised Operand Exception Mask <p>
  0  Invalid Operation Exception Mask <p>
  BASM }
function Get87ControlWord: TBitList;

{: Sets the 80x87 Control Word <p>
  15-12 Reserved <p>
	On 8087/80287 12 was Infinity Control <p>
	 0 Projective <p>
	 1 Affine <p>
  11-10 Rounding Control <p>
    00 Round to nearest even <p>
    01 Round Down <p>
    10 Round Up <p>
    11 Chop - Truncate towards Zero <p>
  9-8  Precision Control <p>
    00 24 bits Single Precision <p>
    01 Reserved <p>
    10 53 bits Double Precision <p>
    11 64 bits Extended Precision (Default) <p>
  7-6  Reserved <p>
	On 8087 7 was Interrupt Enable Mask <p>
  5  Precesion Exception Mask <p>
  4  Underflow Exception Mask <p>
  3  Overflow Exception Mask <p>
  2  Zero Divide Exception Mask <p>
  1  Denormalised Operand Exception Mask <p>
  0  Invalid Operation Exception Mask <p>
  BASM }
procedure Set87ControlWord (const CWord: TBitList); register;

{: Returns  <p>
	-1 if X < 0 <p>
	 0 if X = 0 <p>
	 1 if X > 0 <p>
 }
function Sgn (const X: Extended): ShortInt;

{: Returns the straight line Distance between (X1, Y1) and (X2, Y2) }
function Distance (const X1, Y1, X2, Y2: Extended): Extended;

{: Performs Floating Point Modulus. ExtMod := X - Floor ( X / Y ) * Y }
function ExtMod (const X, Y: Extended): Extended;

{: Performs Floating Point Remainder. ExtRem := X - Int ( X / Y ) * Y }
function ExtRem (const X, Y: Extended): Extended;

{: Returns X mod Y for Comp Data Types }
function CompMOD (const X, Y: Comp): Comp;

{: Converts Polar Co-ordinates into Cartesion Co-ordinates }
procedure Polar2XY (const Rho, Theta: Extended; var X, Y: Extended);

{: Converts Cartesian Co-ordinates to Polar Co-ordinates }
procedure XY2Polar (const X, Y: Extended; var Rho, Theta: Extended);

{: Converts Degrees/Minutes/Seconds into an Extended Real }
function DMS2Extended (const Degs, Mins, Secs: Extended): Extended;

{: Converts an Extended Real into Degrees/Minutes/Seconds }
procedure Extended2DMS (const X: Extended; var Degs, Mins, Secs: Extended);

{: Returns the Maximum of two Extended Reals }
function MaxExt (const X, Y: Extended): Extended;

{: Returns the Minimum of two Extended Reals }
function MinExt (const X, Y: Extended): Extended;

{: Returns the Maximum of an array of Extended Reals }
function MaxEArray (const B: array of Extended): Extended;

{: Returns the Minimum of an array of Extended Reals }
function MinEArray (const B: array of Extended): Extended;

{: Returns the Maximum of an array of Single Reals }
function MaxSArray (const B: array of Single): Single;

{: Returns the Maximum of an array of Single Reals }
function MinSArray (const B: array of Single): Single;

{: Returns the Maximum of an array of Comp Reals }
function MaxCArray (const B: array of Comp): Comp;

{: Returns the Minimum of an array of Comp Reals }
function MinCArray (const B: array of Comp): Comp;

{: Returns the Sum of an Array of Single Reals }
function SumSArray (const B: array of Single): Single;

{: Returns the Sum of an Array of Extended Reals }
function SumEArray (const B: array of Extended): Extended;

{: Returns the Sum of the Square of an Array of Extended Reals }
function SumSqEArray (const B: array of Extended): Extended;

{: Returns the Sum of the Square of the difference of
	an Array of Extended Reals from a given Value }
function SumSqDiffEArray (const B: array of Extended; Diff: Extended): Extended;

{: Returns the Sum of the Pairwise Product of two
	Arrays of Extended Reals }
function SumXYEArray (const X, Y: array of Extended): Extended;

{: Returns the Sum of an Array of Comp Reals }
function SumCArray (const B: array of Comp): Comp;

{: Returns A! i.e Factorial of A - only values up to 1547 are handled
	returns 0 if larger }
function FactorialX (A: Cardinal): Extended;

{: Returns nPr i.e Permutation of r objects from n.
	Only values of N up to 1547 are handled	returns 0 if larger
	If R > N  then 0 is returned }
function PermutationX (N, R: Cardinal): Extended;

{: Returns nCr i.e Combination of r objects from n.
	These are also known as the Binomial Coefficients
	Only values of N up to 1547 are handled	returns 0 if larger
	If R > N  then 0 is returned }

function BinomialCoeff (N, R: Cardinal): Extended;

{: Returns True if all elements of X > ESBTolerance }
function IsPositiveEArray (const X: array of Extended): Boolean;

{: Returns the Geometric Mean of the values }
function GeometricMean (const X: array of Extended): Extended;

{: Returns the Harmonic Mean of the values }
function HarmonicMean (const X: array of Extended): Extended;

{: Returns the Arithmetic Mean of the Values }
function ESBMean (const X: array of Extended): Extended;

{: Returns the Variance of the Values, assuming a Sample.
	Square root this value to get Standard Deviation }
function SampleVariance (const X: array of Extended): Extended;

{: Returns the Variance of the Values, assuming a Population.
	Square root this value to get Standard Deviation }
function PopulationVariance (const X: array of Extended): Extended;

{: Returns the Mean and Variance of the Values, assuming a Sample.
	Square root the Variance to get Standard Deviation }
procedure SampleVarianceAndMean (const X: array of Extended;
	var Variance, Mean: Extended);

{: Returns the Mean and Variance of the Values, assuming a Population.
	Square root the Variance to get Standard Deviation }
procedure PopulationVarianceAndMean (const X: array of Extended;
	var Variance, Mean: Extended);

{: Returns the Median (2nd Quartiles) of the Values. The array
	MUST be sorted before using this operation }
function GetMedian (const SortedX: array of Extended): Extended;

{: Returns the Mode (most frequent) of the Values. The array
	MUST be sorted before using this operation. Function is False
	if no Mode exists }
function GetMode (const SortedX: array of Extended; var Mode: Extended): Boolean;

{: Returns the 1st and 3rd Quartiles - Median is 2nd Quartiles - of the Values.
	The array	MUST be sorted before using this operation }
procedure GetQuartiles (const SortedX: array of Extended; var Q1, Q3: Extended);

implementation

uses
{$IFDEF UseMath}
	Math,
{$ENDIF}
	SysUtils;

procedure IncLim (var B: Byte; const Limit: Byte);
begin
	if B < Limit then
		Inc (B);
end;

procedure IncLimSI (var B: ShortInt; const Limit: ShortInt);
begin
	if B < Limit then
		Inc (B);
end;

procedure IncLimW (var B: Word; const Limit: Word);
begin
	if B < Limit then
		Inc (B);
end;

procedure IncLimI (var B: Integer; const Limit: Integer);
begin
	if B < Limit then
		Inc (B);
end;

procedure IncLimL (var B: LongInt; const Limit: LongInt);
begin
	if B < Limit then
		Inc (B);
end;

procedure DecLim (var B: Byte; const Limit: Byte);
begin
	if B > Limit then
		Dec (B);
end;

procedure DecLimSI (var B: ShortInt; const Limit: ShortInt);
begin
	if B > Limit then
		Dec (B);
end;

procedure DecLimW (var B: Word; const Limit: Word);
begin
	if B > Limit then
		Dec (B);
end;

procedure DecLimI (var B: Integer; const Limit: Integer);
begin
	if B > Limit then
		Dec (B);
end;

procedure DecLimL (var B: LongInt; const Limit: LongInt);
begin
	if B > Limit then
		Dec (B);
end;

function MaxB (const B1, B2: Byte): Byte;
begin
	if B1 > B2 then
		Result := B1
	 else
		Result := B2;
end;

function MinB (const B1, B2: Byte): Byte;
begin
	if B1 < B2 then
		Result := B1
	else
		Result := B2;
end;

function MaxSI (const B1, B2: ShortInt): ShortInt;
begin
	if B1 > B2 then
		Result := B1
	else
		Result := B2;
end;

function MinSI (const B1, B2: ShortInt): ShortInt;
begin
	if B1 < B2 then
		Result := B1
	else
		Result := B2;
end;

function MaxW (const B1, B2: Word): Word;
begin
	if B1 > B2 then
		Result := B1
	else
		Result := B2;
end;

function MinW (const B1, B2: Word): Word;
begin
	if B1 < B2 then
		Result := B1
	else
		Result := B2;
end;

function MaxI (const B1, B2: Integer): Integer;
begin
	if B1 > B2 then
		Result := B1
	else
		Result := B2;
end;

function MinI (const B1, B2: Integer): Integer;
begin
	if B1 < B2 then
		Result := B1
	else
		Result := B2;
end;

function MaxL (const B1, B2: LongInt): LongInt;
begin
	if B1 > B2 then
		Result := B1
	else
		Result := B2;
end;

function MinL (const B1, B2: LongInt): LongInt;
begin
	if B1 < B2 then
		Result := B1
	else
		Result := B2;
end;

procedure SwapB (var B1, B2: Byte); assembler;
{Rory Daulton suggested this better optimised version}
asm
  mov  cl, Byte Ptr [EAX];
  xchg cl, Byte Ptr [EDX]
  mov  Byte Ptr [EAX], cl
end;

procedure SwapSI (var B1, B2: ShortInt); assembler;
{Rory Daulton suggested this better optimised version}
asm
  mov  cl, Byte Ptr [EAX];
  xchg cl, Byte Ptr [EDX]
  mov  Byte Ptr [EAX], cl
end;

procedure SwapW (var B1, B2: Word); assembler;
{Rory Daulton suggested this better optimised version}
asm
  mov  cx, Word Ptr [EAX];
  xchg cx, Word Ptr [EDX]
  mov  word Ptr [EAX], cx
end;

procedure SwapI (var B1, B2: SmallInt); assembler;
{Rory Daulton suggested this better optimised version}
asm
  mov  cx, Word Ptr [EAX];
  xchg cx, Word Ptr [EDX]
  mov  word Ptr [EAX], cx
end;

procedure SwapL (var B1, B2: LongInt); assembler;
{Rory Daulton suggested this better optimised version}
asm
  mov  ecx, [EAX];
  xchg ecx, [EDX]
  mov  [EAX], ecx
end;

procedure SwapI32 (var B1, B2: Integer); assembler;
{Rory Daulton suggested this better optimised version}
asm
  mov  ecx, [EAX];
  xchg ecx, [EDX]
  mov  [EAX], ecx
end;

procedure SwapC (var B1, B2: Cardinal); assembler;
{Rory Daulton suggested this better optimised version}
asm
  mov  ecx, [EAX];
  xchg ecx, [EDX]
  mov  [EAX], ecx
end;

function Sign (const B: LongInt): ShortInt;
begin
	if B < 0 then
		Result := -1
	else if B = 0 then
		Result := 0
	else
		Result := 1;
end;

function Max4Word (const X1, X2, X3, X4: Word): Word;
begin
	Result := X1;
	if X2 > Result then
		Result := X2;
	if X3 > Result then
		Result := X3;
	if X4 > Result then
		Result := X4;
end;

function Min4Word (const X1, X2, X3, X4: Word): Word;
begin
	Result := X1;
	if X2 < Result then
		Result := X2;
	if X3 < Result then
		Result := X3;
	if X4 < Result then
		Result := X4;
end;

function Max3Word (const X1, X2, X3: Word): Word; assembler;
begin
	Result := X1;
	if X2 > Result then
		Result := X2;
	if X3 > Result then
		Result := X3;
end;

function Min3Word (const X1, X2, X3: Word): Word; assembler;
begin
	Result := X1;
	if X2 < Result then
		Result := X2;
	if X3 < Result then
		Result := X3;
end;

function MaxBArray (const B: array of Byte): Byte;
var
	I: Integer;
begin
	Result := B [Low (B)];
	for I:= Low (B) + 1 to High (B) do
		if B [I] > Result then
			Result := B [I];
end;

function MaxWArray (const B: array of Word): Word;
var
	I: Integer;
begin
	Result := B [Low (B)];
	for I:= Low (B) + 1 to High (B) do
		if B [I] > Result then
		Result := B [I];
end;

function MaxIArray (const B: array of Integer): Integer;
var
	I: Integer;
begin
	Result := B [Low (B)];
	for I:= Low (B) + 1 to High (B) do
		if B [I] > Result then
			   Result := B [I];
end;

function MaxSIArray (const B: array of ShortInt): ShortInt;
var
	 I: Integer;
begin
	Result := B [Low (B)];
	for I := Low (B) + 1 to High (B) do
	if B [I] > Result then
		Result := B [I];
end;

function MaxLArray (const B: array of LongInt): LongInt;
var
	I: Integer;
begin
	Result := B [Low (B)];
	for I := Low (B) + 1 to High (B) do
		if B [I] > Result then
			Result := B [I];
end;

function MinBArray (const B: array of Byte): Byte;
var
	I: Integer;
begin
	Result := B [Low (B)];
	for I := Low (B) + 1 to High (B) do
		if B [I] < Result then
		   Result := B [I];
end;

function MinWArray (const B: array of Word): Word;
var
	I: Integer;
begin
	Result := B [Low (B)];
	for I := Low (B) + 1 to High (B) do
		if B [I] < Result then
			  Result := B [I];
end;

function MinIArray (const B: array of Integer): Integer;
var
	I: Integer;
begin
	Result := B [Low (B)];
	for I := Low (B) + 1 to High (B) do
		if B [I] < Result then
			 Result := B [I];
end;

function MinSIArray (const B: array of ShortInt): ShortInt;
var
	I: Integer;
begin
	Result := B [Low (B)];
	for I := Low (B) + 1 to High (B) do
		if B [I] < Result then
			Result := B [I];
end;

function MinLArray (const B: array of LongInt): LongInt;
var
	I: Integer;
begin
	Result := B [Low (B)];
	for I := Low (B) + 1 to High (B) do
		if B [I] < Result then
			Result := B [I];
end;

function ISqrt (const I: SmallInt): SmallInt; assembler;
asm
	push ebx

	MOV   CX, AX  { load argument }
	MOV   AX, -1  { init result }
	CWD           { init odd numbers to -1 }
	XOR   BX, BX  { init perfect squares to 0 }
@loop:
	INC   AX      { increment result }
	INC   DX      { compute }
	INC   DX      {  next odd number }
	ADD   BX, DX  { next perfect square }
	CMP   BX, CX  { perfect square > argument ? }
	JBE   @loop   { until square greater than argument }

	pop ebx
end;

function SumBArray (const B: array of Byte): Byte;
var
	I: Integer;
begin
	Result := B [Low (B)];
	for I:= Low (B) + 1 to High (B) do
		Result := Result + B [I];
end;

function SumBArray2 (const B: array of Byte): Word;
var
	I: Integer;
begin
	Result := B [Low (B)];
	for I:= Low (B) + 1 to High (B) do
		Result := Result + B [I];
end;

function SumSIArray (const B: array of ShortInt): ShortInt;
var
	I: Integer;
begin
	Result := B [Low (B)];
	for I:= Low (B) + 1 to High (B) do
		Result := Result + B [I];
end;

function SumSIArray2 (const B: array of ShortInt): Integer;
var
	I: Integer;
begin
	Result := B [Low (B)];
	for I:= Low (B) + 1 to High (B) do
		Result := Result + B [I];
end;

function SumWArray (const B: array of Word): Word;
var
	I: Integer;
begin
	Result := B [Low (B)];
	for I:= 0 to High (B) do
		Result := Result + B [I];
end;

function SumWArray2 (const B: array of Word): LongInt;
var
	I: Integer;
begin
	Result := B [Low (B)];
	for I:= Low (B) + 1 to High (B) do
		Result := Result + B [I];
end;

function SumIArray (const B: array of Integer): Integer;
var
	I: Integer;
begin
	Result := B [Low (B)];
	for I:= Low (B) + 1 to High (B) do
		Result := Result + B [I];
end;

function SumLArray (const B: array of LongInt): LongInt;
var
	I: Integer;
begin
	Result := B [Low (B)];
	for I:= Low (B) + 1 to High (B) do
		Result := Result + B [I];
end;

function SumLWArray (const B: array of LongWord): LongWord;
var
	I: Integer;
begin
	Result := B [Low (B)];
	for I:= Low (B) + 1 to High (B) do
		Result := Result + B [I];
end;

function Get87ControlWord: TBitList; assembler;
var
	Temp: Word;
asm
	fstcw [Temp]  		{ Get '87 Control Word }
	mov  ax, [Temp] 	{ Leave in AX for function }
end;

procedure Set87ControlWord (const CWord: TBitList); assembler;
var
	Temp: Word;
asm
	mov   [Temp], ax
	fldcw [Temp] 		{ Load '87 Control Word }
end;

procedure Polar2XY (const Rho, Theta: Extended; var X, Y: Extended);
begin
	{$IFDEF UseMath}
	SinCos (Theta, X, Y); { quite fast }
	{$ELSE}
	X := Sin (Theta);
	Y := Cos (Theta);
	{$ENDIF}
	 X := Rho * X;
	 Y := Rho * Y;
end;

procedure XY2Polar (const X, Y: Extended; var Rho, Theta: Extended);
begin
	Rho := Sqrt (Sqr (X) + Sqr (Y));
	if Abs (X) > ESBTolerance then
	begin
		Theta := ArcTan (abs (Y) / abs (X));
		if Sgn (X) = 1 then
		begin
			if Sgn (Y) = -1 then
				Theta := 2 * Pi - Theta
		end
		else
		begin
			if Sgn (Y) = 1 then
				Theta := Pi - Theta
			else
				Theta := Pi + Theta
		end;
	end
	else
		Theta := Sgn (Y) * Pi / 2.0
end;

function Sgn (const X: Extended): ShortInt;
begin
	if X < 0.0 then
		Result := -1
	else if X = 0.0 then
		Result := 0
	else
		Result := 1
end;

function DMS2Extended (const Degs, Mins, Secs: Extended): Extended;
begin
	Result := Degs + Mins / 60.0 + Secs / 3600.0
end;

procedure Extended2DMS (const X: Extended; var Degs, Mins, Secs: Extended);
var
	Y: Extended;
begin
	Degs := Int (X);
	Y := Frac (X) * 60;
	Mins := Int (Y);
	Secs := Frac (Y) * 60;
end;

function Distance (const X1, Y1, X2, Y2: Extended): Extended;
{ Rory Daulton suggested this more tolerant routine }
var
	X, Y: Extended;
begin
	X := Abs (X1 - X2);
	Y := Abs (Y1 - Y2);
	if X > Y then
		Result := X * Sqrt (1 + Sqr (Y / X))
	else if Y <> 0 then
		Result := Y * Sqrt (1 + Sqr (X / Y))
	else
		Result := 0
end;

function ExtMod (const X, Y: Extended): Extended;
var
	Z: Extended;
begin
	 Result := X / Y;
	 Z := Int (Result);
	 if Result < 0 then
		Z := Z - 1.0;
	 { Z now has Floor (X / Y) }
	 Result := X - Z * Y
end;

function ExtRem (const X, Y: Extended): Extended;
begin
	Result := X - Int (X / Y) * Y
end;

function MaxExt (const X, Y: Extended): Extended;
begin
	if X > Y then
		Result := X
	else
		Result := Y
end;

function MinExt (const X, Y: Extended): Extended;
begin
	if X < Y then
		Result := X
	else
		Result := Y
end;

function CompMOD (const X, Y: Comp): Comp;
begin
	Result := X - Y * Int (X / Y)
end;

function MaxEArray (const B: array of Extended): Extended;
var
	I: Integer;
begin
	Result := B [Low (B)];
	for I := Low (B) + 1 to High (B) do
		if B [I] > Result then
			Result := B [I];
end;

function MinEArray (const B: array of Extended): Extended;
var
	I: Integer;
begin
	Result := B [Low (B)];
	for I := Low (B) + 1 to High (B) do
	if B [I] < Result then
		Result := B [I];
end;

function MaxSArray (const B: array of Single): Single;
var
	I: Integer;
begin
	Result := B [Low (B)];
	for I := Low (B) + 1 to High (B) do
		if B [I] > Result then
			Result := B [I];
end;

function MinSArray (const B: array of Single): Single;
var
	I: Integer;
begin
	Result := B [Low (B)];
	for I := Low (B) + 1 to High (B) do
		if B [I] < Result then
			  Result := B [I];
end;

function MaxCArray (const B: array of Comp): Comp;
var
	I: Integer;
begin
	Result := B [Low (B)];
	for I := Low (B) + 1 to High (B) do
		if B [I] > Result then
			Result := B [I];
end;

function MinCArray (const B: array of Comp): Comp;
var
	I: Integer;
begin
	Result := B [Low (B)];
	for I := Low (B) + 1 to High (B) do
		if B [I] < Result then
			Result := B [I];
end;

function SumSArray (const B: array of Single): Single;
var
	I: Integer;
begin
	Result := B [Low (B)];
	for I := Low (B) + 1 to High (B) do
		Result := Result + B [I];
end;

function SumEArray (const B: array of Extended): Extended;
var
	I: Integer;
begin
	Result := B [Low (B)];
	for I := Low (B) + 1 to High (B) do
		Result := Result + B [I];
end;

function SumSqEArray (const B: array of Extended): Extended;
var
	I: Integer;
begin
	Result := Sqr (B [Low (B)]);
	for I := Low (B) + 1 to High (B) do
		Result := Result + Sqr (B [I]);
end;

function SumSqDiffEArray (const B: array of Extended; Diff: Extended): Extended;
var
	I: Integer;
begin
	Result := Sqr (B [Low (B)] - Diff);
	for I := Low (B) + 1 to High (B) do
		Result := Result + Sqr (B [I] - Diff);
end;

function SumXYEArray (const X, Y: array of Extended): Extended;
var
	I: Integer;
	M, N: Integer;
begin
	{$IFDEF UseMath}
	M := max (Low (X), Low (Y));
	N := min (High (X), High (Y));
	{$ELSE}
		M := Low (X);
		if Low (Y) > M then
			M := Low (Y);
		N := High (X);
		if High (Y) < N then
			N := High (Y);
	{$ENDIF}
	Result := X [M] * Y [M];
	for I := M + 1 to N do
		Result := Result + X [I] * Y [I];
end;

function SumCArray (const B: array of Comp): Comp;
var
	I: Integer;
begin
	Result := Low (B);
	for I := Low (B) + 1 to High (B) do
		Result := Result + B [I];
end;

function SameFloat (const X1, X2: Extended): Boolean;
begin
	Result := abs (X1 - X2) < ESBTolerance
end;

function FloatIsZero (const X: Extended): Boolean;
begin
	Result := abs (X) < ESBTolerance
end;

function FactorialX (A: Cardinal): Extended;
var
	I: Integer;
begin
	if A  > 1547 then
	begin
		Result := 0.0;
		Exit;
	end;
	Result := 1.0;
	for I := 2 to A do
		Result := Result * I;
end;

function PermutationX (N, R: Cardinal): Extended;
var
	I : Integer;
begin
	if (N = 0) or (R > N) or (N > 1547) then
	begin
		Result := 0.0;
		Exit;
	end;
	Result := 1.0;
	if (R = 0) then
		Exit;
	try
		for I := N downto N - R + 1 do
			Result := Result * I;
		Result := Int (Result + 0.5);
	except
		Result := -1.0
	end;
end;

function BinomialCoeff (N, R: Cardinal): Extended;
var
	I: Integer;
	K: LongWord;
begin
	if (N = 0) or (R > N) or (N > 1547) then
	begin
		Result := 0.0;
		Exit;
	end;
	Result := 1.0;
	if (R = 0) or (R = N) then
		Exit;
	if R > N div 2 then
		R := N - R;
	K := 2;
	try
		for I := N - R + 1 to N do
		begin
			Result := Result * I;
			if K <= R then
			begin
				Result := Result / K;
				Inc (K);
			end;
		end;
		Result := Int (Result + 0.5);
	except
		Result := -1.0
	end;
end;

function IsPositiveEArray (const X: array of Extended): Boolean;
var
	I: Integer;
begin
	Result := False;
	for I := 0 to High (X) do
		if X [I] <= ESBTolerance then
			Exit;
	Result := True;
end;

function GeometricMean (const X: array of Extended): Extended;
var
	I: Integer;
begin
	if High (X) < 0 then
		raise Exception.Create ('Array is Empty!')
	else if not IsPositiveEArray (X) then
		raise Exception.Create ('Array contains values <= 0!')
	else
	begin
		Result := 1;
		for I := 0 to High (X) do
			Result := Result * X [I];
		{$IFDEF UseMath}
		Result := Power (Result, 1 / (High (X) + 1));
		{$ELSE}
		Result := exp (Ln (Result) / (High (X) + 1));
		{$ENDIF}
	end;
end;

function HarmonicMean (const X: array of Extended): Extended;
var
	I: Integer;
begin
	if High (X) < 0 then
		raise Exception.Create ('Array is Empty!')
	else if not IsPositiveEArray (X) then
		raise Exception.Create ('Array contains values <= 0!')
	else
	begin
		Result := 0;
		for I := 0 to High (X) do
			Result := Result + 1 / X [I];
		Result := Result / (High (X) + 1);
		Result := 1 / Result;
	end;
end;

function ESBMean (const X: array of Extended): Extended;
begin
	Result := SumEArray (X) / (High (X) - Low (X) + 1)
end;

function SampleVariance (const X: array of Extended): Extended;
var
	I: Integer;
	SumSq: Extended;
	Mean: Extended;
begin
	Mean := ESBMean (X);
	SumSq := 0.0;
	for I := Low (X) to High (X) do
		SumSq := SumSq + Sqr (X [I] - Mean);
	Result := SumSq / (High (X) - Low (X))
end;

function PopulationVariance (const X: array of Extended): Extended;
var
	I: Integer;
	SumSq: Extended;
	Mean: Extended;
begin
	Mean := ESBMean (X);
	SumSq := 0.0;
	for I := Low (X) to High (X) do
		SumSq := SumSq + Sqr (X [I] - Mean);
	Result := SumSq / (High (X) - Low (X) + 1)
end;

procedure SampleVarianceAndMean (const X: array of Extended;
	var Variance, Mean: Extended);
var
	I: Integer;
	SumSq: Extended;
begin
	Mean := ESBMean (X);
	SumSq := 0.0;
	for I := Low (X) to High (X) do
		SumSq := SumSq + Sqr (X [I] - Mean);
	if High (X) > Low (X) then
		Variance := SumSq / (High (X) - Low (X))
	else
		Variance := 0;
end;

procedure PopulationVarianceAndMean (const X: array of Extended;
	var Variance, Mean: Extended);
var
	I: Integer;
	SumSq: Extended;
begin
	Mean := ESBMean (X);
	SumSq := 0.0;
	for I := Low (X) to High (X) do
		SumSq := SumSq + Sqr (X [I] - Mean);
	Variance := SumSq / (High (X) - Low (X) + 1)
end;

function GetMedian (const SortedX: array of Extended): Extended;
var
	N: Integer;
begin
	N := High (SortedX) + 1;
	if N <= 0 then
		raise Exception.Create ('Array is Empty!')
	else if N = 1 then
		Result := SortedX [0]
	else if Odd (N) then
		Result := SortedX [N div 2]
	else
		Result := (SortedX [N div 2 - 1] + SortedX [N div 2]) / 2;
end;

function GetMode (const SortedX: array of Extended; var Mode: Extended): Boolean;
var
	I, Freq, HiFreq: Integer;
	Matched: Boolean;
begin
	if High (SortedX) < 0 then
	begin
		raise Exception.Create ('Array is Empty!')
	end
	else if High (SortedX) = 0 then
	begin
		Mode := SortedX [0];
		Result := True;
	end
	else
	begin
		Mode := 0;
		Freq := 1;
		HiFreq := 0;
		Matched := False;
		for I := 1 to High (SortedX) do
		begin
			if SameFloat (SortedX [I - 1], SortedX [I]) then
				Inc (Freq)
			else
			begin
				if Freq <> 1 then
				begin
					if Freq = HiFreq then
						Matched := True
					else if Freq > HiFreq then
					begin
						Mode := SortedX [I - 1];
						HiFreq := Freq;
						Matched := False;
					end;
					Freq := 1;
				end;
			end;
		end;
		if HiFreq > 0 then
		begin
			if Freq = HiFreq then
				Matched := True
			else if Freq > HiFreq then
			begin
				Mode := SortedX [High (SortedX)];
				Matched := False;
			end;
		end
		else if Freq > 1 then
		begin
			HiFreq := Freq;
			Mode := SortedX [0];
			Matched := False;
		end;
		Result := (HiFreq > 0) and not Matched;
	end;
end;

procedure GetQuartiles (const SortedX: array of Extended; var Q1, Q3: Extended);
var
	N: Single;
	I: Integer;
begin
	if High (SortedX) < 0 then
		raise Exception.Create ('Array is Empty!')
	else if High (SortedX) = 0 then
	begin
		Q1 := SortedX [0];
		Q3 := SortedX [0];
	end
	else
	begin
		N := (High (SortedX) + 1) / 4 + 0.5;
		I := Trunc (N);
		N := Frac (N);
		if I - 1 < High (SortedX) then
			Q1 := SortedX [I - 1] + (SortedX [I] - SortedX [I - 1]) * N
		else
			Q1 := SortedX [I - 1];

		N := 3 * (High (SortedX) + 1) / 4 + 0.5;
		I := Trunc (N);
		N := Frac (N);
		if I - 1 < High (SortedX) then
			Q3 := SortedX [I - 1] + (SortedX [I] - SortedX [I - 1]) * N
		else
			Q3 := SortedX [I - 1];
	end;
end;

end.
