{ ====================================================================
  This program provides a pop up task bar similar to the
  one provided in Windows95.  The window lurks at the top
  of the screen.  It responds to mouse movements and pops
  into view if mouse movement is detected.

  To create a new button, right click on the form.
  Make sure you click the form and not an existing button.

  To change the properties of a button, right click on
  the button.

  To run a program, left click a button. }

unit Launchf;

interface

uses
  SysUtils, WinTypes, WinProcs, Messages, Classes, Graphics, Controls,
  Forms, Dialogs, StdCtrls, myDlgF, shellApi, Buttons, ExtCtrls;

{ ==================================================================== }

type
                               { A new class based on a TImage is
                                 described here.  TImage can display
                                 icons extracted from EXE files. }
  TMyImage = class(TImage)
    next : TMyImage;
    prev : TMyImage;           { The create procedure (correctly called
                                 a constructor) has been modified in
                                 this new class.  Described below. }
    constructor create(AOwner: TComponent); override;
                               { The destroy procedure (correctly
                                 called a destructor) has been modified
                                 in this new class.  Described below. }
    destructor  destroy; override;
                               { The mouseDown procedure has been
                                 modified in this new class.  Described
                                 below. }
    procedure mouseDown(Button: TMouseButton; Shift: TShiftState;
                        X, Y: Integer); override;
    procedure drawSelf;
  public                       { BitButtons have many properties.  The
                                 TMyImage class adds three more
                                 properties below. }
    caption   : String;
    exeName   : string;       { The name of the EXE file to run.      }
    exePath   : string;       { The working directory of the program. }
    glyphName : String;       { The button BMP bitmap file name.      }
    gotIcon   : Boolean;
    gotPic    : Boolean;
  end;

{ ====================================================================
  This is the standard stuff created by Delphi. }
  TForm1 = class(TForm)
    Timer1: TTimer;
    procedure FormMouseDown(Sender: TObject; Button: TMouseButton;
      Shift: TShiftState; X, Y: Integer);
    procedure FormCreate(Sender: TObject);
    procedure FormActivate(Sender: TObject);
    procedure FormClose(Sender: TObject; var Action: TCloseAction);
    procedure FormMouseMove(Sender: TObject; Shift: TShiftState; X,
      Y: Integer);
    procedure Timer1Timer(Sender: TObject);
    procedure FormPaint(Sender: TObject);
  private
    { Private declarations }
                              { The buttons are held in a linked list.
                                The head and tail pointers point to
                                the head and tail of this linked list. }
    head : TMyImage;
    tail : TMyImage;
                              { This procedure handles all the steps
                                needed to create a new button. }
    procedure newButton;
  public
    { Public declarations }
                              { The buttons are held in a linked list.
                                The curr pointer points to the current
                                item in the linked list.  This is the
                                button that has just beek clicked. }
    curr      : TMyImage;
                              { countDown is used to hide the form after
                                a time delay. }
    countDown : Integer;

    procedure saveButtons;    { save the buttons to a text file. }

    procedure openButtons;    { loads the buttons from a text file. }
    procedure pack;           { repositions buttons after a deletion. }
    procedure showForm;       { reposition and resize the form }
    procedure hideForm;       { reposition and resize the form }
  end;

{ ==================================================================== }

var
  Form1: TForm1;

implementation

{$R *.DFM}

{ ====================================================================
  This constructor creates a button, inserts it into a linked
  list and inserts the button into form1 to make it visible.

  Input           : None
  Process         : Create a new button.  Make it visible.
                    Insert it into a linked list.
  Test Data       : None
  Expected Output : New blank button appears on screen.
                    List linked correctly.
  Actual Output   : As above
  Action Needed   : None.                                           }

constructor TMyImage.create(AOwner: TComponent);
var scan : TMyImage;   { scan is used to scan the list to decide
                         where to position the new button. }
    max  : Integer;    { max keeps track of the position of the
                         buttons in the list.  The new button is
                         inserted at a position beyond max. }
begin
  { This runs the create procedure of the original BitButton. }
  inherited create(AOwner);

  { Set these to default values. }
  exeName   := '';
  exePath   := '';
  glyphName := '';
  gotIcon   := False;
  gotPic    := False;

  { Insert the new button into a linked list. }
  { If the list is empty do these steps.      }
  if form1.head = Nil then
  begin
    next := Nil;     { Label the ends of the list with Nil pointers.    }
    prev := Nil;     { Label the ends of the list with Nil pointers.    }

    form1.head := self;      { Make the head and tail pointers point    }
    form1.tail := self;      { to the new button.                       }
  end
  else   { If the list is not empty then add to the tail of the list.   }
  begin
    prev := form1.Tail;   { Join the prev pointer to the old list tail  }
    next := Nil;          { Label next with the end marker Nil          }

    form1.tail.next := self;    { Make old tail point to new  button    }
    form1.tail := self;         { Make the tail point to the new button }
  end;

  { This makes the button visible in the form. }
  form1.insertControl(self);

  { Scan will scan from the head, all the items in the linked list }
  scan := form1.head;

  { Initialise max }
  max := 0;

  { repeat until the nil end marker is found. }
  while scan <> Nil do
  begin
    { make max equal the biggest top value found in the list. }
    if scan.top > max then
    begin
      max := scan.top
    end;

    { scan is made to point to the next item in the list. }
    scan := scan.next
  end;

  { First button is placed 8 pixels down the form. }
  if max = 0 then
    top := 8
  else
    top := max + 40;
  { Other buttons are placed 40 pixels below the lowest other button. }

  { Set the left and width values of the new button. }
  left   := 8;
  width  := 36;
  height := 36
end;

{ ====================================================================
  The destructor removes a button from the form,  removes it from the
  linked list and frees the memory used to store itself.

  Input           : None
  Process         : Remove the button from the form.  Delete it from
                    the linked list.
  Test Data       : None
  Expected Output : Button removed from form.  Deletect from list.
                    Pointers linked correctly.
  Actual Output   : As above.
  Action Needed   : None. }

destructor TMyImage.destroy;
begin
  { Remove the button from the form. }
  form1.removeControl(self);

  if form1.head = Nil then                   { There are no buttons. }
  begin
                                             { Should never happen }
  end
  else if (next = Nil) AND (prev = Nil) then { There is only one button }
  begin
    { Set the head, tail and curr pointers to Nil. }
    form1.head := Nil;
    form1.curr := Nil;
    form1.tail := Nil;
  end
  else if (prev = Nil) then                  { Remove the head button }
  begin
    { Make the head pointer point to the next item in the list. }
    { Label the new list end with Nil                           }
    form1.head := form1.head.next;
    form1.head.prev := Nil;
  end
  else if (next = Nil) then                  { Remove the tail button }
  begin
    { Make the tail pointer point to the previous item in the list. }
    { Label the new list end with Nil                               }
    form1.tail := form1.tail.prev;
    form1.tail.next := Nil;
  end
  else                                       { remove a mid list button }
  begin
    { Make the pointers bypass the current button. }
    next.prev := prev;
    prev.next := next;
  end;

  { This frees the memory occupied by the button. }
  inherited destroy;

  { This re-positions the buttons after one has been deleted. }
  form1.pack
end;

{ ====================================================================
  This procedure responds to mouse presses in new ways not available
  in the original BitButton control.

  Input           : Mouse down event.
  Process         : Right click - set button properties.
                    Left click  - run the specified program.
  Test Data       : Use button properties known to be correct.
                    Button should work as expected.
                    Use button properties known to be wrong.
                    Button should give appropriate error messages.
  Expected Output : Should work as described in the process.
  Actual Output   : As expected
  Action Needed   : None. }

procedure TMyImage.mouseDown(Button: TMouseButton; Shift: TShiftState;
                              X, Y: Integer);

        { This procedure displays an error message if the button press
          failed to start the program for any reason.                 }
        procedure errorTest(aTest : THandle);
        var s : string;
        begin
          { Here are the possible text messages linked
            to the numerical error codes.                              }
          case aTest of
             0 : s := 'System was out of memory, executable file was corrupt, or relocations were invalid.';
             1 : s := 'Unknown Error';
             2 : s := 'File was not found.';
             3 : s := 'Path was not found.';
             4 : s := 'Unknown Error';
             5 : s := 'Attempt was made to dynamically link to a task, or there was a sharing or network-protection error.';
             6 : s := 'Library required separate data segments for each task.';
             7 : s := 'Unknown Error';
             8 : s := 'There was insufficient memory to start the application.';
             9 : s := 'Unknown Error';
            10 : s := 'Windows version was incorrect.';
            11 : s := 'Invalid executable file. Not Windows application or error in .EXE image.';
            12 : s := 'Application was designed for a different operating system.';
            13 : s := 'Application was designed for MS-DOS 4.0.';
            14 : s := 'Type of executable file was unknown.';
            15 : s := 'Attempt was made to load a real-mode application (developed for an earlier version of Windows).';
            16 : s := 'Attempted to load second instance of an exe file with multiple data segments that were non read-only.';
            17 : s := 'Unknown Error';
            18 : s := 'Unknown Error';
            19 : s := 'Attempt was made to load a compressed executable file. Decompresse it before loading.';
            20 : s := 'Dynamic-link library (DLL) file invalid. A DLL required to run this application was corrupt.';
            21 : s := 'Application requires Windows 32-bit extensions.';
            22 : s := 'Unknown Error';
            23 : s := 'Unknown Error';
            24 : s := 'Unknown Error';
            25 : s := 'Unknown Error';
            26 : s := 'Unknown Error';
            27 : s := 'Unknown Error';
            28 : s := 'Unknown Error';
            29 : s := 'Unknown Error';
            30 : s := 'Unknown Error';
            31 : s := 'There is no association for the specified file type or action';
            32 : s := 'Unknown Error';
          else
            s := ''  { This happens if there was no error. }
          end;

          { This displau=ys the error message if there is one. }
          if s <> '' then
          begin
            messageDlg(s, mtError, [mbOK], 0)
          end
        end;

var P : PChar;
begin
  { Handle the mouse down event in the original BitButton way. }
  inherited mouseDown(Button, Shift, X, Y);

  { Additional behaviours are described here. }
  form1.curr := self;

  { If it was a right mouse click do these two steps. }
  if shift = [ssRight] then
  begin
    form1.hideForm;
    { show the dialog box where the button properties are set. }
    myDlg.showModal
  end
  else if exeName <> '' then    { If there's an EXE file name, do this }
  begin
    form1.hideForm;
    { Allocate memory to store EXE file name as PChar (Null terminated) }
    P := StrAlloc(length(exeName) + 1);
    { Copy the string to the PChar }
    P := StrPCopy(P, exeName);
    { Attempt to run the EXE file.  Process the error code returned. }
    errorTest(ShellExecute(form1.handle, 'Open', P, Nil, Nil, SW_SHOWNORMAL));
    { Free the memory used to hold the PChar. }
    strDispose(p)
  end;
end;

{ ====================================================================
  drawSelf

  Input           : None
  Process         : Display a default bitmap or an icon extracted from
                    an EXE file if there is one.
  Test Data       : None
  Expected Output : Default bitmap or extracted icon displayed
  Actual Output   : As expected
  Action Needed   : None }

procedure TMyImage.drawSelf;
var ExeFileName : PChar;      { Null terminated file name }
    Icon        : HIcon;      { Icon handle }
begin
  if not gotIcon then         { get the icon if not already done }
  begin
    { Convert string to PChar }
    ExeFileName := stralloc(length(exename));
    ExeFileName := strPcopy(ExeFileName, exeName);

    { Extracts the first icon out of the named EXE file. }
    Icon := ExtractIcon(Form1.Handle, ExeFileName, 0);

    { Free the memory used for the Null terminated string. }
    strDispose(ExeFileName);

    { If there was an icon ... }
    if icon > 1 then
    begin
      picture.icon.handle := icon;
      gotIcon := true
    end
    else   { otherwise display the default bitmap. }
    begin
      if (not gotPic) AND (not gotIcon) then
      begin
        picture.loadFromFile(extractFilePath(application.exeName) + '\foo.bmp');
        gotPic := true
      end
    end;
  end
end;

{ ====================================================================
  newButton creates the new button.

  Input           : None
  Process         : Call constructor of TMyImage.  Additional features
                    could be added here later.
  Test Data       : None
  Expected Output : Button should be created.
  Actual Output   : As expected
  Action Needed   : None.  }

procedure TForm1.newButton;
begin
  { Create the new button. }
  TMyImage.Create(Form1)
end;

{ ====================================================================
  This procedure responds to a mouse click on form1

  Input           : Mouse click
  Process         : Right Click - call newButton
                    Left click  - do nothing
  Test Data       : None
  Expected Output : Right click should create a new button.
  Actual Output   : As expected
  Action Needed   : None. }

procedure TForm1.FormMouseDown(Sender: TObject; Button: TMouseButton;
  Shift: TShiftState; X, Y: Integer);
begin
  { if it was a right click then make a new button }
  if shift = [ssRight] then
  begin
    newButton;
    height := height + 40  { Increase the form height or the new button }
  end                      { won't be visible. }
end;

{ ====================================================================
  The formCreate procedure runs once when the program starts.

  Input           : None
  Process         : Initialise pointers and countdown counter.
                    Hide the form lurking at the top of the screen.
  Test Data       : None
  Expected Output : As per process
  Actual Output   : As expected
  Action Needed   : None. }

procedure TForm1.FormCreate(Sender: TObject);
begin
  { Set these pointers to Nil because the list does not exist yet. }
  head := Nil;
  tail := Nil;
  curr := Nil;

  { No count down needed when the program starts. }
  countDown := 0
end;

{ ====================================================================
  A procedure to save the buttons to a text file.

  Input           : Data taken from the linked list of buttons.
  Process         : Save data as text in a text file named foo.dat.
                    Foo.dat should save to the same directory as
                    launch.exe.
  Test Data       : Linked list button data.
  Expected Output : Text file correctly describing the buttons.
  Actual Output   : As expected.
  Action Needed   : None. }

procedure TForm1.saveButtons;
var f    : textFile;        { f refers to the file                    }
    scan : TMyImage;        { scan used to visit every button in list }
begin
  { associate f with a file name.
    extractFilePath is used to get the directory of the program exe file.
    The data file (foo.dat) name is tagged onto the directory.
    this ensures thet the program looks in the right directory.  }
  assignFile(f, extractFilePath(application.exeName) + '\foo.dat');
  reWrite(f);    { Open the file for writing.  Old copy is destroyed. }

  { Scan strating from the head. }
  scan := head;
  { Visit every item in the list. }
  while scan <> Nil do
  begin
    { Write lines into the text file. }
    { The button position and size.   }
    Writeln(f, scan.top);
    Writeln(f, scan.left);
    Writeln(f, scan.width);
    Writeln(f, scan.height);
    Writeln(f, scan.caption);      { The button caption.              }
    Writeln(f, scan.exeName);      { The exe file name.               }
    Writeln(f, scan.exePath);      { The working directory.           }
    { Make scan point to next list item or nil if at end of the list. }
    scan := scan.next
  end;
  { Close text file.  It is important to free this Windows resource.  }
  closeFile(f);
end;

{ ====================================================================
  A procedure to create buttons and load data from the saved file to
  position the buttons and set all the other button properties.

  Input           : Text data taken from the file foo.dat
  Process         : read foo.dat.
                    create buttons
                    butons should appear as they were before saving.
  Test Data       : Text from foo.dat
  Expected Output : Buttons correctly positioned, labeled and marked
                    with a bitmap image.
  Actual Output   : As expected
  Action Needed   : None. }

procedure TForm1.openButtons;
var f    : textFile;         { f refers to the text file               }
    pNew : TMyImage;         { pNew will point to newly created button }
    s    : String;           { s holds a text string read from file    }
    n    : Integer;          { n holds an integer read from the file   }
begin
  { Only do this procedure if the data file exists.                    }
  if fileExists(extractFilePath(application.exeName) + '\foo.dat') then
  begin
    { extractFilePath gives directory of the exe file of this program  }
    { foo.dat will be loaded from the program file directory           }
    assignFile(f, extractFilePath(application.exeName) + '\foo.dat');
    reSet(f);                  { Open file for reading. Get this right }

    while not eof(f) do        { end of file }
    begin
      { Create the new button and make pNew point to it. }
      pNew := TMyImage.Create(Form1);

      { Read a number from the file. }
      Readln(f, n);
      { This number is the value of pNew.top }
      pNew.top := n;

      { Read left, width and height as above. }
      Readln(f, n);
      pNew.left := n;

      Readln(f, n);
      pNew.width := n;

      Readln(f, n);
      pNew.height := n;

      { read s from the file. }
      Readln(f, s);
      { s is the caption of pNew }
      pNew.caption := s;

      { read the exeName, exePath, and glyph name as above. }
      Readln(f, s);
      pNew.exeName := s;

      Readln(f, s);
      pNew.exePath := s
    end;

    closeFile(f);    { Close file. Remember to free windows resource.  }
  end
end;

{ ====================================================================
  When the for activates, open and load the buttons file.

  Input           : None
  Process         : call openButtons to load the data file foo.dat
  Test Data       : none
  Expected Output : Buttons should be correctly loaded
  Actual Output   : As expected
  Action Needed   : None. }

procedure TForm1.FormActivate(Sender: TObject);
begin
  openButtons;    { Load the data file. }
  showForm        { Show the form.      }
end;

{ ====================================================================
  Reposition the buttons when one has been deleted.

  Input           : Position data from the button linked list.
  Process         : Position the first button 8 pixels from the form top.
                    Position each subsequent button 40 pixels below
                    the prevoius button.
  Test Data       : None
  Expected Output : As described in the process.
  Actual Output   : As expected.
  Action Needed   : None. }

procedure TForm1.pack;
var scan : TMyImage;    { scan is used to visit every button in list }
begin
  scan := head;          { scan from the head }

  if scan <> Nil then
  begin
    scan.top := 8        { make the top value of the first button 8 }
  end;

  while scan <> Nil do   { scan until a Nil end marker pointer is found }
  begin
     scan := scan.next;  { scan now points at the next button or Nil    }

     if scan <> Nil then { if scan points at a button then position it  }
     begin               { 40 pixels below the previous button.         }
       scan.top := scan.prev.top + 40
     end
  end
end;

{ ====================================================================
  When the form is closed then save all the buttons.  Free them too.

  Input           : None
  Process         : Scan the linked list freeing all the buttons.
  Test Data       : None
  Expected Output : All buttons should be freed.
  Actual Output   : As expected
  Action Needed   : None. }

procedure TForm1.FormClose(Sender: TObject; var Action: TCloseAction);
Var scan : TMyImage;    { scan points at each button in turn.         }
    pDel : TMyImage;    { pDel points to the one to be deleted.       }
begin
  saveButtons;           { save the buttons                            }

  scan := head;          { scan from the head                          }

  while scan <> Nil do   { repeat until the Nil end marker is found.   }
  begin
    pDel := scan;        { pDel points to the scan item.               }
    scan := scan.next;   { scan now points to the next item.           }
    pDel.free            { pDel is freed.  This restores the memory    }
  end                    { allocated earlier.                          }
end;

{ ====================================================================
  If a form1 mouse move is detected, pop up the form to make it
  visible and set the count down timer to 5.

  Input           : Mouse movement over the form.
  Process         : Reposition the form to make it visible.
  Test Data       : None
  Expected Output : Repsoitioned form.
  Actual Output   : As expected.
  Action Needed   : None. }

procedure TForm1.FormMouseMove(Sender: TObject; Shift: TShiftState; X,
  Y: Integer);
begin
  showForm;
end;                 { Form is hidden when the countdown hits zero. }

{ ====================================================================
  The timer decrements the count down value to zero.
  When it reaches zero, the form hides itself.

  Input           : Counter value.
  Process         : If counter is greater than zero then decrement it.
                    If counter reaches zero then reposition the form.
                    If counter equals zero then do nothing.
  Test Data       : None
  Expected Output : Form hides itself when counter reaches zero.
  Actual Output   : As expected.
  Action Needed   : None. }

procedure TForm1.Timer1Timer(Sender: TObject);
begin
  if countDown > 0 then         { Count down if not zero already.      }
  begin
    countDown := countDown - 1; { Decrement the counter.               }

    if countDown = 0 then       { If the counter is zero, reposition   }
    begin
      hideForm
    end
  end
end;

{ ==================================================================== }
{ if the form needs re-painting, it also processes this code to
  draw every button.  This is done by scanning the linked list
  and using drawSelf on each item. }
procedure TForm1.FormPaint(Sender: TObject);
Var scan : TMyImage;
begin
  scan := head;

  while scan <> Nil do
  begin
    scan.drawSelf;

    scan := scan.next
  end;
end;

{ ==================================================================== }
{ This sizes and positions the form to make it visible. }
procedure TForm1.showForm;
begin
  { The height of the form is 40 pixels more than the top of
    the last button OR the height is 48.  This gives space to
    insert the first button. }
  if tail <> Nil then
  begin
    height := tail.top + 40
  end
  else
  begin
    height := 48
  end;

  { Position the form at the top of the screen }
  top := 0;
  width := 48;
  { Centre the form }
  left  := screen.width div 2 - width div 2;
  countDown := 5     { Set form hide count down to 5. }
end;

{ ==================================================================== }
{ This hides the form. }
procedure TForm1.hideForm;
begin
  { Position the form so only one pixel height of it is showing. }
  top := -form1.height + 1
end;

end.

{ ==================================================================== }

