{-----------------------------------------------------------------------------}
{ A component to read version info resources.  It is intended for Delphi 3,   }
{ but should work with any file that contains a properly formatted resource.  }
{ Copyright 1996, Brad Stowers.  All Rights Reserved.                         }
{ This component can be freely used and distributed in commercial and private }
{ environments, provied this notice is not modified in any way and there is   }
{ no charge for it other than nomial handling fees.  Contact me directly for  }
{ modifications to this agreement.                                            }
{-----------------------------------------------------------------------------}
{ Feel free to contact me if you have any questions, comments or suggestions  }
{ at bstowers@pobox.com or 72733,3374 on CompuServe.                          }
{ The lateset version will always be available on the web at:                 }
{   http://www.pobox.com/~bstowers/delphi/                                    }
{-----------------------------------------------------------------------------}
{ Date last modified:  May 21, 1997                                           }
{-----------------------------------------------------------------------------}


{ ----------------------------------------------------------------------------}
{ TVersionInfoResource v0.90 Beta                                             }
{ ----------------------------------------------------------------------------}
{ Description:                                                                }
{   A component to read version info resources.  It is intended for Delphi 3, }
{   but should work with any file that contains a properly formatted resource.}
{ Notes:                                                                      }
{   * I have not tested this on anything but Delphi 3 generated EXEs with     }
{     proper version info resources.                                          }
{ ----------------------------------------------------------------------------}
{ Revision History:                                                           }
{ 0.90:  + Initial public beta release.                                       }
{ ----------------------------------------------------------------------------}

unit VersInfo;

{$IFNDEF WIN32}
  ERROR!  This unit only available on Win32!
{$ENDIF}

interface

uses
  Windows, Messages, SysUtils, Classes, Graphics, Controls, Forms, Dialogs;

const
  IDX_FILEDESCRIPTION       = 0;
  IDX_FILEVERSION           = 1;
  IDX_INTERNALNAME          = 2;
  IDX_LEGALCOPYRIGHT        = 3;
  IDX_LEGALTRADEMARKS       = 4;
  IDX_ORIGINALFILENAME      = 5;
  IDX_PRODUCTNAME           = 6;
  IDX_PRODUCTVERSION        = 7;
  IDX_COMMENTS              = 8;

type
  TVersionInfoResource = class(TComponent)
  private
    FVersionInfo: PChar;
    FVersionInfoSize: DWORD;
    FFilename: string;
    FTranslationIDs: TStringList;
    FTranslationIDIndex: integer;
    FFixedInfo: PVSFixedFileInfo;
  protected
    procedure SetFilename(Val: string);
    procedure SetTranslationIDIndex(Val: integer);
    function GetTranslationIDs: TStrings;

    procedure ReadVersionInfoData;
    function GetVersionInfoString(Index: integer): string;
    function GetResourceStr(Index: string): string;

    procedure Loaded; override;
  public
    constructor Create(AOwner: TComponent); override;
    destructor Destroy; override;

    property TranslationIDIndex: integer
      read FTranslationIDIndex
      write SetTranslationIDIndex;
    property TranslationIDs: TStrings
      read GetTranslationIDs;
    property FixedInfo: PVSFixedFileInfo
      read FFixedInfo;
  published
    property Filename: string
      read FFilename
      write SetFilename;
    property CompanyName: string index IDX_FILEDESCRIPTION
      read GetVersionInfoString;
    property FileVersion: string index IDX_FILEVERSION
      read GetVersionInfoString;
    property InternalName: string index IDX_INTERNALNAME
      read GetVersionInfoString;
    property LegalCopyright: string index IDX_LEGALCOPYRIGHT
      read GetVersionInfoString;
    property LegalTrademarks: string index IDX_LEGALTRADEMARKS
      read GetVersionInfoString;
    property OriginalFilename: string index IDX_ORIGINALFILENAME
      read GetVersionInfoString;
    property ProductName: string index IDX_PRODUCTNAME
      read GetVersionInfoString;
    property ProductVersion: string index IDX_PRODUCTVERSION
      read GetVersionInfoString;
    property Comments: string index IDX_COMMENTS
      read GetVersionInfoString;

    property UserResource[Index: string]: string
      read GetResourceStr;
  end;

procedure Register;

implementation

const
  PREDEF_RESOURCES: array[IDX_FILEDESCRIPTION..IDX_COMMENTS] of string = (
     'FileDescription', 'FileVersion', 'InternalName', 'LegalCopyright',
     'LegalTrademarks', 'OriginalFilename', 'ProductName', 'ProductVersion', 'Comments'
    );


procedure Register;
begin
  RegisterComponents('Davids', [TVersionInfoResource]);
end;


constructor TVersionInfoResource.Create(AOwner: TComponent);
begin
  inherited Create(AOwner);

  FVersionInfo := NIL;
  FVersionInfoSize := 0;
  FFilename := '';
  FTranslationIDIndex := 0;
  FTranslationIDs := TStringList.Create;
  FFixedInfo := NIL;
end;


destructor TVersionInfoResource.Destroy;
begin
  FTranslationIDs.Free;
  if FVersionInfo <> NIL then
    FreeMem(FVersionInfo, FVersionInfoSize);

  inherited Destroy;
end;


procedure TVersionInfoResource.Loaded;
begin
  inherited Loaded;

  ReadVersionInfoData;
end;


procedure TVersionInfoResource.SetFilename(Val: string);
begin
  FFilename := Val;
  ReadVersionInfoData;
end;

procedure TVersionInfoResource.ReadVersionInfoData;
const
  TRANSLATION_INFO = '\VarFileInfo\Translation';
type
  TTranslationPair = packed record
    Lang,
    CharSet: word;
  end;
  PTranslationIDList = ^TTranslationIDList;
  TTranslationIDList = array[0..MAXINT div SizeOf(TTranslationPair)-1] of TTranslationPair;
var
  Dummy: DWORD;
  TempFilename: string;
  IDs: PTranslationIDList;
  IDsLen: UINT;
  IDCount: integer;
begin
  FTranslationIDs.Clear;
  FFixedInfo := NIL;
  if FVersionInfo <> NIL then
    FreeMem(FVersionInfo, FVersionInfoSize);

  if FFilename = '' then
    TempFileName := Application.EXEName
  else
    TempFileName := FFilename;

  FVersionInfoSize := GetFileVersionInfoSize(PChar(TempFileName), Dummy);
  if FVersionInfoSize = 0 then
    FVersionInfo := NIL
  else begin
    GetMem(FVersionInfo, FVersionInfoSize);
    GetFileVersionInfo(PChar(TempFileName), Dummy, FVersionInfoSize, FVersionInfo);

    VerQueryValue(FVersionInfo, '\', pointer(FFixedInfo), Dummy);
    if VerQueryValue(FVersionInfo, TRANSLATION_INFO, Pointer(IDs), IDsLen) then
    begin
      IDCount := IDsLen div SizeOf(TTranslationPair);
      for Dummy := 0 to IDCount-1 do
        FTranslationIDs.Add(Format('%.4x%.4x', [IDs[Dummy].Lang, IDs[Dummy].CharSet]));
    end;
  end;

  if FTranslationIDIndex >= FTranslationIDs.Count then
    FTranslationIDIndex := 0;
end;


function TVersionInfoResource.GetVersionInfoString(Index: integer): string;
begin
  if (Index >= Low(PREDEF_RESOURCES)) and (Index <= High(PREDEF_RESOURCES)) then
    Result := GetResourceStr(PREDEF_RESOURCES[Index])
  else
    Result := ''
end;


function TVersionInfoResource.GetResourceStr(Index: string): string;
var
  ResStr: PChar;
  StrLen: UINT;
  SubBlock: string;
  LangCharSet: string;
begin
  if FTranslationIDIndex < FTranslationIDs.Count then
    LangCharSet := FTranslationIDs[FTranslationIDIndex]
  else
    LangCharSet := '';
  SubBlock := '\StringFileInfo\' + LangCharSet + '\' + Index;
  if (FVersionInfo <> NIL) and
     VerQueryValue(FVersionInfo, PChar(SubBlock), Pointer(ResStr), StrLen)
  then
    Result := string(ResStr)
  else
    Result := '';
end;

procedure TVersionInfoResource.SetTranslationIDIndex(Val: integer);
begin
  if (Val > 0) and (Val < FTranslationIDs.Count) then
    FTranslationIDIndex := Val;
end;

function TVersionInfoResource.GetTranslationIDs: TStrings;
begin
  Result := FTranslationIDs;
end;


end.
