unit UnZipLib;

{
Revision history:

Code here derived from Info-ZIP group's portable UnZip zipfile-extraction
program by David J Taylor, Edinburgh

V1.0.0  1997 Feb 09  First version - only handles "stored" files, no compression
V1.1.0  1997 May 14  Copied unchanged for Delphi 3.0
V1.2.0  1997 Jun 14  Add stream functionality with TZipStream class
}

interface

uses
  Windows, Classes, SysUtils;

// True sizes of the various headers, as defined by PKWARE -- so it is not
// likely that these will ever change.  But if they do, make sure both these
// defines AND the typedefs below get updated accordingly.
const
  LREC_SIZE  = 26;   // lengths of local file headers, central
  CREC_SIZE  = 42;   //  directory headers, and the end-of-
  ECREC_SIZE = 18;   //  central-dir record, respectively

type
  PLocalFileHdr = ^TLocalFileHdr;
  TLocalFileHdr = packed record              // local_file_header
    version_needed_to_extract: array [0..1] of Byte;
    general_purpose_bit_flag: Word;
    compression_method: Word;
    last_mod_file_time: Word;
    last_mod_file_date: Word;
    crc32: Longint;
    csize: Longint;
    ucsize: Longint;
    filename_length: Word;
    extra_field_length: Word;
  end;

// this stuff isn't used yet, but might be for later versions

type
  cdir_file_hdr = packed record   // central_directory_file_header
    version_made_by: array [0..1] of Byte;
    version_needed_to_extract: array [0..1] of Byte;
    general_purpose_bit_flag: Word;
    compression_method: Word;
    last_mod_file_time: Word;
    last_mod_file_date: Word;
    crc32: Longint;
    csize: Longint;
    ucsize: Longint;
    filename_length: Word;
    extra_field_length: Word;
    file_comment_length: Word;
    disk_number_start: Word;
    internal_file_attributes: Word;
    external_file_attributes: Longint;
    relative_offset_local_header: Longint;
  end;

type
  ec_byte_rec = packed record
    signature: Longint;   // space-holder only
    number_this_disk: Word;
    num_entries_central_dir_ths_disk: Word;
    total_entries_central_dir: Word;
    size_central_directory: Longint;
    offset_start_central_directory: Longint;
    zipfile_comment_length: Word;
  end;

type
  ecdir_rec = packed record   // end_central_dir_record
    number_this_disk: Word;
    num_disk_start_cdir: Word;
    num_entries_centrl_dir_ths_disk: Word;
    total_entries_central_dir: Word;
    size_central_directory: Longint;
    offset_start_central_directory: Longint;
    zipfile_comment_length: Word;
  end;

const
  CENTRAL_HDR_SIG = $0201;  // the infamous "PK" signature bytes,
  LOCAL_HDR_SIG = $0403;    // sans "PK" (so unzip executable not
  END_CENTRAL_SIG = $0605;  // mistaken for zipfile itself)

const
  max_file_path = 127;  // should pick up a proper definition of this...

// combined type for reading file header records and searching
// in a FindFirst, FindNext sequence
type
  TPKSearchRecord = packed record
    pk_check_field: array [0..1] of char;
    pk_signature: Word;
    local: TLocalFileHdr;
    local_file_name: array [0..max_file_path] of char;
    header_base: Pointer;   // point to header in memory
  end;

type
  TZipFile = class
  private
    FZipFilename: String;
    file_handle: THandle;
    file_mapping: THandle;
    FFilebase: Pointer;
    FFindBuffer: ^TPKSearchRecord;
    FFindPointer: Pointer;
    FFileList: TStringList;
    procedure SetZipFilename (name: string);
    function decode_time (const time: Word): string;
    function decode_date (const date: Word): string;
  public
    constructor Create;
    destructor Destroy;  override;
    function PKSrchRecToStr (const pk: TPKSearchRecord): string;
    function PKFindFirst (var buffer: TPKSearchRecord): boolean;
    function PKFindNext: boolean;
    function UnzipFileToBuffer (const name: string;
                                var buffer;  buffer_size: integer): integer;
    function GetFileSize (const filename: string): integer;
  published
    property ZipFilename: String read FZipFilename write SetZipFilename;
  end;

type
  EErrorInZip = class(Exception);

type
// This stuff only slightly modified from TCustomMemoryStream,
// except that all the write functionality has gone
{ TCustomZipStream }

  TCustomZipStream = class(TStream)
  private
    FMemory: Pointer;
    FSize, FPosition: Longint;
  protected
    procedure SetPointer(Ptr: Pointer; Size: Longint);
  public
    function Read(var Buffer; Count: Longint): Longint; override;
    function Seek(Offset: Longint; Origin: Word): Longint; override;
    property Memory: Pointer read FMemory;
  end;

{ TZipStream }

  TZipStream = class(TCustomZipStream)
  private
    FCapacity: Longint;
    FZipFile: TZipFile;
    procedure SetCapacity(NewCapacity: Longint);
  protected
    function Realloc(var NewCapacity: Longint): Pointer; virtual;
    property Capacity: Longint read FCapacity write SetCapacity;
  public
    constructor CreateFromArchive(const ArchiveName: string);
    destructor Destroy; override;
    procedure Clear;
    procedure LoadFileFromArchive(const FileName: string);
    procedure SetSize(NewSize: Longint); override;
  end;


implementation

{ TCustomZipStream }

procedure TCustomZipStream.SetPointer(Ptr: Pointer; Size: Longint);
begin
  FMemory := Ptr;
  FSize := Size;
end;

function TCustomZipStream.Read(var Buffer; Count: Longint): Longint;
begin
  if (FPosition >= 0) and (Count >= 0) then
    begin
    Result := FSize - FPosition;
    if Result > 0 then
      begin
      if Result > Count then Result := Count;
      Move(Pointer(Longint(FMemory) + FPosition)^, Buffer, Result);
      Inc(FPosition, Result);
      Exit;
      end;
    end;
  Result := 0;
end;

function TCustomZipStream.Seek(Offset: Longint; Origin: Word): Longint;
begin
  case Origin of
    0: FPosition := Offset;
    1: Inc(FPosition, Offset);
    2: FPosition := FSize + Offset;
  end;
  Result := FPosition;
end;


{ TZipStream }

const
  MemoryDelta = $2000; { Must be a power of 2 }

constructor TZipStream.CreateFromArchive(const ArchiveName: string);
begin
  FZipFile := TZipFile.Create;
  FZipFile.ZipFileName := ArchiveName;
end;

destructor TZipStream.Destroy;
begin
  Clear;
  FZipFile.Free;
  inherited Destroy;
end;

procedure TZipStream.Clear;
begin
  SetCapacity(0);
  FSize := 0;
  FPosition := 0;
end;

procedure TZipStream.LoadFileFromArchive(const FileName: string);
var
  count: Longint;
begin
  count := FZipFile.GetFileSize (FileName);
  if count = -1 then
    raise EErrorInZip.CreateFmt ('File %s not found in archive %s',
                                 [FileName, FZipFile.ZipFileName]);
  SetSize (count);
  FZipFile.UnzipFileToBuffer (FileName, FMemory^, count);
end;

procedure TZipStream.SetCapacity(NewCapacity: Longint);
begin
  SetPointer(Realloc(NewCapacity), FSize);
  FCapacity := NewCapacity;
end;

procedure TZipStream.SetSize(NewSize: Longint);
begin
  Clear;
  if FPosition > NewSize then Seek(0, soFromEnd);
  SetCapacity(NewSize);
  FSize := NewSize;
end;

function TZipStream.Realloc(var NewCapacity: Longint): Pointer;
begin
  if NewCapacity > 0 then
    NewCapacity := (NewCapacity + (MemoryDelta - 1)) and not (MemoryDelta - 1);
  Result := Memory;
  if NewCapacity <> FCapacity then
    begin
    if NewCapacity = 0
    then
      begin
      GlobalFreePtr(Memory);
      Result := nil;
      end
    else
      begin
      if Capacity = 0
        then Result := GlobalAllocPtr(HeapAllocFlags, NewCapacity)
        else Result := GlobalReallocPtr(Memory, NewCapacity, HeapAllocFlags);
      if Result = nil then
        raise EErrorInZip.Create('Can''t allocate memory for ZipStream');
      end;
    end;
end;


constructor TZipFile.Create;
begin
  Inherited;
  file_handle := INVALID_HANDLE_VALUE;
  FFilebase := nil;
  FFileList := TStringList.Create;
  with FFileList do
    begin
    Sorted := True;
    Duplicates := dupIgnore;
    end;
end;

destructor TZipFile.Destroy;
begin
  if file_handle <> INVALID_HANDLE_VALUE then
    begin
    UnmapViewOfFile (FFilebase);
    CloseHandle (file_mapping);
    CloseHandle (file_handle);
    end;
  FFileList.Free;
  Inherited;
end;


procedure TZipFile.SetZipFilename (name: string);
var
  pk_buf: TPKSearchRecord;
begin
  FZipFilename := name;

  if file_handle <> INVALID_HANDLE_VALUE then
    begin
    UnmapViewOfFile (FFilebase);
    CloseHandle (file_mapping);
    CloseHandle (file_handle);
    FFileList.Clear;
    end;

  file_handle := CreateFile (PChar (FZipFilename),
                              GENERIC_READ, FILE_SHARE_READ, nil,
                              OPEN_EXISTING, FILE_ATTRIBUTE_NORMAL, 0);

  if file_handle = INVALID_HANDLE_VALUE then
    raise EErrorInZip.Create ('Couldn''t open: ' + FZipFilename);

  file_mapping := CreateFileMapping (file_handle, nil, PAGE_READONLY, 0, 0, nil);
  if file_mapping = 0 then
    raise EErrorInZip.Create ('CreateFileMapping failed');

  FFilebase := MapViewOfFile (file_mapping, FILE_MAP_READ, 0, 0, 0);
  if FFilebase = nil then
    raise EErrorInZip.Create ('MapViewOfFile failed');

  if PKFindFirst (pk_buf) then
    repeat
      with pk_buf do
        FFileList.AddObject (LowerCase (local_file_name), header_base);
    until not PKFindNext;
end;

function TZipFile.PKFindFirst (var buffer: TPKSearchRecord): boolean;
begin
  FFindBuffer := @buffer;
  FFindPointer := FFileBase;
  Result := PKFindNext;
end;

function TZipFile.PKFindNext: boolean;
var
  chars_to_copy: integer;
begin
  Result := False;
  try
    Move (FFindPointer^, FFindBuffer^, 2);
    with FFindBuffer^ do
      if pk_check_field = 'PK' then
        begin
        Inc (Longint (FFindPointer), 2);        // point past the character field
        Move (FFindPointer^, pk_signature, 2);  // get the signature
        Inc (Longint (FFindPointer), 2);        // point past the signature field
        Result := False;
        if pk_signature = local_hdr_sig then
          begin
          Move (FFindPointer^, local, SizeOf (TLocalFileHdr));
          header_base := FFindPointer;
          with local do
            begin
            Inc (Longint (FFindPointer), SizeOf (TLocalFileHdr));
            chars_to_copy := filename_length + 1;
            if chars_to_copy > max_file_path then
              chars_to_copy := max_file_path;
            FillChar (local_file_name, chars_to_copy, 0);
            Move (FFindPointer^, local_file_name, chars_to_copy - 1);
            Inc (Longint (FFindPointer), filename_length + extra_field_length + csize);
            Result := True;
            end;
          end;
        end;
  except
  end;
end;


function TZipFile.GetFileSize (const filename: string): integer;
var
  where: integer;
begin
  Result := -1;
  if FFileList.Find (LowerCase (filename), where)
    then Result := PLocalFileHdr (FFileList.Objects [where])^.ucsize;
end;


// Reads a file from a Zip library into a user's buffer, returns the number
// of bytes actually read (or raises exceptions for the problems)

function TZipFile.UnzipFileToBuffer (const name: string;
                                     var buffer;  buffer_size: integer): integer;
var
  p: pointer;
  ph: PLocalFileHdr;
  where: integer;
begin
  if IsBadWritePtr (@buffer, buffer_size) then
    raise EErrorInZip.Create ('Incomplete write access to buffer (in UnzipFileToBuffer)');

  if not FFileList.Find (LowerCase (name), where)
  then
    raise EErrorInZip.Create ('File not found (in UnzipFileToBuffer)')
  else
    begin
    p := FFileList.Objects [where];
    ph := p;
    with ph^ do
      begin
      if buffer_size < ucsize then
        raise EErrorInZip.Create ('Buffer size too small (in UnzipFileToBuffer)');

      if compression_method <> 0 then
        raise EErrorInZip.Create ('Compression method not handled (in UnzipFileToBuffer)');

      // "stored" compression method - just copy the bytes
      Inc (Longint (p), SizeOf (TLocalFileHdr) + filename_length + extra_field_length);
      Move (p^, buffer, ucsize);
      Result := ucsize;
      end;
    end;
end;


function TZipFile.decode_date (const date: word): string;
const
  months: array [1..12] of string =
       ('Jan', 'Feb', 'Mar', 'Apr', 'May', 'Jun',
        'Jul', 'Aug', 'Sep', 'Oct', 'Nov', 'Dec');
var
  year, month, day: word;
begin
  year := ((date shr 9) and $7F) + 1980;
  month := (date shr 5) and $0F;
  day := date and $1F;
  Result := Format ('%4.4d-%3s-%2.2d', [year, months [month], day]);
end;

function TZipFile.decode_time (const time: word): string;
var
  hh, mm, ss: word;
begin
  hh := (time shr 11) and $1F;
  mm := (time shr 5) and $3F;
  ss := 2 * (time and $1F);
  Result := Format ('%2.2d:%2.2d:%2.2d', [hh, mm, ss]);
end;

function TZipFile.PKSrchRecToStr (const pk: TPKSearchRecord): string;
const
  methods: array [0..8] of string = (
    'Stored ', 'Shrunk ', 'Reduce1', 'Reduce2', 'Reduce3', 'Reduce4',
    'Implode', 'Token  ', 'Deflatn');
var
  s: string;
  ratio: double;
begin
  s := '';
  with pk.local do
    begin
    s := s + Format ('%9d ', [ucsize]);
    s := s + methods [compression_method] + ' ';
    s := s + Format ('%9d ', [csize]);
    if ucsize = 0
      then ratio := 0.0
      else ratio := 100.0 * (1.0 - csize / ucsize);
    s := s + Format ('%2.0f%% ', [ratio]);
    s := s + decode_date (last_mod_file_date) + ' ';
    s := s + decode_time (last_mod_file_time) + ' ';
    s := s + IntToHex (crc32, 8) + ' ';
    s := s + Format ('%*s', [filename_length, pk.local_file_name]);
    end;
  Result := s;
end;

end.


