unit TreeScan;

{
Description:

Component to help with traversing folder structures
Will notify the owner when file or folder is found.

Note that files are scanned before sub-folders, and that
both file and folder scans are in alphabetical order

Revision history:

V1.0.0  1996 Dec 15  First version
V1.0.2  1997 Jan 12  Add file mask (e.g. *.exe)
V1.0.4  1997 Mar 28  Add event when leaving folder
                     Add BeginUpdate/EndUpdate around list updates
V1.0.6  1997 Apr 26  Add report of found object size (for files)
                     Add Recursive flag - normally True
                     (suggestions of Sean Mathews)
V1.1.0  1997 May 12  Version for Delphi 3.0
V1.2.0  1997 Sep 14  Version for Delphi 3.01
                     Add "hidden" properties - these will make the routine find
                     Hidden files or folders in addition to the normal ones
                     (suggestion from Paolo Faccini)
V1.2.2  1997 Oct 22  Add OnSearchDone event (suggestions of Ed Butler)
                     Allow for coding error in Borland's RTL in FindClose
V1.3.0  1998 Mar 15  Version for Delphi 3.02
                     Add CommaText for input file spec, allows multiple searches
                     The simple FileMask property overrides FileMaskCommaText
}

interface

uses
  SysUtils, Classes, Forms;              // seems to be the minimum we need

type
  TTreeScanner = class(TComponent)
  private
    { Private declarations }
    FOnFileFound: TNotifyEvent;          // where to notify for file found
    FOnDirectoryFound: TNotifyEvent;     // where to notify for start of folder
    FOnDirectoryDone: TNotifyEvent;      // where to notify for end of folder
    FOnSearchDone: TNotifyEvent;         // where to notify for end of search
    FFoundObject: string;                // full file or folder name
    FFoundObjectSize: integer;           // size of file found
    FInitialDirectory: string;           // starting point in the tree
    FFileMask: string;                   // files to search for e.g. *.EXE
    FFileMaskList: TStringList;               // files to search for e.g. *.EXE, *.DLL
    FSearchInProgress: boolean;          // true whilst searching
    FRecursive: boolean;
    FFindHiddenFiles: boolean;
    FFindHiddenFolders: boolean;
  protected
    { Protected declarations }
    procedure scan_directory (const dir: string);
    function GetFileMask: string;
    procedure SetFileMask (s: string);
    function GetFileMaskCommaText: string;
    procedure SetFileMaskCommaText (s: string);
  public
    { Public declarations }
    Continue: Boolean;                   // caller sets the false to interrupt
    constructor Create (AOwner: TComponent);  override;
    destructor Destroy;  override;
    procedure ScanTree;                  // call here to scan tree
  published
    { Published declarations }
    property OnDirectoryFound: TNotifyEvent read FOnDirectoryFound write FOnDirectoryFound;
    property OnDirectoryDone: TNotifyEvent read FOnDirectoryDone write FOnDirectoryDone;
    property OnSearchDone: TNotifyEvent read FOnSearchDone write FOnSearchDone;
    property OnFileFound: TNotifyEvent read FOnFileFound write FOnFileFound;
    property FileFound: string read FFoundObject;
    property FileSize: integer read FFoundObjectSize;
    property SearchInProgress: boolean read FSearchInProgress;
    property InitialDirectory: string read FInitialDirectory write FInitialDirectory;
    property FileMask: string read GetFileMask write SetFileMask;
    property FileMaskCommaText: string read GetFileMaskCommaText write SetFileMaskCommaText;
    property Recursive: boolean read FRecursive write FRecursive default True;
    property FindHiddenFiles: boolean read FFindHiddenFiles write FFindHiddenFiles default False;
    property FindHiddenFolders: boolean read FFindHiddenFolders write FFindHiddenFolders default False;
  end;

procedure Register;


implementation

uses
  Windows, Dialogs;

procedure Register;
begin
  RegisterComponents('Davids', [TTreeScanner]);
end;

constructor TTreeScanner.Create (AOwner: TComponent);
begin
  Inherited Create (AOwner);
  FOnDirectoryFound := nil;       // no callback pointer yet for folder start
  FOnDirectoryDone := nil;        // or for folder complete
  FOnSearchDone := nil;           // or for search complete
  FOnFileFound := nil;            // or for found file
  FSearchInProgress := False;     // and we're not yet searching
  FRecursive := True;             // normally, scan sub-folders as well
  FFindHiddenFiles := False;      // normally, don't show hidden files
  FFindHiddenFolders := False;    // normally, don't show hidden folders
  FFileMask := '';
  FFileMaskList := TStringList.Create;
  with FFileMaskList do
    begin
    Sorted := True;
    Duplicates := dupIgnore;
    end;
end;

destructor TTreeScanner.Destroy;
begin
  FFileMaskList.Free;
  Inherited;
end;

procedure TTreeScanner.SetFileMask (s: string);
begin
  if FSearchInProgress then Exit;
  FFileMaskList.Clear;
  FFileMaskList.Add (s);
end;

function TTreeScanner.GetFileMask: string;
begin
  Result := '';
  if FFileMaskList.Count > 0 then Result := FFileMaskList.Strings [0];
end;

procedure TTreeScanner.SetFileMaskCommaText (s: string);
begin
  if FSearchInProgress then Exit;
  FFileMaskList.Clear;
  FFileMaskList.CommaText := s;
end;

function TTreeScanner.GetFileMaskCommaText: string;
begin
  Result := FFileMaskList.CommaText;
end;

procedure TTreeScanner.ScanTree;
begin
  FSearchInProgress := True;      // now searching
  FFoundObject := '';             // nothing found as yet
  FFoundObjectSize := 0;

  // Try to get a valid folder string.  If the user hasn't
  // specified a folder, use the current directory...
  if FInitialDirectory = '' then FInitialDirectory := GetCurrentDir;

  // ensure the folder ends in a '\'
  if FInitialDirectory [Length (FInitialDirectory)] <> '\' then
    FInitialDirectory := FInitialDirectory + '\';

  // now remove any file specification components...
  FInitialDirectory := ExtractFileDir (FInitialDirectory);
  // and again ensure that trailing backslash is present....
  if FInitialDirectory [Length (FInitialDirectory)] <> '\' then
    FInitialDirectory := FInitialDirectory + '\';
  // assume we wish to continue (at least to start with!)
  Continue := True;
  // and call the recursive scanning procedure
  scan_directory (FInitialDirectory);
  FSearchInProgress := False;      // note we have finished searching
  if Assigned (FOnSearchDone)      // finished search
    then FOnSearchDone (self);
end;

procedure TTreeScanner.scan_directory (const dir: string);
var
  f: TSearchRec;             // used for scanning a folder
  status: integer;           // value returned from FindFirst/FindNext
  file_list: TStringList;    // sorted list of files in the folder
  dir_list: TStringList;     // sorted list of sub-folders
  i: integer;                // used to traverse the above lists
  attributes: integer;       // what sort of things to find
begin
  Application.ProcessMessages;    // allow the caller to do some work
  if not Continue then Exit;      // interrupted? - simply return to caller

  // a folder doesn't actually have zero size - it's a file after all
  // but the size isn't reported by FindFirst so say it's zero.
  FFoundObjectSize := 0;             // "a folder has zero size"
  FFoundObject := dir;               // prepare to tell owner that we've
  if Assigned (FOnDirectoryFound)    // found a new folder
    then FOnDirectoryFound (self);

  // Create the string lists which are used to store the names returned from
  // the folder scan.  By making the list sorted, the names returned to the
  // caller will be in alphabetic order.  Note one slight "trick"; the most
  // wanted characteristic of a file is its size.  We therefore store the
  // size in the string list by using its associated "objects" list.  In
  // reality, these are 32-bit pointers to objects.  So rather than storing
  // pointers, we can store and recover 32-bit integers by type-casting.
  // In Win32, a file size is actually a 64-bit field, although Borland only
  // store the low 32-bits.  Once file sizes exceed 2GB this code will break.

  file_list := TStringList.Create;   // prepare list of files
  file_list.Sorted := True;          // and note is is to be in sorted order
  dir_list := TStringList.Create;    // prepare list of folders
  dir_list.Sorted := True;           // also sorted

  // There's a fault in Borland's Run-Time Library under Windows NT where the 
  // system function FindClose can get called with an invalid file handle if
  // the inital result is not zero, i.e. no files were found.  
  // The try..finally blocks below was found on the DDJ to fix this problem

  dir_list.BeginUpdate;              // probably not worth doing this, but...
  attributes := faDirectory;
  if FFindHiddenFolders then attributes := attributes or faHidden;
  // initialise the handle just in case of an exception
  f.FindHandle := INVALID_HANDLE_VALUE;
  try
    status := FindFirst (dir + '*.*', attributes, f);  // try the first find call
    if status <> 0 
    then f.FindHandle := INVALID_HANDLE_VALUE
    else
      while status = 0 do with f do    // keep looping while more entries
        begin
        if ((faDirectory and Attr) <> 0) and     // is this a folder?
           (Name <> '.') and (Name <> '..')      // ignore backlinks
          then dir_list.Add (dir + Name + '\');  // add it to folder list
        status := FindNext (f);        // see if there's another folder entry
        end;
  finally
    SysUtils.FindClose (f);            // all done, clean up from folder scan
  end;
  dir_list.EndUpdate;

  file_list.BeginUpdate;

  for i := 0 to FFileMaskList.Count-1 do
    begin
    // initialise the handle just in case of an exception
    f.FindHandle := INVALID_HANDLE_VALUE;
    try
      status := FindFirst (dir + FFileMaskList.Strings [i], faAnyFile, f);  // try the first find call
      if status <> 0
      then f.FindHandle := INVALID_HANDLE_VALUE
      else
        while status = 0 do with f do    // keep looping while more entries
          begin
          if ((faDirectory and Attr) = 0) and   // is this not a folder?
             ((faVolumeID and Attr) = 0) then   // is this not a volume ID?
            begin
            // don't add hidden files if we're not asked for them
            if ((Attr and faHidden) = faHidden) and (not FFindHiddenFiles)
              then
              else file_list.AddObject (dir + Name, Pointer (Size));    // add to file list
            end;
          status := FindNext (f);        // see if there's another entry
          end;
    finally
      SysUtils.FindClose (f);            // all done, clean up from the scan
    end;
    end;

  file_list.EndUpdate;

  // scan the sorted list of files and call back the owner for each
  for i := 0 to file_list.Count - 1 do
    begin
    FFoundObject := file_list.Strings [i];
    // type cast the stored pointer into a 32-bit integer
    FFoundObjectSize := Integer (file_list.Objects [i]);
    if Assigned (FOnFileFound) then FOnFileFound (self);
    end;
  file_list.Free;                     // return memory owned by the list

  FFoundObjectSize := 0;              // just to keep things clean.....
  // scan the sorted list of folders and scan each sub-folder
  for i := 0 to dir_list.Count - 1 do
    if FRecursive
    then
      scan_directory (dir_list.Strings [i])
    else
      begin
      FFoundObject := dir_list.Strings [i];
      if Assigned (FOnDirectoryFound) then FOnDirectoryFound (self);
      end;

  dir_list.Free;                     // return memory owned by the list

  if Assigned (FOnDirectoryDone)     // finished this folder
    then FOnDirectoryDone (self);
end;

end.

