unit PieChart;

{
V1.0.0  1996 May 26  First public version
V1.1.0  1996 Dec 08  Revised the Icon to match the chart colours better
V1.2.0  1997 Feb 08  Add right-click response
                     Add ClickedString property
V1.2.2  1997 Feb 16  Use Begin/EndUpdate around List and ListBox updates
                     Don't draw many small segments - improves speed
                     Use a Metafile as an intermediate drawing buffer
V1.2.4  1997 Mar 15  Move check for small segments to _before_ drawing them
                     - avoids final small segment filling 360 degrees!
V1.2.6  1997 Apr 12  Add Canvas property as public to allow copying chart
V1.3.0  1997 May 12  Version for Delphi 3.0
                     Make Colours an array rather than 6 values, keep as properties
                     Define both 32 x 32 and 16 x 16 icons
V1.3.2  1997 May 18  Derive from MultiColourControl (from TGraphicControl)
                     Allow for up to 8 colours, user defined
                     Add chart outline if no data (for inital "ghost" display)
                     Update with Howard Harvey's ideas
V1.4.0  1997 Dec 05  Compile with Delphi 3.02
                     Allow negative numbers in source data
                       - treat as positive for display, copy as-is to list box
                       - display in a "background", ghosted light grey colour
}

interface

uses
  SysUtils, WinTypes, WinProcs, Messages, Classes, Graphics, Controls,
  Forms, Dialogs, StdCtrls, MultiColour;

// the objects on our local string list contain both the original
// object and the real number which is the value of the data
type
  TObjectAndDouble = class(TObject)
    source_object: TObject;
    value: double;
  end;

type
  TStringListWithDouble = class(TStringList)
    destructor Destroy;  override;
  end;

const
  min_height = 65;
  min_width = 65;

type
  TPieChart = class(TMultiColourControl)
  private
    { Private declarations }
    FData: TStringListWithDouble; // computed internal data
    FListBox: TListBox;
    FOnDblClick: TNotifyEvent;
    FOnRightClick: TNotifyEvent;
    FMouseX, FMouseY: integer;
    FTotal: double;
    procedure SetListBox (ListBox: TListBox);
    procedure MouseUp (Button: TMouseButton; Shift: TShiftState; X, Y: Integer);  override;
    function get_object_from_mouse_coords: boolean;
  protected
    { Protected declarations }
    procedure Paint;  override;
    procedure DblClick;  override;
    procedure MouseDown (Button: TMouseButton; Shift: TShiftState; X, Y: Integer);  override;
  public
    { Public declarations }
    ClickedObject: TObject;
    ClickedString: string;
    constructor Create (AOwner: TComponent);  override;
    destructor Destroy;  override;
    procedure SetDataAndLabels (source_data: TStringList);
    procedure Clear;
    property Canvas;
  published
    { Published declarations }
    property Height default min_height;
    property Width default min_width;
    property Font;
    property ParentFont;
    property ListBox: TListBox read FListBox write SetListBox;
    property OnDblClick: TNotifyEvent read FOnDblClick write FOnDblClick;
    property OnRightClick: TNotifyEvent read FOnRightClick write FOnRightClick;
  end;

procedure Register;

implementation

procedure Register;
begin
  RegisterComponents('Davids', [TPieChart]);
end;

destructor TStringListWithDouble.Destroy;
var
  index: integer;
begin
  for index := 0 to Count - 1 do
    if Objects [index] <> nil then Objects [index].Free;
  Inherited Destroy;
end;

constructor TPieChart.Create (AOwner: TComponent);
var
  lst: TStringList;
begin
  Inherited Create (AOwner);
  Width := min_width;
  Height := min_height;
  FData := TStringListWithDouble.Create;
  FData.Sorted := False;
  FData.Duplicates := dupAccept;
  FListBox := nil;
  FTotal := 0.0;
  if csDesigning in ComponentState then
    begin
    lst := TStringList.Create;
    lst.Add ('3 Smallest');
    lst.Add ('5 Smaller');
    lst.Add ('7 Small');
    lst.Add ('9 Average-');
    lst.Add ('11 Average+');
    lst.Add ('13 Large');
    lst.Add ('17 Larger');
    lst.Add ('20 Largest');
    SetDataAndLabels (lst);
    lst.Free;
    end;
  FOnDblClick := nil;
  FOnRightClick := nil;
end;

destructor TPieChart.Destroy;
begin
  FData.Free;
  Inherited Destroy;
end;

procedure TPieChart.SetListBox (ListBox: TListBox);
begin
  FListBox := ListBox;
end;

procedure TPieChart.Clear;
begin
  FData.Clear;
  if FListBox <> nil
    then FListBox.Clear;   // remove any items in the list box
  Invalidate;
end;

procedure TPieChart.SetDataAndLabels (source_data: TStringList);

  procedure QuickSort (L, R: Integer);
  // sorts FData into reverse numerical order
  var
    I, J: integer;
    X: double;
  begin
    I := L;
    J := R;
    X := TObjectAndDouble (FData.Objects [(L + R) shr 1]).Value;
    repeat
      while TObjectAndDouble (FData.Objects[I]).Value > X do Inc(I);
      while TObjectAndDouble (FData.Objects[J]).Value < X do Dec(J);
      if I <= J then
      begin
        FData.Exchange(I, J);
        Inc(I);
        Dec(J);
      end;
    until I > J;
    if L < J then QuickSort(L, J);
    if I < R then QuickSort(I, R);
  end;

var
  index: integer;
  d: double;
  s: string;
  num: string;
  lbl: string;
  space: integer;
  code: integer;
  dd: TObjectAndDouble;
begin
  FData.Clear;
  if FListBox <> nil
    then FListBox.Clear;   // remove any items in the list box

  FData.BeginUpdate;
  FTotal := 0.0;
  for index := 0 to source_data.Count - 1 do
    begin
    s := Trim (source_data.Strings[index]);       // get the source string
    space := Pos (' ', s);
    if space = 0
    then
      begin
      num := s;
      lbl := '';                // assume no label part
      end
    else
      begin
      lbl := Trim (Copy (s, space, 999));
      num := Copy (s, 1, space-1);
      end;
    Val (num, d, code);
    d := Abs (d);               // V1.4 - allow for negative data
    if code = 0
    then
      begin
      FTotal := FTotal + d;
      dd := TObjectAndDouble.Create;
      dd.value := d;
      dd.source_object := source_data.Objects[index];
      FData.AddObject (s, dd);
      end
    else
      // should we raise an error here?
    end;

  if FData.Count <> 0 then
    begin
    QuickSort (0, FData.Count - 1);
    if FListBox <> nil then
      begin
      // copy the user's strings and objects to the list box
      FListBox.Items.BeginUpdate;
      for index := 0 to FData.Count - 1 do
         FListBox.Items.AddObject (
             FData.strings[index],
             TObjectAndDouble (Fdata.objects[index]).source_object);
      FListBox.Items.EndUpdate;
      end;
    end;

  FData.EndUpdate;

  Invalidate;
end;

procedure TPieChart.MouseDown (Button: TMouseButton; Shift: TShiftState;
                               X, Y: Integer);
begin
  // record the mouse co-ordinates in case of a subsequent double-click
  FMouseX := X;
  FMouseY := Y;
end;

procedure TPieChart.MouseUp (Button: TMouseButton; Shift: TShiftState;
                               X, Y: Integer);
begin
  if Button <> mbRight then Exit;
  if Assigned (FOnRightClick) then
    if get_object_from_mouse_coords then FOnRightClick (Self);
end;

procedure TPieChart.DblClick;
begin
  Inherited;
  if Assigned (FOnDblClick) then
    if get_object_from_mouse_coords then FOnDblClick (Self);
end;

function TPieChart.get_object_from_mouse_coords: boolean;

  function atan2 (y, x: double): double;
  var
     a: double;
  begin
    if x = 0.0
      then
        if y < 0.0
          then atan2 := -pi / 2 else atan2 := pi / 2
      else
        if y = 0.0
          then
            if x < 0.0
              then atan2 := pi else atan2 := 0.0
          else
            begin
            a := arctan (abs (y/x));
            if x > 0.0
              then
                if y > 0.0
                  then atan2 := a else atan2 := -a
            else
                if y > 0.0
                  then atan2 := pi - a else atan2 := -(pi - a)
            end;
  end;

var
  dx, dy, dr: double;
  pie_radius: double;
  test_theta, theta, d_theta, next_theta: double;
  found: boolean;
  index: integer;
  d: double;
  s: string;
  space: integer;
begin
  // find out where we were clicked - in client co-ordinates
  // translate this relative to the centre of the pie chart
  dx := FMouseX - Width div 2;
  dy := Height div 2 - FMouseY;
  dr := sqrt (sqr (dx) + sqr (dy));
  pie_radius := Width div 2;
  if Height > Width then pie_radius := Height;

  found := False;
  if (dr < pie_radius) and (FData.Count <> 0) then
    begin
    theta := atan2 (dy, dx);
    if theta < 0.0 then theta := theta +  2.0 * pi;
    test_theta := 0.0;
    index := 0;
    while (not found) and (index < FData.Count) do
      begin
      d := TObjectAndDouble (FData.Objects [index]).Value;
      d_theta := (2.0 * pi * d) / FTotal;
      next_theta := test_theta + d_theta;
      found := (theta > test_theta) and (theta < next_theta);
      if found
      then
        begin
        ClickedObject := TObjectAndDouble (FData.Objects [index]).source_object;
        s := Trim (FData.Strings [index]);
        space := Pos (' ', s);
        if space = 0
        then s := ''
        else s := Trim (Copy (s, space, 999));
        ClickedString := s;
        end
      else
        begin
        test_theta := next_theta;
        Inc (index);
        end;
      end;
    end;
  Result := found;
end;


procedure TPieChart.Paint;
const
  radius = 1000;    // nominal radius just for more accurate line edges
type
  Twhat_to_draw = (do_segment, do_label);
var
  colour_number: integer;
  theta, next_theta, d_theta: double;
  x0, y0: integer;
  x, y: integer;
  x1, y1: integer;
  segments_are_distinct: boolean;
  mf: TMetafile;
  cnv: TMetafileCanvas;

  procedure draw_label (const s: string);
  var
    pie_radius: integer;
    semi_width, semi_height: integer;
    x_mid, y_mid, x1, x2, y1, y2: integer;
    mid_theta: double;
    max_radius: double;
    text_radius: double;
    OldBkMode: integer;
  begin
    if (d_theta > 0.13) and (length (s) <> 0) then
      begin
      OldBkMode := SetBkMode (cnv.Handle, TRANSPARENT);
      if Width < Height
        then pie_radius := Width div 2
        else pie_radius := Height div 2;
      semi_width := cnv.TextWidth (s) div 2;
      semi_height := cnv.TextHeight (s) div 2;
      mid_theta := (theta + next_theta) / 2.0;
      // compute the central point, if it was on the rim
      x_mid := x0 + round (pie_radius * cos (mid_theta));
      y_mid := y0 - round (pie_radius * sin (mid_theta));
      // compute the bounding rectangle
      x1 := x_mid - semi_width;  x2 := x_mid + semi_width;
      y1 := y_mid - semi_height;  y2 := y_mid + semi_height;
      // find the maximum radius from the centre to the
      // four corners of the bounding rectangle
      max_radius := 0.0;
      text_radius := round (sqrt (sqr (x1 - x0) + sqr (y1 - y0)));
      if text_radius > max_radius then max_radius := text_radius;
      text_radius := round (sqrt (sqr (x2 - x0) + sqr (y1 - y0)));
      if text_radius > max_radius then max_radius := text_radius;
      text_radius := round (sqrt (sqr (x1 - x0) + sqr (y2 - y0)));
      if text_radius > max_radius then max_radius := text_radius;
      text_radius := round (sqrt (sqr (x2 - x0) + sqr (y2 - y0)));
      if text_radius > max_radius then max_radius := text_radius;
      // compute the text radius that will just fit inside the circle
      text_radius := 2.0 * pie_radius - max_radius;
      x_mid := x0 + round (text_radius * cos (mid_theta));
      y_mid := y0 - round (text_radius * sin (mid_theta));
      cnv.TextOut (x_mid - semi_width, y_mid - semi_height, s);
      SetBkMode (cnv.Handle, OldBkMode);
      end;
  end;

  procedure draw_pie_segment (ghost: boolean);
  begin
    if not ghost
      then cnv.Brush.Color := ColourArray [colour_number]
      else cnv.Brush.Color := clLtGray;
    Inc (colour_number);
    colour_number := colour_number mod FNumColours;
    cnv.Pie (0, 0, Width, Height, x, y, x1, y1);
  end;

  procedure compute_segment (delta: double;  s: string;
                             what: Twhat_to_draw;  ghost: boolean);
  begin
    if delta = 0.0 then Exit;
    d_theta := (2.0 * pi * delta) / FTotal;
    segments_are_distinct := d_theta > 0.007;
    if not segments_are_distinct then Exit;

    next_theta := theta + d_theta;
    x1 := x0 + round (radius * cos (next_theta));
    y1 := y0 - round (radius * sin (next_theta));
    if what = do_segment
      then draw_pie_segment (ghost)
      else draw_label (s);
    theta := next_theta;
    x := x1;
    y := y1;
  end;

var
  d: double;
  index: integer;
  s: string;
  space: integer;
  ghost: boolean;       // display segment colour or not
begin
  x0 := Width div 2;
  y0 := Height div 2;
  mf := TMetafile.Create;
  mf.Height := Height;
  mf.Width := Width;
  mf.Enhanced := True;
  cnv := TMetafileCanvas.Create (mf, 0);

  cnv.Pen.Color := clBlack;

  // draw a full segment to start with - later covered by distinct segments
  // make it slightly smaller so that later rounding errors don't show
  cnv.Brush.Color := clLtGray;
  cnv.Pen.Style := psDot;            // dotted to indicate incomplete display
  cnv.Ellipse (1, 1, Width-1, Height-1);
  cnv.MoveTo (x0, y0);
  cnv.LineTo (Width-2, y0);
  cnv.Pen.Style := psSolid;          // restore solid line for normal segments

  if FTotal > 0.0 then
    begin
    // prepare to draw the segments - set initial co-ordinates
    x := x0 + radius;
    y := y0;

    // set starting colour and inital angle
    colour_number := 0;
    theta := 0.0;
    segments_are_distinct := True;

    index := 0;
    while (index < FData.Count) and (segments_are_distinct) do
      begin
      d := TObjectAndDouble (FData.Objects [index]).Value;
      s := Trim (FData.Strings [index]);
      ghost := Pos ('-', s) = 1;
      compute_segment (d, '', do_segment, ghost);
      Inc (index);
      end;

    // prepare to label the segments - set initial co-ordinates
    x := x0 + radius;
    y := y0;
    theta := 0.0;
    segments_are_distinct := True;
    cnv.Font := Self.Font;
    index := 0;
    while (index < FData.Count) and (segments_are_distinct) do
      begin
      d := TObjectAndDouble (FData.Objects [index]).Value;
      s := Trim (FData.Strings [index]);
      ghost := Pos ('-', s) = 1;
      space := Pos (' ', s);
      if space = 0
      then s := ''
      else s := Trim (Copy (s, space, 999));
      compute_segment (d, s, do_label, ghost);
      Inc (index);
      end;

    end;

  // freeing the canvas causes the draw into the metafile
  cnv.Free;
  // now zap it all into the graphic's canvas at one go
  Canvas.Draw (1, 1, mf);
  mf.Free;
end;

end.

