unit MultiColour;

{
Revision history:

V1.0.0  1997 May 18  First version


Notes:

This unit defines the TMultiColourControl component.

The component itself has an array of TColor, and a count of the 
number of colours.  As presently implemented, there is a fixed 
maximum number of colours (8), but this is defined by a constant.


Properties:

NumColours      Integer       read/write  Number of colours

Colours         String        read/write  Comma separated list of 
                                          integers (hex returned)

OnColourChange  TNotifyEvent  read/write  Called if colours changed


The component will be redrawn if the NumColours or Colours 
properties are changed, but not if the ColourArray is directly 
modified.

The component is implemented as a derivative of TGraphicControl 
since it is expected the component will be used to derive the 
PieChart, or other TGraphicControl components yet to be written.

The property editor dialog has a private copy of the 
TMultiColourControl, and the editor's Edit method uses the 
GetStrValue and SetStrValue to convert between comma separated 
strings and an array of TColor.  The Edit method displays a small 
dialog box with all the colours displayed as label backgrounds, 
and responds to double-clicks on each label by calling up a 
standard TColorDialog editor.
}

interface

uses Windows, SysUtils, Classes, Graphics, Forms, Controls, StdCtrls,
  Buttons, ExtCtrls, Dialogs, DsgnIntf;

const
  max_colours = 8;

type
  TMultiColourControl = class(TGraphicControl)
  private
    FOnColourChange: TNotifyEvent;
    procedure SetStandardColours;
    procedure SetNumColours (Value: integer);
    function GetNumColours: integer;
  protected
    FNumColours: integer;
  public
    ColourArray: array [0..max_colours-1] of TColor;
    constructor Create (AOwner: TComponent);  override;
    function ListToString: string;
    procedure StringToList (const Value: string);
  published
    property NumColours: integer read GetNumColours write SetNumColours;
    property Colours: string read ListToString write StringToList;
    property OnColourChange: TNotifyEvent read FOnColourChange write FOnColourChange;
  end;

type
  TColourArrayPropertyEditor = class(TStringProperty)
  public
    function GetAttributes: TPropertyAttributes;  override;
    procedure Edit;  override;
  end;

type
  TColourListDlg = class(TForm)
    OKBtn: TButton;
    CancelBtn: TButton;
    Bevel1: TBevel;
    lblColour0: TLabel;
    Label1: TLabel;
    btnRevert: TButton;
    Bevel2: TBevel;
    procedure lblColour0DblClick(Sender: TObject);
    procedure FormShow(Sender: TObject);
    procedure FormCreate(Sender: TObject);
    procedure FormDestroy(Sender: TObject);
    procedure btnRevertClick(Sender: TObject);
  private
    { Private declarations }
    labels_done: boolean;
    colour_array: TMultiColourControl;
  public
    { Public declarations }
  end;

procedure Register;


implementation

{$R *.DFM}

procedure Register;
begin
  RegisterPropertyEditor (TypeInfo (string), TMultiColourControl,
                         'Colours', TColourArrayPropertyEditor);
end;

constructor TMultiColourControl.Create (AOwner: TComponent);
begin
  Inherited Create (AOwner);
  // now set up some default values
  FNumColours := 6;
  SetStandardColours;
  FOnColourChange := nil;
end;

procedure TMultiColourControl.SetStandardColours;
begin
  ColourArray [0] := RGB ($FF, $E0, $E0);
  ColourArray [1] := RGB ($FF, $FF, $E0);
  ColourArray [2] := RGB ($E0, $FF, $E0);
  ColourArray [3] := RGB ($E0, $FF, $FF);
  ColourArray [4] := RGB ($E0, $E0, $FF);
  ColourArray [5] := RGB ($FF, $E0, $FF);
  ColourArray [6] := RGB ($FF, $FF, $FF);
  ColourArray [7] := RGB ($E0, $E0, $E0);
end;

function TMultiColourControl.GetNumColours: integer;
begin
  Result := FNumColours;
end;

procedure TMultiColourControl.SetNumColours (value: integer);
begin
  if value in [1..max_colours] then
    begin
    FNumColours := value;
    Invalidate;
    end;
end;

function TMultiColourControl.ListToString: string;
var
  i: integer;
  s: string;
begin
  s := '';
  for i := 0 to FNumColours - 1 do
    begin
    s := s + '$' + IntToHex (ColourArray [i], 8);
    if i <> FNumColours - 1 then s := s + ',';
    end;
  Result := s;
end;

procedure TMultiColourControl.StringToList (const Value: string);
var
  s: TStringList;
  i: integer;
begin
  s := TStringList.Create;
  s.CommaText := Value;
  try
    i := s.Count;
    if i > max_colours then i := max_colours;
    FNumColours := i;
    for i := 0 to FNumColours - 1 do
      ColourArray [i] := StrToInt (s.Strings[i]);
    Invalidate;
    if Assigned (FOnColourChange) then FOnColourChange (Self);
  except
    FNumColours := 0;
  end;
  s.Free;
end;


function TColourArrayPropertyEditor.GetAttributes: TPropertyAttributes;
begin
  Result := [paDialog, paRevertable];
end;

procedure TColourArrayPropertyEditor.Edit;
var
  dlg: TColourListDlg;
begin
  dlg := TColourListDlg.Create (Application);
  try
    dlg.colour_array.Colours := GetStrValue;
    if dlg.ShowModal = mrOK then
      begin
      // update the copy held by the property editor from the dialog box
      SetStrValue (dlg.colour_array.Colours);
      end;
  finally
    dlg.Free;
  end;
end;


procedure TColourListDlg.FormCreate(Sender: TObject);
begin
  labels_done := False;
  colour_array := TMultiColourControl.Create (Self);
end;

procedure TColourListDlg.FormDestroy(Sender: TObject);
begin
  colour_array.Free;
end;

procedure TColourListDlg.FormShow(Sender: TObject);
// respond to the Show by adding all those coloured labels
var
  i: integer;
  lbl: TLabel;
  x: integer;
  w: integer;
  y: integer;
  dy: integer;
begin
  if labels_done then Exit;
  x := lblColour0.Left;
  dy := lblColour0.Height + 2;
  y := lblColour0.Top + dy;
  w := lblColour0.Width;
  lblColour0.Color := colour_array.ColourArray [0];
  for i := 1 to colour_array.FNumColours - 1 do
    begin
    lbl := TLabel.Create (Self);
    with lbl do
      begin
      Tag := i;           // for subsequent updating of the colour list
      Name := 'lblColour' + IntToStr (i);
      Caption := IntToStr (i);
      Width := w;
      Alignment := taCenter;
      AutoSize := False;
      OnDblClick := lblColour0.OnDblClick;
      Color := colour_array.ColourArray [i];
      Left := x;
      Top := y;
      Inc (y, dy);
      end;
    lbl.Parent := Self;
    end;
  labels_done := True;
end;

procedure TColourListDlg.lblColour0DblClick(Sender: TObject);
var
  dlg: TColorDialog;
begin
  dlg := TColorDialog.Create (Application);
  try
    dlg.Color := (sender as TLabel).Color;
    if dlg.Execute then
      with sender as TLabel do
        begin
        Color := dlg.Color;
        colour_array.ColourArray [Tag] := Color;
        end;
  finally
   dlg.Free;
  end;
end;

procedure TColourListDlg.btnRevertClick(Sender: TObject);
var
  i: integer;
  lbl: TLabel;
begin
  colour_array.SetStandardColours;
  for i := 0 to colour_array.FNumColours - 1 do
    begin
    lbl := TLabel (FindComponent ('lblColour' + IntToStr (i)));
    if Assigned (lbl) then lbl.Color := colour_array.ColourArray [i];
    end;
end;

end.
