unit FileProp;

{
Revision history:

V1.0.0   1997 Jan 16  Add version information page
V1.0.2   1997 Jan 19  Add more code for getting the Icon, using code from
                      Borland's Resource Explorer example
V1.0.4   1997 Feb 12  Handle zero 64-bit date/time as "unknown"
V1.0.6   1997 Mar 01  Add Compressed Size and Compression Attribute
V1.0.8   1997 Mar 22  Derive Icon using ShellAPI function - Win95/NT only
V1.0.10  1997 Mar 28  Remove references to Borland's Resource Explorer units
V1.0.12  1997 May 16  Make starting tab the "General" one
                      Display error message if the file isn't found
V1.1.0   1997 May 18  Version for Delphi 3.0
                      Correct failing to hide compression attributes after
                        one compressed file has been displayed
}

interface

uses
  Windows, Messages, SysUtils, Classes, Graphics, Controls, Forms, Dialogs,
  StdCtrls, ExtCtrls, ComCtrls, ShellAPI;

type
  TfrmFileProperties = class(TForm)
    PageControl1: TPageControl;
    tabGeneral: TTabSheet;
    tabVersion: TTabSheet;
    btnOK: TButton;
    Bevel1: TBevel;
    Bevel2: TBevel;
    Bevel3: TBevel;
    lblName: TLabel;
    Label1: TLabel;
    Label2: TLabel;
    Label3: TLabel;
    lblType: TLabel;
    lblLocation: TLabel;
    lblSize: TLabel;
    Label4: TLabel;
    Label5: TLabel;
    Label6: TLabel;
    Label7: TLabel;
    lblDOSName: TLabel;
    lblCreated: TLabel;
    lblModified: TLabel;
    lblAccessed: TLabel;
    Label8: TLabel;
    chkReadOnly: TCheckBox;
    chkArchive: TCheckBox;
    chkHidden: TCheckBox;
    chkSystem: TCheckBox;
    lblFileDescription: TLabel;
    Label9: TLabel;
    Label10: TLabel;
    lblFileVersion: TLabel;
    Label11: TLabel;
    lblLegalCopyright: TLabel;
    grpOtherVersionInformation: TGroupBox;
    lbxNames: TListBox;
    Label12: TLabel;
    Label13: TLabel;
    memValues: TMemo;
    Image1: TImage;
    lblCompressedSize: TLabel;
    lblCompressedSizeCaption: TLabel;
    chkCompressed: TCheckBox;
    pnlError: TPanel;
    procedure btnOKClick(Sender: TObject);
    procedure FormShow(Sender: TObject);
    procedure FormHide(Sender: TObject);
    procedure lbxNamesClick(Sender: TObject);
    procedure chkCompressedClick(Sender: TObject);
    procedure chkReadOnlyClick(Sender: TObject);
    procedure chkArchiveClick(Sender: TObject);
    procedure chkHiddenClick(Sender: TObject);
    procedure chkSystemClick(Sender: TObject);
  private
    { Private declarations }
    value_list: TStringList;
    hIcon: THandle;
    attr_hidden: boolean;
    attr_system: boolean;
    attr_readonly: boolean;
    attr_archive: boolean;
    attr_compressed: boolean;
  public
    { Public declarations }
    filename: string;
  end;

var
  frmFileProperties: TfrmFileProperties;

implementation

{$R *.DFM}


procedure TfrmFileProperties.btnOKClick(Sender: TObject);
begin
  ModalResult := mrOK;
end;


procedure TfrmFileProperties.FormShow(Sender: TObject);

  function FileTimeToStr (ft: TFileTime): string;
  // convert 64-bit date/time to string
  var
    lo, hi: double;
  begin
    if (ft.dwLowDateTime = 0) and (ft.dwHighDateTime = 0) then
      begin
      Result := '(unknown)';
      Exit;
      end;

    with ft do
      begin
      lo := dwLowDateTime and MaxLongInt;   // get the low 32 bits
      if dwLowDateTime < 0 then
        begin
        lo := lo + MaxLongInt;
        lo := lo + 1.0;
        end;
      hi := dwHighDateTime and MaxLongInt;  // get the low 32 bits
      if dwHighDateTime < 0 then
        begin
        hi := hi + MaxLongInt;
        hi := hi + 1.0;
        end;
      end;
    hi := hi * 65536.0;                   // left shift 32 bits
    hi := hi * 65536.0 + lo;              // and add in low 32 bits
    hi := hi / 1e7;                       // convert to seconds
    hi := hi / (60 * 60 *24);             // convert to days
    hi := hi - (299 * 365 + 70);          // to 1899 Dec 30
    Result := FormatDateTime ('yyyy-mmm-dd hh:nn:ss', hi);
  end;

const
  num_items = 9;
  items: array [1..num_items] of string =
    ('LegalCopyright', 'FileDescription', 'FileVersion',
     'Comments', 'CompanyName', 'InternalName',
     'OriginalFilename', 'ProductName',
     'ProductVersion');
var
  version_info: pointer;

  function get_version_info (const prefix, item: string): string;
  // get a particular item of version info
  var
    len: integer;
    buf: PChar;
  begin
    Result := '';
    if VerQueryValue (version_info, PChar (prefix + item), pointer (buf), len)
    then Result := buf;
  end;

  procedure display_version_info (const prefix, item: string);
  // find the right label component for this version info item (will be prefixed
  // by "lbl", get the version info and display it
  var
    s: string;
    comp: TLabel;
  begin
    comp := FindComponent ('lbl' + item) as TLabel;
    if (comp <> nil) then
      begin
      s := Get_version_info (prefix, item);
      comp.Caption := s;
      end;
  end;

const
  FILE_ATTRIBUTE_COMPRESSED = $00000800;  // missing from Borland's WINDOWS.PAS
var
  search: TSearchRec;
  s: string;
  ext: string;
  info_size: integer;
  len: integer;
  hVerInfo: THandle;
  buf: pointer;             // will point to locations within version_info
  prefix: string;
  item: 1..num_items;
  compressed_size: integer;
  ShInfo: TShFileInfo;
begin
  // prepare the list of values from the version information fields
  value_list := TStringList.Create;
  value_list.Sorted := False;
  tabVersion.TabVisible := False;
  PageControl1.ActivePage := tabGeneral;
  lbxNames.Clear;
  value_list.Clear;

  hIcon := 0;

  // we *must* have a real file to open.....
  if FindFirst (filename, faAnyFile, search) = 0
  then
    begin
    PageControl1.Visible := True;
    pnlError.Visible := False;
    // find out if there's any version information
    info_size := GetFileVersionInfoSize (PChar (filename), hVerInfo);
    if info_size > 0 then
      begin
      // process the version info
      GetMem (version_info, info_size);
      tabVersion.TabVisible := True;
      if GetFileVersionInfo (PChar (filename), hVerInfo, info_size, version_info) then
        begin
        // get the language prefix and make a hex string from it
        if VerQueryValue (version_info, '\VarFileInfo\Translation', buf, len) then
          begin
          if len = 2
          then s := '0000' + IntToHex (Pinteger(buf)^, 2)
          else s := IntToHex (Pinteger(buf)^, 4);
          // ensure the string is eight characters long
          while Length (s) < 8 do s := '0' + s;
          // and swap the bytes
          s := Copy (s, 5, 4) + Copy (s, 1, 4);
          // build the prefix
          Prefix := '\StringFileInfo\' + s + '\';

          // display the fixed items in the dialog box
          for item := 1 to 3 do
            display_version_info (prefix, items [item]);

          // prepare to build the list of variable items, the item names go
          // into the list box, and the values of the relevant version info
          // items go into value_list for subsequent display
          for item := 4 to num_items do
            begin
            // see if we can get the desired version information
            s := get_version_info (prefix, items [item]);
            if s <> '' then
              begin
              lbxNames.Items.Add (items [item]);  // add the name to the list box
              value_list.Add (s);                 // and the value to the values
              end;
            end;
          end;
        end;
      FreeMem (version_info);
      end;

    s := '';
    if ShGetFileInfo (PChar (filename), 0, ShInfo, SizeOf (ShInfo),
                   SHGFI_ICON or SHGFI_LARGEICON or SHGFI_TYPENAME) <> 0 then
      begin
      Image1.Picture.Icon.Handle := ShInfo.hIcon;
      s := ShInfo.szTypeName;
      end;

    with search, search.FindData do
      begin

      ext := ExtractFileExt (filename);   // get the file extension
      if s = '' then    // not known by the registry so just
        begin           // call it an EXT file (or whatever)
        Delete (ext, 1, 1);
        s := UpperCase (ext) + ' file';
        end;

      lblType.Caption := s;
      lblLocation.Caption := ExtractFilePath (filename);
      lblLocation.Hint := lblLocation.Caption;
      lblLocation.ShowHint := True;
      lblName.Caption := cFileName;
      if cAlternateFileName <> ''
      then lblDOSName.Caption := cAlternateFileName
      else lblDOSName.Caption := cFileName;
      Caption := cFileName + ' properties';
      if size < 1024
        then s := IntToStr (Size) + ' bytes'
        else
        if size < 1048576
          then s := Format ('%.2n KB', [Size / 1024])
          else s := Format ('%.2n MB', [Size / 1048576]);
      lblSize.Caption := s + Format (' (%.0n bytes)', [Size + 0.0]);
      lblModified.Caption := FileTimeToStr (ftLastWriteTime);
      lblCreated.Caption := FileTimeToStr (ftCreationTime);
      lblAccessed.Caption := Copy (FileTimeToStr (ftLastAccessTime), 1, 11);

      // now look at all the attributes and set the check boxes appropriately
      attr_hidden := (dwFileAttributes and faHidden) <> 0;
      chkHidden.Checked := attr_hidden;
      attr_system := (dwFileAttributes and faSysFile) <> 0;
      chkSystem.Checked := attr_system;
      attr_readonly := (dwFileAttributes and faReadOnly) <> 0;
      chkReadonly.Checked := attr_readonly;
      attr_archive := (dwFileAttributes and faArchive) <> 0;
      chkArchive.Checked := attr_archive;
      attr_compressed := (dwFileAttributes and File_Attribute_Compressed) <> 0;
      chkCompressed.Checked := False;
      chkCompressed.Visible := False;
      lblCompressedSize.Visible := False;
      lblCompressedSizeCaption.Visible := False;
      if attr_compressed then
        begin
        chkCompressed.Checked := True;
        chkCompressed.Visible := True;
        compressed_size := GetCompressedFileSize (PChar (filename), nil);
        if compressed_size <> -1 then
          begin
          if compressed_size < 1024
            then s := IntToStr (compressed_size) + ' bytes'
            else
            if compressed_size < 1048576
              then s := Format ('%.2n KB', [compressed_size / 1024])
              else s := Format ('%.2n MB', [compressed_size / 1048576]);
          lblCompressedSize.Caption := s + Format (' (%.0n bytes)', [compressed_size + 0.0]);
          lblCompressedSize.Visible := True;
          lblCompressedSizeCaption.Visible := True;
          end;
        end;
      end;
    end
  else
    begin
    PageControl1.Visible := False;
    Caption := 'Error - can''t find this file';
    pnlError.Caption := filename;
    pnlError.Visible := True;
    end;
  FindClose (search);
end;


procedure TfrmFileProperties.lbxNamesClick(Sender: TObject);
begin
  memValues.Clear;
  memValues.Text := value_list [lbxNames.ItemIndex];
end;


procedure TfrmFileProperties.FormHide(Sender: TObject);
begin
  value_list.Free;
  if hIcon <> 0 then DestroyIcon (hIcon);
end;


procedure TfrmFileProperties.chkCompressedClick(Sender: TObject);
begin
  with Sender as TCheckBox do
    begin
    if attr_compressed
    then State := cbChecked
    else State := cbUnchecked;
    end;
end;


procedure TfrmFileProperties.chkReadOnlyClick(Sender: TObject);
begin
  with Sender as TCheckBox do
    begin
    if attr_readonly
    then State := cbChecked
    else State := cbUnchecked;
    end;
end;


procedure TfrmFileProperties.chkArchiveClick(Sender: TObject);
begin
  with Sender as TCheckBox do
    begin
    if attr_archive
    then State := cbChecked
    else State := cbUnchecked;
    end;
end;


procedure TfrmFileProperties.chkHiddenClick(Sender: TObject);
begin
  with Sender as TCheckBox do
    begin
    if attr_hidden
    then State := cbChecked
    else State := cbUnchecked;
    end;
end;


procedure TfrmFileProperties.chkSystemClick(Sender: TObject);
begin
  with Sender as TCheckBox do
    begin
    if attr_system
    then State := cbChecked
    else State := cbUnchecked;
    end;
end;


end.

