unit FileList;

{
Description:

This unit produces a list of files, which can be sorted in various ways by
clicking on the top of the column, like Explorer.  It extends the TListView
component adding an AddFile method to add a file to the end of the list.  The
details for the file are extracted from the directory using FindFirst and the
ShellFileOperations.  Files that don't exist will not be added to the list.

Notes:

This unit requires the ShortDateFormat and LongTimeFormat to be
defined so for the time being, these have been defined in the unit as:
  ShortDateFormat := 'yyyy/mm/dd';
  LongTimeFormat := 'hh:nn';

Revision history:

V1.0.0  1997 Apr 27  First version
V1.1.0  1997 May 12  Version for Delphi 3.0
                     Add compile-time live data, but don't save it
V1.1.2  1997 May 18  Make RowSelect true by default
V1.1.4  1998 Feb 23  Add file version information column
V1.1.6  1998 Mar 15  Correct sort error when columns don't contain valid data
                     Force attributes column to have spaces where attr are unset
                     Add product version info column
V1.1.8  1998 Mar 16  Add even more defensive programming for text compare errors
                     Alter columns to make width depend on header, not body text
V1.1.10 1998 Mar 31  Add Begin/End update pairs to minimise repaints
                     Don't LowerCase the filename, the caller can do this
}

interface

uses
  Windows, Messages, SysUtils, Classes, Graphics, Controls, Forms, Dialogs,
  ComCtrls;

type
  TFileListView = class(TListView)
  private
    { Private declarations }
    FSortColumn: integer;               // which column we sort on
    FSortForward: boolean;              // and the sort order
    procedure FCompare(Sender: TObject; Item1,
      Item2: TListItem; Data: Integer; var Compare: Integer);
    procedure FColumnClick(Sender: TObject;
      Column: TListColumn);
  protected
    { Protected declarations }
  public
    { Public declarations }
    constructor Create(AOwner: TComponent);  override;
    procedure CreateWnd;  override;
    function AddFile (filename: string;  var filesize: integer): boolean;
  published
    { Published declarations }
    property AllocBy default 32;
    property HideSelection default False;
    property MultiSelect default True;
    property ReadOnly default True;
    property RowSelect default True;
    property ShowColumnHeaders default True;
    property ViewStyle default vsReport;
    property Items stored False;        // don't save the compile-time demo data
  end;

procedure Register;


implementation

uses ShellAPI;

procedure Register;
begin
  RegisterComponents('Davids', [TFileListView]);
end;


constructor TFileListView.Create (AOwner: TComponent);
begin
  Inherited Create (AOwner);
  // properties which are changed from the default
  AllocBy := 32;
  HideSelection := False;
  MultiSelect := True;
  ReadOnly := True;
  ShowColumnHeaders := True;
  ViewStyle := vsReport;
  // point to the sort method function
  OnCompare := FCompare;
  // point to the column click function
  OnColumnClick := FColumnClick;
  FSortForward := True;
  FSortColumn := 1;        // default is sort on size column
  ShortDateFormat := 'yyyy/mm/dd';
  LongTimeFormat := 'hh:nn';
end;


procedure TFileListView.CreateWnd;
var
  F: TSearchRec;
  stat: integer;
  dummy: integer;
  win_dir: array [0..240] of char;
  demo_dir: string;
begin
  Inherited;
  // we do the stuff here that needs a valid Window handle to work with
  RowSelect := True;
  if Columns.Count = 0 then
    begin
    with Columns.Add do
      begin
      Caption := 'Name';
      Width := ColumnHeaderWidth;
      end;
    with Columns.Add do
      begin
      Caption := 'Size';
      Width := ColumnHeaderWidth;
      Alignment := taRightJustify;
      end;
    with Columns.Add do
      begin
      Caption := 'Type';
      Width := ColumnHeaderWidth;
      end;
    with Columns.Add do
      begin
      Caption := 'Modified';
      Width := ColumnHeaderWidth;
      end;
    with Columns.Add do
      begin
      Caption := 'Attr';
      Alignment := taRightJustify;
      Width := ColumnHeaderWidth;
      end;
    with Columns.Add do
      begin
      Caption := 'File Ver';
      Alignment := taRightJustify;
      Width := ColumnHeaderWidth;
      end;
    with Columns.Add do
      begin
      Caption := 'Prod Ver';
      Alignment := taRightJustify;
      Width := ColumnHeaderWidth;
      end;
    if csDesigning in ComponentState then
      begin
      // as a demo, find all the VCL files in the Windows System folder
      GetSystemDirectory (@win_dir, 240);
      demo_dir := win_dir + '\';
      stat := FindFirst (demo_dir + 'vcl*.dpl', faReadOnly + faArchive, F);
      while stat = 0 do
        begin
        AddFile (ExtractFilePath (demo_dir) + F.Name, dummy);
        stat := FindNext (F);
        end;
      FindClose (F);
      end;
    end;
end;


procedure TFileListView.FColumnClick(Sender: TObject;
  Column: TListColumn);
var
  required_column: integer;
begin
  required_column := Column.Index;     // find out which column was clicked
  if required_column = FSortColumn     // are we already on that column?
  then FSortForward := not FSortForward   // yes - change the sort order
  else
    begin
    FSortColumn := required_column;    // note this is the column for sorting
    FSortForward := True;              // define a "normal" sort
    end;
  SortType := stData;                  // changing the sort type forces
  SortType := stNone;                  // the TListView to actually sort
end;


procedure TFileListView.FCompare(Sender: TObject; Item1,
  Item2: TListItem; Data: Integer; var Compare: Integer);
// this procedure returns a variable Compare based on the file size
// given in the list box
var
  size1, size2: integer;
  date1, date2: TDateTime;
  result: integer;
begin
  case FSortColumn of
    0: try       // sort on path name
         result := CompareText (Item1.Caption, Item2.Caption);
       except
         result := 0;
       end;
    1: begin     // sort on file size
       try
         size1 := StrToInt (Item1.SubItems.Strings [0]);
       except
         size1 := 0;
       end;
       try
         size2 := StrToInt (Item2.SubItems.Strings [0]);
       except
         size2 := 0;
       end;
       result := (size1 - size2);
       end;
    3: begin     // sort on file modified date
       try
         date1 := StrToDateTime (Item1.SubItems.Strings [2]);
       except
         date1 := Now;
       end;
       try
         date2 := StrToDateTime (Item2.SubItems.Strings [2]);
       except
         date2 := Now;
       end;
       result := Round (1E5*(date1 - date2));
       end;
    else
      try
        result := CompareText (Item1.SubItems.Strings [FSortColumn - 1],
                               Item2.SubItems.Strings [FSortColumn - 1])
      except
        result := 0;
      end;
  end;
  if FSortForward
  then Compare := result
  else Compare := -result;
end;


function TFileListView.AddFile (filename: string;  var filesize: integer): boolean;
// Format a single file entry for the ListView, recovering
// the size, attributes and modification time from the FindFirst,
// and the type name from a SHGetFileInfo call.
// Returns the file size of the named file

  function get_version_info (wanted: string): string;
  // This code is adapted from Brad Stowers VersionInfo resource from
  // http://www.pobox.com/~bstowers/delphi/  The original was a component, so
  // did not easily fit in with another component!
  const
    TRANSLATION_INFO = '\VarFileInfo\Translation';
  type
    TTranslationPair = packed record
      Lang,
      CharSet: word;
    end;
    PTranslationIDList = ^TTranslationIDList;
    TTranslationIDList = array [0..MAXINT div SizeOf (TTranslationPair)-1] of TTranslationPair;
  var
    version_info_size: integer;
    dummy: integer;
    version_info: PChar;
    fixed_info: PVSFixedFileInfo;
    IDs: PTranslationIDList;
    IDsLen: UINT;
    TranslationID: string;
    sub_block: string;
    ResStr: PChar;
    StrLen: UINT;
  begin
    Result := '';
    version_info_size := GetFileVersionInfoSize (PChar (filename), dummy);
    if version_info_size <> 0 then
      begin
      GetMem (version_info, version_info_size);
      GetFileVersionInfo (PChar (filename), dummy, version_info_size, version_info);
      VerQueryValue (version_info, '\', Pointer (fixed_info), dummy);
      if VerQueryValue (version_info, TRANSLATION_INFO, Pointer(IDs), IDsLen) then
        begin
        translationID := Format ('%.4x%.4x', [IDs[0].Lang, IDs[0].CharSet]);
        sub_block := '\StringFileInfo\' + translationID + '\' + wanted;
        if VerQueryValue (version_info, PChar (sub_block), Pointer (ResStr), StrLen) then
          Result := String (ResStr);
        end;
      FreeMem (version_info, version_info_size);
      end;
  end;

var
  s: TSearchRec;            // for the attributes, size and date-time
  ShInfo: TSHFileInfo;      // for the type name and icon info
  attributes: string;       // where we build "RHSA"
begin
  Result := False;            // assume failure
  if (SHGetFileInfo (PChar (filename),
                    0,
                    ShInfo, SizeOf (ShInfo),
                    SHGFI_TYPENAME) <> 0)
    and
      (FindFirst (filename, faAnyFile, s) = 0) then
    with s do
      begin
      // build the attributes string
      if (faReadonly and Attr) <> 0 then attributes := 'R'
                                    else attributes := ' ';
      if (faHidden and Attr) <> 0 then attributes := attributes + 'H'
                                  else attributes := attributes + ' ';
      if (faSysfile and Attr) <> 0 then attributes := attributes + 'S'
                                   else attributes := attributes + ' ';
      if (faArchive and Attr) <> 0 then attributes := attributes + 'A'
                                   else attributes := attributes + ' ';
      Items.BeginUpdate;      // freeze the UI
      with Items.Add do       // returns a TListItem for the "with" clause
        begin
        Caption := filename;                         // set main item caption
        SubItems.BeginUpdate;
        SubItems.Add (Trim (IntToStr (s.Size)));     // add the size
        SubItems.Add (Trim (ShInfo.szTypeName));     // add the TypeName
        SubItems.Add (DateTimeToStr (FileDateToDateTime (Time)));
        SubItems.Add (attributes);                   // add the attributes
        SubItems.Add (get_version_info ('FileVersion'));
        SubItems.Add (get_version_info ('ProductVersion'));
        SubItems.EndUpdate;
        end;
      Items.EndUpdate;        // unfreeze the UI
      filesize := s.Size;     // return file size to caller
      Result := True;         // with a success code
      end;
  FindClose (s);
end;


end.

