unit DiskList;

{
This component provdies a ListView component that displays disk drives.  As an
added convenience to the user, fixed drives which are locally attached are
automatically selected.  This component scans the drives when it is inserted
into a form so that the user can better size and judge the component.

So that the user can scan drives (which might take a few moments) at a time when
it is appropriate, the component does NOT scan drives when it first appears on a
form.  The user is expected to call the ScanDrives method.

As each drive is selected, the OnChange event for the control may be captured to
discover the current list of selected drives using the SelectedDrives function.

The SelectedDrives function returns a string of selected drive letters
  e.g. CDE

Finally, the SelectLocalDrives property enables the automatic selection of local
disk drives during a call to ScanDrives.

The behaviour of this component with differing ImageLists and ViewStyles has not
been investigated.

V1.0.0  1997 Mar 30  First version
V1.0.2  1997 Apr 02  Move adding Columns to CreateWnd method (thanks, Brad Stowers)
        1997 May 04  Remove a couple of unused variables
V1.1.0  1997 May 13  Version for Delphi 3.0
                     Don't save compile-time demo data
                     Remove yet another unused variable! (column_list)
                     Define default value for SelectLocalDrives property

Author: david.taylor@gecm.com
}

interface

uses
  Windows, Messages, SysUtils, Classes, Graphics, Controls, Forms, Dialogs,
  ComCtrls, ExtCtrls, ShellAPI;

type
  TDiskListView = class(TListView)
  private
    { Private declarations }
    image_list: TImageList;
    FSelectLocalDrives: boolean;
  protected
    { Protected declarations }
  public
    { Public declarations }
    constructor Create (AOwner: TComponent);  override;
    destructor Destroy;  override;
    procedure CreateWnd;  override;
    procedure ScanDrives;
    function SelectedDrives: string;
  published
    { Published declarations }
    property SelectLocalDrives: boolean read FSelectLocalDrives write FSelectLocalDrives default true;
    property Items stored False;        // don't save the compile-time demo data
  end;

procedure Register;


implementation

procedure Register;
begin
  RegisterComponents('Davids', [TDiskListView]);
end;

const
  expand_allocation = 8;   // expected number of drives visible


constructor TDiskListView.Create (AOwner: TComponent);
begin
  Inherited Create (AOwner);

  image_list := TImageList.Create (Self);   // create the image list
  with image_list do
    begin
    AllocBy := expand_allocation;           // default expand size
    ImageType := itImage;                   // and the other values
    Masked := False;
    Name := 'DiskListViewImageList';        // a unique name required?
    end;

  AllocBy := expand_allocation;
  ColumnClick := False;
  HideSelection := False;
  IconOptions.Arrangement := iaLeft;
  MultiSelect := True;
  ReadOnly := True;
  ShowColumnHeaders := True;
  SmallImages := image_list;
  SortType := stNone;
  ViewStyle := vsReport;

  FSelectLocalDrives := True;
end;


destructor TDiskListView.Destroy;
begin
  image_list.Free;
  Inherited Destroy;
end;


procedure TDiskListView.CreateWnd;
begin
  Inherited;
  if Columns.Count = 0 then
    with Columns.Add do
      begin
      Caption := 'Select drives';
      Width := ColumnTextWidth;
      if csDesigning in ComponentState then ScanDrives;
      end;
end;


function TDiskListView.SelectedDrives: string;

  function parse_drive (s: string): string;
  // extract the drive specification (e.g. A) from the
  // display name such as "3.5 inch floppy (A:)"
  var
    bra, ket: integer;
  begin
    Result := '';
    bra := Pos (' (', s);   // look for the starting string ...
    ket := Pos (':)', s);   // ... and ending string
    if (bra <> 0) and (ket <> 0) then
      Result := UpperCase (Copy (s, bra + 2, 1));
  end;

var
  drives: string;
  i: integer;
begin
  drives := '';                      // assume no drives selected
  with Items do
    for i := 0 to Count - 1 do       // scan down the list view of drives
      with Item [i] do
        if Selected then drives := drives + parse_drive (Caption);
  Result := drives;
end;


procedure TDiskListView.ScanDrives;
var
  image: TImage;
  drive: char;
  drive_root: string;
  drive_type: integer;
  ShInfo: TSHFileInfo;
begin
  image := TImage.Create (Self);
  image.Parent := Self;
  with image do
    begin
    Height := 16;
    Width := 16;
    AutoSize := False;
    Stretch := False;
    Visible := False;
    Name := 'DiskListViewTemporaryImage';
    end;

  // remove any existing images, and any existing disk drives
  image_list.Clear;
  Items.Clear;

  // we're going to do quite a few updates, so batch them together
  Items.BeginUpdate;

  // now find out what drives are available, add an icon and the display name to
  // then Drives list view.  If a drive is local, pre-select it for the user.
  for drive := 'A' to 'Z' do
    begin
    drive_root := drive + ':\';     // the root, e.g. A:\
    drive_type := GetDriveType (PChar (drive_root));   // get the drive type
    if (drive_type <> 0) and (drive_type <> 1) then    // it must be known
      begin
      // get the icon for this drive and the user-friendly display name
      if SHGetFileInfo (PChar (drive_root), 0, ShInfo, SizeOf (ShInfo),
              SHGFI_DISPLAYNAME or SHGFI_ICON or SHGFI_SMALLICON) <> 0 then
        begin
        with image do  // draw the icon returned into a bitmap
          begin
          Canvas.Brush.Color := clWindow;
          Canvas.FillRect (ClientRect);
          DrawIconEx (Canvas.Handle, 0, 0, ShInfo.hIcon,
            16, 16, 0, 0, DI_NORMAL);
          end;

        // Item.Add returns a TListItem that is used in the "with" statement
        with Items.Add do            // new item for the list view
          begin
          Caption := ShInfo.szDisplayName;   // named appropriately
          // add the bitmap to the Image List and note the resulting index
          ImageIndex := image_list.Add (image.Picture.Bitmap,
                                        image.Picture.Bitmap);
          if (drive_type = DRIVE_FIXED) and (FSelectLocalDrives) then
            Selected := True;
          end;
        end;
      end;
    end;

  // finally, submit all the batched updates
  Items.EndUpdate;

  image.Free;
end;


end.

