/* Copyright (c) 2002 Alex Derbeev (derbeev2@tut.by)
 * Portions (C)Eugene Roshal
 *
 * crx -- flexible crx/xck/etc apply utility
 *
 * You may distribute crx under the terms of either the GNU General Public
 * License or the Perl Artistic License.
 */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <assert.h>
#include <ctype.h>
#include <signal.h>
/* Perl-compatible regular expressions. */
/* Copyright (c) 1997-2001 University of Cambridge. */
/* ftp://ftp.csx.cam.ac.uk/pub/software/programming/pcre/pcre-xxx.tar.gz */
#include "pcre.h"

#define CRX /* comment it out if you want to build CRX "library" :) */

#include "crx.h"

#ifdef NDEBUG
#undef assert
#define assert(COND) if (!(COND)) quit ( mk_assert_str(#COND, __LINE__), RC_EXCEPT )
#endif

#if 1
#define dbg(msg) fprintf msg
#elif 0 /*doesnt work*/
#define dbg(msg)
#endif

#define NO_STUDY
#define NO_LICENSE
#if (defined(__CCDL__) && defined(DPMI)) || defined(__DJGPP__)
#define IO_SPEEDUP
#endif

#define OPEN_LOCAL_BLOCK {
#define CLOSE_LOCAL_BLOCK }

#if defined(JIBZ_CRC_INLINE)
#define JIBZCRC JibzCRC_inline
#elif defined(JIBZ_CRC)
#define JIBZCRC JibzCRC_asm
#else
#define JIBZCRC JibzCRC
#endif

BOOL o_deactivate = FALSE;
BOOL o_check_crc = TRUE;
BOOL o_fuzzy_test_only = FALSE;
BOOL o_quiet = FALSE;
BOOL o_verbose = FALSE;
BOOL o_abort = FALSE;
BOOL o_calc_crc = FALSE;

#if defined(__LCC__) || defined(_MSC_VER) || defined(__MINGW32__)
#define COMPILER_BAD_FILE_ASSIGN
#endif

#ifdef COMPILER_BAD_FILE_ASSIGN
#define GLOB_LOG stdout
#else
FILE* GLOB_LOG = stdout;
#endif

int Crc_type = ctCRC32;

const_ptr NO_FILE = "*";
dword CRCTab[256];

const_ptr license = "See the GNU General Public License for the details\n";

const_ptr my_regs[REG_COUNT] =
{
    "\\[([^\\]]+)\\](\\s*)size(\\s*)\\:(\\s*)\\d+(\\s*)\\,(\\s*)crc32(\\s*)\\:(\\s*)[0-9a-f]{8}",
    "^(.{2})size(\\s*)\\:(\\s*)\\d+",
    "[0-9a-f]{8}(\\s*)\\:(\\s*)[0-9a-f]{2}(\\s*)[0-9a-f]{2}",
    "^\\<(\\s([0-9a-f]|\\*|\\?|\\-|x){2})+",
    "\\#search(\\s([0-9a-f]|\\*|\\?|\\-|x){2})+",
    "^(\\s*)(\\;(.*))?$",
    "^\\+(\\s([0-9a-f]|\\*|\\?|\\-|x){2})+",
    "^\\>(\\s([0-9a-f]|\\*|\\?|\\-|x){2})+",
    "^(\\^file\\^(\\s+)\\\"([^\\\"]+)\\\")|(\\^filecl\\^((\\s+)\\\"([^\\\"]+)\\\")?)",
    "^((\\^(getkey|print|cls|file|filecl|size|crc|backup|compare|" \
    "search|test|replace|write|goto|(goto\\+)|(goto\\-)|exit|end)\\^))" /* {0} */,
    "(\\s([0-9a-f]|\\*|\\?|\\-|x){2})+"
};
pcre* pcre_data[REG_COUNT];
pcre_extra* pcre_extra_data[REG_COUNT];

const_ptr PREFIX[NUM_PREF] = {"Dsk://", "Chr://"};
dword crx_flags = 0;

const_ptr Jibz_commands[Jibz_Magic] = {
    "getkey", "print", "cls", "file", "filecl", "size", "crc", "backup",
    "compare", "search", "test", "replace", "write", "goto", "goto+", "goto-",
    "exit", "end"
};

int     apply_error_fuzzy_blocks_patched = 0;
char*   apply_error_bad_file = "";
dword   apply_error_offset = 0;
byte    apply_error_b = '\0';
BOOL    apply_error_Jibz_exit = FALSE;

#if defined(IO_SPEEDUP)
static FILE *myfopen(char const *fn, char const *flags)
{
    FILE *f = fopen(fn, flags);
#define fopen myfopen
    static char mybuf[65536];
    if (f)
    {
        setvbuf(f, mybuf, _IOFBF, sizeof(mybuf));
    }
    return f;
}
#endif

static char const* mk_assert_str(char const *msg, int line)
{
    static char buf[256];
    sprintf(buf, "Assertion failed: `%s', source line %d\n", msg, line);
    return buf;
}

void quit(char const *msg, int code)
{
    if (msg && *msg)
        fprintf(GLOB_LOG, "%s\n", msg);
    exit(code);
}

static char* mk_str(char *buf, char const *start, char const *end)
{
    memcpy(buf, start, end-start);
    buf[end-start] = '\0';
    return buf;
}

static BOOL eq_str(char const *src, char const *dst)
{
    return (BOOL)(strcmp(src, dst) == 0);
}

static void swap(int *x, int *y)
{
    int t = *x;
    *x = *y;
    *y = t;
}

static void swapb(byte *x, byte *y)
{
    byte t = *x;
    *x = *y;
    *y = t;
}

INLINE char *mystrdup(char const *s)
{
    int l;
    char *t;
    if (!s)
        return NULL;
    l = strlen(s);
    t = (char*)malloc(l+1);
    memcpy(t, s, l+1);
    return t;
}

INLINE int Check_match(Crx_parser_data *d, int r, int I, const_ptr s)
{
    return pcre_exec(pcre_data[r], pcre_extra_data[r],
            s? s : d->crx_data[I],
            s? (int)strlen((char*)s) : (int)strlen(d->crx_data[I]),
            0, 0,
            d->ovector, OVECCOUNT);
}

void InitCRC(void)
{
    int I, J;
    dword C;
    for (I = 0; I < 256; ++I)
    {
        for (C = I, J = 0; J < 8; ++J)
            C = (C & 1) ? (C >> 1) ^ 0xEDB88320L : (C >> 1);
        CRCTab[I] = C;
    }
}

dword CalcCRC32(dword StartCRC, byte const *Addr, int Size)
{
    int I;
    for (I = 0; I < Size; ++I)
        StartCRC = CRCTab[(byte) StartCRC ^ Addr[I]] ^ (StartCRC >> 8);
    return StartCRC;
}

#if !defined(JIBZ_CRC)
word JibzCRC(word StartCRC, byte const *buf, int buflen)
{
    word crc16 = StartCRC;
    int k, j;

    for (k = 0; k < buflen; k++)
    {
#if defined(__DMC__)
        word tt = crc16;
        tt ^= (word)buf[k] << 8;
        crc16 &= 0xff;
        crc16 |= tt & 0xff00;
#else
#if defined(LITTLE_ENDIAN)
        ((byte*)&crc16)[1] ^= buf[k];
#elif defined(BIG_ENDIAN)
        ((byte*)&crc16)[0] ^= buf[k];
#endif
#endif
        for (j = 0; j < 8; j++)
        {
            dword t = (dword)crc16+crc16;
            crc16 = (word) ((t <= 65535u)? t : t ^ 0x1021);
        }
    }
    return crc16;
}
#endif

int CalculateCRC(const_ptr fileName, int ct, char* CRC)
{
    char msg[256];
    FILE *f;
    int real_size;
    char *buf;
    unsigned C;

    f = fopen((char*)fileName, "rb");
    if (!f)
        return CRC_IO;

    fseek(f, 0, SEEK_END);
    real_size = (int)ftell(f);
    assert(real_size > 0);
    buf = (char*)malloc(real_size);

    #define RETURN(CODE) {if (f) fclose(f); free(buf); return CODE;}

    OPEN_LOCAL_BLOCK
    fprintf(GLOB_LOG, "Reading ");
    fseek(f, 0, SEEK_SET);
    if (fread(buf, real_size, 1, f) != 1)
        RETURN(CRC_IO)
    fputs("Ok\n", GLOB_LOG);
    CLOSE_LOCAL_BLOCK

    fclose(f); f = NULL;

    switch (ct)
    {
        case ctCRC32:
            OPEN_LOCAL_BLOCK

            C = (unsigned)~CalcCRC32((dword)4294967295u, (byte const*)buf, real_size);
            sprintf(msg, "CRC32=%08X", C);

            CLOSE_LOCAL_BLOCK
            break;
        case ctJibz:
            OPEN_LOCAL_BLOCK

            C = (unsigned)JIBZCRC((word)0, (byte const*)buf, real_size);
            sprintf(msg, "JibzCRC16=%04X", C);

            CLOSE_LOCAL_BLOCK
            break;
        default:
            assert(0);
    }
    strcpy(CRC, msg);
    RETURN(CRC_OK)
    #undef RETURN
}

void sanity_check(void)
{
    char const* msg;
    {
        assert(sizeof(byte) == 1);
        assert(sizeof(word) == 2);
        assert(sizeof(dword) == 4);
    }
    {
/*      assert(0xdeadbeef == *(dword*)"\xef\xbe\xad\xde"); */
        assert(sizeof(char) == 1);
    }
    {
        msg = "Beneath my wing it's gonna be alright";
        assert(0x4B5FF5C5 == ~CalcCRC32(4294967295u, (byte const*)msg, strlen(msg)));
        assert(0xACAA == JIBZCRC((word)0, (byte const*)msg, strlen(msg)));
    }
}

void usage(void)
{
    quit("crx [-dctaqvsrL] crxfile", RC_USAGE);
}

void myabort(int rc)
{
    quit("abort()", RC_ABORT | (rc&0));
}

Crx_parser_data *init_crx_parser_data(Crx_parser_data *p)
{
    p = (Crx_parser_data *) malloc(sizeof(Crx_parser_data));

    p->file = mystrdup((char const*)NO_FILE);
    p->extended_info = NULL;
    p->crx_data = NULL;
    p->file_data = NULL;
    p->crc_data = NULL;

    p->sz_crx_data = 0;
    p->sz_file_data = 0;
    p->sz_crc_data = 0;

    p->linecount = 0;
    p->filecount = 0;
    p->fuzzy = FALSE;
    p->frk_here = FALSE;
    p->sp_here = FALSE;
    p->jibz_here = FALSE;

    return p;
}

void init_task(void)
{
    static int called_once = 0;
    assert(++called_once == 1);
    InitCRC();
    init_regexp();
    setbuf(stdout, NULL);
    signal(SIGABRT, myabort);
    sanity_check();
}

void init_regexp(void)
{
    char const *error;
    int err_offset;
    int i;

    for (i = 0; i < REG_COUNT; i++)
    {
        pcre_data[i] = pcre_compile((const char *)my_regs[i], PCRE_CASELESS, &error, &err_offset, NULL);
        assert(pcre_data[i] != NULL);
#if defined(NO_STUDY)
        pcre_extra_data[i] = NULL;
#else
        pcre_extra_data[i] = pcre_study(pcre_data[i], 0, &error);
#endif
    }
}

static char** CharPtr_push_back(char **ptr, int *sz, char *data)
{
    (*sz)++;
    if (*sz == 1)
        ptr = (char**)malloc(sizeof(char**));
    else ptr = (char**)realloc(ptr, (*sz)*sizeof(char**));
    ptr[*sz-1] = data;
    return ptr;
}

static int* Int_push_back(int *ptr, int *sz, int data)
{
    (*sz)++;
    if (*sz == 1)
        ptr = (int*)malloc(sizeof(int));
    else ptr = (int*)realloc(ptr, (*sz)*sizeof(int));
    ptr[*sz-1] = data;
    return ptr;
}

static IVEC* IVEC_push_back(IVEC* ptr, int *sz, IVEC data)
{
    (*sz)++;
    if (*sz == 1)
        ptr = (IVEC*)malloc(sizeof(IVEC));
    else ptr = (IVEC*)realloc(ptr, (*sz)*sizeof(IVEC));
    ptr[*sz-1] = data;
    return ptr;
}

static JIBZ_CMD *JIBZ_CMD_push_back(JIBZ_CMD *ptr, int *sz, JIBZ_CMD data)
{
    (*sz)++;
    if (*sz == 1)
        ptr = (JIBZ_CMD*)malloc(sizeof(JIBZ_CMD));
    else ptr = (JIBZ_CMD*)realloc(ptr, (*sz)*sizeof(JIBZ_CMD));
    ptr[*sz-1] = data;
    return ptr;
}

static FILE_DATA *FILE_DATA_push_back(FILE_DATA *ptr, int *sz, FILE_DATA data)
{
    (*sz)++;
    if (*sz == 1)
        ptr = (FILE_DATA*)malloc(sizeof(FILE_DATA));
    else ptr = (FILE_DATA*)realloc(ptr, (*sz)*sizeof(FILE_DATA));
    ptr[*sz-1] = data;
    return ptr;
}

static PATCH_DATA *PATCH_DATA_push_back(PATCH_DATA *ptr, int *sz, PATCH_DATA data)
{
    (*sz)++;
    if (*sz == 1)
        ptr = (PATCH_DATA*)malloc(sizeof(PATCH_DATA));
    else ptr = (PATCH_DATA*)realloc(ptr, (*sz)*sizeof(PATCH_DATA));
    ptr[*sz-1] = data;
    return ptr;
}

static CRC_DATA *CRC_DATA_push_back(CRC_DATA *ptr, int *sz, CRC_DATA data)
{
    (*sz)++;
    if (*sz == 1)
        ptr = (CRC_DATA*)malloc(sizeof(CRC_DATA));
    else ptr = (CRC_DATA*)realloc(ptr, (*sz)*sizeof(CRC_DATA));
    ptr[*sz-1] = data;
    return ptr;
}

void crx_read_file(Crx_parser_data *this)
{
    int PL[NUM_PREF];
    static char buf[4096];

    char const *file = this->file;
    char *tok;
    FILE *crx_file;
    int l;

    PL[0] = strlen((char*)PREFIX[0]);
    PL[1] = strlen((char*)PREFIX[1]);

    if ((int)strlen(file) > PL[PX_DSK] && eq_str(mk_str(buf, file+0, file+PL[PX_DSK]), (char const*)PREFIX[PX_DSK]))
    {
        crx_file = fopen(file+PL[PX_DSK], "r");
        while (fgets(buf, sizeof(buf), crx_file) != NULL)
        {
            l = strlen(buf);
            if (l > 0 && buf[l-1] == '\n')
                buf[--l] = '\0';
            this->crx_data = CharPtr_push_back(this->crx_data, &this->sz_crx_data, mystrdup(buf));
            this->linecount++;
        }
        fclose(crx_file);
    }
    else if ((int)strlen(file) > PL[PX_CHR] && eq_str(mk_str(buf, file+0, file+PL[PX_CHR]), (char const*)PREFIX[PX_CHR]))
    {
        file += PL[PX_CHR];
        strcpy(buf, file);
        for (tok = strtok(buf, "\n"); tok; tok = strtok(0, "\n"))
        {
            this->crx_data = CharPtr_push_back(this->crx_data, &this->sz_crx_data, mystrdup(tok) );
            this->linecount++;
        }
    }
    else assert(0);
}

int make_fuzzy_matrix(char const *asc_src, char const *asc_dst, int** src, int** dst, int *sz_src, int *sz_dst)
{
/* src example: "DE AD ** EF", no leading or trailing whitespace */
    char const *end_src = asc_src + strlen(asc_src), *endp_src = asc_src;
    char const *end_dst = asc_dst + strlen(asc_dst), *endp_dst = asc_dst;
    int count = 0;

    char const *spc_src = strchr(asc_src, ' ');
    char const *spc_dst = strchr(asc_dst, ' ');
    if (spc_src && spc_dst && (spc_src-asc_src != spc_dst-asc_dst))
        return FUZZY_NO_SYNC;

    while (endp_src < end_src && endp_dst < end_dst)
    {
        int int_src = (int)strtol(asc_src, (char**)(&endp_src), 16);
        int int_dst = (int)strtol(asc_dst, (char**)(&endp_dst), 16);
        if (endp_src == asc_src)
            int_src = EOF, endp_src += 2;
        if (endp_dst == asc_dst)
            int_dst = EOF, endp_dst += 2;
        *src = Int_push_back(*src, sz_src, int_src);
        *dst = Int_push_back(*dst, sz_dst, int_dst);
        count++;
        asc_src = ++endp_src, asc_dst = ++endp_dst;
    }

    return count;
}

int make_fuzzy_matrix_single(char const *asc_src, int **src, int *sz_src)
{
    char const *end_src = asc_src + strlen(asc_src), *endp_src = asc_src;
    int count = 0;
    while (endp_src < end_src)
    {
        int int_src = (int)strtol(asc_src, (char**)(&endp_src), 16);
        if (endp_src == asc_src)
            int_src = EOF, endp_src += 2;
        *src = Int_push_back(*src, sz_src, int_src);
        count++;
        asc_src = ++endp_src;
    }
    return count;
}

void print_fuzzy_matrix(FILE *s, const_ptr prefix, int* x, int sz_x, const_ptr suffix, char blank)
{
    int i;
    fputs((char const*)prefix, s);
    for (i = 0; i < sz_x; i++)
    {
        fputc(' ', s);
        if (x[i] == EOF)
            fprintf(s, "%c%c", blank, blank);
        else fprintf(s, "%02x", x[i]);
    }
    fputs((char const*)suffix, s);
}

void crx_parse(Crx_parser_data *this)
{
    FILE_DATA fd;
    char cur_filename[256];
    int i, rc, J, k, size, r;
    int Cindex;

    JIBZ_CMD *JC = NULL;
    int sz_JC = 0;
    JIBZ_CMD JC_dummy;
    BOOL Q = FALSE;
    BOOL completed = TRUE;
    BOOL file_pending = TRUE;
    BOOL virgin = TRUE;

    static char C[256], Cdata[1024], s[256], asc_pattern[1024], str_src[4096], str_dst[4096], temp_str[1024];

    char const *closing, *Src;

    int *pattern = NULL, sz_pattern = 0;
    IVEC *IVECptr;

    char *endptr;
    int x;

    IVEC *cont_src = NULL, *cont_dst = NULL;
    int sz_cont_src = 0, sz_cont_dst = 0;
    IVEC IVECdummy;

    int *src = NULL, *dst = NULL;
    int sz_src = 0, sz_dst = 0;

    int X1;

    BOOL inside_patch_data = FALSE;

    unsigned v[3];
    dword offset;
    byte oldbyte, newbyte;

    PATCH_DATA PATCH_DATA_dummy;

    char scanbuf[256];
    char buf[256];
    int Size;
    unsigned CRC32;
    char *myptr;

    CRC_DATA CRC_DATA_dummy;

    BOOL match;

    fd.sz_patch_data = 0;
    fd.patch_data = NULL;
    fd.sz_cont_src = 0, fd.sz_cont_dst = 0;
    fd.cont_src = NULL, fd.cont_dst = NULL;
    fd.q = FALSE;
    fd.sz_cmd = 0;
    fd.cmd = NULL;
    fd.file = NULL;

    strcpy(cur_filename, (char const*)NO_FILE);

    for (i = 0; i < this->linecount; i++)
    {
        if ((rc = Check_match(this, REG_SEARCH_FRK, i, NULL)) > 0)
        {
            this->fuzzy = this->frk_here = TRUE;
            break;
        }
        if ((rc = Check_match(this, REG_SEARCH_SP, i, NULL)) > 0)
        {
            this->fuzzy = this->sp_here = TRUE;
            break;
        }
        if ((rc = Check_match(this, REG_JIBZ_FILE, i, NULL)) > 0)
        {
            this->fuzzy = this->jibz_here = TRUE;
            break;
        }
    }

    if (this->fuzzy)
    {
        fprintf(GLOB_LOG, "fuzzy %s\n", (this->frk_here? "(FRK FMT)" : (this->jibz_here? "(JIBZ FMT)" : "(SP FMT)")) );

        for (i = 0; i < this->sz_crx_data; )
        {
            rc = Check_match(this, this->jibz_here? REG_JIBZ_BLANK : REG_FUZZY_BLANK, i, NULL);
            match = this->jibz_here? (rc <= 0) : (rc > 0);
            if (match)
            {
                free(this->crx_data[i]);
                if (i < this->sz_crx_data-1)
                    memmove(this->crx_data+i, this->crx_data+i+1, ((this->sz_crx_data-1)-(i+1)+1)*sizeof(char*));
                this->sz_crx_data--;
            }
            else i++;
        }
        this->linecount = this->sz_crx_data;

        if (this->jibz_here)
        {
            if (o_verbose)
            {
#if 0
                for (i = 0; i < this->linecount; i++) fprintf(GLOB_LOG, "%s\n", this->crx_data[i]);
#endif
            }

            for (J = 0; J < this->linecount; J++)
            {

                if (Check_match(this, REG_JIBZ_BLANK, J, NULL) > 0)
                {
                    mk_str(C, this->crx_data[J]+this->ovector[0]+1, this->crx_data[J]+this->ovector[1]-1);

                    for (k = 0, size = strlen(C); k < size; C[k] = (char)tolower(C[k]), k++) ;

                    Cindex = -1;
                    for (k = 0; k < Jibz_Magic; k++)
                        if (eq_str(C, (char const*)Jibz_commands[k]))
                            Cindex = k;
                    assert(Cindex != -1);

                    if (file_pending)
                    {
                        if (Cindex != jbGetkey && Cindex != jbCls && Cindex != jbPrint)
                        {
                            if (Cindex != jbFile && Cindex != jbFilecl)
                                continue;
                            file_pending = FALSE;
                        }
                        else if (!virgin)
                            continue;
                    }

                    mk_str(Cdata, this->crx_data[J]+this->ovector[1], this->crx_data[J]+strlen(this->crx_data[J]));
                    Src = Cdata;
                    while (*Src && isspace(*Src))
                        Src++;

                    switch (Cindex)
                    {
                        case jbPrint:
                            JC_dummy.c = Cindex, JC_dummy.data = NULL;
                            JC = JIBZ_CMD_push_back(JC, &sz_JC, JC_dummy);
                            break;
                        case jbFile: case jbBackup: case jbCompare:
                            OPEN_LOCAL_BLOCK

                            assert(strlen(Src) > 0);
                            assert(Src[0] == '\"');
                            closing = strchr(Src+1, '\"');
                            if (closing == 0)
                            {
                                closing = Src+strlen(Src);
                                while ( isspace(*(closing-1)) )
                                    closing--;
                            }
                            mk_str(s, Src+1, closing);
                            if (Cindex == jbFile)
                            {
                                if (this->filecount > 0)
                                {
                                    fd.q = Q, Q = FALSE;
                                    fd.cmd = JC, fd.sz_cmd = sz_JC;
                                    JC = NULL;
                                    this->file_data = FILE_DATA_push_back(this->file_data, &this->sz_file_data, fd);
                                }
                                fd.file = mystrdup(s), completed = FALSE;
                                this->filecount++;
                            }
                            else
                            {
                                JC_dummy.c = Cindex, JC_dummy.data = mystrdup(s);
                                JC = JIBZ_CMD_push_back(JC, &sz_JC, JC_dummy);
                            }

                            CLOSE_LOCAL_BLOCK
                            break;
                        case jbFilecl:
                            assert(0);
                            break;
                        case jbSearch: case jbTest: case jbReplace: case jbWrite:
                            OPEN_LOCAL_BLOCK

                            assert(strlen(Src) > 0);
                            if (Cindex == jbReplace)
                                assert(isxdigit(*Src));
                            rc = Check_match(this, REG_PATTERN, 0, --Src);
                            assert(rc > 0);
                            mk_str(asc_pattern, Src+this->ovector[0]+1, Src+this->ovector[1]);
                            make_fuzzy_matrix_single(asc_pattern, &pattern, &sz_pattern);
                            IVECptr = (IVEC*)malloc(sizeof(IVEC));
                            IVECptr->v = pattern, IVECptr->sz = sz_pattern;
                            pattern = NULL, sz_pattern = 0;

                            JC_dummy.c = Cindex, JC_dummy.data = IVECptr;
                            JC = JIBZ_CMD_push_back(JC, &sz_JC, JC_dummy);

                            CLOSE_LOCAL_BLOCK
                            break;
                        case jbSize:    /* fall through */
                        case jbCrc:     /* fall through */
                        case jbGoto: case jbGoto_plus: case jbGoto_minus:
                            OPEN_LOCAL_BLOCK

                            assert(strlen(Src) > 0);
                            assert(isxdigit(*Src));
                            x = (int)strtol(Src, &endptr, 16);
                            assert(endptr > Src);
                            JC_dummy.c = Cindex;
                            JC_dummy.data = malloc(sizeof(int)), *(int*)JC_dummy.data = x;
                            JC = JIBZ_CMD_push_back(JC, &sz_JC, JC_dummy);

                            CLOSE_LOCAL_BLOCK
                            break;
                        case jbGetkey:   /* fall through */
                        case jbCls:      /* fall through */
                        case jbExit: case jbEnd:
                            JC_dummy.c = Cindex, JC_dummy.data = NULL;
                            JC = JIBZ_CMD_push_back(JC, &sz_JC, JC_dummy);
                            if (Cindex == jbExit)
                                Q = file_pending = TRUE, virgin = FALSE;
                            if (Cindex == jbExit || Cindex == jbEnd)
                            {
                                if (this->filecount > 0)
                                {
                                    fd.q = Q, Q = FALSE;
                                    fd.cmd = JC, fd.sz_cmd = sz_JC;
                                    JC = NULL;
                                    this->file_data = FILE_DATA_push_back(this->file_data, &this->sz_file_data, fd);
                                    completed = TRUE;
                                }
                            }
                            if (Cindex == jbEnd)
                                goto QUIT_JIBZ;
                            break;
                        default:
                            assert(0);
                    }
                }
            }
        QUIT_JIBZ:
            if (!completed)
            {
                fd.q = Q;
                fd.cmd = JC, fd.sz_cmd = sz_JC;
                this->file_data = FILE_DATA_push_back(this->file_data, &this->sz_file_data, fd);
            }
        }
        else if (this->frk_here)
        {
            for (i = 0; i < this->linecount; )
            {
                if ((rc = Check_match(this, REG_SEARCH_FRK, i, NULL)) > 0)
                {
                    assert(i < this->linecount-2);

                    src = NULL, dst = NULL;
                    sz_src = 0, sz_dst = 0;

                    X1 = this->ovector[0], /*X2 = this->ovector[1]-1,*/ J = i;

                    mk_str(str_src, this->crx_data[J]+X1+2, this->crx_data[J]+this->ovector[1]);

                    {
                        for (J++; J < this->linecount && (rc = Check_match(this, REG_PLUS_FRK, J, NULL)) > 0; J++)
                        {
                            strcat(str_src, " ");
                            strcat(str_src, mk_str(temp_str, this->crx_data[J]+this->ovector[0]+2, this->crx_data[J]+this->ovector[1]));
                        }
                        assert(J < this->linecount);
                    }

                    rc = Check_match(this, REG_CHANGE_FRK, J, NULL);
                    assert(rc > 0 && this->ovector[0] == X1);

                    mk_str(str_dst, this->crx_data[J]+X1+2, this->crx_data[J]+this->ovector[1]);

                    {
                        for (J++; J < this->linecount && (rc = Check_match(this, REG_PLUS_FRK, J, NULL)) > 0; J++)
                        {
                            strcat(str_dst, " ");
                            strcat(str_dst, mk_str(temp_str, this->crx_data[J]+this->ovector[0]+2, this->crx_data[J]+this->ovector[1]));
                        }
                        assert(J < this->linecount);
                    }

                    r = make_fuzzy_matrix(str_src, str_dst, &src, &dst, &sz_src, &sz_dst);
                    assert(r != FUZZY_NO_SYNC && r > 0);

                    if ((crx_flags & crx_fVerbose) != 0)
                    {
                        print_fuzzy_matrix(GLOB_LOG, "<", src, sz_src, "\n", '*');
                        print_fuzzy_matrix(GLOB_LOG, ">", dst, sz_dst, "\n", '*');
                    }

                    IVECdummy.v = src, IVECdummy.sz = sz_src;
                    cont_src = IVEC_push_back(cont_src, &sz_cont_src, IVECdummy);
                    IVECdummy.v = dst, IVECdummy.sz = sz_dst;
                    cont_dst = IVEC_push_back(cont_dst, &sz_cont_dst, IVECdummy);

                    if (this->crx_data[J][X1] == '=')
                    {
                        fd.file = mystrdup(this->crx_data[J]+2);
                        fd.cont_src = cont_src, fd.cont_dst = cont_dst;
                        fd.sz_cont_src = sz_cont_src, fd.sz_cont_dst = sz_cont_dst;
                        this->file_data = FILE_DATA_push_back(this->file_data, &this->sz_file_data, fd);
                        cont_src = NULL, cont_dst = NULL;
                        sz_cont_src = 0, sz_cont_dst = 0;
                        this->filecount++;
                        i = J+1;
                    }
                    else i = J;
                }
                else i++;
            }
        }
        else if (this->sp_here)
        {
            quit("SP format's not supported yet ;(", RC_BAD_FMT);
        }
        else assert(0);

        return;
    } /* fuzzy */

    for (i = 0; i < this->linecount; i++)
    {
        if ((rc = Check_match(this, REG_OFFSET_OLD_NEW, i, NULL)) > 0)
        {
            if (!inside_patch_data && i > 0)
            {
                strcpy(cur_filename, this->crx_data[i-1]);
                this->filecount++;
                fd.file = mystrdup(cur_filename);
                this->file_data = FILE_DATA_push_back(this->file_data, &this->sz_file_data, fd);
            }
            assert(!eq_str(cur_filename, (char const*)NO_FILE));

            if (this->sz_crc_data == 1 && this->filecount == 1 && this->crc_data[0].crct == crcNO_CRC && eq_str(this->crc_data[0].file, (char const*)NO_FILE))
                this->crc_data[0].file = mystrdup(fd.file); /* file name for size checking in xck */

            sscanf(this->crx_data[i]+this->ovector[0], "%08x :%02x%02x", &v[0], &v[1], &v[2]);

            offset = (dword)v[0];
            oldbyte = (byte)v[1], newbyte = (byte)v[2];

            PATCH_DATA_dummy.offset = offset;
            PATCH_DATA_dummy.oldbyte = oldbyte;
            PATCH_DATA_dummy.newbyte = newbyte;

            this->file_data[this->filecount-1].patch_data = PATCH_DATA_push_back(this->file_data[this->filecount-1].patch_data,
                &this->file_data[this->filecount-1].sz_patch_data, PATCH_DATA_dummy);

            if (this->file_data[this->filecount-1].sz_patch_data == 1)
                fprintf(GLOB_LOG, "found byte patch section for `%s'\n", cur_filename);
            inside_patch_data = TRUE;
        }
        else if ((rc = Check_match(this, REG_CHECK_SIZE, i, NULL)) > 0)
        {
            fprintf(GLOB_LOG, "found weak check section\n");
            strcpy(scanbuf, this->crx_data[i]+this->ovector[0]);
            myptr = scanbuf;
            while (*++myptr)
                *myptr = (char)tolower(*myptr);
            sscanf(scanbuf, "%*c size :%d", &Size);

            CRC_DATA_dummy.file = mystrdup((char const*)NO_FILE);
            CRC_DATA_dummy.crc = 0;
            CRC_DATA_dummy.crct = crcNO_CRC;
            CRC_DATA_dummy.size = Size;

            this->crc_data = CRC_DATA_push_back(this->crc_data, &this->sz_crc_data, CRC_DATA_dummy);

            inside_patch_data = FALSE;
        }
        else if ((rc = Check_match(this, REG_CHECK_SIZE_CRC32, i, NULL)) > 0)
        {
            fprintf(GLOB_LOG, "found extended check section\n");
            strcpy(scanbuf, this->crx_data[i]+this->ovector[0]);
            myptr = strchr(scanbuf, ']');
            while (*++myptr)
                *myptr = (char)tolower(*myptr);

            sscanf(scanbuf, "[%s size :%d , crc32 :%08x", buf, &Size, &CRC32);

            assert(strlen(buf) > 0);
            buf[strlen(buf)-1] = '\0'; /*destroy closing bracket*/

            CRC_DATA_dummy.file = mystrdup(buf);
            CRC_DATA_dummy.crc = (dword)CRC32;
            CRC_DATA_dummy.crct = crcCRC32;
            CRC_DATA_dummy.size = Size;

            this->crc_data = CRC_DATA_push_back(this->crc_data, &this->sz_crc_data, CRC_DATA_dummy);

            inside_patch_data = FALSE;
        }
        else inside_patch_data = FALSE;
    }
}

int test_pattern(int* src, int sz_src, char const *ptr, int size)
{
    int bytes = sz_src;
    BOOL match = TRUE;
    int J;

    assert(bytes > 0);

    if (bytes > size)
        return -1;

    for (J = 0; J < bytes; J++)
    {
        if (src[J] != EOF && (byte)src[J] != (byte)ptr[J])
        {
            match = FALSE;
            break;
        }
    }

    return match? 0 : -1;
}

int write_pattern(int* src, int sz_src, char *ptr, int size)
{
    int bytes = sz_src;
    int bytes_altered = 0;
    int J;
    assert(bytes > 0);

    if (bytes > size)
        return -1;

    for (J = 0; J < bytes; J++)
    {
        if (src[J] != EOF)
        {
            ptr[J] = (byte)src[J];
            bytes_altered++;
        }
    }

    return bytes_altered;
}

int search_pattern(int* src, int sz_src, char const *ptr, int size)
{
    int bytes = sz_src;
    int sequence = 0;
    int sig = -1;
    int i, J;
    char const *bufptr = ptr;
    int chunk_size;
    char const *susp_block;
    BOOL found;

    assert(bytes > 0);

    if (bytes > size)
        return -1;

    for (i = 0; i < bytes; i++)
        if (src[i] != EOF)
        {
            sig = i;
            break;
        }
    assert(sig != -1);

    while (size-(ptr-bufptr) > 0 && (ptr = (char*)memchr(ptr, src[sig], size-(ptr-bufptr))) != NULL)
    {
        chunk_size = size-(ptr-bufptr);
        if (chunk_size+sig >= bytes && ptr-bufptr >= sig)
        {
            susp_block = ptr-sig;
            found = TRUE;
            for (J = sig+1; J < bytes; J++)
            {
                if (src[J] != EOF && (byte)src[J] != (byte)susp_block[J])
                {
                    found = FALSE;
                    break;
                }
            }
            if (found)
            {
                sequence++;
                return ptr-bufptr;
                /*ptr += bytes-1;*/
            }
        }
        ptr++;
    }
    return -1;
}

int apply_jibz_patch(FILE_DATA fd, BOOL get_back, BOOL test_only)
{
    BOOL Q = fd.q;
    JIBZ_CMD *cmd = fd.cmd;
    int cmdsize = fd.sz_cmd;
    BOOL patched = FALSE;
    int sz_seq_ovec = 0, sz_seq_lvec = 0;
    int *seq_ovec = NULL, *seq_lvec = NULL;
    int blocks_patched = 0;
    BOOL good = TRUE;
    int real_size;
    char *buf;
    byte *B;
    int myptr;
    int Command;
    void *data;
    char const *prefix;
    FILE *f;
    IVEC *IVECptr, *vx;
    int *x;
    word crc16;
    FILE *baq;
    int r;
    int write_offset;
    int write_size;

    fprintf(GLOB_LOG, "patching file `%s' (Jibz mode)...\n", fd.file);

    if (get_back)
        quit("Can't reverse Jibz script yet :)", RC_EXCEPT);

    assert(fd.sz_cmd > 0);

    f = fopen(fd.file, "r+b");
    if (!f)
        return APPLY_FILE_NOT_FOUND;

    fseek(f, 0, SEEK_END);
    real_size = (int)ftell(f);
    assert(real_size > 0);

    buf = (char*)malloc(real_size);
    #define RETURN(CODE) {free(buf); fclose(f); return CODE;}

    OPEN_LOCAL_BLOCK
    fprintf(GLOB_LOG, "Reading ");
    fseek(f, 0, SEEK_SET);
    if (fread(buf, real_size, 1, f) != 1)
        RETURN(APPLY_FUZZY_IO)
    fprintf(GLOB_LOG, "Ok\n");
    CLOSE_LOCAL_BLOCK

    fprintf(GLOB_LOG, "Open Jibz console(TM)\n");

    OPEN_LOCAL_BLOCK /* apply jibz patch */

    B = (byte*)buf;
    myptr = 0;

    for (Command = 0; good && Command < cmdsize; Command++)
    {
        data = cmd[Command].data;

        if (o_verbose)
        {
            prefix = 0;
            switch (cmd[Command].c)
            {
                case jbSearch:      prefix = "<"; break;
                case jbTest:        prefix = "!"; break;
                case jbWrite:       prefix = ">"; break;
            }
            if (prefix)
            {
                IVECptr = (IVEC*)data;
                print_fuzzy_matrix(GLOB_LOG, prefix, IVECptr->v, IVECptr->sz, "\n", '*');
            }

            fprintf(GLOB_LOG, "Running %s", Jibz_commands[cmd[Command].c]);
        }

        switch (cmd[Command].c)
        {
            case jbGetkey:
            case jbCls:
                break;
            case jbPrint:
                break;
            case jbSize:
                good = *(int*)data == real_size;
                break;
            case jbCrc:
                OPEN_LOCAL_BLOCK

                crc16 = JIBZCRC((word)0, B, real_size);

                if (o_verbose)
                    fprintf(GLOB_LOG, " [ crc16=%04X (%04X) ]", crc16, (word)*(int*)data);

                good = (word)*(int*)data == crc16;

                CLOSE_LOCAL_BLOCK
                break;
            case jbBackup:
                OPEN_LOCAL_BLOCK

                if (!test_only)
                {
                    assert(!patched);
                    fflush(f);
                    baq = fopen((char const*)data, "wb");
                    good = fwrite((char const*)B, real_size, 1, baq) == 1;
                    fclose(baq);
                }

                CLOSE_LOCAL_BLOCK
                break;
            case jbCompare:
                good = FALSE;
                break;
            case jbSearch:
                OPEN_LOCAL_BLOCK

                vx = (IVEC*)data, x = vx->v;
                assert(vx->sz > 0);

                if (myptr < 0 || myptr >= real_size || (int)vx->sz > real_size-myptr)
                {
                    good = FALSE;
                    break;
                }

                r = search_pattern(x, vx->sz, (char const*)(B+myptr), real_size-myptr);
                if (r == -1)
                    good = FALSE;
                else myptr += r;

                CLOSE_LOCAL_BLOCK
                break;
            case jbTest:
                OPEN_LOCAL_BLOCK

                vx = (IVEC*)data, x = vx->v;
                assert(vx->sz > 0);

                if (myptr < 0 || myptr >= real_size || (int)vx->sz > real_size-myptr)
                {
                    good = FALSE;
                    break;
                }

                r = test_pattern(x, vx->sz, (char const*)(B+myptr), real_size-myptr);
                good = r != -1;

                CLOSE_LOCAL_BLOCK
                break;
            case jbWrite:
                OPEN_LOCAL_BLOCK

                vx = (IVEC*)data, x = vx->v;
                assert(vx->sz > 0);

                if (myptr < 0 || myptr >= real_size || (int)vx->sz > real_size-myptr)
                {
                    good = FALSE;
                    break;
                }

                r = write_pattern(x, vx->sz, (char*)(B+myptr), real_size-myptr);
                if (r == -1)
                    good = FALSE;
                else
                {
                    if (r > 0)
                    {
                        if (!test_only)
                            patched = TRUE;
                        blocks_patched++;
                        seq_ovec = Int_push_back(seq_ovec, &sz_seq_ovec, myptr);
                        seq_lvec = Int_push_back(seq_lvec, &sz_seq_lvec, vx->sz);
                    }
                    myptr += vx->sz;
                }

                CLOSE_LOCAL_BLOCK
                break;
            case jbReplace:
                OPEN_LOCAL_BLOCK

                vx = (IVEC*)data, x = vx->v;
                assert(vx->sz == 2);
                assert(x[0] != EOF && x[1] != EOF);

                if (myptr < 0 || myptr >= real_size)
                {
                    good = FALSE;
                    break;
                }

                if (B[myptr] == (byte)x[0])
                {
                    B[myptr] = (byte)x[1];
                    if (!test_only)
                        patched = TRUE;
                    blocks_patched++;
                    seq_ovec = Int_push_back(seq_ovec, &sz_seq_ovec, myptr);
                    seq_lvec = Int_push_back(seq_lvec, &sz_seq_lvec, 1);
                    myptr++;
                }
                else good = FALSE;

                CLOSE_LOCAL_BLOCK
                break;
            case jbGoto:
                myptr = *(int*)data;
                if (myptr < 0 || myptr >= real_size)
                    good = FALSE;
                break;
            case jbGoto_plus:
                myptr += *(int*)data;
                if (myptr < 0 || myptr >= real_size)
                    good = FALSE;
                break;
            case jbGoto_minus:
                myptr -= *(int*)data;
                if (myptr < 0 || myptr >= real_size)
                    good = FALSE;
                break;
            case jbExit: case jbEnd:
                break;
            default:
                assert(0);
        }

        if (o_verbose)
        {
            fprintf(GLOB_LOG, " - %s\n", (good? "Ok" : "error ;(") );
        }

    }
    CLOSE_LOCAL_BLOCK

    fprintf(GLOB_LOG, "Close Jibz console(TM)\n");

    OPEN_LOCAL_BLOCK
    if (patched)
    {
        fprintf(GLOB_LOG, "Writing ");
        write_offset = 0;
        write_size = real_size;
        if (blocks_patched == 1)
        {
            write_offset = seq_ovec[0];
            write_size = seq_lvec[0];
        }
        fseek(f, write_offset, SEEK_SET);
        if (fwrite(buf+write_offset, write_size, 1, f) != 1)
            RETURN(APPLY_FUZZY_IO)
        fprintf(GLOB_LOG, "Ok\n");
    }
    CLOSE_LOCAL_BLOCK

    if (patched)
        fprintf(GLOB_LOG, "done\n");

    apply_error_fuzzy_blocks_patched = blocks_patched;
    apply_error_Jibz_exit = Q && good;

    RETURN(patched? APPLY_OK : APPLY_NOTHING_TODO)
    #undef RETURN
}

int apply_fuzzy_patch(FILE_DATA fd, BOOL get_back, BOOL test_only)
{
    BOOL patched = FALSE;
    int sz_seq_ovec = 0, sz_seq_lvec = 0;
    int *seq_ovec = NULL, *seq_lvec = NULL;
    int blocks_patched = 0;
    int fuzzy_blocks = fd.sz_cont_src;
    FILE *f;
    int real_size;
    char *buf;
    int B;
    int *src, *dst;
    IVEC *vsrc, *vdst;
    int bytes;
    int sequence;
    int k, i, J;
    int sig;
    char *ptr, *bufptr;
    int chunk_size;
    char *susp_block;
    BOOL found;
    BOOL altered;
    int write_offset;
    int write_size;

    fprintf(GLOB_LOG, "patching file `%s' (SP/FRK mode)...\n", fd.file);

    assert(fuzzy_blocks > 0);

    f = fopen(fd.file, "r+b");
    if (!f)
        return APPLY_FILE_NOT_FOUND;

    fseek(f, 0, SEEK_END);
    real_size = (int)ftell(f);
    assert(real_size > 0);
    buf = (char*)malloc(real_size);
    #define RETURN(CODE) {free(buf); fclose(f); return CODE;}

    OPEN_LOCAL_BLOCK
    fprintf(GLOB_LOG, "Reading ");
    fseek(f, 0, SEEK_SET);
    if (fread(buf, real_size, 1, f) != 1)
        RETURN(APPLY_FUZZY_IO)
    fprintf(GLOB_LOG, "Ok\n");
    CLOSE_LOCAL_BLOCK

    fprintf(GLOB_LOG, "Searching\n");

    OPEN_LOCAL_BLOCK /* apply fuzzy patch */

    for (B = 0; B < fuzzy_blocks; B++)
    {
        vsrc = &fd.cont_src[B], vdst = &fd.cont_dst[B];
        src = vsrc->v, dst = vdst->v;
        assert(vsrc->sz == vdst->sz);
        bytes = vsrc->sz;
        sequence = 0;

        assert(bytes > 0);

        if (get_back)
        {
            for (k = 0; k < bytes; k++)
                if (dst[k] != EOF)
                    swap(&src[k], &dst[k]);
        }

        sig = -1;
        for (i = 0; i < bytes; i++)
            if (src[i] != EOF)
            {
                sig = i;
                break;
            }
        assert(sig != -1);

        ptr = buf, bufptr = ptr;

        while (real_size-(ptr-bufptr) > 0 && (ptr = (char*)memchr(ptr, src[sig], real_size-(ptr-bufptr))) != NULL)
        {
            chunk_size = real_size-(ptr-bufptr);
            if (chunk_size+sig >= bytes && ptr-bufptr >= sig)
            {
                susp_block = ptr-sig;
                found = TRUE;
                for (J = sig+1; J < bytes; J++)
                {
                    if (src[J] != EOF && (byte)src[J] != (byte)susp_block[J])
                    {
                        found = FALSE;
                        break;
                    }
                }
                if (found)
                {
                    if (!test_only)
                    {
                        altered = FALSE;
                        for (J = 0; J < bytes; J++)
                        {
                            if (dst[J] != EOF)
                            {
                                susp_block[J] = (byte)dst[J];
                                altered = TRUE;
                            }
                        }
                        if (altered)
                        {
                            blocks_patched++;
                            seq_ovec = Int_push_back(seq_ovec, &sz_seq_ovec, susp_block-bufptr);
                            seq_lvec = Int_push_back(seq_lvec, &sz_seq_lvec, bytes);
                            patched = TRUE;
                        }
                    }
                    sequence++;
                    fprintf(GLOB_LOG, "found matching sequence N%d at 0x%X (fuzzy block %d), patched :)\n",
                        sequence, (unsigned)(susp_block-bufptr), B+1);
                    ptr += bytes-1;
                }
            }
            ptr++;
        }
    }
    CLOSE_LOCAL_BLOCK

    OPEN_LOCAL_BLOCK
    if (patched)
    {
        fprintf(GLOB_LOG, "Writing ");
        write_offset = 0;
        write_size = real_size;
        if (blocks_patched == 1)
        {
            write_offset = seq_ovec[0];
            write_size = seq_lvec[0];
        }
        fseek(f, write_offset, SEEK_SET);
        if (fwrite(buf+write_offset, write_size, 1, f) != 1)
            RETURN(APPLY_FUZZY_IO)
        fprintf(GLOB_LOG, "Ok\n");
    }
    CLOSE_LOCAL_BLOCK

    if (patched)
        fprintf(GLOB_LOG, "done\n");

    apply_error_fuzzy_blocks_patched = blocks_patched;

    RETURN(patched? APPLY_OK : APPLY_NOTHING_TODO)
    #undef RETURN
}

int apply_dummy_patch(FILE_DATA fd, BOOL get_back, BOOL test_only)
{
    int bytecount = fd.sz_patch_data;
    FILE *f;
    int i;
    dword offset;
    byte oldbyte;
    byte newbyte;
    int b;

    fprintf(GLOB_LOG, "patching file `%s' (dummy mode)...\n", fd.file);

    if (!bytecount)
        return APPLY_NOTHING_TODO;

    f = fopen(fd.file, "r+b");
    if (!f)
        return APPLY_FILE_NOT_FOUND;
    #define RETURN(CODE) {fclose(f); return CODE;}

    for (i = 0; i < bytecount; i++)
    {
        offset = fd.patch_data[i].offset;
        oldbyte = fd.patch_data[i].oldbyte;
        newbyte = fd.patch_data[i].newbyte;
        if (get_back)
            swapb(&oldbyte, &newbyte);
        fprintf(GLOB_LOG, "[%#08x: %#02x->%#02x]___", offset, oldbyte, newbyte);
        apply_error_offset = offset;

        OPEN_LOCAL_BLOCK

        if (fseek(f, offset, SEEK_SET) != 0)
            RETURN(APPLY_SEEK_ERROR)
        b = fgetc(f);
        if (b == EOF)
            RETURN(APPLY_SEEK_ERROR)
        if ((byte)b != oldbyte)
        {
            apply_error_b = (byte)b;
            RETURN(APPLY_BAD_BYTE)
        }
        if (!test_only)
        {
            fseek(f, -1, SEEK_CUR);
            if (fputc(newbyte, f) == EOF)
                RETURN(APPLY_WRITE_ERROR)
            fflush(f);
        }

        CLOSE_LOCAL_BLOCK

        fprintf(GLOB_LOG, "ok\n");
    }
    fprintf(GLOB_LOG, "done\n");
    RETURN(APPLY_OK)
    #undef RETURN
}

int apply_patch(FILE_DATA fd, BOOL get_back, BOOL fuzzy, BOOL jibz, BOOL test_only)
{
    int r = APPLY_OK;

    assert(!eq_str(fd.file, (char const*)NO_FILE));
    apply_error_bad_file = mystrdup(fd.file);

    if (fuzzy)
    {
        if (jibz)
            r = apply_jibz_patch(fd, get_back, test_only);
        else r = apply_fuzzy_patch(fd, get_back, test_only);
    }
    else r = apply_dummy_patch(fd, get_back, test_only);

    return r;
}

int check_crc(CRC_DATA crc)
{
    BOOL size_only = crc.crct == crcNO_CRC;
    int real_size;
    #define BLOCK_SIZE 65536
    static char buf[BLOCK_SIZE];
    int block_count;
    int block_last_size;
    int i;
    FILE *f;
    dword real_crc;

    assert(crc.crct == crcCRC32 || (crc.crct == crcNO_CRC && crc.size != -1));

    fprintf(GLOB_LOG, "checking %s for `%s'...", (size_only? "size" : "crc"), crc.file);

    f = fopen(crc.file, "rb");
    if (!f)
        return CHECKCRC_IO;
    #define RETURN(CODE) {fclose(f); return CODE;}
    fseek(f, 0, SEEK_END);
    real_size = (int)ftell(f);
    if (crc.size != -1 && crc.size != real_size)
        RETURN(CHECKCRC_BAD_SIZE)
    if (!size_only)
    {
        fseek(f, 0, SEEK_SET);
        block_count = real_size / BLOCK_SIZE;
        block_last_size = real_size % BLOCK_SIZE;
        real_crc = (dword)4294967295u;

        OPEN_LOCAL_BLOCK
        for (i = 0; i < block_count; i++)
        {
            if (fread(buf, BLOCK_SIZE, 1, f) != 1)
                RETURN(CHECKCRC_IO)
            real_crc = CalcCRC32(real_crc, (byte const*)buf, BLOCK_SIZE);
        }
        if (fread(buf, block_last_size, 1, f) != 1)
            RETURN(CHECKCRC_IO)
        real_crc = ~CalcCRC32(real_crc, (byte const*)buf, block_last_size);
        CLOSE_LOCAL_BLOCK

        if (real_crc != crc.crc)
            RETURN(CHECKCRC_BADCRC)
    }
    fprintf(GLOB_LOG, "ok\n");
    RETURN(CHECKCRC_OK)
    #undef RETURN
    #undef BLOCK_SIZE
}

int mygetopt(int argc, char **argv)
{
    int a = 1;
    int r;
    char CRC[256];

    if (argc == 1)
        usage();

    for (; a < argc && argv[a][0] == '-'; a++)
    {
        while (*++argv[a]) switch (*argv[a])
        {
            case 'd':
                o_deactivate = TRUE;
                break;
            case 'c':
                o_check_crc = FALSE;
                break;
            case 't':
                o_fuzzy_test_only = TRUE;
                break;
            case 'a':
                o_abort = TRUE;
                break;
            case 'q':
#if defined(IO_SPEEDUP)
# undef fopen
#endif
                o_quiet = TRUE;
#ifndef COMPILER_BAD_FILE_ASSIGN
                GLOB_LOG = fopen(NULL_DEV, "wb");
#endif
                break;
            case 'v':
                o_verbose = TRUE;
                crx_flags |= crx_fVerbose;
                break;
            case 's':
                OPEN_LOCAL_BLOCK

                if (a+2 < argc)
                {
                    if (eq_str(argv[a+1], "Jibz"))
                        Crc_type = ctJibz;
                    else if (eq_str(argv[a+1], "CRC32"))
                        Crc_type = ctCRC32;
                    else usage();
                    r = CalculateCRC(argv[a+2], Crc_type, CRC);
                    if (r != CRC_OK)
                        quit("Can't calculate CRC ;(", r);
                    fprintf(GLOB_LOG, "%s\n", CRC);
                    quit(0, 0xdead);
                }
                else usage();

                CLOSE_LOCAL_BLOCK
                break;
            case 'r':
                quit(PROG_NAME" v"PROG_VERSION" compiled at "__DATE__" "__TIME__" by "PROG_WHO_COMPILES, RC_REVISION);
            case 'L':
                quit((char const*)license, RC_LICENSE);
            case '?': case 'h':
                usage();
            case '-':
                a++;
                goto AIJAJAI;
            default:
                quit("bad option ;(", RC_BAD_OPTION);
        }
    }

AIJAJAI:

    if (a >= argc)
        usage();

    return a;
}

#if defined( CRX ) && !defined( CRX_LIB )

int crx_main(int argc, char **argv)
{
    static char buf[4096];
    int crc_count;
    char e[256] = {""};
    int r;
    int first_error;
    BOOL everything_ok;
    int i;
    Crx_parser_data *crx_parser_data = NULL;

    init_task();
    crx_parser_data = init_crx_parser_data(crx_parser_data);

    strcpy(buf, (char const*)PREFIX[PX_DSK]);
    strcat(buf, argv[mygetopt(argc, argv)]);
    crx_parser_data->file = mystrdup(buf);

    crx_read_file(crx_parser_data);
    crx_parse(crx_parser_data);

    assert(crx_parser_data->filecount > 0);

    crc_count = crx_parser_data->sz_crc_data;

    if (!o_check_crc || o_deactivate)
        crc_count = 0;

    for (i = 0; i < crc_count; i++)
    {
        r = check_crc(crx_parser_data->crc_data[i]);

        if (r != CHECKCRC_OK)
        {
            switch (r)
            {
                case CHECKCRC_BADCRC:
                    strcpy(e, "bad crc");
                    break;
                case CHECKCRC_IO:
                    strcpy(e, "i/o error");
                    break;
                case CHECKCRC_BAD_SIZE:
                    strcpy(e, "bad size");
                    break;
            }

            fprintf(GLOB_LOG, "\nerror(check_crc): %s\n", e);
            quit("sorry, bad crc ;(", r);
        }
    }

    first_error = 1;
    everything_ok = TRUE;

    for (i = 0; i < crx_parser_data->filecount; i++)
    {
        r = apply_patch(crx_parser_data->file_data[i], o_deactivate, crx_parser_data->fuzzy, crx_parser_data->jibz_here, o_fuzzy_test_only);

        if (apply_error_Jibz_exit)
            quit("premature exit (Jibz mode)", 0xdead);

        if (r != APPLY_OK)
        {
            everything_ok = FALSE;

            if (first_error == 1)
                first_error = r;

            switch (r)
            {
                case APPLY_FILE_NOT_FOUND:
                    strcpy(e, "file not found");
                    break;
                case APPLY_SEEK_ERROR:
                    sprintf(e, "seek error, offset=%#08x(%u)", apply_error_offset, apply_error_offset);
                    break;
                case APPLY_BAD_BYTE:
                    sprintf(e, "bad byte (%#02x) at offset %#08x(%u)", apply_error_b, apply_error_offset, apply_error_offset);
                    break;
                case APPLY_NOTHING_TODO:
                    strcpy(e, "nothing to do ;(");
                    break;
                case APPLY_FUZZY_IO:
                    strcpy(e, "fuzzy patch i/o error");
                    break;
                case APPLY_WRITE_ERROR:
                    sprintf(e, "write error at offset %#08x(%u)", apply_error_offset, apply_error_offset);
                    break;
            }

            fprintf(GLOB_LOG, "\nerror(apply_patch): %s\n", e);
            if (o_abort)
                quit("premature exit ;(", r);
        }
    }

    return everything_ok? 0xdead : first_error; /* 173 means ok ;) */
}

#if defined(DJGPP)
#include <crt0.h>
char **__crt0_glob_function (char *arg)
{
    return 0;
}
#endif

int main(int argc, char **argv)
{
    int r = crx_main(argc, argv);
    return r;
}

#endif
