/* Copyright (c) 2002 Alex Derbeev (derbeev2@tut.by)
 * Portions (C)Eugene Roshal
 *
 * crx -- flexible crx/xck/etc apply utility
 *
 * You may distribute crx under the terms of either the GNU General Public
 * License or the Perl Artistic License.
 */

#ifndef CRX_H
#define CRX_H

#include <iostream>
#include <string>
#include <vector>
#include <exception>
#include <cstdio>
#include <cassert>
// Perl-compatible regular expressions.
// Copyright (c) 1997-2001 University of Cambridge.
// ftp://ftp.csx.cam.ac.uk/pub/software/programming/pcre/pcre-xxx.tar.gz
#include "pcre.h"

#define PROG_NAME "crx"
#define PROG_AUTHOR "alex derbeev"
#define PROG_WHO_COMPILES "alex derbeev"
#define PROG_VERSION "0.10"
#define AUTHOR_EMAIL "derbeev2@tut.by"

#define CRX_BEGIN_NAMESPACE namespace Crx {
#define CRX_END_NAMESPACE }

CRX_BEGIN_NAMESPACE

using namespace std;
#if defined(STLPORT)
using namespace stlport;
#endif

// START OF PLATFORM SPECIFIC STUFF

typedef ostream assignable_ostream;

typedef char const *const const_ptr;
typedef unsigned char byte;

#ifndef LITTLE_ENDIAN
# define LITTLE_ENDIAN
#endif

#if defined(__64BIT__)

#error "Swell!"

typedef unsigned short word;
typedef unsigned int dword;
#if defined(JIBZ_CRC)
#undef JIBZ_CRC
#endif

#else
typedef unsigned short word;
typedef unsigned int dword;
#endif

#if defined(JIBZ_CRC)
extern "C" {
    word JibzCRC_asm(word StartCRC, byte *buf, int buflen);
}
#endif

#if defined(__GNUC__)&&defined(__i386__)
#define JIBZ_CRC_INLINE
inline word JibzCRC_inline(word StartCRC, byte const *buf, int buflen)
{
    word r = 0;
    asm volatile (
        "movw %3,%%ax\n\t"
        "movl %2,%%ebx\n\t"
        "movl %1,%%edi\n\t"
        "1:xorb (%%edi),%%ah\n\t"
        "incl %%edi\n\t"
        "movl $8,%%ecx\n\t"
        "2:addw %%ax,%%ax\n\t"
        "jnc 3f\n\t"
        "xorw $0x1021,%%ax\n\t"
        "3:loopw 2b\n\t"
        "decl %%ebx\n\t"
        "jnz 1b\n\t"
        "movw %%ax,%0\n\t"
        : "=g" (r)
        : "g" (buf), "g" (buflen), "g" (StartCRC)
        : "cc", "%ax", "%ebx", "%ecx", "%edi"
    );
    return r;
}
#endif

// END OF PLATFORM SPECIFIC STUFF

#define Z static

class CrxAssertException : public exception
{
public:
    CrxAssertException(char const *cause, int l) : msg(cause), SrcLine(l)  { }
    virtual ~CrxAssertException() throw() { };
    char const *getMsg() const {return msg;}
    int getSrcLine() const {return SrcLine;}
private:
    char const *msg;
    int SrcLine;
};

class Sync
{
public:
    Sync()
    {
        if (called_once)
        {
            ios::sync_with_stdio(0);
            called_once = false;
        }
    }
    ~Sync() {}

private:
    Z bool called_once;

public:
    Z int const IOBUFSZ = 65536;
    Z char iobuf[IOBUFSZ];
};

class glob
{
public:
    Z void InitCRC();
    Z dword CalcCRC32(dword StartCRC, byte const *Addr, int Size);
#if !defined(JIBZ_CRC)
    Z word JibzCRC(word StartCRC, byte const *buf, int buflen);
#endif

    Z int CalculateCRC(const_ptr fileName, int ct, string& CRC);
    enum {CRC_OK = 0, CRC_IO = 1};

    Z void quit(char const *msg, int code);
    Z void myabort(int);
    Z void sanity_check();
    Z void init_task();

public:                                 // of course it should'nt be over here :)
    Z int mygetopt(int argc, char **argv);
    Z void usage();

public:
    Z const_ptr license;
    Z bool o_deactivate, o_check_crc, o_fuzzy_test_only, o_quiet, o_verbose, o_abort, o_calc_crc, o_xcode;
    enum {ctCRC32 = 0, ctJibz};
    Z int Crc_type;

public:
    Z const_ptr NO_FILE;

    enum {RC_EXCEPT = 1, RC_USAGE, RC_ABORT, RC_BAD_FMT, RC_REVISION,
        RC_LICENSE, RC_BAD_OPTION};

    Z assignable_ostream *mylog;

private:
    Z dword CRCTab[256];
    Z Sync sync;
};

struct PATCH_DATA
{
    dword offset;
    byte oldbyte, newbyte;

    PATCH_DATA(dword new_offset, byte new_oldbyte, byte new_newbyte) :
        offset(new_offset),
        oldbyte(new_oldbyte),
        newbyte(new_newbyte)
    {}
};

struct CRC_DATA
{
    string file;
    dword crc;
    enum {NO_CRC = -1, CRC32 = 0};
    int crct;
    int size; //-1 means no size

    CRC_DATA(const_ptr new_file, dword new_crc, int new_crct, int new_size) :
        file(new_file),
        crc(new_crc),
        crct(new_crct),
        size(new_size)
    {}
    CRC_DATA(CRC_DATA const& x)
    {
        file = x.file, crc = x.crc, crct = x.crct, size = x.size;
    }
    CRC_DATA &operator=(CRC_DATA const& x)
    {
        file = x.file, crc = x.crc, crct = x.crct, size = x.size;
        return *this;
    }
};

struct JIBZ_CMD
{
    int c;
    void *data;
    JIBZ_CMD() : c(-1), data(NULL) {} // vector ctor
    JIBZ_CMD(JIBZ_CMD const&); // push_back() support
    JIBZ_CMD(int cmd, void *dataptr) : c(cmd), data(dataptr) {}
    ~JIBZ_CMD();
};

struct FILE_DATA
{
    vector<PATCH_DATA> patch_data;
    vector<vector<int> > cont_src, cont_dst;
    bool q;
    vector<JIBZ_CMD> cmd;
    string file;

    FILE_DATA() : file(glob::NO_FILE)   { q = false; }
    FILE_DATA(FILE_DATA const &x)
    {
        patch_data = x.patch_data;
        cont_src = x.cont_src, cont_dst = x.cont_dst;
        q = x.q;
        cmd = x.cmd;
        file = x.file;
    }
    FILE_DATA &operator=(FILE_DATA const &x)
    {
        patch_data = x.patch_data;
        cont_src = x.cont_src, cont_dst = x.cont_dst;
        q = x.q;
        cmd = x.cmd;
        file = x.file;
        return *this;
    }
};

class Crx_parser
{
public:
    Crx_parser(string crx, byte *XI = 0) :
        file(crx),
        extended_info(XI)
    {
        linecount = filecount = 0;
        fuzzy = frk_here = sp_here = jibz_here = false;
        read_file();
        parse();
    }
    ~Crx_parser() {shutdown();}

    Z void init_regexp();

    vector<FILE_DATA> const& get_file_data() const
    {return file_data;}
    vector<CRC_DATA> const& get_crc_data() const
    {return crc_data;}

    Z void print_fuzzy_matrix(ostream& s, const_ptr prefix, vector<int> const& x, const_ptr suffix, char blank = '*');
    bool is_fuzzy() const {return fuzzy;}
    bool jibz_fmt() const {return jibz_here;}

    int get_filecount() const {return filecount;}
    int get_linecount() const {return linecount;}

public:

    Z int const NUM_PREF = 2;
    enum {PX_DSK = 0, PX_CHR};
    Z const_ptr PREFIX[NUM_PREF];

    Z dword const
        fVerbose = 1;
    Z dword flags;

    Z int const Jibz_Magic = 18;

    enum {
        jbGetkey = 0, jbPrint, jbCls, jbFile, jbFilecl, jbSize, jbCrc, jbBackup,
        jbCompare = 8, jbSearch, jbTest, jbReplace, jbWrite, jbGoto, jbGoto_plus, jbGoto_minus,
        jbExit = 16, jbEnd = 17
    };

    Z const_ptr Jibz_commands[Jibz_Magic];

private:
    void read_file();
    void parse();

    void shutdown() {}

    int make_fuzzy_matrix(char const *asc_src, char const *asc_dst, vector<int> &src, vector<int> &dst);
    int make_fuzzy_matrix_single(char const *asc_src, vector<int> &src);

    enum {FUZZY_NO_SYNC = -99};

    int Check_match(int r, int I, const_ptr s = 0)
    {
        return pcre_exec(pcre_data[r], pcre_extra_data[r],
                s? s : crx_data[I].c_str(),
                s? (int)strlen(s) : (int)crx_data[I].length(),
                0, 0,
                ovector, OVECCOUNT);
    }
    int Check_match(int r, vector<string>::iterator const& I)
    {
        return pcre_exec(pcre_data[r], pcre_extra_data[r],
                (*I).c_str(), (*I).length(),
                0, 0,
                ovector, OVECCOUNT);
    }

private:
    string file;
    byte *extended_info;
    vector<string> crx_data;
    vector<FILE_DATA> file_data;
    vector<CRC_DATA> crc_data;
    int linecount;
    int filecount;
    bool fuzzy, frk_here, sp_here, jibz_here;

    Z int const OVECCOUNT = 300;   /* should be a multiple of 3 */
    int ovector[OVECCOUNT];

    Z int const REG_COUNT = 11;
    enum {
        REG_CHECK_SIZE_CRC32 = 0, REG_CHECK_SIZE, REG_OFFSET_OLD_NEW,
        REG_SEARCH_FRK, REG_SEARCH_SP, REG_FUZZY_BLANK, REG_PLUS_FRK, REG_CHANGE_FRK,
        REG_JIBZ_FILE, REG_JIBZ_BLANK, REG_PATTERN
    };

    Z const_ptr my_regs[REG_COUNT];

    Z pcre* pcre_data[REG_COUNT];
    Z pcre_extra* pcre_extra_data[REG_COUNT];
};

class Patch
{
public:
    struct apply_error
    {
        Z string bad_file;
        Z dword offset;
        Z byte b;
        Z int fuzzy_blocks_patched;
        Z bool Jibz_exit;
    };

    enum {APPLY_OK = 0, APPLY_FILE_NOT_FOUND = 100, APPLY_SEEK_ERROR, APPLY_BAD_BYTE, APPLY_NOTHING_TODO,
        APPLY_WRITE_ERROR, APPLY_FUZZY_IO, APPLY_JIBZ_EXIT};

    enum {CHECKCRC_OK = 0, CHECKCRC_BADCRC = 200, CHECKCRC_IO,
        CHECKCRC_BAD_SIZE};

    Z int apply_patch(FILE_DATA const& fd, bool get_back, bool fuzzy, bool jibz, bool test_only);
    Z int check_crc(CRC_DATA const& crc);

private:

    Z int search_pattern(vector<int> const& src, char const *ptr, int size);
    Z int test_pattern(vector<int> const& src, char const *ptr, int size);
    Z int write_pattern(vector<int> const& src, char *ptr, int size);

    Z int apply_dummy_patch(FILE_DATA const& fd, bool get_back, bool test_only);
    Z int apply_fuzzy_patch(FILE_DATA const& fd, bool get_back, bool test_only);
    Z int apply_jibz_patch(FILE_DATA const& fd, bool get_back, bool test_only);

};

#undef Z

CRX_END_NAMESPACE

#endif //!CRX_H
