/* Copyright (c) 2002 Alex Derbeev (derbeev2@tut.by)
 * Portions (C)Eugene Roshal
 *
 * crx -- flexible crx/xck/etc apply utility
 *
 * You may distribute crx under the terms of either the GNU General Public
 * License or the Perl Artistic License.
 */

#include <iostream>
#include <fstream>
#if defined(__GNUC__) && __GNUC__ < 3
#include <strstream>
typedef std::ostrstream myostrstream;
#else
#include <sstream>
typedef std::basic_ostringstream<char> myostrstream;
#endif
#include <string>
#include <vector>
#include <memory>
#include <cstdio>
#include <cstdlib>
#include <cstring>
#include <cassert>
#include <cctype>
#include <csignal>
// Perl-compatible regular expressions.
// Copyright (c) 1997-2001 University of Cambridge.
// ftp://ftp.csx.cam.ac.uk/pub/software/programming/pcre/pcre-xxx.tar.gz
#include "pcre.h"

#define CRX // comment it out if you want to build CRX "library" :)

#include "crx.h"

CRX_BEGIN_NAMESPACE

#if defined(__GNUC__) && (__GNUC__ < 3)
#define IOS_NOCREATE ios::nocreate
#else
#define IOS_NOCREATE ((ios::openmode const)0)
#endif

#ifdef NDEBUG
#undef assert
#define assert(COND) if (!(COND)) throw ( Crx::CrxAssertException(#COND, __LINE__) )
#endif

#define OPEN_LOCAL_BLOCK {
#define CLOSE_LOCAL_BLOCK }

#if defined(JIBZ_CRC_INLINE)
#define JIBZCRC JibzCRC_inline
#elif defined(JIBZ_CRC)
#define JIBZCRC JibzCRC_asm
#else
#define JIBZCRC glob::JibzCRC
#endif

bool glob::o_deactivate = false;
bool glob::o_check_crc = true;
bool glob::o_fuzzy_test_only = false;
bool glob::o_quiet = false;
bool glob::o_verbose = false;
bool glob::o_abort = false;
bool glob::o_calc_crc = false;
bool glob::o_xcode = false;

int glob::Crc_type = ctCRC32;

const_ptr glob::NO_FILE = "*";
dword glob::CRCTab[256];

assignable_ostream *glob::mylog = (assignable_ostream*)&cout;

#define GLOB_LOG (*glob::mylog)

const_ptr glob::license = "See the GNU General Public License for the details\n";

char Sync::iobuf[IOBUFSZ];
bool Sync::called_once = true;
Sync glob::sync;

const_ptr Crx_parser::my_regs[REG_COUNT] =
{
    "\\[([^\\]]+)\\](\\s*)size(\\s*)\\:(\\s*)\\d+(\\s*)\\,(\\s*)crc32(\\s*)\\:(\\s*)[0-9a-f]{8}",
    "^(.{2})size(\\s*)\\:(\\s*)\\d+",
    "[0-9a-f]{8}(\\s*)\\:(\\s*)[0-9a-f]{2}(\\s*)[0-9a-f]{2}",
    "^\\<(\\s([0-9a-f]|\\*|\\?|\\-|x){2})+",
    "\\#search(\\s([0-9a-f]|\\*|\\?|\\-|x){2})+",
    "^(\\s*)(\\;(.*))?$",
    "^\\+(\\s([0-9a-f]|\\*|\\?|\\-|x){2})+",
    "^\\>(\\s([0-9a-f]|\\*|\\?|\\-|x){2})+",
    "^(\\^file\\^(\\s+)\\\"([^\\\"]+)\\\")|(\\^filecl\\^((\\s+)\\\"([^\\\"]+)\\\")?)",
    "^((\\^(getkey|print|cls|file|filecl|size|crc|backup|compare|" \
    "search|test|replace|write|goto|(goto\\+)|(goto\\-)|exit|end)\\^))" /* {0} */,
    "(\\s([0-9a-f]|\\*|\\?|\\-|x){2})+"
};
pcre* Crx_parser::pcre_data[REG_COUNT];
pcre_extra* Crx_parser::pcre_extra_data[REG_COUNT];

const_ptr Crx_parser::PREFIX[NUM_PREF] = {"Dsk://", "Chr://"};
dword Crx_parser::flags = 0;

const_ptr Crx_parser::Jibz_commands[Jibz_Magic] = {
    "getkey", "print", "cls", "file", "filecl", "size", "crc", "backup",
    "compare", "search", "test", "replace", "write", "goto", "goto+", "goto-",
    "exit", "end"
};

int     Patch::apply_error::fuzzy_blocks_patched = 0;
string  Patch::apply_error::bad_file = "";
dword   Patch::apply_error::offset = 0;
byte    Patch::apply_error::b = '\0';
bool    Patch::apply_error::Jibz_exit = false;

void glob::InitCRC()
{
    int I, J;
    dword C;
    for (I = 0; I < 256; ++I)
    {
        for (C = I, J = 0; J < 8; ++J)
            C = (C & 1) ? (C >> 1) ^ 0xEDB88320L : (C >> 1);
        CRCTab[I] = C;
    }
}

dword glob::CalcCRC32(dword StartCRC, byte const *Addr, int Size)
{
    for (int I = 0; I < Size; ++I)
        StartCRC = CRCTab[(byte) StartCRC ^ Addr[I]] ^ (StartCRC >> 8);
    return StartCRC;
}

#if !defined(JIBZ_CRC)
word glob::JibzCRC(word StartCRC, byte const *buf, int buflen)
{
    word crc16 = StartCRC;

    for (int k = 0; k < buflen; k++)
    {
#if 0
        word tt = crc16;
        tt ^= (word)buf[k] << 8;
        crc16 &= 0xff;
        crc16 |= tt & 0xff00;
#else
#if defined(LITTLE_ENDIAN)
        ((byte*)&crc16)[1] ^= buf[k];
#elif defined(BIG_ENDIAN)
        ((byte*)&crc16)[0] ^= buf[k];
#endif
#endif
        for (int j = 0; j < 8; j++)
        {
            dword t = (dword)crc16+crc16;
            crc16 = (word) ((t <= 65535u)? t : t ^ 0x1021);
        }
    }
    return crc16;
}
#endif

int glob::CalculateCRC(const_ptr fileName, int ct, string& CRC)
{
    fstream f(fileName, ios::in | ios::binary | IOS_NOCREATE);
    if (!f)
        return CRC_IO;

    f.seekg(0, ios::end);
    int real_size = (int)f.tellg();
    assert(real_size > 0);
    auto_ptr<char> buf(new char[real_size]);

    OPEN_LOCAL_BLOCK
    Sync sync;
    flush(GLOB_LOG << "Reading ");
    if ((f.seekg(0).read(buf.get(), real_size)).fail())
        return CRC_IO;
    GLOB_LOG << "Ok\n";
    CLOSE_LOCAL_BLOCK

    f.close();

    char msg[256] = {""};

    switch (ct)
    {
        case ctCRC32:
            OPEN_LOCAL_BLOCK

            unsigned C = (unsigned)~CalcCRC32((dword)4294967295u, (byte const*)buf.get(), real_size);
            sprintf(msg, "CRC32=%08X", C);

            CLOSE_LOCAL_BLOCK
            break;
        case ctJibz:
            OPEN_LOCAL_BLOCK

            unsigned C = (unsigned)JIBZCRC((word)0, (byte const*)buf.get(), real_size);
            sprintf(msg, "JibzCRC16=%04X", C);

            CLOSE_LOCAL_BLOCK
            break;
        default:
            assert(0);
    }
    CRC = msg;
    return CRC_OK;
}

void glob::sanity_check()
{
    {
        assert(sizeof(byte) == 1);
        assert(sizeof(word) == 2);
        assert(sizeof(dword) == 4);
    }
    {
//      assert(0xdeadbeef == *(dword*)"\xef\xbe\xad\xde");
        assert(sizeof(char) == 1);
    }
    {
        const_ptr msg = "Beneath my wing it's gonna be alright";
        assert(0x4B5FF5C5 == ~CalcCRC32(4294967295u, (byte const*)msg, strlen(msg)));
        assert(0xACAA == JIBZCRC((word)0, (byte const*)msg, strlen(msg)));
    }
}

void glob::quit(char const *msg, int code)
{
    if (msg && *msg)
        (*mylog) << msg << '\n';

    if (!o_xcode)
    {
        switch (code)
        {
            case 0xdead:            // all ok
                code = 0; break;
            case 1:                 // some exception
            case 3:                 // abort()
            case 4:                 // bad format
            case 7:                 // bad option
                code = 1; break;
            case 2:                 // help printed
            case 5:                 // revision printed
            case 6:                 // license printed
                code = 0; break;
            case 100:               // file not found
            case 101:               // seek error
            case 102:               // bad byte
            case 103:               // nothing to do
            case 104:               // write error
            case 105:               // fuzzy mode IO error
                code = 2; break;
            case 200:               // bad crc
            case 201:               // crc IO error
            case 202:               // bad size
                code = 3; break;
            default:
                assert(0);
        }
    }

    exit(code);
}

void glob::usage()
{
    quit("crx [-dctaqvsxrL] crxfile", RC_USAGE);
}

void glob::myabort(int)
{
    quit("abort()", RC_ABORT);
}

void glob::init_task()
{
    static int called_once = 0;
    assert(++called_once == 1);
    InitCRC();
    Crx_parser::init_regexp();
    setbuf(stdout, NULL);
    signal(SIGABRT, myabort);
    sanity_check();
}

void Crx_parser::init_regexp()
{
    char const *error;
    int err_offset;

    for (int i = 0; i < REG_COUNT; i++)
    {
        pcre_data[i] = pcre_compile(my_regs[i], PCRE_CASELESS, &error, &err_offset, NULL);
        assert(pcre_data[i] != NULL);
        pcre_extra_data[i] = pcre_study(pcre_data[i], 0, &error);
    }
}

void Crx_parser::read_file()
{
    int const PL[NUM_PREF] = {strlen(PREFIX[0]), strlen(PREFIX[1])};

    if ((int)file.size() > PL[PX_DSK] && string(file, 0, PL[PX_DSK]) == PREFIX[PX_DSK])
    {
        string s;
        fstream crx_file(file.c_str()+PL[PX_DSK], ios::in | IOS_NOCREATE);
        while (getline(crx_file, s).good())
        {
            crx_data.push_back(s);
            linecount++;
        }
    }
    else if ((int)file.size() > PL[PX_CHR] && string(file, 0, PL[PX_CHR]) == PREFIX[PX_CHR])
    {
        file.erase(0, PL[PX_CHR]);
        char *buf = new char[file.size()+1];
        memcpy(buf, file.c_str(), file.size()+1);
        for (char *tok = strtok(buf, "\n"); tok; tok = strtok(0, "\n"))
        {
            crx_data.push_back(string(tok));
            linecount++;
        }
        delete[] buf;
    }
    else assert(0);
}

int Crx_parser::make_fuzzy_matrix(char const *asc_src, char const *asc_dst, vector<int> &src, vector<int> &dst)
{
// src example: "DE AD ** EF", no leading or trailing whitespace
    char const *end_src = asc_src + strlen(asc_src), *endp_src = asc_src;
    char const *end_dst = asc_dst + strlen(asc_dst), *endp_dst = asc_dst;
    int count = 0;

    char const *spc_src = strchr(asc_src, ' ');
    char const *spc_dst = strchr(asc_dst, ' ');
    if (spc_src && spc_dst && (spc_src-asc_src != spc_dst-asc_dst))
        return FUZZY_NO_SYNC;

    while (endp_src < end_src && endp_dst < end_dst)
    {
        int int_src = (int)strtol(asc_src, const_cast<char**>(&endp_src), 16);
        int int_dst = (int)strtol(asc_dst, const_cast<char**>(&endp_dst), 16);
        if (endp_src == asc_src)
            int_src = EOF, endp_src += 2;
        if (endp_dst == asc_dst)
            int_dst = EOF, endp_dst += 2;
        src.push_back(int_src);
        dst.push_back(int_dst);
        count++;
        asc_src = ++endp_src, asc_dst = ++endp_dst;
    }

    return count;
}

int Crx_parser::make_fuzzy_matrix_single(char const *asc_src, vector<int> &src)
{
    char const *end_src = asc_src + strlen(asc_src), *endp_src = asc_src;
    int count = 0;
    while (endp_src < end_src)
    {
        int int_src = (int)strtol(asc_src, const_cast<char**>(&endp_src), 16);
        if (endp_src == asc_src)
            int_src = EOF, endp_src += 2;
        src.push_back(int_src);
        count++;
        asc_src = ++endp_src;
    }
    return count;
}

void Crx_parser::print_fuzzy_matrix(ostream& s, const_ptr prefix, vector<int> const& x, const_ptr suffix, char blank)
{
    s << prefix;
    for (int i = 0; i < (int)x.size(); i++)
    {
        s << ' ';
        if (x[i] == EOF)
            s << blank << blank;
        else
        {
            s.width(2);
            s << hex << x[i] << dec;
        }
    }
    s << suffix;
}

JIBZ_CMD::~JIBZ_CMD()
{
    switch (c)
    {
        case Crx_parser::jbBackup:
        case Crx_parser::jbCompare:
            delete (string*)data;
            break;
        case Crx_parser::jbSearch:
        case Crx_parser::jbTest:
        case Crx_parser::jbWrite:
        case Crx_parser::jbReplace:
            delete (vector<int>*)data;
            break;
        case Crx_parser::jbSize:
        case Crx_parser::jbCrc:
        case Crx_parser::jbGoto:
        case Crx_parser::jbGoto_plus:
        case Crx_parser::jbGoto_minus:
            delete (int*)data;
            break;
    }
}

JIBZ_CMD::JIBZ_CMD(JIBZ_CMD const& x)
{
    switch (c = x.c)
    {
        case Crx_parser::jbBackup:
        case Crx_parser::jbCompare:
            data = new string(*(string*)x.data);
            break;
        case Crx_parser::jbSearch:
        case Crx_parser::jbTest:
        case Crx_parser::jbWrite:
        case Crx_parser::jbReplace:
            data = new vector<int>(*(vector<int>*)x.data);
            break;
        case Crx_parser::jbSize:
        case Crx_parser::jbCrc:
        case Crx_parser::jbGoto:
        case Crx_parser::jbGoto_plus:
        case Crx_parser::jbGoto_minus:
            data = new int(*(int*)x.data);
            break;
    }
}

void Crx_parser::parse()
{
    FILE_DATA fd;
    string cur_filename = glob::NO_FILE;

    for (int i = 0, rc; i < linecount; i++)
    {
        if ((rc = Check_match(REG_SEARCH_FRK, i)) > 0)
        {
            fuzzy = frk_here = true;
            break;
        }
        if ((rc = Check_match(REG_SEARCH_SP, i)) > 0)
        {
            fuzzy = sp_here = true;
            break;
        }
        if ((rc = Check_match(REG_JIBZ_FILE, i)) > 0)
        {
            fuzzy = jibz_here = true;
            break;
        }
    }

    if (fuzzy)
    {
        GLOB_LOG << "fuzzy " << (frk_here? "(FRK FMT)" : (jibz_here? "(JIBZ FMT)" : "(SP FMT)")) << '\n';

        for (vector<string>::iterator i = crx_data.begin(); i < crx_data.end(); )
        {
            int rc = Check_match(jibz_here? REG_JIBZ_BLANK : REG_FUZZY_BLANK, i);
            bool match = jibz_here? (rc <= 0) : (rc > 0);
            if (match)
                i = crx_data.erase(i);
            else i++;
        }
        linecount = crx_data.size();

        if (jibz_here)
        {
            vector<JIBZ_CMD> JC;
            bool Q = false;
            bool completed = true;
            bool file_pending = true;
            bool virgin = true;

            if (glob::o_verbose)
            {
                //copy(crx_data.begin(), crx_data.end(), ostream_iterator<string>(GLOB_LOG, "\n"));
            }

            for (int J = 0; J < linecount; J++)
            {
                if (Check_match(REG_JIBZ_BLANK, J) > 0)
                {
                    string C(crx_data[J].c_str()+ovector[0]+1, crx_data[J].c_str()+ovector[1]-1);

                    for (int k = 0; k < (int)C.size(); C[k] = (char)tolower(C[k]), k++) ;

                    int Cindex = -1;
                    for (int k = 0; k < Jibz_Magic; k++)
                        if (C == Jibz_commands[k])
                            Cindex = k;
                    assert(Cindex != -1);

                    if (file_pending)
                    {
                        if (Cindex != jbGetkey && Cindex != jbCls && Cindex != jbPrint)
                        {
                            if (Cindex != jbFile && Cindex != jbFilecl)
                                continue;
                            file_pending = false;
                        }
                        else if (!virgin)
                            continue;
                    }

                    string Cdata(crx_data[J].c_str()+ovector[1], crx_data[J].c_str()+crx_data[J].size());
                    char const *src = Cdata.c_str();
                    while (*src && isspace(*src))
                        src++;

                    switch (Cindex)
                    {
                        case jbPrint:
                            JC.push_back(JIBZ_CMD(Cindex, NULL));
                            break;
                        case jbFile: case jbBackup: case jbCompare:
                            OPEN_LOCAL_BLOCK

                            assert(strlen(src) > 0);
                            assert(src[0] == '"');
                            char const *closing = strchr(src+1, '"');
                            if (closing == 0)
                            {
                                closing = src+strlen(src);
                                while ( isspace(*(closing-1)) )
                                    closing--;
                            }
                            string *s = new string(src+1, closing);
                            if (Cindex == jbFile)
                            {
                                if (filecount > 0)
                                {
                                    fd.q = Q, Q = false;
                                    fd.cmd = JC;
                                    JC.clear();
                                    file_data.push_back(fd);
                                }
                                fd.file = *s, delete s, completed = false;
                                filecount++;
                            }
                            else JC.push_back(JIBZ_CMD(Cindex, s));

                            CLOSE_LOCAL_BLOCK
                            break;
                        case jbFilecl:
                            assert(0);
                            break;
                        case jbSearch: case jbTest: case jbReplace: case jbWrite:
                            OPEN_LOCAL_BLOCK

                            assert(strlen(src) > 0);
                            if (Cindex == jbReplace)
                                assert(isxdigit(*src));
                            int rc = Check_match(REG_PATTERN, 0, --src);
                            assert(rc > 0);
                            string asc_pattern(src+ovector[0]+1, src+ovector[1]);
                            vector<int> *pattern = new vector<int>();
                            make_fuzzy_matrix_single(asc_pattern.c_str(), *pattern);
                            JC.push_back(JIBZ_CMD(Cindex, pattern));

                            CLOSE_LOCAL_BLOCK
                            break;
                        case jbSize:    // fall through
                        case jbCrc:     // fall through
                        case jbGoto: case jbGoto_plus: case jbGoto_minus:
                            OPEN_LOCAL_BLOCK

                            assert(strlen(src) > 0);
                            assert(isxdigit(*src));
                            char *endptr;
                            int x = (int)strtol(src, &endptr, 16);
                            assert(endptr > src);
                            JC.push_back(JIBZ_CMD(Cindex, new int(x)));

                            CLOSE_LOCAL_BLOCK
                            break;
                        case jbGetkey:   // fall through
                        case jbCls:      // fall through
                        case jbExit: case jbEnd:
                            JC.push_back(JIBZ_CMD(Cindex, NULL));
                            if (Cindex == jbExit)
                                Q = file_pending = true, virgin = false;
                            if (Cindex == jbExit || Cindex == jbEnd)
                            {
                                if (filecount > 0)
                                {
                                    fd.q = Q, Q = false;
                                    fd.cmd = JC;
                                    JC.clear();
                                    file_data.push_back(fd);
                                    completed = true;
                                }
                            }
                            if (Cindex == jbEnd)
                                goto QUIT_JIBZ;
                            break;
                        default:
                            assert(0);
                    }
                }
            }
        QUIT_JIBZ:
            if (!completed)
            {
                fd.q = Q;
                fd.cmd = JC;
                file_data.push_back(fd);
            }
        }
        else if (frk_here)
        {
            vector<vector<int> > cont_src, cont_dst;

            for (int i = 0, rc; i < linecount; )
            {
                if ((rc = Check_match(REG_SEARCH_FRK, i)) > 0)
                {
                    assert(i < linecount-2);
                    vector<int> src, dst;

                    int X1 = ovector[0], /*X2 = ovector[1]-1,*/ J = i;

                    string str_src(crx_data[J].c_str()+X1+2, crx_data[J].c_str()+ovector[1]);

                    {
                        for (J++; J < linecount && (rc = Check_match(REG_PLUS_FRK, J)) > 0; J++)
                            str_src += ' ' + string(crx_data[J].c_str()+ovector[0]+2, crx_data[J].c_str()+ovector[1]);
                        assert(J < linecount);
                    }

                    rc = Check_match(REG_CHANGE_FRK, J);
                    assert(rc > 0 && ovector[0] == X1);

                    string str_dst(crx_data[J].c_str()+X1+2, crx_data[J].c_str()+ovector[1]);

                    {
                        for (J++; J < linecount && (rc = Check_match(REG_PLUS_FRK, J)) > 0; J++)
                            str_dst += ' ' + string(crx_data[J].c_str()+ovector[0]+2, crx_data[J].c_str()+ovector[1]);
                        assert(J < linecount);
                    }

                    int r = make_fuzzy_matrix(str_src.c_str(), str_dst.c_str(), src, dst);
                    assert(r != FUZZY_NO_SYNC && r > 0);

                    if ((flags & fVerbose) != 0)
                    {
                        print_fuzzy_matrix(GLOB_LOG, "<", src, "\n");
                        print_fuzzy_matrix(GLOB_LOG, ">", dst, "\n");
                    }

                    cont_src.push_back(src);
                    cont_dst.push_back(dst);

                    if (crx_data[J][X1] == '=')
                    {
                        fd.file = crx_data[J].c_str()+2;
                        fd.cont_src = cont_src, fd.cont_dst = cont_dst;
                        file_data.push_back(fd);
                        cont_src.clear();
                        cont_dst.clear();
                        filecount++;
                        i = J+1;
                    }
                    else i = J;
                }
                else i++;
            }
        }
        else if (sp_here)
        {
            glob::quit("SP format's not supported yet ;(", glob::RC_BAD_FMT);
        }
        else assert(0);

        return;
    } // fuzzy

    bool inside_patch_data = false;

    for (int i = 0, rc; i < linecount; i++)
    {
        if ((rc = Check_match(REG_OFFSET_OLD_NEW, i)) > 0)
        {
            if (!inside_patch_data && i > 0)
            {
                cur_filename = crx_data[i-1];
                filecount++;
                fd.file = cur_filename;
                file_data.push_back(fd);
            }
            assert(cur_filename != glob::NO_FILE);

            if (crc_data.size() == 1 && filecount == 1 && crc_data[0].crct == CRC_DATA::NO_CRC && crc_data[0].file == glob::NO_FILE)
                crc_data[0].file = fd.file; /* file name for size checking in xck */

            unsigned v[3];
            sscanf(crx_data[i].c_str()+ovector[0], "%08x :%02x%02x", &v[0], &v[1], &v[2]);
            dword offset = (dword)v[0];
            byte oldbyte = (byte)v[1], newbyte = (byte)v[2];

            file_data[filecount-1].patch_data.push_back(PATCH_DATA(offset, oldbyte, newbyte));

            if (file_data[filecount-1].patch_data.size() == 1)
                GLOB_LOG << "found byte patch section for `" << cur_filename << "'\n";
            inside_patch_data = true;
        }
        else if ((rc = Check_match(REG_CHECK_SIZE, i)) > 0)
        {
            GLOB_LOG << "found weak check section\n";
            char scanbuf[256];
            int Size;
            strcpy(scanbuf, crx_data[i].c_str()+ovector[0]);
            char *ptr = scanbuf;
            while (*++ptr)
                *ptr = (char)tolower(*ptr);
            sscanf(scanbuf, "%*c size :%d", &Size);

            crc_data.push_back(CRC_DATA(glob::NO_FILE, 0, CRC_DATA::NO_CRC, Size));

            inside_patch_data = false;
        }
        else if ((rc = Check_match(REG_CHECK_SIZE_CRC32, i)) > 0)
        {
            GLOB_LOG << "found extended check section\n";
            char scanbuf[256];
            char buf[256];
            int Size;
            unsigned CRC32;
            strcpy(scanbuf, crx_data[i].c_str()+ovector[0]);
            char *ptr = strchr(scanbuf, ']');
            while (*++ptr)
                *ptr = (char)tolower(*ptr);
            sscanf(scanbuf, "[%s size :%d , crc32 :%08x", buf, &Size, &CRC32);
            assert(strlen(buf) > 0);
            buf[strlen(buf)-1] = '\0'; //destroy closing bracket

            crc_data.push_back(CRC_DATA(buf, (dword)CRC32, CRC_DATA::CRC32, Size));

            inside_patch_data = false;
        }
        else inside_patch_data = false;
    }
}

int Patch::test_pattern(vector<int> const& src, char const *ptr, int size)
{
    int bytes = src.size();
    assert(bytes > 0);

    if (bytes > size)
        return -1;

    bool match = true;

    for (int J = 0; J < bytes; J++)
    {
        if (src[J] != EOF && (byte)src[J] != (byte)ptr[J])
        {
            match = false;
            break;
        }
    }

    return match? 0 : -1;
}

int Patch::write_pattern(vector<int> const& src, char *ptr, int size)
{
    int bytes = src.size();
    assert(bytes > 0);

    if (bytes > size)
        return -1;

    int bytes_altered = 0;

    for (int J = 0; J < bytes; J++)
    {
        if (src[J] != EOF)
        {
            ptr[J] = (byte)src[J];
            bytes_altered++;
        }
    }

    return bytes_altered;
}

int Patch::search_pattern(vector<int> const& src, char const *ptr, int size)
{
    int bytes = src.size();
    int sequence = 0;

    assert(bytes > 0);

    if (bytes > size)
        return -1;

    int sig = -1;
    for (int i = 0; i < bytes; i++)
        if (src[i] != EOF)
        {
            sig = i;
            break;
        }
    assert(sig != -1);

    char const *bufptr = ptr;

    while (size-(ptr-bufptr) > 0 && (ptr = (char*)memchr(ptr, src[sig], size-(ptr-bufptr))) != NULL)
    {
        int chunk_size = size-(ptr-bufptr);
        if (chunk_size+sig >= bytes && ptr-bufptr >= sig)
        {
            char const *susp_block = ptr-sig;
            bool found = true;
            for (int J = sig+1; J < bytes; J++)
            {
                if (src[J] != EOF && (byte)src[J] != (byte)susp_block[J])
                {
                    found = false;
                    break;
                }
            }
            if (found)
            {
                sequence++;
                return ptr-bufptr;
//              ptr += bytes-1;
            }
        }
        ptr++;
    }
    return -1;
}

int Patch::apply_jibz_patch(FILE_DATA const& fd, bool get_back, bool test_only)
{
    GLOB_LOG << "patching file `" << fd.file << "' (Jibz mode)...\n";

    if (get_back)
        glob::quit("Can't reverse Jibz script yet :)", glob::RC_EXCEPT);

    bool Q = fd.q;
    vector<JIBZ_CMD> cmd = fd.cmd;
    assert(cmd.size() > 0);

    bool patched = false;
    vector<int> seq_ovec, seq_lvec;
    int blocks_patched = 0;
    bool good = true;

    fstream f(fd.file.c_str(), ios::in | ios::out | ios::binary | IOS_NOCREATE);
    if (!f)
        return APPLY_FILE_NOT_FOUND;

    f.seekg(0, ios::end);
    int real_size = (int)f.tellg();
    assert(real_size > 0);
    auto_ptr<char> buf(new char[real_size]);

    OPEN_LOCAL_BLOCK
    Sync sync;
    flush(GLOB_LOG << "Reading ");
    if ((f.seekg(0).read(buf.get(), real_size)).fail())
        return APPLY_FUZZY_IO;
    GLOB_LOG << "Ok\n";
    CLOSE_LOCAL_BLOCK

    GLOB_LOG << "Open Jibz console(TM)\n";

    OPEN_LOCAL_BLOCK // apply jibz patch

    byte *B = (byte*)buf.get();
    int myptr = 0;

    for (int Command = 0; good && Command < (int)cmd.size(); Command++)
    {
        void *data = cmd[Command].data;

        if (glob::o_verbose)
        {
            char const *prefix = 0;
            switch (cmd[Command].c)
            {
                case Crx_parser::jbSearch:      prefix = "<"; break;
                case Crx_parser::jbTest:        prefix = "!"; break;
                case Crx_parser::jbWrite:       prefix = ">"; break;
            }
            if (prefix)
                Crx_parser::print_fuzzy_matrix(GLOB_LOG, prefix, *(vector<int>*)data, "\n");

            flush(GLOB_LOG << "Running " << Crx_parser::Jibz_commands[cmd[Command].c]);
        }

        switch (cmd[Command].c)
        {
            case Crx_parser::jbGetkey:
            case Crx_parser::jbCls:
                break;
            case Crx_parser::jbPrint:
                break;
            case Crx_parser::jbSize:
                good = *(int*)data == real_size;
                break;
            case Crx_parser::jbCrc:
                OPEN_LOCAL_BLOCK

                word crc16 = JIBZCRC((word)0, B, real_size);

                if (glob::o_verbose)
                    flush(GLOB_LOG << " [ crc16=" << hex << crc16 << " (" << ((word)*(int*)data) << dec << ") ]");

                good = (word)*(int*)data == crc16;

                CLOSE_LOCAL_BLOCK
                break;
            case Crx_parser::jbBackup:
                OPEN_LOCAL_BLOCK

                if (!test_only)
                {
                    assert(!patched);
                    fstream baq(((string*)data)->c_str(), ios::out | ios::binary);
                    good = baq.write((char const*)B, real_size).good();
                }

                CLOSE_LOCAL_BLOCK
                break;
            case Crx_parser::jbCompare:
                good = false;
                break;
            case Crx_parser::jbSearch:
                OPEN_LOCAL_BLOCK

                vector<int> &x = *(vector<int>*)data;
                assert(x.size() > 0);

                if (myptr < 0 || myptr >= real_size || (int)x.size() > real_size-myptr)
                {
                    good = false;
                    break;
                }

                int r = search_pattern(x, (char const*)(B+myptr), real_size-myptr);
                if (r == -1)
                    good = false;
                else myptr += r;

                CLOSE_LOCAL_BLOCK
                break;
            case Crx_parser::jbTest:
                OPEN_LOCAL_BLOCK

                vector<int> &x = *(vector<int>*)data;
                assert(x.size() > 0);

                if (myptr < 0 || myptr >= real_size || (int)x.size() > real_size-myptr)
                {
                    good = false;
                    break;
                }

                int r = test_pattern(x, (char const*)(B+myptr), real_size-myptr);
                good = r != -1;

                CLOSE_LOCAL_BLOCK
                break;
            case Crx_parser::jbWrite:
                OPEN_LOCAL_BLOCK

                vector<int> &x = *(vector<int>*)data;
                assert(x.size() > 0);

                if (myptr < 0 || myptr >= real_size || (int)x.size() > real_size-myptr)
                {
                    good = false;
                    break;
                }

                int r = write_pattern(x, (char*)(B+myptr), real_size-myptr);
                if (r == -1)
                    good = false;
                else
                {
                    if (r > 0)
                    {
                        if (!test_only)
                            patched = true;
                        blocks_patched++;
                        seq_ovec.push_back(myptr);
                        seq_lvec.push_back(x.size());
                    }
                    myptr += x.size();
                }

                CLOSE_LOCAL_BLOCK
                break;
            case Crx_parser::jbReplace:
                OPEN_LOCAL_BLOCK

                vector<int> &x = *(vector<int>*)data;
                assert(x.size() == 2);
                assert(x[0] != EOF && x[1] != EOF);

                if (myptr < 0 || myptr >= real_size)
                {
                    good = false;
                    break;
                }

                if (B[myptr] == (byte)x[0])
                {
                    B[myptr] = (byte)x[1];
                    if (!test_only)
                        patched = true;
                    blocks_patched++;
                    seq_ovec.push_back(myptr);
                    seq_lvec.push_back(1);
                    myptr++;
                }
                else good = false;

                CLOSE_LOCAL_BLOCK
                break;
            case Crx_parser::jbGoto:
                myptr = *(int*)data;
                if (myptr < 0 || myptr >= real_size)
                    good = false;
                break;
            case Crx_parser::jbGoto_plus:
                myptr += *(int*)data;
                if (myptr < 0 || myptr >= real_size)
                    good = false;
                break;
            case Crx_parser::jbGoto_minus:
                myptr -= *(int*)data;
                if (myptr < 0 || myptr >= real_size)
                    good = false;
                break;
            case Crx_parser::jbExit: case Crx_parser::jbEnd:
                break;
            default:
                assert(0);
        }

        if (glob::o_verbose)
        {
            //if (!good) GLOB_LOG << " - error ;(\n";
            GLOB_LOG << " - " << (good? "Ok" : "error ;(") << '\n';
        }

    }
    CLOSE_LOCAL_BLOCK

    GLOB_LOG << "Close Jibz console(TM)\n";

    OPEN_LOCAL_BLOCK
    Sync sync;
    if (patched)
    {
        flush(GLOB_LOG << "Writing ");
        int write_offset = 0;
        int write_size = real_size;
        if (blocks_patched == 1)
        {
            write_offset = seq_ovec[0];
            write_size = seq_lvec[0];
        }
        flush(f);
        if ((f.seekp(write_offset).write(buf.get()+write_offset, write_size)).fail())
            return APPLY_FUZZY_IO;
        GLOB_LOG << "Ok\n";
    }
    CLOSE_LOCAL_BLOCK

    GLOB_LOG << (patched? "done" : "nothing to do ;(") << '\n';

    apply_error::fuzzy_blocks_patched = blocks_patched;
    apply_error::Jibz_exit = Q && good;

    return APPLY_OK;
}

int Patch::apply_fuzzy_patch(FILE_DATA const& fd, bool get_back, bool test_only)
{
    GLOB_LOG << "patching file `" << fd.file << "' (SP/FRK mode)...\n";

    bool patched = false;
    vector<int> seq_ovec, seq_lvec;
    int blocks_patched = 0;

    int fuzzy_blocks = fd.cont_src.size();
    assert(fuzzy_blocks > 0);

    fstream f(fd.file.c_str(), ios::in | ios::out | ios::binary | IOS_NOCREATE);
    if (!f)
        return APPLY_FILE_NOT_FOUND;

    f.seekg(0, ios::end);
    int real_size = (int)f.tellg();
    assert(real_size > 0);
    auto_ptr<char> buf(new char[real_size]);

    OPEN_LOCAL_BLOCK
    Sync sync;
    flush(GLOB_LOG << "Reading ");
    if ((f.seekg(0).read(buf.get(), real_size)).fail())
        return APPLY_FUZZY_IO;
    GLOB_LOG << "Ok\n";
    CLOSE_LOCAL_BLOCK

    GLOB_LOG << "Searching\n";

    OPEN_LOCAL_BLOCK // apply fuzzy patch

    for (int B = 0; B < fuzzy_blocks; B++)
    {
        vector<int> src = fd.cont_src[B], dst = fd.cont_dst[B];
        assert(src.size() == dst.size());
        int bytes = src.size();
        int sequence = 0;

        assert(bytes > 0);

        if (get_back)
        {
            for (int k = 0; k < bytes; k++)
                if (dst[k] != EOF)
                    swap(src[k], dst[k]);
        }

        int sig = -1;
        for (int i = 0; i < bytes; i++)
            if (src[i] != EOF)
            {
                sig = i;
                break;
            }
        assert(sig != -1);

        char *ptr = buf.get(), *bufptr = ptr;

        while (real_size-(ptr-bufptr) > 0 && (ptr = (char*)memchr(ptr, src[sig], real_size-(ptr-bufptr))) != NULL)
        {
            int chunk_size = real_size-(ptr-bufptr);
            if (chunk_size+sig >= bytes && ptr-bufptr >= sig)
            {
                char *susp_block = ptr-sig;
                bool found = true;
                for (int J = sig+1; J < bytes; J++)
                {
                    if (src[J] != EOF && (byte)src[J] != (byte)susp_block[J])
                    {
                        found = false;
                        break;
                    }
                }
                if (found)
                {
                    if (!test_only)
                    {
                        bool altered = false;
                        for (int J = 0; J < bytes; J++)
                        {
                            if (dst[J] != EOF)
                            {
                                susp_block[J] = (byte)dst[J];
                                altered = true;
                            }
                        }
                        if (altered)
                        {
                            blocks_patched++;
                            seq_ovec.push_back(susp_block-bufptr);
                            seq_lvec.push_back(bytes);
                            patched = true;
                        }
                    }
                    sequence++;
                    GLOB_LOG << "found matching sequence N" << sequence
                        << " at 0x" << hex << (susp_block-bufptr) << dec
                        << " (fuzzy block " << (B+1) << ")"
                        << ", patched :)\n";
                    ptr += bytes-1;
                }
            }
            ptr++;
        }
    }
    CLOSE_LOCAL_BLOCK

    OPEN_LOCAL_BLOCK
    Sync sync;
    if (patched)
    {
        flush(GLOB_LOG << "Writing ");
        int write_offset = 0;
        int write_size = real_size;
        if (blocks_patched == 1)
        {
            write_offset = seq_ovec[0];
            write_size = seq_lvec[0];
        }
        flush(f);
        if ((f.seekp(write_offset).write(buf.get()+write_offset, write_size)).fail())
            return APPLY_FUZZY_IO;
        GLOB_LOG << "Ok\n";
    }
    CLOSE_LOCAL_BLOCK

    GLOB_LOG << (patched? "done" : "nothing to do ;(") << '\n';

    apply_error::fuzzy_blocks_patched = blocks_patched;

    return APPLY_OK;
}

int Patch::apply_dummy_patch(FILE_DATA const& fd, bool get_back, bool test_only)
{
    GLOB_LOG << "patching file `" << fd.file << "' (dummy mode)...\n";

    int bytecount = fd.patch_data.size();
    if (!bytecount)
        return APPLY_NOTHING_TODO;

    fstream f(fd.file.c_str(), ios::in | ios::out | ios::binary | IOS_NOCREATE);
    if (!f)
        return APPLY_FILE_NOT_FOUND;

    for (int i = 0; i < bytecount; i++)
    {
        dword offset = fd.patch_data[i].offset;
        byte oldbyte = fd.patch_data[i].oldbyte;
        byte newbyte = fd.patch_data[i].newbyte;
        if (get_back)
            swap(oldbyte, newbyte);
        char msg[50];
        sprintf(msg, "[%#08x: %#02x->%#02x]___", offset, oldbyte, newbyte);
        flush(GLOB_LOG << msg);
        apply_error::offset = offset;

        OPEN_LOCAL_BLOCK

        Sync sync;
        if ((f.seekg(offset)).fail())
            return APPLY_SEEK_ERROR;
        int b = f.peek();
#if (defined(__GNUC__) && (__GNUC__ < 3))
        if (b == EOF) // silly seek bug workaround
            return APPLY_SEEK_ERROR;
#endif
        if ((byte)b != oldbyte)
        {
            apply_error::b = (byte)b;
            return APPLY_BAD_BYTE;
        }
        if (!test_only)
        {
            if ((f.put(newbyte)).fail())
                return APPLY_WRITE_ERROR;
            flush(f);
        }

        CLOSE_LOCAL_BLOCK

        GLOB_LOG << "ok\n";
    }
    GLOB_LOG << "done\n";
    return APPLY_OK;
}

int Patch::apply_patch(FILE_DATA const& fd, bool get_back, bool fuzzy, bool jibz, bool test_only)
{
    assert(fd.file != glob::NO_FILE);
    apply_error::bad_file = fd.file;

    int r = APPLY_OK;

    if (fuzzy)
    {
        if (jibz)
            r = apply_jibz_patch(fd, get_back, test_only);
        else r = apply_fuzzy_patch(fd, get_back, test_only);
    }
    else r = apply_dummy_patch(fd, get_back, test_only);

    return r;
}

int Patch::check_crc(CRC_DATA const& crc)
{
    assert(crc.crct == CRC_DATA::CRC32 || (crc.crct == CRC_DATA::NO_CRC && crc.size != -1));
    bool size_only = crc.crct == CRC_DATA::NO_CRC;
    GLOB_LOG << "checking " << (size_only? "size" : "crc") << " for `" << crc.file << "'...";
    flush(GLOB_LOG);
    fstream f(crc.file.c_str(), ios::in | ios::binary | IOS_NOCREATE);
    if (!f)
        return CHECKCRC_IO;
    f.seekg(0, ios::end);
    int real_size = (int)f.tellg();
    if (crc.size != -1 && crc.size != real_size)
        return CHECKCRC_BAD_SIZE;
    if (!size_only)
    {
        f.seekg(0);
        int const BLOCK_SIZE = 65536;
        static char buf[BLOCK_SIZE];
        int block_count = real_size / BLOCK_SIZE;
        int block_last_size = real_size % BLOCK_SIZE;
        dword real_crc = (dword)4294967295u;
/*      f.rdbuf()->setbuf(Sync::iobuf, Sync::IOBUFSZ); */

        OPEN_LOCAL_BLOCK
        Sync sync;
        for (int i = 0; i < block_count; i++)
        {
            if ((f.read(buf, BLOCK_SIZE)).fail())
                return CHECKCRC_IO;
            real_crc = glob::CalcCRC32(real_crc, (byte const*)buf, BLOCK_SIZE);
        }
        if ((f.read(buf, block_last_size)).fail())
            return CHECKCRC_IO;
        real_crc = ~glob::CalcCRC32(real_crc, (byte const*)buf, block_last_size);
        CLOSE_LOCAL_BLOCK

        if (real_crc != crc.crc)
            return CHECKCRC_BADCRC;
    }
    GLOB_LOG << "ok\n";
    return CHECKCRC_OK;
}

int glob::mygetopt(int argc, char **argv)
{
    if (argc == 1)
        usage();

    int a = 1;

    for (; a < argc && argv[a][0] == '-'; a++)
    {
        while (*++argv[a]) switch (*argv[a])
        {
            case 'd':
                o_deactivate = true;
                break;
            case 'c':
                o_check_crc = false;
                break;
            case 't':
                o_fuzzy_test_only = true;
                break;
            case 'a':
                o_abort = true;
                break;
            case 'q':
                o_quiet = true;
                mylog = (assignable_ostream*)new myostrstream();
                break;
            case 'v':
                o_verbose = true;
                Crx_parser::flags |= Crx_parser::fVerbose;
                break;
            case 's':
                OPEN_LOCAL_BLOCK

                if (a+2 < argc)
                {
                    string crct(argv[a+1]);
                    if (crct == "Jibz")
                        Crc_type = ctJibz;
                    else if (crct == "CRC32")
                        Crc_type = ctCRC32;
                    else usage();
                    string CRC;
                    int r = CalculateCRC(argv[a+2], Crc_type, CRC);
                    if (r != CRC_OK)
                        quit("Can't calculate CRC ;(", r);
                    GLOB_LOG << CRC << '\n';
                    quit(0, 0xdead);
                }
                else usage();

                CLOSE_LOCAL_BLOCK
                break;
            case 'x':
                o_xcode = true;
                break;
            case 'r':
                quit(PROG_NAME" v"PROG_VERSION" compiled at "__DATE__" "__TIME__" by "PROG_WHO_COMPILES, RC_REVISION);
            case 'L':
                quit(license, RC_LICENSE);
            case '?': case 'h':
                usage();
            case '-':
                a++;
                goto NIKLAUS;
            default:
                quit("bad option ;(", RC_BAD_OPTION);
        }
    }

NIKLAUS:

    if (a >= argc)
        usage();

    return a;
}

CRX_END_NAMESPACE

//#define CRX_TEST

#if defined( CRX_TEST )
#undef CRX

int main(int argc, char **argv)
{
    try
    {
        Crx::glob::init_task();

        Crx::glob::o_quiet = true, Crx::glob::mylog = (Crx::assignable_ostream*)new std::myostrstream();

        Crx::Crx_parser crx_parser( Crx::Crx_parser::PREFIX[Crx::Crx_parser::PX_CHR] + std::string(
            "< A1 xx xx xx xx 85 C0 75 xx 6A 00 E8 xx xx xx xx 83 C4 04 A8 07\n" \
            "> -- -- -- -- -- -- -- EB -- -- -- -- -- -- -- -- -- -- -- -- --\n" \
            "= HIEW32.EXE\n"
            ) );

        Crx::Patch::apply_error::fuzzy_blocks_patched = 0;

        int r = Crx::Patch::apply_patch(crx_parser.get_file_data()[0], Crx::glob::o_deactivate,
            crx_parser.is_fuzzy(), crx_parser.jibz_fmt(),
            Crx::glob::o_fuzzy_test_only);

        if (r != Crx::Patch::APPLY_OK || 1 != Crx::Patch::apply_error::fuzzy_blocks_patched)
        {
            std::cerr << "Not Ok ;(\n";
            return Crx::glob::RC_EXCEPT;
        }
    }
    catch (Crx::CrxAssertException const& cause)
    {
        std::cerr << "\n""assert exception: `" << cause.getMsg() << "', source line " << cause.getSrcLine() << "\n";
        Crx::glob::quit(0, Crx::glob::RC_EXCEPT);
    }
    catch (...)
    {
        Crx::glob::quit("\n""unhandled exception ;(", Crx::glob::RC_EXCEPT);
    }

    return 0xdead;
}

#endif //CRX_TEST


#if defined( CRX ) && !defined( CRX_LIB )

int crx_main(int argc, char **argv)
{
    using namespace Crx;

    glob::init_task();

    Crx_parser crx_parser( Crx_parser::PREFIX[Crx_parser::PX_DSK] + string(argv[glob::mygetopt(argc, argv)]) );

    assert(crx_parser.get_filecount() > 0);

    int crc_count = crx_parser.get_crc_data().size();
    char e[256] = {""};

    if (!glob::o_check_crc || glob::o_deactivate)
        crc_count = 0;

    for (int i = 0; i < crc_count; i++)
    {
        int r = Patch::check_crc(crx_parser.get_crc_data()[i]);

        if (r != Patch::CHECKCRC_OK)
        {
            switch (r)
            {
                case Patch::CHECKCRC_BADCRC:
                    strcpy(e, "bad crc");
                    break;
                case Patch::CHECKCRC_IO:
                    strcpy(e, "i/o error");
                    break;
                case Patch::CHECKCRC_BAD_SIZE:
                    strcpy(e, "bad size");
                    break;
            }

            GLOB_LOG << "\nerror(check_crc): " << e << '\n';
            glob::quit("sorry, bad crc ;(", r);
        }
    }

    int first_error = 1;
    bool everything_ok = true;

    for (int i = 0; i < crx_parser.get_filecount(); i++)
    {
        int r = Patch::apply_patch(crx_parser.get_file_data()[i], glob::o_deactivate,
            crx_parser.is_fuzzy(), crx_parser.jibz_fmt(),
            glob::o_fuzzy_test_only);

        if (Patch::apply_error::Jibz_exit)
            glob::quit("premature exit (Jibz mode)", 0xdead);

        if (r != Patch::APPLY_OK)
        {
            everything_ok = false;

            if (first_error == 1)
                first_error = r;

            switch (r)
            {
                case Patch::APPLY_FILE_NOT_FOUND:
                    strcpy(e, "file not found");
                    break;
                case Patch::APPLY_SEEK_ERROR:
                    sprintf(e, "seek error, offset=%#08x(%u)", Patch::apply_error::offset, Patch::apply_error::offset);
                    break;
                case Patch::APPLY_BAD_BYTE:
                    sprintf(e, "bad byte (%#02x) at offset %#08x(%u)", Patch::apply_error::b, Patch::apply_error::offset, Patch::apply_error::offset);
                    break;
                case Patch::APPLY_NOTHING_TODO:
                    strcpy(e, "nothing to do");
                    break;
                case Patch::APPLY_FUZZY_IO:
                    strcpy(e, "fuzzy patch i/o error");
                    break;
                case Patch::APPLY_WRITE_ERROR:
                    sprintf(e, "write error at offset %#08x(%u)", Patch::apply_error::offset, Patch::apply_error::offset);
                    break;
            }

            GLOB_LOG << "\nerror(apply_patch): " << e << '\n';
            if (glob::o_abort)
                glob::quit("premature exit ;(", r);
        }
    }

    return everything_ok? 0xdead : first_error; // 173 means ok ;)
}

int main(int argc, char **argv)
{
    using namespace Crx;

    int r = glob::RC_EXCEPT;
    try
    {
        r = crx_main(argc, argv);
    }
    catch (CrxAssertException const& cause)
    {
        GLOB_LOG << "\n""assert exception: `" << cause.getMsg() << "', source line " << cause.getSrcLine() << "\n";
        glob::quit(0, glob::RC_EXCEPT);
    }
    catch (...)
    {
        glob::quit("\n""unhandled exception ;(", glob::RC_EXCEPT);
    }
    return r;
}

#endif //CRX
