{
  'maxtrack.pas' maxTRACK midi tracker for maxMUSIC sound system
  (c) 1997 by florian haller
  started 19970527.1941.GRAZ.AT
  finished 19970627.2223.GRAZ.AT
}

program maxTRACK;

  {$M 65521, 4096, 4096}

  uses Crt, Windos;

{ data }

  const Tracksnr = 9999;

  type MPU_percrec = record
         nr: byte;
         nm: string [24];
         sc: string [6];
       end;
       MPU_instrumentrec = record
         iprogram: byte;
         inumber: byte;
       end;
       Songrec = record
         Instruments: array [0..15] of MPU_instrumentrec;
         Active: array [0..7] of boolean;
         Tracks: array [0..Tracksnr] of word;
         Instrument, Velocity, Octave, Speed: byte;
         Maxlength, Endmark: word;
         Filename: string;
       end;
       Menurec = record
         Channelpos, Line: byte;
         Position: word;
       end;
       Playerrec = record
         Counter: byte;
         Position, Played: word;
         Channels: array [0..7, 1..2] of byte;
       end;
       Filerec = record
         name: string [12];
         size: longint;
         typeof: byte;
       end;

  const MPU_dataport: word = $330;
        MPU_statusport: word = $331;
        MPU_patchtypes: array [1..16] of string [16] =
        ('Piano', 'Chrom Percussion', 'Organ', 'Guitar', 'Bass',
         'Strings', 'Ensemble', 'Brass', 'Reed', 'Pipe', 'Synth Lead',
         'Synth Pad', 'Synth Effects', 'Ethnic', 'Percussive',
         'Sound Effects');
        MPU_patches: array [1..16, 1..8] of string [24] =
        (('Acoustic Grand', 'Bright Acoustic', 'Electric Grand',
          'Honkey-Tonk', 'Electric Piano 1', 'Electric Piano 2',
          'Harpsichord', 'Clavinet'),
         ('Celesta', 'Glockenspiel', 'Music Box', 'Vibraphone',
          'Marimba', 'Xylophone', 'Tubular Bells', 'Dulcimer'),
         ('Drawbar Organ', 'Percussive Organ', 'Rock Organ',
          'Church Organ', 'Reed Organ', 'Accordian', 'Harmonica',
          'Tango Accordian'),
         ('Acoustic Guitar (nylon)', 'Acoustic Guitar (steel)',
          'Electric Guitar (jazz)', 'Electric Guitar (clean)',
          'Electric Guitar (muted)', 'Overdriven Guitar',
          'Distortion Guitar', 'Guitar Harmonics'),
         ('Acoustic Bass', 'Electric Bass (finger)',
          'Electric Bass (pick)', 'Fretless Bass',
          'Slap Bass 1', 'Slap Bass 2',
          'Synth Bass 1', 'Synth Bass 2'),
         ('Violin', 'Viola', 'Cello', 'Contrabass', 'Tremolo Strings',
          'Pizzicato Strings', 'Orchestral Strings', 'Timpani'),
         ('String Ensemble 1', 'String Ensemble 2', 'SynthStrings 1',
          'SynthStrings 2', 'Choir Aahs', 'Voice Oohs', 'Synth Voice',
          'Orchestra Hit'),
         ('Trumpet', 'Trombone', 'Tuba', 'Muted Trumpet', 'French Horn',
          'Brass Section', 'SynthBrass 1', 'SynthBrass 2'),
         ('Soprano Sax', 'Alto Sax', 'Tenor Sax', 'Baritone Sax',
          'Oboe', 'English Horn', 'Bassoon', 'Clarinet'),
         ('Piccolo', 'Flute', 'Recorder', 'Pan Flute', 'Blown Bottle',
          'Skakuhachi', 'Whistle', 'Ocarino'),
         ('Lead 1 (square)', 'Lead 2 (sawtooth)', 'Lead 3 (calliope)',
          'Lead 4 (chiff)', 'Lead 5 (charang)', 'Lead 6 (voice)',
          'Lead 7 (fifths)', 'Lead 8 (bass+lead)'),
         ('Pad 1 (new age)', 'Pad 2 (warm)', 'Pad 3 (polysynth)',
          'Pad 4 (choir)', 'Pad 5 (bowed)', 'Pad 6 (metallic)',
          'Pad 7 (halo)', 'Pad 8 (sweep)'),
         ('FX 1 (rain)', 'FX 2 (soundtrack)', 'FX 3 (crystal)',
          'FX 4 (atmosphere)', 'FX 5 (brightness)', 'FX 6 (goblins)',
          'FX 7 (echoes)', 'FX 8 (sci-fi)'),
         ('Sitar', 'Banjo', 'Shamisen', 'Koto', 'Kalimba', 'Bagpipe',
          'Fiddle', 'Shanai'),
         ('Tinkle Bell', 'Agogo', 'Steel Drums', 'Woodblock',
          'Taiko Drum', 'Melodic Tom', 'Synth Drum', 'Reverse Cymbal'),
         ('Guitar Fret Noise', 'Breath Noise', 'Seashore', 'Bird Tweet',
          'Telephone Ring', 'Helicopter', 'Applause', 'Gunshot'));
        MPU_percussion: array [1..47] of MPU_percrec =
        ((nr: 35; nm: 'Acoustic Bass Drum'; sc: 'ABassD'),
         (nr: 36; nm: 'Bass Drum 1'; sc: 'BassD1'),
         (nr: 37; nm: 'Side Stick'; sc: 'SStick'),
         (nr: 38; nm: 'Acoustic Snare'; sc: 'ASnare'),
         (nr: 40; nm: 'Electric Snare'; sc: 'ESnare'),
         (nr: 39; nm: 'Hand Clap'; sc: 'HClap'),
         (nr: 41; nm: 'Low Floor Tom'; sc: 'LwFTom'),
         (nr: 43; nm: 'High Floor Tom'; sc: 'HiFTom'),
         (nr: 45; nm: 'Low Tom'; sc: 'LowTom'),
         (nr: 47; nm: 'Low-Mid Tom'; sc: 'LwMTom'),
         (nr: 48; nm: 'Hi-Mid Tom'; sc: 'HiMTom'),
         (nr: 50; nm: 'High Tom'; sc: 'HghTom'),
         (nr: 42; nm: 'Closed Hi-Hat'; sc: 'CHiHat'),
         (nr: 44; nm: 'Pedal Hi-Hat'; sc: 'PHiHat'),
         (nr: 46; nm: 'Open Hi-Hat'; sc: 'OHiHat'),
         (nr: 49; nm: 'Crash Cymbal 1'; sc: 'CCmbl1'),
         (nr: 57; nm: 'Crash Cymbal 2'; sc: 'CCmbl2'),
         (nr: 51; nm: 'Ride Cymbal 1'; sc: 'RCmbl1'),
         (nr: 59; nm: 'Ride Cymbal 2'; sc: 'RCmbl2'),
         (nr: 52; nm: 'Chinese Cymbal'; sc: 'CCymbl'),
         (nr: 55; nm: 'Splash Cymbal'; sc: 'SCymbl'),
         (nr: 53; nm: 'Ride Bell'; sc: 'RBell'),
         (nr: 54; nm: 'Tambourine'; sc: 'Tambrn'),
         (nr: 56; nm: 'Cowbell'; sc: 'Cowbll'),
         (nr: 58; nm: 'Vibraslap'; sc: 'Vbslap'),
         (nr: 60; nm: 'Hi Bongo'; sc: 'HBongo'),
         (nr: 61; nm: 'Low Bongo'; sc: 'LBongo'),
         (nr: 62; nm: 'Mute Hi Conga'; sc: 'MConga'),
         (nr: 63; nm: 'Open Hi Conga'; sc: 'OConga'),
         (nr: 64; nm: 'Low Conga'; sc: 'LConga'),
         (nr: 65; nm: 'High Timbale'; sc: 'HTimbl'),
         (nr: 66; nm: 'Low Timbale'; sc: 'LTimbl'),
         (nr: 67; nm: 'High Agogo'; sc: 'HAgogo'),
         (nr: 68; nm: 'Low Agogo'; sc: 'LAgogo'),
         (nr: 69; nm: 'Cabasa'; sc: 'Cabasa'),
         (nr: 70; nm: 'Maracas'; sc: 'Mracas'),
         (nr: 71; nm: 'Short Whistle'; sc: 'SWhstl'),
         (nr: 72; nm: 'Long Whistle'; sc: 'LWhstl'),
         (nr: 73; nm: 'Short Guiro'; sc: 'SGuiro'),
         (nr: 74; nm: 'Long Guiro'; sc: 'LGuiro'),
         (nr: 75; nm: 'Claves'; sc: 'Claves'),
         (nr: 76; nm: 'Hi Wood Block'; sc: 'HBlock'),
         (nr: 77; nm: 'Low Wood Block'; sc: 'LBlock'),
         (nr: 78; nm: 'Mute Cuica'; sc: 'MCuica'),
         (nr: 79; nm: 'Open Cuica'; sc: 'OCuica'),
         (nr: 80; nm: 'Mute Triangle'; sc: 'MTrngl'),
         (nr: 81; nm: 'Open Triangle'; sc: 'OTrngl'));
        MPU_notes: array [1..12] of string [2] =
        ('C-', 'C#', 'D-', 'D#', 'E-', 'F-',
         'F#', 'G-', 'G#', 'A-', 'A#', 'B-');
        MPU_keytonote: array [1..12] of byte =
        (44, 31, 45, 32, 46, 47, 34, 48, 35, 49, 36, 50);
        MPU_perchannel: byte = 9;
        cmd_no = 0;
        cmd_off = 1;
        cmd_end = 2;
        cmd_note = 3;
        cmd_continue = 4;
        Pnoev: byte = 255;
        Logo: string [80] = ('maxTRACK midi tracker.  Version 1.  (c) 1997 by florian haller/aeiou');
        file_Maxfiles = 1024;
        file_File = 1;
        file_Dir = 0;

  var Song: Songrec;
      Menu: Menurec;
      Player: Playerrec;

{ mpu procedures }

  procedure MPU_sendbyte; assembler;
  asm
    push     ax
    sub      ax, ax
    mov      dx, MPU_statusport
    @MPU_sendbyte_loop:
      dec      ah
      jz       @MPU_sendbyte_escape
      in       al, dx
      and      al, 40h
      jnz      @MPU_sendbyte_loop
    @MPU_sendbyte_escape:
    pop      ax
    dec      dx
    out      dx, al
  end;

  procedure MPU_readbyte; assembler;
  asm
    push     ax
    sub      ax, ax
    mov      dx, MPU_statusport
    @MPU_sendbyte_loop:
      dec      ah
      jz       @MPU_sendbyte_escape
      in       al, dx
      and      al, 80h
      jnz      @MPU_sendbyte_loop
    @MPU_sendbyte_escape:
    pop      ax
    dec      dx
    in       al, dx
  end;

  procedure MPU_reset; assembler;
  asm
    mov     dx, MPU_statusport
    mov     al, 0ffh
    out     dx, al
    call    MPU_readbyte
    inc     dx
    mov     al, 3Fh
    out     dx, al
    call    MPU_readbyte
    mov     al, 0ffh
    call    MPU_sendbyte
  end;

  procedure MPU_stop;
  var x: byte;
  begin
    for x := 0 to 15 do
      asm
        mov      al, 0b0h
        add      al, x
        call     MPU_sendbyte
        mov      al, 123
        call     MPU_sendbyte
        mov      al, 0
        call     MPU_sendbyte
      end;
  end;

  procedure MPU_setpatch (channel, patch: byte); assembler;
  asm
    mov      al, channel
    add      al, 0c0h
    call     MPU_sendbyte
    mov      al, patch
    call     MPU_sendbyte
  end;

  procedure MPU_note (channel, note, velocity: byte); assembler;
  asm
    mov      al, channel
    add      al, 90h
    call     MPU_sendbyte
    mov      al, note
    call     MPU_sendbyte
    mov      al, velocity
    call     MPU_sendbyte
  end;

  procedure MPU_setinstrument (channel, iprogram, inumber: byte);
  begin
    MPU_setpatch (channel, iprogram * 8 + inumber - 9);
  end;

  function MPU_realnote (octave, note: byte): byte;
  begin
    MPU_realnote := octave * 12 + note + 23;
  end;

  procedure MPU_playinstrument (channel, octave, note, velocity: byte);
  begin
    MPU_note (channel, MPU_realnote (octave, note), velocity);
  end;

{ memory routines }

  function mem_Get (size: word): word; assembler;
  asm
    mov      bx, size
    mov      ah, 48h
    int      21h
    jnc      @Getmem_end
    mov      ax, 0
    @Getmem_end:
  end;

  procedure mem_Free (segment: word); assembler;
  asm
    mov      ax, segment
    mov      es, ax
    mov      ah, 49h
    int      21h
  end;

  procedure mem_Init;
  var x, y, z: word;
  begin
    x := 0;
    repeat
      y := mem_Get (2000);
      if y <> 0 then
        for z := 0 to 999 do
          Song.Tracks [z + x] := y + z * 2;
      Inc (x, 1000);
    until (y = 0) or (x > Tracksnr);
    Song.Maxlength := x - 1;
  end;

  function file_Readdir (var files: array of Filerec): word;
  var filecount: word;
      dir: array [0..70] of char;
      x, z: byte;
      dirinfo: tsearchrec;
      s: string;
  begin
    GetCurDir (dir, 0);
    x := 0;
    while dir [x] <> #0 do Inc (x);
    if dir [x - 1] = '\' then Dec (x);
    dir [x] := '\';
    dir [x + 1] := '*';
    dir [x + 2] := '.';
    dir [x + 3] := '*';
    dir [x + 4] := #0;
    filecount := 0;
    Findfirst (dir, fadirectory, dirinfo);
    while Doserror = 0 do begin
      if dirinfo.attr = fadirectory then begin
        if filecount > file_Maxfiles then Exit;
        z := 0;
        s := '';
        while dirinfo.name [z] <> #0 do begin
          s := s + dirinfo.name [z];
          Inc (z);
        end;
        files [filecount].name := s;
        files [filecount].typeof := file_Dir;
        Inc (filecount);
      end;
      Findnext (dirinfo);
    end;
    dir [x + 3] := 'S';
    dir [x + 4] := 'M';
    dir [x + 5] := '0';
    dir [x + 6] := #0;
    Findfirst (dir, faanyfile - fadirectory, dirinfo);
    while Doserror = 0 do begin
      if filecount > file_Maxfiles then Exit;
      z := 0;
      s := '';
      while dirinfo.name [z] <> #0 do begin
        s := s + dirinfo.name [z];
        Inc (z);
      end;
      files [filecount].name := s;
      files [filecount].size := dirinfo.size;
      files [filecount].typeof := file_File;
      Findnext (dirinfo);
      Inc (filecount);
    end;
    file_Readdir := filecount;
  end;

{ input functions }

  function in_Readkey: char; assembler;
  asm
    mov      ah, 0
    int      16h
    mov      al, ah
  end;

  function in_Readascii: char; assembler;
  asm
    mov      ah, 0
    int      16h
  end;

{ output procedures/definitions }

  var screen: array [1..4000] of byte;

  procedure out_Savescreen;
  begin
    Move (mem [$b800: 0], screen, 4000);
  end;

  procedure out_Restorescreen;
  begin
    Move (screen, Mem [$b800: 0], 4000);
  end;

  procedure out_Writexy (x, y, color: byte; text: string);
  var z: byte;
  begin
    for z := 1 to length (text) do
      memw [$b800: (y * 80 + x + z - 1) * 2] := color shl 8 + ord (text [z]);
  end;

  function out_Getchar (x, y: byte): char;
  begin
    out_Getchar := chr (mem [$b800: (y * 80 + x) * 2]);
  end;

  function out_Getcolor (x, y: byte): byte;
  begin
    out_Getcolor := mem [$b800: (y * 80 + x) * 2 + 1];
  end;

  procedure out_Gotoxy (x, y: byte); assembler;
  asm
    mov      ah, 2
    mov      bh, 0
    mov      dh, y
    mov      dl, x
    int      10h
  end;

  procedure out_Clearscreen (editscr: boolean);
  var x, y, z: byte;
      s: string;
  const help: array [1..10] of string [6] =
        ('Help', 'Save', 'Load', 'New', 'Play/a', 'Play/c', 'Stop', 'Instr', 'Speed', 'Quit');
  begin
    asm
      push     0b800h
      pop      es
      mov      di, 0
      mov      ax, 1100h
      mov      cx, 2000
      rep      stosw
      mov      di, 0
      mov      ax, 3000h
      mov      cx, 80
      rep      stosw
    end;
    if not editscr then z := 1 else z := 0;
    out_Writexy (5, 0, $30, Logo);
    for y := 1 to 22 + z do begin
      out_Writexy (0, y, $1b, '');
      out_Writexy (79, y, $1b, '');
    end;
    for y := 1 to 78 do begin
      out_Writexy (y, 1, $1b, '');
      out_Writexy (y, 23 + z, $1b, '');
    end;
    out_Writexy (0, 1, $1b, '');
    out_Writexy (79, 1, $1b, '');
    out_Writexy (0, 23 + z, $1b, '');
    out_Writexy (79, 23 + z, $1b, '');
    if not editscr then Exit;
    for x := 1 to 78 do out_Writexy (x, 3, $1b, '');
    for y := 2 to 20 do begin
      for x := 0 to 6 do begin
        out_Writexy (x * 9 + 16, y + 2, $1b, '');
        out_Writexy (x * 9 + 16, 3, $1b, '');
        out_Writexy (x * 9 + 16, 23, $1b, '');
      end;
      out_Writexy (7, y + 2, $1b, '');
    end;
    out_Writexy (7, 3, $1b, '');
    out_Writexy (7, 23, $1b, '');
    out_Writexy (0, 3, $1b, '');
    out_Writexy (79, 3, $1b, '');
    out_Writexy (2, 2, $1e, 'Instr');
    out_Writexy (50, 2, $1e, 'Oct');
    out_Writexy (58, 2, $1e, 'Veloc');
    out_Writexy (69, 2, $1e, 'Pos');
    z := 0;
    x := 1;
    repeat
      Str (x, s);
      if x <> 1 then s := ' ' + s;
      out_Writexy (z, 24, 7, s);
      Inc (z, length (s));
      s := help [x];
      while length (s) < 6 do s := s + ' ';
      out_Writexy (z, 24, $30, s);
      Inc (z, 6);
      Inc (x);
    until z >= 79;
  end;

  procedure out_Window (x, y, tx, ty: byte; title: string);
  var a, b: byte;
  begin
    for a := x - 3 to tx + 3 do
      for b := y - 1 to ty + 1 do
        out_Writexy (a, b, $70, ' ');
    for a := x + 1 to tx - 1 do begin
      out_Writexy (a, y, $70, '');
      out_Writexy (a, ty, $70, '');
    end;
    for a := y + 1 to ty - 1 do begin
      out_Writexy (x, a, $70, '');
      out_Writexy (tx, a, $70, '');
    end;
    out_Writexy (x, y, $70, '');
    out_Writexy (x, ty, $70, '');
    out_Writexy (tx, y, $70, '');
    out_Writexy (tx, ty, $70, '');
    for a := x to tx + 4 do
      out_Writexy (a, ty + 2, $08, out_Getchar (a, ty + 2));
    for a := y to ty + 2 do begin
      out_Writexy (tx + 4, a, $08, out_Getchar (tx + 4, a));
      out_Writexy (tx + 5, a, $08, out_Getchar (tx + 5, a));
    end;
    if title <> '' then
      out_Writexy (x + (tx - x) div 2 - length (title) div 2 - 1,
        y, $70, ' ' + title + ' ');
  end;

  procedure out_Initscreen; assembler;
  asm
    mov      ax, 3
    int      10h
    mov      ah, 1
    mov      cx, 2000h
    int      10h
  end;

  procedure out_Closescreen;
  begin
    asm
      mov      ax, 3
      int      10h
      mov      bh, 0
      mov      dx, 1900h
      mov      ah, 2
      int      10h
    end;
    out_Writexy (0, 23, 7, Logo);
  end;

  procedure out_Box (x, y, tx, ty, color: byte);
  var z: byte;
  begin
    for z := y + 1 to ty - 1 do begin
      out_Writexy (x, z, color, '');
      out_Writexy (tx, z, color, '');
    end;
    for z := x + 1 to tx - 1 do begin
      out_Writexy (z, y, color, '');
      out_Writexy (z, ty, color, '');
    end;
    out_Writexy (x, y, color, '');
    out_Writexy (tx, y, color, '');
    out_Writexy (x, ty, color, '');
    out_Writexy (tx, ty, color, '');
  end;

  procedure status_Refreshinfo;
  var s: string;
      x: byte;
  begin
    for x := 11 to 43 do out_Writexy (x, 2, $1b, ' ');
    Str (Song.Instrument, s);
    if Song.Instrument < 10 then s := '0' + s;
    if Song.Instrument <> MPU_perchannel then
      out_Writexy (8, 2, $1e, s + ': ' + MPU_patches [Song.Instruments
        [Song.Instrument].iprogram, Song.Instruments [Song.Instrument].inumber])
    else
      out_Writexy (8, 2, $1e, '09: Percussion (' + MPU_percussion [Song.Instruments
        [MPU_perchannel].inumber].nm + ')');
    Str (Song.Octave, s);
    out_Writexy (54, 2, $1e, s);
    Str (Song.Velocity, s);
    if Song.Velocity < 10 then s := '0' + s;
    out_Writexy (64, 2, $1e, s);
    Str (Menu.Position, s);
    for x := 1 to 4 - length (s) do s := '0' + s;
    out_Writexy (73, 2, $1e, s);
  end;

{ status procedures }

  procedure status_Displaychannels;
  var x: byte;
  begin
    for x := 0 to 7 do
      if Song.Active [x] then
        out_Writexy (x * 9 + 10, 3, $30, ' c' + chr (x + 48) + ' ')
      else
        out_Writexy (x * 9 + 10, 3, $38, ' c' + chr (x + 48) + ' ');
  end;

  procedure status_DisplayInstruments (patchtype: byte);
  var x, z: byte;
  begin
    for x := 29 to 57 do out_Writexy (x, 6, $30, ' ');
    out_Writexy (29, 6, $30, MPU_patchtypes [patchtype]);
    for x := 1 to 8 do begin
      for z := 23 to 55 do out_Writexy (z, 7 + x, $70, ' ');
      out_Writexy (23, 7 + x, $70, MPU_patches [patchtype, x]);
    end;
  end;

  procedure status_Displaypercussion (y: byte);
  var x, z: byte;
  begin
    for x := y to y + 10 do begin
      for z := 23 to 55 do out_Writexy (z, x - y + 6, $70, ' ');
      out_Writexy (23, x - y + 6, $70, MPU_percussion [x].nm);
      out_Writexy (56 - length (MPU_percussion [x].sc), x - y + 6, $70,
        MPU_percussion [x].sc);
    end;
  end;

  procedure status_Writeevent (x, y: byte; color: boolean; pos: word; channel: byte);
  const Hextable: array [0..15] of char =
        ('0', '1', '2', '3', '4', '5', '6', '7',
         '8', '9', 'A', 'B', 'C', 'D', 'E', 'F');
  var c, a, b: byte;
      s: string;
  begin
    if color then c := 0 else c := $10;
    case mem [Song.Tracks [pos]: channel * 4] of
      cmd_no: out_Writexy (x * 9 + 8, y + 4, c + 7, '  ');
      cmd_note: begin
                  a := mem [Song.Tracks [pos]: channel * 4 + 1];
                  if a <> MPU_perchannel then begin
                    Str (a, s);
                    if s [0] = #1 then s := '0' + s;
                    out_Writexy (x * 9 + 8, y + 4, c + 7, s + ' ');
                    a := mem [Song.Tracks [pos]: channel * 4 + 2];
                    b := a shr 4;
                    a := a and 15;
                    Str (b, s);
                    out_Writexy (x * 9 + 11, y + 4, c + 15,
                      MPU_notes [a] + s + ' ');
                  end else begin
                    out_Writexy (x * 9 + 8, y + 4, c + 15,
                      MPU_percussion [mem [Song.Tracks [pos]: channel * 4 + 2]].
                        sc + ' ');
                  end;
                  out_Writexy (x * 9 + 15, y + 4, c + 9,
                    Hextable [mem [Song.Tracks [pos]: channel * 4 + 3]]);
                end;
      cmd_off: begin
                 out_Writexy (x * 9 + 11, y + 4, c + 7, ' ');
                 out_Writexy (x * 9 + 8, y + 4, c + 15, 'off');
               end;
      cmd_continue: out_Writexy (x * 9 + 8, y + 4, c + 15, 'continue');
      cmd_end: out_Writexy (x * 9 + 8, y + 4, c + 15, 'end ');
    end;
  end;

  procedure status_Displayevents (start: word);
  var x, y, z: word;
      s: string;
  begin
    for x := 0 to 18 do
      if start + x <= Song.Maxlength then begin
        Str (start + x, s);
        for y := 1 to 4 - length (s) do s := '0' + s;
        out_Writexy (2, x + 4, $1b, s);
        for y := 0 to 7 do status_Writeevent (y, x, false, start + x, y);
      end else begin
        out_Writexy (2, x + 4, $1b, '    ');
        for y := 0 to 7 do
          out_Writexy (y * 9 + 8, x + 4, $17, '        ');
      end;
  end;

  procedure Moveup;
  var s: string;
      x: byte;
  begin
    if Menu.Position > 0 then begin
      Dec (Menu.Position);
      if Menu.Line > 0 then Dec (Menu.Line) else begin
        Move (mem [$b800: 640], mem [$b800: 800], 2880);
        Str (Menu.Position, s);
        for x := 1 to 4 - length (s) do s := '0' + s;
        out_Writexy (2, 4, $1b, s);
      end;
      for x := 0 to 7 do status_Writeevent (x, Menu.Line, false, Menu.Position, x);
    end;
  end;

  procedure Movedown;
  var s: string;
      x: byte;
  begin
    if Menu.Position < Song.Maxlength then begin
      Inc (Menu.Position);
      if Menu.Line < 18 then Inc (Menu.Line) else begin
        Move (mem [$b800: 800], mem [$b800: 640], 2880);
        Str (Menu.Position, s);
        for x := 1 to 4 - length (s) do s := '0' + s;
        out_Writexy (2, 22, $1b, s);
      end;
      for x := 0 to 7 do status_Writeevent (x, Menu.Line, false, Menu.Position, x);
    end;
  end;

{ event handling }

  procedure Setevent (pos: word; channel, subpos, cont: byte);
  begin
    mem [Song.Tracks [pos]: channel * 4 + subpos] := cont;
  end;

  function Getevent (pos: word; channel, subpos: byte): byte;
  begin
    Getevent := mem [Song.Tracks [pos]: channel * 4 + subpos];
  end;

  procedure status_Resetall (display: boolean);
  var z: word;
  begin
    for Song.Instrument := 0 to 15 do begin
      Song.Instruments [Song.Instrument].iprogram := 1;
      Song.Instruments [Song.Instrument].inumber := 1;
      MPU_setinstrument (Song.Instrument, Song.Instruments [Song.Instrument].iprogram,
        Song.Instruments [Song.Instrument].inumber);
    end;
    for z := 0 to 7 do
      Song.Active [z] := true;
    for z := 0 to Song.Maxlength do
      Fillchar (mem [Song.Tracks [z]: 0], 32, cmd_no);
    Menu.Position := 0;
    Menu.Channelpos := 0;
    Menu.Line := 0;
    Song.Instrument := 0;
    Song.Velocity := 15;
    Song.Octave := 3;
    Song.Endmark := 0;
    Song.Speed := 6;
    Song.Filename := '';
    Setevent (0, 0, 0, cmd_end);
    if display then begin
      status_Displaychannels;
      status_Displayevents (0);
      status_Refreshinfo;
    end;
  end;

  {$F+}

  procedure Newtimer; interrupt;
  var x, a, b, c: byte;
  begin
    Inc (Player.Counter);
    if Player.Counter = Song.Speed then begin
      if Getevent (Player.Position, 0, 0) <> cmd_end then begin
        for x := 0 to 7 do begin
          if Song.Active [x] then
            case Getevent (Player.Position, x, 0) of
              cmd_note: begin
                          if Player.Channels [x, 1] <> Pnoev then
                            MPU_note (Player.Channels [x, 1],
                              Player.Channels [x, 2], 0);
                          c := Getevent (Player.Position, x, 1);
                          if Getevent (Player.Position, x, 3) = 0 then
                            c := Pnoev;
                          Player.Channels [x, 1] := c;
                          if Getevent (Player.Position, x, 1) <>
                            MPU_perchannel then begin
                            a := Getevent (Player.Position, x, 2);
                            b := a shr 4;
                            a := a and 15;
                            a := MPU_realnote (b, a);
                          end else
                            a := MPU_percussion [Getevent (Player.Position,
                              x, 2)].nr;
                          Player.Channels [x, 2] := a;
                          MPU_note (Getevent (Player.Position, x, 1), a,
                            Getevent (Player.Position, x, 3) shl 3);
                        end;
              cmd_off: if Player.Channels [x, 1] <> Pnoev then begin
                         MPU_note (Player.Channels [x, 1],
                           Player.Channels [x, 2], 0);
                         Player.Channels [x, 1] := Pnoev;
                       end;
              cmd_continue: Player.Channels [x, 1] := Pnoev;
            end;
        end;
        Player.Played := Player.Position;
        Inc (Player.Position);
        if Player.Position > Song.Maxlength then Player.Position := 0;
      end else begin
        for x := 0 to 7 do
          if Player.Channels [x, 1] <> Pnoev then begin
            MPU_note (Player.Channels [x, 1],
              Player.Channels [x, 2], 0);
            Player.Channels [x, 1] := Pnoev;
          end;
        Player.Position := 0;
      end;
      Player.Counter := 0;
    end;
    Port [$20] := $20;
  end;

  {$F-}

{ main routines }

  procedure mn_Help;
  var z, page: byte;
      r: char;
  const title: string [4] = 'Help';
        textlength = 20;
        text: array [1..2, 0..textlength] of string [76] =
        (('                                                                    Page 1/2',
          'Function keys                          Edit keys',
          '',
          'F1   Help (twice for page 2)           ' + chr (30) + ' ' + chr (31) +
            ' ' + chr (17) + ' ' + chr (16) + '        Move around',
          'F2   Save file                         Pos1/End       Move to begin/end',
          'F3   Load file                         Page' + chr (30) + '/Page' +
            chr (31) + '    Move one page up/down',
          'F4   New (reset everything)            Backspace      Delete note',
          'F5   Play whole song from start on     ''I''            Set end command',
          'F6   Play from cursor on               ''O''            Set off command',
          'F7   Stop playing                      ''P''            Set continue command',
          'F8   Select instrument                 + -            Next/last instrument',
          'F9   Speed                             Insert/Delete  Insert/delete line',
          'F10  Quit',
          '                                       Notation format',
          'Instrument selection',
          '                                         ',
          chr (17) + ' ' + chr (16) + '    Select program                   ' + chr (24) + '  ' + chr (24) + '  ' + chr (24),
          chr (30) + ' ' + chr (31) + '    Select instrument                   Velocity (0..15 or 0..F in hex)',
          'Space  Play instrument                   Note (CDEFGAB) and octave (0..6)',
          'Enter  Select instrument               Instrument (0..15, if 9 a shortcut',
          'ESC    Cancel                            for the instrument is displayed)'),
          ('                                                                    Page 2/2',
           'Keyboard layout                                Velocity',
           '',
           'Notes     C#   D#        F#   G#   A#          Enter the velocity with the',
           '                                  number keys, values between',
           '                                  0 and 15 are allowed. For',
           'Keys     S  D      G  H  J        values below 10 enter a zero',
           '                   first then the number.',
           '        Z   X   C   V   B   N  M       Cancel with ESC.',
           '      ',
           'Notes    C    D    E    F    G    A    B       Country specifics',
           '',
           'Select the octave with Q (octave 0) to U       The keyboard layout shown',
           '(octave 6) keys. De/Activate channels          here is for US keyboards,',
           '1 to 8 with keys 2 (channel 1) to 9            other country sets have Z and',
           '(channel 8).                                   X exchanged (but the layout',
           '                                               keeps the same).',
           'Contacting the author',
           '',
           'florian haller, jaritzweg 15, 8045 graz, austria',
           'e-mail: florian.haller@kornt02.ortwein.big.ac.at (until June 1999 ;)'));
  begin
    out_Savescreen;
    page := 2;
    repeat
      if page = 2 then page := 1 else page := 2;
      out_Clearscreen (false);
      out_Writexy (36, 1, $1b, ' ' + title + ' ');
      for z := 0 to textlength do
        out_Writexy (2, z + 2, $1b, text [page, z]);
      r := in_Readkey;
    until r <> #59;
    out_Restorescreen;
  end;

  procedure mn_Save;
  var z, x: word;
      a, b, last, format: byte;
      y: integer;
      r: char;
      s: string;
      out: file;
      temp, ins: array [0..15] of byte;
      act: array [0..7] of boolean;
      ply: array [0..7, 1..2] of byte;
      cmds, nx: boolean;
  const formats: array [1..2] of string [31] =
        ('SimpleMidi level 0 (raw data) ',
         'SimpleMidi level 1 (optimized)');
  type tempevent = record
         dat: array [1..32] of byte;
         lng: byte;
       end;

  procedure Save_instruments;
  var x, a: byte;
      z: word;
  begin
    for x := 0 to 15 do begin temp [x] := 0; ins [x] := 0; end;
    for z := 0 to Song.Endmark do
      for x := 0 to 7 do
        if Getevent (z, x, 0) = cmd_note then
          temp [Getevent (z, x, 1)] := 1;
    z := 0;
    for x := 0 to 15 do
      if (temp [x] = 1) and (x <> 9) then begin
        ins [z] := Song.Instruments [x].iprogram * 8 + Song.Instruments [x].inumber - 9;
        temp [x] := z;
        Inc (z);
        if z = 9 then z := 10;
      end;
    if z = 10 then z := 9;
    if z > 0 then Dec (z);
    for x := 0 to z do begin
      a := ins [x] shl 1;
      if x = z then Inc (a);
      Blockwrite (out, a, 1);
    end;
  end;

  procedure Save_ate (var event: tempevent; data: byte);
  begin
    Inc (event.lng);
    event.dat [event.lng] := data;
  end;

  procedure Save_event (nr: word; var event: tempevent);
  var x, a, b, c: byte;
  begin
    event.lng := 0;
    for x := 0 to 7 do
      case Getevent (Player.Position, x, 0) of
        cmd_note: begin
                    if Player.Channels [x, 1] <> Pnoev then begin
                      a := Player.Channels [x, 1] shl 4;
                      Save_ate (event, a);
                      Save_ate (event, Player.Channels [x, 2]);
                    end;
                    c := Getevent (Player.Position, x, 1);
                    if Getevent (Player.Position, x, 3) = 0 then
                      c := Pnoev;
                    Player.Channels [x, 1] := c;
                    if Getevent (Player.Position, x, 1) <>
                      MPU_perchannel then begin
                      a := Getevent (Player.Position, x, 2);
                      b := a shr 4;
                      a := a and 15;
                      a := MPU_realnote (b, a);
                    end else
                      a := MPU_percussion [Getevent (Player.Position,
                        x, 2)].nr;
                    b := Getevent (Player.Position, x, 1) shl 4 +
                      Getevent (Player.Position, x, 3);
                    Save_ate (event, b);
                    a := a shl 1;
                    Save_ate (event, a);
                    Player.Channels [x, 2] := a;
                  end;
        cmd_off: if Player.Channels [x, 1] <> Pnoev then begin
                   Save_ate (event, Player.Channels [x, 1] shl 4);
                   Save_ate (event, Player.Channels [x, 2]);
                   Player.Channels [x, 1] := Pnoev;
                 end;
        cmd_continue: Player.Channels [x, 1] := Pnoev;
      end;
  end;

  procedure Save_temp (temp: tempevent);
  var a, b: byte;
  begin
    a := (Player.Counter * Song.Speed) shl 1;
    if temp.lng > 0 then Inc (a);
    Blockwrite (out, a, 1);
    for a := 1 to temp.lng do
      if a < temp.lng then
        Blockwrite (out, temp.dat [a], 1)
      else begin
        b := temp.dat [a] + 1;
        Blockwrite (out, b, 1);
      end;
    Player.Counter := 0;
  end;

  procedure Save_data;
  var x: byte;
      temp, temp1: tempevent;
  begin
    for x := 0 to 7 do Player.Channels [x, 1] := Pnoev;
    Save_event (0, temp);
    Player.Position := 0;
    Player.Counter := 0;
    repeat
      Inc (Player.Position);
      Inc (Player.Counter);
      Save_event (Player.Position, temp1);
      if (temp1.lng > 0) or (Player.Position = Song.Endmark) then begin
        Save_temp (temp);
        temp := temp1;
      end;
    until Player.Position = Song.Endmark;
    temp.lng := 0;
    for x := 0 to 7 do
      if Player.Channels [x, 1] <> Pnoev then begin
        a := Player.Channels [x, 1] shl 4;
        Save_ate (temp, a);
        Save_ate (temp, Player.Channels [x, 2]);
      end;
    Player.Counter := 1;
    Save_temp (temp);
  end;

  begin
    if Song.Endmark = 0 then Exit;
    out_Savescreen;
    format := 0;
    out_Window (22, 11, 55, 14, 'Select format');
    out_Writexy (24, 12, $70, formats [1]);
    out_Writexy (24, 13, $70, formats [2]);
    y := 1;
    repeat
      out_Writexy (24, 11 + y, 15, formats [y]);
      r := in_Readkey;
      out_Writexy (24, 11 + y, $70, formats [y]);
      case r of
        #72: if y = 2 then y := 1;
        #80: if y = 1 then y := 2;
      end;
    until (r = #1) or (r = #28);
    out_Restorescreen;
    if r = #1 then Exit;
    out_Savescreen;
    format := y - 1;
    out_Window (6, 12, 73, 14, 'Enter file name');
    for z := 8 to 71 do out_Writexy (z, 13, $30, ' ');
    s := Song.Filename;
    out_Gotoxy (8, 13);
    asm
      mov      cx, 607h
      mov      ah, 1
      int      10h
    end;
    repeat
      out_Writexy (8, 13, $30, s);
      out_Gotoxy (8 + ord (s [0]), 13);
      r := in_Readascii;
      if (r <> #13) and (r <> #27) then
        if r = #8 then begin
          if ord (s [0]) > 0 then begin
            out_Writexy (7 + ord (s [0]), 13, $30, ' ');
            Dec (s [0]);
          end;
        end else
          if ord (s [0]) < 63 then
            s := s + r;
    until (r = #27) or (r = #13);
    asm
      mov      ah, 1
      mov      cx, 2000h
      int      10h
    end;
    if r = #27 then begin
      out_Restorescreen;
      Exit;
    end;
    y := - 1;
    repeat
      Inc (y);
    until (y = 3) or (s [ord (s [0]) - y] = '.');
    if s [ord (s [0]) - y] = '.' then
      Delete (s, ord (s [0]) - y, y + 1);
    Song.Filename := s;
    if format = 0 then s := s + '.SM0' else s := s + '.SM1';
    if r = #13 then begin
      {$I-}
      Assign (out, s);
      Rewrite (out, 1);
      if IOResult <> 0 then begin
        out_Window (6, 12, 73, 14, 'Error');
        out_Writexy (12, 13, $70, 'Warning: could not save file (maybe not enough space).');
        in_Readkey;
      end else begin
        if format = 1 then begin
          Save_instruments;
          Save_data;
          a := 0;
          Blockwrite (out, a, 1);
        end else begin
          Blockwrite (out, Song.Speed, 1);
          Blockwrite (out, Song.Instruments, sizeof (Song.Instruments));
          Blockwrite (out, Song.Endmark, 2);
          for z := 0 to Song.Endmark do
            Blockwrite (out, mem [Song.Tracks [z]: 0], 32);
        end;
        Close (out);
        {$I+}
      end;
    end;
    out_Restorescreen;
  end;

  procedure mn_Load;
  var files: array [1..file_Maxfiles] of Filerec;
      filecount, pos: word;
      line: byte;
      r: char;
      s: string;
      inp: file;
      y: integer;

  procedure display_Entry (y, color: byte; nr: word);
  var z: byte;
  begin
    for z := 26 to 55 do out_Writexy (z, y, color, ' ');
    out_Writexy (26, y, color, files [nr].name);
    if files [nr].typeof = file_File then begin
      Str (files [nr].size, s);
      out_Writexy (56 - ord (s [0]), y, color, s);
    end else
      out_Writexy (51, y, color, '<DIR>');
  end;

  procedure display_Dir (nr: word);
  var x, y, z: byte;
  begin
    if filecount < 12 then y := filecount else y := 11;
    for x := 1 to y do display_Entry (6 + x, $70, x + nr);
  end;

  procedure mn_Load_init;
  var x, y: byte;
      s: string;
  begin
    filecount := file_Readdir (files);
    for x := 26 to 55 do
      for y := 7 to 17 do
        out_Writexy (x, y, $70, ' ');
    line := 0;
    pos := 1;
    display_Dir (line);
    r := #255;
  end;

  begin
    out_Savescreen;
    out_Window (24, 6, 57, 18, 'Load file');
    mn_Load_init;
    repeat
      display_Entry (7 + line, 15, pos);
      r := in_Readkey;
      display_Entry (7 + line, $70, pos);
      case r of
        #72: if pos > 1 then begin
               Dec (pos);
               if line > 0 then Dec (line) else
                 display_Dir (pos - 1);
             end;
        #80: if pos < filecount then begin
               Inc (pos);
               if line < 10 then Inc (line) else
                 display_Dir (pos - 11);
             end;
        #28: if files [pos].typeof = file_Dir then begin
               Chdir (files [pos].name);
               mn_Load_init;
             end;
      end;
    until (r = #1) or (r = #28);
    out_Restorescreen;
    if r = #28 then begin
      status_Resetall (false);
      s := files [pos].name;
      y := - 1;
      repeat
        Inc (y);
      until (y = 3) or (s [ord (s [0]) - y] = '.');
      if s [ord (s [0]) - y] = '.' then
        Delete (s, ord (s [0]) - y, y + 1);
      Song.Filename := s;
      Assign (inp, files [pos].name);
      Reset (inp, 1);
      pos := 0;
      {$I-}
      Blockread (inp, Song.Speed, 1);
      Blockread (inp, Song.Instruments, sizeof (Song.Instruments));
      Blockread (inp, Song.Endmark, 2);
      for pos := 0 to Song.Endmark do
        Blockread (inp, mem [Song.Tracks [pos]: 0], 32);
      Close (inp);
      {$I+}
      status_Displaychannels;
      status_Displayevents (0);
      status_Refreshinfo;
    end;
  end;

  procedure mn_Reset (askfor: boolean);
  const choices: array [1..2] of string [8] =
        ('[ Yes  ]', '[  No  ]');
  var z: word;
      r: char;
  begin
    if askfor then begin
      out_Savescreen;
      out_Window (17, 11, 60, 14, '');
      out_Writexy (20, 12, $70, 'Warning: all data will be lost. Reset?');
      for z := 1 to 2 do out_Writexy (29 + (z - 1) * 10, 13, $70, choices [z]);
      z := 2;
      repeat
        out_Writexy (29 + (z - 1) * 10, 13, $30, choices [z]);
        r := in_Readkey;
        out_Writexy (29 + (z - 1) * 10, 13, $70, choices [z]);
        case r of
          #75, #77: if z = 1 then z := 2 else z := 1;
        end;
      until (r = #1) or (r = #28) or (r = #44) or (r = #49);
      out_Restorescreen;
      if (r = #1) or (r = #49) or ((z = 2) and (r = #28)) then exit;
    end;
    status_Resetall (true);
  end;

  procedure mn_Play (start: word);
  var Oldtimer: pointer;
      x: byte;
      y: word;
  begin
    MPU_stop;
    while Keypressed do Readkey;
    out_Savescreen;
    Player.Position := start;
    Player.Counter := 0;
    Player.Played := 65535;
    y := Player.Played;
    for x := 0 to 7 do Player.Channels [x, 1] := Pnoev;
    for x := 0 to 15 do
      MPU_setinstrument (x, Song.Instruments [x].iprogram,
        Song.Instruments [x].inumber);
    Getintvec (8, Oldtimer);
    Setintvec (8, @Newtimer);
    repeat
      if Player.Played <> y then begin
        status_Displayevents (Player.Played);
        for x := 0 to 79 do
          out_Writexy (x, 4, out_Getcolor (x, 4) and 15, out_Getchar (x, 4));
        y := Player.Played;
      end;
    until Keypressed;
    Setintvec (8, Oldtimer);
    MPU_reset;
    out_Restorescreen;
  end;

  procedure mn_SelectInstrument;
  var r: char;
      temp: MPU_instrumentrec;
      x, y, last: byte;
      playing: boolean;
  begin
    out_Savescreen;
    playing := false;
    temp := Song.Instruments [Song.Instrument];
    if Song.Instrument = MPU_perchannel then begin
      last := 0;
      out_Window (21, 5, 57, 17, 'Select Percussion instrument');
      if temp.inumber > 37 then y := 37 else y := temp.inumber;
      status_Displaypercussion (y);
      if temp.inumber > 37 then y := temp.inumber - 36 else y := 1;
      repeat
        for x := 23 to 55 do out_Writexy (x, 5 + y, $f, ' ');
        out_Writexy (23, 5 + y, $f, MPU_percussion [temp.inumber].nm);
        out_Writexy (56 - length (MPU_percussion [temp.inumber].sc),
          5 + y, $f, MPU_percussion [temp.inumber].sc);
        r := in_Readkey;
        for x := 23 to 55 do out_Writexy (x, 5 + y, $70, ' ');
        out_Writexy (23, 5 + y, $70, MPU_percussion [temp.inumber].nm);
        out_Writexy (56 - length (MPU_percussion [temp.inumber].sc),
          5 + y, $70, MPU_percussion [temp.inumber].sc);
        case r of
          #72: if y > 1 then begin
                 Dec (y);
                 Dec (temp.inumber);
               end else
                 if temp.inumber > 1 then begin
                   Dec (temp.inumber);
                   status_Displaypercussion (temp.inumber);
                 end;
          #80: if (y < 11) and (temp.inumber < 47) then begin
                 Inc (y);
                 Inc (temp.inumber);
               end else
                 if temp.inumber < 47 then begin
                   Inc (temp.inumber);
                   status_Displaypercussion (temp.inumber - 10);
                 end;
          #57: begin
                 MPU_note (9, last, 0);
                 last := MPU_percussion [temp.inumber].nr;
                 MPU_note (9, last, 127);
                 playing := true;
               end;
        end;
      until (r = #28) or (r = #1);
      if r = #28 then Song.Instruments [Song.Instrument] := temp;
      if playing then MPU_note (9, last, 0);
      out_Restorescreen;
      status_Refreshinfo;
      Exit;
    end;
    out_Window (19, 5, 59, 17, 'Select instrument');
    out_Writexy (21, 6, $70, 'Program');
    out_Box (21, 7, 57, 16, $70);
    status_DisplayInstruments (temp.iprogram);
    repeat
      for x := 23 to 55 do out_Writexy (x, 7 + temp.inumber, $f, ' ');
      out_Writexy (23, 7 + temp.inumber, $f, MPU_patches [temp.iprogram, temp.inumber]);
      r := in_Readkey;
      for x := 23 to 55 do out_Writexy (x, 7 + temp.inumber, $70, ' ');
      out_Writexy (23, 7 + temp.inumber, $70, MPU_patches [temp.iprogram, temp.inumber]);
      case r of
        #72: if temp.inumber > 1 then Dec (temp.inumber);
        #80: if temp.inumber < 8 then Inc (temp.inumber);
        #75: if temp.iprogram > 1 then begin
               Dec (temp.iprogram);
               status_DisplayInstruments (temp.iprogram);
             end;
        #77: if temp.iprogram < 16 then begin
               Inc (temp.iprogram);
               status_DisplayInstruments (temp.iprogram);
             end;
        #57: begin
               MPU_playinstrument (Song.Instrument, 3, 1, 0);
               MPU_setinstrument (Song.Instrument, temp.iprogram, temp.inumber);
               MPU_playinstrument (Song.Instrument, 3, 1, 127);
               playing := true;
             end;
      end;
    until (r = #28) or (r = #1);
    if r = #28 then Song.Instruments [Song.Instrument] := temp;
    if playing then MPU_playinstrument (Song.Instrument, 3, 1, 0);
    MPU_setinstrument (Song.Instrument, Song.Instruments [Song.Instrument].iprogram,
      Song.Instruments [Song.Instrument].inumber);
    out_Restorescreen;
  end;

  procedure mn_Speed;
  var temp: byte;
      s: string;
      r: char;
  begin
    out_Savescreen;
    out_Window (29, 11, 49, 13, 'Enter speed');
    out_Writexy (32, 12, $30, '  ' + chr (18));
    out_Writexy (36, 12, $70, 'tick(s)/row');
    temp := Song.Speed;
    repeat
      Str (temp, s);
      if s [0] = #1 then s := ' ' + s;
      Out_Writexy (32, 12, $30, s);
      r := in_Readkey;
      case r of
        #72: if temp < 18 then Inc (temp);
        #80: if temp > 1 then Dec (temp);
      end;
    until (r = #1) or (r = #28);
    if r = #28 then Song.Speed := temp;
    out_Restorescreen;
  end;

  function mn_Quit: boolean;
  const choices: array [1..2] of string [8] =
        ('[ Yes  ]', '[  No  ]');
  var z: word;
      r: char;
  begin
    out_Savescreen;
    out_Window (23, 11, 55, 14, '');
    out_Writexy (26, 12, $70, 'Do you really want to quit?');
    for z := 1 to 2 do out_Writexy (31 + (z - 1) * 10, 13, $70, choices [z]);
    z := 2;
    repeat
      out_Writexy (31 + (z - 1) * 10, 13, $30, choices [z]);
      r := in_Readkey;
      out_Writexy (31 + (z - 1) * 10, 13, $70, choices [z]);
      case r of
        #75, #77: if z = 1 then z := 2 else z := 1;
      end;
    until (r = #1) or (r = #28) or (r = #44) or (r = #49);
    out_Restorescreen;
    if (r = #1) or (r = #49) or ((z = 2) and (r = #28)) then
      mn_Quit := false else mn_Quit := true;
  end;

  procedure mn_Mainprogram;
  var r: char;
      x, y: word;
      s: string;
      Lastinst, Lastoct, Lastnote: byte;
  begin
    repeat
      status_Writeevent (Menu.Channelpos, Menu.Line, true, Menu.Position, Menu.Channelpos);
      r := in_Readkey;
      status_Writeevent (Menu.Channelpos, Menu.Line, false, Menu.Position, Menu.Channelpos);
      case r of
        #75: if Menu.Channelpos > 0 then Dec (Menu.Channelpos);
        #77: if Menu.Channelpos < 7 then Inc (Menu.Channelpos);
        #72: Moveup;
        #80: Movedown;
        #73: for x := 1 to 18 do Moveup;
        #81: for x := 1 to 18 do Movedown;
        #71: begin
               status_Displayevents (0);
               Menu.Position := 0;
               Menu.Line := 0;
             end;
        #79: begin
               Menu.Position := Song.Endmark;
               if Song.Endmark < 19 then begin
                 Menu.Line := Song.Endmark;
                 status_Displayevents (0);
               end else begin
                 Menu.Line := 18;
                 status_Displayevents (Song.Endmark - 18);
               end;
             end;
        #82: begin
               if Menu.Channelpos = 0 then Setevent (Song.Endmark, 0, 0, cmd_no);
               for x := Song.Maxlength - 1 downto Menu.Position do
                 Move (mem [Song.Tracks [x]: Menu.Channelpos * 4],
                   mem [Song.Tracks [x + 1]: Menu.Channelpos * 4], 4);
               Fillchar (mem [Song.Tracks [Menu.Position]: Menu.Channelpos * 4], 4, 0);
               if Menu.Channelpos = 0 then
                 if (Menu.Position <= Song.Endmark) and (Song.Endmark + 1 <= Song.Maxlength) then
                   Inc (Song.Endmark);
               if Menu.Channelpos = 0 then Setevent (Song.Endmark, 0, 0, cmd_end);
               status_Displayevents (Menu.Position - Menu.Line);
             end;
        #83: begin
               if Menu.Channelpos = 0 then Setevent (Song.Endmark, 0, 0, cmd_no);
               for x := Menu.Position + 1 to Song.Maxlength do
                 Move (mem [Song.Tracks [x]: Menu.Channelpos * 4],
                   mem [Song.Tracks [x - 1]: Menu.Channelpos * 4], 4);
               Fillchar (mem [Song.Tracks [Song.Maxlength]: Menu.Channelpos * 4], 4, 0);
               if Menu.Channelpos = 0 then
                 if (Menu.Position < Song.Endmark) and (Song.Endmark > 0) then
                   Dec (Song.Endmark);
               if Menu.Channelpos = 0 then Setevent (Song.Endmark, 0, 0, cmd_end);
               status_Displayevents (Menu.Position - Menu.Line);
             end;
        #59: mn_Help;
        #60: mn_Save;
        #61: mn_Load;
        #62: mn_Reset (true);
        #63: mn_Play (0);
        #64: mn_Play (Menu.Position);
        #65: MPU_stop;
        #66: mn_SelectInstrument;
        #67: mn_Speed;
        #1, #68: if not mn_Quit then r := #255;
        #78: if Song.Instrument < 15 then Inc (Song.Instrument) else Song.Instrument := 0;
        #74: if Song.Instrument > 0 then Dec (Song.Instrument) else Song.Instrument := 15;
        #23: begin
               Setevent (Song.Endmark, 0, 0, cmd_no);
               Song.Endmark := Menu.Position;
               Setevent (Menu.Position, 0, 0, cmd_end);
               status_Displayevents (Menu.Position - Menu.Line);
             end;
        #24: if Getevent (Menu.Position, Menu.Channelpos, 0) <> cmd_end then begin
               Setevent (Menu.Position, Menu.Channelpos, 0, cmd_off);
               status_Writeevent (Menu.Channelpos, Menu.Line, false, Menu.Position, Menu.Channelpos);
               Movedown;
             end;
        #25: if Getevent (Menu.Position, Menu.Channelpos, 0) <> cmd_end then begin
               Setevent (Menu.Position, Menu.Channelpos, 0, cmd_continue);
               status_Writeevent (Menu.Channelpos, Menu.Line, false, Menu.Position, Menu.Channelpos);
               Movedown;
             end;
        #14: if Getevent (Menu.Position, Menu.Channelpos, 0) <> cmd_end then
               Setevent (Menu.Position, Menu.Channelpos, 0, cmd_no);
        #16..#22: Song.Octave := ord (r) - 16;
        #2, #11: begin
                   y := 12;
                   if r = #11 then x := 48 else begin x := 49; y := 7; end;
                   out_Writexy (64, 2, $0f, chr (x) + ' ');
                   repeat
                     r := in_Readkey;
                   until ((r > #1) and (ord (r) < y)) or
                     (r = #1) or (r = #11);
                   if r <> #1 then begin
                     if r = #11 then r := #1;
                     x := (x - 48) * 10 + (ord (r) - 1);
                     Song.Velocity := x;
                     Setevent (Menu.Position, Menu.Channelpos, 3, x);
                   end;
                   r := #255;
                 end;
        #3..#10: begin
                   x := ord (r) - 3;
                   if Song.Active [x] then Song.Active [x] := false
                   else Song.Active [x] := true;
                   status_Displaychannels;
                 end;
        #44..#50, #31..#32, #34..#36:
          if Getevent (Menu.Position, Menu.Channelpos, 0) <> cmd_end then begin
            if Lastinst <> MPU_perchannel then
              MPU_playinstrument (Lastinst, Lastoct, Lastnote, 0)
            else
              MPU_note (9, Lastnote, 0);
            Setevent (Menu.Position, Menu.Channelpos, 0, cmd_note);
            Setevent (Menu.Position, Menu.Channelpos, 1, Song.Instrument);
            if Song.Instrument <> MPU_perchannel then begin
              for y := 1 to 12 do
                if MPU_keytonote [y] = ord (r) then x := y;
              Lastinst := Song.Instrument;
              Lastoct := Song.Octave;
              Lastnote := x;
              MPU_playinstrument (Lastinst, Lastoct, Lastnote, Song.Velocity shl 3);
              x := Song.Octave shl 4 + x;
              Setevent (Menu.Position, Menu.Channelpos, 2, x);
              Setevent (Menu.Position, Menu.Channelpos, 3, Song.Velocity);
            end else begin
              Setevent (Menu.Position, Menu.Channelpos, 2,
                Song.Instruments [Song.Instrument].inumber);
              Setevent (Menu.Position, Menu.Channelpos, 3, Song.Velocity);
              Lastinst := MPU_perchannel;
              Lastnote := MPU_percussion [Song.Instruments [Song.Instrument].inumber].nr;
              MPU_note (9, Lastnote, Song.Velocity shl 3);
            end;
            status_Writeevent (Menu.Channelpos, Menu.Line, false, Menu.Position, Menu.Channelpos);
            Movedown;
          end;
      end;
      status_Refreshinfo;
    until (r = #1) or (r = #68);
  end;

{ main program }

  begin
    mem_Init;
    MPU_reset;
    out_Initscreen;
    out_Clearscreen (true);
    status_Resetall (true);
    mn_Mainprogram;
    out_Closescreen;
    MPU_reset;
  end.