/* SockAPI.cpp -- Cross platform socket library         William Garrison a.k.a. Moby Disk
               -- Unfinished: 05/98                     mobydisk@home.com

  Copyright (C) 1998 William Garrison 
  This program is free software; you can redistribute it and/or modify it under the terms of
  the GNU General Public License as published by the Free Software Foundation; either version 
  2 of the License, or (at your option) any later version. 
  This program is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY; 
  without even the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. 
  See the GNU General Public License for more details. 
*/


#include "SockAPI.h"

// Static initializers for socket library
void Socket::init() {
#ifdef WIN32
  WSADATA BillGates;
  WSAStartup(MAKEWORD(2,0),&BillGates);
#endif
}

void Socket::done() {
#ifdef WIN32
  WSACleanup();
#endif
}

// Copy constructor: needed since streambuf cannot be copied -- so we construct a new one
Socket::Socket(Socket &copy) {
  sock      = copy.sock;
  remote    = copy.remote;
  local     = copy.local;
  connected = copy.connected;
  memcpy(_back,copy._back,sizeof(_back));
}

// Construct socket w/o connection
Socket::Socket() : streambuf(), connected(0) {
  sock=socket(PF_INET,SOCK_STREAM,IPPROTO_TCP);
}

// Construct socket from a SOCKET
Socket::Socket(SOCKET s, BOOL connFlag, int ignore) : streambuf(), connected(connFlag), sock(s) {}

// Create and connect to host by name
Socket::Socket(const char *hostname, int port) : streambuf(), connected(0) {
  sock=::socket(PF_INET,SOCK_STREAM,IPPROTO_TCP);
  connect(hostname,port);
}

// Create and connect to host by address
Socket::Socket(long hostaddr, int port) : streambuf(), connected(0) {
  sock=::socket(PF_INET,SOCK_STREAM,IPPROTO_TCP);
  connect(hostaddr,port);
}

// Create and begin serving on port
Socket::Socket(int port) : streambuf(), connected(0) {
  local.sin_family      = PF_INET;
  local.sin_port        = htons(port);
  local.sin_addr.s_addr = htonl(INADDR_ANY);

  sock=::socket(PF_INET,SOCK_STREAM,IPPROTO_TCP);
  ::bind(sock,(sockaddr *)&local,sizeof(local));
}

// Shutdown, disconnect, disappear
Socket::~Socket() {
  ::shutdown(sock,2);    ::closesocket(sock); 
  connected = 0;
}

// Close connection to client, create new socket(not shutdown)
void Socket::close() {
  connected=0;
  ::closesocket(sock);
  sock = socket(PF_INET,SOCK_STREAM,IPPROTO_TCP); // Re-open for re-use
}

// Server listen
int Socket::listen(int backlog) {
  return ::listen(sock,backlog);
}

// Server blocking accept
Socket *Socket::accept() {
  int socksize = sizeof(remote);
  SOCKET client = ::accept(sock,((sockaddr *)&remote),&socksize);
  if (client!=INVALID_SOCKET) {
    Socket *newConnect = new Socket(*this);
    newConnect->sock = client;
    newConnect->connected = TRUE;
    return newConnect;
  } else
    return NULL;
}

// Connect to host by name
int Socket::connect(const char *hostname, int port) {
  hostent *hn = gethostbyname(hostname);
  if (hn==NULL) return -1;

  long screwy_conv = *(unsigned long *)hn->h_addr;
  // delete hn; // Do not delete these
  return connect(ntohl(screwy_conv),port);
}

// Connect to host by address
int Socket::connect(long hostaddr, int port) {
  memset(&remote,0,sizeof(remote));
  remote.sin_family      = PF_INET;
  remote.sin_port        = htons(port);
  remote.sin_addr.s_addr = htonl(hostaddr);
  int result = ::connect(sock,(sockaddr *)&remote, sizeof(remote));
  connected = (result==0);
  return result;
}

// Get other end of connected socket
char * Socket::dotName(long addr)  { 
  in_addr inaddr;
  inaddr.S_un.S_addr = htonl(addr);
  return inet_ntoa(inaddr);
}
int   Socket::peerPort()    { return remote.sin_port; }
long  Socket::peerAddr()    { return ntohl((long)remote.sin_addr.S_un.S_addr); }
char *Socket::peerAddrStr() { return inet_ntoa(remote.sin_addr); }
char *Socket::peerName() {
  long addr = remote.sin_addr.S_un.S_addr;
  hostent *theName=gethostbyaddr((const char *)&addr,4/*sizeof(remote.sin_addr)*/,PF_INET);
  if (theName==NULL) return "Unknown";
  //delete theName; // Do not delete these
  return theName->h_name;
}
char *Socket::localName() {
  hostent *theName=gethostbyname("localhost");
  if (theName==NULL) return "localhost";
  return theName->h_name;
}

// Streaming functions: I have no clue how this works...

int Socket::overflow(int c) {
  int written;

// Unbuffered case
  if (unbuffered()) {
    if (c == EOF)          // Special case, this only flushes
      return 0;
    char ch = char(c);     // Write the byte directly
    written = send(sock,&ch,1,0);
    return (written) ? c : EOF;
  }

// Need to allocate a buffer
  if (!base()) allocate();

// Buffered case
  if (base()) {       // Test for memory allocation error
    char * ep = base() + (blen() / 2);
    if (!pbase())    // Set put pointers if not set up
        setp (base(), ep);
    int bytes = pptr() - pbase();   // Bytes to write
    if (bytes) {
      written = send(sock,pbase(),bytes,0);
      if (!written)
          return EOF;
/**************** OKAY BILL:
   One day, Netsuite crashed, for the 100th time.  THIS time, we caught it -- running in
   debug mode too!  And it was boatload that caused the problem!
   Okay, so we trace through.  it was client.write(buffer,512) that caused it.  So we trace
   to the memcpy below, and find the repz movsb is what did it.

   So, we notice that bytes += written, then we compare bytes to zero, then copy... what?
   Just look at it.  It makes no sense.
   So we changed bytes += written to bytes-=written.  That looks like it will move unsent
   bytes to the head of the buffer.  THAT makes sense.

   Why did this now always occurr?  Simple: Because USUALLY the whole buffer got transferred.
   But depending on Windows buffers, MTU, etc -- that may not happen.  Apparently, it was
   seldom.  This modification is done HERE and IN NETSUITE to see if it fixes things...
**********/
//      bytes += written;
      bytes -=written;
      if (bytes)  // Some is still waiting to be written
          memcpy (base(), base() + written, bytes);
    }
    setp (base() + bytes, ep);  // Reset 'put' pointers
    return (c == EOF) ? 0 : sputc (c);  // Put pending chr in buf
  }
  return EOF;
}

int Socket::underflow (void)
{
  int bytes;

// Handle an unbuffered stream
  if (unbuffered()) {
    bytes = recv(sock,&_back[1],1,0);
    if (!bytes) {
      setg (0, 0, 0);
      return EOF;
    }
    setg (_back, _back + 1, _back + 2);
    return (unsigned char)_back[1];
  }

// Need to allocate a buffer
  if (!base()) allocate();

// Handle a buffered stream
  if (base()) {
    char * gp = base() + blen() / 2;
    if (gptr() >= egptr()) {
                      // Read into the buffer from stream
      overflow ();    // Flush output in case we need it
      bytes = recv(sock,gp+1,blen()/2-1,0);
      if (bytes == SOCKET_ERROR)
        connected = 0;
      else
        setg (gp, gp + 1, gp + bytes + 1);
    }
    if (gptr() < egptr())   // Get from buffer
      return (unsigned char) *gptr();
  }
  return EOF;
}

int Socket::sync (void) {
  if (!unbuffered()) {
    overflow ();                // Force output
    char * gp = base();
    setp (gp, gp + blen() / 2);
    gp = base() + blen() / 2;
    setg (0, 0, 0);
  }
  return 0;
}
