/* ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
   LockAPI.cpp, Implementation for class Lockable and LockServer
   designed for Cross Platform API Library by Moby Disk
   ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~ */


#include "LockAPI.h"
#include <iostream.h>

LockServer * Lockable::locker = NULL;

// Initialization for Lockable creates a LockServer and spawn()s it.
void Lockable::init() {
  locker = new LockServer();
  locker->spawn();
}

// Cleanup for Lockable tells the server to end, Threadable will delete it for us
void Lockable::done(int threadID) {
  locker->done(threadID);
}

// Send a message to the lock server with this as the parameter
void Lockable::send(int command, int who, Lockable *object) {
  LockStruct message;
  message.command = command;
  message.count   = 1;
  message.who     = who;
  message.item[0] = object;
  locker->lockPipe.write((char *)&message,sizeof(message));
}

// Lock this
void Lockable::lock(int threadID) {
  send(LOCKAPI_LOCK,threadID,this);
  while (lockVal!=threadID) { Threadable::yield(); }
}

// Unlock this
void Lockable::unlock(int threadID) {
  send(LOCKAPI_UNLOCK,threadID,this);
}

// Lock multiple objects
void Lockable::lockMany(int threadID, Lockable *a, Lockable *b) {
  LockStruct message;
  message.command = LOCKAPI_LOCK;
  message.who     = threadID;
  message.count   = 2;
  message.item[0] = a;
  message.item[1] = b;
  locker->lockPipe.write((char *)&message,sizeof(message));
  while (a->lockVal!=threadID && b->lockVal!=threadID) { Threadable::yield(); }
}

// Shut down the lock server
void LockServer::done(int threadID) { Lockable::send(LOCKAPI_SHUTDOWN, threadID, NULL); }

// Add an item to the queue: returns 0 for success
int LockServer::queue(Lockable *toLock, int who) {
  if (lockTail>=LS_QUEUESIZE) return 1;
  lockList[lockTail].item = toLock;
  lockList[lockTail].who  = who;
  lockTail++;
  return 0;
}

// Scan for the specified item, and lock the first occurrance
int LockServer::scanlock(Lockable *toLock, int who) {
  for (int scan=0; scan<lockTail; scan++)
    if (lockList[scan].item==toLock) {
      lockList[scan].item->lockVal = who;
      memmove(&lockList[scan],&lockList[scan+1],sizeof(LockNode)*(lockTail-scan-1));
      lockTail--;
      return 0;
    }
  return 1;
}

// Main lock server body
void LockServer::main() {
  LockStruct message;
  lockTail = 0;
  cout << "Lockserver started" << endl;

  do {
// Wait for request
    lockPipe.read((char *)&message,sizeof(message));
    switch(message.command) {
// Lock n objects
      case LOCKAPI_LOCK:
        for (n=0; n<message.count; n++)
          if (message.item[n]->lockVal!=0)                   // Locked already? Queue request
            if (queue(message.item[n], message.who)==1)
              break;                                         // Queue failed
          else                                               // Unlocked?
            if (scanlock(message.item[n], message.who)==1);  // See if someone else wants it
              message.item[n]->lockVal = message.who;        // No? Then lock it
        break;
// Unlock n objects
      case LOCKAPI_UNLOCK:
        for (n=0; n<message.count; n++) {
          message.item[n]->lockVal = 0;                      // Unlock it
          scanlock(message.item[n],message.who);             // See if someone else wants it
        }
        break;
// Shutdown lock server
      case LOCKAPI_SHUTDOWN: 
        cout << "Lockserver received shutdown request" << endl;
        break;
      default:
        cout << "Lock server received unknown command: " << message.command << endl;
    }
  } while (lockTail<LS_QUEUESIZE && message.command!=LOCKAPI_SHUTDOWN);
// Cleanup
  if (lockTail>=LS_QUEUESIZE)
    cout << "Lockserver shutdown: Queue overflow(queuesize = " << LS_QUEUESIZE << ")" << endl;
}