/****
  PUNCH -- a DT105S paper tape punch machine control program
  Copyright (C) MXMII by L.Georgiev, Technical University, Varna

  You may freely use and distribute this source code
  alone or with the PUNCH.COM file (even .COM alone).
  If you intend to alter it, or you want to have some
  changes done, please call the author at TUV-207E:2.

#define CENTRONICS	/* Comment this if your DT105S is DEC interfaced */

#include <stdio.h>
#include <stdlib.h>
#include <conio.h>
#include <errno.h>
#include <fcntl.h>
#include <io.h>
#include <sys\types.h>
#include <sys\stat.h>
#include <ctype.h>
#include <time.h>

#define VER 100         /* The 1st bugfix version */

#define RWSTOP_DEFAULT  37
#define HOLES_PER_FOOT  120
#define HOLES_PER_METER 394
#define CHARS_PER_SECOND 150

#define LPT1_DATA   0x378
#define LPT1_STAT   0x379
#define LPT1_CTRL   0x37A

#define TIMEOUT     3000L   /* ms */

main( int argc, char *argv[] )
{
    enum BOOLEAN { NO, YES };

    enum BOOLEAN header = YES;
    char *buf;
    register int data;
    int infile, size, count, feet, meters, minutes, seconds, ch,
        rwstop = RWSTOP_DEFAULT, ones, percent, old_perc = 100;
    long len;
    clock_t ticks, old_tks;

    if( argc < 2 )  {
        printf("Usage: punch <drill file> [RWSTOP]\n");
        printf("(RWSTOP = rewind stop char in decimal, default 37)");
        return 1;
    }
    if( ( infile = open( argv[1], O_BINARY | O_RDONLY ) ) == -1 )   {
        printf("Can't open \"%s\"", argv[1] );
        return 2;
    }
    if( ( len = filelength( infile ) ) > 0x7FFFL )  {
        printf("\"%s\" is too long: split into smaller parts and try again",
            argv[1] );
        close( infile );
        return 3;
    }
    if( ( size = (int)len ) < HOLES_PER_METER )     {
        printf("\"%s\" is too short: could be entered manually",
            argv[1] );
        close( infile );
        return 4;
    }
    if( ( buf = (char *)malloc( (size_t)size ) ) == NULL )      {
        printf("Not enough memory");
        close( infile );
        return 5;
    }
    if( read( infile, (void *)buf, (unsigned)size ) != size )   {
        printf("Error reading \"%s\"", argv[1] );
        close( infile );
        return 6;
    }
    close( infile );
    if( argc == 3 )     /* RWSTOP specified */
        if( !isascii( rwstop = atoi( argv[2] ) ) )  /* Wrong RWSTOP in  */
            rwstop = RWSTOP_DEFAULT;                /* the command line */

    /*---------------- Add parity bit after RWSTOP char ----------------*/

    for( count = 0; count < size; count++ )     {
        data = (int)buf[ count ];
        if( data == rwstop )        /* Start of data */
            header = NO;
        if( !header )    {          /* If header, don't add parity bit */
            if( data & 0x80 )       /* Binary data after rwstop: error */
                break;
            ones = 0;
            while( data )   {
                ones += data & 1;   /* Count if bit 0 = 1 */
                data >>= 1;
            }
            if( ones & 1 )          /* Odd */
                buf[count] |= 0x80;
        }
    }
    if( count != size )  {  /* After break */
        printf("\"%s\" is not an ASCII drill file", argv[1] );
        return 7;
    }
    if( header )    {
        printf("RWSTOP char \'%c\' not found in \"%s\"", rwstop, argv[1] );
        return 8;
    }
    /*----------------- Banners, statistics etc. -----------------*/

    feet = size / HOLES_PER_FOOT + 1;
    meters = size / HOLES_PER_METER + 1;
    seconds = size / CHARS_PER_SECOND + 1;
    minutes = seconds / 60;

    printf("DT105S Paper Tape Punch  Version %u.%02u\n", VER/100, VER%100 );
    printf("Copyright (C) MXMII L.Georgiev, Varna\n");
#ifdef CENTRONICS
    printf("Interface type: Centronics\n\n");
#else
    printf("Interface type: DEC\n\n");
#endif
    printf("You need %d feet ( %d meters ) of tape.\n", feet, meters );
    printf("Punching will take ");
    if( minutes )
        printf("%d min %02d sec.", minutes, seconds % 60 );
    else
        printf("%d seconds.", seconds );
    printf("  Ready ?  ");
    ch = getche();
    if( ch != 'Y'  &&   ch != 'y' )
        return 9;
    printf("es\n\n");

    /*------------- Output data. The essential job (at last !) -------------*/

    for( count = 0; count < size; count++ )     {
        percent = (int)( (long)count * 100L / len );
        if( percent != old_perc )   {           /* Print only when percents */
            printf("\r%d%% done", percent );    /* are changed. This greatly*/
            old_perc = percent;                 /* speeds things up because */
        }                                       /* printf function is slow. */
	old_tks = clock();			    /* Reset "timer" */
#ifdef CENTRONICS
	while( !( inp( LPT1_STAT )  &  0x80 ) )     /*Wait while device busy*/
#else
	while( inp( LPT1_STAT )	 &  0x80 )
#endif
	{
            ticks = clock();
            if( ticks - old_tks > TIMEOUT )   {     /*Timeout period expired*/
                printf("\rDevice timeout.  Resume operation ?  ");
                ch = getche();
                if( ch != 'Y'  &&   ch != 'y' )
                    return 10;
                printf("es\n\n");
                old_tks = ticks = clock();          /*Give it another chance*/
            }
        }
	outp( LPT1_DATA, (int)buf[ count ] );	    /* Write data byte */
#ifdef CENTRONICS
	outp( LPT1_CTRL, inp( LPT1_CTRL )  |  1 );  /* Output strobe */
	outp( LPT1_CTRL, inp( LPT1_CTRL )  &  0xFE );
#else
	outp( LPT1_CTRL, inp( LPT1_CTRL )  &  0xFE );
	outp( LPT1_CTRL, inp( LPT1_CTRL )  |  1 );
#endif
    }                                /* NOTE: Busy and Strobe are inverted) */
    printf("\rPunch complete.");
    return 0;
}
