//======================================================================
//
// NotMyFault.c
//
// Copyright (C) 2002-2012 Mark Russinovich
// Sysinternals - www.sysinternals.com
//
// Simple interface to myfault device driver.
// 
//======================================================================
#include <windows.h>
#include <process.h>
#include <winioctl.h>
#include <shellapi.h>
#include <shlwapi.h>
#include <commdlg.h>
#include <prsht.h>
#include <stdio.h>
#include <stdlib.h>
#include <errno.h>
#include "resource.h"
#include "ioctlcmd.h"
#include "notmyfault.h"

COLORREF	g_BsodFg = RGB(0xFF,0xFF,0xFF);
COLORREF	g_BsodBg = RGB(0x00,0x00,0x80);

ULONGLONG	g_ulPoolTrack[2]; // Track pool allocations

static char	g_szPreview[] = "A problem has been detect\n"
						"to your computer.\n\n"
						"DRIVER_IRQL_NOT_LESS_OR_E\n\n"
						"If this is the first time\n"
						"restart your computer. If\n"
						"these steps:";

HFONT		g_hFont; // Used in the color preview window

#pragma comment(linker,"\"/manifestdependency:type='win32' name='Microsoft.Windows.Common-Controls' version='6.0.0.0' processorArchitecture='*' publicKeyToken='6595b64144ccf1df' language='*'\"")


//----------------------------------------------------------------------
//  
// Abort
//
// Exit with a fatal error.
//
//----------------------------------------------------------------------
LONG Abort( HWND hWnd, char * Msg, DWORD Error )
{
	LPVOID	lpMsgBuf;
	char	errmsg[MAX_PATH*2];
 
	UnloadDeviceDriver( SYS_NAME );
	FormatMessage( FORMAT_MESSAGE_ALLOCATE_BUFFER | FORMAT_MESSAGE_FROM_SYSTEM,
					NULL, Error, 
					MAKELANGID(LANG_NEUTRAL, SUBLANG_DEFAULT),
					(LPTSTR) &lpMsgBuf, 0, NULL );
	sprintf(errmsg, "%s %s", Msg, lpMsgBuf );
	if( (Error == ERROR_INVALID_HANDLE || Error == ERROR_ACCESS_DENIED ||
		 Error == ERROR_FILE_NOT_FOUND)) 
		 sprintf(errmsg, "%s\nVerify that you have administrative privileges. "
			"Make sure that NotMyFault is not already running and that you have "
			"the correct version that matches the architecture of your system.",
			errmsg  );
	MessageBox( hWnd, errmsg, "Not My Fault", MB_OK|MB_ICONERROR );
	EndDialog( hWnd, 1 );
	PostQuitMessage( 1 );
	LocalFree( lpMsgBuf );
	return (DWORD) -1;
}


//----------------------------------------------------------------------
//  
// UpdateLeak
// 
// Updates the leak information on the screen.
//
//----------------------------------------------------------------------
void UpdateLeak( HWND hDlg )
{
	int		i;
	char	szBuffer[64];
	char	szBytes[32];

	for ( i = 0; i < 2; i++ ) {

		if ( g_ulPoolTrack[i] ) {

			StrFormatKBSize( g_ulPoolTrack[i], szBytes, ARRAYSIZE( szBytes ) );
			sprintf( szBuffer, "%s of %s pool has been leaked.", szBytes,
					 i ? "nonpaged" : "paged" );

			SetDlgItemText ( hDlg, IDC_PAGED_STATUS + i, szBuffer );
		}
	}
}


//----------------------------------------------------------------------
//  
// DrawPreview
// 
// Draws the Preview window on the screen.
//
//----------------------------------------------------------------------
VOID DrawPreview( COLORREF newBg, COLORREF newFg, LPDRAWITEMSTRUCT drawItem )
{
	HDC		hdcMem;
	HBITMAP	hbmMem;
	HBRUSH	hBack;

	hdcMem = CreateCompatibleDC( drawItem->hDC );

	if (hdcMem) {

		hbmMem = CreateCompatibleBitmap( drawItem->hDC,
										 drawItem->rcItem.right,
										 drawItem->rcItem.bottom );

		if ( hbmMem ) {

			SelectObject( hdcMem, hbmMem );

			hBack = CreateSolidBrush( newBg );
			FillRect( hdcMem, &drawItem->rcItem, hBack );
			DeleteObject( hBack );

			if ( g_hFont ) {

				SelectObject( hdcMem, g_hFont );
			}

			SetTextColor( hdcMem, newFg );
			SetBkMode( hdcMem, TRANSPARENT );
			
			DrawText( hdcMem, g_szPreview, -1, &drawItem->rcItem, DT_NOCLIP );

			BitBlt( drawItem->hDC,
					0, 0,
					drawItem->rcItem.right, drawItem->rcItem.bottom,
					hdcMem,
					0, 0,
					SRCCOPY );

			DeleteObject( hbmMem );
		}

		DeleteDC( hdcMem );
	}
}


//----------------------------------------------------------------------
//  
// BsodColorsCallback
//
//----------------------------------------------------------------------
UINT_PTR CALLBACK BsodColorsCallback( HWND hDlg, UINT uiMsg, WPARAM wParam, LPARAM lParam )
{
	static COLORREF	newFg, newBg;
	static UINT		wm_setRgbString;
	HDC				hDC;
	LOGFONT			lpLf = { 0 };
	BOOL			bOK;
	BYTE			*rgbCurrent, bRGB;
	int				i;

	switch( uiMsg ) {
		case WM_INITDIALOG:
			newFg = g_BsodFg;
			newBg = g_BsodBg;
			wm_setRgbString  = RegisterWindowMessage( SETRGBSTRING ); 
			CheckRadioButton( hDlg, IDC_RADIOFG, IDC_RADIOBG, IDC_RADIOBG );
			hDC = GetDC ( hDlg );
			lpLf.lfHeight = -MulDiv( 10,
									 GetDeviceCaps( hDC, LOGPIXELSY ),
									 72 );
			ReleaseDC( hDlg, hDC );
			strcpy( lpLf.lfFaceName, "Lucida Console" );
			g_hFont = CreateFontIndirect( &lpLf );

			SetFocus( GetDlgItem( hDlg, IDOK ));
			break;

		case WM_DRAWITEM:
			if ( wParam == IDC_PREVIEW ) {
				DrawPreview( newBg, newFg, (LPDRAWITEMSTRUCT)lParam );
			}
			break;

		case WM_CTLCOLORSTATIC:
			if ( (HWND) lParam == GetDlgItem(hDlg,709) ) {

				rgbCurrent = IsDlgButtonChecked( hDlg, IDC_RADIOBG ) ?
					(BYTE *)&newBg : (BYTE *)&newFg;

				for ( i = 0 ; i < 3 ; i++ ) {

					bRGB = (BYTE)GetDlgItemInt( hDlg, 706 + i, &bOK, FALSE );

					if (bOK) {
						*rgbCurrent++ = (BYTE)bRGB;
					}
				}

				InvalidateRect( GetDlgItem(hDlg,IDC_PREVIEW), NULL, FALSE );
			}
			break;

		case WM_DESTROY:
			if ( g_hFont ) {
				DeleteObject( g_hFont );
			}
			break;

		case WM_COMMAND:
			switch ( LOWORD( wParam ) ) {
			case IDC_RADIOBG:
				SendMessage( hDlg, wm_setRgbString, 0, newBg );
				break;

			case IDC_RADIOFG:
				SendMessage( hDlg, wm_setRgbString, 0, newFg );
				break;

			case IDOK:
				g_BsodFg = newFg;
				g_BsodBg = newBg;
				PostMessage( hDlg, WM_COMMAND, IDABORT, 1 );
				return FALSE;
			}
			break;
	}
	return 0;
}


//----------------------------------------------------------------------
//  
//  StartMyFaultDriver
//
// Loads and starts the driver.
//
//----------------------------------------------------------------------
LONG StartMyFaultDriver( HWND hDlg )
{
	char	driverPath[MAX_PATH];
	char	systemRoot[MAX_PATH];
	char	path[MAX_PATH];
	WIN32_FIND_DATA findData;
	HANDLE	findHandle;
	char	*file;
	DWORD	error;
	char	msgbuf[MAX_PATH*2];

	//
	// Load the myfault driver
	//
	GetCurrentDirectory( sizeof path, path );
	sprintf( path+lstrlen(path), "\\%s", SYS_FILE );

	findHandle = FindFirstFile( path, &findData );
	if( findHandle == INVALID_HANDLE_VALUE ) {

		if( !SearchPath( NULL, SYS_FILE, NULL, sizeof(path), path, &file ) ) {

			sprintf( msgbuf, "The file %s was not found.", SYS_FILE );
			return Abort( hDlg, msgbuf, GetLastError() );
		}

	} else FindClose( findHandle );

	if( !GetEnvironmentVariable( "SYSTEMROOT", systemRoot, sizeof(systemRoot))) {

		strcpy( msgbuf, "Could not resolve the SYSTEMROOT environment variable" );
		return Abort( hDlg, msgbuf, GetLastError() );
	}
	sprintf( driverPath, "%s\\system32\\drivers\\myfault.sys", systemRoot );
	SetFileAttributes( driverPath, FILE_ATTRIBUTE_NORMAL );
	CopyFile( path, driverPath, FALSE );
	if( !LoadDeviceDriver( SYS_NAME, driverPath, &SysHandle, &error )) {

		if( !CopyFile( path, driverPath, FALSE )) {

			sprintf( msgbuf, "Unable to copy the file %s. "
				"Make sure that %s is in the current directory.", 
				SYS_FILE, SYS_FILE );
			return Abort( hDlg, msgbuf, GetLastError() );
		}
		SetFileAttributes( driverPath, FILE_ATTRIBUTE_NORMAL );
		if( !LoadDeviceDriver( SYS_NAME, driverPath, &SysHandle, &error ) )  {

			UnloadDeviceDriver( SYS_NAME );
			if( !LoadDeviceDriver( SYS_NAME, driverPath, &SysHandle, &error ) )  {

				sprintf( msgbuf, "Can't load %s.", SYS_FILE );
				DeleteFile( driverPath );
				return Abort( hDlg, msgbuf, error );
			}
		}
	}
	return TRUE;
}

//----------------------------------------------------------------------
//  
// IoctlThreadProc
//
//----------------------------------------------------------------------
void IoctlThreadProc( PVOID Context )
{
	DWORD nb;
	DeviceIoControl(SysHandle, (DWORD)(ULONG_PTR) Context, NULL, 0, NULL, 0, &nb, NULL );
}


//---------------------------------------------------------------------
//  
// LeakPool
//
//---------------------------------------------------------------------
void LeakPool( UINT PoolType, DWORD allocSize ) 
{
	DWORD	maxAlloc, bytesAllocated, nb;
	DWORD	tickCount = GetTickCount();

	maxAlloc = allocSize;
	bytesAllocated = 0;
	while( bytesAllocated < maxAlloc && tickCount - GetTickCount() < 1000 ) {
		
		if( !DeviceIoControl(SysHandle, 
				PoolType ? IOCTL_LEAK_NONPAGED:IOCTL_LEAK_PAGED, &allocSize, sizeof(allocSize), 
				NULL, 0, &nb, NULL )) {

			// can't even allocate 1 byte
			if( allocSize == 1 ) break;

			allocSize /= 2;
			if( allocSize == 0 ) allocSize = 1;

		} else {

			bytesAllocated += allocSize;
		}
	}

	// one more try going from 2 to 8192
	if( bytesAllocated < maxAlloc ) {

		allocSize = 8192;
		while( allocSize > 1 && bytesAllocated < maxAlloc 
			&& tickCount - GetTickCount() < 1000 ) {
			
			while( DeviceIoControl(SysHandle, 
				PoolType ? IOCTL_LEAK_NONPAGED:IOCTL_LEAK_PAGED, &allocSize, sizeof(allocSize), 
				NULL, 0, &nb, NULL ) && bytesAllocated < maxAlloc ) {

				bytesAllocated += allocSize;
			}
			allocSize /= 2;
		}
	}

	if ( bytesAllocated ) {
		g_ulPoolTrack[PoolType] += bytesAllocated;
	}
}


//----------------------------------------------------------------------
//  
// CrashDialogProc
//
// This is the Crash property sheet page. 
//
//----------------------------------------------------------------------
INT_PTR CALLBACK CrashDialogProc( HWND hDlg, UINT message, WPARAM wParam,
                       		LPARAM lParam ) 
{
	DWORD	nb;
	DWORD	ioctl = 0;
	CHOOSECOLOR colorArgs;
	
	switch (message) {
	case WM_INITDIALOG :

		CheckDlgButton( hDlg, IDC_IRQL, BST_CHECKED );
		break;

	case WM_COMMAND :
		switch (LOWORD( wParam )) {
	
		case IDC_CRASH_OK :

			if( IsDlgButtonChecked( hDlg, IDC_BUFFEROVERFLOW ) == BST_CHECKED ) {

				ioctl = IOCTL_BUFFER_OVERFLOW;
				
			} else if( IsDlgButtonChecked( hDlg, IDC_WILDPOINTER ) == BST_CHECKED ) {

				ioctl = IOCTL_WILD_POINTER;

			} else if( IsDlgButtonChecked( hDlg, IDC_STACKTRASH ) == BST_CHECKED ) {

				ioctl = IOCTL_TRASH_STACK;

			} else if( IsDlgButtonChecked( hDlg, IDC_PAGEFAULT ) == BST_CHECKED ) {

				ioctl = IOCTL_PAGE_FAULT;

			} else if( IsDlgButtonChecked( hDlg, IDC_IRQL ) == BST_CHECKED ) {

				ioctl = IOCTL_IRQL;
			
			} else if( IsDlgButtonChecked( hDlg, IDC_STACKOVERFLOW ) == BST_CHECKED ) {

				ioctl = IOCTL_STACK_OVERFLOW;

			} else if( IsDlgButtonChecked( hDlg, IDC_BREAKPOINT ) == BST_CHECKED ) {

				ioctl = IOCTL_HARD_BREAKPOINT;

			} else if( IsDlgButtonChecked( hDlg, IDC_DOUBLEFREE ) == BST_CHECKED ) {

				ioctl = IOCTL_DOUBLE_FREE;

			}

			DeviceIoControl(SysHandle, ioctl, NULL, 0, NULL, 0, &nb, NULL );
			break;

		case IDC_COLOR: {

			COLORREF	CustomColors[ 16 ];
			int i;
			for ( i = 0; i < 16; i++ )  {
				CustomColors[i] = RGB( 255, 255, 255 );
			}
			colorArgs.lStructSize	= sizeof colorArgs;
			colorArgs.Flags			= CC_RGBINIT|CC_ENABLEHOOK|CC_ENABLETEMPLATE|CC_FULLOPEN;
			colorArgs.hwndOwner		= hDlg;
			colorArgs.hInstance		= (HWND)GetModuleHandle(NULL);
			colorArgs.rgbResult		= g_BsodBg;
			colorArgs.lpCustColors	= CustomColors;
			colorArgs.lCustData		= 0;
			colorArgs.lpTemplateName = "BSODCOLORS";
			colorArgs.lpfnHook		= BsodColorsCallback;
			if( ChooseColor( &colorArgs ) == TRUE ) {

				LARGE_INTEGER Color;
				Color.LowPart = RGB( GetRValue(g_BsodBg)/4,
								 GetGValue(g_BsodBg)/4,
								 GetBValue(g_BsodBg)/4 );
				Color.HighPart = RGB( GetRValue(g_BsodFg)/4,
								 GetGValue(g_BsodFg)/4,
								 GetBValue(g_BsodFg)/4 );
				DeviceIoControl(SysHandle, IOCTL_BSOD_COLOR, &Color, sizeof(LARGE_INTEGER), NULL, 0, &nb, NULL );
			}
			}
			break;
		}
		break;

	default:

		return FALSE;
	}

	return TRUE;
}

//----------------------------------------------------------------------
//  
// HangDialogProc
//
// This is the Hang property sheet page. 
//
//----------------------------------------------------------------------
INT_PTR CALLBACK HangDialogProc( HWND hDlg, UINT message, WPARAM wParam,
                       		LPARAM lParam ) 
{
	SYSTEM_INFO sysInfo;
	DWORD	ioctl = 0;
	DWORD	i, nb;

	switch (message) {
	case WM_INITDIALOG :

		CheckDlgButton( hDlg, IDC_HANGIRP, BST_CHECKED );
		break;

	case WM_COMMAND :

		switch (LOWORD( wParam )) {
	
		case IDC_HANG_OK :

			if( IsDlgButtonChecked( hDlg, IDC_DEADLOCK ) == BST_CHECKED ) {

				ioctl = IOCTL_DEADLOCK;

			} else if( IsDlgButtonChecked( hDlg, IDC_HANG ) == BST_CHECKED ) {

				ioctl = IOCTL_HANG;

			} else if( IsDlgButtonChecked( hDlg, IDC_HANGIRP ) == BST_CHECKED ) {

				_beginthread( IoctlThreadProc, 0, (PVOID) IOCTL_HANG_IRP );
				break;
			}

			//
			// Execute hang and deadlock on each CPU
			//
			GetSystemInfo( &sysInfo );
			for( i = 0; i < sysInfo.dwNumberOfProcessors; i++ ) {

				DeviceIoControl(SysHandle, ioctl, NULL, 0, NULL, 0, &nb, NULL );
			}
			break;
		}
		break;

	default:

		return FALSE;
	}

	return TRUE;
}

//----------------------------------------------------------------------
//  
// LeakDialogProc
//
// This is the Leak property sheet page. 
//
//----------------------------------------------------------------------
INT_PTR CALLBACK LeakDialogProc( HWND hDlg, UINT message, WPARAM wParam,
                       		LPARAM lParam ) 
{
	char	label[MAX_PATH];
	DWORD	allocSize, maxAlloc;
	static BOOLEAN	leakPaged = FALSE;
	static BOOLEAN	leakNonpaged = FALSE;

	switch (message) {
	case WM_INITDIALOG :

		CheckDlgButton( hDlg, IDC_LEAK_PAGE, BST_CHECKED );
		SetDlgItemText( hDlg, IDC_LEAKMB, "1000" );
		break;

	case WM_TIMER:

		GetDlgItemText( hDlg, IDC_LEAKMB, label, _countof(label));
		allocSize = maxAlloc = (atoi( label ) * 1024);
		LeakPool( (UINT)wParam, allocSize );
		UpdateLeak( hDlg );
		break;

	case WM_COMMAND :

		switch (LOWORD( wParam )) {

		case IDC_LEAK_PAGE:

			if( leakPaged ) {

				KillTimer( hDlg, 0 );
				SetDlgItemText( hDlg, IDC_LEAK_PAGE, "Leak &Paged" );
			} else {
 
				SetTimer( hDlg, 0, 1000, NULL );
				SetDlgItemText( hDlg, IDC_LEAK_PAGE, "Stop &Paged" );
			}			
			leakPaged = !leakPaged;
			break;

		case IDC_LEAK_NONPAGE:

			if( leakNonpaged ) {

				KillTimer( hDlg, 1 );
				SetDlgItemText( hDlg, IDC_LEAK_NONPAGE, "Leak &Nonpaged" );
			} else {
 
				SetTimer( hDlg, 1, 1000, NULL );
				SetDlgItemText( hDlg, IDC_LEAK_NONPAGE, "Stop &Nonpaged" );
			}			
			leakNonpaged = !leakNonpaged;
			break;

		}
		break;

	default:

		return FALSE;
	}

	return TRUE;
}

//----------------------------------------------------------------------
//  
// MainProc
//
// This is the main window proc. 
//
//----------------------------------------------------------------------
int CALLBACK MainProc( HWND hDlg, UINT message, LPARAM lParam )
{
	switch (message) {
	case PSCB_INITIALIZED:

		//
		// Start driver
		//
		if( !StartMyFaultDriver( hDlg )) {

			return FALSE;
		}

		ShowWindow( GetDlgItem( hDlg, IDOK ), SW_HIDE );
		break;

	}
	return 0;
}

//----------------------------------------------------------------------
//
// WinMain
//
// Initialize a propery sheet and pop it.
//
//----------------------------------------------------------------------
int WINAPI WinMain(	HINSTANCE hInstance, 
				   HINSTANCE hPrevInstance,	
				   LPSTR lpCmdLine,
				   int nCmdShow )
{
	static TCHAR	szAppName [] = TEXT("Not My Fault") ;
	PROPSHEETHEADER	psh = { 0 };
	PROPSHEETPAGE	psp[3] = { 0 };
	PWSTR			*cmdLine;
	int				numArgs;
	DWORD			nb;
	BOOL			fUsage = FALSE;
	DWORD			ioctl = IOCTL_IRQL;
	DWORD			ioctlBuf = 0;
	PWSTR			endPtr;

	cmdLine = CommandLineToArgvW( GetCommandLineW(), &numArgs );
	if ( numArgs > 1 && ( cmdLine[1][0] == '/' ||
						  cmdLine[1][0] == '-' ) ) {

		if ( numArgs == 3 && !_wcsicmp( &cmdLine[1][1], L"bugcheck" ) ) {

			errno = 0;
			ioctlBuf = wcstoul( cmdLine[2], &endPtr, 16 );
			ioctl = IOCTL_BUG_CHECK;

			if ( errno == ERANGE || cmdLine[2] == endPtr ) {
				fUsage = TRUE;
			}
		} else if ( numArgs != 2 || _wcsicmp( &cmdLine[1][1], L"crash" ) ) {
			fUsage = TRUE;
		}
		
		if (!fUsage) {

			if( StartMyFaultDriver( NULL )) {

				DeviceIoControl(SysHandle,
					ioctl, &ioctlBuf, sizeof(ioctlBuf), NULL, 0, &nb, NULL );
			}
		} else {

			MessageBox( NULL,
						"notmyfault [/crash] [/bugcheck <code>]\n\n"
						"/crash\t\t"       "Crashes the system with a high IRQL fault\n\n"
						"/bugcheck code\t" "Crashes the system with the specified stop code\n"
						"\t\t"             "e.g. notmyfault /bugcheck 0xe2",
						szAppName,
						MB_OK );
			return -1;
		}
	}

	psp[0].dwSize		= sizeof(PROPSHEETPAGE);
	psp[0].hInstance	= hInstance;
	psp[0].pszTemplate	= "CRASH";
	psp[0].pfnDlgProc	= CrashDialogProc;

	psp[1].dwSize		= sizeof(PROPSHEETPAGE);
	psp[1].hInstance	= hInstance;
	psp[1].pszTemplate	= "HANG";
	psp[1].pfnDlgProc	= HangDialogProc;

	psp[2].dwSize		= sizeof(PROPSHEETPAGE);
	psp[2].hInstance	= hInstance;
	psp[2].pszTemplate	= "LEAK";
	psp[2].pfnDlgProc	= LeakDialogProc;

	psh.dwSize			= sizeof(PROPSHEETHEADER);
	psh.dwFlags			= PSH_PROPSHEETPAGE | PSH_USECALLBACK | PSH_NOCONTEXTHELP | PSH_NOAPPLYNOW;
	psh.hInstance		= hInstance;
	psh.pszCaption		= (LPCSTR) szAppName;
	psh.nPages			= sizeof(psp) / sizeof(PROPSHEETPAGE);
	psh.ppsp			= (LPCPROPSHEETPAGE) &psp;
	psh.pfnCallback		= MainProc;

	return (int) PropertySheet( &psh );
}
