unit vDither;

interface
Uses Windows,Classes,SysUtils;
var
   SystemPalette:array[0..15] of TRGBQuad;

procedure dither (Info:PBitMapInfo;
		     input_buf : Pointer;
                     output_buf: Pointer;
                     NumberColors:Integer);


implementation
//Uses JpegLib, jmoreCfg,jInclude,jMemMgr;
const
  BITS_IN_JSAMPLE = 8;

{$IFDEF WIN32}
{ Delphi 2.0 }
type
  int = Integer;
  uInt = Cardinal;
  short = SmallInt;
  ushort = Word;
  long = longint;
{$ELSE}
  {$IFDEF VIRTUALPASCAL}
  type
    int = longint;
    uInt = longint; { unsigned int }
    short = system.Integer;
    ushort = system.Word;
    long = longint;
  {$ELSE}
  type
    int = Integer;
    uInt = Word; { unsigned int }
    short = Integer;
    ushort = Word;
    long = longint;
  {$ENDIF}
{$ENDIF}

type
  JSAMPLE = byte; { Pascal unsigned char }
  GETJSAMPLE = int;

const
  MAXJSAMPLE = 255;
  CENTERJSAMPLE = 128;

{const
  BITS_IN_JSAMPLE = 8;
  MAXJSAMPLE = 255;

type
  JSAMPLE = BYTE;
  int = smallint;}
  
const
  R_SCALE = 2;          { scale R distances by this much }
  G_SCALE = 3;          { scale G distances by this much }
  B_SCALE = 1;          { and B by this much }

const
  C0_SCALE = R_SCALE;
  C1_SCALE = G_SCALE;
  C2_SCALE = B_SCALE;

const
  MAXNUMCOLORS = 256;        { maximum size of colormap }

const
  HIST_C0_BITS = 5;             { bits of precision in R/B histogram }
  HIST_C1_BITS = 6;             { bits of precision in G histogram }
  HIST_C2_BITS = 5;             { bits of precision in B/R histogram }

{ Number of elements along histogram axes. }
const
  HIST_C0_ELEMS = (1 shl HIST_C0_BITS);
  HIST_C1_ELEMS = (1 shl HIST_C1_BITS);
  HIST_C2_ELEMS = (1 shl HIST_C2_BITS);

{ These are the amounts to shift an input value to get a histogram index. }
const
  C0_SHIFT = (BITS_IN_JSAMPLE-HIST_C0_BITS);
  C1_SHIFT = (BITS_IN_JSAMPLE-HIST_C1_BITS);
  C2_SHIFT = (BITS_IN_JSAMPLE-HIST_C2_BITS);

const
  BOX_C0_LOG = (HIST_C0_BITS-3);
  BOX_C1_LOG = (HIST_C1_BITS-3);
  BOX_C2_LOG = (HIST_C2_BITS-3);

  BOX_C0_ELEMS = (1 shl BOX_C0_LOG); { # of hist cells in update box }
  BOX_C1_ELEMS = (1 shl BOX_C1_LOG);
  BOX_C2_ELEMS = (1 shl BOX_C2_LOG);

  BOX_C0_SHIFT = (C0_SHIFT + BOX_C0_LOG);
  BOX_C1_SHIFT = (C1_SHIFT + BOX_C1_LOG);
  BOX_C2_SHIFT = (C2_SHIFT + BOX_C2_LOG);

const
  DCTSIZE             = 8;      { The basic DCT block is 8x8 samples }
  DCTSIZE2            = 64;     { DCTSIZE squared; # of elements in a block }
  NUM_QUANT_TBLS      = 4;      { Quantization tables are numbered 0..3 }
  NUM_HUFF_TBLS       = 4;      { Huffman tables are numbered 0..3 }
  NUM_ARITH_TBLS      = 16;     { Arith-coding tables are numbered 0..15 }
  MAX_COMPS_IN_SCAN   = 4;      { JPEG limit on # of components in one scan }
  MAX_SAMP_FACTOR     = 4;      { JPEG limit on sampling factors }
{ Unfortunately, some bozo at Adobe saw no reason to be bound by the standard;
  the PostScript DCT filter can emit files with many more than 10 blocks/MCU.
  If you happen to run across such a file, you can up D_MAX_BLOCKS_IN_MCU
  to handle it.  We even let you do this from the jconfig.h file.  However,
  we strongly discourage changing C_MAX_BLOCKS_IN_MCU; just because Adobe
  sometimes emits noncompliant files doesn't mean you should too. }
  C_MAX_BLOCKS_IN_MCU = 10;     { compressor's limit on blocks per MCU }
  D_MAX_BLOCKS_IN_MCU = 10;     { decompressor's limit on blocks per MCU }

type
  JCOEF = int;
  JCOEF_PTR = ^JCOEF;

type
{ for typecasting }
  JSAMPLE_PTR = ^JSAMPLE;
  JSAMPROW_PTR = ^JSAMPROW;
  JBLOCKROW_PTR = ^JBLOCKROW;

  jTSample = 0..(MaxInt div SIZEOF(JSAMPLE))-1;
  JSAMPLE_ARRAY = Array[jTSample] of JSAMPLE;  {far}
  JSAMPROW = ^JSAMPLE_ARRAY;  { ptr to one image row of pixel samples. }

  jTRow = 0..(MaxInt div SIZEOF(JSAMPROW))-1;
  JSAMPROW_ARRAY = Array[jTRow] of JSAMPROW;
  JSAMPARRAY = ^JSAMPROW_ARRAY;  { ptr to some rows (a 2-D sample array) }

  jTArray = 0..(MaxInt div SIZEOF(JSAMPARRAY))-1;
  JSAMP_ARRAY = Array[jTArray] of JSAMPARRAY;
  JSAMPIMAGE = ^JSAMP_ARRAY;  { a 3-D sample array: top index is color }

  JBLOCK = Array[0..DCTSIZE2-1] of JCOEF;   { one block of coefficients }
  JBLOCK_PTR = ^JBLOCK;

  jTBlockRow = 0..(MaxInt div SIZEOF(JBLOCK))-1;
  JBLOCK_ROWS = Array[jTBlockRow] of JBLOCK;
  JBLOCKROW = ^JBLOCK_ROWS; {far} { pointer to one row of coefficient blocks }


  jTBlockArray = 0..(MaxInt div SIZEOF(JBLOCKROW))-1;
  JBLOCK_ARRAY = Array[jTBlockArray] of JBLOCKROW;
  JBLOCKARRAY = ^JBLOCK_ARRAY;    { a 2-D array of coefficient blocks }

  jTBlockImage = 0..(MaxInt div SIZEOF(JBLOCKARRAY))-1;
  JBLOCK_IMAGE = Array[jTBlockImage] of JBLOCKARRAY;
  JBLOCKIMAGE = ^JBLOCK_IMAGE;   { a 3-D array of coefficient blocks }

  jTCoef = 0..(MaxInt div SIZEOF(JCOEF))-1;
  JCOEF_ROW = Array[jTCoef] of JCOEF;
  JCOEFPTR = ^JCOEF_ROW; {far}   { useful in a couple of places }


type
  jTByte = 0..(MaxInt div SIZEOF(byte))-1;
  JByteArray = Array[jTByte] of byte;
  JBytePtr = ^JByteArray;
type
  byteptr = ^byte;


type                            { Nomssi }
  RGBptr = ^RGBtype;
  RGBtype = packed record
    r,g,b : JSAMPLE;
  end;

type
  histcell = WORD;            { histogram cell; prefer an unsigned type }

type
  histptr = ^histcell {FAR};       { for pointers to histogram cells }

type
  hist1d = array[0..HIST_C2_ELEMS-1] of histcell; { typedefs for the array }

  hist1d_field = array[0..HIST_C1_ELEMS-1] of hist1d;
                                  { type for the 2nd-level pointers }
  hist2d = ^hist1d_field;
  hist2d_field = array[0..HIST_C0_ELEMS-1] of hist2d;
  hist3d = ^hist2d_field;	  { type for top-level pointer }

type
  FSERROR = smallint;              { 16 bits should be enough }
  LOCFSERROR = int;             { use 'int' for calculation temps }

type                            { Nomssi }
  RGB_FSERROR_PTR = ^RGB_FSERROR;
  RGB_FSERROR = packed record
    r,g,b : FSERROR;
  end;
  LOCRGB_FSERROR = packed record
    r,g,b : LOCFSERROR;
  end;

type
  FSERROR_PTR = ^FSERROR;
  jFSError = 0..(MaxInt div SIZEOF(RGB_FSERROR))-1;
  FS_ERROR_FIELD = array[jFSError] of RGB_FSERROR;
  FS_ERROR_FIELD_PTR = ^FS_ERROR_FIELD;{far}
                                { pointer to error array (in FAR storage!) }

type
  error_limit_array = array[-MAXJSAMPLE..MAXJSAMPLE] of int;
  { table for clamping the applied error }
  error_limit_ptr = ^error_limit_array;


{ Private subobject }
type
  my_cquantize_ptr = ^my_cquantizer;
  my_cquantizer = record
//    pub : jpeg_color_quantizer; { public fields }

    { Space for the eventually created colormap is stashed here }
    sv_colormap : JSAMPARRAY;	{ colormap allocated at init time }
    desired : int;              { desired # of colors = size of colormap }

    { Variables for accumulating image statistics }
    histogram : hist3d;         { pointer to the histogram }

    needs_zeroed : boolean;     { TRUE if next pass must zero histogram }

    { Variables for Floyd-Steinberg dithering }
    fserrors : FS_ERROR_FIELD_PTR;        { accumulated errors }
    on_odd_row : boolean;       { flag to remember which row we are on }
    error_limiter : error_limit_ptr; { table for clamping the applied error }
  end;
type
  INT32 = longint;
type
  INT32PTR = ^INT32;

type
  box = record
  { The bounds of the box (inclusive); expressed as histogram indexes }
    c0min, c0max : int;
    c1min, c1max : int;
    c2min, c2max : int;
    { The volume (actually 2-norm) of the box }
    volume : INT32;
    { The number of nonzero histogram cells within this box }
    colorcount : long;
  end;

type
  jBoxList = 0..(MaxInt div SizeOf(box))-1;
  box_field = array[jBoxlist] of box;
  boxlistptr = ^box_field;
  boxptr = ^box;

  range_limit_table = array[-(MAXJSAMPLE+1)..(4*(MAXJSAMPLE+1)+ CENTERJSAMPLE -1)] of JSAMPLE;
  range_limit_table_ptr = ^range_limit_table;

type
  int_ptr = ^int;
  size_t = int;

type
  JDIMENSION = uInt;

const
  JPEG_MAX_DIMENSION = 65500;  { a tad under 64K to prevent overflows }

var
  Desired_number_colors:Integer;
  actual_number_Colors:Integer;
  colormap : JSAMPARRAY;

  Header:TBitMapInfoHeader;
  LocalQuant:my_cquantize_ptr;
  range_limit:range_limit_table_ptr;

procedure MEMZERO(target : pointer; size : size_t);
begin
  FillChar(target^, size, 0);
end;

procedure MEMCOPY(dest, src : pointer; size : size_t);
begin
  Move(src^, dest^, size);
end;


procedure prepare_range_limit_table;
{ Allocate and fill in the sample_range_limit table }
var
  table : range_limit_table_ptr;
  idct_table : JSAMPROW;
  i : int;
begin
  New(table);
//  GetMem(table,(5 * (MAXJSAMPLE+1) + CENTERJSAMPLE) * SIZEOF(JSAMPLE) );

  { First segment of "simple" table: limit[x] := 0 for x < 0 }
  FillChar(table^, (MAXJSAMPLE+1) * SIZEOF(JSAMPLE), #0);

  range_limit := (table);
  { allow negative subscripts of simple table }
  { is noop, handled via type definition (Nomssi) }
  { Main part of "simple" table: limit[x] := x }
  for i := 0 to MAXJSAMPLE do
    table^[i] := JSAMPLE (i);
  idct_table := JSAMPROW(@ table^[CENTERJSAMPLE]);
                        { Point to where post-IDCT table starts }
  { End of simple table, rest of first half of post-IDCT table }
  for i := CENTERJSAMPLE to pred(2*(MAXJSAMPLE+1)) do
    idct_table^[i] := MAXJSAMPLE;
  { Second half of post-IDCT table }
  MEMZERO(@(idct_table^[2 * (MAXJSAMPLE+1)]),
	  (2 * (MAXJSAMPLE+1) - CENTERJSAMPLE) * SIZEOF(JSAMPLE));
  MemCopy(@(idct_table^[(4 * (MAXJSAMPLE+1) - CENTERJSAMPLE)]),
	  @range_limit^[0], CENTERJSAMPLE * SIZEOF(JSAMPLE));

end;



procedure init_error_limit ;
const
  STEPSIZE = ((MAXJSAMPLE+1) div 16);
{ Allocate and fill in the error_limiter table }
var
  table : error_limit_ptr;
  inp, out : int;
begin
  GetMem(table, (MAXJSAMPLE*2+1) * SIZEOF(int));
  { not needed: Inc(table, MAXJSAMPLE);
                so can index -MAXJSAMPLE .. +MAXJSAMPLE }
  LocalQuant^.error_limiter := table;
  { Map errors 1:1 up to +- MAXJSAMPLE/16 }
  out := 0;
  for inp := 0 to pred(STEPSIZE) do
  begin
    table^[inp] := out;
    table^[-inp] := -out;
    Inc(out);
  end;
  { Map errors 1:2 up to +- 3*MAXJSAMPLE/16 }
  inp := STEPSIZE;       { Nomssi: avoid problems with Delphi2 optimizer }
  while (inp < STEPSIZE*3) do
  begin
    table^[inp] := out;
    table^[-inp] := -out;
    Inc(inp);
    if Odd(inp) then
      Inc(out);
  end;
  { Clamp the rest to final out value (which is (MAXJSAMPLE+1)/8) }
  inp := STEPSIZE*3;     { Nomssi: avoid problems with Delphi 2 optimizer }
  while inp <= MAXJSAMPLE do
  begin
    table^[inp] := out;
    table^[-inp] := -out;
    Inc(inp);
  end;
end;


procedure start_pass_2_quant (is_pre_scan : boolean); far;
var
  histogram : hist3d;
  i : int;
var
  arraysize : size_t;
begin
  histogram := LocalQuant^.histogram;
  { Only F-S dithering or no dithering is supported. }
  { If user asks for ordered dither, give him F-S. }
  { Set up method pointers }
//  LocalQuant^.pub.color_quantize := pass2_fs_dither

//  LocalQuant^.pub.finish_pass := nil;

  if not is_pre_scan then
  begin
    { Make sure color count is acceptable }

    arraysize := size_t ((Header.biWidth + 2) *
                                 (3 * SIZEOF(FSERROR)));
    { Allocate Floyd-Steinberg workspace if we didn't already. }
    if (LocalQuant^.fserrors = NIL) then
      GetMem(LocalQuant^.fserrors,arraysize);

    { Initialize the propagated errors to zero. }
    FillChar(LocalQuant^.fserrors^, arraysize,#0);

    { Make the error-limit table if we didn't already. }
    if (LocalQuant^.error_limiter = NIL) then
      init_error_limit;
    LocalQuant^.on_odd_row := FALSE;
  end;

  { Zero the histogram or inverse color map, if necessary }
  if (LocalQuant^.needs_zeroed) then
  begin
    for i := 0 to pred(HIST_C0_ELEMS) do
    begin
      FillChar( histogram^[i]^,
		HIST_C1_ELEMS*HIST_C2_ELEMS * SIZEOF(histcell), #0);
    end;
    LocalQuant^.needs_zeroed := FALSE;
  end;
end;

procedure new_color_map_2_quant; far;
begin
  { Reset the inverse color map }
  LocalQuant^.needs_zeroed := TRUE;
end;



{GLOBAL}
function init_:my_cquantize_ptr;
var
  i : int;
var
  desired : int;
begin
  New(result);
  LocalQuant:=result;
//  result^.pub.start_pass := start_pass_2_quant;
//  result^.pub.new_color_map := new_color_map_2_quant;
  result^.fserrors := NIL;	{ flag optional arrays not allocated }
  result^.error_limiter := NIL;

  { Allocate the histogram/inverse colormap storage }
  GetMem(result^.histogram , HIST_C0_ELEMS * SIZEOF(hist2d));

  for i := 0 to pred(HIST_C0_ELEMS) do
  begin
    GetMem(result^.histogram^[i],HIST_C1_ELEMS*HIST_C2_ELEMS * SIZEOF(histcell));
  end;

  result^.needs_zeroed := TRUE; { histogram is garbage now }

  { Allocate storage for the completed colormap, if required.
    We do this now since it is FAR storage and may affect
    the memory manager's space calculations. }

  { Make sure color count is acceptable }
  desired := desired_number_colors;
  { Lower bound on # of colors ... somewhat arbitrary as long as > 0 }
  if (desired < 8)or(desired > MAXNUMCOLORS) then
    exit;
  { Make sure colormap indexes can be represented by JSAMPLEs }

//  alloc_sarray (nil,JPOOL_IMAGE, JDIMENSION(desired), JDIMENSION(3));
  With LocalQuant^ do
  begin
    GetMem(sv_colormap,Sizeof(pointer)*3);
    GetMem(sv_colormap^[0],25000);FillChar(sv_colormap^[0]^,25000,#0);
    GetMem(sv_colormap^[1],25000);FillChar(sv_colormap^[1]^,25000,#0);
    GetMem(sv_colormap^[2],25000);FillChar(sv_colormap^[2]^,25000,#0);
  end;
  
  result^.desired := desired;

  GetMem(result^.fserrors,
       size_t ((Header.biwidth + 2) * (3 * SIZEOF(FSERROR)))  );
    { Might as well create the error-limiting table too. }
  init_error_limit;
end;

function find_nearby_colors (minc0 : int; minc1 : int; minc2 : int;
		             var colorlist : array of JSAMPLE) : int;
{ Locate the colormap entries close enough to an update box to be candidates
  for the nearest entry to some cell(s) in the update box.  The update box
  is specified by the center coordinates of its first cell.  The number of
  candidate colormap entries is returned, and their colormap indexes are
  placed in colorlist[].
  This routine uses Heckbert's "locally sorted search" criterion to select
  the colors that need further consideration. }

var
  numcolors : int;
  maxc0, maxc1, maxc2 : int;
  centerc0, centerc1, centerc2 : int;
  i, x, ncolors : int;
  minmaxdist, min_dist, max_dist, tdist : INT32;
  mindist : array[0..MAXNUMCOLORS-1] of INT32;
  	{ min distance to colormap entry i }
begin
  numcolors := actual_number_colors;

  { Compute true coordinates of update box's upper corner and center.
    Actually we compute the coordinates of the center of the upper-corner
    histogram cell, which are the upper bounds of the volume we care about.
    Note that since ">>" rounds down, the "center" values may be closer to
    min than to max; hence comparisons to them must be "<=", not "<". }

  maxc0 := minc0 + ((1 shl BOX_C0_SHIFT) - (1 shl C0_SHIFT));
  centerc0 := (minc0 + maxc0) shr 1;
  maxc1 := minc1 + ((1 shl BOX_C1_SHIFT) - (1 shl C1_SHIFT));
  centerc1 := (minc1 + maxc1) shr 1;
  maxc2 := minc2 + ((1 shl BOX_C2_SHIFT) - (1 shl C2_SHIFT));
  centerc2 := (minc2 + maxc2) shr 1;

  { For each color in colormap, find:
     1. its minimum squared-distance to any point in the update box
        (zero if color is within update box);
     2. its maximum squared-distance to any point in the update box.
    Both of these can be found by considering only the corners of the box.
    We save the minimum distance for each color in mindist[];
    only the smallest maximum distance is of interest. }

  minmaxdist := long($7FFFFFFF);

  for i := 0 to pred(numcolors) do
  begin
    { We compute the squared-c0-distance term, then add in the other two. }
    x := GETJSAMPLE(colormap^[0]^[i]);
    if (x < minc0) then
    begin
      tdist := (x - minc0) * C0_SCALE;
      min_dist := tdist*tdist;
      tdist := (x - maxc0) * C0_SCALE;
      max_dist := tdist*tdist;
    end
    else
      if (x > maxc0) then
      begin
        tdist := (x - maxc0) * C0_SCALE;
        min_dist := tdist*tdist;
        tdist := (x - minc0) * C0_SCALE;
        max_dist := tdist*tdist;
      end
      else
      begin
        { within cell range so no contribution to min_dist }
        min_dist := 0;
        if (x <= centerc0) then
        begin
          tdist := (x - maxc0) * C0_SCALE;
          max_dist := tdist*tdist;
        end
        else
        begin
          tdist := (x - minc0) * C0_SCALE;
          max_dist := tdist*tdist;
        end;
      end;

    x := GETJSAMPLE(colormap^[1]^[i]);
    if (x < minc1) then
    begin
      tdist := (x - minc1) * C1_SCALE;
      Inc(min_dist, tdist*tdist);
      tdist := (x - maxc1) * C1_SCALE;
      Inc(max_dist, tdist*tdist);
    end
    else
      if (x > maxc1) then
      begin
        tdist := (x - maxc1) * C1_SCALE;
        Inc(min_dist, tdist*tdist);
        tdist := (x - minc1) * C1_SCALE;
        Inc(max_dist, tdist*tdist);
      end
      else
      begin
        { within cell range so no contribution to min_dist }
        if (x <= centerc1) then
        begin
	  tdist := (x - maxc1) * C1_SCALE;
	  Inc(max_dist, tdist*tdist);
        end
        else
        begin
	  tdist := (x - minc1) * C1_SCALE;
	  Inc(max_dist, tdist*tdist);
        end
      end;

    x := GETJSAMPLE(colormap^[2]^[i]);
    if (x < minc2) then
    begin
      tdist := (x - minc2) * C2_SCALE;
      Inc(min_dist, tdist*tdist);
      tdist := (x - maxc2) * C2_SCALE;
      Inc(max_dist, tdist*tdist);
    end
    else
      if (x > maxc2) then
      begin
        tdist := (x - maxc2) * C2_SCALE;
        Inc(min_dist, tdist*tdist);
        tdist := (x - minc2) * C2_SCALE;
        Inc(max_dist, tdist*tdist);
      end
      else
      begin
        { within cell range so no contribution to min_dist }
        if (x <= centerc2) then
        begin
	  tdist := (x - maxc2) * C2_SCALE;
	  Inc(max_dist, tdist*tdist);
        end
        else
        begin
	  tdist := (x - minc2) * C2_SCALE;
	  Inc(max_dist, tdist*tdist);
        end;
      end;

    mindist[i] := min_dist;	{ save away the results }
    if (max_dist < minmaxdist) then
      minmaxdist := max_dist;
  end;

  { Now we know that no cell in the update box is more than minmaxdist
    away from some colormap entry.  Therefore, only colors that are
    within minmaxdist of some part of the box need be considered. }

  ncolors := 0;
  for i := 0 to pred(numcolors) do
  begin
    if (mindist[i] <= minmaxdist) then
    begin
      colorlist[ncolors] := JSAMPLE(i);
      Inc(ncolors);
    end;
  end;
  find_nearby_colors := ncolors;
end;

procedure find_best_colors (minc0 : int; minc1 : int; minc2 : int;
                            numcolors : int;
                            var colorlist : array of JSAMPLE;
                            var bestcolor : array of JSAMPLE);
{ Find the closest colormap entry for each cell in the update box,
  given the list of candidate colors prepared by find_nearby_colors.
  Return the indexes of the closest entries in the bestcolor[] array.
  This routine uses Thomas' incremental distance calculation method to
  find the distance from a colormap entry to successive cells in the box. }
const
  { Nominal steps between cell centers ("x" in Thomas article) }
  STEP_C0 = ((1 shl C0_SHIFT) * C0_SCALE);
  STEP_C1 = ((1 shl C1_SHIFT) * C1_SCALE);
  STEP_C2 = ((1 shl C2_SHIFT) * C2_SCALE);
var
  ic0, ic1, ic2 : int;
  i, icolor : int;
  {register} bptr : INT32PTR;     { pointer into bestdist[] array }
  cptr : JSAMPLE_PTR;              { pointer into bestcolor[] array }
  dist0, dist1 : INT32;         { initial distance values }
  {register} dist2 : INT32;	{ current distance in inner loop }
  xx0, xx1 : INT32;             { distance increments }
  {register} xx2 : INT32;
  inc0, inc1, inc2 : INT32;	{ initial values for increments }
  { This array holds the distance to the nearest-so-far color for each cell }
  bestdist : array[0..BOX_C0_ELEMS * BOX_C1_ELEMS * BOX_C2_ELEMS-1] of INT32;
begin
  { Initialize best-distance for each cell of the update box }
  for i := BOX_C0_ELEMS*BOX_C1_ELEMS*BOX_C2_ELEMS-1 downto 0 do
    bestdist[i] := $7FFFFFFF;

  { For each color selected by find_nearby_colors,
    compute its distance to the center of each cell in the box.
    If that's less than best-so-far, update best distance and color number. }

  for i := 0 to pred(numcolors) do
  begin
    icolor := GETJSAMPLE(colorlist[i]);
    { Compute (square of) distance from minc0/c1/c2 to this color }
    inc0 := (minc0 - GETJSAMPLE(colormap^[0]^[icolor])) * C0_SCALE;
    dist0 := inc0*inc0;
    inc1 := (minc1 - GETJSAMPLE(colormap^[1]^[icolor])) * C1_SCALE;
    Inc(dist0, inc1*inc1);
    inc2 := (minc2 - GETJSAMPLE(colormap^[2]^[icolor])) * C2_SCALE;
    Inc(dist0, inc2*inc2);
    { Form the initial difference increments }
    inc0 := inc0 * (2 * STEP_C0) + STEP_C0 * STEP_C0;
    inc1 := inc1 * (2 * STEP_C1) + STEP_C1 * STEP_C1;
    inc2 := inc2 * (2 * STEP_C2) + STEP_C2 * STEP_C2;
    { Now loop over all cells in box, updating distance per Thomas method }
    bptr := @bestdist[0];
    cptr := @bestcolor[0];
    xx0 := inc0;
    for ic0 := BOX_C0_ELEMS-1 downto 0 do
    begin
      dist1 := dist0;
      xx1 := inc1;
      for ic1 := BOX_C1_ELEMS-1 downto 0 do
      begin
	dist2 := dist1;
	xx2 := inc2;
	for ic2 := BOX_C2_ELEMS-1 downto 0 do
        begin
	  if (dist2 < bptr^) then
          begin
	    bptr^ := dist2;
	    cptr^ := JSAMPLE (icolor);
	  end;
	  Inc(dist2, xx2);
	  Inc(xx2, 2 * STEP_C2 * STEP_C2);
	  Inc(bptr);
	  Inc(cptr);
	end;
	Inc(dist1, xx1);
	Inc(xx1, 2 * STEP_C1 * STEP_C1);
      end;
      Inc(dist0, xx0);
      Inc(xx0, 2 * STEP_C0 * STEP_C0);
    end;
  end;
end;


procedure fill_inverse_cmap (c0 : int; c1 : int; c2 : int);
{ Fill the inverse-colormap entries in the update box that contains }
{ histogram cell c0/c1/c2.  (Only that one cell MUST be filled, but }
{ we can fill as many others as we wish.) }
var
  histogram : hist3d;
  minc0, minc1, minc2 : int;    { lower left corner of update box }
  ic0, ic1, ic2 : int;
  {register} cptr : JSAMPLE_PTR;	{ pointer into bestcolor[] array }
  {register} cachep : histptr;	{ pointer into main cache array }
  { This array lists the candidate colormap indexes. }
  colorlist : array[0..MAXNUMCOLORS-1] of JSAMPLE;
  numcolors : int;		{ number of candidate colors }
  { This array holds the actually closest colormap index for each cell. }
  bestcolor : array[0..BOX_C0_ELEMS * BOX_C1_ELEMS * BOX_C2_ELEMS-1] of JSAMPLE;
begin
  histogram := LocalQuant^.histogram;

  { Convert cell coordinates to update box ID }
  c0 := c0 shr BOX_C0_LOG;
  c1 := c1 shr BOX_C1_LOG;
  c2 := c2 shr BOX_C2_LOG;

  { Compute true coordinates of update box's origin corner.
    Actually we compute the coordinates of the center of the corner
    histogram cell, which are the lower bounds of the volume we care about.}

  minc0 := (c0 shl BOX_C0_SHIFT) + ((1 shl C0_SHIFT) shr 1);
  minc1 := (c1 shl BOX_C1_SHIFT) + ((1 shl C1_SHIFT) shr 1);
  minc2 := (c2 shl BOX_C2_SHIFT) + ((1 shl C2_SHIFT) shr 1);

  { Determine which colormap entries are close enough to be candidates
    for the nearest entry to some cell in the update box. }

  numcolors := find_nearby_colors(minc0, minc1, minc2, colorlist);

  { Determine the actually nearest colors. }
  find_best_colors( minc0, minc1, minc2, numcolors, colorlist,
		   bestcolor);

  { Save the best color numbers (plus 1) in the main cache array }
  c0 := c0 shl BOX_C0_LOG;		{ convert ID back to base cell indexes }
  c1 := c1 shl BOX_C1_LOG;
  c2 := c2 shl BOX_C2_LOG;
  cptr := @(bestcolor[0]);
  for ic0 := 0 to pred(BOX_C0_ELEMS) do
    for ic1 := 0 to pred(BOX_C1_ELEMS) do
    begin
      cachep := @(histogram^[c0+ic0]^[c1+ic1][c2]);
      for ic2 := 0 to pred(BOX_C2_ELEMS) do
      begin
	cachep^ := histcell (GETJSAMPLE(cptr^) + 1);
        Inc(cachep);
        Inc(cptr);
      end;
    end;
end;

procedure update_box (var boxp : box);
label
  have_c0min, have_c0max,
  have_c1min, have_c1max,
  have_c2min, have_c2max;
{ Shrink the min/max bounds of a box to enclose only nonzero elements, }
{ and recompute its volume and population }
var
  histogram : hist3d;
  histp : histptr;
  c0,c1,c2 : int;
  c0min,c0max,c1min,c1max,c2min,c2max : int;
  dist0,dist1,dist2 : INT32;
  ccount : long;
begin
  histogram := LocalQuant^.histogram;

  c0min := boxp.c0min;  c0max := boxp.c0max;
  c1min := boxp.c1min;  c1max := boxp.c1max;
  c2min := boxp.c2min;  c2max := boxp.c2max;

  if (c0max > c0min) then
    for c0 := c0min to c0max do
      for c1 := c1min to c1max do
      begin
	histp := @(histogram^[c0]^[c1][c2min]);
	for c2 := c2min to c2max do
        begin
	  if (histp^ <> 0) then
          begin
            c0min := c0;
	    boxp.c0min := c0min;
	    goto have_c0min;
	  end;
          Inc(histp);
        end;
      end;
 have_c0min:
  if (c0max > c0min) then
    for c0 := c0max downto c0min do
      for c1 := c1min to c1max do
      begin
	histp := @(histogram^[c0]^[c1][c2min]);
	for c2 := c2min to c2max do
        begin
	  if ( histp^ <> 0) then
          begin
            c0max := c0;
	    boxp.c0max := c0;
	    goto have_c0max;
	  end;
          Inc(histp);
        end;
      end;
 have_c0max:
  if (c1max > c1min) then
    for c1 := c1min to c1max do
      for c0 := c0min to c0max do
      begin
	histp := @(histogram^[c0]^[c1][c2min]);
	for c2 := c2min to c2max do
        begin
	  if (histp^ <> 0) then
          begin
            c1min := c1;
	    boxp.c1min := c1;
	    goto have_c1min;
	  end;
          Inc(histp);
        end;
      end;
 have_c1min:
  if (c1max > c1min) then
    for c1 := c1max downto c1min do
      for c0 := c0min to c0max do
      begin
	histp := @(histogram^[c0]^[c1][c2min]);
	for c2 := c2min to c2max do
        begin
	  if (histp^ <> 0) then
          begin
            c1max := c1;
	    boxp.c1max := c1;
	    goto have_c1max;
	  end;
          Inc(histp);
        end;
      end;
 have_c1max:
  if (c2max > c2min) then
    for c2 := c2min to c2max do
      for c0 := c0min to c0max do
      begin
	histp := @(histogram^[c0]^[c1min][c2]);
	for c1 := c1min to c1max do
        begin
	  if (histp^ <> 0) then
          begin
	    c2min := c2;
	    boxp.c2min := c2min;
	    goto have_c2min;
	  end;
          Inc(histp, HIST_C2_ELEMS);
        end;
      end;
 have_c2min:
  if (c2max > c2min) then
    for c2 := c2max downto c2min do
      for c0 := c0min to c0max do
      begin
	histp := @(histogram^[c0]^[c1min][c2]);
	for c1 := c1min to c1max do
        begin
	  if (histp^ <> 0) then
          begin
	    c2max := c2;
	    boxp.c2max := c2max;
	    goto have_c2max;
	  end;
          Inc(histp, HIST_C2_ELEMS);
        end;
      end;
 have_c2max:

  { Update box volume.
    We use 2-norm rather than real volume here; this biases the method
    against making long narrow boxes, and it has the side benefit that
    a box is splittable iff norm > 0.
    Since the differences are expressed in histogram-cell units,
    we have to shift back to JSAMPLE units to get consistent distances;
    after which, we scale according to the selected distance scale factors.}

  dist0 := ((c0max - c0min) shl C0_SHIFT) * C0_SCALE;
  dist1 := ((c1max - c1min) shl C1_SHIFT) * C1_SCALE;
  dist2 := ((c2max - c2min) shl C2_SHIFT) * C2_SCALE;
  boxp.volume := dist0*dist0 + dist1*dist1 + dist2*dist2;

  { Now scan remaining volume of box and compute population }
  ccount := 0;
  for c0 := c0min to c0max do
    for c1 := c1min to c1max do
    begin
      histp := @(histogram^[c0]^[c1][c2min]);
      for c2 := c2min to c2max do
      begin
	if (histp^ <> 0) then
	  Inc(ccount);
        Inc(histp);
      end;
    end;
  boxp.colorcount := ccount;
end;

function find_biggest_color_pop (boxlist : boxlistptr; numboxes : int) : boxptr;
{ Find the splittable box with the largest color population }
{ Returns NIL if no splittable boxes remain }
var
  boxp : boxptr ; {register}
  i : int;        {register}
  maxc : long;    {register}
  which : boxptr;
begin
  which := NIL;
  boxp := @(boxlist^[0]);
  maxc := 0;
  for i := 0 to pred(numboxes) do
  begin
    if (boxp^.colorcount > maxc) and (boxp^.volume > 0) then
    begin
      which := boxp;
      maxc := boxp^.colorcount;
    end;
    Inc(boxp);
  end;
  find_biggest_color_pop := which;
end;

function find_biggest_volume (boxlist : boxlistptr; numboxes : int) : boxptr;
{ Find the splittable box with the largest (scaled) volume }
{ Returns NULL if no splittable boxes remain }
var
  {register} boxp : boxptr;
  {register} i : int;
  {register} maxv : INT32;
  which : boxptr;
begin
  maxv := 0;
  which := NIL;
  boxp := @(boxlist^[0]);
  for i := 0 to pred(numboxes) do
  begin
    if (boxp^.volume > maxv) then
    begin
      which := boxp;
      maxv := boxp^.volume;
    end;
    Inc(boxp);
  end;
  find_biggest_volume := which;
end;

{LOCAL}
function median_cut (boxlist : boxlistptr;
                     numboxes : int; desired_colors : int) : int;
{ Repeatedly select and split the largest box until we have enough boxes }
var
  n,lb : int;
  c0,c1,c2,cmax : int;
  {register} b1,b2 : boxptr;
begin
  while (numboxes < desired_colors) do
  begin
    { Select box to split.
      Current algorithm: by population for first half, then by volume. }

    if (numboxes*2 <= desired_colors) then
      b1 := find_biggest_color_pop(boxlist, numboxes)
    else
      b1 := find_biggest_volume(boxlist, numboxes);

    if (b1 = NIL) then          { no splittable boxes left! }
      break;
    b2 := @(boxlist^[numboxes]);	{ where new box will go }
    { Copy the color bounds to the new box. }
    b2^.c0max := b1^.c0max; b2^.c1max := b1^.c1max; b2^.c2max := b1^.c2max;
    b2^.c0min := b1^.c0min; b2^.c1min := b1^.c1min; b2^.c2min := b1^.c2min;
    { Choose which axis to split the box on.
      Current algorithm: longest scaled axis.
      See notes in update_box about scaling distances. }

    c0 := ((b1^.c0max - b1^.c0min) shl C0_SHIFT) * C0_SCALE;
    c1 := ((b1^.c1max - b1^.c1min) shl C1_SHIFT) * C1_SCALE;
    c2 := ((b1^.c2max - b1^.c2min) shl C2_SHIFT) * C2_SCALE;
    { We want to break any ties in favor of green, then red, blue last.
      This code does the right thing for R,G,B or B,G,R color orders only. }

    cmax := c1; n := 1;
    if (c0 > cmax) then
    begin
      cmax := c0;
      n := 0;
    end;
    if (c2 > cmax) then
      n := 2;
    { Choose split point along selected axis, and update box bounds.
      Current algorithm: split at halfway point.
      (Since the box has been shrunk to minimum volume,
      any split will produce two nonempty subboxes.)
      Note that lb value is max for lower box, so must be < old max. }

    case n of
    0:begin
        lb := (b1^.c0max + b1^.c0min) div 2;
        b1^.c0max := lb;
        b2^.c0min := lb+1;
      end;
    1:begin
        lb := (b1^.c1max + b1^.c1min) div 2;
        b1^.c1max := lb;
        b2^.c1min := lb+1;
      end;
    2:begin
        lb := (b1^.c2max + b1^.c2min) div 2;
        b1^.c2max := lb;
        b2^.c2min := lb+1;
      end;
    end;
    { Update stats for boxes }
    update_box( b1^);
    update_box( b2^);
    Inc(numboxes);
  end;
  median_cut := numboxes;
end;


{LOCAL}
procedure compute_color (const boxp : box; icolor : int);
{ Compute representative color for a box, put it in colormap[icolor] }
var
  { Current algorithm: mean weighted by pixels (not colors) }
  { Note it is important to get the rounding correct! }
  histogram : hist3d;
  histp : histptr;
  c0,c1,c2 : int;
  c0min,c0max,c1min,c1max,c2min,c2max : int;
  count : long;
  total : long;
  c0total : long;
  c1total : long;
  c2total : long;
begin
  histogram := LocalQuant^.histogram;
  total := 0;
  c0total := 0;
  c1total := 0;
  c2total := 0;

  c0min := boxp.c0min;  c0max := boxp.c0max;
  c1min := boxp.c1min;  c1max := boxp.c1max;
  c2min := boxp.c2min;  c2max := boxp.c2max;

  for c0 := c0min to c0max do
    for c1 := c1min to c1max do
    begin
      histp := @(histogram^[c0]^[c1][c2min]);
      for c2 := c2min to c2max do
      begin
	count := histp^;
        Inc(histp);
	if (count <> 0) then
        begin
	  Inc(total, count);
	  Inc(c0total, ((c0 shl C0_SHIFT) + ((1 shl C0_SHIFT) shr 1)) * count);
	  Inc(c1total, ((c1 shl C1_SHIFT) + ((1 shl C1_SHIFT) shr 1)) * count);
	  Inc(c2total, ((c2 shl C2_SHIFT) + ((1 shl C2_SHIFT) shr 1)) * count);
	end;
      end;
    end;

  colormap^[0]^[icolor] := JSAMPLE ((c0total + (total shr 1)) div total);
  colormap^[1]^[icolor] := JSAMPLE ((c1total + (total shr 1)) div total);
  colormap^[2]^[icolor] := JSAMPLE ((c2total + (total shr 1)) div total);
end;


procedure select_colors (desired_colors : int);
{ Master routine for color selection }
var
  boxlist : boxlistptr;
  numboxes : int;
  i : int;
begin
  { Allocate workspace for box list }
  GetMem(boxlist, desired_colors * SIZEOF(box));
  { Initialize one box containing whole space }
  numboxes := 1;
  boxlist^[0].c0min := 0;
  boxlist^[0].c0max := MAXJSAMPLE shr C0_SHIFT;
  boxlist^[0].c1min := 0;
  boxlist^[0].c1max := MAXJSAMPLE shr C1_SHIFT;
  boxlist^[0].c2min := 0;
  boxlist^[0].c2max := MAXJSAMPLE shr C2_SHIFT;
  { Shrink it to actually-used volume and set its statistics }
  update_box( boxlist^[0]);
  { Perform median-cut to produce final box list }
  numboxes := median_cut( boxlist, numboxes, desired_colors);
  { Compute the representative color for each box, fill colormap }
  for i := 0 to pred(numboxes) do
    compute_color( boxlist^[i], i);
  actual_number_colors := numboxes;
  FreeMem(boxlist, desired_colors * SIZEOF(box));
//  TRACEMS1(j_common_ptr(cinfo), 1, JTRC_QUANT_SELECTED, numboxes);
end;

procedure prescan_quantize (input_buf : PChar;
                            num_rows : int); far;
var
  {register} ptr : PChar;
  {register} histp : histptr;
  {register} histogram : hist3d;
  row : int;
  col : JDIMENSION;
  width : JDIMENSION;
  i,j,k:Integer;
begin
  histogram := LocalQuant^.histogram;
  width := Header.biwidth;
  ptr := input_buf;
  for row := 0 to pred(num_rows) do
  begin
    for col := pred(width) downto 0 do
    begin
      i:=GETJSAMPLE(ptr^) shr C0_SHIFT;inc(ptr);
      j:=GETJSAMPLE(ptr^) shr C1_SHIFT;inc(ptr);
      k:=GETJSAMPLE(ptr^) shr C2_SHIFT;inc(ptr);
      { get pixel value and index into the histogram }
      histp := @(histogram^[i]^
                           [j]
			   [k]);
      { increment, check for overflow and undo increment if so. }
      Inc(histp^);
      if (histp^ <= 0) then
	Dec(histp^);
    end;
  end;
end;

procedure fs_dither (input_buf : Pointer;
                     output_buf: Pointer;
                     NumLines:Integer);
{ This version performs Floyd-Steinberg dithering }
var
  histogram : hist3d;
  {register} cur : LOCRGB_FSERROR;	{ current error or pixel value }
  belowerr : LOCRGB_FSERROR; { error for pixel below cur }
  bpreverr : LOCRGB_FSERROR; { error for below/prev col }
  prev_errorptr,
  {register} errorptr : RGB_FSERROR_PTR;	{ => fserrors[] at column before current }
  inptr : RGBptr;		{ => current input pixel }
  outptr : JSAMPLE_PTR;		{ => current output pixel }
  cachep : histptr;
  dir : int;			{ +1 or -1 depending on direction }
  row : int;
  col : JDIMENSION;
  width : JDIMENSION;
  error_limit : error_limit_ptr;
  colormap0 : JSAMPROW;
  colormap1 : JSAMPROW;
  colormap2 : JSAMPROW;
  {register} pixcode : int;
  {register} bnexterr, delta : LOCFSERROR;
begin

  histogram := LocalQuant^.histogram;
  width := Header.biwidth;
  error_limit := LocalQuant^.error_limiter;

  colormap0 := colormap^[0];
  colormap1 := colormap^[1];
  colormap2 := colormap^[2];

  for row := 0 to pred(NumLines) do
  begin

    inptr := RGBptr(PChar(Input_Buf) + 3*width*row);
    outptr := JSAMPLE_PTR(PChar(Output_Buf) + width*row);

    errorptr := RGB_FSERROR_PTR(LocalQuant^.fserrors); { => entry before first real column }
    if (LocalQuant^.on_odd_row) then
    begin
      { work right to left in this row }
      Inc(inptr, (width-1));     { so point to rightmost pixel }
      Inc(outptr, width-1);
      dir := -1;
      Inc(errorptr, (width+1)); { => entry after last column }
      LocalQuant^.on_odd_row := FALSE; { flip for next time }
    end
    else
    begin
      { work left to right in this row }
      dir := 1;
      LocalQuant^.on_odd_row := TRUE; { flip for next time }
    end;

    { Preset error values: no error propagated to first pixel from left }
    cur.r := 0;
    cur.g := 0;
    cur.b := 0;
    { and no error propagated to row below yet }
    belowerr.r := 0;
    belowerr.g := 0;
    belowerr.b := 0;
    bpreverr.r := 0;
    bpreverr.g := 0;
    bpreverr.b := 0;

    for col := pred(width) downto 0 do
    begin
      prev_errorptr := errorptr;
      Inc(errorptr, dir);	{ advance errorptr to current column }

      { curN holds the error propagated from the previous pixel on the
        current line.  Add the error propagated from the previous line
        to form the complete error correction term for this pixel, and
        round the error term (which is expressed * 16) to an integer.
        RIGHT_SHIFT rounds towards minus infinity, so adding 8 is correct
        for either sign of the error value.
        Note: prev_errorptr points to *previous* column's array entry. }

      { Nomssi Note: Borland Pascal SHR is unsigned }
      cur.r := (cur.r + errorptr^.r + 8) div 16;
      cur.g := (cur.g + errorptr^.g + 8) div 16;
      cur.b := (cur.b + errorptr^.b + 8) div 16;
      { Limit the error using transfer function set by init_error_limit.
        See comments with init_error_limit for rationale. }

      cur.r := error_limit^[cur.r];
      cur.g := error_limit^[cur.g];
      cur.b := error_limit^[cur.b];
      { Form pixel value + error, and range-limit to 0..MAXJSAMPLE.
        The maximum error is +- MAXJSAMPLE (or less with error limiting);
        this sets the required size of the range_limit array. }

      Inc(cur.r, GETJSAMPLE(inptr^.r));
      Inc(cur.g, GETJSAMPLE(inptr^.g));
      Inc(cur.b, GETJSAMPLE(inptr^.b));

      cur.r := GETJSAMPLE(range_limit^[cur.r]);
      cur.g := GETJSAMPLE(range_limit^[cur.g]);
      cur.b := GETJSAMPLE(range_limit^[cur.b]);
      { Index into the cache with adjusted pixel value }
      cachep := @(histogram^[cur.r shr C0_SHIFT]^
                            [cur.g shr C1_SHIFT][cur.b shr C2_SHIFT]);
      { If we have not seen this color before, find nearest colormap }
      { entry and update the cache }
      if (cachep^ = 0) then
	fill_inverse_cmap(cur.r shr C0_SHIFT,
                          cur.g shr C1_SHIFT,
                          cur.b shr C2_SHIFT);
      { Now emit the colormap index for this cell }

      pixcode := cachep^ - 1;
      outptr^ := JSAMPLE (pixcode);

      { Compute representation error for this pixel }
      Dec(cur.r, GETJSAMPLE(colormap0^[pixcode]));
      Dec(cur.g, GETJSAMPLE(colormap1^[pixcode]));
      Dec(cur.b, GETJSAMPLE(colormap2^[pixcode]));

      { Compute error fractions to be propagated to adjacent pixels.
        Add these into the running sums, and simultaneously shift the
        next-line error sums left by 1 column. }

      bnexterr := cur.r;	{ Process component 0 }
      delta := cur.r * 2;
      Inc(cur.r, delta);		{ form error * 3 }
      prev_errorptr^.r := FSERROR (bpreverr.r + cur.r);
      Inc(cur.r, delta);		{ form error * 5 }
      bpreverr.r := belowerr.r + cur.r;
      belowerr.r := bnexterr;
      Inc(cur.r, delta);		{ form error * 7 }
      bnexterr := cur.g;	{ Process component 1 }
      delta := cur.g * 2;
      Inc(cur.g, delta);		{ form error * 3 }
      prev_errorptr^.g := FSERROR (bpreverr.g + cur.g);
      Inc(cur.g, delta);		{ form error * 5 }
      bpreverr.g := belowerr.g + cur.g;
      belowerr.g := bnexterr;
      Inc(cur.g, delta);		{ form error * 7 }
      bnexterr := cur.b;	{ Process component 2 }
      delta := cur.b * 2;
      Inc(cur.b, delta);		{ form error * 3 }
      prev_errorptr^.b := FSERROR (bpreverr.b + cur.b);
      Inc(cur.b, delta);		{ form error * 5 }
      bpreverr.b := belowerr.b + cur.b;
      belowerr.b := bnexterr;
      Inc(cur.b, delta);		{ form error * 7 }

      { At this point curN contains the 7/16 error value to be propagated
        to the next pixel on the current line, and all the errors for the
        next line have been shifted over.  We are therefore ready to move on.}

      Inc(inptr, dir); { Advance pixel pointers to next column }
      Inc(outptr, dir);
    end;
    { Post-loop cleanup: we must unload the final error values into the
      final fserrors[] entry.  Note we need not unload belowerrN because
      it is for the dummy column before or after the actual array. }

    errorptr^.r := FSERROR (bpreverr.r); { unload prev errs into array }
    errorptr^.g := FSERROR (bpreverr.g);
    errorptr^.b := FSERROR (bpreverr.b);
  end;
end;


procedure dither (Info:PBitMapInfo;
		  input_buf : Pointer;
                  output_buf: Pointer;
                  NumberColors:Integer);

var
  i:Integer;
  p,pp:PChar;
begin
  Desired_number_colors:=NumberColors;
  Header:=Info^.bmiheader;
  actual_number_colors:=NumberColors;
  LocalQuant:=Init_;

  start_pass_2_quant(True);
  prescan_quantize(input_buf,Header.biHeight);

  colormap := LocalQuant^.sv_colormap;
  select_colors(LocalQuant^.desired);

  prepare_range_limit_table;

  { Force next pass to zero the color index table }
  LocalQuant^.needs_zeroed := TRUE;

  start_pass_2_quant(False);

  p:=input_buf;pp:=output_buf;
  fs_Dither(p,pp,Header.biHeight-1);

  for i:=0 to 255 do
  begin
    With Info^.bmiColors[i] do
    begin
      rgbRed:=colormap[2]^[i];
      rgbGreen:=colormap[1]^[i];
      rgbBlue:=colormap[0]^[i];
      rgbReserved:=0;
    end;
  end;

  Dispose(range_limit);
  FreeMem(LocalQuant^.error_Limiter, (MAXJSAMPLE*2+1) * SIZEOF(int));
  FreeMem(LocalQuant^.fserrors,(Header.biWidth + 2) *
                                 (3 * SIZEOF(FSERROR)));
  for i:= 0 to pred(HIST_C0_ELEMS) do
  begin
    FreeMem(LocalQuant^.histogram^[i],HIST_C1_ELEMS*HIST_C2_ELEMS * SIZEOF(histcell));
  end;
  FreeMem(LocalQuant^.histogram , HIST_C0_ELEMS * SIZEOF(hist2d));

  With LocalQuant^ do
  begin
    FreeMem(sv_colormap^[0],25000);
    FreeMem(sv_colormap^[1],25000);
    FreeMem(sv_colormap^[2],25000);
    FreeMem(sv_colormap,Sizeof(pointer)*3);
  end;
{  FreeMem(LocalQuant^.fserrors,
       size_t ((Header.biwidth + 2) * (3 * SIZEOF(FSERROR)))  );}
//  FreeMem(boxlist, desired_number_colors * SIZEOF(box));
  Dispose(LocalQuant);
end;
end.
