unit dpmsdrv;

// DPMS Screen Saver for Win 9X/NT to prolong your the lifetime of your monitor
// Copyright (C) 1999-2000, Daniel Marczisovszky
// mailto:marczi@kurt.hu
// http://www.kurt.hu/~marczi

// This program is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation; either version 2 of the License, or
// (at your option) any later version.

// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.

// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

// TDPMSDriver communicates with service control manager and DPMSdrv.
// If running on Win9X then it will use DPMS16.EXE to switch to power
// saving modes.
// Based on : GWIOPM.PAS by Graham Wideman (http://www.wideman-one.com)


interface
uses forms, windows, winsvc, constant, sysutils, ShellAPI;

type

  TDPMSDriver = class
  private
    HomeDir         : string;
    DriverDir       : string;
    DriverName      : string;
    DriverPath      : string; // the whole thing
    hSCMan          : SC_HANDLE; // Service Control Manager
    hDevice         : SC_HANDLE; // Handle for device

  public
    LastError       : longint;
    constructor Create;

    //---------------------------------------
    // Interact with Service Control Manager
    //---------------------------------------
    function OpenSCM: DWORD;
    function CloseSCM: DWORD;

    //---------------------------------------
    // Install/Start/Stop/Remove driver
    //---------------------------------------
    function Install(newdriverpath: string): DWORD; { use '' for default }
    function Start:   DWORD;
    function Stop:    DWORD;
    function Remove:  DWORD;

    //--------------------------------
    // Device Open/Close
    //--------------------------------
    function DeviceOpen: DWORD;   // get a valid hDevice
    function DeviceClose: DWORD;

    procedure PowerSavingMode(dpms_mode: integer);

    function ErrorLookup(ErrorNum: integer): string;
  end;

const
  DPMS_MODE_NORMAL     = $900;
  DPMS_MODE_STANDBY    = $901;
  DPMS_MODE_SUSPEND    = $902;
  DPMS_MODE_OFF        = $903;
  DPMS_MODE_REDUCED_ON = $904;

var
  DPMSDriver: TDPMSDriver;

implementation

uses common;

constructor TDPMSDriver.Create;
begin
  hSCMan  := 0;
  hDevice := INVALID_HANDLE_VALUE;
  HomeDir := ExtractFilePath(ParamStr(0));
  DriverName  := DEVICE_NAME_STRING;
end;

function TDPMSDriver.ErrorLookup(ErrorNum: integer): string;
var i: integer;
begin
  result := '';

  for i := 1 to ErrorMsgCt do begin
    if ErrorNum = ErrorMsgs[i].Num then begin
      result := result + ErrorMsgs[i].Msg;
      break;
    end;
  end;
end;

function TDPMSDriver.OpenSCM:  DWORD;
begin
  result := ERROR_SUCCESS;
  hSCMan := OpenSCManager(nil, nil, SC_MANAGER_ALL_ACCESS);
  if hSCMan = 0 then result := ERROR_SCM_CANT_CONNECT;
end;

function TDPMSDriver.CloseSCM:  DWORD;
begin
  result := ERROR_SUCCESS;
  CloseServiceHandle(hSCMan);
  hSCMan := 0;
end;

function TDPMSDriver.Install(newdriverpath: string): DWORD; { use '' for default }
var hService: SC_HANDLE;
begin
  hService := 0;
  result := 0;

  if newdriverpath = '' then begin
    DriverDir   := HomeDir;
    DriverName  := DEVICE_NAME_STRING;
  end 
  else begin
    DriverDir  := ExtractFilePath(DriverPath);
    DriverName := ExtractFileName(DriverPath);
  end;
  DriverPath  := DriverDir + DriverName + '.sys';

  // add to service control manager's database
  hService := CreateService(hSCMan, PChar(DriverName),PChar(DriverName),
              SERVICE_ALL_ACCESS, SERVICE_KERNEL_DRIVER, SERVICE_DEMAND_START,
              SERVICE_ERROR_NORMAL, PChar(DriverPath),
              nil, nil, nil, nil, nil);
  if (hService = 0) then
    result := GetLastError()
  else
    CloseServiceHandle(hService);
end;

function TDPMSDriver.Start:   DWORD;
var hService: SC_HANDLE;
    ServiceArg: PChar;
begin
  hService := 0;
  result := 0;
  ServiceArg := nil;

  // get a handle to the service
  hService := OpenService(hSCMan, PChar(DriverName), SERVICE_ALL_ACCESS);
  if hService <> 0 then begin
     // start the driver
     if not StartService(hService, 0, PChar(ServiceArg)) then
       result := GetLastError();
  end 
  else 
    result := GetLastError();

  if (hService <> 0) then CloseServiceHandle(hService);
end;

function TDPMSDriver.Stop: DWORD;
var hService: SC_HANDLE;
    serviceStatus: TServiceStatus;
begin
  hService := 0;
  result := 0;

  // get a handle to the service
  hService := OpenService(hSCMan, PChar(DriverName), SERVICE_ALL_ACCESS);
  if hService <> 0 then begin
    if not ControlService(hService, SERVICE_CONTROL_STOP, serviceStatus) then // stop the driver
      result := GetLastError();
  end 
  else 
    result := GetLastError();

  if (hService <> 0) then CloseServiceHandle(hService);
end;

function TDPMSDriver.Remove: DWORD;
var hService: SC_HANDLE;
begin
  hService := 0;
  result := 0;

  // get a handle to the service
  hService := OpenService(hSCMan, PChar(DriverName), SERVICE_ALL_ACCESS);
  if hService <> 0 then begin
    if not DeleteService(hService) then
      result := GetLastError();
  end 
  else 
    result := GetLastError();

  if (hService <> 0) then CloseServiceHandle(hService);
end;

//=============================================================
// Device Open/Close functions
//=============================================================

function TDPMSDriver.DeviceOpen:  DWORD;
begin
  result := 0;
  hDevice := CreateFile('\\.\' + DEVICE_NAME_STRING, GENERIC_READ or GENERIC_WRITE, 0, nil, OPEN_EXISTING, 0, 0);
  if hDevice = INVALID_HANDLE_VALUE then result := GetLastError();
end;

function TDPMSDriver.DeviceClose:  DWORD;
begin
  result := 0;
  if (hDevice <> INVALID_HANDLE_VALUE) then CloseHandle(hDevice);
  hDevice := INVALID_HANDLE_VALUE;
end;

procedure TDPMSDriver.PowerSavingMode(dpms_mode: integer);
var BytesReturned: DWORD;
    CtlCode: DWORD;
    path, param: string;
    b: boolean;
    s: string;
begin
  // Windows NT
  DeviceOpen;
  if Win32Platform = VER_PLATFORM_WIN32_NT then begin
    CtlCode := IOCTL or (dpms_mode shl 2);
    b := DeviceIoControl(hDevice, CtlCode, nil, 0, nil, 0, BytesReturned, nil);
    case dpms_mode of
      DPMS_MODE_NORMAL:      s := 'powersaving mode: normal - result:';
      DPMS_MODE_STANDBY:     s := 'powersaving mode: standby - result:';
      DPMS_MODE_SUSPEND:     s := 'powersaving mode: suspend - result:';
      DPMS_MODE_OFF:         s := 'powersaving mode: off - result:';
      DPMS_MODE_REDUCED_ON:  s := 'powersaving mode: reduced - result:';
    end;
    if b then
      s := s + ' success'
    else
      s := s + ' failed';
    Log(s);
  end
  // Windows 95
  else begin
    case dpms_mode of
      DPMS_MODE_NORMAL:      param := '';
      DPMS_MODE_STANDBY:     param := '1';
      DPMS_MODE_SUSPEND:     param := '2';
      DPMS_MODE_OFF:         param := '4';
      DPMS_MODE_REDUCED_ON:  param := '8';
    end;
    path := ExtractFilePath(Application.ExeName);;
    ShellExecute(Application.Handle, nil,
                 PChar(path + '\dpms16.exe'),
                 PChar(param), PChar(path), SW_SHOWNORMAL);
  end;
  DeviceClose;
end;


initialization
  DPMSDriver := TDPMSDriver.Create;

finalization

end.

