unit Config;

// DPMS Screen Saver for Win 9X/NT to prolong your the lifetime of your monitor
// Copyright (C) 1999-2000, Daniel Marczisovszky
// mailto:marczi@kurt.hu
// http://www.kurt.hu/~marczi

// This program is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation; either version 2 of the License, or
// (at your option) any later version.

// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.

// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

interface

uses
  Windows, Messages, SysUtils, Classes, Graphics, Controls, Forms, Dialogs,
  StdCtrls, Buttons, IniFiles, Spin, ExtCtrls, Mask, ShellAPI, ComCtrls,
  CommCtrl, Registry;

type
  TFormConfig = class(TForm)
    GroupBox1: TGroupBox;
    Label4: TLabel;
    CheckStandby: TCheckBox;
    EditStandby: TSpinEdit;
    EditSuspend: TSpinEdit;
    EditOff: TSpinEdit;
    CheckSuspend: TCheckBox;
    CheckOff: TCheckBox;
    LabelStandby: TLabel;
    LabelSuspend: TLabel;
    LabelOff: TLabel;
    Label1: TLabel;
    Label2: TLabel;
    Bevel1: TBevel;
    CheckMouseClick: TCheckBox;
    CheckMouseMove: TCheckBox;
    CheckKeypress: TCheckBox;
    EditMouseSens: TSpinEdit;
    LabelMouseSens1: TLabel;
    LabelMouseSens2: TLabel;
    CheckInterval: TCheckBox;
    BtnAbout: TBitBtn;
    BtnOK: TBitBtn;
    BtnClose: TBitBtn;
    BtnRemove: TBitBtn;
    BtnHelp: TBitBtn;
    LabelAnd: TLabel;
    Bevel2: TBevel;
    Bevel3: TBevel;
    Image1: TImage;
    DateTimePicker1: TDateTimePicker;
    DateTimePicker2: TDateTimePicker;
    CheckExternal: TCheckBox;
    cbScreenSavers: TComboBox;
    BtnSettings: TBitBtn;
    MemoLog: TMemo;
    BtnPreview: TBitBtn;
    CheckBeforeLogon: TCheckBox;
    Bevel4: TBevel;
    EditBeforeLogon: TSpinEdit;
    LabelAfterStarted: TLabel;
    Bevel5: TBevel;
    procedure FormShow(Sender: TObject);
    procedure BtnOKClick(Sender: TObject);
    procedure BtnCloseClick(Sender: TObject);
    procedure BtnAboutClick(Sender: TObject);
    procedure CheckClick(Sender: TObject);
    procedure EditChange(Sender: TObject);
    procedure EditKeyPress(Sender: TObject; var Key: Char);
    procedure FormDestroy(Sender: TObject);
    procedure BtnRemoveClick(Sender: TObject);
    procedure CheckMouseMoveClick(Sender: TObject);
    procedure BtnHelpClick(Sender: TObject);
    procedure BtnSettingsClick(Sender: TObject);
    procedure CheckExternalClick(Sender: TObject);
    procedure CheckIntervalClick(Sender: TObject);
    procedure BtnPreviewClick(Sender: TObject);
    procedure CheckBeforeLogonClick(Sender: TObject);
  private
    startup: boolean;
    procedure EnableEdit(Edit: TEdit; en: boolean; defaultText: string);
    { Private declarations }
  public
    { Public declarations }
  end;

var
  FormConfig: TFormConfig;

implementation

uses About, common, DPMSdrv, constant, scrnsave, ExtPrev;

{$R *.DFM}

{ FORM SHOW }
procedure TFormConfig.FormShow(Sender: TObject);
var i, p: integer;
    s: string;
begin
  Application.HelpFile := ExtractFilePath(Application.ExeName) + 'dpms.hlp';

  ReadSettings;

  CheckKeypress.Checked := stop_keypress;
  CheckMouseClick.Checked := stop_mouseclick;
  CheckMouseMove.Checked := stop_mousemove;
  if CheckMouseMove.Checked then
    EditMouseSens.Text := IntToStr(mouse_sens)   // set mouse sensitivity
  else
    CheckMouseMoveClick(self);                   // gray out mouse sens. edit box

  CheckStandby.Checked := use_standby; CheckClick(CheckStandby);
  CheckSuspend.Checked := use_suspend; CheckClick(CheckSuspend);
  CheckOff.Checked := use_off;         CheckClick(CheckOff);

  if use_standby then EditStandby.Text := IntToStr(time_standby);
  if use_suspend then EditSuspend.Text := IntToStr(time_suspend);
  if use_off then EditOff.Text := IntToStr(time_off);

  DateTimePicker1.Time := interval_from;
  DateTimePicker2.Time := interval_till;
  CheckInterval.Checked := use_time_interval;

  // read "before logon" settings
  with TRegistry.Create do begin
    RootKey := HKEY_USERS;
    OpenKey('.DEFAULT\Control Panel\Desktop', false);
    s := ReadString('SCRNSAVE.EXE');
    // check if dpms.scr is the currently active logon screen saver
    CheckBeforeLogon.Checked := (pos(ExtractFileName(Application.ExeName), s) > 0);
    s := ReadString('ScreenSaveTimeOut');
    i := StrToIntDef(s, 1);
    i := i div 60;
    if i < 1 then i := 1;
    CheckBeforeLogonClick(self);
    if CheckBeforeLogon.Checked then EditBeforeLogon.Text := IntToStr(i);
    CloseKey;
    Free;
  end;

  // read external screen savers
  ExtScrSavers := TStringList.Create;
  ExtScrSavers.Sorted := True;
  ExtScrSavers.Duplicates := dupIgnore;
  GetScreenSaverNames(ExtScrSavers);

  for i := 0 to ExtScrSavers.Count - 1 do begin
    s := ExtScrSavers.Strings[i];
    p := Pos('===', s);
    if p <> 0 then Delete(s, p, Length(s) - p + 1);
    cbScreenSavers.Items.Add(s);
  end;

  with cbScreenSavers do begin
    ItemIndex := Items.IndexOf(ext_screen_name);
  end;
  CheckExternal.Checked := use_ext;

  // try to init the DPMSDRV Driver
  if InitDriver(MemoLog.Lines) <> ERROR_SUCCESS then begin
    // if it was not successful, then remove the driver and install again
    RemoveDriver(nil);
    TerminateDriver(true);
    InitDriver(MemoLog.Lines);
  end;

  // set datetimepicker datetime format 
  DateTime_SetFormat(DateTimePicker1.Handle, 'hh:mm tt');
  DateTime_SetFormat(DateTimePicker2.Handle, 'hh:mm tt');
end;

{ REMOVE DRIVER }
procedure TFormConfig.BtnRemoveClick(Sender: TObject);
begin
  RemoveDriver(MemoLog.Lines);
end;

{ OK - SAVE CHANGES }
procedure TFormConfig.BtnOKClick(Sender: TObject);
var min: integer;
begin
  with TIniFile.Create(ExtractFilePath(Application.ExeName) + 'dpms_scr.ini') do begin
    WriteBool('Stop', 'keypress', CheckKeypress.Checked);
    WriteBool('Stop', 'mouse_click', CheckMouseClick.Checked);
    WriteBool('Stop', 'mouse_move', CheckMouseMove.Checked);
    WriteInteger('Stop', 'mouse_sensitivity', StrToIntDef(EditMouseSens.Text, 5));

    WriteBool('Modes', 'standby', CheckStandby.Checked);
    WriteBool('Modes', 'suspend', CheckSuspend.Checked);
    WriteBool('Modes', 'off', CheckOff.Checked);
    WriteInteger('Timeout', 'standby', StrToIntDef(EditStandby.Text, 0));
    WriteInteger('Timeout', 'suspend', StrToIntDef(EditSuspend.Text, 0));
    WriteInteger('Timeout', 'off', StrToIntDef(EditOff.Text, 0));

    WriteString('External', 'screen_saver', cbScreenSavers.Text);
    WriteBool('External', 'use_external', CheckExternal.Checked);

    WriteString('Interval', 'from', TimeToStr(DateTimePicker1.Time));
    WriteString('Interval', 'till', TimeToStr(DateTimePicker2.Time));
    WriteBool('Interval', 'use_time_interval', CheckInterval.Checked);
    Free;
  end;

  // start screen saver before logon
  with TRegistry.Create do begin
    RootKey := HKEY_USERS;
    if OpenKey('.DEFAULT\Control Panel\Desktop', false) then begin
      if CheckBeforeLogon.Checked then begin
        min := StrToIntDef(EditBeforeLogon.Text, 0);
        if min < 1 then min := 1;
        min := min * 60;
        WriteString('SCRNSAVE.EXE', Application.ExeName);
        WriteString('ScreenSaveTimeOut', IntToStr(min));
      end
      else begin
        WriteString('SCRNSAVE.EXE', 'logon.scr');
        WriteString('ScreenSaveTimeOut', '600');
      end;
      CloseKey;
    end;
    Free;
  end;

  Close;
end;

{ DESTROY }
procedure TFormConfig.FormDestroy(Sender: TObject);
begin
  ExtScrSavers.Free;
  TerminateDriver(false);
end;

{ CANCEL }
procedure TFormConfig.BtnCloseClick(Sender: TObject);
begin
  Close;
end;

{ HELP }
procedure TFormConfig.BtnHelpClick(Sender: TObject);
begin
  Application.HelpCommand(HELP_CONTEXT, 0);
end;

{ ABOUT }
procedure TFormConfig.BtnAboutClick(Sender: TObject);
begin
  if FormAbout = nil then
    Application.CreateForm(TFormAbout, FormAbout);
  FormAbout.Show;
end;

{ SETTINGS }
procedure TFormConfig.BtnSettingsClick(Sender: TObject);
var s: string;
begin
  with cbScreenSavers do s := Items[ItemIndex];
  s := ScreenSaveFileName(s);
  StartExtScreenSaver(s, '/C');
end;

{ PREVIEW }
procedure TFormConfig.BtnPreviewClick(Sender: TObject);
var s: string;
begin
  if FormExtPreview <> nil then FormExtPreview.Free;
  FormExtPreview := TFormExtPreview.Create(Self);
  FormExtPreview.Show;
  with cbScreenSavers do s := Items[ItemIndex];
  s := ScreenSaveFileName(s);
  StartExtScreenSaver(s, '/P '+IntToStr(FormExtPreview.Panel.Handle));
end;

{ ONLY NUMBERS ARE ALLOWED IN THE EDIT BOXES }
procedure TFormConfig.EditKeyPress(Sender: TObject; var Key: Char);
begin
  if not (key in [#8, '0'..'9']) then key := #0;
end;

{ ENABLE/DISABLE AND EDIT BOX }
procedure TFormConfig.EnableEdit(Edit: TEdit; en: boolean; defaultText: string);
begin
  with Edit do begin
    if en then begin
      Color := clWindow;
      Enabled := true;
      Text := defaultText;
    end
    else begin
      Color := clBtnFace;
      Enabled := false;
      Text := '';
    end;
  end;
end;

{ ENABLE/DISABLE MOUSE SENSITIVITY EDIT BOX }
procedure TFormConfig.CheckMouseMoveClick(Sender: TObject);
begin
  LabelMouseSens1.Enabled := CheckMouseMove.Checked;
  LabelMouseSens2.Enabled := CheckMouseMove.Checked;
  EnableEdit(TEdit(EditMouseSens), CheckMouseMove.Checked, '5');
end;

{ ENABLE/DISABLE EXTERNAL SCREEN SAVER }
procedure TFormConfig.CheckExternalClick(Sender: TObject);
begin
  cbScreenSavers.Enabled := CheckExternal.Checked;
  BtnSettings.Enabled := CheckExternal.Checked;
  BtnPreview.Enabled := CheckExternal.Checked;
end;

{ ENABLE/DISABLE ONLY BETWEEN CERTAIN TIME }
procedure TFormConfig.CheckIntervalClick(Sender: TObject);
begin
  DateTimePicker1.Enabled := CheckInterval.Checked;
  DateTimePicker2.Enabled := CheckInterval.Checked;
  LabelAnd.Enabled := CheckInterval.Checked;
end;

{ ENABLE/DISABLE BEFORE LOGON SCREEN SAVER MODE }
procedure TFormConfig.CheckBeforeLogonClick(Sender: TObject);
begin
  EnableEdit(TEdit(EditBeforeLogon), CheckBeforeLogon.Checked, '10');
  LabelAfterStarted.Enabled := CheckBeforeLogon.Checked;
end;

{ DPMS MODE CHECKBOXES }
procedure TFormConfig.CheckClick(Sender: TObject);
var name: string;
    checked: boolean;
    spinEdit: TSpinEdit;
begin
  name := (Sender as TCheckBox).Name; Delete(name, 1, 5);
  checked := (Sender as TCheckBox).Checked;

  spinEdit := FindComponent('Edit' + name) as TSpinEdit;
  EnableEdit(TEdit(spinEdit), Checked, '0');
  with FindComponent('Label' + name) as TLabel do Enabled := checked;
end;

{ DPMS MODE EDIT BOXES }
procedure TFormConfig.EditChange(Sender: TObject);
var standby, suspend, shutoff: integer;
begin
  if startup then exit;

  standby := StrToIntDef(EditStandby.Text, -1);
  suspend := StrToIntDef(EditSuspend.Text, -1);
  shutoff := StrToIntDef(EditOff.Text, -1);

  if CheckSuspend.Checked then begin
    if suspend <= standby then EditSuspend.Text := IntToStr(standby + 1);
  end
  else begin
    suspend := standby;
  end;

  if CheckOff.Checked then begin
    if shutoff <= suspend then EditOff.Text := IntToStr(suspend + 1);
  end;
end;



end.
