library hp200lx;

{
***** version 1.13 ***** 
(c) Pavel Zampach (zampach@volny.cz), 2004
GNU GENERAL PUBLIC LICENSE 
Source code for Borland Delphi Compiler (originally version 7.0)
}

{**$Define debug}
{$E wfx}
{$IOChecks Off}
{$BoolEval On}
{$ImageBase $40008000}
{$OverflowChecks On}
{$RangeChecks On}
{$Resource HP200LX.RES}

uses
  Windows, SysUtils, ShellAPI,
  DosIni   in 'DOSINI.PAS',
  Fsplugin in 'FSPLUGIN.PAS',
  Filer    in 'FILER.PAS',
  DirCache in 'DIRCACHE.PAS';

const
  PLUGIN_TITLE = 'HPLX file system';
  HELLO_TITLE  = 'HPLX plugin 1.13/GNU GPL, (c) P. Zampach, 2004';
  DIR_HPLX     = 0;
  DIR_CACHE    = 1;
  DIR_ROOT     = 2;

type
  CharArr = array [0..MAX_PATH-1] of char;

var
  ProgressProc : tProgressProc;
  LogProc      : tLogProc;
  RequestProc  : tRequestProc;
  PluginNumber : integer;
  DiskNr       : byte;
  SDrives, IniFileName, CachedDir : string;

{ ------------------------------------------------------------------ }

function getShortFileName (const FileName : string) : string;

var
  Tmp : array[0..255] of char;
  
begin
  if getShortPathName (PChar (FileName), Tmp, sizeOf (Tmp) - 1) = 0 then
    Result := FileName
  else
    Result := strPas (Tmp);
end;

{ ------------------------------------------------------------------ }

procedure dirEntryToFindData (const DirEntry : TDirEntry; var FindData : tWIN32FINDDATA);

var
  FT : TFileTime;
  
begin
  fillChar (FindData, sizeOf (tWIN32FINDDATA), 0);

  FindData.dwFileAttributes := DirEntry.fattr;
  DosDateTimeToFileTime (DirEntry.fdate, DirEntry.ftime, FT);
  LocalFileTimeToFileTime (FT, FindData.ftLastWriteTime);
  FindData.nFileSizeLow := DirEntry.fsize;
  strPLCopy (FindData.cFileName, DirEntry.fname, FNAME_LENGTH);
end;

{ ------------------------------------------------------------------ }

procedure fileTime2DosDateTime (const FT : TFileTime; var DFD : TDosFileDate);

var
  LFT : TFileTime;

begin
  fileTimeToLocalFileTime (FT, LFT);
  fileTimeToDosDateTime (LFT, DFD.fdate, DFD.ftime);
end;

{ ------------------------------------------------------------------ }

procedure rootToFindData (var FindData : tWIN32FINDDATA);

begin
  fillChar (FindData, sizeOf (tWIN32FINDDATA), 0);

  FindData.dwFileAttributes := FILE_ATTRIBUTE_DIRECTORY;
  FindData.ftLastWriteTime.dwHighDateTime := $FFFFFFFF;
  FindData.ftLastWriteTime.dwLowDateTime  := $FFFFFFFE;
  strPLCopy (FindData.cFileName, SDrives[DiskNr] + ':', 2);
end;

{ ------------------------------------------------------------------ }

function getDirEntry (const RemoteName : CharArr; var dirEntry : TDirEntry) : boolean;

begin
  Result := false;
  
  if not filerRequest (CMD_ASK_DIR, RemoteName) then begin
    filerReset;
    exit;
  end;
  Packet.data := @dirEntry;
  if not filerRequest (CMD_GET_DIR) then begin
    filerReset;
    exit;
  end;
  
  Result := true;
end;

{ ------------------------------------------------------------------ }

function cacheAdd (const RemoteName : CharArr) : boolean;

var
  DirEntry : TDirEntry;

begin
  Result := false;
  if not getDirEntry (RemoteName, DirEntry) then
    exit;
  Result := cacheWrite (DirEntry);  
end;

{ ------------------------------------------------------------------ }

function HPLXConnect : boolean;

var
  CommPort    : string;
  CommSpeed   : integer;

begin
  Result := true;
  if Connected then exit;
  
  CommPort  := getProfileStr (IniFileName, 'HPLX', 'Port', '');
  CommSpeed := getProfileInt (IniFileName, 'HPLX', 'Speed', -1);
  SDrives   := getProfileStr (IniFileName, 'HPLX', 'Drives', '');
  LogProc (PluginNumber, MsgTYPE_CONNECT, 'CONNECT \');
  
  if CommSpeed = -1 then begin
    Result := false;
    SetLastError (ERROR_FILE_NOT_FOUND);
    LogProc (PluginNumber, MsgTYPE_IMPORTANTERROR, 'FSPLUGIN.INI not found');
    exit;
  end;
  
  if not filerConnect (CommPort, CommSpeed) then begin
    Result := false;
    SetLastError (ERROR_FILE_NOT_FOUND);
    LogProc (PluginNumber, MsgTYPE_IMPORTANTERROR, 'Connect failed');
    exit;
  end;
  
  LogProc (PluginNumber, MsgTYPE_OPERATIONCOMPLETE, 'HPLX connected');
  LogProc (PluginNumber, MsgTYPE_DETAILS, HELLO_TITLE);
end;

{ ****************************************************************** }

function FsInit (PluginNr : integer; pProgressProc : tProgressProc; pLogProc : tLogProc;
                pRequestProc : tRequestProc) : integer; stdcall;

begin
  ProgressProc := pProgressProc;
  LogProc      := pLogProc;
  RequestProc  := pRequestProc;
  PluginNumber := PluginNr;

  Result := 0;
end;

{ ------------------------------------------------------------------ }

function FsFindFirst (Path : PChar; var FindData : tWIN32FINDDATA) : thandle; stdcall;

var
  Buf         : CharArr;
  DirEntry    : TDirEntry;
  CacheResult : byte;
  SPath       : string;

begin
  if not HPLXConnect then begin             // Login
    Result := INVALID_HANDLE_VALUE;
    exit;
  end;  

  if Path = '\' then begin                  // Root processing
    Result := DIR_ROOT;
    DiskNr := 1;
    rootToFindData (FindData);
    exit;
  end;

{$Warnings Off}
  SPath := IncludeTrailingBackslash (Path + 1); // Skip over leading backslash
{$Warnings On}
  
  if CachedDir = SPath then begin           // reading dir from cache
    Result := DIR_CACHE;
    cacheReset;
    CacheResult := cacheRead (DirEntry);
    if CacheResult = CACHE_NOTREAD then begin
      SetLastError (ERROR_FILE_NOT_FOUND);
      Result := INVALID_HANDLE_VALUE;
      exit;
    end;
    if CacheResult = CACHE_EMPTY then begin
      SetLastError (ERROR_NO_MORE_FILES);
      Result := INVALID_HANDLE_VALUE;
      exit;
    end;
  end
  else begin                                // reading dir from HPLX
    Result := DIR_HPLX;
    strCopy (Buf, PChar (SPath));
    strCat  (Buf, '*.*');
    cacheInit;
    CachedDir := SPath;
    if not getDirEntry (Buf, DirEntry) then begin
      SetLastError (ERROR_FILE_NOT_FOUND);
      Result := INVALID_HANDLE_VALUE;
      exit;
    end;
    if not Packet.existFlag then begin      // Empty dir!
      SetLastError (ERROR_NO_MORE_FILES);
      Result := INVALID_HANDLE_VALUE;
      exit;
    end;
    if not cacheWrite (DirEntry) then begin
      SetLastError (ERROR_FILE_NOT_FOUND);
      Result := INVALID_HANDLE_VALUE;
      exit;
    end;
  end;  

  dirEntryToFindData (DirEntry, FindData);
end;                                        // function FsFindFirst

{ ------------------------------------------------------------------ }

function FsFindNext (Hdl : thandle; var FindData : tWIN32FINDDATA) : bool; stdcall;

var
  DirEntry : TDirEntry;

begin
  Result := false;

  case Hdl of
    DIR_HPLX :
    begin
      Packet.data := @DirEntry;
      if not filerRequest (CMD_GET_DIR) then begin
        filerReset;
        exit;
      end;
      Result := Packet.existFlag;
      if Result then
        Result := cacheWrite (DirEntry);
      if Result then
        dirEntryToFindData (DirEntry, FindData);
    end;  
    
    DIR_CACHE :
    begin
      Result := (cacheRead (DirEntry) = CACHE_OK);
      if Result then
        dirEntryToFindData (DirEntry, FindData);
    end;
    
    DIR_ROOT :
    begin
      Result := (DiskNr < length (SDrives));
      if Result then begin
       inc (DiskNr);
       rootToFindData (FindData);
      end;
    end;
  end;                                      // case
end;                                        // function FsFindNext

{ ------------------------------------------------------------------ }

function FsFindClose (Hdl : thandle) : integer; stdcall;

begin
  Result := 0;
end;

{ ------------------------------------------------------------------ }

function FsDisconnect (DisconnectRoot : PChar) : bool; stdcall;

begin
  Result := filerDisconnect;
  CachedDir := #0;
end;

{ ------------------------------------------------------------------ }

function FsDeleteFile (RemoteName : PChar) : bool; stdcall;

begin
  inc (RemoteName);                         // skip over leading backslash
  Result := false;
  LogProc (PluginNumber, MsgTYPE_DETAILS, PChar ('Del ' + RemoteName));
  if not filerRequest (CMD_DEL_FILE, RemoteName) then begin;
    filerReset;
    exit;
  end;

  if extractFilePath (RemoteName) = CachedDir then
    if not cacheDelete (extractFileName (RemoteName)) then exit;

  Result := true;  
end;                                        // function FsDeleteFile

{ ------------------------------------------------------------------ }

function FsMkDir (RemoteDir : PChar) : bool; stdcall;

var
  FName : CharArr;

begin
  inc (RemoteDir);                          // skip over leading backslash
  Result := false;
  LogProc (PluginNumber, MsgTYPE_DETAILS, PChar ('MkDir ' + RemoteDir));
  if not filerRequest (CMD_MAKE_DIR, RemoteDir) then begin
    filerReset;
    exit;
  end;

  if extractFilePath (RemoteDir) = CachedDir then begin
    strCopy (FName, RemoteDir);
    if not cacheAdd (FName) then exit;
  end;  
  Result := true;  
end;                                        // function FsMkDir

{ ------------------------------------------------------------------ }

function FsRemoveDir (RemoteName : PChar) : bool; stdcall;

begin
  inc (RemoteName);                         // skip over leading backslash
  Result := false;
  LogProc (PluginNumber, MsgTYPE_DETAILS, PChar ('DelDir ' + RemoteName));
  if not filerRequest (CMD_DEL_DIR, RemoteName) then begin
    filerReset;
    exit;
  end;

  if extractFilePath (RemoteName) = CachedDir then
    if not cacheDelete (extractFileName (RemoteName)) then
      exit;
  Result := true;  
end;                                        // function FsRemoveDir

{ ------------------------------------------------------------------ }

function FsSetTime (RemoteName : PChar; CreationTime, LastAccessTime,
                    LastWriteTime : PFileTime) : bool; stdcall;

var
  DosTime : TDOSFileDate;
  FName   : CharArr;

begin  
  if LastWriteTime = nil then begin
    Result := true;
    exit;
  end;  

  inc (RemoteName);                         // skip over leading backslash
  Result := false;
  
  LogProc (PluginNumber, MsgTYPE_DETAILS, PChar ('Set FileTime ' + RemoteName));

  if not filerRequest (CMD_RESET_FILE, RemoteName) then begin
    filerReset;
    exit;
  end;
    
  fileTime2DosDateTime (LastWriteTime^, DosTime);
  Packet.data := @DosTime;
  if not filerRequest (CMD_SET_FILETIME) then begin
    filerReset;
    exit;
  end;
  
  if not filerRequest (CMD_CLOSE_FILE) then begin
    filerReset;
    exit;
  end;

  if extractFilePath (RemoteName) = CachedDir then begin
    if not cacheDelete (extractFileName (RemoteName)) then exit;
    strCopy (FName, RemoteName);
    if not cacheAdd (FName) then exit;
  end;  

  Result := true;  
end;  

{ ------------------------------------------------------------------ }

function FsSetAttr (RemoteName : PChar; NewAttr : integer) : bool; stdcall;

var
  DosAttr : byte;
  FName   : CharArr;

begin
  inc (RemoteName);                         // skip over leading backslash
  Result := false;
  LogProc (PluginNumber, MsgTYPE_DETAILS, PChar ('Set FileAttr ' + RemoteName));

  DosAttr := Lo (NewAttr);
  Packet.data2 := @DosAttr;
  if not filerRequest (CMD_SET_FILEATTR, RemoteName) then begin
    filerReset;
    exit;
  end;
  
  if extractFilePath (RemoteName) = CachedDir then begin
    if not cacheDelete (extractFileName (RemoteName)) then exit;
    strCopy (FName, RemoteName);
    if not cacheAdd (FName) then exit;
  end;  

  Result := true;
end;  

{ ------------------------------------------------------------------ }

function FsGetFile (RemoteName, LocalName : PChar; CopyFlags : integer ;
                    RemoteInfo : pRemoteInfo) : integer; stdcall;

var
  FileLength, Transferred, Delivered : longword;
  PercentDone  : integer;
  WorkFile     : file;
  Buffer       : TData;

begin
  if (CopyFlags and FS_COPYFLAGS_OVERWRITE) = 0 then
    if fileExists (LocalName) then begin
      Result := FS_FILE_EXISTS;
      exit
    end;  

  inc (RemoteName);                         // skip over leading backslash
  LogProc (PluginNumber, MsgTYPE_DETAILS, PChar ('Copy ' + RemoteName + ' to ' + LocalName));

  assignFile (WorkFile, string (LocalName));
  rewrite (WorkFile, 1);
  if IOResult <> 0 then begin
    Result := FS_FILE_WRITEERROR;
    exit;
  end;

  if not filerRequest (CMD_RESET_FILE, RemoteName) then begin
    Result := FS_FILE_NOTFOUND;
    filerReset;
    exit;
  end;
 
 if ProgressProc (PluginNumber, RemoteName, LocalName, 0) <> 0 then begin
    Result := FS_FILE_USERABORT;
    closeFile (WorkFile);
    filerRequest (CMD_CLOSE_FILE);
    exit;
  end;

  Transferred := 0;
  FileLength  := RemoteInfo^.SizeLow;

// ------- main copy loop -------
  repeat
    Packet.size := PACKET_DATA_SIZE;
    Packet.data := @Buffer;
    if not filerRequest (CMD_GET_DATA) then begin
      closeFile (WorkFile);
      filerRequest (CMD_CLOSE_FILE);
      filerReset;
      Result := FS_FILE_READERROR;
      exit;
    end;

    Delivered   := Packet.size;
    Transferred := Transferred + Delivered;
    if FileLength = 0 then
      PercentDone := 100
    else
      PercentDone := (Transferred * 100) div FileLength;

    blockWrite (WorkFile, Buffer, Delivered);
    if IOResult <> 0 then begin
      closeFile (WorkFile);
      filerRequest (CMD_CLOSE_FILE);
      Result := FS_FILE_WRITEERROR;
      exit;
    end;

    if ProgressProc (PluginNumber, RemoteName, LocalName, PercentDone) <> 0 then begin
      closeFile (WorkFile);
      filerRequest (CMD_CLOSE_FILE);
      Result := FS_FILE_USERABORT;
      exit;
    end;

  until Delivered < PACKET_DATA_SIZE;
// ------- end of main copy loop -------    

  setFileTime (TFileRec (WorkFile).Handle, nil, nil, @RemoteInfo^.LastWriteTime);

  closeFile (WorkFile);
  if not filerRequest (CMD_CLOSE_FILE) then begin
    filerReset;
    Result := FS_FILE_READERROR;
    exit;
  end;
  if IOResult <> 0 then begin
    filerReset;
    Result := FS_FILE_WRITEERROR;
    exit;
  end;

  setFileAttributes (LocalName, RemoteInfo^.Attr);

  if (CopyFlags and FS_COPYFLAGS_MOVE) <> 0 then begin          // delete remote file
    dec (RemoteName);
    if not FsDeleteFile (RemoteName) then begin
      Result := FS_FILE_READERROR;
      exit;
    end;  
  end;

  Result := FS_FILE_OK;

end;                                        // function FsGetFile

{ ------------------------------------------------------------------ }

function FsPutFile (LocalName, RemoteName : PChar; CopyFlags : integer) : integer; stdcall;

var
  FileLength, Transferred, Delivered : longword;
  PercentDone : integer;
  WorkFile    : file;
  Buffer      : TData;
  FName       : CharArr;
  SRemoteName : string;
  FileTime    : TFileTime;
  DosTime     : TDOSFileDate;
  DosAttr     : byte;

begin
  SRemoteName := RemoteName + 1;
  SRemoteName := extractFilePath (SRemoteName) + getShortFileName (extractFileName (SRemoteName));
  strCopy (FName, PChar (SRemoteName));
  Result := FS_FILE_WRITEERROR;

  if (CopyFlags and FS_COPYFLAGS_OVERWRITE) = 0 then begin      // does remote file exist?
    if not filerRequest (CMD_CHECK_FILE, FName) then exit;
    if Packet.existFlag then begin
      Result := FS_FILE_EXISTS;
      exit
    end;  
  end
  else                                      // if exists, delete it before copying 
    if extractFilePath (SRemoteName) = CachedDir then
      if not cacheDelete (extractFileName (SRemoteName)) then exit;

  LogProc (PluginNumber, MsgTYPE_DETAILS, PChar ('Copy ' + LocalName + ' to ' + FName));

  assignFile (WorkFile, string (LocalName));
  FileMode := 0;                            // open file only for reading
  reset (WorkFile, 1);
  if IOResult <> 0 then begin
    Result := FS_FILE_READERROR;
    exit;
  end;

  if not filerRequest (CMD_REWRITE_FILE, FName) then begin
    filerReset;
    Result := FS_FILE_NOTFOUND;
    exit;
  end;

  if ProgressProc (PluginNumber, @FName, LocalName, 0) <> 0 then begin
    closeFile (WorkFile);
    filerRequest (CMD_CLOSE_FILE);
    if not filerRequest (CMD_DEL_FILE, FName) then
      filerReset;
    Result := FS_FILE_USERABORT;
    exit;
  end;

  Transferred := 0;
  FileLength  := fileSize (WorkFile);
  
// --------- main copy loop -----------  
  repeat
    blockRead (WorkFile, Buffer, PACKET_DATA_SIZE, Delivered);
    if IOResult <> 0 then begin
      closeFile (WorkFile);
      filerRequest (CMD_CLOSE_FILE);
      if not filerRequest (CMD_DEL_FILE, FName) then
        filerReset;
      Result := FS_FILE_READERROR;
      exit;
    end;

    Packet.size := Delivered;
    Packet.data := @Buffer;
    if not filerRequest (CMD_SEND_DATA) then begin
      closeFile (WorkFile);
      filerReset;
      filerRequest (CMD_CLOSE_FILE);
      filerRequest (CMD_DEL_FILE, FName);
      exit;
    end;

    Transferred := Transferred + Delivered;
    if FileLength = 0 then
      PercentDone := 100
    else
      PercentDone := (Transferred * 100) div FileLength;

    if ProgressProc (PluginNumber, @FName, LocalName, PercentDone) <> 0 then begin
      closeFile (WorkFile);
      filerRequest (CMD_CLOSE_FILE);
      if not filerRequest (CMD_DEL_FILE, FName) then
        filerReset;
      Result := FS_FILE_USERABORT;
      exit;
    end;

  until Delivered < PACKET_DATA_SIZE;
// -------- end of main copy loop --------

  getFileTime (TFileRec (WorkFile).Handle, nil, nil, @FileTime); // copy file time/date
  fileTime2DosDateTime (FileTime, DosTime);
  Packet.data := @DosTime;
  if not filerRequest (CMD_SET_FILETIME) then begin
    filerReset;
    exit;
  end;

  closeFile (WorkFile);
  if not filerRequest (CMD_CLOSE_FILE) then begin
    filerReset;
    exit;
  end;
  if IOResult <> 0 then begin
    Result := FS_FILE_READERROR;
    exit;
  end;

  DosAttr := getFileAttributes (LocalName); // copy file attributes
  Packet.data2 := @DosAttr;
  if not filerRequest (CMD_SET_FILEATTR, FName) then begin
    filerReset;
    exit;
  end;

  if (CopyFlags and FS_COPYFLAGS_MOVE) <> 0 then
    if not deleteFile (LocalName) then begin
      Result := FS_FILE_READERROR;
      filerReset;
      exit;
    end;

  if extractFilePath (SRemoteName) = CachedDir then
    if not cacheAdd (FName) then exit;

  Result := FS_FILE_OK;
end;                                        // function FsPutFile

{ ------------------------------------------------------------------ }

function FsRenMovFile (OldName, NewName : PChar; Move, OverWrite : bool;
         RemoteInfo : pRemoteInfo) : integer; stdcall;

var
  FName : CharArr;

begin
  inc (OldName);                            // skip over leading backslash
  inc (NewName);

  if OldName[0] <> NewName[0] then begin    // moving to different drive is not supported
    Result := FS_FILE_NOTSUPPORTED;
    exit;
  end;  
  
  LogProc (PluginNumber, MsgTYPE_DETAILS, PChar ('Ren ' + OldName + ' to ' + NewName));
  Packet.size2 := strLen (NewName);
  Packet.data2 := TPData (NewName);
  Result := FS_FILE_OK;
  if not filerRequest (CMD_REN_FILE, OldName) then begin
    Result := FS_FILE_NOTFOUND;
    filerReset;
    exit;
  end;

  if extractFilePath (OldName) = CachedDir then
     if not cacheDelete (extractFileName (OldName)) then begin
       Result := FS_FILE_NOTFOUND;
       exit;
     end;
  if extractFilePath (NewName) = CachedDir then begin
    strCopy (FName, NewName);
    if not cacheAdd (FName) then begin
      Result := FS_FILE_WRITEERROR;
      exit;
    end;
  end;  
  
end;
  
{ ------------------------------------------------------------------ }

function FsExecuteFile (MainWin : thandle; RemoteName, Verb : PChar) : integer; stdcall;

begin
  Result := FS_EXEC_OK;

  if Verb = 'open' then
    Result := FS_EXEC_YOURSELF;
  
  if (Verb = 'properties') and (RemoteName = '\') then
    shellExecute (0, nil, 'notepad.exe', PChar (IniFileName), '', SW_SHOWNORMAL);
// else do nothing    
end;

{ ------------------------------------------------------------------ }

procedure FsGetDefRootName (DefRootName : PChar; maxlen : integer); stdcall;

begin
  strLCopy (DefRootName, PLUGIN_TITLE, maxlen - 1);
end;

{ ------------------------------------------------------------------ }

procedure FsSetDefaultParams (dps : pFsDefaultParamStruct); stdcall;

begin
  IniFileName := dps^.DefaultIniName;
end;

{ ------------------------------------------------------------------ }

exports
  FsDeleteFile,
  FsDisconnect,
  FsExecuteFile,
  FsFindClose,
  FsFindFirst,
  FsFindNext,
  FsGetDefRootName,
  FsGetFile,
  FsInit,
  FsMkDir,
  FsPutFile,
  FsRemoveDir,
  FsRenMovFile,
  FsSetAttr,
  FsSetDefaultParams,
  FsSetTime;
  
{ ------------------------------------------------------------------ }

begin
  CachedDir := #0;
end.