{****************************************}
{*           DOSINI.PAS 1.20            *}
{*                                      *}
{*       Author : Thomas Bargholz       *}
{*     Written : August 10th, 1994      *}
{*                                      *}
{*    Donated to the Public Domain      *}
{****************************************}

{ adapted for Delphi Pascal by Pavel Zampach, 2004

This unit implements a Windows style .INI file for DOS applications, by
supplying easy and fast access to the exact infomation in the .INI file
of your choise.
The Windows API supplies functions to retrieve information in .INI files,
but are (ofcource) directed at Windows. Now DOS Double and pmode programs
can use .INI files just as easy.
As in Windows, the DOS .INI files must have the format:

[section]
entry=profile

By supplying the .INI filename, the section name, and the entry name,
DOSINI will return the profile.
The .INI file can be placed anywhere, but this unit supplies two functions
to locate the DOS directory and your applications home directory, so if
you place your .INI file in one of those directories, you can locate
your .INI files easily.

if you have any comments, suggestions or bug reports, please contact me:

e-mail    : tba@m.dia.dk
snail mail: Thomas Bargholz
            Smallegade 20, 3 tv.
            DK-2000 Frederiksberg
            Denmark

Changes:

06.09.94:     Added DEFINE OPRO. Create simmilar functions of those in
              OPRO for those of you that don't have OPRO.
26.09.94:     Removed bug in WriteProfilestring. Tanks to Germano Rossi
              for giving me this bugfix. (germano@chiostro.univr.it)

}

{$I-}
unit DosIni;

interface

{------ Read a profile in a .INI file ------}

function getProfileStr (const IniName, Section, Entry, Default : string) : string;
  {- Read a string in a .INI file}

function getProfileInt (const IniName, Section, Entry : string; const Default : integer) : integer;
  {- Read a integer in a .INI file}

function getProfileDouble (const IniName, Section, Entry : string; const Default : double) : double;
  {- Read a Double in a .INI file}

{------ Write a profile to .INI file ------}

function writeProfileStr (const IniName, Section, Entry, Str : string) : integer;
  {- Write a string to a .INI file}

function writeProfileInt (const IniName, Section, Entry : string; const Int : integer) : integer;
  {- Write a integer to a .INI file}

function writeProfileDouble (const IniName, Section, Entry : string; const R : double) : integer;
  {- Write a Double to a .INI file}

{ ****************************************************************** }

implementation

uses
  SysUtils;

const
  NONSENSE = '{@}';
  
{ ------------------------------------------------------------------ }

function getProfileStr (const IniName, Section, Entry, Default : string) : string;
  {- Read a string in a .INI file}
var
  F : text;
  S : string;
  I : integer;
  
begin
  assignFile (F, IniName);
  reset (F);                                // Open the file
  if IOResult <> 0 then begin               // Success?}
    Result := Default;
    exit;
  end;
  repeat                                    // Find the right section
    readLn (F, S);
  until (ansiUpperCase (S) = ansiUpperCase('[' + Section + ']')) or (eof(F));
  
  if not eof(F) then begin
    repeat
      readLn (F, S);                        // Find the right entry
      I := Pos (ansiUpperCase (Entry + '='), ansiUpperCase (S));
    until (I = 1) or (eof(F));
    if I = 1 then                           // Extract the profile string
      S := copy (S, length(Entry) + 2, length(S))
    else
      S := '';
  end
  else
    S := '';
  closeFile (F);                            // Close the .INI file, we're done
  if S <> '' then
    Result := S
  else
    Result := Default;
end;

{ ------------------------------------------------------------------ }

function getProfileInt (const IniName, Section, Entry : string; const Default : integer) : integer;
  {- Read a integer in a .INI file}
var
  S : string;
  
begin
  S := getProfileStr (IniName, Section, Entry, NONSENSE);
  if S = NONSENSE then
    Result := Default
  else
    Result := StrToIntDef (S, Default);
end;

{ ------------------------------------------------------------------ }

function getProfileDouble (const IniName, Section, Entry : string; const Default : Double) : Double;
  {- Read a Double in a .INI file}
var
  S : string;
  R : Double;
  Error : integer;
  
begin
  S := getProfileStr(IniName, Section, Entry, NONSENSE);
  if S = NONSENSE then
    Result := Default
  else begin
    val (S, R, Error);
    if Error <> 0 then
      Result := Default
    else
      Result := R;
  end;
end;

{ ------------------------------------------------------------------ }

function WriteProfileStr (const IniName, Section, Entry, Str : string) : integer;
  {- Write a string to a .INI file}
var
  F1, F2 : Text;
  I : integer;
  S : string;
  SectionOK, EntryOK : boolean;
  
begin
  if not FileExists (IniName) then          // if the file doesn't exist, create it
  begin
    assignFile (F1, IniName);
    rewrite (F1);
    I := IOResult;
    if I <> 0 then begin
      closeFile (F1);
      Result := I;
      exit;
    end;
    writeLn (F1,'[' + Section + ']');       // write the section header
    writeLn (F1, Entry + '=' + Str);        // write the entry and the profile string
    writeLn (F1);
    closeFile (F1);
  end
  else                                      // if the file do exist
  begin
    assignFile (F1, IniName);
    reset (F1);
    I := IOResult;
    if I <> 0 then begin
      closeFile (F1);
      Result := I;
      exit;
    end;
    assignFile (F2, getEnvironmentVariable ('TEMP') + '\DOSINI$$.$$$');
    rewrite (F2);
    I := IOResult;
    if I <> 0 then begin
      Result := I;
      closeFile (F1);
      closeFile (F2);
      exit;
    end;
    SectionOK := false;
    EntryOK   := false;
    while not eof(F1) do begin
      readLn (F1, S);
      if AnsiUpperCase (S) = AnsiUpperCase ('[' + Section + ']') then begin
        SectionOK := True;                  // we've found the section
        writeLn (F2, S);
        repeat
          readLn (F1,S);
          I := Pos (AnsiUpperCase (Entry + '='), AnsiUpperCase(S));
          if I = 1 then begin
            EntryOK := True;                // we've found the entry
            writeLn (F2, Entry + '=' + Str);
          end
          else
            if (S = '') or (Pos ('[', S) = 1) then begin
              writeLn (F2, Entry + '=' + Str);
              EntryOK := True;
            end
            else
              writeLn (F2, S);
        until (Pos ('[', S) = 1) or (Eof (F1)) or (EntryOK);
        if not EntryOK then
        begin
          writeLn (F2, Entry + '=' + Str);
          writeLn (F2);
        end;
      end
      else
        writeLn (F2, S);
      if (SectionOK) and (EntryOK) then begin // we have made the change}
        repeat
          readLn (F1, S);
          writeLn (F2, S);
        until eof (F1);
        closeFile (F1);
        erase (F1);
        closeFile (F2);
        rename (F2, IniName);
        Result := 0;                        // everything OK -> Return 0
        exit;
      end;
    end;
    if not SectionOK then begin
      writeLn (F2);
      writeLn (F2, '[' + Section + ']');
      writeLn (F2, Entry + '=' + Str);
    end;
    closeFile (F1);
    erase (F1);
    closeFile (F2);
    rename (F2, IniName);
  end;
  Result := 0;                              // everything OK -> Return 0
end;

{ ------------------------------------------------------------------ }

function WriteProfileInt (const IniName, Section, Entry : string; const Int : integer) : integer;
  {- Write a integer to a .INI file}
  
begin
  Result := WriteProfileStr (IniName, Section, Entry, IntToStr (Int));
end;

{ ------------------------------------------------------------------ }

function WriteProfileDouble (const IniName, Section, Entry : string; const R : double) : integer;
  {- Write a double to a .INI file}
  
begin
  Result := WriteProfileStr (IniName, Section, Entry, FloatToStr (R));
end;



end.