unit Filer;

{
 ***** version 1.1 ***** 
 (c) Pavel Zampach (zampach@volny.cz), 2003, 2004
 GNU GENERAL PUBLIC LICENSE 
 Source code for Borland Delphi Compiler (originally ver. 7.0)
}

interface

const
  PACKET_DATA_SIZE = $0800;
  FNAME_LENGTH     = 13;

  CMD_CLOSE_FILE        = $01;              //   Filer communications commands
  CMD_CREATE_FILE       = $03;
  CMD_DEL_FILE          = $04;
  CMD_REWRITE_FILE      = $05;
  CMD_GET_ACTIVEDIR     = $06;
  CMD_GET_ACTIVEDISK    = $07;
  CMD_CHECK_FILE        = $08;
  CMD_GET_DIR           = $09;
  CMD_MAKE_DIR          = $0A;
  CMD_RESET_FILE        = $0B;
  CMD_GET_DATA          = $0C;
  CMD_REN_FILE          = $0D;
  CMD_DEL_DIR           = $0E;
  CMD_SET_FILEPOINTER   = $0F;
  CMD_SET_DIR           = $10;
  CMD_SET_DISK          = $11;
  CMD_ASK_DIR           = $12;
  CMD_GET_FILEPOINTER   = $13;
  CMD_GET_DISKINFO      = $14;
  CMD_SEND_DATA         = $15;
  CMD_INIT_GET          = $16;
  CMD_GET_FILETIME      = $17;
  CMD_SET_FILETIME      = $18;
  CMD_GET_FILEATTR      = $19;
  CMD_SET_FILEATTR      = $1A;
  CMD_GET_DISKVECTOR    = $1B;
  CMD_CONNECT_SERVER    = $40;
  CMD_DISCONNECT_SERVER = $41;
  
 
type
  TData = array [1..PACKET_DATA_SIZE] of byte;
  TPData = ^TData;

  TDirEntry = packed record
    fattr  : byte;
    ftime  : word;
    fdate  : word;
    fsize  : longword;
    fname  : array [0..FNAME_LENGTH-1] of char;
  end;

  TDOSFileDate = packed record
    ftime  : word;
    fdate  : word;
  end;

  TDiskInfo = packed record
    volume  : array [1..11] of char;
    free    : longword;
  end;

var
  Packet :  record                          //  Data for packet traffic
    size      : word;
    size2     : word;
    data      : TPData;
    data2     : TPData;
    existFlag : boolean;
  end;
       
  Connected : boolean;

function filerReset : boolean;
function filerRequest (Funct : byte) : boolean; overload;
function filerRequest (Funct : byte; Data : PChar) : boolean; overload;
function filerConnect (const CommPort : string; const CommSpeed : integer) : boolean;
function filerDisconnect : boolean;

{ ****************************************************************** }

implementation

uses
  SysUtils,
  Windows;

const
  NUL         = 0;
  SOH         = 1;
  STX         = 2;
  STATUS_OK   = $81;
  MAX_PKT_LEN = PACKET_DATA_SIZE*2+100;
  SIG_LEN     = 5;
  
// packet signature
  Signature : array [1..SIG_LEN] of byte = ($16, $16, $16, $10, $02);

type
  TWorkBuf = array [1..MAX_PKT_LEN] of byte;

var
  hCommFile : THandle;
  PacketFunct, PacketCount, PacketStatus : byte;

{ ------------------------------------------------------------------ }

function updateCRC16 (CRC : word; Data : byte) : word;

var
  i : byte;

begin
  Result := CRC xor Data;
  for i := 1 to 8 do
    if (Result and 1) = 1 then
      Result := (Result shr 1) xor $A001
    else
      Result := Result shr 1;
end;

{ ------------------------------------------------------------------ }

procedure sendPacket;

var
  Checksum, i, WorkSize : word;
  WorkBuf : TWorkBuf;
  BytesWritten : LongWord;

{ -------------- }

procedure sendByte (value : byte; Chsum : boolean);

begin
  inc (WorkSize);
  WorkBuf[WorkSize] := value;
  if (value = $10) and Chsum then begin     // '$10' CRC trick, got to send it twice
    inc (WorkSize);
    WorkBuf[WorkSize] := value;
  end;
  if Chsum then                             // update checksum
    Checksum := updateCRC16 (Checksum, value);
end;                                        // function sendByte

{ -------------- }

begin                                       // function sendPacket
  Checksum := 0;
  WorkSize := 0;
  
  if not Connected then                     // synchonization - 4 NUL bytes
    for i := 1 to 4 do
      sendByte (NUL, false);
      
  for i := 1 to SIG_LEN do                  // send signature, no CRC yet
    sendByte (Signature[i], false);
  sendByte (SOH, true);
  sendByte (PacketFunct, true);
  sendByte (PacketCount, true);
  sendByte (SOH, true);                     // send Data marker
  sendByte (STX, true);

  if PacketFunct = CMD_SEND_DATA then begin
    sendByte (0, true);
    sendByte (0, true);
    sendByte (Lo (Packet.size), true);
    sendByte (Hi (Packet.size), true);
    for i := 1 to Packet.size do
      sendByte (Packet.data^[i], true);
  end;

  if (PacketFunct = CMD_MAKE_DIR) or
     (PacketFunct = CMD_DEL_DIR) or
     (PacketFunct = CMD_DEL_FILE) or
     (PacketFunct = CMD_CHECK_FILE) or
     (PacketFunct = CMD_CREATE_FILE) or
     (PacketFunct = CMD_REWRITE_FILE) or
     (PacketFunct = CMD_RESET_FILE) or
     (PacketFunct = CMD_REN_FILE) or
     (PacketFunct = CMD_SET_DIR) or
     (PacketFunct = CMD_SET_FILEATTR) or
     (PacketFunct = CMD_GET_FILEATTR) or
     (PacketFunct = CMD_ASK_DIR) then
  begin
    sendByte (Lo (Packet.size), true);      // send path/filename (FROM) size LO, HI
    sendByte (Hi (Packet.size), true);
    for i := 1 to Packet.size do            // send path/filename (FROM)
      sendByte (Packet.data^[i], true);
    if  PacketFunct = CMD_SET_FILEATTR then
      sendByte (Packet.data2^[1], true)     // send FileAttr
    else  
      sendByte (0, true);                 // send Data End Marker
    sendByte (0, true);
  end;

  if PacketFunct = CMD_REN_FILE then begin
    sendByte (Lo (Packet.size2), true);     // send filename TO size LO, HI
    sendByte (Hi (Packet.size2), true);
    for i := 1 to Packet.size2 do           // send filename TO
      sendByte (Packet.data2^[i], true);
    sendByte (0, true);                   // send Data End Marker
    sendByte (0, true);
  end;

  if PacketFunct = CMD_GET_DATA then begin
    sendByte (0, true);                     // double word size, MSB always zero
    sendByte (0, true);
    sendByte (Lo (Packet.size), true);
    sendByte (Hi (Packet.size), true);
  end;

  if PacketFunct = CMD_SET_FILETIME then begin
    sendByte (0, true);                     // double word size, MSB always zero
    sendByte (0, true);
    for i := 1 to 4 do                      // send date and time
      sendByte (Packet.data^[i], true);
  end;

  if (PacketFunct = CMD_CLOSE_FILE) or
     (PacketFunct = CMD_GET_ACTIVEDISK) or
     (PacketFunct = CMD_GET_DISKVECTOR) or
     (PacketFunct = CMD_GET_FILEPOINTER) or
     (PacketFunct = CMD_GET_FILETIME) then
  begin
    sendByte (0, true);
    sendByte (0, true);
  end;  

  if (PacketFunct = CMD_SET_DISK) or
     (PacketFunct = CMD_GET_ACTIVEDIR) then
  begin
    sendByte (Packet.data^[1], true);       // send disk
    sendByte (0, true);
  end;

  if PacketFunct = CMD_SET_FILEPOINTER then begin
    sendByte (0, true);
    sendByte (0, true);
    sendByte (0, true);
    sendByte (0, true);
    for i := 1 to 4 do                      // send date and time
      sendByte (Packet.data^[i], true);
  end;  

// ----- COMMON PART -----
  sendByte ($10, false);                    // no CRC on this one (tricky!)
  sendByte ($03, true);
  sendByte (Lo (Checksum), false);          // finally send Checksum LO, HI
  sendByte (Hi (Checksum), false);

  writeFile (hCommFile, WorkBuf, WorkSize, BytesWritten, nil);  // really send packet from buffer
end;                                        // function sendPacket

{ ------------------------------------------------------------------ }

function getPacket : boolean;

var
  c, Data, ByteHi, ByteLo : byte;
  i, Checksum, EstimatePktLen, WorkPtr  : word;
  BytesRead : longword;
  WorkBuf : TWorkBuf;

{ -------------- }

function getByte (Chsum : boolean) : byte;

begin
  if WorkPtr = BytesRead then abort;
  inc (WorkPtr);
  Result := WorkBuf[WorkPtr];
  if (Result = $10) and Chsum then begin    // '$10' CRC trick, receive it twice
    if WorkPtr = BytesRead then abort;
    inc (WorkPtr);
    Result := WorkBuf[WorkPtr];
  end;
  if Chsum then                             // update checksum
    Checksum := updateCRC16 (Checksum, Result);
end;                                        // function getByte

{ -------------- }

procedure testNUL (NoN : integer);

VAR i : integer;

begin
  for i := 1 to NoN do
    if getByte (true) <> NUL then abort;
end;

{ -------------- }

begin
  Checksum := 0;
  Result   := true;
  WorkPtr  := 0;
  if PacketFunct = CMD_GET_DATA then
    EstimatePktLen := MAX_PKT_LEN
  else
    EstimatePktLen := 100;

  try

  readFile (hCommFile, WorkBuf, EstimatePktLen, BytesRead, nil);
                                            // really read packet into buffer, end by timeout
  c := 0;                                   // search signature
  repeat
    inc (c);
    if getByte (false) <> Signature[c] then c := 0;
  until c = SIG_LEN;

  if getByte (true) <> SOH then abort;      // get SOH
  PacketFunct  := getByte (true);
  PacketCount  := getByte (true);
  PacketStatus := getByte (true);
  if getByte (true) <> STX then abort;      // get Data marker

  if (PacketFunct = CMD_CONNECT_SERVER) or
     (PacketFunct = CMD_DISCONNECT_SERVER) then
    testNUL (2);                            // Data end marker

  if  (PacketFunct = CMD_SEND_DATA) then
  begin
    testNUL (4);                            // get Data marker and size MSB (always 0)
    ByteLo := getByte (true);               // get Data size
    ByteHi := getByte (true);
    Packet.size := (ByteHi * $100) + ByteLo;
  end;

  if (PacketFunct = CMD_CHECK_FILE) then begin
    testNUL (4);                            // get Data marker and size MSB (always 0)
    Data := getByte (true);                 // result marker    
    if Data > 2 then abort;
    Packet.existFlag := (Data <> NUL);
    testNUL (1);
  end;

  if (PacketFunct = CMD_GET_FILETIME) or
     (PacketFunct = CMD_GET_FILEPOINTER) then
  begin
    testNUL (4);                            // get Data marker and size MSB (always 0)
    Packet.size := 4;
    for i := 1 to 4 do                      // get date and time
      Packet.data^[i] := getByte (true);
  end;

  if (PacketFunct = CMD_MAKE_DIR) or
     (PacketFunct = CMD_DEL_DIR) or
     (PacketFunct = CMD_DEL_FILE) or
     (PacketFunct = CMD_REN_FILE) or
     (PacketFunct = CMD_CREATE_FILE) or
     (PacketFunct = CMD_REWRITE_FILE) or
     (PacketFunct = CMD_RESET_FILE) or
     (PacketFunct = CMD_CLOSE_FILE) or
     (PacketFunct = CMD_INIT_GET) or
     (PacketFunct = CMD_SET_FILETIME) or
     (PacketFunct = CMD_SET_FILEATTR) or
     (PacketFunct = CMD_SET_FILEPOINTER) or
     (PacketFunct = CMD_SET_DISK) or
     (PacketFunct = CMD_SET_DIR) or
     (PacketFunct = CMD_ASK_DIR) then
    testNUL (6);                            // get Data size & end marker (6 bytes)

  if PacketFunct = CMD_GET_DATA then begin
    testNUL (4);                            // get Data marker and size MSB (always 0)
    ByteLo := getByte (true);               // get Data size
    ByteHi := getByte (true);
    Packet.size := (ByteHi * $100) + ByteLo;
    for i := 1 to Packet.size do
      Packet.data^[i] := getByte (true);
  end;

  if (PacketFunct = CMD_GET_ACTIVEDIR) or
     (PacketFunct = CMD_GET_DISKVECTOR) then
  begin
    testNUL (4);                            // get Data marker and size MSB (always 0)
    ByteLo := getByte (true);               // get Data size
    ByteHi := getByte (true);
    Packet.size := (ByteHi * $100) + ByteLo;
    for i := 1 to Packet.size do
      Packet.data^[i] := getByte (true);
    testNUL (4);
  end;

  if PacketFunct = CMD_GET_DIR then begin
    testNUL (4);                            // get Data marker and size MSB (always 0)
    ByteLo := getByte (true);               // get Data size
    ByteHi := getByte (true);
    Packet.size := (ByteHi * $100) + ByteLo;
    for i := 1 to Packet.size do            // get Data
      Packet.data^[i] := getByte (true);
    Data := getByte (true);                 // special 'dir' Data end marker    
    if Data > 1 then abort;
    Packet.existFlag := (Data <> NUL);
    testNUL (3);
  end;

  if PacketFunct = CMD_GET_DISKINFO then begin
    testNUL (4);                            // get Data marker and size MSB (always 0)
    ByteLo := getByte (true);               // get volume size
    ByteHi := getByte (true);
    Packet.size := ((ByteHi * $100) + ByteLo) + 4; // Packet.size = 11 + 4 (always)
    for i := 1 to Packet.size do            // get data
      Packet.data^[i] := getByte (true);
  end;

  if PacketFunct = CMD_GET_ACTIVEDISK then begin
    testNUL (4);                            // get Data marker and size MSB (always 0)
    Packet.size := 1;
    Packet.data^[1] := getByte (true);
    testNUL (1);
  end;

  if PacketFunct = CMD_GET_FILEATTR then begin
    testNUL (4);                            // get Data marker and size MSB (always 0)
    Packet.size := 1;
    Packet.data2^[1] := getByte (true);
    testNUL (1);
  end;

// ----- COMMON PART -----
  if getByte (false) <> $10 then abort;     // get CRC Marker (no CRC on the this byte!)
  if getByte (true)  <> $03 then abort;
  ByteLo := getByte (false);                // get received CRC
  ByteHi := getByte (false);
  if ((ByteHi * $100) + ByteLo) <> Checksum then abort; 
                                            // check if CRC is good
  except
    Result := false;
  end;                                      // except  
end;                                        // function getPacket

{ ------------------------------------------------------------------ }

function filerRequest (Funct : byte) : boolean;  overload;

const
  MAX_ATT = 3;

var
  Att, Count : byte;
  BgetPacket : boolean;

begin
  Result := false;
  if Packet.size > PACKET_DATA_SIZE then exit;
  if (Funct = CMD_CONNECT_SERVER) or
     (Funct = CMD_DISCONNECT_SERVER) then PacketCount := 0;
  Count := PacketCount;
  PacketFunct := Funct;

  Att := 0;                                 // packet handshake
  repeat
    purgeComm (hCommFile, PURGE_TXCLEAR or PURGE_RXCLEAR);
    sendPacket;
    inc (Att);
    BgetPacket := getPacket;
  until BgetPacket or (Att = MAX_ATT);

  if (not BgetPacket) or                    // check received values
     (PacketStatus <> STATUS_OK) or
     (PacketCount  <> Count) or
     (PacketFunct  <> Funct) then exit;

  if PacketCount = $FF then                 // cyclic update packet counter
    PacketCount := 0
  else
    inc (PacketCount);
    
  Result := true;
end;                                        // funct filerRequest

{ ------------------------------------------------------------------ }

function filerRequest (Funct : byte; Data : Pchar) : boolean; overload;

begin
  Packet.data := TPData (Data);
  Packet.size := strLen (Data);
  Result := filerRequest (Funct);
end;                                        // funct filerRequest with filename

{ ------------------------------------------------------------------ }

function filerReset : boolean;

begin
  sleep (300);
  purgeComm (hCommFile, PURGE_TXABORT or PURGE_RXABORT or PURGE_TXCLEAR or PURGE_RXCLEAR);
  Result := filerRequest (CMD_CONNECT_SERVER);
end;

{ ------------------------------------------------------------------ }

function filerConnect (const CommPort : string; const CommSpeed : integer) : boolean;

const
  RX_BUF = $C00;
  TX_BUF = $C00;

  fEmpty               = $00000000;
  fBinary              = $00000001;
  fParity              = $00000002;
  fOutxCtsFlow         = $00000004;
  fOutxDsrFlow         = $00000008;
  fDtrControlEnable    = $00000010;
  fDtrControlHandshake = $00000020;
  fDsrSensitivity      = $00000040;
  fTXContinueOnXoff    = $00000080;
  fOutX                = $00000100;
  fInX                 = $00000200;
  fErrorChar           = $00000400;
  fNull                = $00000800;
  fRtsControlEnable    = $00001000;
  fRtsControlHandshake = $00002000;
  fAbortOnError        = $00004000;
  fDummy2              = $FFFF8000;


var
  DcbPort          : TDCB;
  CommPortTimeouts : TCommTimeouts;
  SpeedCorr        : word;

begin
  Result := true;
  if Connected then exit;

  try

// open file for serial communications
  hCommFile := CreateFile (PChar(CommPort), GENERIC_READ or GENERIC_WRITE, 0, nil, OPEN_EXISTING, 0, 0);
  if hCommFile = INVALID_HANDLE_VALUE then abort;
  purgeComm (hCommFile, PURGE_TXABORT or PURGE_RXABORT or PURGE_TXCLEAR or PURGE_RXCLEAR);

  if not getCommState (hCommFile, dcbPort) then abort; 
  with DcbPort do begin                     // set port parameters
    BaudRate := CommSpeed;
    ByteSize := 8;
    Parity   := NOPARITY;
    StopBits := ONESTOPBIT;
    Flags    := fEmpty;                     // no handshake!
  end;
  setCommState (hCommFile, DcbPort);

  SpeedCorr := 10000 div CommSpeed;         // set port timeouts
  if not getCommTimeouts (hCommFile, CommPortTimeouts) then abort;
  with CommPortTimeouts do begin
    ReadIntervalTimeout         := 50 + SpeedCorr;
    ReadTotalTimeoutMultiplier  := 1 + SpeedCorr;
    ReadTotalTimeoutConstant    := 1000;
    WriteTotalTimeoutMultiplier := 1 + SpeedCorr;
    WriteTotalTimeoutConstant   := 1000;
  end;
  setCommTimeouts (hCommFile, CommPortTimeouts);

  setupComm (hCommFile, RX_BUF, TX_BUF);    // set recieve and transmit buffer size

  if not filerRequest (CMD_CONNECT_SERVER) then abort;

  Connected := true;

  except
    sleep (300);
    purgeComm (hCommFile, PURGE_TXABORT or PURGE_RXABORT or PURGE_TXCLEAR or PURGE_RXCLEAR);
    closeHandle (hCommFile);
    Result := false;
  end;                                      // except  
end;                                        // funct FilerConnect

{ ------------------------------------------------------------------ }

function filerDisconnect : boolean;

begin
  if not Connected then
    Result := true
  else begin
    Result := filerRequest (CMD_DISCONNECT_SERVER);
    closeHandle (hCommFile);
    Connected := false;
  end;
end;

{ ------------------------------------------------------------------ }

initialization
  Connected := false;
end.