#include <dos.h>
#include <stdarg.h>

#define VERSION "1.33"

#include "lang.h"
#include "wmix.h"
#include "wio.h"
#include "ftn.h"
#include "bbs.h"
#include "if.h"
#include "name.h"
#include "tcoll.h"

#include "syslog.h"

static const char *ssw[][3] =
{
   { "COPY",      "Y", LNG_HLP_COPY      }
  ,{ "MOVE",      "M", LNG_HLP_MOVE      }
  ,{ "OVERWRITE", "O", LNG_HLP_OVERWRITE }
  ,{ "APPEND",    "A", LNG_HLP_APPEND    }
  ,{ "SKIP",      "X", LNG_HLP_SKIP      }
  ,{ "DIFFONLY",  "D", LNG_HLP_DIFFONLY  }
  ,{ "EXISTONLY", "E", LNG_HLP_EXISTONLY }
  ,{ "NEWONLY",   "N", LNG_HLP_NEWONLY   }
  ,{ "CLEARATTR", "C", LNG_HLP_CLEARATTR }
  ,{ "DIGITSET",  "9", LNG_HLP_DIGITSET  }
  ,{ "KEEPEXT",   "K", LNG_HLP_KEEPEXT   }
  ,{ "NOSOUND",   "S", LNG_HLP_NOSOUND   }
  ,{ "USEPATH",   "P", LNG_HLP_USEPATH   }
  ,{ "BINKSTYLE", "B", LNG_HLP_BINKSTYLE }
  ,{ "ADDHOME",   "H", LNG_HLP_ADDHOME   }
  ,{ "EXTMASK",   "+", LNG_HLP_EXTMASK   }
  ,{ "FILESBBS",  "F", LNG_HLP_FILESBBS  }
  ,{ "RECURSIVE", "R", LNG_HLP_RECURSIVE }
  ,{ "TREE",      "T", LNG_HLP_TREE      }
  ,{ "FLOPPY",    "#", LNG_HLP_FLOPPY    }
  ,{ "REWRITE",   "W", LNG_HLP_REWRITE   }
  ,{ "TRYRENAME", "=", LNG_HLP_TRYRENAME }
#ifdef __WIN32__
  ,{ "DOSNAMES",  "8", LNG_HLP_DOSNAMES  }
  ,{ "OEM2ANSI",  "2", LNG_HLP_OEM2ANSI  }
#endif
};

static int usage(void)
{
  static const char *sw2[] =
  {
    LNG_HLP_NOENV, LNG_HLP_EXCLUDE, LNG_HLP_ADDRESS, LNG_HLP_BASEADDR, LNG_HLP_REN,
    LNG_HLP_LOG,   LNG_HLP_MSG,     LNG_HLP_TRY,     LNG_HLP_LIMIT,    LNG_HLP_IF
  };
  static const char *sw3[] =
  {
    LNG_HLP_IF_LT, LNG_HLP_IF_LE, LNG_HLP_IF_EQ, LNG_HLP_IF_NE, LNG_HLP_IF_GE, LNG_HLP_IF_GT
  };
  static const char *sw4[] =
  {
    LNG_HLP_IF_FILE, LNG_HLP_IF_DATE, LNG_HLP_IF_DAYS, LNG_HLP_IF_TIME, LNG_HLP_IF_SIZE
  };
  static const char *sw5[] =
  {
    LNG_HLP_MACRO_A, LNG_HLP_MACRO_B, LNG_HLP_MACRO_L, LNG_HLP_MACRO_D, LNG_HLP_MACRO_M,
    LNG_HLP_MACRO_C, LNG_HLP_MACRO_Y, LNG_HLP_MACRO_H, LNG_HLP_MACRO_T, LNG_HLP_MACRO_S
  };
  static const char *ex[][3] =
  {
    { LNG_HLP_EX_S1, LNG_HLP_EX_T11, LNG_HLP_EX_T12 },
    { LNG_HLP_EX_S2, LNG_HLP_EX_T21, LNG_HLP_EX_T22 },
    { LNG_HLP_EX_S3, LNG_HLP_EX_T31, LNG_HLP_EX_T32 },
    { LNG_HLP_EX_S4, LNG_HLP_EX_T41, LNG_HLP_EX_T42 },
    { LNG_HLP_EX_S5, LNG_HLP_EX_T51, LNG_HLP_EX_T52 },
    { LNG_HLP_EX_S6, LNG_HLP_EX_T61, LNG_HLP_EX_T62 },
    { LNG_HLP_EX_S7, LNG_HLP_EX_T71, LNG_HLP_EX_T72 }
  };
  printf(LNG_HLP_USAGE0 LNG_HLP_USAGE1 LNG_HLP_USAGE2 LNG_HLP_USAGE3 LNG_HLP_USAGE4);
  for ( int i = 0 ; i < dimensionOf(ssw) ; i++ )
    printf(" /%-14s/%s - %s\n", ssw[i][0], ssw[i][1], ssw[i][2]);
  for ( int i = 0 ; i < dimensionOf(sw2) ; i++ )
    printf(" /%s\n", sw2[i]);
  printf(LNG_HLP_USAGE5);
  for ( int i = 0 ; i < dimensionOf(sw3) ; i++ )
    printf(" %s\n", sw3[i]);
  printf(LNG_HLP_USAGE6);
  for ( int i = 0 ; i < dimensionOf(sw4) ; i++ )
    printf(" %s\n", sw4[i]);
  printf(LNG_HLP_USAGE7);
  for ( int i = 0 ; i < dimensionOf(sw5) ; i++ )
    printf(" =%s\n", sw5[i]);
  printf(LNG_HLP_EXAMPLES);
  for ( int i = 0 ; i < dimensionOf(ex) ; i++ )
    printf(" WCOPY %-35s - %s\n%45s%s\n", ex[i][0], ex[i][1], "", ex[i][2]);
  return 2;
}

static TFTNAddress addr;
static unsigned short gl_defFTNZone = 2, bk_defFTNZone = 2;

#ifdef __WIN32__
static int   gl_dosNames       = 0, bk_dosNames       = 0,
             gl_Oem2Ansi       = 0, bk_Oem2Ansi       = 0;
#endif
static int   gl_maxTryes       = 0, bk_maxTryes       = 0,
             gl_noSoundMode    = 0, bk_noSoundMode    = 0,
             gl_logMode        = 0, bk_logMode        = 0,
             gl_msgMode        = 0, bk_msgMode        = 0,
             gl_moveMode       = 0, bk_moveMode       = 0,
             gl_binkStyle      = 0, bk_binkStyle      = 0,
             gl_overWrite      = 0, bk_overWrite      = 0,
             gl_keepExt        = 0, bk_keepExt        = 0,
             gl_digitSet       = 0, bk_digitSet       = 0,
             gl_addHome        = 0, bk_addHome        = 0,
             gl_newOnly        = 0, bk_newOnly        = 0,
             gl_diffOnly       = 0, bk_diffOnly       = 0,
             gl_existOnly      = 0, bk_existOnly      = 0,
             gl_clearAttr      = 0, bk_clearAttr      = 0,
             gl_usePath        = 0, bk_usePath        = 0,
             gl_exclude        = 0, bk_exclude        = 0,
             gl_extMask        = 0, bk_extMask        = 0,
             gl_append         = 0, bk_append         = 0,
             gl_filesBBS       = 0, bk_filesBBS       = 0,
             gl_recursive      = 0, bk_recursive      = 0,
             gl_tree           = 0, bk_tree           = 0,
             gl_copyAlwaysMode = 0, bk_copyAlwaysMode = 0,
             gl_floppyMode     = 0, bk_floppyMode     = 0,
             gl_address        = 0, bk_address        = 0,
             gl_rewriteMode    = 0, bk_rewriteMode    = 0,
             gl_useRen         = 1, bk_useRen         = 1, ///???
             gl_skipAll        = 0, bk_skipAll        = 0,
             gl_renMode        = 0, bk_renMode        = 0,
             gl_addr_b         = 0, bk_addr_b         = 0,
             gl_addr_r         = 0, bk_addr_r         = 0;
static long  gl_sizeLimit     = -1, bk_sizeLimit     = -1;
static char  gl_adb[40]      = "",  bk_adb[40]      = "",
             gl_adr[40]      = "",  bk_adr[40]      = "",
             gl_ren[MAXPATH] = "*", bk_ren[MAXPATH] = "*",
             gl_log[MAXPATH] = "",  bk_log[MAXPATH] = "",
             gl_msg[MAXPATH] = "",  bk_msg[MAXPATH] = "",
             gl_xcl[MAXPATH] = "",  bk_xcl[MAXPATH] = "";
char        *gl_ifSize     = NULL, *gl_ifDate     = NULL, *gl_ifTime     = NULL;
static char *bk_ifSize     = NULL, *bk_ifDate     = NULL, *bk_ifTime     = NULL;

static char *copyMode[] = { LNG_MODE_MOVE, LNG_MODE_COPY };
static char *descFile = "FILES.BBS";

#ifdef __WIN32__
static inline char* char2oem(char *s) { CharToOem(s, s); return s; }
static inline char* oem2char(char *s) { OemToChar(s, s); return s; }
#else
static inline char* char2oem(char *s) { return s; }
static inline char* oem2char(char *s) { return s; }
#endif

static int copyFile(HANDLE inHandle, HANDLE outHandle, long size)
{
  long tr = 0;
  FILETIME t;
  size_t r;
  GetFileTime(inHandle, NULL, NULL, &t);
  do
  {
    r = readBlock(inHandle, IOBuffer, IOBufferSize);
    if ( writeBlock(outHandle, IOBuffer, r) != r )
      return 0;
    cprintf("\r%3ld%%%s\r", (tr += r)*100/(size ? size : 1), LNG_MODE_FIXLEN);
  } while ( r );
  SetFileTime(outHandle, NULL, NULL, &t);
  return tr == size;
}

static char *strCurrentTime(void)
{
  static char s[30];
#ifdef __WIN32__
  SYSTEMTIME t;
  GetLocalTime(&t);
  sprintf(s, "%02d/%02d %02d:%02d:%02d ", t.wMonth, t.wDay, t.wHour, t.wMinute, t.wSecond);
#else
  time_t t = time(NULL);
  strftime(s, 30, "%m/%d %H:%M:%S ", localtime(&t));
#endif
  return s;
}

static void lprintf(unsigned char attr, const char * fmt, ...)
{
  static char buffer[128];
  HANDLE f = INVALID_HANDLE_VALUE;
  if ( gl_logMode)
    f = openFile(gl_log, 1, 1, "a");
  if ( fmt )
  {
    va_list v;
    va_start(v, fmt);
    if ( f != INVALID_HANDLE_VALUE )
    {
      strcpy(buffer, strCurrentTime());
      switch ( attr )
      {
        case WHITE:
        case LIGHTCYAN:
          strcat(buffer, "> ");
          break;
        case CYAN:
          strcat(buffer, "- ");
          break;
        case YELLOW:
          strcat(buffer, "! ");
          break;
         default:
          strcat(buffer, "  ");
          break;
      }
    }
    else
      *buffer = 0;
    char *cp = strchr(buffer, 0);
    vsprintf(cp, fmt, v);
    va_end(v);
    if ( f )
    {
      putString(f, buffer);
      putString(f, "\r\n");
    }
    text_info textInfo;
    gettextinfo(&textInfo);
    textattr(attr ? attr : LIGHTGRAY);
#ifdef __WIN32__
    clreolattr(attr ? attr : LIGHTGRAY);
#endif
    putch('\r');
    cputs(cp);
    clreol();
    cputs("\r\n");
    textattr(textInfo.attribute);
  }
  CloseHandle(f);
}

static inline int parmMatch(const char *p, int n)
{
  return !stricmp(p, ssw[n][0]) || !stricmp(p, ssw[n][1]);
}

static void makeLine(char *(*w)[2], size_t n, char *buffer, const char *mask)
{
  char *m = (char*)mask;
  char *b = buffer;
  *b = 0;
  while ( *m )
  {
    int found = 0;
    for ( size_t i = 0 ; i < n ; i++ )
    {
      int l = strlen(w[i][0]);
      if ( !strnicmp(m, w[i][0], l) )
      {
        m += l;
        strcpy(b, w[i][1]);
        b += strlen(w[i][1]);
        found = 1;
        break;
      }
    }
    if ( !found )
      *b++ = *m++;
  }
  *b = 0;
}

static char *to32(unsigned long n, int len)
{
  register i;
  static char s[12], t[12];
  ultoa(n, t, 32);
  for ( i = 0 ; i < len ; i++ )
    s[i] = '0';
  i = len-strlen(t);
  strcpy(s+i, t);
  return strupr(s);
}

static char *expandMacro(char *s)
{
  char buffer[256];
  char dt[][32] = { "", "", "", "", "", "", "", "", "", "" };
  for ( int i = 0 ; i < 5 ; i++ )
    *dt[i] = 0;
  if ( gl_address )
  {
    sprintf(dt[0], "%u.%u.%u.%u", addr.zone, addr.net, addr.node, addr.point);
    strcpy(dt[1], to32(addr.zone,  2));
    strcat(dt[1], to32(addr.net,   3));
    strcat(dt[1], to32(addr.node,  3));
    strcat(dt[1], ".");
    strcat(dt[1], to32(addr.point, 2));
    if ( addr.zone == gl_defFTNZone )
      *dt[2] = 0;
    else
      sprintf(dt[2], ".%03X", addr.zone);
    sprintf(buffer, "%s\\%04X%04X", dt[2], addr.net, addr.node);
    if ( addr.point )
      sprintf(dt[2], "%s.PNT\\%08X", buffer, addr.point);
    else
      strcpy(dt[2], buffer);
  }
#ifdef __WIN32__
  SYSTEMTIME t;
  GetLocalTime(&t);
  sprintf(dt[3], "%02d", t.wDay);
  sprintf(dt[4], "%02d", t.wMonth);
  sprintf(dt[5], "%04d", t.wYear);
  sprintf(dt[6], "%02d", t.wYear % 100);
  sprintf(dt[7], "%02d", t.wHour);
  sprintf(dt[8], "%02d", t.wMinute);
  sprintf(dt[9], "%02d", t.wSecond);
#else
  time_t t = time(NULL);
  strftime(dt[3], 5, "%d", localtime(&t));
  strftime(dt[4], 5, "%m", localtime(&t));
  strftime(dt[5], 5, "%Y", localtime(&t));
  strftime(dt[6], 5, "%y", localtime(&t));
  strftime(dt[7], 5, "%H", localtime(&t));
  strftime(dt[8], 5, "%M", localtime(&t));
  strftime(dt[9], 5, "%S", localtime(&t));
#endif
  char * w[][2] =
  {
    { "=A", dt[0] },
    { "=B", dt[1] },
    { "=L", dt[2] },
    { "=D", dt[3] },
    { "=M", dt[4] },
    { "=C", dt[5] },
    { "=Y", dt[6] },
    { "=H", dt[7] },
    { "=T", dt[8] },
    { "=S", dt[9] }
  };
  makeLine(w, dimensionOf(w), buffer, s);
  return strcpy(s, buffer);
}

static int parmM_arg(const char *p, const char *v, char **arg)
{
  size_t len = strlen(v);
  if ( !strnicmp(p, v, len) )
  {
    if ( *(*arg = (char*)(p+len)) == ':' )
      ++*arg;
    return 1;
  }
  return 0;
}

static int setParameter(int ga, char *p)
{
  if ( stricmp(p, "NOENV") )
  {
    char *a;
    if      ( parmMatch(p,  0) )       gl_copyAlwaysMode         = 1;
    else if ( parmMatch(p,  1) )       gl_moveMode               = 1;
    else if ( parmMatch(p,  2) )       gl_overWrite              = 1;
    else if ( parmMatch(p,  3) )       gl_append                 = 1;
    else if ( parmMatch(p,  4) )       gl_skipAll                = 1;
    else if ( parmMatch(p,  5) )       gl_diffOnly               = 1;
    else if ( parmMatch(p,  6) )       gl_existOnly              = 1;
    else if ( parmMatch(p,  7) )       gl_newOnly                = 1;
    else if ( parmMatch(p,  8) )       gl_clearAttr              = 1;
    else if ( parmMatch(p,  9) )       gl_digitSet               = 1;
    else if ( parmMatch(p, 10) )       gl_keepExt                = 1;
    else if ( parmMatch(p, 11) )       gl_noSoundMode            = 1;
    else if ( parmMatch(p, 12) )       gl_usePath                = 1;
    else if ( parmMatch(p, 13) )       gl_binkStyle = gl_usePath = 1;
    else if ( parmMatch(p, 14) )       gl_addHome                = 1;
    else if ( parmMatch(p, 15) )       gl_extMask                = 1;
    else if ( parmMatch(p, 16) )       gl_filesBBS               = 1;
    else if ( parmMatch(p, 17) )       gl_recursive              = 1;
    else if ( parmMatch(p, 18) )       gl_recursive = gl_tree    = 1;
    else if ( parmMatch(p, 19) )       gl_floppyMode             = 1;
    else if ( parmMatch(p, 20) )       gl_rewriteMode            = 1;
    else if ( parmMatch(p, 21) )       gl_useRen                 = 1;
#ifdef __WIN32__
    else if ( parmMatch(p, 22) )       gl_dosNames               = 1;
    else if ( parmMatch(p, 23) )       gl_Oem2Ansi               = 1;
#endif
    else if ( parmM_arg(p, "@@",      &a) )
      if ( ga )
        gl_addr_b = strcpy(gl_adb, a) != NULL;
      else
      {
        lprintf(YELLOW, "%s: /%s", LNG_MSG_MUSTBEGLOBAL);
        return 0;
      }
    else if ( parmM_arg(p, "@",       &a) )
      if ( ga )
        gl_addr_r   = strcpy(gl_adr, a) != NULL;
      else
      {
        lprintf(YELLOW, "%s: /%s", LNG_MSG_MUSTBEGLOBAL);
        return 0;
      }
    else if ( parmM_arg(p, "-",       &a) ) gl_exclude  = strcpy(gl_xcl, a) != NULL;
    else if ( parmM_arg(p, "REN",     &a) ) gl_renMode  = strcpy(gl_ren, a) != NULL;
    else if ( parmM_arg(p, "LOG",     &a) ) gl_logMode  = strcpy(gl_log, a) != NULL;
    else if ( parmM_arg(p, "MSG",     &a) ) gl_msgMode  = strcpy(gl_msg, a) != NULL;
    else if ( parmM_arg(p, "TRY",     &a) ) gl_maxTryes = atoi(a);
    else if ( parmM_arg(p, "LIMIT",   &a) ) parseSizeMacro(a, gl_sizeLimit);
    else if ( parmM_arg(p, "IFSIZE.", &a) ) gl_ifSize   = a;
    else if ( parmM_arg(p, "IFDATE.", &a) ) gl_ifDate   = a;
    else if ( parmM_arg(p, "IFTIME.", &a) ) gl_ifTime   = a;
    else
    {
      lprintf(YELLOW, "%s /%s", LNG_MSG_INVALID_KEY, p);
      return 0;
    }
  }
  return 1;
}

static int setParameterBlock(int ga, char *p1)
{
  if ( !strnicmp(p1, "/IFDATE", 7) || !strnicmp(p1, "/@", 2) )
    return setParameter(ga, p1+1);
  char *p;
  while ( ( p = strtok(p1, "/ ") ) != NULL )
  {
    p1 = NULL;
    if ( !setParameter(ga, p) )
      return 0;
  }
  return 1;
}

static int setParameters(void)
{
  char *p, env[128];
  int processEnv = 0;
  if ( _argc < 2 )
    return 0;
  for ( register i = 1 ; i < _argc ; i++ )
    if ( *(p = strchr(_argv[i], 0)-1) == '\\' )
      *p = 0;
#ifdef __WIN32__
  if ( GetEnvironmentVariable("WCOPY", env, sizeof env) )
    processEnv = 1;
#else
  if ( ( p = getenv("WCOPY") ) != NULL )
  {
    strcpy(env, p);
    processEnv = 1;
  }
#endif
  int tac = _argc-1;
  while ( *(p = _argv[tac--]) == '/' )
  {
    if ( !stricmp(++p, "NOENV") )
    {
      processEnv = 0;
      break;
    }
  }
  if ( processEnv && !setParameterBlock(1, env) )
    return 0;
  while ( *(p = _argv[_argc-1]) == '/' )
  {
    if ( !setParameterBlock(1, p) )
      return 0;
    _argc--;
  }
  expandMacro(addPoint(gl_xcl));
  expandMacro(gl_ren);
  expandMacro(gl_log);
  expandMacro(gl_msg);
  if ( gl_addr_r )
  {
    if ( gl_addr_b )
    {
      TFTNAddress addrbase(gl_adb);
      gl_defFTNZone = addrbase.zone;
      gl_address = addr.parse(gl_adr, &addrbase);
    }
    else
    {
      gl_address = addr.parse(gl_adr);
      gl_defFTNZone = addr.zone;
    }
  }
  return 1;
}

static long expTotalSize = 0, totalSize = 0;

struct reqItem
{
  int found:1, move:1, count:14;
  int list;
  long size;
  char prefix, *file;
};

static inline char *newStr(const char *s)
{
  char *p = NULL;
  if ( s && ( ( p = new char[strlen(s)+1] ) != NULL ) )
    strcpy(p, s);
  return p;
}

static void reqItem_init(reqItem& f, int aMove, const char* aFile, int aList, char aPrefix)
{
  f.size = f.count = f.found = 0;
  f.move = aMove;
  f.list = aList;
  f.prefix = aPrefix;
  f.file = newStr(aFile);
}

static void reqItem_done(reqItem& f)
{
  if ( f.file )
  {
    delete [] f.file;
    f.file = NULL;
  }
}

static reqItem& reqItem_inc(reqItem& f, long aSize)
{
  f.count++;
  f.size += aSize;
  totalSize += aSize;
  return f;
}

static void delReqItem(void *v)
{
  reqItem *r = (reqItem*)v;
  reqItem_done(*r);
  delete r;
}

static reqItem *newReqItem(int aMove, char *realBuff, int listN, char aPrefix)
{
  reqItem *item = new reqItem;
  if ( item )
  {
    reqItem_init(*item, aMove, realBuff, listN, aPrefix);
    if ( !item->file )
    {
      delete item;
      item = NULL;
    }
  }
  return item;
}

struct copyItem
{
  char *file, *dir;
  reqItem *opt;
};

static void copyItem_init(copyItem& f, const char* aFile, char* aDir, reqItem *aOpt)
{
  f.file = newStr(aFile);
  f.dir = aDir;
  f.opt = aOpt;
}

static void copyItem_done(copyItem& f)
{
  if ( f.file )
  {
    delete [] f.file;
    f.file = NULL;
  }
}

static int cmpCopyItem(void *v1, void *v2)
{
  copyItem *r1 = (copyItem*)v1;
  copyItem *r2 = (copyItem*)v2;
  return !stricmp(r1->file, r2->file);
}

static void delCopyItem(void *v)
{
  copyItem *r = (copyItem*)v;
  copyItem_done(*r);
  delete r;
}

static copyItem *newCopyItem(char *file, char *dir, reqItem *opt)
{
  copyItem *item = new copyItem;
  if ( item )
  {
    copyItem_init(*item, file, dir, opt);
    if ( !item->file )
    {
      delete item;
      item = NULL;
    }
  }
  return item;
}

struct dirItem
{
  char *dir, *add;
};

static void dirItem_init(dirItem& f, const char* aDir, char* aAdd)
{
  f.dir = newStr(aDir);
  f.add = aAdd;
}

static void dirItem_done(dirItem& f)
{
  if ( f.dir )
  {
    delete [] f.dir;
    f.dir = NULL;
  }
}

static void delDirItem(void *v)
{
  dirItem *r = (dirItem*)v;
  dirItem_done(*r);
  delete r;
}

static dirItem *newDirItem(char *dir, char *add)
{
  dirItem *item = new dirItem;
  if ( item )
  {
    dirItem_init(*item, dir, add);
    if ( !item->dir )
    {
      delete item;
      item = NULL;
    }
  }
  return item;
}

static TCollection cRequest(128, 128, delReqItem);
static TCollection cFileList(16, 16);
static TCollection cWorkList(256, 256, delCopyItem);
static TCollection cSrcFolder(64, 64, delDirItem);
static TCollection cDstFolder(16, 16, delDirItem);

static void showStatus(int toLog, const char *dest, const char *action, const char *f1, const char *f2, long size, const char *status = "", int errorMode = 0)
{
  static char buffer[400] = "", line[MAXPATH*2];
  char prc[8] = "";
  if ( !toLog )
  {
    long pp = expTotalSize ? totalSize*100l/expTotalSize : 100l;
    sprintf(prc, "%3ld%% ", pp > 100l ? 100l : pp);
  }
  int y = dest ? 50 : 60;
  strcat(strcpy(buffer, action), " ");
  strcat(buffer, char2oem(strcpy(line, f1)));
  if ( f2 && stricmp(f1, f2) )
    strcat(strcat(buffer, LNG_STATUS_AS), char2oem(strcpy(line, f2)));
  if ( (int)strlen(buffer) > y )
    strcpy(buffer+(y-3), "...");
  if ( dest )
  {
    char2oem(strcpy(line, dest));
    strcat(buffer, LNG_STATUS_TO);
    int rlen = (errorMode ? 50 : 60)-strlen(buffer);
    int dlen = strlen(dest);
    if ( dlen > rlen )
    {
      char *p = strchr(buffer, 0);
      strcat(strcat(buffer, "   ..."), line+(dlen-rlen+6));
      for ( int i = 0 ; i < 3 ; i++ )
        p[i] = line[i];
    }
    else
      strcat(buffer, line);
  }
  if ( errorMode )
    sprintf(line, "%-50s %s", buffer, status);
  else if ( size >= 0 )
    sprintf(line, "%-60s%10s %s%s", buffer, commas(size), prc, status);
  else
    sprintf(line, "%-60s %s%s", buffer, prc, status);
  char * p = strchr(line, 0);
  while ( *--p == ' ')
    *p = 0;
  putch('\r');
  if ( toLog )
  {
    lprintf(0, "%s", line);
    y = wherey()-1;
  }
  else
  {
    cputs(line);
    y = wherey();
    clreol();
  }
  if ( errorMode )
  {
    text_info textInfo;
    gettextinfo(&textInfo);
    gotoxy(52, y);
    textattr(YELLOW);
    cputs(status);
    textattr(textInfo.attribute);
    clreol();
    gotoxy(textInfo.curx, textInfo.cury);
  }
}

static int restartNeeded = 0;

static char *replExtMask(char *s)
{
  for ( char *p = s ; *p ; p++ )
    if ( *p == '+' )
      *p = '?';
  return s;
}

static int doOneFile(char *srcFile, char *addDir, char *dstDir0, reqItem *fopt, int fitem, int fcount)
{
  FIND_DATA ffInDir, ffOutDir;
  char outFile[MAXPATH], dir[MAXPATH], dstDir[MAXPATH];
  char tempDrive[MAXDRIVE], tempDir[MAXDIR], tempFile[MAXFILE], tempExt[MAXEXT];
  int r = restartNeeded = 0;
  if ( existFile(srcFile, &ffInDir) )
  {
    int erase = gl_moveMode;
    if ( gl_binkStyle )
    {
      if ( gl_copyAlwaysMode )
        erase = 0;
      else if ( !gl_moveMode )
        erase = fopt->move;
    }
    if ( fitem != fcount-1 )
      erase = 0;
    fnsplit(srcFile, tempDrive, tempDir, tempFile, tempExt);
    strcpy(dstDir, dstDir0);
    if ( addDir )
      strcat(dstDir, addDir);
    fnmerge(outFile, "", dstDir, tempFile, tempExt);
    fnmerge(tempDir, "", "", tempFile, tempExt);
    const char *destDir = (fcount > 1) ? dstDir : NULL;
    int attr = GetFileAttributes(srcFile);
    int sizeOfFile = ff_fsize(&ffInDir);
    showStatus(0, destDir, copyMode[erase ? 0 : 1], tempDir, NULL, sizeOfFile);
    int CopyNeeded = 1, outFileExists = 1;
    if ( gl_renMode )
      makeNewName(addPoint(outFile), gl_ren);
#ifdef __WIN32__
    if ( gl_dosNames )
#endif
      trim83(outFile);
    if ( !existFile(outFile, &ffOutDir) )
    {
      outFileExists = 0;
      if ( gl_existOnly )
        CopyNeeded = 0;
    }
    else if ( gl_skipAll )
      CopyNeeded = 0;
    else
    {
      if ( gl_newOnly )
      {
        if ( ff_fdate(&ffInDir) < ff_fdate(&ffOutDir) )
          CopyNeeded = 0;
        else if ( ff_fdate(&ffInDir) == ff_fdate(&ffOutDir) )
          if ( ff_ftime(&ffInDir) <= ff_ftime(&ffOutDir) )
            CopyNeeded = 0;
      }
      if ( gl_diffOnly )
      {
        if ( ff_fsize(&ffInDir) == ff_fsize(&ffOutDir) &&
             ff_fdate(&ffInDir) == ff_fdate(&ffOutDir) &&
             ff_ftime(&ffInDir) == ff_ftime(&ffOutDir)
           )
          CopyNeeded = 0;
      }
      if ( CopyNeeded && !gl_overWrite && !gl_append )
        CopyNeeded = makeDiffName(!gl_keepExt, gl_digitSet, outFile);
    }
    if ( CopyNeeded && checkFile(srcFile) )
    {
      if ( gl_sizeLimit < 0 || totalSize+sizeOfFile <= gl_sizeLimit )
      {
        char *p2 = outFile, *p1;
        char *ok = ( gl_append && outFileExists ) ? LNG_STATUS_APPEND : LNG_STATUS_OK;
        char *p = ok;
        while ( ( p1 = strchr(p2, '\\') ) != NULL )
          p2 = p1+1;
        showStatus(0, destDir, copyMode[erase ? 0 : 1], tempDir, NULL, sizeOfFile, ".");
        int renameIsSuccess = 0;
        if ( gl_useRen && erase && !gl_append && RenameFile(srcFile, outFile) )
          renameIsSuccess = 1;
        if ( !renameIsSuccess )
        {
          HANDLE inHandle = openFile(srcFile, gl_maxTryes, gl_noSoundMode, erase ? "r+" : "r", LNG_CHOICE_THIS);
          if ( inHandle != INVALID_HANDLE_VALUE )
          {
            if ( !createDir(addBackslash(fexpand(dstDir))) )
            {
              p = LNG_ERR_TARGETFOLDER;
              r = 1;
            }
            else
            {
              HANDLE outHandle = openFile(outFile, gl_maxTryes, gl_noSoundMode, gl_append ? "a" : "w", LNG_CHOICE_THIS);
              showStatus(0, destDir, copyMode[erase ? 0 : 1], tempDir, p2, sizeOfFile, "..");
              if ( outHandle != INVALID_HANDLE_VALUE )
              {
                FIND_DATA ff2;
                existFile(srcFile, &ff2);
                int success = copyFile(inHandle, outHandle, sizeOfFile = ff_fsize(&ff2));
                if ( success && !existFile(outFile) )
                {
                  p = LNG_ERR_TARGETCHECK;
                  r = 1;
                  success = 0;
                }
                CloseHandle(inHandle);
                CloseHandle(outHandle);
                if ( success )
                {
                  if ( !gl_clearAttr && attr != -1)
                    SetFileAttributes(outFile, attr);
                  if ( erase && !DeleteFile(srcFile) )
                  {
                    p = LNG_ERR_MOVEERR0;
                    if ( DeleteFile(outFile) )
                      p = LNG_ERR_MOVEERR1;
                    r = 1;
                  }
                  else
                    reqItem_inc(*fopt, sizeOfFile);
                }
                else
                {
                  DeleteFile(outFile);
                  if ( gl_floppyMode )
                  {
                    if ( !gl_noSoundMode )
                      putch('\a');
                    restartNeeded = choiceYesNo(LNG_CHOICE_DISKFULL, NULL);
                    p = restartNeeded ? LNG_CHOICE_NEXTDISK : LNG_ERR_SKIPPED;
                  }
                  else
                  {
                    p = LNG_ERR_COPYERROR;
                    r = 1;
                  }
                }
              }
              else
              {
                p = LNG_ERR_CANTCREATE;
                r = 1;
              }
            }
          }
          else
            p = LNG_ERR_BUSY;
        }
        int copyOK = p == ok;
        showStatus(0, destDir, copyMode[erase ? 0 : 1], tempDir, p2, sizeOfFile, "...");
        if ( gl_filesBBS && copyOK )
          updateDescription(dir, dstDir, descFile, tempDir, gl_maxTryes, gl_noSoundMode);
        showStatus(restartNeeded ? 0 : 1, destDir, copyMode[erase ? 0 : 1], tempDir, p2, sizeOfFile, p, !copyOK);
      }
      else
        showStatus(1, NULL, LNG_MODE_SKIP, tempDir, NULL, sizeOfFile, LNG_STATUS_SIZE, 1);
    }
    else
      showStatus(0, NULL, LNG_MODE_SKIP, tempDir, NULL, sizeOfFile);
  }
  return r;
}

static void addOne(reqItem *f)
{
  char inFile[MAXPATH], outFile[MAXPATH], dir[MAXPATH];
  char findMask[MAXPATH], realMask[MAXFILE+MAXEXT-1];
  char tempDrive[MAXDRIVE], tempDir[MAXDIR], tempFile[MAXFILE], tempExt[MAXEXT];
  for ( unsigned item = 0 ; item < cSrcFolder.getCount() ; item++ )
  {
    dirItem *li = (dirItem*)(cSrcFolder[item]);
    char *srcDir = li->dir;
    char *addDir = li->add;
    if ( gl_usePath )
    {
      int fp = f->file[0] == '\\' || f->file[1] == ':';
      fnmerge(inFile, "", fp ? "" : srcDir, f->file, "");
      fnsplit(inFile, dir, outFile, tempFile, tempExt);
      strcat(dir, outFile);
    }
    else
    {
      fnsplit(f->file, dir, outFile, tempFile, tempExt);
      fnmerge(inFile, "", strcpy(dir, srcDir), tempFile, tempExt);
    }
    fnmerge(realMask, "", "", tempFile, tempExt);
    strcpy(findMask, inFile);
    if ( gl_extMask )
      replExtMask(findMask);
    FIND_DATA ff;
    FF_HANDLE ffh = findFirst(findMask, &ff, IN_ATTR);
    if ( ffh != INVALID_HANDLE_VALUE )
    {
      do
      {
        expTotalSize += ff_fsize(&ff);
        fnmerge(inFile, "", dir, ff_fname(&ff), "");
        if ( isFile(&ff) )
        {
#ifdef __WIN32__
          if ( ff_attrib(&ff) & (OUT_ATTR) )
            continue;
          strcpy(outFile, inFile);
          if ( gl_dosNames )
            GetShortPathName(outFile, inFile, sizeof inFile);
          else
          {
            char *dummy;
            GetFullPathName(outFile, sizeof inFile, inFile, &dummy);
          }
#endif
          fnsplit(inFile, tempDrive, tempDir, tempFile, tempExt);
          fnmerge(tempDir, "", "", tempFile, tempExt);
          if ( !gl_extMask || matched(addPoint(realMask), addPoint(tempDir), gl_extMask) )
          {
            if ( ( !gl_exclude  || ( gl_exclude  && !matched(gl_xcl, tempDir, gl_extMask) ) ) &&
                 ( !gl_filesBBS || ( gl_filesBBS && stricmp(descFile, tempDir) ) ) )
            {
              f->found = 1;
              copyItem *ci = newCopyItem(inFile, addDir, f);
              if ( cWorkList.find(cmpCopyItem, ci) )
                delCopyItem(ci);
              else
              {
                cWorkList.insert(ci);
                cprintf("\r%4d", cWorkList.getCount());
              }
            }
          }
        }
      } while ( findNext(ffh, &ff) );
    }
    findClose(ffh, &ff);
  }
}

static char *allTrim(char * s)
{
  char *p;
  while ( *s && strchr(" \t\n\r", *s) )
    strcpy(s, s+1);
  if ( *s == '\"' )
  {
    strcpy(s, s+1);
    p = strchr(s, '\"');
    if ( p )
      *p = 0;
  }
  p = strchr(s, 0);
  while ( (p-- > s ) && strchr(" \t\n\r", *p) )
    *p = 0;
  return s;
}

static int loadList(int fileList, char *fn, TCollection *list)
{
  static char buff[MAXPATH];
  const char *title = fileList ? LNG_STATUS_FILELIST : LNG_STATUS_FOLDERLIST;
  strcpy(buff, fn);
  int n = 0, listN = 0;
  if ( fileList )
    listN = cFileList.insert(newStr(buff));
  cprintf(LNG_STATUS_LOAD, title, char2oem(buff));
  HANDLE lf = openFile(fn, gl_maxTryes, gl_noSoundMode, "r", LNG_CHOICE_LOAD);
  if ( lf != INVALID_HANDLE_VALUE )
  {
    while ( getString(lf, buff, MAXPATH) )
    {
      if ( *allTrim(buff) )
      {
#ifdef __WIN32__
        if ( gl_Oem2Ansi )
          oem2char(buff);
#endif
        char *realBuff = buff, prefix = 0;
        int add2list = 1, aMove = 0;
        if ( fileList && gl_binkStyle )
        {
          switch ( *buff )
          {
            case '^':
            case '#':
              prefix = *buff;
              realBuff++;
              aMove = 1;
              break;
            case '~':
              add2list = 0;
              break;
            default:
              break;
          }
        }
        if ( add2list )
        {
          if ( fileList )
            list->insert(newReqItem(aMove, realBuff, listN, prefix));
          else
            list->insert(newDirItem(realBuff, NULL));
          cprintf("\r%4d", ++n);
        }
      }
    }
    CloseHandle(lf);
    putch('\r');
    lprintf(CYAN, LNG_STATUS_LOADED, title, char2oem(strcpy(buff, fn)), n);
    if ( !n )
      lprintf(YELLOW, LNG_STATUS_EMPTY);
  }
  else
    lprintf(YELLOW, LNG_STATUS_CANTLOAD, title, buff);
  return n;
}

static char *parseListEnd(char *s)
{
  char *p = strchr(s, ']');
  if ( !p )
  {
    lprintf(YELLOW, "%s: %s", LNG_MSG_INVALID_PARM, LNG_MSG_MISSINGEND);
    return NULL;
  }
  if ( p[1] )
  {
    lprintf(YELLOW, "%s: %s", LNG_MSG_INVALID_PARM, LNG_MSG_EXTRASYMBOLS);
    return NULL;
  }
  *p = 0;
  return p;
}

int postDiags(int ok)
{
  if ( !ok )
  {
    lprintf(YELLOW, "%s. %s", LNG_MSG_INVALID_CMD, LNG_MSG_USEHELP);
    return 2;
  }
  if ( gl_overWrite+gl_append+gl_skipAll > 1 )
  {
    lprintf(YELLOW, "%s: %s", LNG_MSG_INCOMPATIBLE, LNG_MSG_INCOMPAT1);
    return 2;
  }
  if ( gl_copyAlwaysMode+gl_moveMode > 1 )
  {
    lprintf(YELLOW, "%s: %s", LNG_MSG_INCOMPATIBLE, LNG_MSG_INCOMPAT2);
    return 2;
  }
  if ( !parseIf() )
  {
    lprintf(YELLOW, "%s. %s", LNG_MSG_INVALID_IF, LNG_MSG_USEHELP);
    return 2;
  }
  return 0;
}

static int getLocParam(char *mask)
{
  char *p = strchr(mask, '/');
  if ( p )
  {
    if ( !setParameterBlock(0, p) )
      return postDiags(0);
    *p = 0;
  }
  return postDiags(1);
}

static void backupParam(void)
{
#ifdef __WIN32__
  bk_dosNames       = gl_dosNames;
  bk_Oem2Ansi       = gl_Oem2Ansi;
#endif
  bk_maxTryes       = gl_maxTryes;
  bk_noSoundMode    = gl_noSoundMode;
  bk_logMode        = gl_logMode;
  bk_msgMode        = gl_msgMode;
  bk_moveMode       = gl_moveMode;
  bk_binkStyle      = gl_binkStyle;
  bk_overWrite      = gl_overWrite;
  bk_keepExt        = gl_keepExt;
  bk_digitSet       = gl_digitSet;
  bk_addHome        = gl_addHome;
  bk_newOnly        = gl_newOnly;
  bk_diffOnly       = gl_diffOnly;
  bk_existOnly      = gl_existOnly;
  bk_clearAttr      = gl_clearAttr;
  bk_usePath        = gl_usePath;
  bk_exclude        = gl_exclude;
  bk_extMask        = gl_extMask;
  bk_append         = gl_append;
  bk_filesBBS       = gl_filesBBS;
  bk_recursive      = gl_recursive;
  bk_tree           = gl_tree;
  bk_copyAlwaysMode = gl_copyAlwaysMode;
  bk_floppyMode     = gl_floppyMode;
  bk_address        = gl_address;
  bk_rewriteMode    = gl_rewriteMode;
  bk_useRen         = gl_useRen;
  bk_skipAll        = gl_skipAll;
  bk_renMode        = gl_renMode;
  bk_addr_b         = gl_addr_b;
  bk_addr_r         = gl_addr_r;
  bk_sizeLimit      = gl_sizeLimit;
  bk_defFTNZone     = gl_defFTNZone;
  strcpy(bk_adb, gl_adb);
  strcpy(bk_adr, gl_adr);
  strcpy(bk_log, gl_log);
  strcpy(bk_msg, gl_msg);
  strcpy(bk_xcl, gl_xcl);
  strcpy(bk_ren, gl_ren);
  bk_ifSize = gl_ifSize;
  bk_ifDate = gl_ifDate;
  bk_ifTime = gl_ifTime;
}

static void restoreParam(void)
{
#ifdef __WIN32__
  gl_dosNames       = bk_dosNames;
  gl_Oem2Ansi       = bk_Oem2Ansi;
#endif
  gl_maxTryes       = bk_maxTryes;
  gl_noSoundMode    = bk_noSoundMode;
  gl_logMode        = bk_logMode;
  gl_msgMode        = bk_msgMode;
  gl_moveMode       = bk_moveMode;
  gl_binkStyle      = bk_binkStyle;
  gl_overWrite      = bk_overWrite;
  gl_keepExt        = bk_keepExt;
  gl_digitSet       = bk_digitSet;
  gl_addHome        = bk_addHome;
  gl_newOnly        = bk_newOnly;
  gl_diffOnly       = bk_diffOnly;
  gl_existOnly      = bk_existOnly;
  gl_clearAttr      = bk_clearAttr;
  gl_usePath        = bk_usePath;
  gl_exclude        = bk_exclude;
  gl_extMask        = bk_extMask;
  gl_append         = bk_append;
  gl_filesBBS       = bk_filesBBS;
  gl_recursive      = bk_recursive;
  gl_tree           = bk_tree;
  gl_copyAlwaysMode = bk_copyAlwaysMode;
  gl_floppyMode     = bk_floppyMode;
  gl_address        = bk_address;
  gl_rewriteMode    = bk_rewriteMode;
  gl_useRen         = bk_useRen;
  gl_skipAll        = bk_skipAll;
  gl_renMode        = bk_renMode;
  gl_addr_b         = bk_addr_b;
  gl_addr_r         = bk_addr_r;
  gl_sizeLimit      = bk_sizeLimit;
  gl_defFTNZone     = bk_defFTNZone;
  strcpy(gl_adb, bk_adb);
  strcpy(gl_adr, bk_adr);
  strcpy(gl_log, bk_log);
  strcpy(gl_msg, bk_msg);
  strcpy(gl_xcl, bk_xcl);
  strcpy(gl_ren, bk_ren);
  gl_ifSize = bk_ifSize;
  gl_ifDate = bk_ifDate;
  gl_ifTime = bk_ifTime;
}

static char *GetCommaWord(char *Src, char *Word)
{
  if ( !*Src )
    return NULL;
  int WordPos;
  bool SkipBrackets = false;
  for ( WordPos = 0 ; *Src ; Src++, WordPos++ )
  {
    if ( *Src == '\"' )
      SkipBrackets = !SkipBrackets;
    if ( *Src==',' && !SkipBrackets )
    {
      Word[WordPos] = 0;
      Src++;
      while ( isspace(*Src) )
        Src++;
      return Src;
    }
    else
      Word[WordPos] = *Src;
  }
  Word[WordPos] = 0;
  return Src;
}

static void addFolder(int recursive, TCollection *list, char *drv, char *path, int addLen = -1)
{
  char drv2[MAXPATH], path2[MAXPATH];
  strcpy(drv2, drv);
  strcpy(path2, path);
  for ( ; ; )
  {
    char *wld = strpbrk(path2, gl_extMask ? "+*?" : "*?");
    if ( wld )
    {
      char ch, *brk = strpbrk(path2, "\\/");
      if ( ( brk != NULL ) && ( brk < wld ) )
      {
        ch = brk[1];
        brk[1] = 0;
        strcat(drv2, path2);
        brk[1] = ch;
        strcpy(path2, brk+1);
      }
      else
      {
        static char drv3[MAXPATH];
        char giv_Mask[MAXPATH], realMask[MAXPATH];
        if ( brk )
        {
          ch = *brk;
          *brk = 0;
        }
        else
          ch = 0;
        strcat(strcpy(realMask, drv2), replExtMask(strcpy(giv_Mask, path2)));
        strcat(strcpy(giv_Mask, drv2), path2);
        if ( brk )
          strcpy(path2, brk+1);
        else
          *path2 = 0;
        FIND_DATA ff;
        FF_HANDLE f = findFirst(realMask, &ff, DIR_ATTR);
        if ( f != INVALID_HANDLE_VALUE )
          do
          {
            char *p = ff_fname(&ff);
            if ( isSubDir(&ff) )
            {
#ifdef __WIN32__
              if ( ff_attrib(&ff) & (OUT_ATTR) & (~DIR_ATTR) )
                continue;
#endif
              char cc[2] = { ch, 0 };
              strcat(strcpy(drv3, drv2), p);
              if ( !gl_extMask || matched(giv_Mask, drv3, gl_extMask) )
                addFolder(recursive, list, drv3, path2, addLen);
              strcat(drv3, cc);
            }
          } while ( findNext(f, &ff) );
        findClose(f, &ff);
        break;
      }
    }
    else
    {
      char temp[MAXPATH];
      strcat(strcpy(temp, drv2), path2);
#ifdef __WIN32__
      char temp2[MAXPATH];
      strcpy(temp2, temp);
      if ( gl_dosNames )
      {
        if ( !GetShortPathName(temp2, temp, sizeof temp) )
          trim83(temp);
      }
      else if ( strcmp(temp2+1, ":") )
      {
        char *dummy;
        GetFullPathName(temp2, sizeof temp, temp, &dummy);
      }
#endif
      addBackslash(temp);
      dirItem *it = newDirItem(temp, NULL);
      if ( addLen >= 0 )
         it->add = it->dir+addLen;
      list->insert(it);
      if ( recursive )
      {
        if ( gl_tree && addLen < 0 )
          addLen = strlen(temp);
        addFolder(recursive, list, temp, "*", addLen);
      }
      break;
    }
  }
}

static void rewriteLists(void)
{
  for ( unsigned i = 0 ; i < cFileList.getCount() ; i++ )
  {
    HANDLE f = INVALID_HANDLE_VALUE;
    char *li = (char*)(cFileList[i]);
    for ( unsigned j = 0 ; j < cRequest.getCount() ; j++ )
    {
      reqItem *lj = (reqItem*)(cRequest[j]);
      if ( lj->list == (int)i+1 && ( lj->found == 0 || lj->count == 0 ) )
      {
        if ( f == INVALID_HANDLE_VALUE )
          f = openFile(li, gl_maxTryes, gl_noSoundMode, "w", LNG_CHOICE_REWRITE);
        if ( f != INVALID_HANDLE_VALUE )
        {
          if ( lj->prefix )
          {
            char prf[2] = { lj->prefix, 0 };
            putString(f, prf);
          }
          putString(f, lj->file);
          putString(f, "\r\n");
        }
      }
    }
    if ( f != INVALID_HANDLE_VALUE )
      CloseHandle(f);
    else
      DeleteFile(li);
  }
}

int main(void)
{
  char *p;
  static char drive[MAXDRIVE], dir[MAXDIR], file[MAXPATH], ext[MAXEXT];
  static char inDir[MAXPATH], inFile[MAXPATH];
  static char buff[MAXPATH];
  static char list2load[MAXPATH], mask2process[MAXPATH*2];
  int listDir, listFile, rc;

  lprintf(WHITE, LNG_MSG_START1" "VERSION"/"TARGET_OS" "LNG_MSG_START2);
  if ( _argc == 2 && !strcmp(_argv[1], "/?") )
    return usage();
  if ( ( rc = postDiags(setParameters()) ) != 0 )
    return rc;
  char *parmOne = ".\\";
  if ( _argc == 3 )
    parmOne = _argv[--_argc];
  char *outdir = parmOne;
  if ( _argc != 2 )
  {
    lprintf(YELLOW, "%s. %s", LNG_MSG_TOOMANY, LNG_MSG_USEHELP);
    return 2;
  }
  if ( gl_msgMode )
    DeleteFile(gl_msg);
  while ( ( parmOne = GetCommaWord(parmOne, mask2process) ) != NULL )
  {
    expandMacro(mask2process);
    if ( *mask2process == '[' )
    {
      if ( parseListEnd(mask2process+1) == NULL )
        return 2;
      loadList(0, mask2process+1, &cDstFolder);
    }
    else
    {
      fexpand(strcat(addBackslash(mask2process), "*"));
      fnsplit(mask2process, drive, dir, file, ext);
      *(strchr(strcpy(mask2process, dir), 0)-1) = 0;
      addFolder(0, &cDstFolder, drive, mask2process);
    }
  }
  parmOne = _argv[1];
  while ( ( parmOne = GetCommaWord(parmOne, mask2process) ) != NULL )
  {
    listDir = listFile = rc = *inDir = *inFile = 0;
    backupParam();
    if ( getLocParam(mask2process) == 0 )
    {
      if ( gl_msgMode )
        DeleteFile(gl_msg);
      if ( *mask2process == '[' )
      {
        p = strchr(mask2process, ']');
        if ( !p )
        {
          lprintf(YELLOW, "%s: %s", LNG_MSG_INVALID_PARM, LNG_MSG_MISSINGEND);
          return 2;
        }
        *p = 0;
        p++;
        switch ( *p )
        {
          case 0:
            listFile = 1;
            strcpy(inFile, mask2process+1);
            break;
          case '[':
            listFile = listDir = 1;
            strcpy(inDir, mask2process+1);
            strcpy(inFile, p+1);
            if ( ( p = parseListEnd(inFile) ) == NULL )
              return 2;
            break;
          default:
            listDir = 1;
            strcpy(inDir, mask2process+1);
            strcpy(inFile, p);
            break;
        }
      }
      else if ( ( p = strchr(mask2process, '[') ) != NULL )
      {
        *p = 0;
        p++;
        listFile = 1;
        strcpy(inDir, mask2process);
        strcpy(inFile, p);
        if ( ( p = parseListEnd(inFile) ) == NULL )
          return 2;
      }
      else
      {
        fnsplit(expandMacro(strcpy(inFile, mask2process)), drive, dir, file, ext);
        fnmerge(inDir, drive, dir, "", "");
        fnmerge(inFile, "", "", file, ext);
      }
      const char *moveModeDescr = gl_moveMode ? LNG_MODE_MOVING : LNG_MODE_COPYING;
      char *qf1 = "", *qf2 = "", *qd1 = "", *qd2 = "";
      if ( listFile )
      {
        qf1 = "[";
        qf2 = "]";
      }
      if ( listDir )
      {
        qd1 = "[";
        qd2 = "]";
      }
      expandMacro(inFile);
      expandMacro(inDir);
      if ( listFile )
      {
        FIND_DATA ff;
        FF_HANDLE ffh = findFirst(inFile, &ff, IN_ATTR);
        fnsplit(inFile, drive, dir, file, ext);
        if ( ffh != INVALID_HANDLE_VALUE )
        {
          do
          {
            if ( isFile(&ff) )
            {
#ifdef __WIN32__
              if ( ff_attrib(&ff) & (OUT_ATTR) )
                continue;
#endif
              fnmerge(list2load, drive, dir, ff_fname(&ff), "");
              loadList(1, list2load, &cRequest);
            }
          } while ( findNext(ffh, &ff) );
        }
        else
          lprintf(YELLOW, LNG_STATUS_CANTFIND, LNG_STATUS_FILELIST, char2oem(strcpy(buff, inFile)));
        findClose(ffh, &ff);
      }
      else
        cRequest.insert(newReqItem(0, inFile, 0, 0));
      if ( cRequest.getCount() )
      {
        allocateIOBuffer();
        if ( listDir )
        {
          FIND_DATA ff;
          FF_HANDLE ffh = findFirst(inDir, &ff, IN_ATTR);
          fnsplit(inDir, drive, dir, file, ext);
          if ( ffh != INVALID_HANDLE_VALUE )
          {
            do
            {
              if ( isFile(&ff) )
              {
#ifdef __WIN32__
                if ( ff_attrib(&ff) & (OUT_ATTR) )
                  continue;
#endif
                fnmerge(list2load, drive, dir, ff_fname(&ff), "");
                loadList(0, list2load, &cSrcFolder);
              }
            } while ( findNext(ffh, &ff) );
          }
          else
            lprintf(YELLOW, LNG_STATUS_CANTFIND, LNG_STATUS_FOLDERLIST, char2oem(strcpy(buff, inDir)));
          findClose(ffh, &ff);
        }
        else
        {
          fexpand(strcat(addBackslash(strcpy(mask2process, inDir)), "*"));
          fnsplit(mask2process, drive, dir, file, ext);
          *(strchr(strcpy(mask2process, dir), 0)-1) = 0;
          addFolder(gl_recursive, &cSrcFolder, drive, mask2process);
        }
        if ( gl_addHome )
          addFolder(gl_recursive, &cSrcFolder, "", "");
        cprintf(LNG_STATUS_SCAN);
        for ( unsigned j = 0 ; j < cRequest.getCount() ; j++ )
        {
          reqItem *lj = (reqItem*)(cRequest[j]);
          if ( lj->file && lj->file[0] )
            addOne(lj);
        }
        putch('\r');
        int wcount = cWorkList.getCount();
        int dcount = cDstFolder.getCount();
        expTotalSize *= dcount;
        if ( wcount )
        {
          lprintf(LIGHTCYAN, "%s %s%s%s%s%s%s%s%s", moveModeDescr, qd1, char2oem(strcpy(dir, inDir)), qd2, qf1, char2oem(strcpy(file, inFile)), qf2, LNG_STATUS_TO, char2oem(expandMacro(strcpy(buff, outdir))));
          for ( int j = 0 ; j < wcount ; j++ )
          {
            copyItem *lj = (copyItem*)(cWorkList[j]);
            int rcone = 1;
            for ( int i = 0 ; i < dcount ; i++ )
            {
              dirItem *li = (dirItem*)(cDstFolder[i]);
              do
              {
                rcone = doOneFile(lj->file, lj->dir, li->dir, lj->opt, i, dcount) ? rcone : 0;
              } while ( restartNeeded );
              if ( rcone )
                break;
            }
            rc = rcone ? rcone : rc;
          }
        }
        else
          lprintf(YELLOW, "%s %s%s%s%s%s%s %s", LNG_MODE_FILE, qd1, char2oem(strcpy(dir, inDir)), qd2, qf1, char2oem(strcpy(file, inFile)), qf2, LNG_STATUS_NOTFOUND);
        destroyIOBuffer();
        if ( wcount )
        {
          if ( gl_rewriteMode )
            rewriteLists();
          for ( unsigned j = 0 ; j < cRequest.getCount() ; j++ )
          {
            reqItem *lj = (reqItem*)(cRequest[j]);
            if ( lj->found == 0 )
            {
              sprintf(buff, "%s%s%s%s", qd1, inDir, qd2, lj->file);
              showStatus(1, NULL, LNG_MODE_FILE, buff, NULL, -1, LNG_STATUS_NOTFOUND, 1);
            }
          }
        }
        if ( gl_msgMode )
        {
          cprintf(LNG_STATUS_CREATEREP, char2oem(strcpy(buff, gl_msg)));
          HANDLE f = openFile(gl_msg, gl_maxTryes, gl_noSoundMode, "a", LNG_CHOICE_REPORT);
          putch('\r');
          clreol();
          if ( f != INVALID_HANDLE_VALUE )
          {
            FPRINTF(f, LNG_REQUEST1, listFile ? LNG_STATUS_FILELIST : LNG_STATUS_FILES, inFile);
            FPRINTF(f, LNG_REQUEST2);
            FPRINTF(f, LNG_REQUEST3);
            for ( unsigned j = 0 ; j < cRequest.getCount() ; j++ )
            {
              reqItem *lj = (reqItem*)(cRequest[j]);
              char tmp[32] = "";
              if ( lj->found )
              {
                if ( lj->count )
                  sprintf(tmp, ( lj->count == 1 ) ? LNG_STATUS_OK : LNG_STATUS_COPYES, lj->count);
                else
                  strcpy(tmp, LNG_STATUS_IOERROR);
              }
              else
                strcpy(tmp, LNG_STATUS_NOTFOUND);
              FPRINTF(f, LNG_REQUEST4, tmp, commas(lj->size), lj->file);
            }
            FPRINTF(f, LNG_REQUEST5);
            FPRINTF(f, LNG_REQUEST6, cRequest.getCount(), commas(totalSize));
            CloseHandle(f);
          }
        }
      }
    }
    cSrcFolder.removeAll();
    cRequest.removeAll();
    cFileList.removeAll();
    cWorkList.removeAll();
    expTotalSize = 0;
    restoreParam();
  }
  putch('\r');
  clreol();
  lprintf(WHITE, LNG_MSG_END, commas(totalSize));
  textattr(7);
  clreol();
  return rc;
}
