/* TTCODE 1.02 (slightly fixed version)
   
   Written by Tom Torfs (2:292/516@fidonet.org)
   I hereby donate this code to the public domain

   Thanks to Arnold Hendriks (2:282/502.1@fidonet.org) for pointing
   out some potential problems

   The CRC-16 routines are based on CRC-16.C from Snippets 10/95

   This code compiles without problems under Borland C++ 3.0 and
   Turbo C++ 1.0; probably other compilers require some changes
   Feel free to add the necessary #ifdef's and redistribute
   
   The compiler & OS-specific parts of this code are:
      - the get/set file time functions
      - filenames are converted to uppercase; remove the strupr() calls
	for case sensitive OSes
*/

/* TTCODE encoded file structure:

   TTCODE <name of software used to encode the file>
   
   FILE <filename> <size in bytes> <filedate DD-MM-YYYY> <filetime HH:MM>
   X <hex codes of the excluded characters>
   
   SECTION <section number (base 1)>/<sections>
   T<encoded data>
   ...
   END CRC16=<section-CRC in hex>

   all statements are case sensitive

   everything before the TTCODE statement and after the END statement is
   ignored

   every TTcoded file must have the TTCODE statement in it; if this
   statement is missing TTDECODE will say the file doesn't contain
   TTcoded data

   the FILE and X statements must come between the TTCODE statement and
   before the first SECTION statement and only in the first section (they
   will be ignored in other sections)

   the 37 excluded hex codes may be spread over multiple lines; however
   there must be exactly 37 hex codes, no more, no less

   there must be exactly one section per file, and TTDECODE requires the
   section files to have extensions 001,002,etc. (the first section may
   have another extension)
*/

/* TTCODE encoded data format

   219 of the 256 characters will be remapped to a character set with only
   219 characters; the following have been removed:
      0..31 (control codes), 32 (space), 127 (delete), 141 (soft return)
      and 255 (fixed space)

   the other 37 characters are encoded with 2 bytes: one id-byte, which is
   character 254, followed by the number of the excluded character (0..36)
   which is remapped to the encoded character set (which simply means adding
   33 here)

   the 37 least frequently used characters are picked as the excluded
   characters, since they take 2 encoded characters instead of 1

   each encoded data line contains the 'T' signature plus 59 code bytes 
*/

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <dir.h>
#include <io.h>

#define CRCPOLY 0x8408

#define FIRSTCHAR 33
#define DELETE 127
#define SOFTRETURN 141
#define MAXCHARS 219
#define EXCLUDED 37
#define XCHARID 254

#define BYTESPERLINE 59

#define MAXFILENAMELEN 80
#define BUFSIZE 512
#define CHARACTERS 256

char idstring[] = "TTCODE 1.02 by Tom Torfs";
   
char *binfile,*ttfile;
unsigned char *buf;
int *convtable;
long *countbuf;
int *excluded;
   
unsigned crc;

/* calculate the CRC-16 for a single byte */
void crc16(int data)
{
   register int i;

   for (i = 0; i<8; i++, data >>= 1)
   {
      if ((crc & 0x0001) ^ (data & 0x0001))
	 crc = (crc >> 1) ^ CRCPOLY;
      else
	 crc >>= 1;
   }
}

void *checkmalloc(unsigned size)
{
   void *ptr;

   ptr = malloc(size);
   if (ptr==NULL)
   {
      printf("Not enough memory :-(\n");
      exit(1);
   }

   return ptr;
}

void allocdata(void)   
{
   binfile = checkmalloc(MAXFILENAMELEN);
   ttfile = checkmalloc(MAXFILENAMELEN);
   buf = checkmalloc(BUFSIZE);
   convtable = checkmalloc(sizeof(int)*CHARACTERS);
   countbuf = checkmalloc(sizeof(long)*CHARACTERS);
   excluded = checkmalloc(sizeof(int)*EXCLUDED);
}

void freedata(void)   
{
   free(excluded);
   free(countbuf);
   free(convtable);
   free(buf);
   free(ttfile);
   free(binfile);
}

/* void because exit() is used */
void main(int argc, char *argv[])
{
   char name[9],ext[5];
   int maxlines,lines;
   long bytespersection;
   int i,j;
   int xcount;
   FILE *binf,*ttf;
   long fsize,count,bytes,codebytes;
   int offset;
   long min;
   int section,sections;
   struct ftime ftime;
   int c;
   int xchar;

   if (argc<2)
   {
      puts("TTCODE 1.02 -- bin-2-ascii encoder\n"
	   "Written By Tom Torfs\n\n"
	   "Syntax: TTCODE <file> [<maxlines>]\n"
	   "The output file(s) will get extensions 001, 002, ... upto 999.\n"
	   "Default for <maxlines> is 750. If you specify a 0 for <maxlines>\n" 
	   "the file won't be split in sections.");
      exit(1);
   }

   allocdata();
   atexit(freedata);

   strncpy(binfile,argv[1],80);
   strupr(binfile);

   fnsplit(binfile,NULL,NULL,name,ext);

   maxlines = 750;
   if (argc>=3)
   {
      maxlines = -1;
      sscanf(argv[2],"%d",&maxlines);
      if (maxlines<0)
      {
	 printf("Invalid number of lines !\n");
	 exit(1);
      }
   }

   if ((binf=fopen(binfile,"rb"))==NULL)
   {
      printf("Source file %s not found !\n",binfile);
      exit(2);
   }

   fseek(binf,0,SEEK_END);
   fsize = ftell(binf);
   rewind(binf);

   getftime(fileno(binf),&ftime);

   printf("Source file: %s, %ld bytes, last modified %02d-%02d-%04d %02d:%02d\n\n",
		    binfile,fsize,
		    ftime.ft_day,ftime.ft_month,ftime.ft_year+1980,
		    ftime.ft_hour,ftime.ft_min);

   printf("Calculating optimal excluded character set...\n\n");

   for (i=0; i<256; i++)
      countbuf[i] = 0;
   for (count=(fsize+BUFSIZE-1)/BUFSIZE; count>0; count--)
   {
      bytes = fread(buf,1,BUFSIZE,binf);
      for (i=0; i<bytes; i++)
	 countbuf[buf[i]]++;
   }
   rewind(binf);

   codebytes = fsize;

   /* pretty dumb way to find the 37 characters with the lowest frequency,
      but hey, at least it works */
   xcount = 0;
   do
   {
      min = 0x7FFFFFFFLU;
      for (i=0; i<256; i++)
      {
	 if (countbuf[i]<min)
	 {
	    min = countbuf[i];
	    j = i;
	 }
      }
      excluded[xcount++] = j;
      codebytes += countbuf[j];
      countbuf[j] = 0x7FFFFFFFLU;
   }
   while (xcount<EXCLUDED);

   /* build conversion table */
   for (i=0; i<256; i++)
      convtable[i] = 0;
   for (i=0; i<EXCLUDED; i++)
      convtable[excluded[i]] = i+1000;
   j = FIRSTCHAR;
   for (i=0; i<256; i++)
   {
      if (convtable[i]==0)
      {
	 convtable[i] = j;
	 j++;
	 if (j==DELETE || j==SOFTRETURN)
	    j++;
      }
   }

   if (maxlines==0)
   {
      sections = 1;
      printf("File won't be split in sections\n\n");
   }
   else
   {
      bytespersection = (long)maxlines * BYTESPERLINE;
      sections = (int)((codebytes+bytespersection-1)/bytespersection);
      printf("File will be split in %d sections of max %d lines\n\n",
	      sections,maxlines);
   }

   fread(buf,1,BUFSIZE,binf);
   offset = 0;
   xchar = 0;
   for (section=1; section<=sections; section++)
   {
      sprintf(ttfile,"%s.%03d",name,section);
      if ((ttf=fopen(ttfile,"wt"))==NULL)
      {
	 printf("Error: can't create %s\n",ttfile);
	 exit(2);
      }
      printf("Writing %s...\n",ttfile);
      fprintf(ttf,"TTCODE %s\n\n",idstring);

      fprintf(ttf,"FILE %s%s %ld %02d-%02d-%04d %02d:%02d\n",
		       name,ext,fsize,
		       ftime.ft_day,ftime.ft_month,ftime.ft_year+1980,
		       ftime.ft_hour,ftime.ft_min);
      fprintf(ttf,"X");
      for (i=0; i<(EXCLUDED+1)/2; i++)
	 fprintf(ttf," %02X",excluded[i]);
      fprintf(ttf,"\n");
      fprintf(ttf,"X");
      for (i=(EXCLUDED+1)/2; i<EXCLUDED; i++)
	 fprintf(ttf," %02X",excluded[i]);
      fprintf(ttf,"\n\n");

      fprintf(ttf,"SECTION %d/%d",section,sections);

      crc = 0xFFFF;

      if (maxlines==0)
	 bytes = codebytes;
      else if (section==sections)
	 bytes = codebytes%bytespersection;
      else
	 bytes = bytespersection;
      lines = 0;
      for (count=0; count<bytes; count++)
      {
	 if (count%BYTESPERLINE==0)
	 {
	    lines++;
	    if (maxlines!=0 && lines>maxlines)
	       break;
	    fprintf(ttf,"\nT");
	 }
	 if (xchar)
	 {
	    crc16(excluded[c]);
	    fputc(c+FIRSTCHAR,ttf);
	    xchar = 0;
	 }
	 else
	 {
	    i = buf[offset];
	    c = convtable[i];
	    if (c>=1000)
	    {
	       fputc(XCHARID,ttf);
	       c = c-1000;
	       xchar = 1;
	    }
	    else
	    {
	       crc16(i);
	       fputc(c,ttf);
	    }
	    offset++;
	    if (offset==BUFSIZE)
	    {
	       fread(buf,1,BUFSIZE,binf);
	       offset = 0;
	    }
	 }
      }

      crc = ~crc;
      crc = (crc << 8) | ((crc >> 8) & 0xff);

      fprintf(ttf,"\nEND CRC16=%04X\n",crc);

      fclose(ttf);
   }

   fclose(binf);

   printf("\nReady.\n");

   exit(0);
}
