//   ⨢ True-Cpp plugin. (c) Andrey Tretjakov,1999 v.1.0

#include "winmem.h"

#include <plugin.hpp>
#include <farkeys.hpp>

#include "eicoll.h"
#include "true-tpl.h"
#include "ttpl-lng.h"
#include "farintf.h"
#include "console.h"

#include "cregexp.h"

#include "syslog.h"

bool IsOldFar = true;

enum TExpandType
{
  expAnyWhere,
  expAtStart,
  expAtMiddle,
  expAtEnd,
  expOnBlank,
  expNone
};

#define MAX_REG_LEN 128
#define MAX_STR_LEN 4096

struct TMacro
{
  char Keyword[MAX_REG_LEN];
  unsigned MinLen;
  int FARKey;
  TExpandType ExpandType;
  char MacroText[MAX_STR_LEN];
};

struct TIndent
{
  char mask[MAX_REG_LEN], relative[MAX_REG_LEN], immChar;
  int indent[2], start, BracketsMode, bracketLink;
};

struct TBracket
{
  char open[MAX_REG_LEN], close[MAX_REG_LEN];
};

struct TComment
{
  char mask[MAX_REG_LEN];
};

enum TJumpType { jtNone, jtFirst, jtMenu, jtSmart };
enum TSaveType { saNone, saAll,  saCurrent };
enum TCDType   { cdNone, cdFile, cdBase };

struct TCompiler
{
  TCompiler();
  TCompiler(const TCompiler&);
  char title[MAX_REG_LEN], err[MAX_REG_LEN];
  int lineMatch, colMatch, fileMatch;
  void defaults(void)
  {
    *title = *err = 0;
    lineMatch = colMatch = fileMatch = -1;
  }
};

TCompiler::TCompiler()
{
  defaults();
}

TCompiler::TCompiler(const TCompiler& e)
{
  lstrcpy(title, e.title);
  lstrcpy(err, e.err);
  lineMatch = e.lineMatch;
  colMatch = e.colMatch;
  fileMatch = e.fileMatch;
}

struct TExec
{
  TExec();
  TExec(const TExec&);
  char title[MAX_REG_LEN], cmd[MAX_STR_LEN], compiler[MAX_STR_LEN];
  char enable[NM], disable[NM];
  TJumpType jumpType;
  TSaveType saveType;
  TCDType cd;
  bool searchBase, ask;
  void defaults(void)
  {
    *title = *cmd = *compiler = *enable = *disable = 0;
    saveType = saCurrent;
    jumpType = jtSmart;
    cd = cdNone;
    searchBase = ask = false;
  }
};

TExec::TExec()
{
  defaults();
}

TExec::TExec(const TExec& e)
{
  lstrcpy(title, e.title);
  lstrcpy(cmd, e.cmd);
  lstrcpy(compiler, e.compiler);
  lstrcpy(enable, e.enable);
  lstrcpy(disable, e.disable);
  jumpType = e.jumpType;
  saveType = e.saveType;
  cd = e.cd;
  ask = e.ask;
  searchBase = e.searchBase;
}

struct TLang
{
  char mask[MAX_REG_LEN], desc[MAX_REG_LEN], imm[MAX_REG_LEN];
  int ignoreCase, TabSize;
  TExec defExec;
  TCollection macroColl, indentColl, bracketColl, commentColl, execColl, compilerColl;
};

static void delLang(void *it)
{
  TLang *lng = (TLang*)it;
  lng->macroColl.removeAll();
  lng->indentColl.removeAll();
  lng->bracketColl.removeAll();
  lng->commentColl.removeAll();
  lng->execColl.removeAll();
  lng->compilerColl.removeAll();
}

class TLangCollection: public TCollection
{
  public:
  TLangCollection() : TCollection(5, 5, delLang) {};
};

static TLangCollection langColl;
static int pluginBusy = 0;
static int pluginStop = 0;

static char *addExt(char *path, char *defExt)
{
  int n = lstrlen(path)-1;
  for ( int i = n ; i >= 0 ; i-- )
  {
    if ( path[i] == '.' )
    {
      lstrcpy(path+i, defExt);
      break;
    }
    if ( ( path[i] == '\\' ) || ( path[i] == ':' ) || ( i == 0 ) )
      return lstrcat(path, defExt);
  }
  return path;
}

static char *lstrchr(char *s, int c)
{
  char *p;
  for ( p = s ; *p ; p++ )
    if ( *p == c )
      return p;
  return c ? NULL : p;
}

static inline int isCharSpace(char c)
{
  return lstrchr(" \t\r\n", c) != NULL;
}

static char *skipSpaces(char*& line)
{
  while ( *line && isCharSpace(*line) )
    line++;
  return line;
}

static char *getWord(char*& line, char *kwd)
{
  char *k = kwd;
  if ( *skipSpaces(line) == '\"')
  {
    while ( *++line )
    {
      if ( *line == '\\' && line[1] == '\"' )
        *k++ = *++line;
      else if ( *line == '\"' )
      {
        if ( *skipSpaces(++line) != '\"' )
          break;
      }
      else
        *k++ = *line;
    }
    *k = 0;
  }
  else
  {
    while ( *line && !isCharSpace(*line) )
      *k++ = *line++;
    *k = 0;
  }
  return kwd;
}

static char *parseItem(char*& line, char *kwd, char *value)
{
  if ( IsCharAlpha(*skipSpaces(line)) )
  {
    char *k = kwd;
    while ( *line && IsCharAlpha(*line) )
      *k++ = *line++;
    *k = *value = 0;
    if ( *skipSpaces(line) == '=' )
    {
      line++;
      getWord(line, value);
    }
    return line;
  }
  return NULL;
}

static char *getItem(char*& line, char *kwd)
{
  if ( *skipSpaces(line) == '<')
  {
    char *k = kwd;
    line++;
    skipSpaces(line);
    while ( *line && IsCharAlpha(*line) )
      *k++ = *line++;
    *k = 0;
    k = line;
    int inQuote = 0;
    while ( *++line )
    {
      if ( inQuote )
      {
        if ( *line == '\\' && line[1] == '\"' )
          line++;
        else if ( *line == '\"' )
          inQuote = 0;
      }
      else
      {
        if ( *line == '>' )
        {
          *line = 0;
          line++;
          break;
        }
        else if ( *line == '\"' )
          inQuote = 1;
      }
    }
    return k;
  }
  return NULL;
}

static void squeeze(char *path)
{
  char *dest = path;
  char *src = path;
  while ( *src != 0 )
  {
    if ( *src != '.' )
      *dest++ = *src++;
    else
    {
      src++;
      if ( *src == '.' )
      {
        src += 2;
        dest--;
        while( *--dest != '\\' )
          ;
        dest++;
      }
      else
      {
        src++;
        dest += 2;
      }
    }
  }
  *dest = 0;
}

enum enSplitType { enDRIVE = 1, enDIRECTORY = 2, enFILENAME = 4, enEXTENTION = 8 };

static int fnSplit(const char *path, char *drive, char *dir, char *file, char *ext)
{
  int flag = *drive = *dir = *file = *ext = 0;
  char ch, *p = lstrchr((char*)path, ':');
  if ( p )
  {
    ch = p[1];
    p[1] = 0;
    lstrcpy(drive, path);
    flag |= enDRIVE;
    p[1] = ch;
    p++;
  }
  else
    p = (char*)path;
  char *dp = strrchr(p, '\\');
  if ( dp )
  {
    ch = dp[1];
    dp[1] = 0;
    lstrcpy(dir, p);
    dp[1] = ch;
    p = dp+1;
    flag |= enDIRECTORY;
  }
  dp = strchr(p, '.');
  if ( dp )
  {
    lstrcpy(ext, dp);
    flag |= enEXTENTION;
    ch = *dp;
    *dp = 0;
    strcpy(file, p);
    *dp = ch;
  }
  else
    lstrcpy(file, p);
  if ( *file  ) flag |= enFILENAME;
  return flag;
}

static inline char *fnMerge(char *path, char *drive, char *dir, char *file, char *ext)
{
  FSF.AddEndSlash(lstrcat(lstrcpy(path, drive), dir));
  return lstrcat(lstrcat(path, file), ext);
}

static char *fExpand(char *rpath, char *defPath = NULL)
{
  static char path[NM], drive[NM], dir[NM], file[NM], ext[NM];
  static char defDrive[4], defDir[NM];
  if ( defPath )
    strcpy(path, defPath);
  else
    GetCurrentDirectory(sizeof(path), path);
  FSF.AddEndSlash(path);
  fnSplit(path, defDrive, defDir, file, ext);
  int flags = fnSplit(rpath, drive, dir, file, ext);
  if ( (flags & enDRIVE) == 0 )
    lstrcpy(drive, defDrive);
  CharUpperBuff(drive, 1);
  if ( ( flags & enDIRECTORY ) == 0 || ( ( *dir != '\\' ) && ( *dir != '/' ) ) )
  {
    static char curdir[NM];
    lstrcpy(curdir, dir);
    lstrcat(lstrcpy(dir, defDir), curdir);
  }
  squeeze(dir);
  fnMerge(path, drive, dir, file, ext);
  return lstrcpy(rpath, path);
}

static char *getFile(char *path, char *fn)
{
  char file[NM], *fileBuff = NULL;
  HANDLE s = CreateFile(fExpand(lstrcpy(file, fn), path), GENERIC_READ, FILE_SHARE_READ, NULL, OPEN_EXISTING, FILE_ATTRIBUTE_NORMAL, NULL);
  if ( s != INVALID_HANDLE_VALUE )
  {
    unsigned fileSize = GetFileSize(s, NULL);
    if ( fileSize )
    {
      if ( ( fileBuff = new char[fileSize+1] ) != NULL )
      {
        DWORD rb = 0;
        if ( ReadFile(s, fileBuff, fileSize, &rb, NULL) && rb )
          fileBuff[rb] = 0;
        else
        {
          delete [] fileBuff;
          fileBuff = NULL;
        }
      }
    }
  }
  CloseHandle(s);
  return fileBuff;
}

static void parseExec(TExec *tmpe, const char *name, const char *value)
{
  if ( !stricmp(name, "Save") )
  {
    if      ( !stricmp(value, "Current") ) tmpe->saveType = saCurrent;
    else if ( !stricmp(value, "All") )     tmpe->saveType = saAll;
    else                                   tmpe->saveType = saNone;
  }
  else if ( !stricmp(name, "CD") )
  {
    if      ( !stricmp(value, "File") )    tmpe->cd = cdFile;
    else if ( !stricmp(value, "Base") )    tmpe->cd = cdBase;
    else                                   tmpe->cd = cdNone;
  }
  else if ( !stricmp(name, "Jump") )
  {
    if      ( !stricmp(value, "Smart") )   tmpe->jumpType = jtSmart;
    if      ( !stricmp(value, "Menu") )    tmpe->jumpType = jtMenu;
    else if ( !stricmp(value, "First") )   tmpe->jumpType = jtFirst;
    else                                   tmpe->jumpType = jtNone;
  }
  else if ( !stricmp(name, "Ask") )        tmpe->ask = FSF.atoi(value) ? true : false;
  else if ( !stricmp(name, "Title") )      lstrcpy(tmpe->title, value);
  else if ( !stricmp(name, "Enable") )
  {
    if ( strncmp(value, "*\\", 2) )
      lstrcpy(tmpe->enable, value);
    else
    {
      lstrcpy(tmpe->enable, value+2);
      tmpe->searchBase = true;
    }
  }
  else if ( !stricmp(name, "Disable") )    lstrcpy(tmpe->disable, value);
  else if ( !stricmp(name, "Compiler") )   lstrcpy(tmpe->compiler, value);
}

static void parseCompiler(TCompiler *tmpc, const char *name, const char *value)
{
  if      ( !stricmp(name, "Error") )      lstrcpy(tmpc->err, value);
  else if ( !stricmp(name, "Line") )       tmpc->lineMatch = FSF.atoi(value);
  else if ( !stricmp(name, "Col") )        tmpc->colMatch = FSF.atoi(value);
  else if ( !stricmp(name, "File") )       tmpc->fileMatch = FSF.atoi(value);
}

static HANDLE hReloadCheck = 0, hEvent = 0;
bool reloadNeeded = false;
bool reloadInProcess = false;

DWORD WINAPI reloadCheck(LPVOID)
{
  HANDLE hIdle[2];
  char path[NM];
  char *p = strrchr(lstrcpy(path, Info.ModuleName), '\\');
  if ( p )
    *p = 0;
  hIdle[0] = FindFirstChangeNotification(path, true, FILE_NOTIFY_CHANGE_LAST_WRITE);
  if ( !hIdle[0] || hIdle[0] == INVALID_HANDLE_VALUE)
    return 0;
  hIdle[1] = hEvent;
  while ( WaitForMultipleObjects(2, hIdle, false, INFINITE) == WAIT_OBJECT_0 )
  {
    FindNextChangeNotification(hIdle[0]);
    if ( pluginStop || reloadInProcess )
      continue;
    reloadNeeded = true;
  };
  FindCloseChangeNotification(hIdle[0]);
  return 0;
};

static void initThread(void)
{
  DWORD id;
  if ( !hEvent )
    hEvent = CreateEvent(NULL, false, false, NULL);
  hReloadCheck = CreateThread(NULL, 0, reloadCheck, 0, 0, &id);
}

static void doneThread(void)
{
  if ( hReloadCheck )
  {
    SetEvent(hEvent);
    WaitForSingleObject(hReloadCheck, INFINITE);
    CloseHandle(hReloadCheck);
    hReloadCheck = 0;
  }
  if ( hEvent )
  {
    CloseHandle(hEvent);
    hEvent = 0;
  }
}

static void InitMacro()
{
  reloadInProcess = true;
  HANDLE hScreen=Info.SaveScreen(0,0,-1,-1);
  const char *MsgItems[]={GetMsg(MMenuString),GetMsg(MLoading)};
  Info.Message(Info.ModuleNumber,0,NULL,MsgItems,sizeof(MsgItems)/sizeof(MsgItems[0]),0);
  char path[NM];
  addExt(lstrcpy(path, Info.ModuleName), ".ini");
  char *file = strrchr(path, '\\');
  *(file++) = 0;
  char *fileBuff = getFile(path, file);
  if ( fileBuff )
  {
    char *item, *p = fileBuff, name[MAX_STR_LEN], value[MAX_STR_LEN];
    TLang *lng = NULL;
    TCollection dummyLang;
    memcpy(&langColl, &dummyLang, sizeof(TLangCollection));
    while ( ( item = getItem(p, name) ) != NULL )
    {
      if ( !stricmp(name, "Include") )
      {
        char incFile[NM];
        getWord(item, incFile);
        char *incBuff = getFile(path, incFile);
        if ( incBuff )
        {
          char *newBuff = new char[lstrlen(incBuff)+lstrlen(p)+1];
          if ( newBuff )
          {
            lstrcat(lstrcpy(newBuff, incBuff), p);
            delete [] fileBuff;
            fileBuff = p = newBuff;
          }
          delete [] incBuff;
        }
      }
      else if ( !stricmp(name, "Language") )
      {
        lng = new TLang;
        lng->ignoreCase = lng->TabSize = 0;
        lng->mask[0] = lng->desc[0] = 0;
        getWord(item, lng->mask);
        lng->imm[0] = 0;
        lng->defExec.defaults();
        while ( parseItem(item, name, value) )
        {
          if      ( !stricmp(name, "Desc") )
            lstrcpy(lng->desc, value);
          else if ( !stricmp(name, "IgnoreCase") )
            lng->ignoreCase = FSF.atoi(value);
          else if ( !stricmp(name, "Tab") )
            lng->TabSize = FSF.atoi(value);
          else
            parseExec(&(lng->defExec), name, value);
        }
        langColl.insert(lng);
      }
      else if ( lng != NULL )
      {
        if ( !stricmp(name, "Expand") )
        {
          TMacro *tmpm = new TMacro;
          tmpm->ExpandType = expOnBlank;
          tmpm->Keyword[0] = tmpm->MacroText[0] = 0;
          tmpm->FARKey = -1;
          getWord(item, tmpm->Keyword);
          int validTag = true;
          if ( tmpm->Keyword[0] == 0 )
            validTag = false;
          char *opDelimiter = lstrchr(tmpm->Keyword, '|');
          if ( opDelimiter )
          {
            *opDelimiter = 0;
            tmpm->MinLen = lstrlen(tmpm->Keyword);
            lstrcpy(opDelimiter, opDelimiter+1);
          }
          else
            tmpm->MinLen = lstrlen(tmpm->Keyword);
          while ( parseItem(item, name, value) )
          {
            if      ( !stricmp(name, "To") )
              lstrcpy(tmpm->MacroText, value);
            else if ( !stricmp(name, "Key") )
            {
              validTag = true;
              tmpm->FARKey = FSF.FarNameToKey(value);
            }
            else if ( !stricmp(name, "At") )
            {
              if      ( !stricmp(value, "AnyWhere") ) tmpm->ExpandType = expAnyWhere;
              else if ( !stricmp(value, "Start" ) )   tmpm->ExpandType = expAtStart;
              else if ( !stricmp(value, "Middle") )   tmpm->ExpandType = expAtMiddle;
              else if ( !stricmp(value, "End") )      tmpm->ExpandType = expAtEnd;
              else if ( !stricmp(value, "Blank") )    tmpm->ExpandType = expOnBlank;
            }
          }
          if ( validTag )
            lng->macroColl.insert(tmpm);
        }
        else if ( !stricmp(name, "Comment") )
        {
          TComment *tmpc = new TComment;
          tmpc->mask[0] = 0;
          getWord(item, tmpc->mask);
          lng->commentColl.insert(tmpc);
        }
        else if ( !stricmp(name, "Exec") )
        {
          TExec *tmpe = new TExec(lng->defExec);
          getWord(item, tmpe->cmd);
          if ( tmpe->cmd[0] )
          {
            while ( parseItem(item, name, value) )
              parseExec(tmpe, name, value);
            if ( !tmpe->title[0] )
              lstrcpy(tmpe->title, tmpe->cmd);
          }
          lng->execColl.insert(tmpe);
        }
        else if ( !stricmp(name, "Compiler") )
        {
          TCompiler *tmpc = new TCompiler();
          getWord(item, tmpc->title);
          if ( tmpc->title[0] )
          {
            while ( parseItem(item, name, value) )
              parseCompiler(tmpc, name, value);
          }
          lng->compilerColl.insert(tmpc);
        }
        else if ( !stricmp(name, "Indent") )
        {
          TIndent *tmpi = new TIndent;
          tmpi->mask[0] = tmpi->relative[0] = 0;
          tmpi->BracketsMode = tmpi->start = tmpi->immChar = 0;
          getWord(item, tmpi->mask);
          for ( int i = 0 ; i < 2 ; i++ )
            tmpi->indent[i] = 0xFFFF;
          if ( tmpi->mask[0] )
          {
            while ( parseItem(item, name, value) )
            {
              if      ( !stricmp(name, "Line") )     tmpi->indent[0] = FSF.atoi(value);
              else if ( !stricmp(name, "Next") )     tmpi->indent[1] = FSF.atoi(value);
              else if ( !stricmp(name, "Imm") )      tmpi->immChar = *value;
              else if ( !stricmp(name, "Start") )    tmpi->start = FSF.atoi(value);
              else if ( !stricmp(name, "Relative") ) lstrcpy(tmpi->relative, value);
            }
            if ( tmpi->immChar )
            {
              if ( !lstrchr(lng->imm, tmpi->immChar) )
              {
                int len = lstrlen(lng->imm);
                lng->imm[len++] = tmpi->immChar;
                lng->imm[len] = 0;
              }
            }
          }
          lng->indentColl.insert(tmpi);
        }
        else if ( !stricmp(name, "Bracket") )
        {
          TBracket *tmpb = new TBracket;
          TIndent  *tmp0 = new TIndent;
          TIndent  *tmp1 = new TIndent;
          tmpb->open[0] = tmpb->close[0] = 0;
          tmp0->relative[0] = 0;
          tmp0->start = tmp0->immChar = 0;
          tmp0->BracketsMode = 0;
          tmp1->relative[0] = 0;
          tmp1->start = tmp1->immChar = 0;
          tmp1->BracketsMode = 1;
          tmp1->bracketLink = lng->bracketColl.getCount();
          getWord(item, tmpb->open);
          if ( parseItem(item, name, tmpb->close) )
          {
            if ( !stricmp(name, "Match") )
            {
              lstrcpy(tmp0->mask, tmpb->open);
              lstrcpy(tmp1->mask, tmpb->close);
              tmp0->indent[0] = 0xFFFF;
              tmp0->indent[1] = 1;
              tmp1->indent[0] = 0;
              tmp1->indent[1] = 0xFFFF;
              if ( tmpb->open[0] && tmpb->close[0] )
              {
                while ( parseItem(item, name, value) )
                {
                  if      ( !stricmp(name, "Line") )     tmp0->indent[0] = FSF.atoi(value);
                  else if ( !stricmp(name, "Shift") )    tmp0->indent[1] = FSF.atoi(value);
                  else if ( !stricmp(name, "Next") )     tmp1->indent[1] = FSF.atoi(value);
                  else if ( !stricmp(name, "ImmOpen") )  tmp0->immChar = *value;
                  else if ( !stricmp(name, "ImmClose") ) tmp1->immChar = *value;
                  else if ( !stricmp(name, "Start") )    tmp0->start = FSF.atoi(value);
                }
                if ( tmp0->immChar )
                {
                  if ( !lstrchr(lng->imm, tmp0->immChar) )
                  {
                    int len = lstrlen(lng->imm);
                    lng->imm[len++] = tmp0->immChar;
                    lng->imm[len] = 0;
                  }
                }
                if ( tmp1->immChar )
                {
                  if ( !lstrchr(lng->imm, tmp1->immChar) )
                  {
                    int len = lstrlen(lng->imm);
                    lng->imm[len++] = tmp1->immChar;
                    lng->imm[len] = 0;
                  }
                }
              }
              lng->bracketColl.insert(tmpb);
              lng->indentColl.insert(tmp0);
              lng->indentColl.insert(tmp1);
            }
          }
        }
      }
    }
    delete [] fileBuff;
  }
  Info.RestoreScreen(hScreen);
  reloadInProcess = reloadNeeded = false;
  doneThread();
  initThread();
}

static void DoneMacro()
{
  langColl.removeAll();
  doneThread();
}

static inline char *StrLower(char *str)
{
  FSF.LLowerBuf(str, lstrlen(str));
  return str;
}

static TMacro *CheckMacroPos(TMacro *mc, TExpandType exp)
{
  if ( exp == expAnyWhere )
    return mc;
  TMacro *rmc = NULL;
  switch ( mc->ExpandType )
  {
    case expAnyWhere:
      rmc = mc;
      break;
    case expAtMiddle:
      if ( exp == expAtMiddle )
        rmc = mc;
      break;
    case expOnBlank:
      if ( exp == expOnBlank )
        rmc = mc;
      break;
    case expAtStart:
      if ( exp == expOnBlank || exp == expAtStart )
        rmc = mc;
      break;
    case expAtEnd:
      if ( exp == expOnBlank || exp == expAtEnd )
        rmc = mc;
      break;
    default:
      break;
  }
  return rmc;
}

static TMacro *FindMacro(TLang *lng, char *line, unsigned l, TExpandType exp)
{
  for ( unsigned i = 0 ; i < lng->macroColl.getCount() ; i++ )
  {
    TMacro *mm = (TMacro*)(lng->macroColl[i]);
    if ( l >= mm->MinLen && l <= (unsigned)lstrlen(mm->Keyword) )
    {
      TMacro *rmc = CheckMacroPos(mm, exp);
      if ( rmc )
      {
        if ( lng->ignoreCase )
        {
          if ( !FSF.LStrnicmp(mm->Keyword, line, l) )
            return rmc;
        }
        else
        {
          if ( !strncmp(mm->Keyword, line, l) )
            return rmc;
        }
      }
    }
  }
  return NULL;
}

TExpandType GetMacroPos(char *line, char *ptr, int len)
{
  char *startOfLine = line, *endOfLine = lstrchr(line, 0);
  while ( isCharSpace(*startOfLine) )
    startOfLine++;
  endOfLine -= len;
  TExpandType exp = expNone;
  if      ( startOfLine == endOfLine )             exp = expOnBlank;
  else if ( ptr > startOfLine && ptr < endOfLine ) exp = expAtMiddle;
  else if ( ptr == startOfLine )                   exp = expAtStart;
  else if ( ptr == endOfLine )                     exp = expAtEnd;
  return exp;
}

static TMacro *FindMacroKey(TLang *lng, const INPUT_RECORD *Rec)
{
  int rKey = FSF.FarInputRecordToKey(Rec);
  for ( unsigned i = 0 ; i < lng->macroColl.getCount() ; i++ )
  {
    TMacro *mm = (TMacro*)(lng->macroColl[i]);
    if ( mm->FARKey == rKey )
      return mm;
  }
  return NULL;
}

static TMacro *FindMacroKeyEx(TLang *lng, TMacro *mc, TExpandType at)
{
  if ( mc->FARKey != -1 )
    for ( unsigned i = 0 ; i < lng->macroColl.getCount() ; i++ )
    {
     TMacro *mm = (TMacro*)(lng->macroColl[i]);
     if ( ( mc->FARKey == mm->FARKey ) && ( CheckMacroPos(mm, at) != NULL ) )
       return mm;
    }
  return NULL;
}

static char userString[10][80] = { "", "", "", "", "", "", "", "", "", "" };
static char titleString[80] = "", *userStr[10], *titleStr;

static void nullUserString(void)
{
  for ( int i = 0 ; i < 10 ; i++ )
    userStr[i] = NULL;
}

static int insertUserString(int n)
{
  if ( userStr[n] == NULL )
  {
    struct EditorInfo ei;
    static char *h = "TrueTpl";
    Info.EditorControl(ECTL_GETINFO, &ei);
    struct InitDialogItem Init[] =
    {
      { DI_DOUBLEBOX, 1, 0, 22, 3, 0,        0,               titleString    },
      { DI_EDIT,      3, 1, 20, 0, (DWORD)h, DIF_HISTORY,     ""             },
      { DI_BUTTON,    0, 2,  0, 0, 0,        DIF_CENTERGROUP, (char*)MOK     },
      { DI_BUTTON,    0, 2,  0, 0, 0,        DIF_CENTERGROUP, (char*)MCancel }
    };
    int CoorX = ei.CurPos-ei.LeftPos;
    int CoorY = ei.CurLine-ei.TopScreenLine;
    int mx = ( CoorX+28 > ei.WindowSizeX ) ? CoorX-32 : CoorX;
    int my = ( CoorY+3  > ei.WindowSizeY ) ? CoorY-2  : CoorY;
    int size = sizeof(Init)/sizeof(Init[0]);
    FarDialogItem *Item = new FarDialogItem[size];
    InitDialogItems(Init, Item, size);
    lstrcpy(Item[1].Data, userString[n]);
    Item[1].Focus = Item[2].DefaultButton = 1;
    int Result = Info.Dialog(Info.ModuleNumber,mx,my,mx+23,my+3,NULL,Item,size);
    if ( Result == 2 )
      userStr[n] = lstrcpy(userString[n], Item[1].Data);
    delete [] Item;
  }
  if ( userStr[n] )
    Info.EditorControl(ECTL_INSERTTEXT, userStr[n]);
  return userStr[n] != NULL;
}

struct TEditorPos
{
  TEditorPos() { Default(); }
  int Row, Col;
  int TopRow,LeftCol;
  void Default(void) { Row = Col = TopRow = LeftCol = -1; }
};

static TEditorPos EditorGetPos(void)
{
  TEditorPos r;
  EditorInfo ei;
  Info.EditorControl(ECTL_GETINFO, &ei);
  r.Row = ei.CurLine;
  r.Col = ei.CurPos;
  r.TopRow = ei.TopScreenLine;
  r.LeftCol = ei.LeftPos;
  return r;
}

static void EditorSetPos(TEditorPos pos)
{
  EditorSetPosition sp;
  sp.CurLine = pos.Row;
  sp.CurPos = pos.Col;
  sp.TopScreenLine = pos.TopRow;
  sp.LeftPos = pos.LeftCol;
  sp.CurTabPos = sp.Overtype = -1;
  Info.EditorControl(ECTL_SETPOSITION,&sp);
}

void EditorSetPos(int line, int col, int topline = -1, int leftcol = -1)
{
  EditorSetPosition sp;
  sp.CurLine = line;
  sp.CurPos = col;
  sp.TopScreenLine = topline;
  sp.LeftPos = leftcol;
  sp.CurTabPos = sp.Overtype = -1;
  Info.EditorControl(ECTL_SETPOSITION, &sp);
}

static void EditorProcessKey(char ascii)
{
  INPUT_RECORD tr;
  tr.EventType = KEY_EVENT;
  tr.Event.KeyEvent.bKeyDown = true;
  tr.Event.KeyEvent.wRepeatCount = 1;
  tr.Event.KeyEvent.wVirtualKeyCode = 0;
  tr.Event.KeyEvent.uChar.AsciiChar = ascii;
  tr.Event.KeyEvent.dwControlKeyState = 0;
  Info.EditorControl(ECTL_PROCESSINPUT, &tr);
}

static inline void EditorProcessFARKey(int key)
{
  Info.EditorControl(ECTL_PROCESSKEY, (void*)key);
}

static inline void EditorInsertText(char *s)
{
  Info.EditorControl(ECTL_INSERTTEXT, (void*)s);
}

static inline void redraw(void)
{
  Info.EditorControl(ECTL_REDRAW, 0);
}

static void EditorProcessTabKey(int TabSize, int pos, int n)
{
  if ( n > 0 )
  {
    int idnt = n;
    char symbol = ' ';
    if ( TabSize ) // if TabSize = 0 use Tabs! [updated by DO]
      idnt = n*TabSize-pos%TabSize;
    else
      symbol = '\t';
    if ( idnt > 0 )
    {
      char *s = new char[idnt+1];
      memset(s, symbol, idnt);
      s[idnt] = 0;
      EditorInsertText(s);
      delete [] s;
      redraw();
    }
  }
}

static void EditorProcessReturnKey(int before = -1, int after = -1)
{
  INPUT_RECORD tr;
  tr.EventType = KEY_EVENT;
  tr.Event.KeyEvent.bKeyDown = true;
  tr.Event.KeyEvent.wRepeatCount = 1;
  tr.Event.KeyEvent.wVirtualKeyCode = VK_RETURN;
  tr.Event.KeyEvent.dwControlKeyState = 0;
  if ( before != -1 )
    pluginBusy = before;
  Info.EditorControl(ECTL_PROCESSINPUT, &tr);
  if ( after != -1 )
    pluginBusy = after;
}

static void EditorGetString(EditorGetString *gs, int line = -1)
{
  gs->StringNumber = line;
  Info.EditorControl(ECTL_GETSTRING, gs);
}

static void EditorSetString(char *src, int line = -1)
{
  struct EditorSetString st;
  st.StringNumber = line;
  st.StringText = src;
  st.StringEOL = 0;
  st.StringLength = lstrlen(src);
  Info.EditorControl(ECTL_SETSTRING, &st);
}

static void EditorGoToHome(void)
{
  struct EditorGetString gs;
  EditorProcessFARKey(KEY_HOME);
  EditorGetString(&gs);
  if ( isCharSpace(gs.StringText[0]) )
    EditorProcessFARKey(KEY_RIGHT|KEY_CTRL);
}

static inline void insIns(char *ins, int& insPos)
{
  if ( *ins )
  {
    EditorInsertText(ins);
    insPos = *ins = 0;
    redraw();
  }
}

static void RunMacro(TMacro *m, TLang *lng)
{
  int userStrN, insPos = 0, setPos = 0;
  char ins[MAX_STR_LEN] = "";
  TEditorPos pos, curPos;
  pluginBusy = 1;
  nullUserString();
  for ( char *p = m->MacroText ; *p ; p++ )
  {
    char *cb;
    if ( *p == '\\' )
    {
      insIns(ins, insPos);
      switch ( *++p )
      {
        case 'p':
          setPos = 1;
          pos = EditorGetPos();
          break;
        case 0:
        case '\\':
          EditorProcessFARKey(KEY_BACKSLASH);
          break;
        case 't':
          curPos = EditorGetPos();
          EditorProcessTabKey(lng->TabSize, curPos.Col, 1);
          break;
        case 'h':
          EditorGoToHome();
          break;
        case 'b':
          EditorProcessFARKey(KEY_BS);
          break;
        case '^':
          EditorProcessFARKey(KEY_UP);
          break;
        case 'v':
          EditorProcessFARKey(KEY_DOWN);
          break;
        case '<':
          EditorProcessFARKey(KEY_LEFT);
          break;
        case '>':
          EditorProcessFARKey(KEY_RIGHT);
          break;
        case '[':
          EditorProcessFARKey(KEY_HOME);
          break;
        case ']':
          EditorProcessFARKey(KEY_END);
          break;
        case '(':
          EditorProcessFARKey(KEY_LEFT|KEY_CTRL);
          break;
        case ')':
          EditorProcessFARKey(KEY_RIGHT|KEY_CTRL);
          break;
        case 'r':
          EditorProcessReturnKey(0, 1);
          break;
        case 'n':
          EditorProcessReturnKey();
          break;
        case '!':
          pluginBusy = 0;
          EditorProcessKey(*++p);
          pluginBusy = 1;
        case '?':
          if ( p[1] == '\'' )
          {
            p++;
            *(titleStr = titleString) = 0;
            while ( *++p && *p != '\'' )
              *titleStr++ = *p;
            *titleStr = 0;
          }
          break;
        case '0': case '1': case '2': case '3': case '4':
        case '5': case '6': case '7': case '8': case '9':
          userStrN = *p-'0';
          if ( p[1] == '\'' )
          {
            p++;
            *(titleStr = userString[userStrN]) = 0;
            while ( *++p && *p != '\'' )
              *titleStr++ = *p;
            *titleStr = 0;
          }
          if ( !insertUserString(userStrN) )
            userStr[userStrN] = lstrcpy(userString[userStrN], "");
          break;
        case '{':
          if ( ( cb = lstrchr(p+1, '}') ) == NULL )
            EditorProcessKey(*p);
          else
          {
            char ch = *cb;
            *cb = 0;
            EditorProcessFARKey(FSF.FarNameToKey(p+1));
            *cb = ch;
            p = cb;
          }
          break;
        default:
          pluginBusy = 0;
          EditorProcessKey(*p);
          pluginBusy = 1;
          break;
      }
    }
    else
    {
      ins[insPos++] = *p;
      ins[insPos] = 0;
    }
  }
  insIns(ins, insPos);
  pluginBusy = 0;
  if ( setPos )
    EditorSetPos(pos);
}

int WINAPI _export GetMinFarVersion(void)
{
  return FARMANAGERVERSION;
//  return MAKEFARVERSION(1,70,1193);
}

HANDLE heap;

void WINAPI _export SetStartupInfo(const struct PluginStartupInfo *Info)
{
  initMem();
  ::Info = *Info;
  IsOldFar = true;
  if ( Info->StructSize >= sizeof(struct PluginStartupInfo) )
  {
    ::FSF = *Info->FSF;
    ::Info.FSF = &::FSF;
    IsOldFar = false;
    FSF.sprintf(PluginRootKey,"%s\\TRUE-TPL",Info->RootKey);
    GetRegKey(HKEY_CURRENT_USER, "", "Disable", pluginStop, 0);
    InitMacro();
  }
}

TEICollection *eList = NULL;

static void initEList(void)
{
  if ( eList == NULL )
    eList = new TEICollection;
}

void WINAPI _export GetPluginInfo(struct PluginInfo *Info)
{
  if ( !IsOldFar )
  {
    static const char *PluginMenuStrings[1];
    Info->StructSize = sizeof(*Info);
    Info->Flags = PF_EDITOR|PF_DISABLEPANELS;
    Info->DiskMenuStringsNumber = 0;
    Info->PluginConfigStringsNumber = 0;
    PluginMenuStrings[0] = GetMsg(MMenuString);
    Info->PluginMenuStrings = PluginMenuStrings;
    Info->PluginMenuStringsNumber = 1;
    static const char *PluginCfgStrings[1];
    PluginCfgStrings[0]=GetMsg(MSetupString);
    Info->PluginConfigStrings=PluginCfgStrings;
    Info->PluginConfigStringsNumber=sizeof(PluginCfgStrings)/sizeof(PluginCfgStrings[0]);
  }
}

int Config(void)
{
  struct InitDialogItem InitItems[] =
  {
    { DI_DOUBLEBOX, 3,1,33,  5,0,         0,               (char*)MSetupTitle },
    { DI_CHECKBOX,  5,2, 0,  0,pluginStop,0,               (char*)MDisable    },
    { DI_TEXT,      5,3, 0,255,0,         0,               NULL               },
    { DI_BUTTON,    0,4, 0,  0,0,         DIF_CENTERGROUP, (char*)MReload     },
    { DI_BUTTON,    0,4, 0,  0,0,         DIF_CENTERGROUP, (char*)MCancel     }
  };
  int ExitCode = ExecDialog(InitItems,sizeof(InitItems)/sizeof(InitItems[0]),1,3,NULL);
  if ( ExitCode == 3 )
  {
    SetRegKey(HKEY_CURRENT_USER, "", "Disable", pluginStop = InitItems[1].Selected);
    DoneMacro();
    InitMacro();
    return true;
  }
  return false;
}

int WINAPI _export Configure(int ItemNumber)
{
  if ( ItemNumber )
    return false;
  return Config();
}

static inline int matched(const char *mask, const char *value)
{
  return FSF.ProcessName(mask, (char*)value, PN_CMPNAMELIST|PN_SKIPPATH);
}

static char *FirstNonSpace(const char *s)
{
  char *i = (char*)s;
  while ( *i )
  {
    if ( !isCharSpace(*i) )
      return i;
    i++;
  }
  return NULL;
}

static int strMatch(char *s, char *pattern, char *prefix, char *suffix, int *l, int n, int *l2 = 0, int n2 = 0, int *l3 = 0, int n3 = 0)
{
  int ret = false;
  char slashPattern[320];
  static CRegExp reg;
  SMatches m;
  lstrcat(lstrcat(lstrcpy(slashPattern, prefix), pattern), suffix);
  if ( reg.SetExpr(slashPattern) )
  {
    if ( slashPattern[1] == '^' )
      reg.SetNoMoves(true);
    ret = reg.Parse(s, &m) ? true : false;
    if ( ret )
    {
      if ( l && n >= 0 )
      {
        l[0] = m.s[n];
        l[1] = m.e[n];
      }
      if ( l2 && n2 >= 0 )
      {
        l2[0] = m.s[n2];
        l2[1] = m.e[n2];
      }
      if ( l3 && n3 >= 0 )
      {
        l3[0] = m.s[n3];
        l3[1] = m.e[n3];
      }
    }
  }
  return ret;
}

static int FindIndent(int spORret, char c, TLang *lng, char *str, int foundBounds[2])
{
  char line[MAX_STR_LEN];
  if ( lng->ignoreCase )
    StrLower(line);
  int bounds[2];
  for ( unsigned j = 0 ; j < lng->indentColl.getCount() ; j++ )
  {
    TIndent *tmpi = (TIndent*)(lng->indentColl[j]);
    lstrcpy(line, tmpi->mask);
    char ic = ((TIndent*)(lng->indentColl[j]))->immChar;
    if ( lng->ignoreCase )
      ic = (char)FSF.LLower(ic);
    if ( strMatch(str, line, "/^", lng->ignoreCase ? "$/i" : "$/", bounds, tmpi->start) )
    {
      if ( spORret || ( ic == c ) )
      {
        foundBounds[0] = bounds[0];
        foundBounds[1] = bounds[1];
        return j;
      }
    }
  }
  return -1;
}

static int FindBrackets(TLang *lng, char *str, char **foundBrackets)
{
  char line[MAX_STR_LEN], *closeReg = "$/";
  if ( lng->ignoreCase )
  {
    StrLower(line);
    closeReg = "$/i";
  }
  int bounds[2], foundBracketsNum = 0;
  for ( unsigned j = 0 ; j < lng->indentColl.getCount() ; j++ )
  {
    TIndent *tmpi = (TIndent*)(lng->indentColl[j]);
    lstrcpy(line, tmpi->mask);
    if ( strMatch(str, line, "/^", closeReg, bounds, tmpi->start) )
    {
      if ( tmpi->indent[0] != 0xFFFF && tmpi->BracketsMode )
      {
        TBracket *br = (TBracket*)(lng->bracketColl[tmpi->bracketLink]);
        foundBrackets[foundBracketsNum++] = br->open;
      }
    }
  }
  return foundBracketsNum;
}

static char *LookForOpenBracket(TEditorPos& p, TLang *lng, char **openBr, int nOpenBr)
{
  static char *c, buff[MAX_STR_LEN];
  static struct EditorGetString gs;
  int ret = -1;
  char *closeReg = lng->ignoreCase ? "$/i" : "$/";
  while ( --p.Row >= 0 )
  {
    EditorSetPos(p);
    EditorGetString(&gs);
    if ( gs.StringLength )
    {
      if ( ( c = FirstNonSpace(gs.StringText) ) != NULL )
      {
        FSF.Trim(lstrcpy(buff, c));
        if ( lng->ignoreCase )
          StrLower(buff);
        for ( int i = 0 ; i < nOpenBr ; i++ )
        {
          if ( strMatch(buff, openBr[i], "/^", closeReg, NULL, 0) )
          {
            ret = c-gs.StringText;
            break;
          }
        }
        if ( ret != -1 )
          break;
        unsigned n = lng->bracketColl.getCount();
        for ( unsigned i = 0 ; i < n ; i++ )
        {
          TBracket *br = (TBracket*)(lng->bracketColl[i]);
          if ( strMatch(buff, br->close, "/^", closeReg, NULL, 0 ) )
          {
            char *result = NULL, **br4look = new char*[n];
            int nbr4look = FindBrackets(lng, buff, br4look);
            if ( nbr4look )
              result = LookForOpenBracket(p, lng, br4look, nbr4look);
            delete [] br4look;
            if ( result )
              break;
            return NULL;
          }
        }
      }
    }
  }
  if ( ret != -1 )
  {
    memcpy(buff, gs.StringText, ret);
    buff[ret] = 0;
    return buff;
  }
  return NULL;
}

static char *FindOpenBracket(TLang *lng, char **openBr, int n)
{
  TEditorPos p, o = EditorGetPos();
  p.Default();
  p.Row = o.Row;
  char *ret = LookForOpenBracket(p, lng, openBr, n);
  EditorSetPos(o);
  return ret;
}

static char *prevStringIndent(void)
{
  static char buff[MAX_STR_LEN];
  struct EditorGetString gs;
  int ret = 0;
  TEditorPos p, o = EditorGetPos();
  p.Default();
  p.Row = o.Row;
  while ( --p.Row >= 0 )
  {
    EditorSetPos(p);
    EditorGetString(&gs);
    if ( gs.StringLength )
    {
      char *c = FirstNonSpace(gs.StringText);
      if ( c )
      {
        ret = c-gs.StringText;
        break;
      }
    }
  }
  EditorSetPos(o);
  memcpy(buff,gs.StringText,ret);
  buff[ret]=0;
  return buff;
}

static int shiftLine(int i, int TabSize)
{
  TEditorPos pos = EditorGetPos();
  pluginBusy = 1;
  if ( i > 0 )
    EditorProcessTabKey(TabSize, pos.Col, i);
  else if ( i < 0 )
  {
    int ind = -i*TabSize;
    for ( int j = 0 ; j < ind ; j++ )
      EditorProcessFARKey(KEY_BS);
  }
  TEditorPos pos2 = EditorGetPos();
  pluginBusy = 0;
  return pos2.Col-pos.Col;
}

static int TryIndent(int ret, TLang *lng, int j, char *str, char *realStr, int oldIndent, int foundBounds[2])
{
  TIndent *tmpi = (TIndent*)(lng->indentColl[j]);
  int iCurr = tmpi->indent[0];
  int iNext = tmpi->indent[1];
  if ( iCurr != 0xFFFF )
  {
    bool processIndent = true;
    char *opencol = NULL;
    TEditorPos pos = EditorGetPos();
    if ( *(tmpi->relative) )
    {
      char *rel = tmpi->relative;
      if ( ( opencol = FindOpenBracket(lng, &rel, 1) ) == NULL )
        processIndent = false;
    }
    else if ( tmpi->BracketsMode )
    {
      unsigned n = lng->bracketColl.getCount();
      char **br4look = new char*[n];
      int nbr4look = FindBrackets(lng, str, br4look);
      processIndent = false;
      if ( nbr4look )
        opencol = FindOpenBracket(lng, br4look, nbr4look);
      if ( opencol )
        processIndent = true;
      delete [] br4look;
    }
    if ( processIndent )
    {
      if ( opencol == NULL )
        opencol = prevStringIndent();
      size_t olen = lstrlen(opencol);
      char *ptr = new char[olen+lstrlen(str)+1];
      lstrcat(lstrcpy(ptr, opencol), realStr);
      EditorSetString(ptr);
      EditorSetPos(-1, olen);
      EditorSetPos(-1, pos.Col+olen-oldIndent+shiftLine(iCurr, lng->TabSize));
    }
  }
  if ( ret )
  {
    EditorProcessReturnKey(1, 0);
    if ( tmpi->start )
    {
      pluginBusy = 1;
      for ( int k = foundBounds[0] ; k < foundBounds[1] ; k++ )
        EditorProcessKey(realStr[k]);
      pluginBusy = 0;
    }
    if ( iNext != 0xFFFF )
      shiftLine(iNext, lng->TabSize);
  }
  else
    redraw();
  return IGNORE_EVENT;
}

static int inComment(TLang *lng, const char *realLine, unsigned pos)
{
  int ic = lng->ignoreCase;
  for ( unsigned i = 0 ; i < lng->commentColl.getCount() ; i++ )
  {
    char line[MAX_STR_LEN];
    TComment *tmpc = (TComment*)(lng->commentColl[i]);
    lstrcpy(line, realLine);
    if ( ic )
      StrLower(line);
    int bounds[2];
    if ( strMatch(line, tmpc->mask, "/", ic ? "/i" : "/", bounds, 0) )
      if ( pos >= (unsigned)bounds[0] && pos <= (unsigned)bounds[1] )
        return 1;
  }
  return 0;
}

static bool checkMultipleChoice(TLang *lng, TMacro*&mc, TExpandType exp)
{
  const char *menuPrefix = "\\~";
  unsigned pl = lstrlen(menuPrefix);
  if ( (unsigned)lstrlen(mc->MacroText) > pl && !strncmp(mc->MacroText, menuPrefix, pl) )
  {
    unsigned lc = 0;
    char *p = mc->MacroText, *p1;
    while ( ( p1 = strstr(p, menuPrefix) ) != NULL )
    {
      p = p1+pl;
      lc++;
    }
    if ( lc )
    {
      FarMenuItem *amenu = new FarMenuItem[lc];
      TMacro **amacro = new TMacro*[lc];
      if ( amenu )
      {
        unsigned i = 0;
        p = mc->MacroText;
        while ( ( p1 = strstr(p, menuPrefix) ) != NULL )
        {
          p = p1+pl;
          char ch, *p2 = strstr(p, menuPrefix);
          if ( p2 )
          {
            ch = *p2;
            *p2 = 0;
          }
          TMacro *tm;
          char *p3 = lstrchr(p, '=');
          if ( p3 )
          {
            *p3 = 0;
            tm = FindMacro(lng, p, lstrlen(p), exp);
            lstrcpy(amenu[i].Text, p3+1);
            *p3 = '=';
          }
          else
          {
            tm = FindMacro(lng, p, lstrlen(p), exp);
            lstrcpy(amenu[i].Text, p);
          }
          if ( p2 )
            *p2 = ch;
          if ( tm )
          {
            amacro[i] = tm;
            amenu[i].Selected = amenu[i].Checked = amenu[i].Separator = false;
            i++;
          }
        }
        int res = Info.Menu(Info.ModuleNumber,-1,-1,0,FMENU_WRAPMODE,GetMsg(MSelectMacro),NULL,NULL,NULL,NULL,amenu,i);
        bool retval = false;
        if ( res != -1 && amacro[res] )
        {
          mc = amacro[res];
          retval = true;
        }
        delete [] amenu;
        delete [] amacro;
        return retval;
      }
    }
  }
  return true;
}

static int TryMacro(TLang *lng, struct EditorGetString& gs, TEditorPos& ep, char *line)
{
  if ( !isCharSpace(gs.StringText[ep.Col-1]) )
  {
    char *ptr = lstrcpy(line, gs.StringText)+ep.Col;
    char ch = *ptr;
    *ptr-- = 0;
    while ( (ptr>=line) && !isCharSpace(*ptr) )
      ptr--;
    int len = lstrlen(++ptr);
    line[ep.Col] = ch;
    if ( len )
    {
      if ( !inComment(lng, line, ep.Col) )
      {
        TExpandType at = GetMacroPos(line, ptr, len);
        TMacro *mc = FindMacro(lng, ptr, len, at);
        if ( mc && checkMultipleChoice(lng, mc, at) )
        {
          pluginBusy = 1;
          for ( int i = 0 ; i < len ; i++ )
            EditorProcessFARKey(KEY_BS);
          pluginBusy = 0;
          RunMacro(mc, lng);
          redraw();
          return 1;
        }
      }
    }
  }
  return 0;
}

static int TryMacroKey(TLang *lng, TMacro *mc, struct EditorGetString& gs, TEditorPos& ep, char *line)
{
  char *ptr = lstrcpy(line, gs.StringText)+ep.Col;
  if ( !inComment(lng, line, ep.Col) )
  {
    TExpandType at = GetMacroPos(line, ptr, 0);
    if ( CheckMacroPos(mc, at) != NULL )
    {
      RunMacro(mc, lng);
      redraw();
      return 1;
    }
    if ( FindMacroKeyEx(lng, mc, at) )
    {
      RunMacro(mc, lng);
      redraw();
      return 1;
    }
  }
  return 0;
}

static int findLngID(const char *fn)
{
  for ( unsigned i = 0 ; i < langColl.getCount() ; i++ )
  {
    TLang *lng = (TLang*)(langColl[i]);
    if ( matched(lng->mask, fn) )
      return i;
  }
  return -1;
}

int WINAPI _export ProcessEditorEvent(int Event, void *Param)
{
  struct EditorInfo ei;
  initEList();
  switch ( Event )
  {
    case EE_READ:
      Info.EditorControl(ECTL_GETINFO, &ei);
      eList->insert(ei.EditorID, findLngID(ei.FileName), ei.FileName);
      return 0;
    case EE_CLOSE:
      eList->removeID(*((int*)Param));
      return 0;
    case EE_REDRAW:
      if ( reloadNeeded )
      {
        reloadInProcess = true;
        DoneMacro();
        InitMacro();
      }
      return 0;
    default:
      return 0;
  }
}

int WINAPI _export ProcessEditorInput(const INPUT_RECORD *Rec)
{
  struct EditorInfo ei;
  if ( pluginStop || pluginBusy || Rec->EventType != KEY_EVENT || Rec->Event.KeyEvent.bKeyDown == 0 )
    return PROCESS_EVENT;
  Info.EditorControl(ECTL_GETINFO, &ei);
  initEList();
  if ( eList->findID(ei.EditorID) == -1 )
    eList->insert(ei.EditorID, findLngID(ei.FileName), ei.FileName);
  int id = eList->findLang(ei.EditorID);
  if ( id != -1 )
  {
    TLang *lng = (TLang*)(langColl[id]);
    if ( lng )
    {
      char line[MAX_STR_LEN];
      struct EditorGetString gs;
      TEditorPos epos = EditorGetPos();
      TMacro *fm;
      if ( ( fm = FindMacroKey(lng, Rec) ) != NULL )
      {
        EditorGetString(&gs);
        if ( checkMultipleChoice(lng, fm, expAnyWhere) )
          return TryMacroKey(lng, fm, gs, epos, line) ? IGNORE_EVENT : PROCESS_EVENT;
        return IGNORE_EVENT;
      }
      if ( Rec->Event.KeyEvent.dwControlKeyState & ( RIGHT_ALT_PRESSED | LEFT_ALT_PRESSED | RIGHT_CTRL_PRESSED | LEFT_CTRL_PRESSED ) )
        return PROCESS_EVENT;
      char c = Rec->Event.KeyEvent.uChar.AsciiChar;
      if ( c == '\t' || Rec->Event.KeyEvent.wVirtualKeyCode == VK_TAB )
      {
        EditorProcessTabKey(lng->TabSize, epos.Col, 1);
        return IGNORE_EVENT;
      }
      int inImm, spORret = ( c == ' ' || Rec->Event.KeyEvent.wVirtualKeyCode == VK_RETURN );
      if ( lng->ignoreCase )
        inImm = lstrchr(lng->imm, FSF.LLower(c)) || lstrchr(lng->imm, FSF.LUpper(c));
      else
        inImm = lstrchr(lng->imm, c) != NULL;
      if ( !inImm && epos.Col == 0 )
        return PROCESS_EVENT;
      if ( ! ( spORret || ( c && inImm ) ) )
        return PROCESS_EVENT;
      if ( c == ' ' )
      {
        EditorGetString(&gs);
        return TryMacro(lng, gs, epos, line) ? IGNORE_EVENT : PROCESS_EVENT;
      }
      else
      {
        int ret = PROCESS_EVENT;
        EditorGetString(&gs);
        if ( !inComment(lng, gs.StringText, epos.Col) )
        {
          if ( !spORret )
          {
            pluginBusy = 1;
            EditorProcessKey(c);
            pluginBusy = 0;
            ret = IGNORE_EVENT;
            EditorGetString(&gs);
          }
          char *str = FirstNonSpace(gs.StringText);
          if ( str == NULL )
            return ret;
          int spacesCount = str-gs.StringText;
          FSF.Trim(lstrcpy(line, str));
          if ( lng->ignoreCase )
          {
            StrLower(line);
            c = (char)FSF.LLower(c);
          }
          int bounds[2];
          int j = FindIndent(spORret, c, lng, line, bounds);
          if ( j != -1 )
            return TryIndent(spORret, lng, j, line, str, spacesCount, bounds);
          return ret;
        }
      }
    }
  }
  return PROCESS_EVENT;
}

static inline char *delBackslash(char *s)
{
  if ( *s && strcmp(s, "\\") )
  {
    char *b = strchr(s, 0)-1;
    if ( *b == '\\' )
      *b = 0;
  }
  return s;
}

static char* QuoteText(char *Str)
{
  if (*Str=='-' || *Str=='^' || strpbrk(Str," &+,")!=NULL)
  {
    unsigned l = lstrlen(Str);
    memcpy(Str+1,Str,l);
    *Str = Str[l+1] = '\"';
    Str[l+2] = 0;
  }
  return Str;
}

static inline char *quote(int doQuote, char *Str)
{
  return doQuote ? QuoteText(Str) : Str;
}

static char* makeCmdLine(char *buffer, const char *mask, const char *fn, const char macro = '=')
{
  static char n1[NM], drv[NM], dir1[NM], fil1[NM], ext1[NM];
  static char n2[NM], tmp[NM], dir2[NM], fil2[NM], ext2[NM];
  char *i, *j, *e1 = ext1, *e2 = ext2, *smartQuote = NULL;
  fExpand(strcpy(tmp, fn));
  GetFullPathName(tmp, sizeof n1, n1, &j);
  fnSplit(n1, drv, dir1, fil1, ext1);
  GetShortPathName(tmp, n2, sizeof n2);
  fnSplit(n2, drv, dir2, fil2, ext2);
  strcat(strcat(strcat(strcpy(n1, drv), dir1), fil1), ext1);
  strcat(strcat(strcat(strcpy(n2, drv), dir2), fil2), ext2);
  if ( *e1 == '.' )
    e1++;
  if ( *e2 == '.' )
    e2++;
  i = (char*)mask;
  *(j = buffer) = 0;
  while ( *i )
  {
    int dosName = 0, noDrive = 0, noBackslash = 0, doQuote = 1;
    char *m = strchr(i, macro);
    if ( m )
    {
      *m = 0;
      strcat(j, i);
      j = strchr(j, 0);
      *m = macro;
      if ( *++m == macro )
      {
        j[0] = macro;
        j[1] = 0;
        j = strchr(j, 0);
        i = m+1;
      }
      else
      {
        int modifyer = 1;
        while ( modifyer )
        {
          switch ( *m )
          {
            case 'd':
              dosName = 1;
              break;
            case 'm':
              noDrive = 1;
              break;
            case 't':
              noBackslash = 1;
              break;
            case 'o':
              doQuote = 0;
              break;
            default:
              m--;
              modifyer = 0;
              break;
          }
          m++;
        }
        switch ( *m )
        {
          case 'M':
            strcat(j, drv);
            break;
          case 'N':
            strcat(j, quote(doQuote, strcpy(tmp, dosName ? fil2 : fil1)));
            break;
          case 'E':
            strcat(j, quote(doQuote, strcpy(tmp, dosName ? e2 : e1)));
            break;
          case 'F':
            strcpy(tmp, dosName ? fil2 : fil1);
            if ( *(dosName ? e2 : e1) )
            {
              strcat(tmp, ".");
              strcat(tmp, dosName ? e2 : e1);
            }
            strcat(j, quote(doQuote, tmp));
            break;
          case 'D':
            if ( noDrive )
              *tmp = 0;
            else
              strcpy(tmp, drv);
            strcat(tmp, dosName ? dir2 : dir1);
            if ( noBackslash )
              delBackslash(tmp);
            strcat(j, quote(doQuote, tmp));
            break;
          case 'P':
            if ( noDrive )
            {
              strcpy(tmp, dosName ? dir2 : dir1);
              strcat(tmp, dosName ? fil2 : fil1);
              if ( *(dosName ? e2 : e1) )
              {
                strcat(tmp, ".");
                strcat(tmp, dosName ? e2 : e1);
              }
              strcat(j, quote(doQuote, tmp));
            }
            else
              strcat(j, quote(doQuote, strcpy(tmp, dosName ? n2 : n1)));
            break;
          case '\'':
            if ( smartQuote )
            {
              for ( char *p = smartQuote ; *p ; p++ )
                if ( *p == '\"' )
                  *p = '\'';
              quote(1, smartQuote);
              smartQuote = NULL;
            }
            else
              smartQuote = strchr(j, 0);
            break;
        }
        i = m+1;
        j = strchr(j, 0);
      }
    }
    else
      break;
  }
  strcat(j, i);
  return buffer;
}

static struct FarMenuItemEx *compilerOut = NULL;
static TCollection *errColl = NULL;
static int compilerOutN = 0;

void WINAPI _export ExitFAR()
{
  DoneMacro();
  if ( eList )
  {
    eList->done();
    delete eList;
  }
  if ( compilerOut )
  {
    delete [] compilerOut;
    compilerOut = NULL;
  }
  if ( errColl )
  {
    errColl->done();
    delete errColl;
  }
}

struct TErrData
{
  char fn[NM];
  int line, col, msgCount;
  char message[64][64];
};

static char cmd[MAX_STR_LEN];

static bool showErrorMsgBox(int res)
{
  int n = (int)compilerOut[res].UserData-1;
  if ( errColl && ( n >= 0 ) && ( n < (int)errColl->getCount() ) )
  {
    TErrData *err = (TErrData*)((*errColl)[n]);
    const char *MsgItems[] =
    {
      cmd, err->fn, "\x01",
      err->message[ 0], err->message[ 1], err->message[ 2], err->message[ 3], err->message[ 4],
      err->message[ 5], err->message[ 6], err->message[ 7], err->message[ 8], err->message[ 9],
      err->message[10], err->message[11], err->message[12], err->message[13], err->message[14],
      err->message[15], err->message[16], err->message[17], err->message[18], err->message[19]
    };
    redraw();
    Info.Message(Info.ModuleNumber, FMSG_MB_OK|FMSG_LEFTALIGN, NULL, MsgItems, err->msgCount+3, 0);
    return true;
  }
  return false;
}

static void jumpToError(int aj, bool showMsgBox)
{
  if ( compilerOut && ( aj >= 0 ) && ( aj <= compilerOutN ) )
  {
    int n = (int)compilerOut[aj].UserData-1;
    if ( errColl && ( n >= 0 ) && ( n < (int)errColl->getCount() ) )
    {
      TErrData *err = (TErrData*)((*errColl)[n]);
      int nLine = err->line <= 0 ? -1 : err->line-1;
      struct WindowInfo wi;
      wi.Pos = -1;
      Info.AdvControl(Info.ModuleNumber, ACTL_GETWINDOWINFO, (void*)&wi);
      if ( err->fn[0] && !FSF.LStricmp(wi.Name, err->fn) )
        EditorSetPos(nLine, err->col);
      else
      {
        bool found = false;
        int n = Info.AdvControl(Info.ModuleNumber, ACTL_GETWINDOWCOUNT, NULL);
        for ( int i = 0 ; i < n ; i++ )
        {
          wi.Pos = i;
          Info.AdvControl(Info.ModuleNumber, ACTL_GETWINDOWINFO, (void*)&wi);
          if ( wi.Type == WTYPE_EDITOR && !FSF.LStricmp(wi.Name, err->fn) )
          {
            Info.AdvControl(Info.ModuleNumber, ACTL_SETCURRENTWINDOW, (void*)i);
            Info.AdvControl(Info.ModuleNumber, ACTL_COMMIT, NULL);
            EditorSetPos(nLine, err->col);
            found = true;
            break;
          }
        }
        if ( !found )
          Info.Editor(err->fn, NULL, 0, 0, -1, -1, EF_NONMODAL|EF_ENABLE_F6, err->line, err->col);
      }
      if ( showMsgBox )
        showErrorMsgBox(aj);
    }
  }
}

static void showCompileOut(void)
{
  if ( compilerOut )
  {
    int Code, BreakKeys[] = { VK_RETURN, VK_F4, VK_F3, 0 };
    for ( ; ; )
    {
      int res = Info.Menu(Info.ModuleNumber, -1, -1, 0,
        FMENU_WRAPMODE|FMENU_SHOWAMPERSAND|FMENU_USEEXT,
        cmd, GetMsg(MMenuBottom), NULL, BreakKeys, &Code,
        (const struct FarMenuItem *)compilerOut, compilerOutN);
      if ( res < 0 )
        break;
      if ( Code == 2 )
      {
        if ( res <= compilerOutN && showErrorMsgBox(res) )
        {
          for ( int i = 0 ; i < compilerOutN ; i++ )
            compilerOut[i].Flags &= ~MIF_SELECTED;
          compilerOut[res].Flags |= MIF_SELECTED;
        }
      }
      else
      {
        jumpToError(res, false);
        break;
      }
    }
  }
}

static inline void execute(TCollection *coll)
{
  HANDLE hScreen = Info.SaveScreen(0,0,-1,-1);
  ExecConsoleApp(cmd, coll, true);
  Info.Control(NULL, FCTL_SETUSERSCREEN, NULL);
  Info.RestoreScreen(hScreen);
  redraw();
}

static void SaveAll()
{
  struct WindowInfo wi;
  wi.Pos = -1;
  Info.AdvControl(Info.ModuleNumber, ACTL_GETWINDOWINFO, (void*)&wi);
  int home = wi.Pos;
  int n = Info.AdvControl(Info.ModuleNumber, ACTL_GETWINDOWCOUNT, NULL);
  for ( int i = 0 ; i < n ; i++ )
  {
    wi.Pos = i;
    Info.AdvControl(Info.ModuleNumber, ACTL_GETWINDOWINFO, (void*)&wi);
    if ( wi.Type == WTYPE_EDITOR && wi.Modified )
    {
      Info.AdvControl(Info.ModuleNumber, ACTL_SETCURRENTWINDOW, (void*)i);
      Info.AdvControl(Info.ModuleNumber, ACTL_COMMIT, NULL);
      Info.EditorControl(ECTL_SAVEFILE, NULL);
    }
  }
  Info.AdvControl(Info.ModuleNumber, ACTL_SETCURRENTWINDOW, (void*)home);
  Info.AdvControl(Info.ModuleNumber, ACTL_COMMIT, NULL);
}

static bool isFile(const char *s)
{
  WIN32_FIND_DATA ff;
  HANDLE h = FindFirstFile(s, &ff);
  bool res = h != INVALID_HANDLE_VALUE;
  FindClose(h);
  return res;
}

static bool findBaseFile(const char *file, char *baseFile)
{
  int n = Info.AdvControl(Info.ModuleNumber, ACTL_GETWINDOWCOUNT, NULL);
  bool found = false;
  for ( int i = 0 ; i < n ; i++ )
  {
    struct WindowInfo wi;
    wi.Pos = i;
    Info.AdvControl(Info.ModuleNumber, ACTL_GETWINDOWINFO, (void*)&wi);
    if ( wi.Type == WTYPE_EDITOR )
    {
      char testFile[NM];
      char *p = (char*)strrchr(strcpy(testFile, wi.Name), '\\');
      if ( p )
      {
        lstrcpy(p+1, FSF.PointToName(file));
        if ( isFile(testFile) )
        {
          if ( baseFile )
            lstrcpy(baseFile, testFile);
          found = true;
          break;
        }
      }
    }
  }
  return found;
}

static bool validMenuItem(const char *fn, TExec *e)
{
  char baseFile[NM] = "";
  bool enable = true;
  if ( e->enable[0] )
  {
    makeCmdLine(baseFile, e->enable, fn);
    enable = isFile(baseFile);
    if ( !enable && e->searchBase )
      enable = findBaseFile(baseFile, NULL);
  }
  if ( enable && e->disable[0] )
    enable = !isFile(makeCmdLine(baseFile, e->disable, fn));
  return enable;
}

static char *makeTitle(const char *line, size_t size, const char *fn)
{
  static char temp[MAX_STR_LEN];
  makeCmdLine(temp, line, fn);
  FSF.TruncStr(temp, size);
  return temp;
}

static bool parseError(TLang *lng, const char *compiler, const char *fn, char *line, unsigned ci, int& aj)
{
  TCompiler *e = NULL;
  bool found = false, res = false;
  for ( unsigned i = 0 ; i < lng->compilerColl.getCount() ; i++ )
  {
    e = (TCompiler*)(lng->compilerColl[i]);
    if ( !stricmp(compiler, e->title) )
    {
      found = true;
      break;
    }
  }
  if ( found && e && e->err[0] )
  {
    int lineBounds[2] = { 0, 0 }, colBounds[2] = { 0, 0 }, fileBounds[2] = { 0, 0 };
    if ( strMatch(line, e->err, "/", ".*/i", lineBounds, e->lineMatch, colBounds, e->colMatch, fileBounds, e->fileMatch) )
    {
      int len, lineNo = -1, colNo = -1;
      compilerOut[ci].Flags = 0;
      char fileName[NM];
      if ( e->lineMatch >= 0 )
      {
        strncpy(fileName, line+lineBounds[0], len = lineBounds[1]-lineBounds[0]);
        fileName[len] = 0;
        if ( len )
          lineNo = FSF.atoi(fileName);
      }
      if ( e->colMatch >= 0 )
      {
        strncpy(fileName, line+colBounds[0], len = colBounds[1]-colBounds[0]);
        fileName[len] = 0;
        if ( len )
          colNo = FSF.atoi(fileName);
      }
      if ( e->fileMatch >= 0 )
      {
        strncpy(fileName, line+fileBounds[0], len = fileBounds[1]-fileBounds[0]);
        fileName[len] = 0;
        if ( !len )
          strcpy(fileName, fn);
      }
      else
        strcpy(fileName, fn);
      TErrData *errData = new TErrData;
      fExpand(strcpy(errData->fn, fileName));
      char *p = line;
      errData->msgCount = 0;
      for ( ; ; )
      {
        char brk[] = " !%^&*)+|{}:>?`-=];,/";
        strncpy(errData->message[errData->msgCount], p, 64);
        errData->message[errData->msgCount][63] = 0;
        if ( strpbrk(errData->message[errData->msgCount], brk) )
        {
          char *pp = lstrchr(errData->message[errData->msgCount], 0);
          while ( !lstrchr(brk, *(--pp)) )
            ;
          pp[1] = 0;
        }
        p += lstrlen(errData->message[errData->msgCount]);
        FSF.Trim(errData->message[errData->msgCount]);
        if ( !errData->message[errData->msgCount][0] )
        {
          lstrcpy(errData->message[errData->msgCount], "\x01");
          errData->msgCount++;
          break;
        }
        errData->msgCount++;
      }
      errData->line = lineNo;
      errData->col = colNo;
      errColl->insert(errData);
      compilerOut[ci].UserData = errColl->getCount();
      if ( aj < 0 )
      {
        compilerOut[ci].Flags = MIF_SELECTED;
        aj = (int)ci;
      }
      res = true;
    }
  }
  return res;
}

static bool runCompiler(TLang *lng, const char *fn, const char *title, TExec *e)
{
  char cwd[NM] = "",  baseFile[NM] = "";
  if ( e->enable[0] )
  {
    char temp[NM];
    makeCmdLine(baseFile, e->enable, fn);
    bool enable = isFile(baseFile);
    if ( !enable && e->searchBase && findBaseFile(baseFile, temp) )
      lstrcpy(baseFile, temp);
  }
  makeCmdLine(cmd, e->cmd, fn);
  if ( e->ask && !Info.InputBox(NULL, makeTitle(title, 70, fn), "TrueTpl-ASK", cmd, cmd, sizeof(cmd), NULL, FIB_EXPANDENV|FIB_BUTTONS) )
    return false;
  FSF.ExpandEnvironmentStr(cmd, cmd, sizeof(cmd));
  if ( e->saveType != saNone )
  {
    if ( e->saveType == saCurrent )
      Info.EditorControl(ECTL_SAVEFILE, NULL);
    else
      SaveAll();
  }
  if ( e->cd != cdNone )
  {
    char newDir[NM] = "", *p;
    if ( e->cd == cdFile )
      p = (char*)strrchr(lstrcpy(newDir, fn), '\\');
    else if ( ( e->cd == cdBase ) && *baseFile )
      p = (char*)strrchr(lstrcpy(newDir, baseFile), '\\');
    if ( p )
      *p = 0;
    if ( *newDir )
    {
      GetCurrentDirectory(sizeof(cwd), cwd);
      SetCurrentDirectory(newDir);
    }
  }
  if ( errColl )
    errColl->removeAll();
  else
    errColl = new TCollection;
  TCollection compilerColl;
  execute(&compilerColl);
  unsigned n = compilerColl.getCount();
  if ( n )
  {
    int nErr = 0, aj = -1;
    compilerOut = new FarMenuItemEx[compilerOutN = n+1];
    for ( unsigned i = 0 ; i < n ; i++ )
    {
      char *line = (char*)(compilerColl[i]);
      strncpy(compilerOut[i].Text.Text, line, sizeof(compilerOut[i].Text.Text));
      compilerOut[i].Text.Text[sizeof(compilerOut[i].Text.Text)-1] = 0;
      compilerOut[i].Flags = MIF_DISABLE;
      compilerOut[i].UserData = 0;
      char compiler[MAX_STR_LEN];
      strcpy(compiler, e->compiler);
      char *pp, *p = compiler;
      bool found = false;
      while ( ( pp = lstrchr(p, ',') ) != NULL )
      {
        *pp = 0;
        if ( parseError(lng, p, fn, line, i, aj) )
        {
          nErr++;
          found = true;
        }
        p = pp+1;
        if ( found )
          break;
      }
      if ( !found && parseError(lng, p, fn, line, i, aj) )
        nErr++;
    }
    if ( ( aj >= 0 ) && ( ( e->jumpType == jtFirst ) ) || ( ( e->jumpType == jtSmart ) && ( nErr == 1 ) ) )
      jumpToError(aj, true);
    else
    {
      if ( aj < 0 )
      {
        compilerOut[n].Text.Text[0] = 0;
        compilerOut[n].Flags = MIF_SELECTED;
      }
      else
        compilerOutN--;
      if ( ( e->jumpType == jtMenu ) || ( e->jumpType == jtSmart ) )
        showCompileOut();
    }
  }
  else if ( compilerOut )
  {
    delete [] compilerOut;
    compilerOut = NULL;
  }
  compilerColl.done();
  if ( *cwd )
    SetCurrentDirectory(cwd);
  return true;
}

static void SelectTemplate(TEInfo *te)
{
  unsigned lc = langColl.getCount();
  FarMenuItem *amenu = new FarMenuItem[lc+2];
  if ( amenu )
  {
    for ( unsigned i = 0 ; i < lc ; i++ )
    {
      FSF.sprintf(amenu[i].Text, "%s", ((TLang*)langColl[i])->desc);
      amenu[i].Selected = (unsigned)(te->lang) == i;
      amenu[i].Checked = amenu[i].Separator = false;
    }
    amenu[lc].Selected = amenu[lc].Checked = false;
    amenu[lc].Separator = true;
    *amenu[lc].Text = 0;
    FSF.sprintf(amenu[lc+1].Text, "%s", GetMsg(MOtherLang));
    amenu[lc+1].Selected = te->lang == -1;
    amenu[lc+1].Checked = amenu[lc+1].Separator = false;
    int res = Info.Menu(Info.ModuleNumber,-1,-1,0,FMENU_WRAPMODE|FMENU_AUTOHIGHLIGHT,GetMsg(MSelectLang),NULL,NULL,NULL,NULL,amenu,lc+2);
    if ( res != -1 )
    {
      if ( pluginStop )
        SetRegKey(HKEY_CURRENT_USER, "", "Disable", pluginStop = 0);
      te->lang = ( (unsigned)res < lc ) ? res : -1;
    }
    delete [] amenu;
  }
}

static bool sep = true;

static void addToCompilerMenu(const char *line, FarMenuItemEx *amenu, int& i, DWORD j, const char *fn, DWORD AccelKey)
{
  FSF.sprintf(amenu[i].Text.Text, "%s", makeTitle(line, sizeof(amenu[i].Text.Text)-1, fn));
  amenu[i].UserData = j;
  amenu[i].Flags = 0;
  amenu[i].AccelKey = AccelKey;
  if ( amenu[i].Text.Text[0] )
  {
    i++;
    sep = false;
  }
  else
  {
    if ( !sep )
    {
      amenu[i].Flags = MIF_SEPARATOR;
      i++;
    }
    sep = true;
  }
}

static void SelectCompiler(TEInfo *te)
{
  struct EditorInfo ei;
  Info.EditorControl(ECTL_GETINFO, &ei);
  char top[NM];
  FSF.TruncPathStr(lstrcpy(top, ei.FileName), 40);
  TLang *lng = (TLang*)(langColl[te->lang]);
  unsigned ec = 0;
  if ( lng )
    ec = lng->execColl.getCount();
  FarMenuItemEx *amenu = new FarMenuItemEx[ec+2];
  if ( amenu )
  {
    const DWORD scId = -1;
    int i = 0;
    sep = true;
    if ( ec )
    {
      for ( unsigned j = 0 ; j < ec ; j++ )
      {
        TExec *ex = ((TExec*)lng->execColl[j]);
        if ( validMenuItem(ei.FileName, ex) )
          addToCompilerMenu(ex->title, amenu, i, j, ei.FileName, 0);
      }
      addToCompilerMenu("", amenu, i, 0, ei.FileName, 0);
    }
    addToCompilerMenu(GetMsg(MShowOutput), amenu, i, scId, ei.FileName, KEY_MULTIPLY);
    if ( !compilerOut )
      amenu[i-1].Flags = MIF_DISABLE;
    for ( int k = 0 ; k < i ; k++ )
      if ( ! ( amenu[k].Flags & ( MIF_DISABLE|MIF_SEPARATOR ) ) )
      {
        amenu[k].Flags |= MIF_SELECTED;
        break;
      }
    int res = Info.Menu(Info.ModuleNumber,-1,-1,0,FMENU_WRAPMODE|FMENU_AUTOHIGHLIGHT|FMENU_USEEXT,top,NULL,NULL,NULL,NULL,(const struct FarMenuItem *)amenu, i);
    if ( res != -1 )
    {
      DWORD id = amenu[res].UserData;
      if ( id == scId )
        showCompileOut();
      else if ( lng )
      {
        TExec *e = (TExec*)lng->execColl[id];
        runCompiler(lng, ei.FileName, e->title, e);
      }
    }
    delete [] amenu;
  }
}

HANDLE WINAPI _export OpenPlugin(int, int)
{
  if ( !IsOldFar )
  {
    EditorInfo ei;
    Info.EditorControl(ECTL_GETINFO, &ei);
    initEList();
    int n = eList->findID(ei.EditorID);
    if ( n == -1 )
      n = eList->insert(ei.EditorID, findLngID(ei.FileName), ei.FileName);
    if ( n != -1 )
    {
      TEInfo *te = (TEInfo*)(*eList)[n];
      if ( te )
      {
        struct FarMenuItem mMenu[2];
        int count = sizeof(mMenu)/sizeof(*mMenu);
        for ( int i = 0 ; i < count ; i++ )
          mMenu[i].Selected = mMenu[i].Checked = mMenu[i].Separator = false;
        strcpy(mMenu[0].Text, GetMsg(MMenuSelectLang));
        strcpy(mMenu[1].Text, GetMsg(MMenuCompile));
        int mode = Info.Menu(Info.ModuleNumber,-1,-1,0,FMENU_WRAPMODE,GetMsg(MMenuString),NULL,NULL,NULL,NULL,mMenu,count);
        if ( mode == 0 )
          SelectTemplate(te);
        else if ( mode == 1 )
          SelectCompiler(te);
      }
    }
  }
  return INVALID_HANDLE_VALUE;
}
