// SIS File Reader and unpacker

#include <stdio.h>
#include <memory.h>
#include <string.h>
#include <malloc.h>
#include "sisread.h"

#define QUIET(code) { if (!pcOptions['q']) { code; } }
//#define QUIET(code) code

char pcOptions[128];
FILE *SisFile;
struct SisFileStruct *psSisHead;
int argc;
char **argv;

void show_help(void)
{
	printf(" usage : SISREAD <cmd> archive [files]\n");
	printf(" examples : sisread e foo EBook.app license.txt, sisread l foo\n");
	printf("\n commands :\n");
	printf("   e[qy] - Extract files\n");
	printf("   l[s]  - List files\n");
	printf("\n switches :\n");
	printf("   q - Quiet operation\n");
	printf("   y - assume Yes on all questions\n");
	printf("   s - Short listing\n");
	printf("\nNote: wildcards are not supported yet.\n");
}

int add(void)
{
	printf("Adding files to SIS pack currently is not supported.\n");
	return 1;
}

char *get_file_name(char *pszPath)
{
	size_t l = strlen(pszPath);
	while (l > 0 && pszPath[l] != '\\') l--;
	return pszPath + l + 1;
}

int get_file_number(char *pszFileName)
{
	int nFile;
	for (nFile = psSisHead->sHeader.wFilesCount - 1; nFile >= 0; nFile--)
	{
		if (!strlen(psSisHead->psFiles[nFile].pszDestFile))
			continue;
		if (_stricmp(get_file_name(psSisHead->psFiles[nFile].pszDestFile), pszFileName) == 0)
			break;
	}
	return nFile;
}

int extract(BOOL bFullPath)
{
	int i, nFile;
	size_t nReadSize;
	FILE *DestFile;
	void *pBuf;
	DWORD dwFileLength;
	char cRes, nSkipMode = pcOptions['q'] ? 1 : 0; // 0 - no skip, 1 - ovr all, 2 - skip all

	bFullPath |= pcOptions['d'];

	for (i = 3; i< argc; i++)
	{
		nFile = get_file_number(argv[i]);
		if (nFile == -1)
			QUIET(printf("Cannot find file %s\n", argv[i]))
		else
		{
			QUIET(printf("Unpacking %s ... ", get_file_name(psSisHead->psFiles[nFile].pszDestFile)))
			
			if (nSkipMode != 1)
			{
				DestFile = fopen(argv[i], "r");
				if (DestFile)
				{
					fclose(DestFile);
					if (nSkipMode == 1)
						continue;
					do {
						fflush(stdin);
						printf("File already exists. Overwrite (y/a/n/s)? ");
						cRes = _fgetchar() | 0x20;
					} while (!(cRes == 'y' || cRes == 'a' || cRes == 'n' || cRes == 's'));
					if (cRes == 's')
						nSkipMode = 2;
					if (cRes == 'a')
						nSkipMode = 1;
					if (cRes != 'y' && cRes != 'a')
						continue;
				}
			}

			DestFile = fopen(argv[i], "wb");
			if (!DestFile)
			{
				QUIET(printf("Error opening file for writing.\n"));
				continue;
			}
			dwFileLength = psSisHead->psFiles[nFile].pdwFileLengths[0];
			pBuf = malloc(dwFileLength);
			if (pBuf == NULL)
			{
				QUIET(printf("Error allocationg %d bytes.\n", dwFileLength));
				continue;
			}
			fseek(SisFile, psSisHead->psFiles[nFile].pfpFiles[0], SEEK_SET);
			nReadSize = fread(pBuf, dwFileLength, 1, SisFile);
			if (!nReadSize)
			{
				QUIET(printf("Error reading archive.\n"));
				continue;
			}
			nReadSize = fwrite(pBuf, dwFileLength, 1, DestFile);
			if (!nReadSize)
			{
				QUIET(printf("Error writing output file.\n"));
				continue;
			}
			fclose(DestFile);
			QUIET(printf("OK\n"));
		}
	}

	return 0;
}

int list(void)
{
	int nFile;
	char pszTemp[12];

	if (!pcOptions['s'])
	{
		printf("%-50s %9s %5s %-5s\n", "destination file name", "length", "multi", "type");
		for (nFile = 0; nFile < 50+9+5+5+2; nFile++)
			printf("-");
		printf("-\n");
	}
	for (nFile = 0; nFile < psSisHead->sHeader.wFilesCount; nFile++)
		if (!pcOptions['s'])
		{
			switch (psSisHead->psFiles[nFile].dwType)
			{
			case 0: sprintf(pszTemp, "FF"); break;
			case 1: sprintf(pszTemp, "FT:T%c", psSisHead->psFiles[nFile].dwDetails == 0 ? 'C' :
				psSisHead->psFiles[nFile].dwDetails == 1 ? 'S' : 'A'); break;
			case 2: sprintf(pszTemp, "FC"); break;
			case 3: sprintf(pszTemp, "FR:R%c", psSisHead->psFiles[nFile].dwDetails == 0 ? 'I' :
				psSisHead->psFiles[nFile].dwDetails == 1 ? 'R' : 'B'); break;
			case 4: sprintf(pszTemp, "FN"); break;
			default: sprintf(pszTemp, "??");
			}
			printf("%-50s %9d %5d %-5s\n", psSisHead->psFiles[nFile].pszDestFile,
				psSisHead->psFiles[nFile].pdwFileLengths[0],
				psSisHead->psFiles[nFile].dwFlags,
				pszTemp);
		}
		else
			printf("%s\n", get_file_name(psSisHead->psFiles[nFile].pszDestFile));

	return 0;
}

int main(int _argc, char *_argv[])
{
	char cCmd, cOption;
	char *pszArcName;
	int nResult = 1, i;

	QUIET(printf("SIS File Reader 0.8 Copyright (c) 2003 Zverik\n\n"));

	argc = _argc;
	argv = _argv;
	
	if (argc < 3)
	{
		show_help();
		return 1;
	}

	cCmd = argv[1][0] | 0x20;
	pszArcName = argv[2];
	memset(pcOptions, 0, sizeof(pcOptions));
	for (i = 1; cOption = argv[1][i] & ~0x80; i++)
	{
		cOption |= 0x20;
		pcOptions[cOption] |= cOption ? 1 : 0;
	}

	if ((SisFile = fopen(pszArcName, "rb")) == NULL)
	{
		QUIET(printf("Cannot open file %s.\nNote that creating or modifying install packs is not possible for now.\n", pszArcName));
		return 2;
	}
	if ((psSisHead = sis_read_header(SisFile)) == NULL)
	{
		QUIET(printf("Cannot read header. File is corrupt.\n"));
		return 2;
	}

	switch (cCmd)
	{
	case 'a': nResult = add(); break;
	case 'e': nResult = extract(FALSE); break;
	case 'x': nResult = extract(TRUE); break;
	case 'l': nResult = list(); break;
	}

    sis_dispose_header(psSisHead);
    if (fclose(SisFile))
		QUIET(printf("Error closing file %s, error = %d\n", pszArcName, ferror(SisFile)));

	return nResult;
}