// SIS file structures
//
// Made by Zverik (zverik@rbcmail.ru)
//
// With the great help of SIS file format documentation from Alexander Thoukydides
// http://www.thouky.co.uk/software/psifs/sis.html
//
// NB: Aligning must be set to 4 bytes. Probably. It seems there must be no issue
//     with this, but just in case.
//
// Used notation:
//      wXXX - word parameter (2 bytes)
//     dwXXX - double word (4 bytes)
//     fpXXX - dword file pointer (4 bytes, specifies offset in the SIS file)
//      sXXX - structure
//     pwXXX,
//    pdwXXX,
//     psXXX - pointers to respective types
//    pszXXX - pointer to zero-terminated string. Like char *pszXXX
//   ppszXXX - array of pszXXX
//

#ifndef _SIS_H
#define _SIS_H

#include <stdio.h>

//#define SIS_DEBUG

#ifndef WORD
#define WORD unsigned short
#endif
#ifndef DWORD
#define DWORD unsigned long
#endif

#define FILEPTR DWORD

// SIS header structure

#define SIS_UID1_NOAPP 0x10000000
#define SIS_UID2 0x1000006D
#define SIS_UID3 0x10000419

enum SisInstallerVersion {
	SIS_ER345	= 0x00000044,
	SIS_ER6		= 0x000000C8
};

enum SisOptions {
	SIS_UNICODE			= 0x0001,	// ER5 & ER6
	SIS_DISTRIBUTABLE	= 0x0002,	// ER5 & ER6
	SIS_NOCOMPRESS		= 0x0008,	// ER6
	SIS_SHUTDOWNAPPS	= 0x0010	// ER6
};

enum SisTypes {
	SIS_APP,
	SIS_SYSTEM,	// ER6
	SIS_OPTION,	// ER6
	SIS_CONFIG,	// ER6
	SIS_PATCH,	// ER6
	SIS_UPGRADE	// ER6
};

struct SisHeader {
	DWORD	dwUID1;		// UID of application to be installed
	DWORD	dwUID2;		// UID2 and UID3 must be set to SIS_UID2 and SIS_UID3
	DWORD	dwUID3;
	DWORD	dwUID4;		// Strange checksum of preceding fields
	WORD	wCheckSum;
	WORD	wLanguagesCount;
	WORD	wFilesCount;
	WORD	wRequisitesCount;
	WORD	wInstallLanguage;
	WORD	wInstallFiles;
	DWORD	dwInstallDrive;
	DWORD	dwInstallerVer;
	WORD	wOptions;
	WORD	wType;
	WORD	wVersionMajor;
	WORD	wVersionMinor;
	DWORD	wVariant;
	FILEPTR	fpLanguages;
    FILEPTR	fpFiles;
	FILEPTR fpRequisites;
	DWORD	dwUnknown;
	FILEPTR	fpComponentName;
};

// Languages

enum SisLanguages {
	SIS_LNG_TEST,	// 0
	SIS_LNG_EN,
	SIS_LNG_FR,
	SIS_LNG_GE,
	SIS_LNG_SP,
	SIS_LNG_IT,
	SIS_LNG_SW,
	SIS_LNG_DA,
	SIS_LNG_NO,
	SIS_LNG_FI,
	SIS_LNG_AM,		// 10
	SIS_LNG_SF,
	SIS_LNG_SG,
	SIS_LNG_PO,
	SIS_LNG_TU,
	SIS_LNG_IC,
	SIS_LNG_RU,
	SIS_LNG_HU,
	SIS_LNG_DU,
	SIS_LNG_BL,
	SIS_LNG_AU,		// 20
	SIS_LNG_BG,
	SIS_LNG_AS,
	SIS_LNG_NZ,
	SIS_LNG_IF,
	SIS_LNG_CS,
	SIS_LNG_SK,
	SIS_LNG_PL,
	SIS_LNG_SL,
	SIS_LNG_TC,
	SIS_LNG_HK,		// 30
	SIS_LNG_ZH,
	SIS_LNG_JA,
	SIS_LNG_TH
};

// File structure

enum SisFileFlags {
	SIS_MULTILANG = 0x00000001
};

enum SisFileTypes {
	SIS_FILE_STANDART,
	SIS_FILE_TEXT,
	SIS_FILE_COMPONENT,
	SIS_FILE_RUN,
	SIS_FILE_NOTEXIST
};

enum SisTextButtons {
	SIS_TEXT_CONTINUE,
	SIS_TEXT_SKIP,
	SIS_TEXT_ABORT
};

enum SisRunMoment {
	SIS_RUN_INSTALLATION,
	SIS_RUN_REMOVAL,
	SIS_RUN_BOTH
};

struct SisFile {
	DWORD	dwFlags;
	DWORD	dwType;
	DWORD	dwDetails;
	DWORD	dwSrcNameLength;
	FILEPTR	fpSrcName;
	DWORD	dwDestNameLength;
	FILEPTR	fpDestName;
	// NB: File offset are cut out. You will need to read them manually as
	// it is described in the documentation: firstly N DWORD are situated
	// that are file lengths for all languages specified in header (N), then
	// N DWORDs with file pointers to file contents.
	// As an alternative, use sis_read_header :)
};

// Requisite structure

struct SisRequisite {
	DWORD	dwUID;
	WORD	wVersionMajor;
	WORD	wVersionMinor;
	DWORD	dwVariant;
	// NB: Requisite names are cut out. Refer to comment in struct SisFile.
};

// Component record structure

// NB: Component names are cut out. Refer to comment in struct SisFile.

// :)))))))

// And now... The Great Structure That Represents Almost The Whole File!
// It can be read with SisFileStruct *sis_read_header(FILE *SisFile) function
// When done, this whole structure can be disposed by void sis_dispose_header(SisFileStruct *sFile).

struct SisFSFile {
	// Yes, it's just copied from there. Not very optimal solution, though.
	DWORD	dwFlags;
	DWORD	dwType;
	DWORD	dwDetails;
	DWORD	dwSrcNameLength;
	FILEPTR	fpSrcName;
	DWORD	dwDestNameLength;
	FILEPTR	fpDestName;

	char	*pszSrcFile;
	char	*pszDestFile;
	WORD	wFilesCount;	// == (dwFlags & SIS_MULTILANG) ? SisHeader.wLanguagesCount : 1;
	DWORD	*pdwFileLengths;
	FILEPTR	*pfpFiles;
};

struct SisFSRequisite {
	DWORD	dwUID;
	WORD	wVersionMajor;
	WORD	wVersionMinor;
	DWORD	dwVariant;

	char	**ppszRequisiteNames;
};

struct SisFileStruct {
	struct	SisHeader		sHeader;
	WORD					*pwLanguages;	// sizeof(enum) == 4 in 32bit compilers, so we have to specify WORD type here.
	struct	SisFSFile		*psFiles;		// array [sHeader.wFilesCount] of SisFSFile
	struct	SisFSRequisite	*psRequisites;
	char					**ppszComponentNames;
};

extern struct SisFileStruct *sis_read_header(FILE *SisFile);
extern void sis_dispose_header(struct SisFileStruct *sFile);

#endif // _SIS_H
