// SIS file handling functions
//
// Made by Zverik (zverik@rbcmail.ru)
//
// Those functions are necessary only if you do not wish to read SIS headers by youself :)
//

#include "sis.h"
#include <stdio.h>
#include <memory.h>
#include <malloc.h>

// A little code-saving routine :)  If (nError == 0) than crash-exit.
#define CHECK_ZERO(nError) if (!(nError)) { sis_dispose_header(psFS); return NULL; }
#define CHECK_NULL(nError) if ((nError) == NULL) { sis_dispose_header(psFS); return NULL; }
// This macro won't be so cruel - disposes what it's told to.
#define CHECK_ZERO_FUNC(pBuf, nError) if (nError) { free(pBuf); return NULL; }
// And this one is for the most unhappy case...
#define DO_IF_ZERO(nError, code) if (!nError) nError = (int)(code);
#ifdef SIS_DEBUG
#define DEBUG_CODE(code) code
#else
#define DEBUG_CODE(code)
#endif

// fread stub
size_t __cdecl sis_fread (void *buffer, size_t size, size_t num, FILE *stream)
{
	size_t result;
	long old_ftell = ftell(stream);
	DEBUG_CODE(printf("fread: buffer = %p, size = %d*%d, file pos = %d\n", buffer, size, num, old_ftell));
	result = fread(buffer, size, num, stream);
	old_ftell = ftell(stream) - old_ftell;
	DEBUG_CODE(printf(" result = %d, ferror = %d, file pos = %d (%c%d)\n", result, ferror(stream), ftell(stream), old_ftell < 0 ? '-' : '+', old_ftell));
	return result;
}
int __cdecl sis_fseek (FILE *stream, long offset, int mode)
{
	int result;
	printf("fseek: %d -> %d\n", ftell(stream), offset);
	result = fseek(stream, offset, mode);
	printf(" result = %d, file pos = %d\n", result, ftell(stream));
	return result;
}
void * __cdecl sis_malloc(size_t size)
{
	void *result;
	printf("malloc: size = %d", size);
	result = malloc(size);
	printf(" result = %p\n", result);
	return result;
}

#ifdef SIS_DEBUG
#define fread sis_fread
#define fseek sis_fseek
#define malloc sis_malloc
#endif

// Private functions. Those all are used in sis_read_header.
char *sis_read_string(DWORD dwLength, FILEPTR fpOffset, FILE *SisFile)
{
	char *pszBuf;
	DEBUG_CODE(printf("\nread_string: dwLength = %d, fpOffset = %d, file pos = %d\n", dwLength, fpOffset, ftell(SisFile)));
	pszBuf = (char *)malloc(dwLength + 1);
	if (pszBuf == NULL) return NULL;
	if (dwLength)
	{
		if (fpOffset != (FILEPTR)ftell(SisFile))
			CHECK_ZERO_FUNC(pszBuf, fseek(SisFile, fpOffset, SEEK_SET));
		CHECK_ZERO_FUNC(pszBuf, fread(pszBuf, dwLength, 1, SisFile) == 0);
	}
	pszBuf[dwLength] = 0;
	DEBUG_CODE(printf("normal leave\n\n"));
	return pszBuf;
}

void *sis_read_structure(DWORD dwLength, FILEPTR fpOffset, FILE *SisFile)
{
	void *pszBuf;
	DEBUG_CODE(printf("\nread_structure: dwLength = %d, fpOffset = %d, file pos = %d\n", dwLength, fpOffset, ftell(SisFile)));
	pszBuf = (void *)malloc(dwLength);
	if (pszBuf == NULL) return NULL;
	if (fpOffset != (FILEPTR)ftell(SisFile))
		CHECK_ZERO_FUNC(pszBuf, fseek(SisFile, fpOffset, SEEK_SET));
	CHECK_ZERO_FUNC(pszBuf, fread(pszBuf, dwLength, 1, SisFile) == 0);
	DEBUG_CODE(printf("normal leave\n\n"));
	return pszBuf;
}

int sis_read_lenghts_offsets(DWORD **pdwLengths, FILEPTR **pfpOffsets, WORD wCount, FILEPTR fpOffset, FILE *SisFile)
{
	int nError = 0;
	DEBUG_CODE(printf("\nread_lengths_offsets: wCount = %d, fpOffset = %d, file pos = %d\n", wCount, fpOffset, ftell(SisFile)));
	DO_IF_ZERO(nError, (*pdwLengths = (DWORD *)malloc(sizeof(DWORD) * wCount)) == NULL);
	DO_IF_ZERO(nError, (*pfpOffsets = (FILEPTR *)malloc(sizeof(FILEPTR) * wCount)) == NULL);
	if (fpOffset != (FILEPTR)ftell(SisFile))
		DO_IF_ZERO(nError, fseek(SisFile, fpOffset, SEEK_SET));
	DO_IF_ZERO(nError, fread(*pdwLengths, sizeof(DWORD), wCount, SisFile) == 0);
	DO_IF_ZERO(nError, fread(*pfpOffsets, sizeof(FILEPTR), wCount, SisFile) == 0);
	if (nError)
	{
		free(*pdwLengths);
		free(*pfpOffsets);
	}
	DEBUG_CODE(printf("leave, nError = %d\n\n", nError));
	return nError;
}

char **sis_read_strings(WORD wCount, FILEPTR fpOffset, FILE *SisFile)
{
	DWORD *pdwLengths = NULL;	// Otherwise we get warning 'local variable used without having been initialized'
	FILEPTR *pfpOffsets = NULL;
	char **ppszBuf = NULL;
	int i, nError;	// Seems that macro CHECK_ZERO_FUNC will not be enough
	
	DEBUG_CODE(printf("\nread_strings: wCount = %d, fpOffset = %d, file pos = %d\n", wCount, fpOffset, ftell(SisFile)));
	nError = sis_read_lenghts_offsets(&pdwLengths, &pfpOffsets, wCount, fpOffset, SisFile);
	DO_IF_ZERO(nError, (ppszBuf = (char **)calloc(sizeof(char *), wCount)) == NULL);
	if (!nError)
	{
		for (i = 0; i < wCount && !nError; i++)
			DO_IF_ZERO(nError, (ppszBuf[i] = sis_read_string(pdwLengths[i], pfpOffsets[i], SisFile)) == NULL);
		if (nError)
		{
			// items ppszBuf[0..i-1] were allocated, so we need to malloc them.
			for (--i; i >= 0; --i)
				free(ppszBuf[i]);
			free(ppszBuf);
			free(pdwLengths);
			free(pfpOffsets);
			DEBUG_CODE(printf("error leave, error allocating items\n\n"));
			return NULL;
		}
	}

	free(pdwLengths);
	free(pfpOffsets);
	DEBUG_CODE(printf("leave, ppszBuf = %p (size = %d)\n\n", ppszBuf, _msize(ppszBuf)));
	return ppszBuf;
}

// sis_read_header itself.
struct SisFileStruct *sis_read_header(FILE *SisFile)
{
	struct SisFileStruct *psFS;
	WORD nFile;
	long lStoredOffset;

	// Memory allocation for header
	CHECK_NULL(psFS = (struct SisFileStruct *)malloc(sizeof(struct SisFileStruct)))
	memset(psFS, 0, sizeof(struct SisFileStruct));

	// Reading header of SIS file
	CHECK_ZERO(fread(&psFS->sHeader, sizeof(struct SisHeader), 1, SisFile));

	// Reading languages array. Yeah, it's so simple now :)
	CHECK_NULL(psFS->pwLanguages = (WORD *)sis_read_structure(sizeof(WORD) * psFS->sHeader.wLanguagesCount, psFS->sHeader.fpLanguages, SisFile));

	if (psFS->sHeader.wFilesCount)
	{
		// Files list
		CHECK_NULL(psFS->psFiles = (struct SisFSFile *)calloc(sizeof(struct SisFSFile), psFS->sHeader.wFilesCount));
		CHECK_ZERO(fseek(SisFile, psFS->sHeader.fpFiles, SEEK_SET) == 0);
		for (nFile = 0; nFile < psFS->sHeader.wFilesCount; nFile++)
		{
			DEBUG_CODE(printf("\nReading info for file #%d\n", nFile));
			CHECK_ZERO(fread(&psFS->psFiles[nFile], sizeof(struct SisFile), 1, SisFile));
			CHECK_ZERO((lStoredOffset = ftell(SisFile)) != -1);
			CHECK_NULL(psFS->psFiles[nFile].pszSrcFile = sis_read_string(psFS->psFiles[nFile].dwSrcNameLength, psFS->psFiles[nFile].fpSrcName, SisFile));
			CHECK_NULL(psFS->psFiles[nFile].pszDestFile = sis_read_string(psFS->psFiles[nFile].dwDestNameLength, psFS->psFiles[nFile].fpDestName, SisFile));
			psFS->psFiles[nFile].wFilesCount = (psFS->psFiles[nFile].dwFlags & SIS_MULTILANG) ? psFS->sHeader.wLanguagesCount : 1;
			CHECK_ZERO(sis_read_lenghts_offsets(&psFS->psFiles[nFile].pdwFileLengths, &psFS->psFiles[nFile].pfpFiles, psFS->psFiles[nFile].wFilesCount, lStoredOffset, SisFile) == 0);
		}
	}

	if (psFS->sHeader.wRequisitesCount)
	{
		CHECK_NULL(psFS->psRequisites = (struct SisFSRequisite *)calloc(sizeof(struct SisFSRequisite), psFS->sHeader.wRequisitesCount));
		CHECK_ZERO(fseek(SisFile, psFS->sHeader.fpRequisites, SEEK_SET) == 0);
		for (nFile = 0; nFile < psFS->sHeader.wRequisitesCount; nFile++)
		{
			DEBUG_CODE(printf("\nReading info for requisite #%d\n", nFile));
			CHECK_ZERO(fread(&psFS->psRequisites[nFile], sizeof(struct SisRequisite), 1, SisFile));
			CHECK_ZERO((lStoredOffset = ftell(SisFile)) != -1);
			// Reading requisite names that are immediately after SisRequisite structure in file.
			CHECK_NULL(psFS->psRequisites[nFile].ppszRequisiteNames = sis_read_strings(psFS->sHeader.wLanguagesCount, psFS->sHeader.fpRequisites + sizeof(struct SisRequisite), SisFile));
			CHECK_ZERO(fseek(SisFile, lStoredOffset, SEEK_SET) == 0);
		}
	}

	// Reading component names - the final property, at last!
	CHECK_NULL(psFS->ppszComponentNames = sis_read_strings(psFS->sHeader.wLanguagesCount, psFS->sHeader.fpComponentName, SisFile));

	// Huh... I wonder if there is a case when all those checks will be passed :)
	return psFS;
}

// This function is used in str_dispose_header
void sis_dispose_strings(char **ppszBuf, WORD wCount)
{
	int i;
	if (ppszBuf)
	{
		for (i = 0; i < wCount; i++)
			free(ppszBuf[i]);
		free(ppszBuf);
	}
}

void sis_dispose_header(struct SisFileStruct *sFile)
{
	WORD nFile;

	if (sFile)
	{
		free(sFile->pwLanguages);
		if (sFile->psFiles)
		{
			for (nFile = 0; nFile < sFile->sHeader.wFilesCount; nFile++)
			{
				free(sFile->psFiles[nFile].pszSrcFile);
				free(sFile->psFiles[nFile].pszDestFile);
				free(sFile->psFiles[nFile].pdwFileLengths);
				free(sFile->psFiles[nFile].pfpFiles);
			}
			free(sFile->psFiles);
		}
		if (sFile->psRequisites)
		{
			for (nFile = 0; nFile < sFile->sHeader.wRequisitesCount; nFile++)
				sis_dispose_strings(sFile->psRequisites[nFile].ppszRequisiteNames, sFile->sHeader.wLanguagesCount);
			free(sFile->psRequisites);
		}
		sis_dispose_strings(sFile->ppszComponentNames, sFile->sHeader.wLanguagesCount);
	}
	free(sFile);
}

#undef fread
#undef fseek