{
 Program: SDS.PAS
  Author: Eric E. A. Schreiber
    Date: April 11, 1998
Function: Size of Directory Structure
}

program SDS;

uses Dos;

const
  HelpSet = ['?', '/', '-'];   { * Command line help characters      * }

  { *** Typed constants *** }
  iTotalDir    : Integer = 0;  { * Total directories accumulator     * }
  iTotalFiles  : Integer = 0;  { * Total files accumulator           * }
  lClusters    : LongInt = 0;  { * Clusters occupied by files        * }
  lTotalSize   : LongInt = 0;  { * Total cumulative size files + dir * }
  lDirSize     : LongInt = 0;  { * lBytesPerDir * iTotalDir          * }
  lFileSize    : LongInt = 0;  { * File size accumulator             * }
  lBytesPerDir : LongInt = 0;  { * Bytes per directory               * }
  lBytesAvail  : LongInt = 0;  { * Bytes available on drive          * }
  lBytesCons   : LongInt = 0;  { * Bytes consumed by files           * }
  btDriveCode  : Byte    = 0;  { * Drive number for DirectorySpace   * }

var
  sSizeDir      : String;      { * Directory specified by user * }
  sBytesCons,                  { * Comma string for BytesCons  * }
  sDirSize,                    { * Comma string for DirSize    * }
  sBytesAvail,                 { * Comma string for BytesAvail * }
  sFileSize,                   { * Comma string for FileSize   * }
  sTotalSize    : String[14];  { * Comma string for TotalSize  * }
  bTempFileMode : Byte;        { * Contain starting File Mode  * }


{ **** Help/Info screen **** }
procedure HelpScreen;
const
  LF  = #13#10;   { * Carriage return & line feed * }

begin

  Write(LF + 'SDS - Kobayashi Size of Directory Structure version 1.3' + LF +
        '      Copyright (c) 1992, 1998 by Eric Schreiber.' + LF +
        '      All rights reserved.' + LF);
  Write('      SDS reports the disk space consumed by entire ' +
              'directory structures.' + LF);
  Write(#10 + 'Usage:' + LF +
        '      SDS <directory>' + LF +
        '      Where <directory> is the directory structure that' + LF +
        '          you want to calculate the size of.'                         + LF);
  Halt;

end;


{ **** Finds and returns current directory **** }
function CurrentDir: String;
var
  CurrDir: String;

begin

  GetDir (0, CurrDir);
  CurrentDir := CurrDir;

end;


{ **** Convert string to uppercase **** }
function UpperCase (S: String): String;
var
  I : Integer;

begin

  for I := 1 to Length(S) do
    S[I] := UpCase(S[I]);
  UpperCase := S;

end;


{ **** Determine bytes per directory **** }
procedure DirectorySpace (DriveCode:Byte);
var
   Regs                : Registers;
   lSectorsPerCluster,
   lBytesPerSector     : LongInt;

begin

   Regs.ah := $36;
   Regs.dl := DriveCode;
   MSDos(Regs);
   lSectorsPerCluster := Regs.ax;
   lBytesPerSector := Regs.cx;
   lBytesPerDir := lBytesPerSector * lSectorsPerCluster;
   lBytesAvail := Regs.bx * lBytesPerDir;

end;


{ **** Comma string from any integer **** }
function IntToCommaStr (N : Longint) : String;
var
  W: String[14];
  I: Byte;
  D: Byte;

begin

  Str(N,W);
  D := Length(W);
  for I := 3 to (D-1) do
    if I mod 3 = 0
    then Insert(',',W,(D-I+1));
  IntToCommaStr := W;

end;


{ **** Parse out directory name **** }
procedure Parse;
var
  sHomeDir,               { Starting directory                }
  sTempStr     : String;  { Holds current dir for root test   }
  cDriveLetter : Char;    { Drive letter found in sSizeDir[1] }

begin

  { *** Find drive letter and set drive number *** }
  if sSizeDir[2] = ':' then
  begin

    cDriveLetter := Upcase(sSizeDir[1]);
    if cDriveLetter in ['A'..'Z']
    then btDriveCode := ord(cDriveLetter) - 64;

  end
  else btDriveCode := 0;

  { *** Determine if sSizeDir is root and seed iTotalDir counter **** }
  { * Append backslash if d: only for chdir * }
  if (sSizeDir[2] = ':') and (Length(sSizeDir) = 2)
  then sSizeDir := sSizeDir + '\';

  sHomeDir := CurrentDir;  { * Set dir for return * }
  {$I-} ChDir (sSizeDir); {$I+}
  if IOResult <> 0 then
  begin

    Writeln;
    Writeln ('Specified directory ',sSizeDir, ' does not exist');
    ChDir (sHomeDir);
    Halt;

  end;

  sTempStr := CurrentDir;
  if Length (sTempStr) > 3
  then iTotalDir := 1;

  ChDir (sHomeDir);

  { *** Append backslash if needed *** }
  if (sSizeDir [Length (sSizeDir)] <> '\')
  then sSizeDir := sSizeDir + '\';

end;


{ **** Main procedure - recurses specified path **** }
procedure SearchDirectory (Path, FileSpec: String);
var
  Fileinfo : Searchrec;
  F        : File;

begin

  FindFirst (Path + Filespec, AnyFile, FileInfo);
  while DosError = 0 do
  begin

    Assign (F, Path + FileInfo.Name);
    {$I-} Reset (F); {$I+}
    if IOResult = 0 then
    begin

      Inc (iTotalFiles);
      if (FileInfo.Size mod lBytesPerDir) > 0.0
        then lClusters := (lClusters + (FileInfo.Size div lBytesPerDir) + 1)
        else lClusters := (lClusters + (FileInfo.Size div lBytesPerDir));
      lFileSize  := lFileSize + FileInfo.Size;
      Close (F);

    end;

    FindNext (FileInfo);

  end;

  FindFirst (Path + '*.*', Directory, FileInfo);
  while DosError = 0 do
  begin

    if ((FileInfo.Attr and Directory) > 0) and
        (FileInfo.Name <> '.') and
        (FileInfo.Name <> '..') then
    begin

      Inc (iTotalDir);
      SearchDirectory (Path + FileInfo.Name + '\', Filespec);

    end;

    FindNext (FileInfo);

  end;

end;


{ **** Calculate all totals and output strings **** }
procedure Calculations;
begin

  lDirSize    := lBytesPerDir * iTotalDir;
  lBytesCons  := lClusters * lBytesPerDir;
  sBytesCons  := IntToCommaStr (lBytesCons);
  lTotalSize  := lDirSize + lBytesCons;
  sFileSize   := IntToCommaStr (lFileSize);
  sDirSize    := IntToCommaStr (lDirSize);
  sTotalSize  := IntToCommaStr (lTotalSize);
  sBytesAvail := IntToCommaStr (lBytesAvail);

end;


{ **** Grand totals output to screen **** }
procedure OutputTotals;
begin

  Writeln;
  Writeln   ('Directory structure ', sSizeDir,' contains:');

  if iTotalDir = 1
    then Writeln (lBytesPerDir:13, ' bytes in 1 directory')
    else Writeln (sDirSize:13, ' bytes in ', iTotalDir, ' directories');

  if iTotalFiles = 1
    then Writeln (sFileSize:13, ' bytes in 1 file consuming ', sBytesCons, ' bytes')
    else Writeln (sFileSize:13, ' bytes in ', iTotalFiles, ' files consuming ', sBytesCons, ' bytes');

  Writeln   (sTotalSize:13, ' bytes total disk space used by ', sSizeDir);
  Writeln;
  Writeln   (sBytesAvail:13, ' bytes available on disk');

end;


{ **** Main Line **** }
begin

  sSizeDir := UpperCase(ParamStr(1));            { * Set sSizeDir        * }
  if sSizeDir = ''
  then sSizeDir := CurrentDir;                   { * Assign if blank     * }

  if sSizeDir[1] in HelpSet
  then HelpScreen;                               { * Show help screen    * }

  Parse;                                   { * Parse directory name      * }
  bTempFileMode := FileMode;               { * Store original FileMode   * }
  FileMode      := 0;                      { * Set read-only FileMode    * }
  DirectorySpace (btDriveCode);            { * Determine allocation size * }
  SearchDirectory (sSizeDir, '*.*');       { * Main search routine       * }
  Calculations;                            { * Perform all calculations  * }
  OutputTotals;                            { * Output totals to screen   * }
  FileMode := bTempFileMode;               { * Restore original FileMode * }

end.
