//////////////////////////////////////////////////////////////////////////
//
// rez.cpp
//
// Copyright (c) 2002 Dennis Trachuk <dennis.trachuk@nm.ru>||<dennis.trachuk@bk.ru>
//
// THIS CODE AND INFORMATION IS PROVIDED "AS IS" WITHOUT WARRANTY OF
// ANY KIND, EITHER EXPRESSED OR IMPLIED, INCLUDING BUT NOT LIMITED TO
// THE IMPLIED WARRANTIES OF MERCHANTABILITY AND/OR FITNESS FOR A
// PARTICULAR PURPOSE.
//

#include <windows.h>
#include <fmt.hpp>
#include "rez.h"

#pragma comment(linker, "/comment:\"Dennis Trachuk <dennis.trachuk@nm.ru>\"")
#ifndef _DEBUG
#pragma comment(linker, "/ignore:4078")
#pragma comment(linker, "/merge:.data=.")
#pragma comment(linker, "/merge:.rdata=.")
#pragma comment(linker, "/merge:.text=.")
#endif

LPSTR PointToName( LPCSTR FileName );
void CreateDirectory( LPSTR FullPath );
void UnixTimeToFileTime( long time, FILETIME * ft );

char ModuleName[ NM ];

static CRez        * rez = NULL;
static CRez::TItem * cur = NULL;

extern "C" DWORD WINAPI LoadFormatModule( const char * ModuleName )
{
	strcpy( ::ModuleName, ModuleName );
	return 0;
}

extern "C" BOOL WINAPI GetFormatName( int Type, char * FormatName, char * DefaultExt )
{
	if ( Type == 0 )
	{
		strcpy( FormatName, "REZ" );
		strcpy( DefaultExt, "REZ" );
		return TRUE;
	}

	return FALSE;
}

extern "C" BOOL WINAPI IsArchive( const char * Name, const unsigned char * Data, int DataSize )
{
	return CRez::IsRez( Data, DataSize );
}

extern "C" BOOL WINAPI OpenArchive( const char * Name, int * Type )
{
	*Type = 0;

	if ( rez )
		delete rez;

	cur = NULL;

	rez = CRez::Create( Name );

	return rez ? TRUE : FALSE;
}

extern "C" BOOL WINAPI CloseArchive( ArcInfo * Info )
{
	if ( rez )
	{
		delete rez;
		rez = NULL;
	}

	cur = NULL;

	return TRUE;
}

extern "C" int WINAPI GetArcItem( PluginPanelItem * Item, ArcItemInfo * Info )
{
	if ( cur == NULL )
	{
		int res = rez->EnumItems();
		if ( res != GETARC_SUCCESS )
			return res;
		cur = rez->GetFirst();
	}
	else
		cur = rez->GetNext( cur );

	if ( cur == NULL )
		return GETARC_EOF;

	memcpy( &Item->FindData, &cur->FindData, sizeof( FAR_FIND_DATA ) );

	return GETARC_SUCCESS;

}

extern "C" BOOL WINAPI GetDefaultCommands( int Type, int Command, char * Dest )
{
	if ( Type == 0 && Command <= 1 )
	{
		strcpy( Dest, "RunDll32 \"" );
		strcat( Dest, ModuleName );
		strcat( Dest, "\",ExecCommand " );
		_itoa( Command, Dest + strlen( Dest ), 10 );
		strcat( Dest, " %%a %%LM" );

		return TRUE;
	}
	return FALSE;
}

extern "C" long	WINAPI ExecCommand( HWND, HINSTANCE, LPCSTR CmdLine, long )
{
	LPSTR CmdLineStart;
	int Command = strtol( CmdLine, &CmdLineStart, 10 );
	if ( Command > 1 )
		return 1;

	CmdLineStart ++;
	LPSTR LstName = strchr( CmdLineStart, '\x20' );
	if ( LstName == NULL )
		return 1;

	LPSTR ArcName = (LPSTR)memcpy( new char[ LstName - CmdLineStart + 1 ], CmdLineStart, LstName - CmdLineStart );
	ArcName[ LstName - CmdLineStart ] = '\0';
	LstName ++;

	HANDLE hFile = CreateFile( LstName, GENERIC_READ, FILE_SHARE_READ,
		NULL, OPEN_EXISTING, FILE_FLAG_SEQUENTIAL_SCAN, NULL );

	if ( hFile == INVALID_HANDLE_VALUE )
		return 1;

	DWORD Size = GetFileSize( hFile, NULL );
	if ( Size == (DWORD)-1 || Size == 0 )
	{
		CloseHandle( hFile );
		return 1;
	}

	LPSTR buffer = new char[ Size + 1 ];

	DWORD rb;

	if ( ReadFile( hFile, buffer, Size, &rb, NULL ) != 0 && rb != Size )
	{
		delete [] buffer;
		CloseHandle( hFile );
		return 1;
	}
	CloseHandle( hFile );

	buffer[ Size ++ ] = '\0';

	DWORD   Count = 0;
	DWORD   Alloc = 32;
	LPSTR * Files = (LPSTR*)malloc( Alloc * sizeof( LPSTR ) );

	LPSTR CurPtr = buffer;
	LPSTR EndPtr = buffer + Size;
	LPSTR NewPtr = CurPtr;

	CRez * RezFile = CRez::Create( ArcName );
	if ( RezFile == NULL )
	{
		delete [] Files;
		delete [] buffer;
		delete [] ArcName;
	}

	while ( CurPtr < EndPtr )
	{
		while ( CurPtr < EndPtr && *CurPtr != '\r' && *CurPtr != '\n' )
			CurPtr++;

		if ( *CurPtr == '\r' )
			*CurPtr++ = '\0';

		if ( *CurPtr == '\n' )
			*CurPtr++ = '\0';

		if ( CurPtr < EndPtr && strlen( NewPtr ) > 0 )
		{
			if ( Alloc < Count + 1 )
			{
				Alloc += 32;
				Files = (LPSTR*)realloc( Files, Alloc * sizeof( LPSTR ) );
			}

			Files[ Count++ ] = NewPtr;
		}

		NewPtr = CurPtr;
	}

	if ( RezFile->EnumItems() == GETARC_SUCCESS )
		for ( DWORD i = 0; i < Count; i ++ )
			RezFile->Extract( Files[ i ], Command == 0 );

	delete [] Files;
	delete [] buffer;
	delete [] ArcName;

	delete RezFile;

	return 0;
}

void UnixTimeToFileTime( long time, FILETIME * ft )
{
	static char Days[ 12 ] = { 31, 28, 31, 30, 31, 30, 31, 31, 30, 31, 30, 31 };

	SYSTEMTIME st;

	time -= 24L * 60L * 60L * 3652L;

	st.wMilliseconds = 0;

	st.wSecond = time % 60; time /= 60;
	st.wMinute = time % 60; time /= 60;

	st.wYear = 1980 + (int)( ( time / ( 1461L * 24L ) ) << 2 );

	time %= 1461L * 24L;
	if ( time >= 366 * 24 )
	{
		time -= 366 * 24;
		st.wYear++;
		st.wYear += (int)( time / ( 365 * 24 ) );
		time %= 365 * 24;
	}

	st.wHour = time % 24; time /= 24;

	time++;
	if ( ( st.wYear & 3 ) == 0 )
	{
		if ( time > 60 )
			time--;
		else if ( time == 60 )
		{
			st.wMonth = 2;
			st.wDay   = 29;
			SystemTimeToFileTime( &st, ft );
		}
	}

	for ( st.wMonth = 0; Days[ st.wMonth ] < time; st.wMonth++ )
		time -= Days[ st.wMonth ];
	st.wMonth ++;
	st.wDay = LOWORD( time );

	SystemTimeToFileTime( &st, ft );
}

BOOL CRez::IsRez( const unsigned char * Data, int DataSize )
{
	if ( DataSize < sizeof( TRezHeader ) )
		return FALSE;

	PRezHeader hdr = (PRezHeader)Data;

	if ( hdr->Mark0 != 1 || hdr->Mark1 != 1 )
		return FALSE;

	if ( strstr( hdr->Signature, "LithTech" ) == NULL )
		return FALSE;

	return TRUE;
}

CRez * CRez::Create( LPCSTR FileName )
{
	HANDLE hFile = CreateFile( FileName, GENERIC_READ, FILE_SHARE_READ|FILE_SHARE_WRITE,
		NULL, OPEN_EXISTING, FILE_FLAG_RANDOM_ACCESS, NULL );

	if ( hFile == INVALID_HANDLE_VALUE )
		return FALSE;

	DWORD rs;

	TRezHeader hdr;

	if ( ReadFile( hFile, &hdr, sizeof( TRezHeader ), &rs, NULL ) == 0 || rs != sizeof( TRezHeader ) )
	{
		CloseHandle( hFile );
		return NULL;
	}

	if ( IsRez( (LPBYTE)&hdr, sizeof( TRezHeader ) ) == FALSE )
	{
		CloseHandle( hFile );
		return NULL;
	}

	CRez * result = new CRez;

	result->m_hFile = hFile;

	result->m_Root.Parent =
	result->m_Root.SubDir =
	result->m_Root.Next   =
	result->m_Root.Prev   = NULL;
	result->m_Root.Offset = hdr.Root.Offset;
	result->m_Root.Length = hdr.Root.Length;
	memset( &result->m_Root.FindData, 0, sizeof( FAR_FIND_DATA ) );

	result->m_MaxIndexSize = max( sizeof( TRezFile ), sizeof( TRezDirectory ) ) +
		max( hdr.Max_Foldername, hdr.Max_Filename ) + 1 + 3;
	result->m_Index = (TRezIndex*) new BYTE[ result->m_MaxIndexSize ];

	return result;
}

CRez::CRez() : m_hFile( INVALID_HANDLE_VALUE ), m_Index( NULL ), m_MaxIndexSize( 0 )
{
}

CRez::~CRez()
{
	delete [] m_Index;      // always valid
	CloseHandle( m_hFile ); // always valid
}

int CRez::EnumItems( CRez::TItem * Parent )
{
	DWORD Offset = Parent->Offset;

	CRez::TItem * Prev = NULL;
	while ( Offset < Parent->Offset + Parent->Length )
	{
		if ( SetFilePointer( m_hFile, Offset, NULL, FILE_BEGIN ) != Offset )
			return GETARC_BROKEN;

		DWORD rb;
		if ( ReadFile( m_hFile, m_Index, m_MaxIndexSize, &rb, NULL ) == 0 )
			return GETARC_READERROR;

		CRez::TItem * Item = new CRez::TItem;
		Item->Parent = Parent;
		Item->Next   = NULL;
		Item->Prev   = Prev;
		Item->SubDir = NULL;
		Item->Offset = m_Index->Data.Offset;
		Item->Length = m_Index->Data.Length;
		if ( Parent->SubDir == NULL )
			Parent->SubDir = Item;
		if ( Prev )
			Prev->Next = Item;
		Prev = Item;

		memset( &Item->FindData, 0, sizeof( FAR_FIND_DATA ) );

		UnixTimeToFileTime( m_Index->Date, &Item->FindData.ftLastWriteTime );

		strcpy( Item->FindData.cFileName, Parent->FindData.cFileName );

		if ( m_Index->Attr == 0 )
		{
			if ( rb < sizeof( TRezFile ) )
			{
				delete Item;
				return GETARC_BROKEN;
			}

			TRezFile * f = (TRezFile*)m_Index;

			strcat( Item->FindData.cFileName, f->GetName() );

			if ( *f->Type != '\0' )
			{
				rb = strlen( Item->FindData.cFileName );
				Item->FindData.cFileName[ rb++ ] = '.';

				for ( int i = 3; i >= 0; i -- )
					if ( f->Type[ i ] )
						Item->FindData.cFileName[ rb++ ] = f->Type[ i ];

			}

			Item->FindData.nFileSizeLow = f->Data.Length;

			Offset += sizeof( TRezFile ) + strlen( f->GetName() ) + 2;
		}
		else
		{
			if ( rb < sizeof( TRezDirectory ) )
			{
				delete Item;
				return GETARC_BROKEN;
			}

			Item->FindData.dwFileAttributes = FILE_ATTRIBUTE_DIRECTORY;

			TRezDirectory * Dir = (TRezDirectory*)m_Index;

			strcat( Item->FindData.cFileName, Dir->GetName() );
			strcat( Item->FindData.cFileName, "\\" );

			int ns = strlen( Dir->GetName() );

			if ( Dir->Data.Length > 0 )
			{
				rb = EnumItems( Item );
				if ( rb != GETARC_SUCCESS )
					return rb;
			}

			Offset += sizeof( TRezDirectory ) + ns + 1;
		}
	}
	return GETARC_SUCCESS;
}

CRez::TItem * CRez::GetNext( CRez::TItem * Prev, CRez::TItem * Root )
{
	if ( Prev->SubDir )
		return Prev->SubDir;

	CRez::TItem * Next = Prev->Next;
	while ( Next == NULL && Prev->Parent != Root )
	{
		Prev = Prev->Parent;
		Next = Prev->Next;
	}

	return Next;
}

#define BUF_SZ 0x10000

bool CRez::Extract( CRez::TItem * Item, LPCSTR Name )
{
	HANDLE hFile = CreateFile( Name, GENERIC_WRITE,
		FILE_SHARE_READ|FILE_SHARE_WRITE, NULL, CREATE_ALWAYS, 0, NULL );

	if ( hFile == INVALID_HANDLE_VALUE )
		return false;

	if ( SetFilePointer( m_hFile, Item->Offset, NULL, FILE_BEGIN ) != Item->Offset )
	{
		CloseHandle( hFile );
		return false;
	}

	LPBYTE buffer = new BYTE[ BUF_SZ ];

	DWORD sz;

	for ( DWORD i = 0; i < Item->Length / BUF_SZ; i++ )
	{
		ReadFile( m_hFile, buffer, BUF_SZ, &sz, NULL );
		WriteFile( hFile, buffer, BUF_SZ, &sz, NULL );
	}
	sz = Item->Length % BUF_SZ;
	ReadFile( m_hFile, buffer, sz, &sz, NULL );
	WriteFile( hFile, buffer, sz, &sz, NULL );

	delete [] buffer;

	CloseHandle( hFile );

	return true;
}

bool CRez::Extract( LPCSTR Name, bool bUsePath )
{
	CRez::TItem * Item = GetFirst();
	while ( Item )
	{
		if ( _stricmp( Name, Item->FindData.cFileName ) == 0 )
			break;
		Item = GetNext( Item );
	}

	if ( Item == NULL )
		return false;

	if ( bUsePath )
	{
		CRez::TItem * Last = Item;
		while( Item )
		{
			if ( Item->SubDir )
				CreateDirectory( Item->FindData.cFileName );
			else
				Extract( Item, Item->FindData.cFileName );

			Item = GetNext( Item, Last );
		}
		return true;
	}

	return Extract( Item, PointToName( Item->FindData.cFileName ) );
}

bool FileExists( LPCSTR FileName )
{
	DWORD attrs = GetFileAttributes( FileName );
	return attrs != (DWORD)-1 && attrs != FILE_ATTRIBUTE_DIRECTORY;
}

void CreateDirectory( LPSTR FullPath )
{
	if ( FileExists( FullPath ) )
		return;

	for ( LPSTR c = FullPath; *c; c ++ )
	{
		if ( *c != '\x20' )
		{
			for ( ; *c; c ++ )
			{
				if ( *c == '\\' )
				{
					*c = 0;
					CreateDirectory( FullPath, NULL );
					*c = '\\';
				}
			}

			CreateDirectory( FullPath, NULL );

			break;
		}
	}
}

LPSTR PointToName( LPCSTR FileName )
{
	LPCSTR p = FileName;
	while ( *FileName )
	{
		if ( *FileName == '\\' || *FileName == '/' || *FileName == ':' )
			p = FileName + 1;
		FileName ++;
	}
	return (LPSTR)p;
}
