/*
 * fsperf.c
 * (C) Grzegorz J.Blinowski
 *
 *	v	0.0	Jan 1995		( devel )
 * v  0.9   Feb 1995    ( pre - 1st release )
 * v  1.0	Jun 1995		( 1st release )
 *
 * Tests performance of file operations: creat / write / read (seq & random)
 *
 * Compiled & tested under:
 *		MS-DOS 5.0 (Borland TC 2.0, BC 3.1 compilers
 *		Solaris 2.3 (SPARCWorks C compiler)
 *
 * Note:
 *   getopt() function is required but not included in this distribution
 *
 *		HP-UX 9 (boundled C compiler)
 * !!!!!!!!
 * TAB = 3
 * !!!!!!!!
 *
 */

#include "config.h"
#include <time.h>
#include <stdio.h>
#include <string.h>
#include <stdlib.h>
#include <fcntl.h>

#include <ctype.h>

#ifdef __TURBOC__
#include <io.h>
#include <dos.h>
#include <dir.h>
#include <sys\stat.h>
#else
#include <limits.h>
#include <sys/stat.h>
#include <sys/file.h>
#endif

#if defined(UNIX)
#include <unistd.h>

#else
int	getopt(int argc, char *argv[], char *optionS);
extern char *optarg;
#endif

#ifdef __TURBOC__
#define MAX_FILE (MAXFILE+MAXEXT)
#define TF_PERMS (S_IREAD | S_IWRITE)
#else
#define MAX_FILE 255

#ifdef POSIX
#define MAXPATH _POSIX_PATH_MAX
#endif

#define TF_PERMS (S_IWUSR | S_IRUSR)
#endif

#ifdef UNIX
#define MAX_BUF_SIZE ((size_t)65536)
#else
#define MAX_BUF_SIZE ((size_t)(65536-12))
#endif

#define STD_BUF_SIZE	8192
#define SMALL_FILE_SLIMIT MAX_BUF_SIZE
#define MAX_N_FILES 1000

#define min(a,b) ((a)>(b) ? (b) : (a))
#define VERB_Stat 	1
#define VERB_Stat2	2
#define VERB_Warn		4
#define VERB_Action	6

/* Copyright string */

static char c[] =  "fsperf v 1.0 (C) by Grzegorz J.Blinowski";

/* Timer global variables */
double TotalTime = 0.0;
double MFileTotalTime = 0.0;
double SeqWriteTime = 0.0,
		 RndWriteTime = 0.0;
double ReadSeqTime = 0.0,
		 ReadRndTime = 0.0;

/* Action flags: */
int FlgM = 1;				/* many small-files test */
int FlgU = 0;			   /* don't unlink until finished */
int FlgV = 5;           /* Verbosity level */
int FlgCd = 0;				/* work in a specified directory */
int FlgRS = 0;			/* test sequential read time of the long file */
int FlgRR = 0;			/* test random read time of the long file */
int FlgL = 0;				/* don't unlink long file */
int FlgWF = 0;				/* use provided work file */
int FlgWS = 0;				/* execute sequential write on long file */
int FlgWR = 0;				/* execute sequential write on long file */
int FlgIR = 0;				/* increment record size after each full test */
int FlgTab = 0;			/* print stats in table format */

long int Size = 1024;
long int BSize = (1024L*1024L);
long int N = 50;
size_t RRec_size = STD_BUF_SIZE;
size_t WRec_size = STD_BUF_SIZE;
long int NReadRecs = 0,
			NWriteRecs = 0;
long int NSteps = 1;

static char *univ_buf;

static char *Tmpnamarr[MAX_N_FILES];

static int Tmparr_i=0;

#if defined(HAS_TEMPNAM) && !defined(HAS_MKTEMP)
#define Tempnam(s) tempnam(WORK_DIR,"fsp")
#endif

#if defined(HAS_MKTEMP)
static char TmpPattern[30];
#define Tempnam(s) mktemp(TmpPattern)
#endif

#if !defined(HAS_TEMPNAM)
#define Tempnam(s) tmpnam(s)
#endif

#ifdef RAND_ONLY
#define random rand
#define RANDOM_0
#endif

#ifdef RANDOM_0
#define Random(x) (random()%(x))
#endif

#ifdef RANDOM_1
#define Random(x) random(x)
#endif


#ifdef MSDOS
#define O_SPECMODES O_BINARY
#else
#define O_SPECMODES 0
#endif

#ifdef __TURBOC__
struct time timer, timer_s;

void start_timer(void)
{
	gettime(&timer);
}

void stop_timer(void)
{
	gettime(&timer_s);
}

double time_cvt(struct time *t1)
{
	return ((( (t1->ti_hour)*60.0 + (t1->ti_min)) * 60.0 + (t1->ti_sec))*100.0
		+t1->ti_hund) / 100.0;
}

double get_timer(void)
{
	return time_cvt(&timer_s)-time_cvt(&timer);
}

double get_timer_error(void)
{
	return 0.18;
}

#else

static time_t timer,timer_s;

void start_timer(void)
{
	time(&timer);
}

void stop_timer(void)
{
	time(&timer_s);
}

double get_timer(void)
{
	return  difftime(timer_s, timer);
}
double get_timer_error(void)
{
	return 1.0;
}
#endif	/* Universal time, sadly 1 sec resolution */

#define print_tr(flg, r) if (flg) printf(" %8.2f ",(r)); else printf("        ");

void PrintTable(void)
{
	static header=0;
	double MinTime;

	if (!header)
		printf(

" Sector Size [B]       -->     Transfer Rate [KB/sec]   <--  Error [%]\n"
" Write       Read        WS        WR        RS        RR"
	"\n" );

	header = 1;

	if (FlgWS || FlgWR)
		printf("%5u      ",WRec_size);
	 else
		printf("   -          ");
	if (FlgRS || FlgRR)
		printf("%5u     ", RRec_size);

	print_tr(FlgWS, BSize/SeqWriteTime/1000.0);
	print_tr(FlgWR, (WRec_size*NWriteRecs)/RndWriteTime/1000.0);
	print_tr(FlgRS, BSize/ReadSeqTime/1000.0);
	print_tr(FlgRR, RRec_size * NReadRecs/ReadRndTime/1000.0);

	MinTime=1e10;

	if (FlgWS)
		MinTime= SeqWriteTime;
	if (FlgWR)
		MinTime = min(MinTime, RndWriteTime);
	if (FlgRS)
		MinTime = min(MinTime, ReadSeqTime);
	if (FlgRR)
		MinTime = min(MinTime, ReadRndTime);

	printf("    %5.2f", 100.0 * get_timer_error()/MinTime);
	printf("\n");
}

void PrintStats(void)
{
		if ( FlgTab ) {
			PrintTable();
			return;
		}

		if ( FlgV >= VERB_Stat) {
			if ( FlgM ) {
				printf("Total %ld files created, total %ld bytes written to file(s)\n",
					(long) N, (long) N * Size );
				printf("Transfer rate: %8.4f [KB/sec]\n", N*Size/MFileTotalTime/1000.0);
				printf("File create/write/(unlink) time: %8.4f [files/sec]\n", N/MFileTotalTime);
				}
			else {
				printf("%ld byte file created\n", BSize);
				if (FlgWR)
					printf("Random write transfer rate: %8.4f [KB/sec]\n", (WRec_size*NWriteRecs)/RndWriteTime/1000.0);
				if ( FlgRS )
					printf("Sequential read transfer rate: %8.4f [KB/sec]\n", BSize/ReadSeqTime/1000.0);
				if ( FlgRR )
					printf("Random read transfer rate: %8.4f [KB/sec]\n", RRec_size * NReadRecs/ReadRndTime/1000.0);
				if (FlgWS)
					printf("Sequential write transfer rate: %8.4f [KB/sec]\n", BSize/SeqWriteTime/1000.0);
			}
		}

		if ( FlgV >= VERB_Stat2) {
			if (FlgM)
				printf("Total file creation time: %8.3f\n", MFileTotalTime);
			else {
				if (FlgWS)
					printf("Long file sequential write time %8.3f\n", SeqWriteTime);
				if (FlgWR)
					printf("Long file random write time %8.3f\n", RndWriteTime);
				if ( FlgRS )
					printf("Sequential Read time %8.3f\n", ReadSeqTime);
				if ( FlgRR )
					printf("Random Read time %8.3f\n", ReadRndTime);
			}
		}
}

void ZeroStats(void)
{
 TotalTime = 0.0;
 MFileTotalTime = 0.0;
 SeqWriteTime = 0.0;
 RndWriteTime = 0.0;
 ReadSeqTime = 0.0;
 ReadRndTime = 0.0;
}

void CleanUp(void)
{
	int i;
	if ( FlgV >= VERB_Action )
		printf("removing files...\n");
	for (i=0; i< Tmparr_i; i++)
		unlink ( Tmpnamarr[ i ]);
}

void Write1Buf(void)
{
	int fd;

	if ( -1 == (fd=open(Tmpnamarr[ Tmparr_i++ ], O_CREAT|O_TRUNC|O_WRONLY|O_SPECMODES , TF_PERMS  ) )) {
			perror("can't create test file ");
			exit(1);
		}

	if (-1 == write( fd, (void*)univ_buf, Size) ) {
		perror("can't write to test file ");
		exit(1);
	}

	close( fd );
	if (! FlgU)
		unlink( Tmpnamarr[ Tmparr_i - 1 ] );
}


void GenFileNames(long int n)
{

	int i;
	for (i=0; i<n; i++) {
		if ( (Tmpnamarr[i] = calloc(1, MAX_FILE)) == NULL) {
			fprintf(stderr, "calloc failed: out of memory allocating name array\n");
			exit(1);
		}
		strcpy(  Tmpnamarr[i], (const char*) Tempnam(NULL) );
	}
}

int DoMany(void)
{
	int i;

	GenFileNames(N);
	Tmparr_i=0;

	if ( FlgV >= VERB_Action )
		printf( "creating files...\n" );
	start_timer();
	for (i=0; i < N; i++)
		Write1Buf();
	stop_timer();

	if ( FlgV >= VERB_Action )
		printf( "finished.\n" );

	if (FlgU)
		CleanUp();

	MFileTotalTime = get_timer();
	TotalTime += MFileTotalTime;

	PrintStats();
	return 0;
}

int WriteSeqBuf( int fd, long int n, size_t b)
{
	long int i;


	for (i=0; i<n; i++)
		if ( b != write(fd, (void*)univ_buf, b) ) {
			perror("write failed on long file ");
			exit(1);
	}

	return fd;
}

int WriteRndBuf(int fd, long int n_recs, long int n_writes, size_t b)
{
	long int i;

	for (i=0; i<n_writes; i++) {
		lseek(fd, b*Random(n_recs), SEEK_SET);
		if ( b != write(fd, (void*)univ_buf, b) ) {
			perror("write failed on long file ");
			exit(1);
	}
}
	return fd;
}


int ReadSeq(int fd, long int n, size_t b)
{
	long int i;

	for (i=0; i<n; i++)
		read(fd, (void*)univ_buf, b);

	return fd;
}

int ReadRand(int fd, long int n_recs, long int n_reads, size_t b)
{
	long int i;

	for (i=0; i<n_reads; i++) {
		lseek(fd, b*Random(n_recs), SEEK_SET);
		if (b != read(fd, (void*)univ_buf, b) ) {
		  perror("read failed ");
		  exit(1);
		}
	}
	return fd;
}

int OpenToWrite(char *fname, int create)
{
	int fd;
	if ( create ) {
		if ( -1 == (fd=open(fname, O_CREAT | O_TRUNC | O_WRONLY | O_SPECMODES , TF_PERMS) )) {
			perror("can't create test file ");
			exit(1);
		}
	}
	 else
		if ( -1 == (fd=open(fname, O_WRONLY | O_SPECMODES , TF_PERMS) )) {
			perror("can't open test file ");
			exit(1);
		}
	return fd;
}


int DoOne(char *pfname)
{
	char fname[ MAX_FILE ];
	int fd;
	int creat = 1;
	int count;

	if ( pfname == NULL )
		strcpy(fname, (const char*) Tempnam(NULL) );
	else {
		creat = 0;
		strcpy( fname, pfname );
	}

	for (count=1; count <= NSteps; count++) {
		if ( FlgWS ) {
			fd = OpenToWrite(fname, creat);

		if (FlgV >= VERB_Action)
				printf("Sequentialy writing to long file:"
						 "%d byte %ld records...\n", WRec_size, (long)BSize/WRec_size);

			start_timer();
			WriteSeqBuf(fd, BSize/WRec_size, WRec_size);
			stop_timer();

			SeqWriteTime = get_timer();
			TotalTime += SeqWriteTime;

			close( fd );
		}

		if ( FlgWR ) {
			if ( creat && FlgWS ) {
				if (FlgV >= VERB_Action)
					printf("Unlinking long file...\n");
				unlink(fname);
			}
			fd = OpenToWrite(fname, creat);
			lseek(fd, BSize-1, SEEK_SET);
			write(fd, "\0x0", 1);

			if (FlgV >= VERB_Action)
					printf("Randomly writing to long file:"
					"%d byte %ld records...\n", WRec_size, NWriteRecs);

			start_timer();
			fd=WriteRndBuf(fd, BSize/WRec_size, NWriteRecs, WRec_size);
			stop_timer();

			RndWriteTime = get_timer();
			TotalTime += RndWriteTime;

			close( fd );
		}

		if (!FlgWS && !FlgWR && creat) {
			fd = OpenToWrite(fname, 1);
			lseek(fd, BSize-1, SEEK_SET);
			write(fd, "\0x0", 1);
			close(fd);
		}

		fd = open( fname, O_SPECMODES | O_RDONLY );
		if (fd == -1) {
			perror("failed to re-open test file " );
			exit(1);
			  }

		if ( FlgRS ) {
		if (FlgV>=VERB_Action)
			printf("Sequential read %ld %d byte records...\n",BSize/RRec_size, RRec_size);
			start_timer();
			ReadSeq(fd, BSize/RRec_size, RRec_size);
			stop_timer();
			ReadSeqTime = get_timer();
			TotalTime += ReadSeqTime;
		}


		if ( FlgRR ) {
		if (FlgV>=VERB_Action)
			printf("Random read %ld %d byte records...\n",NReadRecs, RRec_size);
			start_timer();
			ReadRand(fd, BSize/RRec_size, NReadRecs, RRec_size);
			stop_timer();
			ReadRndTime = get_timer();
			TotalTime += ReadRndTime;
		}

		close(fd);

		if ( ! FlgL ) {
			if (FlgV >= VERB_Action)
				printf("Unlinking long file...\n");
			unlink(fname);
		}

	PrintStats();
	ZeroStats();
	if ( FlgIR ) {
		if (2L*WRec_size > MAX_BUF_SIZE || 2L*WRec_size==0)
			WRec_size = MAX_BUF_SIZE;
		 else
			WRec_size *=2;
		if (2L*RRec_size > MAX_BUF_SIZE || 2L*RRec_size == 0)
			RRec_size = MAX_BUF_SIZE;
		 else
			RRec_size *=2;

	}
	}
   return 0;
}

void CheckWorkFile(char *fname)
{
	struct stat statbuf;
	if ( -1 == stat(fname, &statbuf) ) {
		perror("Can't find work file ");
		exit(1);
	}

	BSize = statbuf.st_size;

	if (BSize < WRec_size) {
		BSize = WRec_size;
		if (FlgV >= VERB_Warn)
			printf("Warning: file size set to %ld (was smaller than record size)\n",BSize);
	}

	if ( BSize % WRec_size != 0) {
		BSize = (BSize/WRec_size)*WRec_size;
		if (FlgV >= VERB_Warn)
			printf("Warning: file size set to %ld\n",BSize);
	}

}

void Usage(void)
{
	printf("\nfsperf [-?] M|L[SRsr] [-U] [-d work_dir] [-v level] [-s size[k|m]] [-n nfiles]\n"
			 "         [-R|W nrecords] [-r rsize] [-w wsize] [-l] [-f file ] [-S nsteps]\n"
			 " Options:\n"
			 "     M           -         Multi-file test\n"
			 "     L...        -         One-file test\n"
			 "     LS          -         sequential write test\n"
			 "     LR          -         random write test\n"
			 "     Ls          -         sequential read test\n"
			 "     Lr          -         random write test\n"
			 "    -U           -         unlink only after test finished\n"
			 "    -d work_dir  -         set work dir\n"
			 "    -v level     -         verbosity level\n"
			 "    -t           -         display the results in a table\n"
			 "    -s size[k|m] -         file size, option: kilo or megabytes\n"
			 "    -n nfiles    -         create nfiles files\n"
			 "    -W nrecords  -         write nrecords during random access test\n"
			 "    -R nrecords  -         read nrecords during random access test\n"
			 "    -r rsize     -         read-record size\n"
			 "    -w wsize     -         write-record size\n"
			 "    -l           -         leave test file\n"
			 "    -f file      -         use file as work file\n"
			 "    -S nsteps    -         test nsteps times doubling record size\n"
			 "    -?           -         help\n"
			);
}


int main(int argc, char **argv)
{
	  int opt;
	  char work_dir[ MAXPATH + 1];
	  char WorkFileName[ MAX_FILE ];

	  if (argc <2) {
		Usage();
		exit(1);
	}

#	if defined(HAS_MKTEMP)
	  strcpy(TmpPattern, "fsf.XXXXXXX");
#	endif

	  switch (argv[1][0]) {
		case 'M':
			FlgM = 1;
			break;
		case 'L':
			FlgM = 0;
			{	int i;
				for (i=1; argv[1][i]; i++)
					switch( argv[1][i] ) {
						case 's':
							FlgRS=1;
						break;
						case 'r':
							FlgRR=1;
						break;
						case 'S':
							FlgWS=1;
						break;
						case 'R':
							FlgWR=1;
						break;
						default:
							if ( FlgV >= VERB_Warn )
								fprintf(stderr, "illegal option to -R (ignored)\n");
						break;
					}
			}			break;
      default:
			Usage();
			exit(1);
		}

	  argv++; argc--;

	  while( (opt=getopt(argc,argv,"tS:W:R:f:lr:w:N:b:d:c:v:UR:s:n:?")) != EOF )
		switch(opt) {
			case 't':
				if ( FlgM ) {
					fprintf(stderr, "Warning: 't' option ignored\n");
				break; }
				FlgTab = 1;
				break;
			case 'S':
				if ( FlgM ) {
					fprintf(stderr, "Warning: 'S' option ignored\n");
					break; }
				sscanf( optarg, "%ld", (long int*)&NSteps);
				FlgIR = 1;
				break;

			case 'f':
				if ( FlgM ) {
					fprintf(stderr, "Warning: 'f' option ignored\n");
					break; }
				sscanf( optarg, "%s", WorkFileName );
				FlgWF = 1;
				break;

			case 'l':
				if ( FlgM )
					fprintf(stderr, "Warning: 'l' option ignored\n");
				FlgL = 1;
				break;

			case 'R':
				sscanf(optarg, "%lu", &NReadRecs);
				if ( NReadRecs == 0 ) {
					FlgRR = FlgRS = 0;
					fprintf(stderr, "read test canceled\n"); }
			break;

			case 'W':
				sscanf(optarg, "%lu", &NWriteRecs);
				if ( NWriteRecs == 0 ) {
					FlgWR = FlgWS = 0;
					fprintf(stderr, "write test canceled\n"); }
			break;

			case 'r':
				sscanf(optarg, "%u", &RRec_size);
				if ( RRec_size == 0 ) {
					fprintf(stderr, "Read-record size can't be 0\n");
					exit(1);
				}
				break;

			case 'w':
				sscanf(optarg, "%u", &WRec_size);
				if ( WRec_size == 0 ) {
					fprintf(stderr, "Write-record size can't be 0\n");
					exit(1);
				}
				break;

			case 'd':
				FlgCd=1;
				strcpy(work_dir, optarg);
				break;

			case 'v':
				if ( optarg[0] >= '0' && optarg[0] <= '9')
					FlgV = optarg[0]-'0';
				break;

			case 'U':
				FlgU = 1;
				break;

			case 's':
					{  int  cnt;
						char unit;
						cnt=sscanf(optarg,"%ld%c", &Size, &unit);
						if (cnt==2)
							Size=Size *
								(tolower(unit)=='k' ? 1024 :  tolower(unit) == 'm' ?
									1048576L : 1);
					}
				BSize = Size;
				break;

			case 'n':
				N=atol(optarg);
				break;

			case '?':
				Usage();
				exit(0);
			default:
				break;
		}

		if ( FlgM && Size > SMALL_FILE_SLIMIT ) {
			fprintf( stderr, "Small file > %ld\n",(long) SMALL_FILE_SLIMIT );
			exit(1);
		}

		if (FlgM && N >= MAX_N_FILES) {
			fprintf(stderr, "too many files with -U option (max %ld)\n", (long) MAX_N_FILES);
			exit(1);
		}

		if ( FlgCd ) {
#		ifdef __TURBOC__
				char drive[3], dir[MAXDIR], name[MAXFILE], ext[MAXEXT];
				if ( fnsplit(work_dir, drive, dir, name, ext) & DRIVE )
					setdisk( tolower(drive[0]) - 'a' );
#		endif
			if ( -1 == chdir(work_dir) ) {
				fprintf(stderr, "can't chdir to %s, no dir or no permition\n",work_dir);
				exit(1);
			}
}

		if ( (univ_buf = malloc( MAX_BUF_SIZE ) ) == NULL) {
			perror("No memory ");
			exit(1);
		}

		if (FlgM)
			DoMany();
		else
			if ( FlgWF ) {
				CheckWorkFile( WorkFileName );
				DoOne( WorkFileName );
			}
			else
				DoOne(NULL);

		return 0;
}

