// Format Block plugin for FAR Manager
// Version 1.06
// (C)opyright Wesha the Leopard, 2000.
// wesha@members.limitless.org
// Dedicated to Vixy.

//  LICENSE TERMS AND CONDITIONS:

// This code may be freely distributed, modified, partially or in
// totality, by any mechanical or electronical means, or used in any
// other way, given that:
//
// 1. Its author is always given full credit for the original code.
// 2. No profit whatsoever may be made for the distribution and/or use
//    of this code.

//======================== definitons ===========================

#define _WINCON_ // to prevent including wincon.h
#include <windows.h>
#undef _WINCON_
#include "plugin.hpp" //this file switches to 1-byte alignment
#pragma pack(8)
#include <wincon.h> //this file wants 8-byte alignment
#include <winuser.h>

//======================== exports ==============================

extern "C"
{
  void WINAPI _export SetStartupInfo(struct PluginStartupInfo *info);
  void WINAPI _export GetPluginInfo(struct PluginInfo *info);
  HANDLE WINAPI _export OpenPlugin(int OpenFrom,int item);
};

//======================== typedefs==============================

struct InitDialogItem {
  unsigned char Type;
  unsigned char X1,Y1,X2,Y2;
  unsigned char Focus;
  bool          Selected;
  unsigned int  Flags;
  bool          DefaultButton;
  signed   char Data;
};

//======================== global variables =====================

struct PluginStartupInfo startupInfo;
#if (FARMANAGERVERSION >= 0x0146)
struct FarStandardFunctions FSF;
#endif //FARMANAGERVERSION

//------------------------ default messages----------------------

enum { msgPluginName,
       msgLeftMargin, msgParagraph, msgRightMargin,
       msgAlignment,
       msgLeft,msgRight,msgCenter,msgFullJustify,msgForceJustify,
       msgDetection,
       msgNone,msgParagraphPerLine,msgBlankLine,msgWhitespace,
       msgSeparateParagraphs,
       msgOK,msgCancel,msgTemplate,
       msgError1,msgError2,msgError3,
       msgTemplate2,
       msgDelete1,msgDelete2
     };

//===================== external function wrappers ==============

#if (FARMANAGERVERSION >= 0x0146)
#define FARitoa(a,b,c)  FSF.itoa(a,b,c)
#define FARatoi(x)      FSF.atoi(x)
#else  // FARMANAGERVERSION
#define FARitoa(a,b,c)  itoa(a,b,c)
#define FARatoi(x)      atoi(x)
#endif // FARMANAGERVERSION

#define GetMsg(msgID)	startupInfo.GetMsg(startupInfo.ModuleNumber,msgID)
#define CenterDialog(width,height,help,items,num)	\
		startupInfo.Dialog(startupInfo.ModuleNumber,-1,-1,width,height,help,items,num)
#define CenterMenu(title,bottom,help,keys,retcode,items,num) \
    startupInfo.Menu(startupInfo.ModuleNumber,-1,-1,0, FMENU_WRAPMODE | FMENU_AUTOHIGHLIGHT, \
                     title,bottom,help,keys,retcode,items,num)

#define GetEditorInfo(param)  startupInfo.EditorControl(ECTL_GETINFO,param)
#define ShowWarning(msgs,num,help) startupInfo.Message(startupInfo.ModuleNumber, \
                                                       FMSG_WARNING,help,msgs,num,1)
#define ShowQuestion(msgs,num,help) startupInfo.Message(startupInfo.ModuleNumber, \
                                                       0,help,msgs,num,2)

//---------------------------------------------------------------

int SetCursorPosition (const int x, const int y) {
struct EditorSetPosition tmp = {-1,-1,-1,-1,-1,-1};
tmp.CurLine = y; tmp.CurPos = x;
return startupInfo.EditorControl(ECTL_SETPOSITION,&tmp);
}

//---------------------------------------------------------------

int GetString(const int y,struct EditorGetString *str) {
char *tmp;
str->StringNumber=y;
if (startupInfo.EditorControl(ECTL_GETSTRING,str)) {
  if(tmp = (char*)malloc(str->StringLength)) {
    str->StringText = (char*)memcpy(tmp,str->StringText,str->StringLength);
    if(tmp = (char*)malloc(strlen(str->StringEOL))) {
      str->StringEOL = strcpy(tmp,str->StringEOL);
      return true;
      }
    }
  }
return false;
}

//---------------------------------------------------------------

inline void ReleaseString(const struct EditorGetString * const str) {
free(str->StringText); free(str->StringEOL);
}

//---------------------------------------------------------------

int DeleteString(const int y) {
if(SetCursorPosition(0,y))
  return startupInfo.EditorControl(ECTL_DELETESTRING,NULL);
return FALSE;
}

//---------------------------------------------------------------

int InsertString(const int y,const char * const str,
                 const int length,const char * const eol) {
if(SetCursorPosition(0,y)) {
  if (startupInfo.EditorControl(ECTL_INSERTSTRING,0)) {
    EditorSetString tmp;
    tmp.StringNumber = y;
    tmp.StringText = (char*)str;
    tmp.StringEOL = (char*)eol;
    tmp.StringLength = length;
    return startupInfo.EditorControl(ECTL_SETSTRING,&tmp);
    }
  }
return FALSE;
}

//---------------------------------------------------------------

inline int UnselectBlock(void) {
struct EditorSelect tmp;
tmp.BlockType = BTYPE_NONE;
return startupInfo.EditorControl(ECTL_SELECT,&tmp);
}

//---------------------------------------------------------------

struct FarDialogItem* ConvertDialog(const struct InitDialogItem * const items,
                                    struct FarDialogItem * const newDialog,const int num) {
int i;
for (i=0;i<num;i++) {
  newDialog[i].Type = items[i].Type;
  newDialog[i].X1=items[i].X1;
  newDialog[i].Y1=items[i].Y1;
  newDialog[i].X2=items[i].X2;
  newDialog[i].Y2=items[i].Y2;
  newDialog[i].Focus=items[i].Focus;
  newDialog[i].Selected=items[i].Selected;
  newDialog[i].Flags=items[i].Flags;
  newDialog[i].DefaultButton=items[i].DefaultButton;
  if(items[i].Data >= 0)
    strncpy(newDialog[i].Data,GetMsg((int)items[i].Data),sizeof(newDialog[0].Data));
  else
    newDialog[i].Data[0] = 0;
  }
return newDialog;
}

//============ internal utility functions =======================

//=============== Configuration related =========================

//---------------------------------------------------------------

#define PARAMNAMELEN (40)
struct formattingParams {
  unsigned int leftMargin;
  unsigned int paragraphIndent;
  unsigned int rightMargin;
  unsigned char formattingType;
  unsigned char paragraphDetection;
  bool separateParagraphs;
  char templateName[PARAMNAMELEN];
  };

//---------------------------------------------------------------

void SetDefaultParameters(struct formattingParams * const dst) {
const struct formattingParams defaultParameters = { 1, 4, 70, msgLeft, msgNone, 0, "Default" };
*dst = defaultParameters;
}

//---------------------------------------------------------------

bool CheckParameters(struct formattingParams * const dst) {
if( dst->leftMargin      &&
    dst->paragraphIndent &&
    dst->rightMargin > dst->paragraphIndent &&
    dst->rightMargin > dst->leftMargin &&
    dst->formattingType >= msgLeft     &&
    dst->formattingType <= msgForceJustify &&
    dst->paragraphDetection >= msgNone     &&
    dst->paragraphDetection <= msgWhitespace ) return true;
SetDefaultParameters(dst);
return false;
}

//---------------------------------------------------------------
#define defPluginKeyName  "\\FBlock"

bool LoadAndCheckParameters(struct formattingParams * const dst,const int templateNumber) {
HKEY hKey;
char strBuf[512];

strncpy(strBuf,startupInfo.RootKey,sizeof(strBuf));
strncat(strBuf,defPluginKeyName,sizeof(strBuf));
SetDefaultParameters(dst);
if (RegOpenKeyEx(HKEY_CURRENT_USER,strBuf,0,KEY_QUERY_VALUE,&hKey)!=ERROR_SUCCESS) return false;
DWORD size = sizeof(*dst);
int retCode = RegQueryValueEx(hKey,FARitoa(templateNumber,strBuf,10),0,NULL,(BYTE*)dst,&size);
RegCloseKey(hKey);
dst->templateName[sizeof(dst->templateName) - 1] = 0;
return ((retCode==ERROR_SUCCESS) && CheckParameters(dst));
}

//---------------------------------------------------------------

bool CheckAndSaveParameters(struct formattingParams * const par,const int templateNumber) {
if (!CheckParameters(par)) return false;
HKEY hKey;
char strBuf[512];
DWORD ignore;

strncpy(strBuf,startupInfo.RootKey,sizeof(strBuf));
strncat(strBuf,defPluginKeyName,sizeof(strBuf));
  
if (RegCreateKeyEx(HKEY_CURRENT_USER,strBuf,0,NULL,0,KEY_WRITE,
                   NULL,&hKey,&ignore)!=ERROR_SUCCESS) return false;
RegSetValueEx(hKey,FARitoa(templateNumber,strBuf,10),0,REG_BINARY,(BYTE*)par,sizeof(*par));
RegCloseKey(hKey);
return true;
}

//---------------------------------------------------------------

bool DeleteParameters(const int templateNumber) {
HKEY hKey;
char strBuf[512];

strncpy(strBuf,startupInfo.RootKey,sizeof(strBuf));
strncat(strBuf,defPluginKeyName,sizeof(strBuf));
if (RegOpenKeyEx(HKEY_CURRENT_USER,strBuf,0,
                 KEY_SET_VALUE,&hKey)!=ERROR_SUCCESS) return false;
int retCode = RegDeleteValue(hKey,FARitoa(templateNumber,strBuf,10));
RegCloseKey(hKey);
return (retCode==ERROR_SUCCESS);
}

//---------------------------------------------------------------

void ParamsToDialog(const struct InitDialogItem * const dialog,struct FarDialogItem * const farDialog,unsigned short int i,
                    struct formattingParams * const params) {
while(--i) {
  if(farDialog[i].Type == DI_RADIOBUTTON) {
    if ((dialog[i].Data == params->formattingType) || (dialog[i].Data == params->paragraphDetection))
      farDialog[i].Selected = true;
    }
  else {
    switch (dialog[i].Data) {
      case msgLeftMargin:	        FARitoa(params->leftMargin,	 farDialog[i+1].Data,10); break;
      case msgRightMargin:        FARitoa(params->rightMargin,    farDialog[i+1].Data,10); break;
      case msgParagraph:          FARitoa(params->paragraphIndent,farDialog[i+1].Data,10); break;
      case msgSeparateParagraphs: farDialog[i].Selected = params->separateParagraphs;      break;
      case -2:                    strncpy(farDialog[i].Data,params->templateName,sizeof(params->templateName)); break;
      }
    }
  }
}

//---------------------------------------------------------------

void DialogToParams(struct InitDialogItem * dialog,/*const*/ struct FarDialogItem * const farDialog,unsigned short int i,
                    struct formattingParams * const params) {
while(--i) {
  if((farDialog[i].Type == DI_RADIOBUTTON) && farDialog[i].Selected) {
    if (dialog[i].Data < msgDetection)
		  params->formattingType = dialog[i].Data;
		else
			params->paragraphDetection = dialog[i].Data;
    }
  else {
    switch (dialog[i].Data) {
      case msgLeftMargin:					params->leftMargin         = FARatoi(farDialog[i+1].Data); break;
      case msgRightMargin:        params->rightMargin        = FARatoi(farDialog[i+1].Data); break;
      case msgParagraph:          params->paragraphIndent    = FARatoi(farDialog[i+1].Data); break;
      case msgSeparateParagraphs: params->separateParagraphs = (farDialog[i].Selected)!=0;   break;
      case -2:                    strncpy(params->templateName,farDialog[i].Data,sizeof(params->templateName)); break;
      }
    }
  }
}

//============ Block formatting related =========================

#define IsWhitespace(c)		(((c)==' ') || ((c)=='\t'))

//---------------------------------------------------------------

int UpdateBuffer(char* const buf,int truelen,const int fulllen) {
int j=0;

if(IsWhitespace(buf[truelen])) truelen++;
while(truelen<fulllen) {
  buf[j++] = buf[truelen++];
  }
return j;
}

//---------------------------------------------------------------

char* AddSpaces(char* ptr,signed int n) {
while((n--)>0) *ptr++=' ';
return(ptr);
}

//---------------------------------------------------------------

void InsertFormattedString(const char *const srcString,const int srcLength,
                           const int leftMargin,const int rightMargin,
                           const int formattingType,const int y,
                           const char * const eol) {

char* const buf = (char*)malloc(rightMargin);  // debug: add checking here
int leftIndent = leftMargin;

switch(formattingType) {
  case msgRight:
    leftIndent = rightMargin - srcLength + 1;
    break;
  case msgCenter:   
    leftIndent = ((rightMargin + leftMargin - srcLength) / 2) + 1;
  }

char *ptr = AddSpaces(buf,leftIndent - 1);

if( (formattingType==msgFullJustify) ||
    (formattingType==msgForceJustify) ) {
  int i;
  unsigned int currentSpaces = 0;
  for(i = 0;i < srcLength;i++) {
    if(srcString[i]==' ') currentSpaces++;
    }
  const unsigned int additionalSpaces = rightMargin - leftMargin - srcLength + 1;
  const unsigned int additionalSpaces1 = currentSpaces ? additionalSpaces / currentSpaces : 0;
  signed int additionalSpaces2 = currentSpaces - 
                                 (currentSpaces ? additionalSpaces % currentSpaces : 0);
   
  for(i = 0;i < srcLength;i++) {
    if(srcString[i]==' ') {
    ptr = AddSpaces(ptr,(( (additionalSpaces2--) > 0) ? 1 : 2) + additionalSpaces1 );
      }
    else {
      *ptr++ = srcString[i];
      }
    }
  InsertString(y,buf,ptr - buf,eol);
  }
else {
  memcpy(ptr,srcString,srcLength);
  InsertString(y,buf,leftIndent - 1 + srcLength ,eol);
  }
free(buf);
}

//---------------------------------------------------------------

void SaveLastLine(unsigned int fullStringLength, const unsigned int leftIndent,
		  const char* const pureString, const char* const strEOL,
		  int *currentInputLine, int *currentOutputLine,
		  const struct formattingParams* const params) {

if(!fullStringLength) return; // Nothing to do!

// remove trailing whitespace in the last line
if(IsWhitespace(pureString[fullStringLength-1])) fullStringLength--;

if(!fullStringLength) return; // Nothing to do!

InsertFormattedString(pureString,fullStringLength,leftIndent,params->rightMargin,
                      ((params->formattingType==msgFullJustify) ? msgLeft : params->formattingType ),
                      *currentOutputLine,strEOL);
(*currentInputLine)++; 
(*currentOutputLine)++;

if(params->separateParagraphs) {
  InsertString(*currentOutputLine,"",0,strEOL);
	(*currentInputLine)++; 
	(*currentOutputLine)++;
  }
}

//---------------------------------------------------------------
 
bool IsFirstLine(const struct formattingParams* const params,const struct EditorGetString* const str) {

switch(params->paragraphDetection) {
	case msgWhitespace:								// empty line starts new paragraph
		return (((str->StringLength)==0) || IsWhitespace(*(str->StringText)));
	case msgBlankLine:								// empty line starts new paragraph
		return ((str->StringLength)==0);
	case msgParagraphPerLine:	// each line starts new paragraph
		return true;
	}
return false;
}

bool IsWorkDone(const struct EditorInfo* const eInfo,const struct EditorGetString* const str) {
if(eInfo->BlockType==BTYPE_NONE) return true;							// no selection = nothing more to do
if((str->SelStart==-1) || (str->SelEnd==0)) return true;	// current line has selection in it
return false;
}

//---------------------------------------------------------------

void PerformFormatting(const struct formattingParams* const params) {
struct EditorInfo eInfo;
GetEditorInfo(&eInfo);

struct EditorGetString str;
int currentInputLine = (eInfo.BlockType==BTYPE_NONE) ? eInfo.CurLine : eInfo.BlockStartLine;
int currentOutputLine = currentInputLine;
char* pureString = (char*)malloc(sizeof(char) * (2 + params->rightMargin - min(params->paragraphIndent,params->leftMargin) ));

unsigned int requiredLength, leftIndent, pureStringLength, fullStringLength = 0;
bool processingWord,firstLoop = true;

for(;;) { // string loop
  // load the string to be processed
  GetString(currentInputLine,&str);
  const char* const currentString = str.StringText;

  if(!firstLoop && IsWorkDone(&eInfo,&str)) { // At least one loop MUST happen
    SaveLastLine(fullStringLength,leftIndent,pureString,str.StringEOL,
									&currentInputLine,&currentOutputLine,params);
    ReleaseString(&str); // clean up
    break;
    }

  // Start a new paragraph	
  if(firstLoop || IsFirstLine(params,&str)) { // We ALWAYS start with a paragraph
    // save last line of previous paragraph (if none, the function will do nothing)
    SaveLastLine(fullStringLength,leftIndent,pureString,str.StringEOL,
									&currentInputLine,&currentOutputLine,params);
    // assign intitial values to all variables
    leftIndent = (params->formattingType==msgCenter) ? params->leftMargin : params->paragraphIndent;
    requiredLength = params->rightMargin - leftIndent + 2;
    processingWord = false;
    pureStringLength = fullStringLength = 0;
    }

  // process the current line
  for(int i=-1;i < str.StringLength;i++) { // character loop
    if (i < 0 || IsWhitespace(currentString[i])) { // line always starts with whitespace
      if(processingWord) { // whitespace ends the word
        pureStringLength = fullStringLength;
        pureString[fullStringLength++] = ' ';
        processingWord = false;
        }
      // else: consequential whitespaces are ignored
      }
    else { // non-whitespace is a part of word
      pureString[fullStringLength++] = currentString[i];
      processingWord = true;
      }
    if(fullStringLength >= requiredLength) {
      if(!pureStringLength) {
        char* errorMessage[] = { GetMsg(msgError1), GetMsg(msgError2),
                                 GetMsg(msgError3), GetMsg(msgOK) };
        ShowWarning(errorMessage,sizeof(errorMessage)/sizeof(errorMessage[0]),"WordTooLong");
        ReleaseString(&str);
        free(pureString);
        return;
        }
      InsertFormattedString(pureString,pureStringLength,
                            leftIndent,params->rightMargin,params->formattingType,
                            currentOutputLine,str.StringEOL);
      currentOutputLine++;
      currentInputLine++; // cause text got shifted down

      fullStringLength = UpdateBuffer(pureString,pureStringLength,fullStringLength);
      pureStringLength = 0;

      leftIndent = params->leftMargin;
      requiredLength = params->rightMargin - leftIndent + 2;
      }
    }	// character loop
  DeleteString(currentInputLine); // string processed 
  ReleaseString(&str);
  firstLoop = false;
  } // string loop

free(pureString);

UnselectBlock();
}

//---------------------------------------------------------------

bool CallTemplateDialog(struct formattingParams * params) {

#define TEMPLATEDLGWIDTH	(55)
#define TEMPLATEDLGHEIGHT	(20)

struct InitDialogItem templateDialog[] = {
  { DI_DOUBLEBOX,  3, 1,TEMPLATEDLGWIDTH-4,TEMPLATEDLGHEIGHT-2,0,0,0,0,msgTemplate },
  { DI_EDIT,       5, 2,49,0,1,0,0,        0,-2 },
  { DI_TEXT,       0, 3, 0,0,0,0,DIF_SEPARATOR|DIF_BOXCOLOR,0,-1},
  { DI_TEXT,       5, 4, 0,0,0,0,0,        0,msgLeftMargin },
  { DI_EDIT,      20, 4,23,0,0,0,0,        0,-1 },
  { DI_TEXT,       5, 5, 0,0,0,0,0,        0,msgParagraph },
  { DI_EDIT,      20, 5,23,0,0,0,0,        0,-1 },
  { DI_TEXT,       5, 6, 0,0,0,0,0,        0,msgRightMargin },
  { DI_EDIT,      20, 6,23,0,0,0,0,        0,-1 },
  { DI_TEXT,       0, 7, 0,0,0,0,DIF_SEPARATOR|DIF_BOXCOLOR,0,-1 },
  { DI_TEXT,       5, 8, 0,0,0,0,0,        0,msgAlignment },
  { DI_RADIOBUTTON,7, 9, 0,0,0,0,DIF_GROUP,0,msgLeft },
  { DI_RADIOBUTTON,7,10, 0,0,0,0,0,        0,msgRight },
  { DI_RADIOBUTTON,7,11, 0,0,0,0,0,        0,msgCenter },
  { DI_RADIOBUTTON,7,12, 0,0,0,0,0,        0,msgFullJustify },
  { DI_RADIOBUTTON,7,13, 0,0,0,0,0,        0,msgForceJustify },

  { DI_TEXT,       26, 8, 0,0,0,0,0,        0,msgDetection },
  { DI_RADIOBUTTON,28, 9, 0,0,0,0,DIF_GROUP,0,msgNone },
  { DI_RADIOBUTTON,28,10, 0,0,0,0,0,        0,msgParagraphPerLine },
  { DI_RADIOBUTTON,28,11, 0,0,0,0,0,        0,msgBlankLine },
  { DI_RADIOBUTTON,28,12, 0,0,0,0,0,        0,msgWhitespace },

  { DI_TEXT,       0,14, 0,0,0,0,DIF_SEPARATOR|DIF_BOXCOLOR,0,-1},
  { DI_CHECKBOX,   7,15, 0,0,0,0,0,        0,msgSeparateParagraphs },
  { DI_TEXT,       0,16, 0,0,0,0,DIF_SEPARATOR|DIF_BOXCOLOR,0,-1},
  { DI_BUTTON,     0,17, 0,0,0,0,DIF_CENTERGROUP,1,msgOK },
  { DI_BUTTON,     0,17, 0,0,0,0,DIF_CENTERGROUP,0,msgCancel }
};

const int dialogSize = sizeof(templateDialog)/sizeof(templateDialog[0]);
struct FarDialogItem farTemplateDialog[dialogSize];

ConvertDialog(templateDialog,farTemplateDialog,dialogSize);
ParamsToDialog(templateDialog,farTemplateDialog,dialogSize,params);
int retCode = CenterDialog(TEMPLATEDLGWIDTH,TEMPLATEDLGHEIGHT,"Contents",farTemplateDialog,dialogSize);
if ((retCode < 0) || (templateDialog[retCode].Data == msgCancel)) return(false);
DialogToParams(templateDialog,farTemplateDialog,dialogSize,params);
return true;
}

//---------------------------------------------------------------

void CallTemplateMenu(struct formattingParams* const dstParams) {
int selectedItem = 0;

for(;;) {
  struct FarMenuItem * menuItems = NULL;
  struct formattingParams tmpParams;
  int i = 0;

  while(LoadAndCheckParameters(&tmpParams,i + 1)) {
    menuItems = (struct FarMenuItem *) realloc(menuItems,sizeof(struct FarMenuItem) * (i + 1));
    menuItems[i].Checked = menuItems[i].Separator = 0;
    menuItems[i].Selected = (i == selectedItem);
    strncpy(menuItems[i].Text,tmpParams.templateName,sizeof(menuItems[0].Text));
    i++;
    }
  /*const*/ int usedKeys[] = { VK_INSERT, VK_F4, VK_DELETE, 0 };

  bool validParams;
  int retKey,retCode = CenterMenu(GetMsg(msgTemplate2),"Ins, F4, Del","Templates",usedKeys,&retKey,
                                  menuItems,i);
  free(menuItems);

  validParams = LoadAndCheckParameters(&tmpParams,retCode + 1);
  switch (retKey) {
    case -1:  // Enter, Esc, F10
             if (retCode == -1) return;  // Esc, F10
             if(validParams) {
               *dstParams = tmpParams;
               return;
               }
             break;
    case 0:  // Insert
             SetDefaultParameters(&tmpParams);
             retCode = i;
    case 1:   // F4
             if (retCode == -1) continue;
             if(CallTemplateDialog(&tmpParams))
               CheckAndSaveParameters(&tmpParams,retCode + 1);
             selectedItem = retCode;
             break;
    case 2:   // Delete
             if (retCode == -1) continue;
             selectedItem = retCode - 1;
             if(validParams) {
               char* question[] = { GetMsg(msgDelete1), GetMsg(msgDelete2),NULL,
                                    GetMsg(msgOK), GetMsg(msgCancel) };
               question[2] = tmpParams.templateName;
               if (!ShowQuestion(question,sizeof(question)/sizeof(question[0]),NULL))
                 while(++retCode < i) {
                   LoadAndCheckParameters(&tmpParams,retCode + 1);
                   CheckAndSaveParameters(&tmpParams,retCode);
                   }
                 DeleteParameters(retCode);
                 }
             break;           
    }
  }
}

//=================== exported functions ========================

void WINAPI _export SetStartupInfo(struct PluginStartupInfo *Info) {
startupInfo=*Info;
#if (FARMANAGERVERSION >= 0x0146)
FSF = *(Info->FSF);
#endif // FARMANAGERVERSION
}
//---------------------------------------------------------------
void WINAPI _export GetPluginInfo(struct PluginInfo *pluginInfo) {
pluginInfo->StructSize = sizeof(*pluginInfo);
pluginInfo->Flags = PF_EDITOR|PF_DISABLEPANELS;
pluginInfo->DiskMenuStringsNumber = 0;
static char *pluginMenuStrings[1];
pluginMenuStrings[0] = GetMsg(msgPluginName);
pluginInfo->PluginMenuStrings = pluginMenuStrings;
pluginInfo->PluginMenuStringsNumber = sizeof(pluginMenuStrings)/sizeof(pluginMenuStrings[0]);
pluginInfo->PluginConfigStringsNumber = 0;
}
//---------------------------------------------------------------

HANDLE WINAPI _export OpenPlugin(int OpenFrom,int Item) {
struct formattingParams params = {0,0,0,0,0,0,"Empty"};
LoadAndCheckParameters(&params,0);

#define MAINDLGWIDTH	(55)
#define MAINDLGHEIGHT	(18)

for(;;) {
  struct InitDialogItem mainDialog[] = {
    { DI_DOUBLEBOX,  3, 1,MAINDLGWIDTH-4,MAINDLGHEIGHT-2,0,0,0,0,msgPluginName },
    { DI_TEXT,       5, 2, 0,0,0,0,0,        0,msgLeftMargin },
    { DI_EDIT,      20, 2,23,0,1,0,0,        0,-1 },
    { DI_TEXT,       5, 3, 0,0,0,0,0,        0,msgParagraph },
    { DI_EDIT,      20, 3,23,0,0,0,0,        0,-1 },
    { DI_TEXT,       5, 4, 0,0,0,0,0,        0,msgRightMargin },
    { DI_EDIT,      20, 4,23,0,0,0,0,        0,-1 },
    { DI_TEXT,       0, 5, 0,0,0,0,DIF_SEPARATOR|DIF_BOXCOLOR,0,-1 },

    { DI_TEXT,       5, 6, 0,0,0,0,0,        0,msgAlignment },
    { DI_RADIOBUTTON,7, 7, 0,0,0,0,DIF_GROUP,0,msgLeft },
    { DI_RADIOBUTTON,7, 8, 0,0,0,0,0,        0,msgRight },
    { DI_RADIOBUTTON,7, 9, 0,0,0,0,0,        0,msgCenter },
    { DI_RADIOBUTTON,7,10, 0,0,0,0,0,        0,msgFullJustify },
    { DI_RADIOBUTTON,7,11, 0,0,0,0,0,        0,msgForceJustify },

    { DI_TEXT,       26, 6, 0,0,0,0,0,        0,msgDetection },
    { DI_RADIOBUTTON,29, 7, 0,0,0,0,DIF_GROUP,0,msgNone },
    { DI_RADIOBUTTON,29, 8, 0,0,0,0,0,        0,msgParagraphPerLine },
    { DI_RADIOBUTTON,29, 9, 0,0,0,0,0,        0,msgBlankLine },
    { DI_RADIOBUTTON,29,10, 0,0,0,0,0,        0,msgWhitespace },
    
    { DI_TEXT,       0,12, 0,0,0,0,DIF_SEPARATOR|DIF_BOXCOLOR,0,-1},
    { DI_CHECKBOX,   7,13, 0,0,0,0,0,        0,msgSeparateParagraphs },
    { DI_TEXT,       0,14, 0,0,0,0,DIF_SEPARATOR|DIF_BOXCOLOR,0,-1},
    { DI_BUTTON,     0,15, 0,0,0,0,DIF_CENTERGROUP,0,msgTemplate },
    { DI_BUTTON,     0,15, 0,0,0,0,DIF_CENTERGROUP,1,msgOK },
    { DI_BUTTON,     0,15, 0,0,0,0,DIF_CENTERGROUP,0,msgCancel }
  };

  const int dialogSize = sizeof(mainDialog)/sizeof(mainDialog[0]);
  struct FarDialogItem farMainDialog[dialogSize];

  ConvertDialog(mainDialog,farMainDialog,dialogSize);
  ParamsToDialog(mainDialog,farMainDialog,dialogSize,&params);
  for(unsigned int i=0;i<dialogSize;i++) {
    if((farMainDialog[i].Type == DI_RADIOBUTTON) &&
       (params.formattingType == mainDialog[i].Data)) {
      farMainDialog[i].Selected = true;
      break;
      }
    }

  int retCode = CenterDialog(MAINDLGWIDTH,MAINDLGHEIGHT,"Contents",farMainDialog,dialogSize);
  if ((retCode < 0) || (mainDialog[retCode].Data == msgCancel)) return(INVALID_HANDLE_VALUE);
  if (mainDialog[retCode].Data != msgTemplate) {
    DialogToParams(mainDialog,farMainDialog,dialogSize,&params);
    CheckAndSaveParameters(&params,0);

    PerformFormatting(&params);
    return(INVALID_HANDLE_VALUE);
    }
  CallTemplateMenu(&params);
  }
}

//#include <stdio.h>
//void sprintbuf(char *b,char *buf,int len) { while(len--) *b++=*buf++; } int zzzz=20;
//void alert(char* zzz) { char* msg[2] = { "msgBlankLine","msgWhitespace" }; msg[1]=zzz; startupInfo.Message(startupInfo.ModuleNumber,FMSG_WARNING,NULL,msg,2,1); }
