/*  Express Extension Definition Utility

    Copyright (c) Express Software 1998.
    All Rights Reserved.
    
    Created by: Joseph Cosentino.
    
*/

// I N C L U D E S //////////////////////////////////////////////////////////

#include <windows.h>
#include <stdio.h>
#include <string.h>
#include <stdlib.h>
#include <io.h>
#include "define.h"

// D E F I N E S ////////////////////////////////////////////////////////////

#define MAXFIELDLENGTH 80                   // Maximum data field length.
#define MAXDATAENTRIES 200                  // Maximum number of records.
#define SEARCH_DATA    WM_USER+1            // Your own messages.

// S T R U C T U R E S //////////////////////////////////////////////////////

// Type definition of data records used.
typedef struct tagDataRecord_t
{
    char szExtension[MAXFIELDLENGTH];
    char szDescript[MAXFIELDLENGTH];
    short nMedium;
    unsigned wIndex;
    BOOL bUsed;

} DataRecord_t;

// G L O B A L S ////////////////////////////////////////////////////////////

HANDLE hInst;
HWND hwMain, hDlgModeless=NULL, hDlgModal=NULL;
HMENU hMainMenu;
char szApplName[MAXFIELDLENGTH];
BOOL bDlgModeless = FALSE;
FILE *FilePointer;
DataRecord_t DataArray[MAXDATAENTRIES];
DataRecord_t DataRecord;
short nRecNr=0, nSearchRecNr;
FARPROC lpfnModal, lpfnModeless;

// P R O T O T Y P E S //////////////////////////////////////////////////////

LONG FAR PASCAL MainWndProc(HWND, unsigned, WORD, LONG);
BOOL FAR PASCAL MdfAboutProc(HWND, unsigned, WORD, LONG);
BOOL FAR PASCAL MdfExtensionProc(HWND, unsigned, WORD, LONG);
BOOL FAR PASCAL NdfSearchProc(HWND, unsigned, WORD, LONG);
BOOL CompareStrings(char *, char *, BOOL, BOOL);
BOOL ReadData(void);
BOOL WriteData(void);
int  InitFirstInstance(HANDLE ,HANDLE, int);
void ReadExtensionData(HWND, DataRecord_t *);
void DisplayData(HWND, DataRecord_t *);
void PaintDialog(HWND);

// F U N C T I O N S ////////////////////////////////////////////////////////

int PASCAL WinMain(HANDLE hInstance,  HANDLE hPrevInstance, LPSTR lpszCmdLine, int nCmdShow)
{
    MSG msg;

    if (!hPrevInstance)
        {
        if (!InitFirstInstance(hInstance, hPrevInstance, nCmdShow))
            return NULL;

        } // end if.
    else
        {
        return NULL;
        } // end else.

    hInst = hInstance;
    hwMain = CreateWindow(szApplName, "Express Extension Definition Utility", WS_OVERLAPPEDWINDOW, CW_USEDEFAULT, CW_USEDEFAULT, CW_USEDEFAULT, CW_USEDEFAULT, NULL, LoadMenu(hInstance, "MainMenu"), hInstance, NULL);

    if (!hwMain)
        return NULL;

    // Execute initialization tasks.
    FilePointer = fopen("EXT.DAT", "rb");
    if (FilePointer == NULL)
        {
        fclose(FilePointer);
        memset(DataArray, 0, sizeof(DataArray));
        WriteData();
        } // end if.
    else
        fclose(FilePointer);

    // Get handle to main menu.
    hMainMenu = GetMenu(hwMain);

    // Read data records.
    ReadData();

    // Display main window.
    ShowWindow(hwMain, nCmdShow);
    UpdateWindow(hwMain);

    // Message loop.
    while (GetMessage(&msg, NULL, 0, 0)) // Message reading.
        {
        if (!(hDlgModeless != NULL && IsDialogMessage(hDlgModeless, &msg)))
            {
            TranslateMessage(&msg);
            DispatchMessage(&msg);
            } // end if.
        
        } // end while.
        
    return (msg.wParam);                // End define.

} // end WinMain.

/////////////////////////////////////////////////////////////////////////////

int InitFirstInstance(HANDLE hInstance, HANDLE hPrevInstance, int nCmdShow)
{
    WNDCLASS MainWndClass;

    // Load application name from resource file.
    LoadString(hInstance, IDS_APPLNAME, (LPSTR)szApplName, 20);

    // Specify window class information
    MainWndClass.style = 0;
    MainWndClass.lpfnWndProc = MainWndProc;
    MainWndClass.cbClsExtra = 0;
    MainWndClass.cbWndExtra = 0;
    MainWndClass.hInstance = hInstance;
    MainWndClass.hIcon = LoadIcon(hInstance, "DEFINEICON");
    MainWndClass.hCursor = LoadCursor(NULL, IDC_ARROW);
    MainWndClass.hbrBackground = GetStockObject(GRAY_BRUSH);
    MainWndClass.lpszMenuName = (LPSTR)NULL;
    MainWndClass.lpszClassName = szApplName;

    // Registering "Define" window class.
    return (RegisterClass(&MainWndClass));

} // end InitFirstInstance.

/////////////////////////////////////////////////////////////////////////////

LONG FAR PASCAL MainWndProc(HWND hWnd, unsigned msg, WORD wP, LONG lP)
{
    int nResult;

    switch (msg)
        {
        case WM_CREATE:                 // For creating window.
            memset(&DataRecord, 0, sizeof(DataRecord));
            break;

        case WM_DESTROY:
            PostQuitMessage(0);         // Send WM_QUIT if window is destroyed.
            break;

        case WM_COMMAND:                // Messages from menu bar.
            switch (wP)
                {
                case MI_QUIT:           // End program.
                    PostMessage(hwMain, WM_SYSCOMMAND, SC_CLOSE, 0L);
                    break;
                
                case MI_INDEX:          // Help.
                    WinHelp(hWnd, "define.hlp", HELP_CONTENTS, 0L);
                    return 0L;
                    break;
                
                case MI_ABOUT:          // Call "About" dialog box.
                    lpfnModal = MakeProcInstance(MdfAboutProc, hInst);
                    nResult = DialogBox(hInst, "ABOUTDLG", hwMain, lpfnModal);
                    FreeProcInstance(lpfnModal);
                    break;

                case MI_ENTERDATA:      // Call "Enter Data" dialog box.
                    lpfnModal = MakeProcInstance(MdfExtensionProc, hInst);
                    nResult = DialogBox(hInst, "ADDEXTDLG", hwMain, lpfnModal);
                    FreeProcInstance(lpfnModal);
                    break;

                case MI_SEARCHDATA:     // Call "Search for Data" dialog box.
                    if (!bDlgModeless)  // Dialog box must be displayed.
                        {
                        lpfnModeless = MakeProcInstance(NdfSearchProc, hInst);
                        hDlgModeless = CreateDialog(hInst, "SEARCHDLG", hwMain, lpfnModeless);
                        bDlgModeless = TRUE;
                        } // end if.
                    else
                        {
                        DestroyWindow(hDlgModeless);
                        FreeProcInstance(lpfnModeless);
                        bDlgModeless = FALSE;
                        hDlgModeless = NULL;
                        } // end else.
                    
                    break;
       
                default:
                    break;
                
                } // end switch.
                
            break;

        case WM_PAINT:                  // Client area update needed.
            PaintDialog(hWnd);
            break;

        default:                        // Pass other messages to default window function.
            return (DefWindowProc(hWnd, msg, wP, lP));
            break;
        
        } // end switch.
        
    return 0L;

} // end MainWndProc.

/////////////////////////////////////////////////////////////////////////////

BOOL FAR PASCAL MdfAboutProc(HWND hDlg, unsigned msg, WORD wP, LONG lP)
{
    switch (msg)
        {
        case WM_INITDIALOG:             // Dialog box display message.
            return TRUE;                // Do not execute initialization.
            break;

        case WM_COMMAND:                // Dialog box control message.
            switch (wP)
                {
                case IDOK:              // Click on OK button.
                    EndDialog(hDlg, TRUE);
                    return TRUE;
                    break;
                
                case PB_CREDITS:        // Credits.
                    CreateDialog(hInst, "CREDITS", hwMain, lpfnModal);
                    return TRUE;
                    break;
                
                default:
                    return FALSE;
                    break;
                
                } // end switch.
                
            break;

        default:
            return FALSE;
            break;
        
        } // end switch.
        
} // end MdfAboutProc.

/////////////////////////////////////////////////////////////////////////////

BOOL FAR PASCAL MdfExtensionProc(HWND hDlg, unsigned msg, WORD wP, LONG lP)
{
    int i;
    BOOL bFound;

    switch (msg)
        {
        case WM_INITDIALOG:             // Dialog box display message.
            hDlgModal = hDlg;           // Mark dialog box handle.

            // Does nRecNr point to a valid data record?
            if (!DataArray[nRecNr].bUsed)
                {
                // If not, display first record found in data field.
                for (i=0;i<MAXDATAENTRIES;i++)
                    {
                    if (DataArray[i].bUsed)
                        {
                        nRecNr = i;
                        break;
                        } // end if.
                        
                    } // end for.
                
                } // end if.
            
            DataRecord = DataArray[nRecNr];
            DisplayData(hDlg, &DataRecord);

            // Set input focus to first edit field.
            SetFocus(GetDlgItem(hDlg, EB_EXTENSION));
            return FALSE;               // Return FALSE unless SetFocus.
            break;

        case SEARCH_DATA:
            nRecNr = (short)wP;
            DataRecord = DataArray[nRecNr];
            DisplayData(hDlg, &DataRecord);
            return TRUE;
            break;

        case WM_COMMAND:                // Message from dialog box controls.
            switch (wP)
                {
                case IDOK:              // Click on OK button.
                    WriteData();
                    EndDialog(hDlg, TRUE);
                    hDlgModal = NULL;
                    return TRUE;
                    break;

                case PB_NEW:            // Save a new data record.
                    if (MessageBox(hDlg, "Add record - are you sure?", "Add Record", MB_ICONQUESTION | MB_YESNO) == IDYES)
                        {
                        ReadExtensionData(hDlg, &DataRecord); // Read values.
      
                        // Search for a free data record in DataArray
                        bFound = FALSE;
                        for (i=0;i<MAXDATAENTRIES;i++)
                            {
                            if (!DataArray[i].bUsed)    // Free entry.
                                {
                                DataArray[i] = DataRecord;
                                DataArray[i].bUsed = TRUE;
                                bFound = TRUE;
                                nRecNr = i;
                                break;
                                } // end if.
                                
                            } // end for.
                        
                        if (!bFound)
                            MessageBox(hDlg, "No more fields available", "Error", MB_OK | MB_ICONEXCLAMATION);
                        
                        } // end if.
                    
                    return TRUE;
                    break;

                case PB_CHANGE:         // Change an existing data record.
                    if (MessageBox(hDlg, "Edit record - are you sure?", "Edit Record", MB_ICONQUESTION | MB_YESNO) == IDYES)
                        {
                        ReadExtensionData(hDlg, &DataRecord); // Read values
                        DataArray[nRecNr] = DataRecord;   // Move to data records - fields
                        } // end if.
                        
                    return TRUE;
                    break;

                case PB_DELETE:         // Delete a data record.
                    if (MessageBox(hDlg, "Delete record - are you sure?", "Delete Record", MB_ICONQUESTION | MB_YESNO) == IDYES)
                        DataArray[nRecNr].bUsed = FALSE;

                    // Go to next data record
                    PostMessage(hDlg, WM_COMMAND, PB_FORWARD, 0L);
                    return TRUE;
                    break;

                case PB_FORWARD:        // Next data record.
                    i = nRecNr+1;
                    if (i >= MAXDATAENTRIES)
                        i = 0;
                    
                    while (!DataArray[i].bUsed)
                        {
                        i++;
                        if (i >= MAXDATAENTRIES)
                            i=0;
                        
                        if (i == nRecNr)
                            break;
                        
                        } // end while.
                        
                    if (DataArray[i].bUsed)
                        {
                        nRecNr = i;
                        DataRecord = DataArray[i];
                        DisplayData(hDlg, &DataRecord);
                        } // end if.
                    else
                        MessageBox(GetFocus(), "No records found", szApplName, MB_ICONEXCLAMATION | MB_OK);
                
                    return TRUE;
                    break;

                case PB_BACKWARD:       // Previous data record.
                    i = nRecNr-1;
                    if (i < 0)
                        i = MAXDATAENTRIES-1; 
                    
                    while (!DataArray[i].bUsed)
                        {
                        i--;
                        if (i < 0)
                            i=MAXDATAENTRIES-1;
                        
                        if (i == nRecNr)
                            break;
                        
                        } // end while.
                        
                    if (DataArray[i].bUsed)
                        {
                        nRecNr = i;
                        DataRecord = DataArray[i];
                        DisplayData(hDlg, &DataRecord);
                        } // end if.
                    else
                        MessageBox(GetFocus(), "No records found", szApplName, MB_ICONEXCLAMATION | MB_OK);
                    
                    return TRUE;
                    break;

                default:
                    return FALSE;
                    break;
                
                } // end switch.
                
            break;

        default:
            return FALSE;
            break;
        
        } // end switch.
        
} // end MdfExtensionProc.

/////////////////////////////////////////////////////////////////////////////

BOOL FAR PASCAL NdfSearchProc(HWND hDlg, unsigned msg, WORD wP, LONG lP)
{
    int i;
    static BOOL bCBNoIgnoreCase, bCBWholeWord;
    BOOL bExtension, bDescription;
    char szDescript[MAXFIELDLENGTH], szExtension[MAXFIELDLENGTH], szListboxString[255];
          
    switch (msg)
        {
        case WM_INITDIALOG:             // Dialog box display message.
            return TRUE;
            break;

        case WM_COMMAND:                // Dialog box control message.
            switch (wP)
                {
                case PB_SEARCH:         // Select Search button.
                    // Read search fields
                    GetDlgItemText(hDlg, EB_EXTENSION, szExtension, MAXFIELDLENGTH);
                    GetDlgItemText(hDlg, EB_DESCRIPT, szDescript, MAXFIELDLENGTH);

                    // Delete preceding list field entries.
                    SendDlgItemMessage(hDlg, LB_FOUND, LB_RESETCONTENT, 0, 0L);

                    // Search data record fields.
                    for (i=0;i<MAXDATAENTRIES;i++)
                        {
                        if (DataArray[i].bUsed)  // Data record used?                     
                            {
                            bExtension = bDescription = FALSE;
                            
                            // Compare strings
                            if (CompareStrings(DataArray[i].szExtension, szExtension, bCBNoIgnoreCase, bCBWholeWord) && (szExtension[0] != 0))
                                bExtension = TRUE;
                            
                            if (CompareStrings(DataArray[i].szDescript, szDescript, bCBNoIgnoreCase, bCBWholeWord) && (szDescript[0] != 0))
                                bDescription = TRUE;

                            if ((bExtension && (szDescript[0] == 0)) || (bDescription && (szExtension[0] == 0)) || (bExtension && bDescription))
                                {
                                // Format list field entry.
                                wsprintf(szListboxString, "%-5.5s %-85.65s %65d", (LPSTR)DataArray[i].szExtension, (LPSTR)DataArray[i].szDescript, i);
                                
                                // Append entry to list field.
                                SendDlgItemMessage(hDlg, LB_FOUND, LB_ADDSTRING, 0, (LONG)(LPSTR) szListboxString);
                                } // end if.
                                
                            } // end if.
                            
                        } // end for.
                        
                    return TRUE;
                    break;

                case LB_FOUND:
                    if (HIWORD(lP) == LBN_DBLCLK)
                        {
                        // Get index for currently selected entry.
                        i = (int) SendDlgItemMessage(hDlg, LB_FOUND, LB_GETCURSEL, 0, 0L); 
                        
                        // Read entry.
                        SendDlgItemMessage(hDlg, LB_FOUND, LB_GETTEXT, i, (LONG)(LPSTR)szListboxString);
                        
                        // Specify field index.
                        sscanf(&szListboxString[57], "%d", &nSearchRecNr);
                        
                        // Adapt current data record.
                        if (DataArray[nSearchRecNr].bUsed)
                            {
                            // Open "File Entry" dialog box if record doesn't exist.
                            if (!hDlgModal)
                                {
                                nRecNr = nSearchRecNr;
                                SendMessage(hwMain, WM_COMMAND, MI_ENTERDATA, 0L);
                                } // end if.
                            else
                                PostMessage(hDlgModal, SEARCH_DATA, nSearchRecNr, 0L);
                            
                            } // end if.
                            
                        return TRUE;
                        } // end if.
                        
                    break;

                default:
                    return FALSE;
                    break;
                
                } // end switch.
                
            break;

        default:
            return FALSE;
            break;
        
        } // end switch.

} // end NdfSearchProc.

/////////////////////////////////////////////////////////////////////////////

void PaintDialog(HWND hWnd)
{
    PAINTSTRUCT ps;
    HDC hDC;

    hDC = BeginPaint(hWnd, &ps); // Get main window display context.

    // Client area output eventually appears in main window from here.
    EndPaint(hWnd, &ps);

} // end PaintDialog.

/////////////////////////////////////////////////////////////////////////////

void ReadExtensionData(HWND hDlg, DataRecord_t *DataRecord)
{
    GetDlgItemText(hDlg, EB_EXTENSION, DataRecord->szExtension, MAXFIELDLENGTH);
    GetDlgItemText(hDlg, EB_DESCRIPT, DataRecord->szDescript, MAXFIELDLENGTH);

} // end ReadExtensionData.

/////////////////////////////////////////////////////////////////////////////

BOOL CompareStrings(char *szString1, char *szString2, BOOL bNoIgnoreCase, BOOL bWholeWord)
{
    char szPattern1[MAXFIELDLENGTH], szPattern2[MAXFIELDLENGTH];
    WORD nLength1, nLength2;

    nLength1 = strlen(szString1);
    nLength2 = strlen(szString2);
    strcpy(szPattern1, szString1);
    strcpy(szPattern2, szString2);
    strupr(szPattern1);
    strupr(szPattern2);
    if (!bWholeWord)                    // Compare strings as whole words only.
        {
        szPattern1[min(nLength1, nLength2)] = '\0';
        szPattern2[min(nLength1, nLength2)] = '\0';
        } // end if.
        
    if (!bNoIgnoreCase)                 // Ignore upper/lowercase.
        {
        // Convert both strings to uppercase.
        AnsiUpper((LPSTR)szPattern1);
        AnsiUpper((LPSTR)szPattern2);
        } // end if.
        
    // Compare.
    if (lstrcmp((LPSTR)szPattern1, (LPSTR)szPattern2) == 0)
        return TRUE;
    else
        return FALSE;

} // end CompareStrings.

/////////////////////////////////////////////////////////////////////////////

void DisplayData(HWND hDlg, DataRecord_t *DataRecord)
{
   // Initialize edit fields with current data record contents.
   SetDlgItemText(hDlg, EB_EXTENSION, DataRecord->szExtension);
   SetDlgItemText(hDlg, EB_DESCRIPT, DataRecord->szDescript);

} // end DisplayData.

/////////////////////////////////////////////////////////////////////////////

BOOL WriteData(void)
{
    int nItems;

    FilePointer = fopen("EXT.DAT", "wb");
    if (!FilePointer)
        {
        MessageBox(GetFocus(), "File could not be opened", "Error", MB_OK | MB_ICONEXCLAMATION);
        return FALSE;
        } // end if.
        
    SetCursor(LoadCursor(NULL, IDC_WAIT));

    // Save data records.
    nItems = fwrite(DataArray, sizeof(DataRecord), MAXDATAENTRIES, FilePointer);

    fclose(FilePointer);
    SetCursor(LoadCursor(NULL, IDC_ARROW));

    // All data records written?
    if (nItems == MAXDATAENTRIES)
        return TRUE;
    else
        return FALSE;

} // end WriteData.

/////////////////////////////////////////////////////////////////////////////

BOOL ReadData(void)
{
    int nItems;

    FilePointer = fopen("EXT.DAT", "rb");
    if (!FilePointer)
        {
        MessageBox(GetFocus(), "File could not be opened", "Error", MB_OK | MB_ICONEXCLAMATION);
        return FALSE;
        } // end if.
        
    SetCursor(LoadCursor(NULL, IDC_WAIT));
    nItems = fread(DataArray, sizeof(DataRecord), MAXDATAENTRIES, FilePointer);
    fclose(FilePointer);
    SetCursor(LoadCursor(NULL, IDC_ARROW));

    // All data records read?
    if (nItems == MAXDATAENTRIES)
        return TRUE;
    else
        return FALSE;

} // end ReadData.
