=head1 COPYRIGHT
/*
 * Copyright (c) Ondrej Brablc, 2001 <far@brablc.com>
 * You can use, modify, distribute this source code  or
 * any other parts of BlockPro plugin only according to
 * BlockPro  License  (see  \Doc\License.txt  for   more
 * information).
 */
=cut

use Strict;

SWITCH: for (uc(shift))
{
    /^GREP$/     && do { &Grep;               last SWITCH; };
    /^SORT$/     && do { &Sort;               last SWITCH; };
    /^UNIQ$/     && do { &Unique;             last SWITCH; };
    /^UNIQC$/    && do { &UniqueCount;        last SWITCH; };
    /^WF$/       && do { &WordFrequency;      last SWITCH; };

    /^PAIR$/     && do { &HtmlAddPair;        last SWITCH; };
    /^MATCH$/    && do { &HtmlMatch;          last SWITCH; };
    /^TABLE$/    && do { &HtmlCnvTable;       last SWITCH; };

    /^ALIGN$/    && do { &AlignBlock;         last SWITCH; };
    /^UNALIGN$/  && do { &UnAlignBlock;       last SWITCH; };
    /^CENTER$/   && do { &CenterBlock;        last SWITCH; };
    /^INDTAB$/   && do { &IndentBlockToTab;   last SWITCH; };
    /^CASE$/     && do { &ToggleCase;         last SWITCH; };
    /^RENUM$/    && do { &RenumberLines;      last SWITCH; };
    /^FILL$/     && do { &FillChar;           last SWITCH; };
    /^SUM$/      && do { &Summator;           last SWITCH; };
    /^STAT$/     && do { &TextStatistics;     last SWITCH; };
    /^ADDCMT$/   && do { &AddComment;         last SWITCH; };
    /^TEST$/     && do { &Test;               last SWITCH; };

    /^CNV$/      && do { &Convert;            last SWITCH; };
    /^MIME$/     && do { &MIME;               last SWITCH; };
    /^SYLK2CSV$/ && do { &ConvertSYLK2CSV;    last SWITCH; };
    /^DOS2UNIX$/ && do { &ConvertDOS2UNIX;    last SWITCH; };
    /^SQL$/      && do { &EmbeddedSQL;        last SWITCH; };
    /^MTCHCOL$/  && do { &MatchColumns;       last SWITCH; };
    /^DUPL$/     && do { &Duplicate;          last SWITCH; };
    /^SPGL$/     && do { &SplitAndGlue;       last SWITCH; };
    /^TMPL$/     && do { &Template;           last SWITCH; };
    /^INCCHNG$/  && do { &IncrementalChange;  last SWITCH; };

    if (/^$/)
    {
        print STDERR "Command missing!\n";
    }
    else
    {
        print STDERR "Unknown command $_!\n";
    }
    exit 1;
}

sub AlignBlock
{
    # TODO: Do not leave short words at line ends

    my $width   = 80;
    my $justify = 0;

    for (@ARGV)
    {
        /^\-w(\d+)$/ && do { $width   = int($1); next; };
        /^\-j(\d+)$/ && do { $justify = int($1); next; };
    }

    my $text = "";

    for (<STDIN>)
    {
        my $newline = 0;
        chop;
        s/\s+/ /g;
        s/\s$//g;

        $newline = 1 if /^$/;
        $text    = $text . (length($text)?" ":"") . $_ unless $newline;

        while (length($text) > $width)
        {
            my $buff = substr($text,0,$width);
            $text    = substr($text,$width);

            # If in the middle of the word then return back the nonspaces
            if ($text =~ /^\S/ && $buff =~ /^(.*)\s(\S+)$/)
            {
                $text = $2 . $text;
                $buff = $1;
            }

            $text =~ s/^\s//;
            $buff =~ s/\s$//;

            if ($justify)
            {
                my $items = scalar(split /\s/, $buff);
                if ($items>1)
                {
                    my $size = ($width - length($buff))/$items;
                    my $fill = " "x($size+2);
                    $buff =~ s/\s/${fill}/g;

                    while (length($buff) > $width)
                    {
                        my $smaller = substr( $fill, 0, length($fill)-1);
                        $buff =~ s/${fill}/$smaller/;
                    }
                }
            }

            print STDOUT "$buff\n";
        }

        if ($newline)
        {
            print STDOUT "$text\n" if length($text)>0;
            print STDOUT "\n";
            $text = "";
        }
    }

    print STDOUT "$text\n" if length($text)>0;
}

sub UnAlignBlock
{
    my $text = "";
    my @file = <STDIN>;
    my $size = scalar @file;

    for (my $i=0; $i< $size; $i++)
    {
        $_ = $file[$i];
        chop;
        $text .= " " . $_;
        if (/^$/ || $i == $size - 1)
        {
            $text =~ s/\s{2,}/ /g;
            $text =~ s/^\s+//;
            $text =~ s/\s+$//;
            print STDOUT "$text\n";
            $text = "";
        }
    }
}

sub CenterBlock
{
    my $width   = 80;

    for (@ARGV)
    {
        /^\-w(\d+)$/ && do { $width   = int($1); next; };
    }

    for (<STDIN>)
    {
        $_ =~ s/^\s*(.*?)\s*$/$1/;

        if (length($_))
        {
            print STDOUT " "x((80-length($_))/2) . $_ . "\n";
        }
        else
        {
            print STDOUT "\n";
        }
    }
}

sub IndentBlockToTab
{
    $_ = shift @ARGV;

    my $tabsize = int(shift @ARGV);
    my $expand  = " "x$tabsize;
    my $tabcnt;
    my $direction;

    SWITCH_DIR:
    {
        /^LEFT$/   && do { $direction = -1; last SWITCH_DIR; };
        /^MIDDLE$/ && do { $direction =  0; last SWITCH_DIR; };
        /^RIGHT$/  && do { $direction =  1; last SWITCH_DIR; };
    }

    while (<STDIN>)
    {
        chop;

        s/\t/$expand/g;

        if (/^(\s*)(.*)$/)
        {
            my $spaces = length($1);
            my $text   = $2;
            my $tabcnt = $spaces / $tabsize + $direction;

            $tabcnt = 0 if $tabcnt<0;

            $_ = "\t"x$tabcnt . $text;
        }

        print STDOUT $_ . "\n";
    }
}

sub RenumberLines
{
    my $start     = 0;
    my $increment = 1;
    my $format    = "%d";

    for (@ARGV)
    {
        /^-s(\d+)$/ && do { $start     = int($1); next; };
        /^-i(\d+)$/ && do { $increment = int($1); next; };
        /^-f(.*)$/  && do { $format    = $1;      next; };
    }

    for (<STDIN>)
    {
        chop;
        printf STDOUT "${format}\n", ($start,$_);
        $start += $increment;
    }
}

sub Grep
{
    my $regexp  = shift @ARGV;
    my $inverse = shift @ARGV;

    my @files = <STDIN>;

    foreach $file (@files)
    {
        chop($file);

        if (!open (IN, "<" . $file))
        {
            print STDOUT "Cannot open " . $file . ": $!\n";
        }

        # $file =~ s/^\w://;

        my $line = 0;

        while (<IN>)
        {
            $line ++;
            my $found = 0;
            eval "\$found= 1 if $regexp;";
            if (($found && !$inverse) || (!$found && $inverse))
            {
                print STDOUT $file . ":" if scalar @files > 1;
                print STDOUT $line . ":" . $_;
            }
        }

        close IN;
    }
}

sub Sort
{
    my ($start,$width,$inverse,$casesensitive) = (-1,-1,0,0);

    for (@ARGV)
    {
        /^\-i(\d+)$/ && do { $inverse       = int($1); next; };
        /^\-c(\d+)$/ && do { $casesensitive = int($1); next; };
        /^\-s(\d+)$/ && do { $start         = int($1); next; };
        /^\-w(\d+)$/ && do { $width         = int($1); next; };
    }

    my $cmd = "print STDOUT ";
    $cmd   .= "reverse " if $inverse;
    $cmd   .= "sort { ";

    my $fmt = '$%s';

    $fmt    = "substr(\$%s,$start,$width)" if     $start>=0;
    $fmt    = "uc(" . $fmt . ")"           unless $casesensitive;

    $cmd   .= sprintf( "$fmt cmp $fmt", ("a","b")) . " } <STDIN>;";
    eval $cmd;
}

sub Unique
{
    my %UNIQ;

    for (<STDIN>)
    {
        chop;
        s/\s+$//;
        $UNIQ{$_}++;
    }

    for (sort keys %UNIQ)
    {
        print STDOUT "$_\n";
    }
}

sub UniqueCount
{
    my %UNIQ;

    for (<STDIN>)
    {
        chop;
        s/\s+$//;
        $UNIQ{$_}++;
    }

    for (sort keys %UNIQ)
    {
        printf STDOUT "%6.6d %s\n", ($UNIQ{$_},$_);
    }
}

sub WordFrequency
{
    my %UNIQ;

    for (<STDIN>)
    {
        while (s/\b([\w\$]+)\b//)
        {
            $UNIQ{$1}++;
        }
    }

    for (sort keys %UNIQ)
    {
        printf STDOUT "%6d %s\n", ($UNIQ{$_},$_);
    }
}

sub HtmlAddPair
{
    my $tag   = shift @ARGV;
    my @file  = <STDIN>;
    my $size  = scalar @file;
    my $param = "";

    if ($tag =~ /^(\w+)(\s.*?)$/)
    {
        $tag   = $1;
        $param = $2;
    }

    if ($size==0)
    {
        print STDOUT "<${tag}${param}>\n";
        print STDOUT "</${tag}>\n";
        return;
    }

    if ($file[0] =~ /^<${tag}>/)
    {
        $file[0]         =~ s%^<${tag}>%%;
        $file[$size - 1] =~ s%</${tag}>$%%;
    }
    else
    {
        $file[0]         =~ s%^%<${tag}${param}>%;
        $file[$size - 1] =~ s%$%</${tag}>%;
    }
    print STDOUT @file;
}

sub HtmlMatch
{
    my @file  = <STDIN>;
    my $size  = scalar @file;

    if ( $file[0] =~ /^(\s*)<(\w*?)(\s.*?)?>(.*)$/)
    {
        my $indent = $1;
        my $tag    = $2;
        my $params = $3;
        my $rest   = $4;

        if ( $rest eq "")
        {
            if ($size == 1)
            {
                $file[0] =~ s%$%</$tag>%;
            }
            else
            {
                push @file, "$indent</$tag>\n";
            }
        }
        elsif ($file[$size - 1] !~ /<\/$tag>$/)
        {
            $file[$size - 1] =~ s%\n$%</$tag>\n%;
        }
        else
        {
            $file[0]         =~ s%^<${tag}>%%;
            $file[$size - 1] =~ s%</${tag}>$%%;
        }
    }

    print STDOUT @file;
}

sub HtmlCnvTable
{
    my $sep   = shift @ARGV;
    my @file  = <STDIN>;

    print STDOUT "<table>\n";

    for (@file)
    {
        chop;

        print STDOUT "\t<tr>\n";

        # Columns are delimited using ;
        foreach $item (split /${sep}/)
        {
            $item =~ s/^\s*(.*?)\s*$/$1/;
            print STDOUT "\t\t<td>$item</td>\n";
        }

        print STDOUT "\t</tr>\n";
    }

    print STDOUT "</table>\n";
}


sub FillChar
{
    my $replace = shift @ARGV;
    my $width   = int(shift @ARGV);
    my @file    = <STDIN>;

    $width = 0 if $width < 0;
    push @file, "\n" if scalar @file == 0;

    for (@file)
    {
        chop;
        s/./${replace}/g;

        $_ = "$replace"x$width if /^$/;

        print STDOUT "$_\n";
    }
}

sub ToggleCase #Implemented in JScript too
{
    my @file  = <STDIN>;

    my $lo = 0;
    my $hi = 0;

    my @LC, @UP, @TC;

    for (@file)
    {
        $lo++ if /[a-z]/;
        $hi++ if /[A-Z]/;
        push @LC, lc($_);
        push @UC, uc($_);
        s/\b(\w)(\w+)\b/\u$1\L\l$2\L/g;
        push @TC, $_;
    }

    if    ($hi == 0)
    {
        print STDOUT @TC;
    }
    elsif ($lo>0 && $hi>0)
    {
        print STDOUT @UC;
    }
    elsif ($lo == 0)
    {
        print STDOUT @LC;
    }
    else
    {
        print STDOUT @file;
    }
}

sub Summator
{
    my ($dec, $tsnd) = (","," ");

    for (@ARGV)
    {
        /^\-d(.+)$/ && do { $dec  = $1; next; };
        /^\-t(.+)$/ && do { $tsnd = $1; next; };
    }

    my $deco = $dec;

    $dec  =~ s/\./\\./g;
    $tsnd =~ s/\./\\./g;

    my $sum = 0;

    for (<STDIN>)
    {
        chop;

        while (s/(^|-|\D)((\d{1,3})(${tsnd}?\d{3})*(${dec}\d+)?)(\D|$)//)
        {
            my $flt  = $2;
            my $sign = $1 eq "-"?-1:1;
            $flt =~ s/${tsnd}//g;
            $flt =~ s/${dec}/./;
            $sum += $sign * $flt;
        }
    }

    $sum =~ s/\./${deco}/;

    print STDOUT $sum;
}

sub TextStatistics
{
    my $lines = 0;
    my $chars = 0;
    my $words = 0;
    my $max   = 0;

    for (<STDIN>)
    {
        chop;
        $max    = length($_) if length($_) > $max;

        s/^\s+//;
        s/\s+$//;
        s/\s+/ /g;
        $lines++;
        $chars += length($_);
        $words += scalar (split / /, $_);
    }

    my $fmtmax = length("$chars");
    my $FMTMIN = 6;
    $fmtmax = $FMTMIN if $fmtmax < $FMTMIN;

    printf STDOUT "Columns    %${fmtmax}d\n", ($max);
    printf STDOUT "Lines      %${fmtmax}d\n", ($lines);
    printf STDOUT "Words      %${fmtmax}d\n", ($words);
    printf STDOUT "Characters %${fmtmax}d\n", ($chars);
}

sub AddComment
{
    my $what   = shift @ARGV;
    my $indent = shift @ARGV;
    my $cmt    = shift @ARGV;

    printf STDOUT " "x$indent . "/**\n";
    printf STDOUT " "x$indent . "* $cmt\n";
    printf STDOUT " "x$indent . "*/\n";
    printf STDOUT <STDIN>;
}

sub Convert
{
    my $what    = shift @ARGV;
    my $reverse = shift @ARGV;
    my $windows = shift @ARGV;

    my $from    = "";
    my $to      = "";
    my %MB      = ();  # Multibyte words
    my %AR      = ();  # Convert after multibyte words

    if ($what eq "CY2CZ")
    {
        if ($windows)
        {
            $from = "";
            $to   = "IAEOUVGKLBDMNPRSTZFJC''Wiaeouvgklbdmnprstzfjc''w";

            %MB = (
                    "" => "Ja",
                    "" => "ja",
                    "" => "Je",
                    "" => "je",
                    "" => "Ji",
                    "" => "ji",
                    "" => "Ju",
                    "" => "ju",
                    "" => "",
                    "" => "",
                    "" => "",
                    "" => "",
                    "" => "ch",
                    "" => "Ch",
                  );

            %AR = (
                    "'" => "",
                    "'" => "",
                    "ks" => "x",
                    "KS" => "X",
                    "t'" => "",
                    "T'" => "",
                    "l'" => "",
                    "L'" => "",
                  );
        }
        else
        {
            $from = "㢣ꢘ";
            $to   = "IAEOUVGKLBDMNPRSTZFJC''Wiaeouvgklbdmnprstzfjc''w";

            %MB = (
                    "" => "Ja",
                    "" => "ja",
                    "" => "Je",
                    "" => "je",
                    "" => "Ji",
                    "" => "ji",
                    "" => "Ju",
                    "" => "ju",
                    "" => "",
                    "" => "",
                    "" => "",
                    "" => "",
                    "" => "ch",
                    "" => "Ch",
                  );

            %AR = (
                    "'" => "",
                    "'" => "",
                    "ks"  => "x",
                    "KS"  => "X",
                    "t'"  => "",
                    "T'"  => "",
                    "l'"  => "",
                    "L'"  => "",
                  );
        }
    }
    elsif ($what eq "CY2US")
    {
        if ($windows)
        {
            $from = "";
            $to   = "IAEOUVGKLBDMNPRSTZFHXYC''WJiaeouvgklbdmnprstzfhxyc''wj";

            %MB = (
                    ""  => "Ch",
                    ""  => "ch",
                    ""  => "Sch",
                    ""  => "sch",
                    ""  => "Sh",
                    ""  => "sh",
                    "" => "x",
                    "" => "X",
                    ""  => "Ya",
                    ""  => "ya",
                    ""  => "Ye",
                    ""  => "ye",
                    ""  => "Yi",
                    ""  => "yi",
                    ""  => "Yu",
                    ""  => "yu",
                    ""  => "Zh",
                    ""  => "zh",
                  );
        }
        else
        {
            $from = "㢣ꢦ";
            $to   = "IAEOUVGKLBDMNPRSTZFHXYC''WJiaeouvgklbdmnprstzfhxyc''wj";

            %MB = (
                    "" => "Ch",
                    "" => "ch",
                    "" => "Sch",
                    "" => "sch",
                    "" => "Sh",
                    "" => "sh",
                    ""=> "x",
                    ""=> "X",
                    "" => "Ya",
                    "" => "ya",
                    "" => "Ye",
                    "" => "ye",
                    "" => "Yi",
                    "" => "yi",
                    "" => "Yu",
                    "" => "yu",
                    "" => "Zh",
                    "" => "zh",
                  );
        }
    }
    elsif ($what eq "HTML")
    {
        %MB = (
               '&'    => '&amp;'  ,
               '<'    => '&lt;'   ,
               '>'    => '&gt;'   ,
               '"'    => '&quot;' ,
               '(R)'  => '&reg;'  ,
               '(r)'  => '&reg;'  ,
               '(C)'  => '&copy;' ,
               '(c)'  => '&copy;' ,
               '(TM)' => '&trade;',
               '(tm)' => '&trade;',
              );
    }
    else
    {
        die "Unknown conversion code \"$what\"!";
    }

    my @order = ("MB", "AR");

    if ($reverse)
    {
        ($from,$to) = ($to,$from);
        &InvertHash(\%MB) ;
        &InvertHash(\%AR) ;
        @order =  ("AR", "MB");
    }

    undef $/;
    my $file = <STDIN>;
    eval( "\$file =~ tr/$from/$to/;");

    my $hash;
    my @keys;

    foreach $hash (@order)
    {
        eval "\@keys  =  keys %$hash;";
        for (my $i=0;$i<scalar @keys;$i++)
        {
            $keys[$i] =~ s/\(/\\(/g;
            $keys[$i] =~ s/\)/\\)/g;
        }
        eval "\$file =~ s/((" . join(")|(", @keys) . "))/\$" . $hash . "{\$1}/g;";
    }
    print STDOUT $file;
}

sub MIME
{
    my $what    = shift @ARGV;
    my $reverse = shift @ARGV;

    if ($what eq  "QP")
    {
        eval("use MIME::QuotedPrint;");

        while (<STDIN>)
        {
            if ($reverse)
            {
                print STDOUT encode_qp($_);
            }
            else
            {
                print STDOUT decode_qp($_);
            }
        }
    }
    else
    {
        die "Unknown method \"$what\"!";
    }
}

sub EmbeddedSQL
{
    $_         = shift @ARGV;
    my $indent = " " x (shift @ARGV);
    my $table  = shift @ARGV;
    my $where  = shift @ARGV;

    print STDOUT "${indent}EXEC SQL";
    $indent = "\n" . $indent . "\t";

    if    (/^SELECT$/)
    {
        print STDOUT "${indent}SELECT *";
        print STDOUT "${indent}INTO   :";
        print STDOUT "${indent}FROM   $table";
        print STDOUT "${indent}WHERE  $where" if defined $where;
        print STDOUT ";\n";
    }
    elsif (/^INSERT$/)
    {
        print STDOUT "INSERT INTO $table${indent}()${indent}VALUES${indent}();\n";
    }
    elsif (/^DELETE$/)
    {
        print STDOUT "${indent}DELETE FROM $table";
        print STDOUT "${indent}WHERE  $where" if defined $where;
        print STDOUT ";\n";
    }
    elsif (/^UPDATE$/)
    {
        print STDOUT "${indent}UPDATE $table";
        print STDOUT "${indent}SET ";
        print STDOUT "${indent}WHERE  $where" if defined $where;
        print STDOUT ";\n";
    }
}

sub MatchColumns
{
    my (%LEFT, %RIGHT);

    while (<STDIN>)
    {
        if (/^(\S+)\s+(\S+)\s*$/)
        {
            $LEFT{$1}  = 1;
            $RIGHT{$2} = 1;
        }
        elsif (/^(\S+)\s*$/)
        {
            $LEFT{$1}  = 1;
        }
        elsif (/^\s+(\S+)\s*$/)
        {
            $RIGHT{$1} = 1;
        }
    }

    for (sort keys %LEFT)
    {
        if ($RIGHT{$_})
        {
            print STDERR $_ . "\n";
            $RIGHT{$_} = 0;
        }
        else
        {
            print STDERR $_ . "+\n";
        }
    }

    for (sort keys %RIGHT)
    {
        if ($RIGHT{$_})
        {
            print STDERR $_ . "-\n";
        }
    }
}

sub ConvertSYLK2CSV
{
    my $fld_sep  = ';';
    my $txt_sep  = '"';

    for (@ARGV)
    {
        /^\-f(.+)$/ && do { $fld_sep = $1; next; };
        /^\-t(.+)$/ && do { $txt_sep = $1; next; };
    }

    my @files = <STDIN>;

    print STDOUT "Conversion report Separator [txt=$txt_sep] [fld=$fld_sep]\n" . "-"x80 . "\n";

    foreach $slkfile (@files)
    {
        chop($slkfile);

        my $csvfile =  $slkfile;
        $csvfile    =~ s/\.\w+$/.csv/;

        print STDOUT $slkfile . " => " . $csvfile . " ";

        if (!open (IN,  "<" . $slkfile))
        {
            print STDOUT "[Canot open input file $!]\n";
            next;
        }
        if (!open (OUT, ">" . $csvfile))
        {
            print STDOUT "[Canot open output file $!]\n";
            next;
        }

        my @fields;
        my $item;
        my $records = 0;

        for (<IN>)
        {
            chop;

            if (/^C;Y\d+;/ || /^E/)
            {
                if (scalar(@fields) >0)
                {
                    print OUT join($fld_sep,@fields) . "\n";
                    $records++;
                }
                @fields = ();
            }

            next unless /^C;(Y\d+;)?X\d+;K"(.*)"$/;

            $item =  $2;
            $item =~ s/;;/;/g;

            if ($item =~  s/([${txt_sep}]+)/$1$1/g || $item =~ /${fld_sep}/)
            {
                $item = $txt_sep . $item . $txt_sep;
            }

            push @fields, $item;
        }

        print STDOUT "[OK $records record(s)]\n";

        close IN;
        close OUT;
    }
}

sub ConvertDOS2UNIX
{
    my $reverse = shift @ARGV;

    if ($reverse)
    {
        $/ = "\r";
    }
    else
    {
        binmode STDOUT;
    }

    while (<STDIN>)
    {
        chomp;
        print $_ . ($reverse?"\n":"\r");
    }
}

sub SplitAndGlue
{
    my $what = uc(shift @ARGV);

    &Split if $what =~ /^SPLIT$/;
    &Glue  if $what =~ /^GLUE$/;
}

sub Split
{
    my $size = shift @ARGV;
    my $fmt  = shift @ARGV;

    if ($size !~ /^([\d\.]+)([km]?)$/i)
    {
        die "Unsuported format of size!\nUse: 99.999[k|K|m|M]!\n";
    }

    my $size = $1;
    my $mult = $2;

    for ($mult)
    {
        /k/ && do { $size *=    1000; next; };
        /K/ && do { $size *=    1024; next; };
        /m/ && do { $size *= 1000000; next; };
        /M/ && do { $size *= 1048576; next; };
    }

    $size = int($size);

    my $file  = <STDIN>;
    my $part;
    my $buffer;
    my $count = 0;

    chop $file;

    $file =~ /^(.*)\.(.*)$/;

    my $name = $1;
    my $ext  = $2;

    open (IN, "<" . $file) || die "Cannot open source file $file: $!\n";
    binmode IN;
    while (read(IN,$buffer,$size)>0)
    {
        $count++;
        $part = sprintf $fmt, ($name,$ext,$count);
        open (OUT, ">" . $part) || die "Cannot open partial file $part: $!\n";
        binmode OUT;
        print OUT $buffer;
        close OUT;

        print STDOUT "$part\n";
    }
    close IN;
}

sub Glue
{
    use File::stat;

    my $name  = shift @ARGV;
    my $sort  = shift @ARGV;

    my @files = <STDIN>;
    my $buffer;

    @files = sort @files if $sort;

    open ( OUT, ">" . $name) || die "Cannot open output file $name: $!\n";
    binmode OUT;

    for (@files)
    {
        open (IN, "<" . $_) || die "Cannot open partial file $_: $!\n";
        binmode IN;
        while (read(IN, $buffer, 4096)>0)
        {
            print OUT $buffer || die "Cannot write: $!";
        }
        close IN;
    }

    close OUT;

    my $sb = stat($name);

    printf STDOUT "%s [%s bytes]", ($name, $sb->size);
}

### Duplicate ##################################################################

sub Duplicate
{
    use File::stat;

    my $recurse = shift @ARGV;
    my $swap    = shift @ARGV;
    my $fmt     = shift @ARGV;

    my @files   = <STDIN>;
    my @sized;
    my %SIZE;
    my %TIME;

    for (my $i=0; $i<scalar @files; $i++)
    {
        my $file = $files[$i];
        chomp $file;

        if (-d $file)
        {
            if ($recurse)
            {
                opendir(DIR, $file) || die "Can't opendir $file: $!";
                my @sub = grep { $_ !~ /^\.+$/ } readdir(DIR);
                closedir DIR;
                for (@sub)
                {
                    push @files, $file . "\\" . $_;
                }
            }
            next;
        }

        if (-f $file)
        {
            my $sb       = stat($file);
            $SIZE{$file} = $sb->size;
            $TIME{$file} = $sb->mtime;

            push @sized, [$file,$sb->size,$sb->mtime];
        }
    }

    undef @files;

    my @last = ("",-1,-1);

    for $file (sort bySizeTime @sized)
    {
        if (@last[1] == @$file[1] && &isSameFile(@last[0],@$file[0]))
        {
            if ($swap)
            {
                printf STDOUT $fmt . "\n", (@$file[0],@last[0]);
            }
            else
            {
                printf STDOUT $fmt . "\n", (@last[0],@$file[0]);
            }
        }
        else
        {
            @last = @$file;
        }
    }
}

sub bySizeTime
{
    if (@$a[1] == @$b[1])
    {
        @$a[2] <=> @$b[2];
    }
    else
    {
        @$a[1] <=> @$b[1];
    }
}

sub isSameFile
{
    open( F1, "<" . shift);
    open( F2, "<" . shift);

    binmode(F1);
    binmode(F2);

    my ($b1, $b1);
    my $equal = 1;

    while (read(F1,$b1,4096)>0)
    {
        read F2,$b2,4096;

        if ($b1 ne $b2)
        {
            $equal = 0;
            last;
        }
    }

    close F1;
    close F2;
    return $equal;
}

### Template ###################################################################

sub Template
{
    my $what = shift @ARGV;

    if ($what eq "JAVA")
    {
        my $class = shift @ARGV;
        my $file  = $class . ".java";

        unless ($class)
        {
            print STDERR "Missing class name!";
            exit 1;
        }

        if (-f $file)
        {
            print STDERR "File $file already exists!";
            exit 1;
        }

        my $c = lc(substr( $class, 0, 1));

        open (OUT, ">" . $file);
        print OUT <<__JAVA;
import java.util.*;

public class $class {

\t$class () {
\t}

\tpublic static void main(String[] args) {
\t\t$class $c = new $class();
\t}
}
__JAVA
        close OUT;

    }
}

### IncrementalChange ##########################################################

sub IncrementalChange
{
    eval("use Text::CSV_XS;"); # If not eval then will crash on comps without

    my $csv  = Text::CSV_XS->new({
                'quote_char'  => '"',
                'escape_char' => '"',
                'sep_char'    => ',',
                'binary'      => 1
               });

    my %COLS;
    my @prev = ();
    my @curr = ();
    my @file = <STDIN>;
    my $len  = 0;

    for (@file)
    {
        if (!$csv->parse($_))
        {
            print STDERR "Failed on: [$_]\n";
            next;
        }

        @prev = @curr;
        @curr = $csv->fields();
        $len  = scalar @curr if scalar @curr > $len;

        if (scalar @prev>0)
        {
            for (my $i=0; $i<$len; $i++)
            {
                if ("$prev[$i]" ne "$curr[$i]")
                {
                    $COLS[$i]++;
                }
            }
        }
    }

    for (@file)
    {
        if (!$csv->parse($_))
        {
            print STDERR "Failed on: [$_]\n";
            next;
        }

        my @out;

        @curr = $csv->fields();

        for (my $i=0; $i<$len; $i++)
        {
            if ($COLS[$i]>0)
            {
                push @out, $curr[$i];
            }
        }

        $csv->combine(@out);
        print STDOUT $csv->string() . "\n";
    }
}

################################################################################

sub InvertHash
{
    my $hash = shift;
    my %TMP  = reverse %$hash;
    %$hash   = %TMP;
    undef %TMP;
}


sub Test
{
    for (@ARGV)
    {
        print STDOUT $_ , "\n";
    }
}
