/*
 * Copyright (c) Ondrej Brablc, 2001 <brablc@yahoo.com>
 * You can use, modify, distribute this source code  or
 * any other parts of BlockPro plugin only according to
 * BlockPro  License  (see  \Doc\License.txt  for  more
 * information).
 */

#include <windows.h>
#include <winmem.h>
#include <filebuf.hpp>
#include <string.h>

FileNameConvertor::FileNameConvertor(const char * pFullLongPath)
{
    int len = strlen(pFullLongPath) + 1;

    longPath  = winNew(char, len);
    shortPath = winNew(char, len);
    strcpy( longPath, pFullLongPath);

    if (!GetShortPathName( longPath, shortPath, len))
    {
        strcpy( shortPath, longPath);
    }
}

FileNameConvertor::~FileNameConvertor()
{
    winDel( longPath);
    winDel( shortPath);
}

char * FileNameConvertor::getShort(char * pBuffer)
{
    strcpy( pBuffer, shortPath);
    return pBuffer;
}

char * FileNameConvertor::get(char * fmt, char * pBuffer)
{
    char * ptr;

    // Full pathname
    if      (strcmp( fmt, "!\\!.!") == 0)
    {
        strcpy(pBuffer, longPath);
    }
    // Current drive
    else if (strcmp( fmt, "!:")     == 0)
    {
        strncpy( pBuffer, longPath, 2);
        pBuffer[2] = 0;
    }
    // Long filename without extension
    else if (strcmp( fmt, "!")      == 0 )
    {
        ptr = strrchr( longPath, '\\');
        strcpy(pBuffer, ptr+1);
        ptr  = strrchr(pBuffer, '.');
        if (ptr) *ptr = 0;
    }
    // Short file name without extension
    else if (strcmp( fmt, "!~")     == 0 )
    {
        ptr  = strrchr( shortPath, '\\');
        strcpy(pBuffer, ptr+1);
        ptr  = strrchr(pBuffer, '.');
        if (ptr) *ptr = 0;
    }
    // Long file name with extension
    else if (strcmp( fmt, "!.!")    == 0 )
    {
        ptr = strrchr( longPath, '\\');
        strcpy(pBuffer, ptr+1);
    }
    // Short file name with extension
    else if (strcmp( fmt, "!-!")    == 0 )
    {
        ptr = strrchr( shortPath, '\\');
        strcpy(pBuffer, ptr+1);
    }
    // Current path
    else if (strcmp( fmt, "!\\")    == 0 )
    {
        strcpy( pBuffer, longPath);
        ptr = strrchr( pBuffer , '\\');
        if (ptr) *ptr = 0;
    }
    // Short name of the current path
    else if (strcmp( fmt, "!/")     == 0 )
    {
        strcpy( pBuffer, shortPath);
        ptr = strrchr( pBuffer , '\\');
        if (ptr) *ptr = 0;
    }
    else *pBuffer = 0;

    return pBuffer;
}

/******************************************************************************/

FileBuf::FileBuf()
{
    file = INVALID_HANDLE_VALUE;
}

FileBuf::~FileBuf()
{
    Close();
}

bool FileBuf::Error()
{
    return (file == INVALID_HANDLE_VALUE);
}

void FileBuf::Close()
{
    if (file != INVALID_HANDLE_VALUE)
    {
        CloseHandle( file);
        file = INVALID_HANDLE_VALUE;
    }
}

/******************************************************************************/

ReadFileBuf::ReadFileBuf(const char * fileName, DWORD mode)
{
    file = CreateFile( fileName,
                       GENERIC_READ,
                       FILE_SHARE_READ,
                       NULL, // pointer to security attributes
                       OPEN_EXISTING,
                       mode,
                       NULL);

    sBuffer = winNew( char, READ_BUFFER_LEN);
    lBuffer = 0;

    sLine   = NULL;
    lLine   = 0;
}

ReadFileBuf::~ReadFileBuf()
{
    winDel( sBuffer);
    if (sLine) winDel(sLine);
}

bool ReadFileBuf::ReadLine()
{
    if (sLine) winDel(sLine);

    if (file == INVALID_HANDLE_VALUE)
    {
        sLine = NULL;
        lLine = 0;
    }

    // Leave there one character which might be 0x0d where 0x0a is in next block
    for (lLine=0; lLine+1<lBuffer; lLine++)
    {
        if (sBuffer[lLine] == 0x0d || sBuffer[lLine] == 0x0a)
        {
            sLine = winNew(char,lLine+1);
            strncpy( sLine, sBuffer, lLine);
            sLine[lLine] = 0;

            DWORD len = lLine + 1;

            if (sBuffer[len] == 0x0a)
            {
                len++;
            }

            // If it is less then zero than we have a problem because it is DWORD
            lBuffer -= len;

            memcpy( sBuffer, sBuffer + len, lBuffer);
            return true;
        }
    }

    // Incomplete line
    sLine = winNew(char, lBuffer+1);
    strncpy( sLine, sBuffer, lBuffer);
    lLine = lBuffer;
    sLine[lLine] = 0;

    if (!ReadFile(file, sBuffer, READ_BUFFER_LEN, &lBuffer, NULL) || lBuffer == 0)
    {
        Close();
        return (lLine>0);
    }
    else
    {
        char * tmp = winNew(char, lLine+lBuffer);
        memcpy( tmp,         sLine,   lLine);
        memcpy( tmp + lLine, sBuffer, lBuffer);

        winDel( sBuffer);
        sBuffer = tmp;
        lBuffer = lLine + lBuffer;

        return ReadLine();
    }
}

void ReadFileBuf::Flush()
{
    if (sLine)
    {
        winDel(sLine);
        sLine = NULL;
        lLine = 0;
    }

    sBuffer[0] = 0;
    lBuffer    = 0;
}

DWORD ReadFileBuf::Length()
{
    return lLine;
}

char * ReadFileBuf::LastLine()
{
    return sLine;
}

long ReadFileBuf::Size()
{
    return GetFileSize(file,NULL);
}

long ReadFileBuf::Tell()
{
    return SetFilePointer(file,0,NULL,FILE_CURRENT);
}


bool ReadFileBuf::Seek(long pos)
{
    Flush();
    return (SetFilePointer (file, pos, NULL, FILE_BEGIN) != 0xFFFFFFFF);
}

/******************************************************************************/

WriteFileBuf::WriteFileBuf(const char * fileName)
{
    file = CreateFile( fileName,
                       GENERIC_WRITE,
                       FILE_SHARE_READ,
                       NULL, // pointer to security attributes
                       CREATE_ALWAYS,
                       FILE_FLAG_SEQUENTIAL_SCAN,
                       NULL);
}

WriteFileBuf::~WriteFileBuf()
{

}

bool WriteFileBuf::Write(const char * string)
{
    DWORD dWritten;

    WriteFile( file,
               string,
               strlen(string),
               &dWritten,
               NULL);
}

bool WriteFileBuf::WriteLine(const char * string)
{
    if (string) Write( string);
    Write( "\x0D\x0A");
}
