/*
    HtmlHelp sub-plugin for Active-Help plugin for FAR Manager
    Copyright (C) 2003-2004 Alex Yaroslavsky

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/
#include "../../ahp.hpp"
#include "registry.hpp"
#include "crt.hpp"
#include "htmlhelp.h"

#if defined(__GNUC__)
extern "C"
{
  BOOL WINAPI DllMainCRTStartup(HANDLE hDll,DWORD dwReason,LPVOID lpReserved);
};

BOOL WINAPI DllMainCRTStartup(HANDLE hDll,DWORD dwReason,LPVOID lpReserved)
{
  (void) hDll;
  (void) dwReason;
  (void) lpReserved;
  return TRUE;
}
#endif

extern "C"
{
  void CALLBACK _export RundllEntry(HWND hwnd, HINSTANCE hinst, LPSTR lpCmdLine, int nCmdShow);
};

static void Config(void);

#define sizeofa(array) (sizeof(array)/sizeof(array[0]))
#define WAHMSG_SHOWHELP (WM_USER+1)

#if defined(__BORLANDC__) || defined(__GNUC__)
extern HWND MHWND;
extern DWORD PID;
extern char PERROR[128];
#else
#if defined(_MSC_VER)
#pragma comment(linker, "/SECTION:Shared,RWS")
#pragma data_seg("Shared")
HWND MHWND=0;
DWORD PID=0;
char PERROR[128];
#pragma data_seg()
#endif
#endif

static struct PluginStartupInfo Info;
FARSTANDARDFUNCTIONS FSF;
static struct AHPluginStartupInfo AHInfo;
char PluginRootKey[80];
static int HowShowWindow;
static unsigned int WCounter=0;

enum
{
  MTitle,
  MOpenIn,
  MSW_NORMAL,
  MSW_MAXIMIZED,
  MSW_DEFAULT,
  MErrMSDN,
  MOk,
  MCancel,
};

static int hswa[] =
{
  SW_SHOWNORMAL,
  SW_SHOWMAXIMIZED,
  SW_SHOWDEFAULT
};

typedef HWND (WINAPI *THtmlHelp)(HWND hwndCaller, LPCSTR pszFile, UINT uCommand, DWORD_PTR dwData);
static THtmlHelp dHtmlHelp;
static WNDPROC OldHHWndProc=0;
static HANDLE event;

struct InitDialogItem
{
  unsigned char Type;
  unsigned char X1, Y1, X2, Y2;
  unsigned int Flags;
  signed char Data;
};

static char *GetMsg(int MsgNum, char *Str)
{
  AHInfo.GetMsg(AHInfo.MessageName,MsgNum,Str);
  return Str;
}

static void InitDialogItems(const struct InitDialogItem *Init, struct FarDialogItem *Item, int ItemsNumber)
{
  int i;
  struct FarDialogItem *PItem = Item;
  const struct InitDialogItem *PInit = Init;

  for (i=0; i<ItemsNumber; i++, PItem++, PInit++)
  {
    PItem->Type = PInit->Type;
    PItem->X1 = PInit->X1;
    PItem->Y1 = PInit->Y1;
    PItem->X2 = PInit->X2;
    PItem->Y2 = PInit->Y2;
    PItem->Focus = 0;
    PItem->Selected = 0;
    PItem->Flags = PInit->Flags;
    PItem->DefaultButton = 0;
    if (PInit->Data==-1)
      *PItem->Data = 0;
    else
      GetMsg(PInit->Data,PItem->Data);
  }
}

LRESULT WINAPI HHWndProc(HWND hwnd,UINT message,UINT wParam,LONG lParam)
{
  LRESULT ret=CallWindowProc(OldHHWndProc,hwnd,message,wParam,lParam);
  if(message==WM_CLOSE)
  {
    if ((--WCounter)<=0)
    {
      CloseHandle(event);
      PostMessage(MHWND,WM_CLOSE,0,0);
      WCounter=0;
    }
  }
  return ret;
}

BOOL ShowHelp(char *FileName, char *Keyword, int SW)
{
  char szPath[512];
  BOOL ret=TRUE;
  if (!lstrcmpi(FileName,"MSDN"))
  {
    ret=FALSE;
    HKEY hKeyCol;
    if( RegOpenKeyEx(HKEY_LOCAL_MACHINE,"SOFTWARE\\Microsoft\\HTML Help Collections\\Developer Collections",0,KEY_READ,&hKeyCol)==ERROR_SUCCESS )
    {
      DWORD dwType, dwSize=sizeof(szPath);
      if( RegQueryValueEx(hKeyCol,"Language",0,&dwType,(LPBYTE)szPath,&dwSize)==ERROR_SUCCESS )
      {
        dwSize=sizeof(szPath);
        HKEY hKeyLang;
        if( RegOpenKeyEx(hKeyCol,szPath,0,KEY_READ,&hKeyLang)==ERROR_SUCCESS )
        {
          dwSize=sizeof(szPath);
          if( RegQueryValueEx(hKeyLang,"Preferred",0,&dwType,(LPBYTE)szPath,&dwSize)==ERROR_SUCCESS )
          {
            HKEY hKeyLoc;
            if( RegOpenKeyEx(hKeyLang,szPath,0,KEY_READ,&hKeyLoc)==ERROR_SUCCESS )
            {
              dwSize=sizeof(szPath);
              if( RegQueryValueEx(hKeyLoc,"Filename",0,&dwType,(LPBYTE)szPath,&dwSize)==ERROR_SUCCESS )
              {
                ret=TRUE;
                FileName=szPath;
              }
              RegCloseKey(hKeyLoc);
            }
          }
          RegCloseKey(hKeyLang);
        }
      }
      RegCloseKey(hKeyCol);
    }
  }
  if (!ret)
  {
    //Error message is already loaded
    if(OldHHWndProc==0 && WCounter<=0)
    {
      CloseHandle(event);
      PostMessage(MHWND,WM_CLOSE,0,0);
    }
    return ret;
  }
  HH_AKLINK link;
  HWND hwndCaller = GetDesktopWindow();
  UINT uiCommand;
  DWORD dwData;
  if (*Keyword)
  {
    link.cbStruct = sizeof(link);
    link.fReserved = FALSE;
    link.pszKeywords = Keyword;
    link.pszUrl = NULL;
    link.pszMsgText = NULL;
    link.pszMsgTitle = NULL;
    link.pszWindow = NULL;
    link.fIndexOnFail = TRUE;
    uiCommand = HH_KEYWORD_LOOKUP;
    dwData = (DWORD)&link;
  }
  else
  {
    uiCommand = HH_HELP_FINDER;
    dwData = 0;
  }
  HWND hwnd = dHtmlHelp(hwndCaller, FileName, uiCommand, dwData);
  ret = hwnd==NULL ? FALSE : TRUE;
  if (!ret)
  {
    HH_LAST_ERROR lasterror;
    hwnd = dHtmlHelp(hwndCaller, NULL, HH_GET_LAST_ERROR, (DWORD)&lasterror);
    if (hwnd && (lasterror.hr>=0))
    {
      ret = TRUE;
    }
    else
    {
      WideCharToMultiByte(CP_ACP,0,lasterror.description,-1,PERROR,128,NULL,NULL);
      CharToOem(PERROR,PERROR);
    }
    if(OldHHWndProc==0 && WCounter<=0)
    {
      CloseHandle(event);
      PostMessage(MHWND,WM_CLOSE,0,0);
    }
  }
  else
  {
    LONG l=SetWindowLong(hwnd,GWL_WNDPROC,(LONG)HHWndProc);
    if(l!=(LONG)HHWndProc)
    {
      WCounter++;
      OldHHWndProc=(WNDPROC)l;
    }
    ShowWindow(hwnd, SW_SHOW);
    ShowWindow(hwnd, hswa[SW]);
  }
  return ret;
}

LRESULT WINAPI WndProc(HWND hwnd,UINT message,UINT wParam,LONG lParam)
{
  switch (message)
  {
    case WAHMSG_SHOWHELP:
    {
      HANDLE hMap=(HANDLE)wParam;
      int ret=FALSE;
      if (dHtmlHelp)
      {
        char *FileName=(char*)MapViewOfFile(hMap,FILE_MAP_WRITE,0,0,0);
        ret=ShowHelp(FileName,FileName+lstrlen(FileName)+1,
                     FileName[lstrlen(FileName)+1+lstrlen(FileName+lstrlen(FileName)+1)+1]);
        UnmapViewOfFile(FileName);
      }
      CloseHandle(hMap);
      if (!ret && WCounter<=0)
        PostQuitMessage(0);
      return ret;
    }

    case WM_CLOSE:
      PostQuitMessage(0);
      return 0;

    default:
      return DefWindowProc(hwnd, message, wParam, lParam);
  }
}

void CALLBACK _export RundllEntry(HWND hwnd, HINSTANCE hinst, LPSTR lpCmdLine, int nCmdShow)
{
  (void)hwnd;
  (void)lpCmdLine;
  (void)nCmdShow;
  //FreeConsole();
  event=OpenEvent(EVENT_ALL_ACCESS,0,"htmlhelpWEvent");
  HMODULE hLib = LoadLibrary("HHCTRL.OCX");
  dHtmlHelp = (THtmlHelp)GetProcAddress(hLib,(char*)14);

  WNDCLASS wchtmlhelp={0};
  wchtmlhelp.lpfnWndProc = (WNDPROC)WndProc;
  wchtmlhelp.lpszClassName = "htmlhelpWClass";
  wchtmlhelp.hIcon = LoadIcon(hLib,"ICON!HTMLHELP");
  RegisterClass(&wchtmlhelp);
  MHWND = CreateWindow("htmlhelpWClass","", WS_OVERLAPPEDWINDOW, 0,0, 0,0, NULL,NULL,hinst,NULL);

  SetEvent(event);

  MSG msg;
  while(GetMessage(&msg,NULL,0,0))
  {
    TranslateMessage(&msg);
    DispatchMessage(&msg);
  }

  if (hLib)
    FreeLibrary(hLib);
}

int WINAPI _export Start(const struct PluginStartupInfo *FarInfo,const struct AHPluginStartupInfo *AHInfo)
{
  Info=*FarInfo;
  FSF=*FarInfo->FSF;
  ::AHInfo=*AHInfo;
  FSF.sprintf(PluginRootKey,"%s\\%s",AHInfo->RootKey,"HTMLHelp");
  GetRegKey("HowShowWindow",&HowShowWindow,0);
  HowShowWindow%=sizeofa(hswa);
  return 0;
}

void WINAPI _export Exit(void)
{
}

int WINAPI _export Message(unsigned long Msg,void *InData,void *OutData)
{
  switch(Msg)
  {
    case AHMSG_GETINFO:
    {
      GetInfoOutData *data=(GetInfoOutData *)OutData;
      data->TypesNumber=1;
      static struct TypeInfo TypesInfo[1];
      memset(TypesInfo,0,sizeof(TypesInfo));
      TypesInfo[0].StructSize=sizeof(TypesInfo[0]);
      lstrcpy(TypesInfo[0].TypeString,"HTMLHelp");
      lstrcpy(TypesInfo[0].Mask,"*.chm,*.col,*.its,msdn");
      lstrcpy(TypesInfo[0].Encoding,"WIN");
      data->TypesInfo=TypesInfo;
      GetMsg(MTitle,data->ConfigString);
      data->Flags=AHMSG_CONFIG;
      return TRUE;
    }

    case AHMSG_SHOWHELP:
    {
      HANDLE event=OpenEvent(EVENT_ALL_ACCESS,0,"htmlhelpWEvent");
      if(event==NULL)
      {
        event=CreateEvent(NULL,1,0,"htmlhelpWEvent");
        STARTUPINFO si={sizeof(STARTUPINFO)};
        PROCESS_INFORMATION pi;

        char shortp[MAX_PATH];
        GetShortPathName(AHInfo.ModuleName,shortp,MAX_PATH);
        char mod[MAX_PATH];
        FSF.sprintf(mod,"rundll32.exe %s,RundllEntry",shortp);
        CreateProcess(0,mod,0,0,0,0,0,0,&si,&pi);
        PID=pi.dwProcessId;
        CloseHandle(pi.hProcess);
        CloseHandle(pi.hThread);
      }
      WaitForSingleObject(event,INFINITE);
      CloseHandle(event);
      HANDLE hProcess=OpenProcess(PROCESS_DUP_HANDLE,0,PID);

      HelpInData *data = (HelpInData *)InData;
      char FileName[512];
      {
        DWORD spr = GetShortPathName(data->FileName,FileName,sizeof(FileName));
        if (spr>0&&spr<512)
          data->FileName=FileName;
      }
      int flen=lstrlen(data->FileName)+1;
      int klen=lstrlen(data->Keyword)+1;

      HANDLE hMap=CreateFileMapping(INVALID_HANDLE_VALUE,0,PAGE_READWRITE,0,flen+klen+1,0);
      char *buf=(char*)MapViewOfFile(hMap,FILE_MAP_WRITE,0,0,0);
      memcpy(buf,data->FileName,flen);
      memcpy(buf+flen,data->Keyword,klen);
      buf[flen+klen]=HowShowWindow;
      DuplicateHandle(GetCurrentProcess(),hMap,hProcess,&hMap,0,0,DUPLICATE_SAME_ACCESS|DUPLICATE_CLOSE_SOURCE);
      UnmapViewOfFile(buf);
      CloseHandle(hProcess);
      SetForegroundWindow(MHWND);
      HelpOutData *odata = (HelpOutData *)OutData;
      GetMsg(MErrMSDN,PERROR); //We can not call GetMsg from ShowHelp, different memory space.
      int ret = SendMessage(MHWND,WAHMSG_SHOWHELP,(WPARAM)hMap,0);
      if (!ret)
        lstrcpy(odata->Error,PERROR);
      return ret;
    }

    case AHMSG_CONFIG:
    {
      Config();
      return TRUE;
    }
  }
  return FALSE;
}

static void Config(void)
{
  static const struct InitDialogItem PreDialogItems[] =
  {
    DI_DOUBLEBOX  ,3  ,1  ,50 ,6  ,0               ,MTitle,
    DI_TEXT       ,5  ,2  ,0  ,0  ,0               ,MOpenIn,
    DI_COMBOBOX   ,5  ,3  ,47 ,0  ,DIF_DROPDOWNLIST,-1,
    DI_TEXT       ,4  ,4  ,0  ,0  ,DIF_SEPARATOR   ,-1,
    DI_BUTTON     ,0  ,5  ,0  ,0  ,DIF_CENTERGROUP ,MOk,
    DI_BUTTON     ,0  ,5  ,0  ,0  ,DIF_CENTERGROUP ,MCancel
  };
  struct FarDialogItem DialogItems[sizeofa(PreDialogItems)];

  InitDialogItems(PreDialogItems,DialogItems,sizeofa(PreDialogItems));
  DialogItems[2].Focus = TRUE;
  struct FarListItem li[sizeofa(hswa)];
  memset(li,0,sizeof(li));
  struct FarList fl = {sizeofa(hswa), li};
  li[HowShowWindow].Flags=LIF_SELECTED;
  for (int i=0, msg=MSW_NORMAL; i<sizeofa(hswa); i++, msg++)
    GetMsg(msg,li[i].Text);
  DialogItems[2].ListItems = &fl;
  DialogItems[4].DefaultButton = 1;

  if (Info.Dialog(Info.ModuleNumber,-1,-1,54,8,NULL,(struct FarDialogItem *)&DialogItems,sizeofa(DialogItems)) != 4)
    return;

  HowShowWindow = DialogItems[2].ListPos;

  SetRegKey("HowShowWindow",HowShowWindow);
}
