/*
    ChmView sub-plugin for Active-Help plugin for FAR Manager
    Copyright (C) 2003-2004 Alex Yaroslavsky

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/
#include "../../ahp.hpp"
#include "registry.hpp"
#include "chmlib.h"
#include "crt.hpp"
#include "memory.hpp"
#define sizeofa(array) (sizeof(array)/sizeof(array[0]))
#define PATH_MAX 512

#if defined(__GNUC__)
extern "C"
{
  BOOL WINAPI DllMainCRTStartup(HANDLE hDll,DWORD dwReason,LPVOID lpReserved);
};

BOOL WINAPI DllMainCRTStartup(HANDLE hDll,DWORD dwReason,LPVOID lpReserved)
{
  (void) hDll;
  (void) dwReason;
  (void) lpReserved;
  return TRUE;
}
#endif

static struct PluginStartupInfo Info;
FARSTANDARDFUNCTIONS FSF;
static struct AHPluginStartupInfo AHInfo;
char PluginRootKey[80];

extern "C"
{
  int chm_getfile(chmfile *c, char *name, ulong *length, ubyte **outbuf);
  chmfile *chm_openfile(const char *fname);
  void chm_close(chmfile *l);
};

static int Modality, Visuality;
enum
{
  MTitle,
  MGeneratingHash,
  MShowIn,
  MModal,
  MNonModal,
  MEditor,
  MViewer,
  MErrOpeningCHM,
  MErrUnsupportedCHM,
  MErrOpeningTopic,
  MErrWritingTemp,
  MOk,
  MCancel,
};

static int ShowCHM(const char *FileName, const char *Keyword, const char *Encoding, char *Error);
static void Config(void);

static char *GetMsg(int MsgNum, char *Str)
{
  AHInfo.GetMsg(AHInfo.MessageName,MsgNum,Str);
  return Str;
}

struct InitDialogItem
{
  unsigned char Type;
  unsigned char X1, Y1, X2, Y2;
  unsigned int Flags;
  signed char Data;
};

static void InitDialogItems(const struct InitDialogItem *Init, struct FarDialogItem *Item, int ItemsNumber)
{
  int i;
  struct FarDialogItem *PItem = Item;
  const struct InitDialogItem *PInit = Init;

  for (i=0; i<ItemsNumber; i++, PItem++, PInit++)
  {
    PItem->Type = PInit->Type;
    PItem->X1 = PInit->X1;
    PItem->Y1 = PInit->Y1;
    PItem->X2 = PInit->X2;
    PItem->Y2 = PInit->Y2;
    PItem->Focus = 0;
    PItem->Selected = 0;
    PItem->Flags = PInit->Flags;
    PItem->DefaultButton = 0;
    if (PInit->Data==-1)
      *PItem->Data = 0;
    else
      GetMsg(PInit->Data,PItem->Data);
  }
}

int WINAPI _export Start(const struct PluginStartupInfo *FarInfo,const struct AHPluginStartupInfo *AHInfo)
{
  Info=*FarInfo;
  FSF=*FarInfo->FSF;
  ::AHInfo=*AHInfo;
  FSF.sprintf(PluginRootKey,"%s\\%s",AHInfo->RootKey,"CHMView");
  GetRegKey("Modality",&Modality,0);
  Modality %= 2;
  GetRegKey("Visuality",&Visuality,0);
  Visuality %= 2;
  return 0;
}

void WINAPI _export Exit(void)
{
}

int WINAPI _export Message(unsigned long Msg,void *InData,void *OutData)
{
  switch(Msg)
  {
    case AHMSG_GETINFO:
    {
      GetInfoOutData *data=(GetInfoOutData *)OutData;
      data->TypesNumber=1;
      static struct TypeInfo TypesInfo[1];
      memset(TypesInfo,0,sizeof(TypesInfo));
      TypesInfo[0].StructSize=sizeof(TypesInfo[0]);
      lstrcpy(TypesInfo[0].TypeString,"CHMView");
      lstrcpy(TypesInfo[0].Mask,"*.chm,*.its");
      lstrcpy(TypesInfo[0].Encoding,"WIN");
      TypesInfo[0].HFlags=F_INPUTKEYWORD;
      data->TypesInfo=TypesInfo;
      GetMsg(MTitle,data->ConfigString);
      data->Flags=AHMSG_CONFIG;
      return TRUE;
    }

    case AHMSG_SHOWHELP:
    {
      HelpInData *data = (HelpInData *)InData;
      if (!data->Keyword || !(*data->Keyword))
        return 1;
      char *Keyword = strdup(data->Keyword);
      if (!Keyword)
        return 0;
      AHInfo.Decode(Keyword,lstrlen(Keyword),data->Encoding);
      HelpOutData *odata = (HelpOutData *)OutData;
      int ret=ShowCHM(data->FileName,Keyword,data->Encoding,odata->Error);
      free(Keyword);
      return ret;
    }

    case AHMSG_CONFIG:
    {
      Config();
      return TRUE;
    }
  }
  return FALSE;
}

static ulong make_dword(ubyte *b)
{
  return ((ulong)(b[0] + b[1]*0x100 + b[2]*0x10000 + b[3]*0x1000000));
}

static int ShowCHM(const char *FileName, const char *Keyword, const char *Encoding, char *Error)
{
  ulong l_topics, l_strings, l_urltbl, l_urlstr;
  ubyte *o_topics, *o_strings, *o_urltbl, *o_urlstr;
  char url[PATH_MAX*2];
  lstrcpy(url,FileName);
  lstrcpy(url+lstrlen(url)-3,"chi");
  chmfile *h = chm_openfile(url);
  if (!h)
    h = chm_openfile(FileName);
  if (!h)
  {
    GetMsg(MErrOpeningCHM,Error);
    return 0;
  }
  ulong i;
  i=chm_getfile(h, "/#TOPICS", &l_topics, &o_topics);
  i+=chm_getfile(h, "/#STRINGS", &l_strings, &o_strings);
  i+=chm_getfile(h, "/#URLTBL", &l_urltbl, &o_urltbl);
  i+=chm_getfile(h, "/#URLSTR", &l_urlstr, &o_urlstr);
  chm_close(h);
  if (i!=0)
  {
    GetMsg(MErrUnsupportedCHM,Error);
    return 0;
  }
  i = 12;
  //while (i<l_topics && make_dword(&o_topics[i])!=0)
    //i += 16;
  if (i<l_topics)
    i -= 8;

  int KeywordLen=lstrlen(Keyword);
  FarMenuItemEx *Menu=NULL;
  int ItemsCount=0;
  while (true)
  {
    bool found=false;
    char topic[512];
    lstrcpyn(topic,(const char *) &o_strings[make_dword(&o_topics[i])],512);
    AHInfo.Decode(topic,lstrlen(topic),Encoding);
    while (i<l_topics && FSF.LStrnicmp(topic,Keyword,KeywordLen))
    {
      i += 16;
      lstrcpyn(topic,(const char *) &o_strings[make_dword(&o_topics[i])],512);
      AHInfo.Decode(topic,lstrlen(topic),Encoding);
    }
    ulong x;
    if (i<l_topics && !FSF.LStrnicmp(topic,Keyword,KeywordLen))
    {
      found = true;
      x = make_dword(&o_urltbl[make_dword(&o_topics[i+4])+8]);
      while (x<l_urlstr && o_urlstr[x]==0)
        x++;
    }
    if (found)
    {
      Menu = (struct FarMenuItemEx *)realloc(Menu,sizeof(struct FarMenuItemEx)*(++ItemsCount));
      memset(Menu+ItemsCount-1,0,sizeof(struct FarMenuItemEx));
      lstrcpyn(Menu[ItemsCount-1].Text.Text,topic,128);
      Menu[ItemsCount-1].UserData=x;
    }
    else
      break;
    i+=16;
  }

  bool Show=false;
  if (ItemsCount)
  {
    int Selected=0;
    if (ItemsCount>1)
    {
      char temp[128];
      Selected = Info.Menu(Info.ModuleNumber,-1,-1,0,FMENU_WRAPMODE|FMENU_AUTOHIGHLIGHT|FMENU_USEEXT,GetMsg(MTitle,temp),NULL,NULL,NULL,NULL,(const struct FarMenuItem *)Menu,ItemsCount);
    }
    if (Selected>=0)
    {
      url[0]='/';
      lstrcpy(&url[1],(const char *) &o_urlstr[Menu[Selected].UserData]);
      char *p = strchr(url,'#');
      if (p)
        *p = 0;
      Show=true;
    }
    if (Menu) free(Menu);
  }

  if (o_topics) free(o_topics);
  if (o_urlstr) free(o_urlstr);
  if (o_strings) free(o_strings);
  if (o_urltbl) free(o_urltbl);

  if (Show)
  {
    char TempFileName[MAX_PATH];
    if (CreateDirectory(FSF.MkTemp(TempFileName,"ACTH"),NULL))
    {
      chmfile *c = chm_openfile(FileName);
      if (!c)
      {
        GetMsg(MErrOpeningCHM,Error);
        return 0;
      }
      lstrcat(TempFileName,"\\chmview.txt");
      ulong length;
      ubyte *outbuf;
      i=chm_getfile(c, url, &length, &outbuf);
      chm_close(c);
      if (i!=0)
      {
        GetMsg(MErrOpeningTopic,Error);
        return 0;
      }
      char *p;
      while ((p = strchr(TempFileName,'\\')) != NULL)
        *p = '/';
      HANDLE f = CreateFile(TempFileName, GENERIC_WRITE, FILE_SHARE_READ,NULL,CREATE_NEW,FILE_ATTRIBUTE_NORMAL,NULL);
      if (f!=INVALID_HANDLE_VALUE)
      {
        DWORD t;
        WriteFile(f,outbuf, length, &t, NULL);
        CloseHandle(f);
      }
      else
       i = 1;
      if (outbuf) free(outbuf);
      if (i!=0)
      {
        GetMsg(MErrWritingTemp,Error);
        return 0;
      }
    }
    else
      return 1;
    int Modal=1;
    if (Info.AdvControl(Info.ModuleNumber,ACTL_GETWINDOWCOUNT,0)>1)
    {
      Modal=0;
    }
    else
    {
      struct WindowInfo wi;
      wi.Pos=-1;
      Info.AdvControl(Info.ModuleNumber,ACTL_GETWINDOWINFO,(void *)&wi);
      if (wi.Type==WTYPE_PANELS)
        Modal=0;
    }
    if (Visuality==0)
      Info.Editor(TempFileName,Keyword,0,0,-1,-1,(Modality==0&&!Modal?EF_NONMODAL:0)|EF_DELETEONCLOSE,0,1);
    else
      Info.Viewer(TempFileName,Keyword,0,0,-1,-1,(Modality==0&&!Modal?VF_NONMODAL:0)|VF_DELETEONCLOSE);
  }

  return 1;
}

static void Config(void)
{
  static const struct InitDialogItem PreDialogItems[] =
  {
    DI_DOUBLEBOX  ,3  ,1  ,41 ,6  ,0               ,MTitle,
    DI_TEXT       ,5  ,2  ,0  ,0  ,0               ,MShowIn,
    DI_COMBOBOX   ,5  ,3  ,20 ,0  ,DIF_DROPDOWNLIST,-1,
    DI_COMBOBOX   ,23 ,3  ,38 ,0  ,DIF_DROPDOWNLIST,-1,
    DI_TEXT       ,4  ,4  ,0  ,0  ,DIF_SEPARATOR   ,-1,
    DI_BUTTON     ,0  ,5  ,0  ,0  ,DIF_CENTERGROUP ,MOk,
    DI_BUTTON     ,0  ,5  ,0  ,0  ,DIF_CENTERGROUP ,MCancel
  };
  struct FarDialogItem DialogItems[sizeofa(PreDialogItems)];

  InitDialogItems(PreDialogItems,DialogItems,sizeofa(PreDialogItems));
  DialogItems[2].Focus = TRUE;
  struct FarListItem li1[2];
  memset(li1,0,sizeof(li1));
  struct FarList fl1 = {2, li1};
  li1[Modality].Flags=LIF_SELECTED;
  GetMsg(MNonModal,li1[0].Text);
  GetMsg(MModal,li1[1].Text);
  DialogItems[2].ListItems = &fl1;
  struct FarListItem li2[2];
  memset(li2,0,sizeof(li2));
  struct FarList fl2 = {2, li2};
  li2[Visuality].Flags=LIF_SELECTED;
  GetMsg(MEditor,li2[0].Text);
  GetMsg(MViewer,li2[1].Text);
  DialogItems[3].ListItems = &fl2;
  DialogItems[5].DefaultButton = 1;

  if (Info.Dialog(Info.ModuleNumber,-1,-1,45,8,NULL,(struct FarDialogItem *)&DialogItems,sizeofa(DialogItems)) != 5)
    return;

  Modality = DialogItems[2].ListPos;
  Visuality = DialogItems[3].ListPos;

  SetRegKey("Modality",Modality);
  SetRegKey("Visuality",Visuality);
}
