//---------------------------------------------------------------------------
#define UNICODE
#include <vcl\condefs.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <windows.h>
#include <winioctl.h>

#pragma hdrstop
//---------------------------------------------------------------------------
USERES("NTCopy.res");
//---------------------------------------------------------------------------

//---------------------------------------------------------------------------
//  NTCopy
//      NTFS     
//---------------------------------------------------------------------------

//---------------------------------------------------------------------------
//    
//---------------------------------------------------------------------------
#define EXIT_NOT_FOUND            1     // ,     
#define EXIT_ACCESS_DENIED        2     //  
#define EXIT_ALREADY_EXISTS       4     //   
#define EXIT_INCOMPATIBLE_TYPES   8     //     
#define EXIT_INVALID_SWITCH      16     //   
#define EXIT_SYNTAX_ERROR        32     //     
#define EXIT_NO_MEMORY           64     //  
#define EXIT_PRIVILEGE_CHECK     96     //   
#define EXIT_INVALID_SYSTEM     128     //    
#define EXIT_INVALID_VOLUME     256     //     
#define EXIT_ERROR_CREATE_LOG   512     //     
#define EXIT_SYSTEM_ERROR      1024     //   

//---------------------------------------------------------------------------
//   
//---------------------------------------------------------------------------
void* WBuffer1 = NULL;                  //    
void* WBuffer2 = NULL;
char* OEMBuffer = NULL;                 //    
wchar_t* SourcePath = NULL;             //    
wchar_t* DestinationPath = NULL;        //    
wchar_t* LoggingPath = NULL;            //    
HANDLE InitialStdOutHandle = INVALID_HANDLE_VALUE;  //    
HANDLE CurrentStdOutHandle = INVALID_HANDLE_VALUE;  //    
char DiskOut = 0;                       //     
char UseRussian = -1;                   //    
char OverwriteFiles = -1;               //    
char SkipCopy = 0;                      //   
char CopyRootRights = 0;                //       
char BatchMode = -1;                    //   
char AppendLog = 0;                     //    
int ExitCode = 0;                       //  
WIN32_FIND_DATAW FileFinder;            //    

//---------------------------------------------------------------------------
//  ShowAlert
//     
//---------------------------------------------------------------------------
int ShowAlert (char* Message, UINT uIcon)
{
    char* pZero = strchr (Message, 0);  //   
    if (UseRussian < 0)
    {
        *pZero++ = '\n'; *pZero++ = '\n';   //    
    }
    else
    {
        pZero++; pZero++;               // 
    }
    if (UseRussian <= 0) pZero = Message;   //  
    return MessageBoxA (GetActiveWindow (), pZero, "NTCopy", uIcon);    //  
}

//---------------------------------------------------------------------------
//  CheckSystemVersion
//      
//---------------------------------------------------------------------------
BOOL CheckSystemVersion (void)
{
    OSVERSIONINFOA OSVersion;           //    
    OSVersion.dwOSVersionInfoSize = sizeof (OSVERSIONINFOA);    //   
    if (!GetVersionExA (&OSVersion))    //      
    {
        ShowAlert ("Can'n get operating system version\0\0    ",
          MB_OK | MB_ICONSTOP);
        ExitCode |= EXIT_INVALID_SYSTEM;    //   
        return FALSE;
    }
    if (OSVersion.dwPlatformId != VER_PLATFORM_WIN32_NT)    //   Windows NT
    {
        ShowAlert ("This utility can work only under Windows NT\0\0      Windows NT",
          MB_OK | MB_ICONSTOP);
        ExitCode |= EXIT_INVALID_SYSTEM;    //   
        return FALSE;
    }
    return TRUE;
}

//---------------------------------------------------------------------------
//  ParseSwitchesAlert
//      
//---------------------------------------------------------------------------
void ParseSwitchesAlert (char* pSwitch, int ErrorMode)
{
    char MBuffer[1024];                 //   
    char* pMBuffer = stpcpy (MBuffer, "Invalid ");  //   
    if (ErrorMode)                      //   
        pMBuffer = stpcpy (pMBuffer, "value of ");  //   
    pMBuffer = stpcpy (pMBuffer, "switch: ");
    if (strlen (pSwitch) > 20)          //   
        pMBuffer = ((char*) memcpy (pMBuffer, pSwitch, 20)) + 20;   //  
    else pMBuffer = stpcpy (pMBuffer, pSwitch); //   
    pMBuffer = stpcpy (pMBuffer, " - ignored"); //   
    *pMBuffer++ = 0; *pMBuffer++ = 0;   //  
    pMBuffer = stpcpy (pMBuffer, ""); //   -
    if (ErrorMode)                      //   
        pMBuffer = stpcpy (pMBuffer, "  : ");
    else pMBuffer = stpcpy (pMBuffer, " : ");
    if (strlen (pSwitch) > 20)          //   
        pMBuffer = ((char*) memcpy (pMBuffer, pSwitch, 20)) + 20;   //  
    else pMBuffer = stpcpy (pMBuffer, pSwitch); //   
    pMBuffer = stpcpy (pMBuffer, " -  ");   //   
    *pMBuffer = 0;                      //  
    ShowAlert (MBuffer, MB_OK | MB_ICONWARNING);    //  
    ExitCode |= EXIT_INVALID_SWITCH;    //   
}

//---------------------------------------------------------------------------
//  ParseSwitches
//      
//---------------------------------------------------------------------------
void ParseSwitches (int argc, char** argv)
{
    for (int nArg = 1; nArg < argc; ++nArg)   //   
    {
        char* pArg = argv[nArg];        //  
        char cc = *pArg++;              //  
        if (cc == '-' || cc == '/')     //  
        {
            cc = *pArg++;               //  
            if (cc == 'l' || cc == 'L') //  
            {
                cc = *pArg++;           //  
                if (cc != ':')          //    
                    ParseSwitchesAlert (argv[nArg], (cc) ? 0 : 1); //  
                else
                {
                    cc = *pArg++;       //  
                    if (!cc || *pArg)   //       
                        ParseSwitchesAlert (argv[nArg], 1); //   
                    else if (cc == 'e' || cc == 'E')    //  
                        UseRussian = 0; // 
                    else if (cc == 'r' || cc == 'R')    //  
                        UseRussian = 1; // 
                    else ParseSwitchesAlert (argv[nArg], 1);    //   
                }
            }
            else if (cc == 'o' || cc == 'O')    //   
            {
                cc = *pArg++;           //  
                if (cc != ':')          //    
                    ParseSwitchesAlert (argv[nArg], (cc) ? 0 : 1);  //  
                else
                {
                    cc = *pArg++;       //  
                    if (!cc || *pArg)   //       
                        ParseSwitchesAlert (argv[nArg], 1); //   
                    else if (cc == 'y' || cc == 'Y')    // 
                        OverwriteFiles = 1; // 
                    else if (cc == 'n' || cc == 'N')    //  
                        OverwriteFiles = 0; // 
                    else ParseSwitchesAlert (argv[nArg], 1);    //   
                }
            }
            else if (cc == 'b' || cc == 'B')    //  
            {
                cc = *pArg++;           //  
                if (cc != ':')          //    
                    ParseSwitchesAlert (argv[nArg], (cc) ? 0 : 1);  //  
                else
                {
                    cc = *pArg++;       //  
                    if (!cc || *pArg)   //       
                        ParseSwitchesAlert (argv[nArg], 1); //   
                    else if (cc == 'a' || cc == 'A')    //   
                        BatchMode = 1;  // 
                    else if (cc == 'c' || cc == 'C')    //   
                        BatchMode = 0;  // 
                    else ParseSwitchesAlert (argv[nArg], 1);    //   
                }
            }
            else if (cc == 's' || cc == 'S')    //  
            {
                if (*pArg)              //   
                    ParseSwitchesAlert (argv[nArg], 0); //  
                else SkipCopy = 1;      // 
            }
            else if (cc == 'r' || cc == 'R')    //      
            {
                if (*pArg)              //   
                    ParseSwitchesAlert (argv[nArg], 0); //  
                else CopyRootRights = 1;   // 
            }
            else ParseSwitchesAlert (argv[nArg], 0);    //  
        }
    }
}

//---------------------------------------------------------------------------
//  RetreivePath
//        
//---------------------------------------------------------------------------

wchar_t* RetrievePath (char* pArg, int nType)
{
    wchar_t* cBuffer;
    int nCoded;
    if (!MultiByteToWideChar (CP_ACP, MB_PRECOMPOSED, pArg, -1,
      (wchar_t*) WBuffer2, 10240 / sizeof (wchar_t) - 1) || //     Unicode
      ((nCoded = GetFullPathNameW ((wchar_t*) WBuffer2,
      10240 / sizeof (wchar_t) - 1, (wchar_t*) WBuffer1, &cBuffer)) == 0))  //     
    {
        ShowAlert ("Unicode path convertion failed\nProbably a syntax error in command line\0\0    Unicode\n,      ",
          MB_OK | MB_ICONSTOP);         //   
        ExitCode |= EXIT_SYNTAX_ERROR;  //   
        return NULL;                    //  
    }
    cBuffer = ((wchar_t*) WBuffer1) + nCoded - 1;   //   
    if (nType >= 0)                     //  
    {
        if (*cBuffer++ != L'\\')        //  
            *cBuffer++ = L'\\';         // 
        if (!nType) cBuffer = _wcspcpy (cBuffer, L"*.*");   //    
    }
    else ++cBuffer;                     //   
    *cBuffer = 0;                       //  
    cBuffer = (wchar_t*) malloc ((wcslen ((wchar_t*) WBuffer1) + 1) * sizeof (wchar_t));    //    
    if (!cBuffer)                       //   
    {
        ShowAlert ("Insufficient memory\0\0    ",
          MB_OK | MB_ICONSTOP);         //   
        ExitCode |= EXIT_NO_MEMORY;     //   
        return NULL;                    //  
    }
    wcscpy (cBuffer, (wchar_t*) WBuffer1);  //  
    return cBuffer;                     //     
}

//---------------------------------------------------------------------------
//  ParseTreePair
//       
//---------------------------------------------------------------------------
BOOL ParseTreePair (int argc, char** argv)
{
    int ArgCount = 0;                   //  
    for (int nArg = 1; nArg < argc; nArg++) //   
    {
        char* pArg = argv[nArg];        //  
        if (*pArg == '>')               //   
        {
            char cAppend = 0;           //   
            if (*(++pArg) == '>')       //  
            {
                cAppend = 1; ++pArg;    //   
            }
            if (LoggingPath)            //     
                ShowAlert ("Multiple logging file definition - ignored\0\0    - ",
                  MB_OK | MB_ICONWARNING);  //   
            else
            {
                wchar_t* cBuffer = RetrievePath (pArg, -1); //  
                if (!cBuffer) return FALSE; //   
                LoggingPath = cBuffer;  //    
                AppendLog = cAppend;    //   
            }
        }
        else if (*pArg != '-' && *pArg != '/' && *pArg != '<')   //       
        {
            if (ArgCount > 1)           //   
            {
                ShowAlert ("Too many path arguments or syntax error in command line\0\0         ",
                  MB_OK | MB_ICONSTOP); //   
                ExitCode |= EXIT_SYNTAX_ERROR;  //   
                return FALSE;           //   
            }
            wchar_t* cBuffer = RetrievePath (pArg, ArgCount);   //  
            if (!cBuffer) return FALSE; //   
            if (ArgCount++) DestinationPath = cBuffer;  //   - 
            else SourcePath = cBuffer;  //   - 
        }
    }
    return TRUE;                        //  
}

//---------------------------------------------------------------------------
//  WriteLogString
//       
//---------------------------------------------------------------------------
void WriteLogString (char* LString, BOOL bBreak)
{
    char* pBuffer = LString;            //    
    if (!DiskOut)                       // ,    
    {
        CharToOemA (LString, OEMBuffer);    // 
        pBuffer = OEMBuffer;            //  
    }
    if (CurrentStdOutHandle == INVALID_HANDLE_VALUE)    //    
    {
        printf ("%s", pBuffer);         // 
        if (bBreak) printf ("\n");      //   
    }
    else
    {
    	DWORD nWritten = strlen (pBuffer);  //  ,    
        if (nWritten) WriteFile (CurrentStdOutHandle, pBuffer, nWritten, &nWritten, NULL);  //  
        if (bBreak) WriteFile (CurrentStdOutHandle, "\r\n", 2, &nWritten, NULL);    //   
    }
}

//---------------------------------------------------------------------------
//  WriteLogMessage
//     
//---------------------------------------------------------------------------
void WriteLogMessage (char* EString, char* RString)
{
    if (UseRussian <= 0) WriteLogString (EString, TRUE);    //   
    if (UseRussian) WriteLogString (RString, TRUE); //   
}

//---------------------------------------------------------------------------
//  WriteLogFile
//       
//---------------------------------------------------------------------------
void WriteLogFile (wchar_t* FileString, void* WBuffer, BOOL bBreak)
{
    int nCoded = WideCharToMultiByte (CP_ACP, 0, FileString, -1, (char*) WBuffer,
      10230, NULL, NULL);               //   
    if (!nCoded)                        //   
        *((char*) WBuffer) = 0;         // 
    WriteLogString ((char*) WBuffer, bBreak);   // 
}

//---------------------------------------------------------------------------
//  LogFileAlert
//        
//---------------------------------------------------------------------------
void LogFileAlert (wchar_t* FileString, int nStrLength, char* EString, char* ETrail,
  char* RString, char* RTrail)
{
    wchar_t* sBuffer = FileString;      //  
    if (nStrLength >= 0)                //   
    {
        sBuffer = (wchar_t*) WBuffer2;  //  
        if (nStrLength) memcpy (WBuffer2, FileString, nStrLength * sizeof (wchar_t));  //   
        sBuffer[nStrLength] = 0;        //  
    }
    if (UseRussian <= 0)                //   
    {
        WriteLogString (EString, FALSE);    //  
        WriteLogFile (sBuffer, WBuffer1, FALSE);    //  
        WriteLogString (ETrail, TRUE);  //  
    }
    if (UseRussian)                     //   
    {
        WriteLogString (RString, FALSE);    //  
        WriteLogFile (sBuffer, WBuffer1, FALSE);    //  
        WriteLogString (RTrail, TRUE);  //  
    }
}

//---------------------------------------------------------------------------
//  ShowFileAlert
//       
//---------------------------------------------------------------------------
int ShowFileAlert (wchar_t* FileString, int nStrLength, char* EString, char* ETrail,
  char* RString, char* RTrail, UINT uIcon)
{
    wchar_t* sBuffer = FileString;      //  
    if (nStrLength >= 0)                //   
    {
        sBuffer = (wchar_t*) WBuffer2;  //  
        if (nStrLength) memcpy (WBuffer2, FileString, nStrLength * sizeof (wchar_t));  //   
        sBuffer[nStrLength] = 0;        //  
    }
    char* cBuffer = (char*) WBuffer1;   //  
    *cBuffer++ = ':'; *cBuffer++ = '\n';    //  
    int nCoded = WideCharToMultiByte (CP_ACP, 0, sBuffer, -1, cBuffer, 10230,
      NULL, NULL);                      //  
    if (!nCoded)                        //   
        *((char*) WBuffer1) = 0;        // 
    cBuffer = (char*) WBuffer2;         //  
    cBuffer = stpcpy (cBuffer, EString);    //   
    cBuffer = stpcpy (cBuffer, (char*) WBuffer1);   //  
    cBuffer = stpcpy (cBuffer, ETrail); //   
    *cBuffer++ = 0; *cBuffer++ = 0;     //  
    cBuffer = stpcpy (cBuffer, RString);    //   
    cBuffer = stpcpy (cBuffer, (char*) WBuffer1);   //  
    stpcpy (cBuffer, RTrail);           //   
    return ShowAlert ((char*) WBuffer2, uIcon); //  
}

//---------------------------------------------------------------------------
//  Concatenate
//    
//---------------------------------------------------------------------------
wchar_t* Concatenate (wchar_t* S1, int n1, wchar_t* S2, int n2, wchar_t* S3, int n3)
{
    if (n1 < 0) n1 = wcslen (S1);       //    
    if (n2 < 0) n2 = wcslen (S2);
    if (n3 < 0) n3 = wcslen (S3);
    wchar_t* Sp = (wchar_t*) malloc ((n1 + n2 + n3 + 1) * sizeof (wchar_t));    //  
    if (Sp)                             //  
    {
        wchar_t* pSp = Sp;              //   
        if (n1 > 0) pSp = ((wchar_t*) memcpy (pSp, S1, n1 * sizeof (wchar_t))) + n1;    //  
        if (n2 > 0) pSp = ((wchar_t*) memcpy (pSp, S2, n2 * sizeof (wchar_t))) + n2;    //  
        if (n3 > 0) pSp = ((wchar_t*) memcpy (pSp, S3, n3 * sizeof (wchar_t))) + n3;    //  
        *pSp = 0;                       //  
    }
    return Sp;                          //  
}

//---------------------------------------------------------------------------
//  GetPathLength
//     
//---------------------------------------------------------------------------
int GetPathLength (wchar_t* FString)
{
    int nResult = 0;                    // 
    int nIndex = 0;                     //  
    while (FString[nIndex])             //    
    {
        if (FString[nIndex++] == L'\\') //  
            nResult = nIndex;           //  
    }
    return nResult;                     //  
}

//---------------------------------------------------------------------------
//  ProcessSystemError
//       
//---------------------------------------------------------------------------

void ProcessSystemError (int ErrorCode, wchar_t* FString, int PathLength, BOOL Type)
{
    char EMessage[49], RMessage[49];
    char* cBuffer = stpcpy (EMessage, "System error ");  //   
    sprintf (cBuffer, "%06d", ErrorCode);   //   
    cBuffer = stpcpy (cBuffer + 6, " while processing ");
    stpcpy (cBuffer, (Type) ? "directory" : " file");   //  
    cBuffer = stpcpy (RMessage, "  ");   //   
    sprintf (cBuffer, "%06d", ErrorCode);   //   
    cBuffer = stpcpy (cBuffer + 6, "   ");
    stpcpy (cBuffer, (Type) ? "" : "");    //  
    if (BatchMode < 0) ShowFileAlert (FString, PathLength, EMessage, "",
      RMessage, "", MB_OK | MB_ICONSTOP);   //    
    strcat (EMessage, ": ");            //   
    strcat (RMessage, ": ");
    LogFileAlert (FString, PathLength, EMessage, " - operation aborted",
      RMessage, " -  ");    //   
}

//---------------------------------------------------------------------------
//  ProcessFileError
//     
//---------------------------------------------------------------------------

BOOL ProcessFileError (int ErrorCode, wchar_t* FString, int PathLength, BOOL Type)
{
    if ((ErrorCode == ERROR_FILE_NOT_FOUND) || (ErrorCode == ERROR_PATH_NOT_FOUND) ||
      (ErrorCode == 67))                //     
    {
        ExitCode |= EXIT_NOT_FOUND;     //   
        int nReply = (!BatchMode) ? IDOK :
          ((BatchMode > 0) ? IDCANCEL :
          ShowFileAlert (FString, PathLength, "File, directory or path not found", "",
          ",     ", "", MB_OKCANCEL | MB_ICONWARNING));   // 
        if (nReply == IDOK)             //  
        {
            LogFileAlert (FString, PathLength,
              "File, directory or path not found: ",
              (Type) ? " - directory skipped" : " - file skipped",
              ",     : ",
              (Type) ? " -  " : " -  "); //   
            return TRUE;                //  
        }
        LogFileAlert (FString, PathLength,
          "File, directory or path not found: ", " - operation aborted",
          ",     : ", " -  ");    //   
        return FALSE;                   // 
    }
    if ((ErrorCode == ERROR_ACCESS_DENIED) ||
      (ErrorCode == 1314) || (ErrorCode == 1307))   //  
    {
        ExitCode |= EXIT_ACCESS_DENIED; //   
        int nReply = (!BatchMode) ? IDOK :
          ((BatchMode > 0) ? IDCANCEL :
          ShowFileAlert (FString, PathLength, "Access denied", "",
          " ", "", MB_OKCANCEL | MB_ICONWARNING));    // 
        if (nReply == IDOK)             //  
        {
            LogFileAlert (FString, PathLength,
              "Access denied: ", (Type) ? " - directory skipped" : " - file skipped",
              " : ", (Type) ? " -  " : " -  ");    //   
            return TRUE;                //  
        }
        LogFileAlert (FString, PathLength, "Access denied: ", " - operation aborted",
          " : ", " -  ");    //   
        return FALSE;                   // 
    }
    if ((ErrorCode == ERROR_NOT_ENOUGH_MEMORY) ||
      (ErrorCode == ERROR_NO_MORE_SEARCH_HANDLES) ||
      (ErrorCode == ERROR_INSUFFICIENT_BUFFER) ||
      (ErrorCode == ERROR_BUFFER_OVERFLOW)) //   
    {
        ExitCode |= EXIT_NO_MEMORY;     //   
        if (BatchMode < 0) ShowFileAlert (FString, PathLength,
          (Type) ? "Insufficient memory or system resources to process directory" :
          "Insufficient memory or system resources to process file", "",
          (Type) ? "       " :
          "       ", "",
          MB_OK | MB_ICONSTOP);         //    
        LogFileAlert (FString, PathLength,
          (Type) ? "Insufficient memory or system resources to process directory: " :
          "Insufficient memory or system resources to process file: ",
          " - operation aborted",
          (Type) ? "       : " :
          "       : ",
          " -  ");      //   
        return FALSE;                   // 
    }
    if ((ErrorCode == ERROR_INVALID_DRIVE) || (ErrorCode == ERROR_INVALID_NAME))    //    
    {
        ExitCode |= EXIT_SYNTAX_ERROR;  //   
        if (BatchMode < 0) ShowFileAlert (FString, PathLength,
          "Invalid drive or another error in the path definition", "",
          "       ", "",
          MB_OK | MB_ICONWARNING);      //    
        return FALSE;                   // 
    }
    ProcessSystemError (ErrorCode, FString, PathLength, Type);  //    
    return FALSE;                       //  
}

//---------------------------------------------------------------------------
//  TransferCompressionStatus
//     
//---------------------------------------------------------------------------
BOOL TransferCompressionStatus (wchar_t* PSource, wchar_t* PDest, BOOL Type)
{
    USHORT StatusBuffer;                //    
    DWORD nResult;                      //    
    int nErrorCode;                     //   
    DWORD dwAttrib = GetFileAttributesW (PSource);  //   
    if (dwAttrib == 0xFFFFFFFF)         //  
        return ProcessFileError (GetLastError (), PSource, -1, Type);   //  
    HANDLE hPtr = CreateFileW (PSource, 0,
      FILE_SHARE_DELETE | FILE_SHARE_READ | FILE_SHARE_WRITE, NULL, OPEN_EXISTING,
      FILE_FLAG_BACKUP_SEMANTICS, NULL);    //  
    if (hPtr == INVALID_HANDLE_VALUE)   //   
        return ProcessFileError (GetLastError (), PSource, -1, Type);   //  
    BOOL bExStat = DeviceIoControl (hPtr, FSCTL_GET_COMPRESSION, NULL, 0,
      &StatusBuffer, sizeof (USHORT), &nResult, NULL);  //   
    nErrorCode = GetLastError ();       //   
    CloseHandle (hPtr);                 //  
    if (!bExStat)                       //   
        return ProcessFileError (nErrorCode, PSource, -1, Type);    //  
    if (!SetFileAttributesW (PDest, FILE_ATTRIBUTE_NORMAL)) //     
        return ProcessFileError (GetLastError (), PDest, -1, Type); //  
    hPtr = CreateFileW (PDest, GENERIC_WRITE,
      FILE_SHARE_DELETE | FILE_SHARE_READ | FILE_SHARE_WRITE, NULL, OPEN_EXISTING,
      FILE_FLAG_BACKUP_SEMANTICS, NULL);    //  
    if (hPtr == INVALID_HANDLE_VALUE)   //   
        return ProcessFileError (GetLastError (), PDest, -1, Type); //  
    bExStat = DeviceIoControl (hPtr, FSCTL_SET_COMPRESSION,
      &StatusBuffer, sizeof (USHORT), NULL, 0, &nResult, NULL); //   
    nErrorCode = GetLastError ();       //   
    CloseHandle (hPtr);                 //  
    if (!bExStat)                       //   
        return ProcessFileError (nErrorCode, PDest, -1, Type);  //  
    if (!SetFileAttributesW (PDest, dwAttrib))  //    
        return ProcessFileError (GetLastError (), PDest, -1, Type); //  
    return TRUE;                        // 
}

//---------------------------------------------------------------------------
//  TransferFile
//     
//---------------------------------------------------------------------------
BOOL TransferFile (wchar_t* PSource, wchar_t* PDest)
{
    char DFlag;                         //    
    int nErrorCode = 0;                 //  
    DWORD dwAttrib = GetFileAttributesW (PDest);    //    
    if (dwAttrib == 0xFFFFFFFF)         //   
        nErrorCode = GetLastError ();   //   
    else
    {
        if (dwAttrib & FILE_ATTRIBUTE_DIRECTORY)    //  
        {
            ExitCode |= EXIT_INCOMPATIBLE_TYPES;    //   
            int nReply = (!BatchMode) ? IDOK :
              ((BatchMode > 0) ? IDCANCEL :
              ShowFileAlert (PDest, -1, "Directory already exists, file cannot be copied",
              "", "  ,   ", "",
              MB_OKCANCEL | MB_ICONWARNING));   // 
            if (nReply == IDOK)         //  
            {
                LogFileAlert (PDest, -1, "Directory already exists, file cannot be copied: ",
                  " - skipped", "  ,   : ",
                  " - ");       //   
                return TRUE;            //  
            }
            LogFileAlert (PDest, -1,
              "Directory already exists, file cannot be copied: ",
              " - operation aborted",
              "  ,   : ",
              " -  ");  //   
            return FALSE;               //  
        }
        int nReply = (!OverwriteFiles) ? IDNO :
          ((OverwriteFiles > 0) ? IDYES :
          ShowFileAlert (PDest, -1, "File already exists", "\nOverwrite?",
          "  ", "\n?",
          MB_YESNOCANCEL | MB_ICONQUESTION | MB_DEFBUTTON2));   //  ?
        if (nReply == IDCANCEL)         //  
        {
            LogFileAlert (PDest, -1, "File already exists: ", " - operation aborted",
            "  : ", " -  ");   //   
            return FALSE;               //  
        }
        if (nReply == IDNO)             //  
        {
            LogFileAlert (PDest, -1, "File already exists: ",
              " - skipped", "  : ", " - ");    //   
            return TRUE;                //  
        }
        if (OverwriteFiles <= 0) ExitCode |= EXIT_ALREADY_EXISTS;   //   
        if (!SetFileAttributesW (PDest, FILE_ATTRIBUTE_NORMAL)) //    
            nErrorCode = GetLastError ();   //   
    }
    if (nErrorCode == ERROR_FILE_NOT_FOUND) nErrorCode = 0; //   
    DFlag = (nErrorCode != 0);          //  
    if (!nErrorCode &&                  //  
      !CopyFileW (PSource, PDest, FALSE))   //    
    {
        nErrorCode = GetLastError ();   //   
        if (nErrorCode == ERROR_FILE_EXISTS)    //  
        {
            ExitCode |= EXIT_ALREADY_EXISTS;    //   
            int nReply = (!BatchMode) ? IDOK :
              ((BatchMode > 0) ? IDCANCEL :
              ShowFileAlert (PDest, -1, "File already exists and cannot be overwritten", "",
              "       ", "",
              MB_OKCANCEL | MB_ICONWARNING));   // 
            if (nReply == IDOK)         //  
            {
                LogFileAlert (PDest, -1, "File already exists and cannot be overwritten: ",
                  " - skipped", "       : ",
                  " - ");       //   
                return TRUE;            //  
            }
            LogFileAlert (PDest, -1, "File already exists and cannot be overwritten: ",
              " - operation aborted", "       : ",
              " -  ");  //   
            return FALSE;               //  
        }
    }
    if (nErrorCode)                     //  
        return ProcessFileError (nErrorCode, (DFlag) ? PDest : PSource, -1, FALSE);   //  
    if (!TransferCompressionStatus (PSource, PDest, FALSE)) return FALSE;   //    
    WriteLogFile (PSource, WBuffer1, FALSE);    // -
    WriteLogString (" => ", FALSE);     // 
    WriteLogFile (PDest, WBuffer1, TRUE);   //  
    return TRUE;                        // 
}

//---------------------------------------------------------------------------
//  MakeTargetDirectory
//     
//---------------------------------------------------------------------------
int MakeTargetDirectory (wchar_t* PSource, wchar_t* PDest)
{
    int nLength = GetPathLength (PDest);    //    
    DWORD dwAttrib = GetFileAttributesW (PDest);    //    / 
    if ((dwAttrib == 0xFFFFFFFF) || !(dwAttrib & FILE_ATTRIBUTE_DIRECTORY)) //   
    {
        if (dwAttrib != 0xFFFFFFFF)     // ,  
        {
            ExitCode |= EXIT_INCOMPATIBLE_TYPES;    //   
            int nReply = (!BatchMode) ? IDOK :
              ((BatchMode > 0) ? IDCANCEL :
              ShowFileAlert (PDest, -1, "File already exists, directory cannot be created", "",
              "  ,   ", "",
              MB_OKCANCEL | MB_ICONWARNING));   //  - 
            if (nReply == IDOK)         //  
            {
                LogFileAlert (PDest, -1, "File already exists, directory cannot be created: ",
                  " - skipped", "  ,   : ",
                  " - ");       //   
                return 0;               //  
            }
            LogFileAlert (PDest, -1, "File already exists, directory cannot be created: ",
              " - operation aborted", "  ,   : ",
              " -  ");  //   
            return -1;                  //  
        }
        int nErrorCode = GetLastError ();   //   
        if (nErrorCode == ERROR_FILE_NOT_FOUND) //   -  
            nErrorCode = CreateDirectoryExW (PSource, PDest, NULL) ? 0 : GetLastError ();   //  
        if (nErrorCode)                 //  
            return (ProcessFileError (nErrorCode, PDest, nLength, TRUE)) ? 0 : -1;  // 
    }
    if (!TransferCompressionStatus (PSource, PDest, TRUE)) return -1;   //    
    if (dwAttrib == 0xFFFFFFFF)         //  
    {
        WriteLogFile (PSource, WBuffer1, FALSE);    // -
        WriteLogString ("\\ => ", FALSE);   // 
    }
    else
    {
        WriteLogFile (PSource, WBuffer1, FALSE);    // -
        WriteLogString ("\\ == ", FALSE);   // 
    }
    WriteLogFile (PDest, WBuffer1, FALSE);  //  
    WriteLogString ("\\", TRUE);        // 
    return 1;                           // 
}

//---------------------------------------------------------------------------
//  TransferAccessRights
//          
//---------------------------------------------------------------------------
int TransferAccessRights (wchar_t* PSource, wchar_t* PDest, BOOL Type)
{
    int nErrorCode;                     //  
    DWORD dwDAttrib = GetFileAttributesW (PDest);   //    
    if (dwDAttrib == 0xFFFFFFFF)        //  
    {
        nErrorCode = GetLastError ();   //   
        if (((nErrorCode == ERROR_FILE_NOT_FOUND) || (nErrorCode == ERROR_PATH_NOT_FOUND) ||
          (nErrorCode == 67)) &&        //     
          !SkipCopy) return 0;          //    - ,  
        return (ProcessFileError (nErrorCode, PDest, -1, Type)) ? 0 : -1;   //  
    }
    DWORD dwSAttrib = GetFileAttributesW (PSource); //   
    if (dwSAttrib == 0xFFFFFFFF)        //  
        return (ProcessFileError (GetLastError (), PSource, -1, Type)) ? 0 : -1;    //  
    dwDAttrib &= FILE_ATTRIBUTE_DIRECTORY;  //    
    dwSAttrib &= FILE_ATTRIBUTE_DIRECTORY;
    if (dwDAttrib != dwSAttrib)         //    
    {
        if (!SkipCopy) return 0;        //    - ,  
        ExitCode |= EXIT_INCOMPATIBLE_TYPES;    //   
        int nReply = (!BatchMode) ? IDOK :
          ((BatchMode > 0) ? IDCANCEL :
          ShowFileAlert (PSource, -1, (dwSAttrib) ? "Target is not a directory" : "Target is not a file",
          "", (dwSAttrib) ? "    " : "    ",
          "", MB_OKCANCEL | MB_ICONWARNING));   // 
        if (nReply == IDOK)             //  
        {
            if (UseRussian <= 0)        //   
            {
                WriteLogFile (PSource, WBuffer1, FALSE);    // -
                WriteLogString ((dwSAttrib) ? "\\ - target is not a directory " : " - target is not a file ", FALSE);
                WriteLogFile (PDest, WBuffer1, FALSE);  //  
                WriteLogString ((dwDAttrib) ? "\\" : "", FALSE);    //  
                WriteLogString (" - skipped", TRUE);
            }
            if (UseRussian)             //   
            {
                WriteLogFile (PSource, WBuffer1, FALSE);    // -
                WriteLogString ((dwSAttrib) ? "\\ -      " : " -      ", FALSE);
                WriteLogFile (PDest, WBuffer1, FALSE);  //  
                WriteLogString ((dwDAttrib) ? "\\" : "", FALSE);    //  
                WriteLogString (" - ", TRUE);
            }
            return 0;                   //  
        }
        if (UseRussian <= 0)            //   
        {
            WriteLogFile (PSource, WBuffer1, FALSE);    // -
            WriteLogString ((dwSAttrib) ? "\\ - target is not a directory " : " - target is not a file ", FALSE);
            WriteLogFile (PDest, WBuffer1, FALSE);  //  
            WriteLogString ((dwDAttrib) ? "\\" : "", FALSE);    //  
            WriteLogString (" - operation aborted", TRUE);
        }
        if (UseRussian)                 //   
        {
            WriteLogFile (PSource, WBuffer1, FALSE);    // -
            WriteLogString ((dwSAttrib) ? "\\ -      " : " -      ", FALSE);
            WriteLogFile (PDest, WBuffer1, FALSE);  //  
            WriteLogString ((dwDAttrib) ? "\\" : "", FALSE);    //  
            WriteLogString (" -  ", TRUE);
        }
        return -1;                      //  
    }
    DWORD nLengthNeeded = 0;            //     
    PSECURITY_DESCRIPTOR pSecurity = NULL;  //     
    nErrorCode = 0;                     //   
    if (!GetFileSecurityW (PSource,
      OWNER_SECURITY_INFORMATION | GROUP_SECURITY_INFORMATION | DACL_SECURITY_INFORMATION | SACL_SECURITY_INFORMATION,
      (PSECURITY_DESCRIPTOR) WBuffer1, 0, &nLengthNeeded))  //       
    {
        nErrorCode = GetLastError ();   //   
        if ((nErrorCode == ERROR_INSUFFICIENT_BUFFER) ||
          (nErrorCode == ERROR_BUFFER_OVERFLOW))    //   
            nErrorCode = 0;             //    
    }
    if (!nErrorCode)                    //   
    {
        if (!nLengthNeeded) return 0;   //    
        pSecurity = (PSECURITY_DESCRIPTOR) malloc (nLengthNeeded);  //    
        if (!pSecurity)                 //   
        {
            ProcessFileError (ERROR_NOT_ENOUGH_MEMORY, PSource, -1, Type);  // 
            return -1;                  //  
        }
        DWORD nBufferLength = nLengthNeeded;    //   
        if (!GetFileSecurityW (PSource,
          OWNER_SECURITY_INFORMATION | GROUP_SECURITY_INFORMATION | DACL_SECURITY_INFORMATION | SACL_SECURITY_INFORMATION,
          pSecurity, nBufferLength, &nLengthNeeded))    //   
            nErrorCode = GetLastError ();   //   
        else if (nLengthNeeded > nBufferLength) nErrorCode = ERROR_INSUFFICIENT_BUFFER; //   
    }
    if (nErrorCode)                     // 
    {
        if (pSecurity) free (pSecurity);    //   
        return (ProcessFileError (nErrorCode, PSource, -1, Type)) ? 0 : -1; //  
    }
    if (!SetFileSecurityW (PDest,
      OWNER_SECURITY_INFORMATION | GROUP_SECURITY_INFORMATION | DACL_SECURITY_INFORMATION | SACL_SECURITY_INFORMATION,
      pSecurity))                       //    
        nErrorCode = GetLastError ();   //   
    free (pSecurity);                   //  
    if (((nErrorCode == ERROR_FILE_NOT_FOUND) || (nErrorCode == ERROR_PATH_NOT_FOUND) ||
      (nErrorCode == 67)) &&            //     
      !SkipCopy) return 0;              //    - ,  
    if (nErrorCode)                     // 
        return (ProcessFileError (nErrorCode, PDest, -1, Type)) ? 0 : -1;   // 
    WriteLogFile (PSource, WBuffer1, FALSE);    // -
    WriteLogString ((dwDAttrib && (*(PSource + wcslen (PSource) - 1) != L'\\')) ?
      "\\ => " : " => ", FALSE);        // 
    WriteLogFile (PDest, WBuffer1, FALSE);   //  
    WriteLogString ((dwSAttrib && (*(PDest + wcslen (PDest) - 1) != L'\\')) ?
      "\\" : "", TRUE);                 //  
    return 1;                           // 
}

//---------------------------------------------------------------------------
//  ScanTree
//         / 
//---------------------------------------------------------------------------
BOOL ScanTree (wchar_t* PSource, wchar_t* PDest, int nPass)
{
    int nLength = GetPathLength (PSource);  //   
    HANDLE FindHandle = FindFirstFileW (PSource, &FileFinder);  //   
    DWORD nErrorCode = GetLastError (); //     
    if (FindHandle != INVALID_HANDLE_VALUE) //  
    {
        BOOL bExStat = TRUE;            //  
        while (bExStat)                 //   
        {
            char cError = 0;            //  
            wchar_t* SourceSpec = Concatenate (PSource, nLength, FileFinder.cFileName, -1,
              PSource, 0);              //   
            wchar_t* DestSpec = Concatenate (PDest, -1, FileFinder.cFileName, -1, PDest, 0);    //   
            if (!SourceSpec || !DestSpec)   //   
            {
                ProcessFileError (ERROR_NOT_ENOUGH_MEMORY, PSource, nLength, TRUE); // 
                cError = 1;             //  
            }
            else if (FileFinder.dwFileAttributes & FILE_ATTRIBUTE_DIRECTORY)    //  
            {
                if (wcscmp (FileFinder.cFileName, L".") &&
                  wcscmp (FileFinder.cFileName, L"..")) //    
                {
                    if (nPass == 1)     //  
                    {
                        int nFlag = MakeTargetDirectory (SourceSpec, DestSpec); //   
                        if (nFlag < 0) cError = 1;  //  
                        else if (nFlag) //     
                        {
                            wchar_t* TempSpec = SourceSpec; //  
                            free (DestSpec);    //  
                            SourceSpec = Concatenate (PSource, nLength,
                              FileFinder.cFileName, -1, L"\\*.*", -1);  //    
                            DestSpec = Concatenate (PDest, -1, FileFinder.cFileName, -1,
                              L"\\", -1);   //  
                            if (!SourceSpec || !DestSpec)   //   
                            {
                                ProcessFileError (ERROR_NOT_ENOUGH_MEMORY, TempSpec,
                                  GetPathLength (TempSpec), TRUE);  // 
                                cError = 1; //  
                                free (TempSpec);    //   
                            }
                            else
                            {
                                free (TempSpec);    //   
                                if (!ScanTree (SourceSpec, DestSpec, 0) ||
                                  !ScanTree (SourceSpec, DestSpec, 1))  //    
                                    cError = 1; //  
                            }
                        }
                    }
                    else if (nPass == 3)    //     
                    {
                        int nFlag = TransferAccessRights (SourceSpec, DestSpec, TRUE);  //   
                        if (nFlag < 0) cError = 1;  //  
                        else if (nFlag)     //   
                        {
                            wchar_t* TempSpec = SourceSpec; //  
                            free (DestSpec);    //  
                            SourceSpec = Concatenate (PSource, nLength,
                              FileFinder.cFileName, -1, L"\\*.*", -1);  //    
                            DestSpec = Concatenate (PDest, -1, FileFinder.cFileName, -1,
                              L"\\", -1);   //  
                            if (!SourceSpec || !DestSpec)   //   
                            {
                                ProcessFileError (ERROR_NOT_ENOUGH_MEMORY, TempSpec,
                                  GetPathLength (TempSpec), TRUE);  // 
                                cError = 1; //  
                                free (TempSpec);    //   
                            }
                            else
                            {
                                free (TempSpec);    //   
                                if (!ScanTree (SourceSpec, DestSpec, 2) ||
                                  !ScanTree (SourceSpec, DestSpec, 3))  //    
                                    cError = 1; //  
                            }
                        }
                    }
                }
            }
            else
            {
                if (nPass == 0)         //  
                {
                    if (!TransferFile (SourceSpec, DestSpec))   //   
                        cError = 1;     // 
                }
                else if (nPass == 2)    //     
                {
                    int nFlag = TransferAccessRights (SourceSpec, DestSpec, FALSE); //   
                    if (nFlag < 0) cError = 1;  //  
                }
            }
            if (SourceSpec) free (SourceSpec);  //   
            if (DestSpec) free (DestSpec);
            if (cError) return FALSE;   //    
            bExStat = FindNextFileW (FindHandle, &FileFinder);  //   
        }
        nErrorCode = GetLastError ();   //   
        if (!FindClose (FindHandle))    //    
        {
            ProcessSystemError (GetLastError (), PSource, nLength, TRUE);   //    
            return FALSE;               //  
        }
    }
    if ((nErrorCode == ERROR_FILE_NOT_FOUND) || (nErrorCode == ERROR_NO_MORE_FILES))    //  
        return TRUE;
    return ProcessFileError (nErrorCode, PSource, nLength, TRUE);   //  
}

//---------------------------------------------------------------------------
//  CheckVolume
//     
//---------------------------------------------------------------------------

BOOL CheckVolume (wchar_t* CheckPath)
{
    wchar_t* cBuffer = wcscpy ((wchar_t*) WBuffer2, CheckPath); //     
    int nSearch = 1;                    //   
    if (*cBuffer == L'\\' && *(cBuffer + 1) == L'\\') nSearch = 4;    //   -  
    while (*cBuffer)                    //    
    {
        if (*cBuffer++ == L'\\')        //  
        {
            if (--nSearch == 0) break;  // 
        }
    }
    if (*(cBuffer - 1) != L'\\') *cBuffer++ = L'\\';    //   
    *cBuffer = 0;                       //  
    if (!GetVolumeInformationW ((wchar_t*) WBuffer2, NULL, 0, NULL, NULL,
      (DWORD*) WBuffer1, NULL, 0))      //      
    {
        int nErrorCode = GetLastError ();   //   
        if ((nErrorCode == ERROR_FILE_NOT_FOUND) || (nErrorCode == ERROR_PATH_NOT_FOUND) ||
          (nErrorCode == 67))           //     
        {
            ExitCode |= EXIT_NOT_FOUND; //   
            ShowFileAlert ((wchar_t*) WBuffer2, -1, "The specified volume not found", "",
              "   ", "", MB_OK | MB_ICONSTOP);   //  - 
            return FALSE;               // 
        }
        if (nErrorCode == ERROR_ACCESS_DENIED)  //  
        {
            ExitCode |= EXIT_ACCESS_DENIED; //   
            ShowFileAlert ((wchar_t*) WBuffer2, -1, "Access denied", "",
              " ", "", MB_OK | MB_ICONSTOP);  //  - 
            return FALSE;               //  
        }
        if (nErrorCode == ERROR_NOT_ENOUGH_MEMORY)  //   
        {
            ShowAlert ("Insufficient memory\0\0    ",
              MB_OK | MB_ICONSTOP);
            ExitCode |= EXIT_NO_MEMORY; //   
            return FALSE;               //  
        }
        if ((nErrorCode == ERROR_INVALID_DRIVE) || (nErrorCode == ERROR_INVALID_NAME))  //    
        {
            ExitCode |= EXIT_SYNTAX_ERROR;  //   
            ShowFileAlert ((wchar_t*) WBuffer2, -1,
              "Invalid drive or another error in the path definition", "",
              "       ", "",
              MB_OK | MB_ICONSTOP);     // 
            return FALSE;               // 
        }
        ShowFileAlert ((wchar_t*) WBuffer2, -1, "Error checking volume", "",
          "   ", "", MB_OK | MB_ICONSTOP); //   
        ExitCode |= EXIT_SYSTEM_ERROR;  //   
        return FALSE;                   //  
    }
    if (!(*((DWORD*) WBuffer1) & FS_PERSISTENT_ACLS) ||
      !(*((DWORD*) WBuffer1) & FS_FILE_COMPRESSION))    //      
    {
        ShowFileAlert ((wchar_t*) WBuffer2, -1,
          "The specified volume is not an NTFS volume", "",
          "     NTFS", "", MB_OK | MB_ICONSTOP); //   
        ExitCode |= EXIT_INVALID_VOLUME;    //   
        return FALSE;                   //  
    }
    return TRUE;                        //   
}

//---------------------------------------------------------------------------
//  CreateProcessLog
//     
//---------------------------------------------------------------------------

BOOL CreateProcessLog (void) {
    InitialStdOutHandle = GetStdHandle (STD_OUTPUT_HANDLE); //    
    if (LoggingPath)                    //   
    {
        CurrentStdOutHandle = CreateFileW (LoggingPath, GENERIC_WRITE, 0, NULL,
          (AppendLog) ? OPEN_ALWAYS : CREATE_ALWAYS,
          FILE_ATTRIBUTE_NORMAL | FILE_FLAG_SEQUENTIAL_SCAN, NULL); //   
        if (CurrentStdOutHandle == INVALID_HANDLE_VALUE)    //     
        {
            if (ShowFileAlert (LoggingPath, -1, "Cannot create log file",
              "The standard output will be used for logging",
              "   ",
              "   ",
              MB_OKCANCEL | MB_ICONWARNING) == IDCANCEL)    //   
            {
                ExitCode |= EXIT_ERROR_CREATE_LOG;  //  
                return FALSE;           //  
            }
        }
        else if (AppendLog) SetFilePointer (CurrentStdOutHandle, 0L, NULL, FILE_END);   //    
    }
    if (CurrentStdOutHandle != INVALID_HANDLE_VALUE)    //   
        DiskOut = (GetFileType (CurrentStdOutHandle) == FILE_TYPE_DISK);   //    
    else if (InitialStdOutHandle != INVALID_HANDLE_VALUE)       //    
        DiskOut = (GetFileType (InitialStdOutHandle) == FILE_TYPE_DISK);   //    
    if (!DiskOut)                       //  ,     
    {
        OEMBuffer = (char*) malloc (5120);      //     
        if (!OEMBuffer)                 //  
        {
            ShowAlert ("Insufficient memory\0\0    ",
              MB_OK | MB_ICONSTOP);
            ExitCode |= EXIT_NO_MEMORY; //   
            return FALSE;               //  
        }
    }
    return TRUE;                        // 
}

//---------------------------------------------------------------------------
//  GetPrivileges
//      
//  Know-How: Mark Russinovich and Bryce Cogswell
//---------------------------------------------------------------------------

BOOL GetPrivileges (void)
{
    HANDLE Client;
    int j;
    wchar_t* P[] = {                    //   
        SE_TAKE_OWNERSHIP_NAME,
        SE_SECURITY_NAME,
        SE_BACKUP_NAME,
        SE_RESTORE_NAME,
        SE_MACHINE_ACCOUNT_NAME,
        SE_CHANGE_NOTIFY_NAME
	};

    if (!OpenProcessToken (GetCurrentProcess (), TOKEN_ADJUST_PRIVILEGES, &Client)) //    
        return FALSE;                   //  
    TOKEN_PRIVILEGES* Priv = (TOKEN_PRIVILEGES*) WBuffer1;  //    
    for (j = 0; j < sizeof (P) / sizeof (P[0]); ++j)    //   
    {
        if (!LookupPrivilegeValueW (NULL, P[j], &Priv->Privileges[j].Luid)) //   
        {
            CloseHandle (Client);       //  
            return FALSE;               //   
        }
        Priv->Privileges[j].Attributes	= SE_PRIVILEGE_ENABLED; //  
    }
    Priv->PrivilegeCount = sizeof (P) / sizeof (P[0]);  //  
    BOOL bExStat = AdjustTokenPrivileges (Client, FALSE, Priv, 0, NULL, NULL);  //   
    CloseHandle (Client);               //  
    return bExStat;                     //  
}

//---------------------------------------------------------------------------
//   
//---------------------------------------------------------------------------
int main (int argc, char** argv)
{
    ParseSwitches (argc, argv);         //  
    WBuffer1 = malloc (10240);          //   
    WBuffer2 = malloc (10240);
    if (!WBuffer1 || !WBuffer2)         //    
    {
        ShowAlert ("Insufficient memory\0\0    ",
          MB_OK | MB_ICONSTOP);
        ExitCode |= EXIT_NO_MEMORY;     //   
    }
    else if (CheckSystemVersion ())     //  Windows NT
    {
        if (ParseTreePair (argc, argv) && SourcePath && DestinationPath)    //   
        {
            if (CheckVolume (SourcePath) && CheckVolume (DestinationPath))  //   
            {
                if (!GetPrivileges ())  //     
                {
                    ShowAlert ("Unable to obtain necessary privilege\0     ",
                      MB_OK | MB_ICONSTOP);
                    ExitCode |= EXIT_PRIVILEGE_CHECK;   //   
                }
                else if (CreateProcessLog ())   //   
                {
                    BOOL bExStat = TRUE;    //   
                    if (!SkipCopy)      //   
                    {
                        WriteLogMessage ("Copying files and folders:",
                          "   :");   //  
                        bExStat = ScanTree (SourcePath, DestinationPath, 0);    //  
                        if (bExStat) bExStat =
                          ScanTree (SourcePath, DestinationPath, 1);    //  
                    }
                    if (bExStat)        //   
                    {
                        WriteLogMessage ("Copying user access rights:",
                          "   :");   //  
                        bExStat = ScanTree (SourcePath, DestinationPath, 2);    //    
                        if (bExStat) bExStat = ScanTree (SourcePath, DestinationPath, 3); //  
                        if (bExStat && CopyRootRights)  //       
                        {
                            wchar_t* LastSlash = wcsrchr (SourcePath, L'\\');   //   
                            *(++LastSlash) = 0; //  
                            if (wcslen (SourcePath) > 3) *(--LastSlash) = 0;    //  ,     
                            LastSlash = wcsrchr (DestinationPath, L'\\');   //   
                            if (wcslen (DestinationPath) > 3) *LastSlash = 0;   //  ,     
                            TransferAccessRights (SourcePath, DestinationPath, TRUE);   //     
                        }
                    }
                }
            }
        }
        else if (CreateProcessLog ())   //   
        {
            if (UseRussian <= 0)        //  
            {
                WriteLogString ("\nNTCopy 1.1 - Copying files with user access rights\n", TRUE);
                WriteLogString ("Usage:\n\tNTCopy [-switch] SourcePath [-switch] DestinationPath [-switch]", TRUE);
                WriteLogString ("\nBoth SourcePath and DestinationPath are paths to the folders (may be relative)\nand can't contain wildcards", TRUE);
                WriteLogString ("\nAvailable switches:", TRUE);
                WriteLogString ("\t-l:e|r\tswitches to English or Russian language; uses both as default", TRUE);
                WriteLogString ("\t-o:y|n\tenables (y) or disables (n) an overwriting of existing files;\n\t\tprompts to the user as default", TRUE);
                WriteLogString ("\t-b:a|c\tforces batch mode and sets an action on error:", TRUE);
                WriteLogString ("\t\tabort (a) or continue (c) operation", TRUE);
                WriteLogString ("\t-s\tskips file copying and sets user access rights for existing\n\t\tfiles", TRUE);
                WriteLogString ("\t-r\tcopies user access rights for root directory", TRUE);
                WriteLogString ("\nYou can use both uppercase and lowercase characters to set switches", TRUE);
            }
            if (UseRussian)             //  
            {
                WriteLogString ("\nNTCopy 1.1 -      \n", TRUE);
                WriteLogString (":\n\tNTCopy [-]  [-]  [-]", TRUE);
                WriteLogString ("\n   -     (, );\n      ", TRUE);
                WriteLogString ("\n :", TRUE);
                WriteLogString ("\t-l:e|r\t    (e)   (r) ;\n\t\t     ", TRUE);
                WriteLogString ("\t-o:y|n\t (y)   (n)   ;\n\t\t    ", TRUE);
                WriteLogString ("\t-b:a|c\t      \n\t\t :", TRUE);
                WriteLogString ("\t\t (a)   (c) ", TRUE);
                WriteLogString ("\t-s\t      \n\t\t  ", TRUE);
                WriteLogString ("\t-r\t     ", TRUE);
                WriteLogString ("\n          ", TRUE);
            }
        }
    }
    if (CurrentStdOutHandle != INVALID_HANDLE_VALUE)    //   
        CloseHandle (CurrentStdOutHandle);  //   
    if (SourcePath) free (SourcePath);  //  
    if (DestinationPath) free (DestinationPath);
    if (LoggingPath) free (LoggingPath);
    if (OEMBuffer) free (OEMBuffer);
    if (WBuffer1) free (WBuffer1);
    if (WBuffer2) free (WBuffer2);
    return ExitCode;
}
//---------------------------------------------------------------------------

