using System ;
using System.Windows.Forms ;

namespace FHTE.LTOOLS.GUI
{
	/// <summary>
	/// The ListViewItemFile is derived from the ListViewItem-Class. It contains more fields
	/// to save information about a file/directory.
	/// It implements the interface IComparabel to compare different objects of this class, which allows the
	/// ArrayList-class to sort objects of this class.
	/// </summary>
    public class ListViewItemFile : ListViewItem, IComparable
    {
		#region Fields
        // ############################################################################################################
        // FIELDS
        // ############################################################################################################
        private bool isDirectory ;                              // true, if the ListViewItem represents an directory
        private bool isReadOnly ;                               // true, if the file is ReadOnly
        private bool isError = false ;                          // true, if the ListViewItem represents an ErrorMessage
        private int fileSymbol ;                                // the index of the icon for the ListViewItem
        private string fileName ;                               // the name of the file
        private string directory ;                              // the directory containing this ListViewItem (file)
        private string fullFilePath ;                           // the full path of this ListViewItem (file)
        private string fileSize ;                               // the size of the file (directory-size: 0)
        private string fileLastModified ;                       // the last-modified date of the ListViewItem (file)
        private string linuxFileInfoString ;                    // the full row of the output of ldir, if it's a 
                                                                // linux-file, e.g.: 
                                                                // drwxr-xr-x  0  0  4096 Tue Apr 08 02:07:59 2003 .
		#endregion

		#region Properties
        // ############################################################################################################
        // PROPERTIES
        // ############################################################################################################
        /// <summary>
        /// returns or sets the IsError-property, which describes an ListView as an ErrorMessage, if true.
        /// If IsError set to true, then set an error-icon for the ListViewItem.
        /// </summary>
        public bool IsError
        {
            get { return isError ; }
            set 
            { 
                isError = value ;  
                if (IsError)
                {
                    fileSymbol = DirFilePanel.IndexIconError ;
                    this.ImageIndex = fileSymbol ;
                }
            }
        }
        /// <summary>
        /// 
        /// </summary>
        public bool IsDirectory
        {
            get { return isDirectory ; }
        }
        /// <summary>
        /// returns true if the linux-file is a symbolic link else returns false
        /// </summary>
        public bool IsLink
        {
            get 
            {
                if (linuxFileInfoString != null)
                {
                    if (linuxFileInfoString[0].ToString().Equals("l"))
                    {
                        return true ;
                    }
                }
                return false ; 
            }
        }
        /// <summary>
        /// returns or sets the IsReadOnly-property
        /// </summary>
        public bool IsReadOnly
        {
            get { return isReadOnly ; }
            set { isReadOnly = value ; }
        }
        /// <summary>
        /// returns or sets the index of the icon for the ListViewItem
        /// </summary>
        public int FileSymbol
        {
            get { return fileSymbol ; }
            set { fileSymbol = value ; }
        }
        /// <summary>
        /// returns or sets the directory-property, which contains the ListViewItem
        /// </summary>
        public string Directory
        {
            get { return directory ; }
            set { directory = value ; }
        }
        /// <summary>
        /// returns or sets the fileName of the ListViewItem. and sets the index of the icon
        /// for the ListViewItem.
        /// </summary>
        public string FileName
        {
            get { return fileName ; }
            set 
            { 
                fileName = value ;
                // if the filetype is file, then set a Symbol for this File
                // default is DirFilePanel.IndexIconFile
                if (!IsDirectory)
                {
                    if (fileName.ToLower().EndsWith(".bmp") ||
                        fileName.ToLower().EndsWith(".gif") ||
                        fileName.ToLower().EndsWith(".jpg") ||
                        fileName.ToLower().EndsWith(".png"))
                    {
                        fileSymbol = DirFilePanel.IndexIconBitmap ;
                    }
                    else if (fileName.ToLower().EndsWith(".pdf"))
                    {
                        fileSymbol = DirFilePanel.IndexIconPdf ;
                    }
                    else if (fileName.ToLower().EndsWith(".html") ||
                        fileName.ToLower().EndsWith(".htm"))
                    {
                        fileSymbol = DirFilePanel.IndexIconHtml ;
                    }
                    else if (fileName.ToLower().EndsWith(".avi") ||
                        fileName.ToLower().EndsWith(".mpg"))
                    {
                        fileSymbol = DirFilePanel.IndexIconAvi ;
                    }
                    else if (fileName.ToLower().EndsWith(".mp3") ||
                        fileName.ToLower().EndsWith(".wav"))
                    {
                        fileSymbol = DirFilePanel.IndexIconMp3 ;
                    }
                    else if (fileName.ToLower().EndsWith(".doc"))
                    {
                        fileSymbol = DirFilePanel.IndexIconWord ;
                    }
					else if (fileName.ToLower().EndsWith(".xls"))
					{
						fileSymbol = DirFilePanel.IndexIconExcel ;
					}
					else if (fileName.ToLower().EndsWith(".mdb"))
					{
						fileSymbol = DirFilePanel.IndexIconAccess ;
					}
					else if (fileName.ToLower().EndsWith(".ppt"))
					{
						fileSymbol = DirFilePanel.IndexIconPowerPoint ;
					}
					else if (fileName.ToLower().EndsWith(".exe") ||
						fileName.ToLower().EndsWith(".com"))
					{
						fileSymbol = DirFilePanel.IndexIconExe ;
					}
					else if (fileName.ToLower().EndsWith(".java"))
					{
						fileSymbol = DirFilePanel.IndexIconJava ;
					}
					else if (fileName.ToLower().EndsWith(".zip"))
					{
						fileSymbol = DirFilePanel.IndexIconZip ;
					}
                    else                                        // the default-icon for files
                    {
                        fileSymbol = DirFilePanel.IndexIconFile ;
                    }
                    if (IsLink)                                 // check it if it's a symbolic link
                    {
                        fileSymbol = DirFilePanel.IndexIconLink ;
                    }
                    this.ImageIndex = fileSymbol ;
                }
            }
        }
        /// <summary>
        /// returns or sets the fullFilePath of the ListViewItem
        /// </summary>
        public string FullFilePath
        {
            get { return fullFilePath ; }
            set { fullFilePath = value ; }
        }
        /// <summary>
        /// returns or sets the fileSize of the ListViewItem (file/directory=0)
        /// </summary>
        public string FileSize
        {
            get { return fileSize ; }
            set { fileSize = value ; }
        }
        /// <summary>
        /// returns or sets the lastModified date of the ListViewItem
        /// </summary>
        public string FileLastModified
        {
            get { return fileLastModified ; }
            set { fileLastModified = value ; }
        }
        /// <summary>
        /// returns or sets the linuxFileInfoString of the ListViewItem. It's just necessary if
        /// the DirFilePanel shows linux-partitions
        /// </summary>
        public string LinuxFileInfoString
        {
            get { return linuxFileInfoString ; }
            set { linuxFileInfoString = value ; }
        }
        /// <summary>
        /// returns the drive in which the ListViewItem is saved. It's in the LinuxFileInfoString.
        /// e.g. /dev/hda6
        /// </summary>
        public string LinuxDrive
        {
            get 
            {
                if (FullFilePath != "" && LinuxFileInfoString != "")
                 {
                    return FullFilePath.Substring(0, FullFilePath.IndexOf(" ")) ;
                 }
                return "" ;
            }
        }
		#endregion

		#region Constructors
        // ############################################################################################################
        // CONSTRUCTORS
        // ############################################################################################################
        /// <summary>
        /// Constructor for ListViewItemFile-Class with just some information for e.g. ErrorMessages.
        /// </summary>
        /// <param name="text">the text shown in the listView under the ListViewItem</param>
        /// <param name="imageIndex">the index of the icon to be shown in the listView</param>
        /// <param name="isDirectory">true if the ListViewItem represents a directory</param>
        public ListViewItemFile(string text, int imageIndex, bool isDirectory) : base(text, imageIndex)
		{
			//
			// TODO: Add constructor logic here
			//
            this.isDirectory = isDirectory ;
		}
        
        /// <summary>
        /// Constructor for ListViewItemFile-Class with all necessary information.
        /// </summary>
        /// <param name="text">the text shown in the listView under the ListViewItem</param>
        /// <param name="imageIndex">the index of the icon to be shown in the listView</param>
        /// <param name="isDirectory">true if the ListViewItem represents a directory</param>
        /// <param name="dir">the directory which contains this ListViewItem</param>
        /// <param name="fSize">the size of file</param>
        /// <param name="fLastModified">the last modified date of the file</param>
        /// <param name="lxFileInfoString">the output-row of ldir for this file</param>
        public ListViewItemFile(string text, int imageIndex, bool isDirectory,
            string dir, string fSize, string fLastModified, string lxFileInfoString) : base(text, imageIndex)
        {
            this.isDirectory         = isDirectory ;    
            // first set the LinuxFileInfoString
            this.LinuxFileInfoString = lxFileInfoString ;
            this.FileName            = text ;
            this.Directory           = dir ;
			this.FullFilePath        = dir + text ;
            this.FileSize            = fSize ;
            this.FileLastModified    = fLastModified ;
        }
		#endregion

		#region Methods
        // ############################################################################################################
        // METHODS
        // ############################################################################################################
		/// <summary>
		/// the method to overwrite from the IComparable-interface to compare this instance of the class with another
		/// instance of this Type.
		/// </summary>
		/// <param name="o">the object to compare to this instance</param>
		/// <returns>-1 if the this instance is less than the object, 0 if the instance is equal to the object
		/// or 1 if the instance is greater than than the object</returns>
		public int CompareTo(object o)
		{
			// check if null
			if (o == null)
			{
				return 1 ;
			}
			// check if it's the same kind of object
			if (o.GetType() != this.GetType())
			{
				throw new ArgumentException() ;
			}
			// sort in ascending order of the name
			ListViewItemFile item = (ListViewItemFile) o ;
			return FileName.CompareTo(item.FileName) ;
		}
        
		/// <summary>
        /// the ToString() method returns a string that represents this ListViewItem
        /// </summary>
        /// <returns>a string that represents the ListViewItem</returns>
        public override string ToString()
        {
            return "ListViewItemFile { " +
                   "isDirectory: " + isDirectory.ToString() + ", " + 
                   "isReadOnly: " + isReadOnly.ToString() + ", " + 
                   "fileSymbol: " + fileSymbol.ToString() + ", " + 
                   "fileName: " + fileName + ", " + 
                   "directory: " + directory + ", " +
                   "fullFilePath: " + fullFilePath + ", " + 
                   "fileSize: " + fileSize +  ", " +
                   "fileLastModified: " + fileLastModified + ", " +
                   "linuxFileInfoString: " + linuxFileInfoString + " }" ;
        }
		#endregion
	}
}

