/*
 *  This program is copyright 1992-1994 by Vadim V. Vlasov, Moscow, Russia.
 *
 * The source is supplied as an example of interface to COMBI-disk (C)
 * device driver. You may change the source as You wish, however I'm not
 * responsible for the results of Your changes.
 *
 * Exit status:
 *            3 - COMBI-disk not found;
 *            2 - incorrect usage;
 *            1 - can't change options or reallocate buffer;
 *            0 - O'K - no errors.
 *
 * The program is written for MS C 6.0 and may be compiled in TINY or
 * SMALL memory model.
 *
 * I apologize if the code looks cumbersome but I tried to make it most
 * compatible with other compilers. (Turbo C 2.0 compiles it O'K).
 *
 */


#include <stdio.h>
#include <string.h>
#include <stdlib.h>
#include <dos.h>
#include <ctype.h>
#include "ioctl.h"    /* definitions of structures and control codes */

/* ANSI escape sequences to control colour output. If ANSI is not detected
 * all they are changed to point to an empty string. */

char *A_bold = "\x1b[1m";
char *A_red  = "\x1b[1;31m";
char *A_yell = "\x1b[1;33m";
char *A_cyan = "\x1b[1;36m";
char *A_norm = "\x1b[0m";

char   drive=2;

struct wr_ioctl ctl = {"COMBI", IOCTL_VER};

static void
show_status(struct combi_status *cmb_st)
{
  char  opt;
  char  cw_opt;

  printf("\t%sCOMBI-disk version %1d.%02d installed as drive %c:%s\n",
          A_yell,
          cmb_st->version >> 8, cmb_st->version & 0xff, drive+'A'-1, A_norm);

  printf("\n\t%sCurrent status:%s\n", A_bold, A_norm);

  printf("buffer size:  %s%5dK%s\n", A_cyan, cmb_st -> buff_size, A_norm);

  printf("current size: %s%5dK%s\n", A_cyan, cmb_st -> curr_size, A_norm);

  printf("total number of blocks:   %s%4d%s (%s%d%s sectors each)\n",
         A_cyan, cmb_st -> n_blocks, A_norm,
         A_cyan, cmb_st -> bl_sect, A_norm);

  printf("current number of blocks: %s%4d%s (%s%d%s - RAM disk, %s%d%s - cache)\n",
         A_cyan, cmb_st -> n_curr, A_norm,
         A_cyan, cmb_st -> n_bl_RD, A_norm,
         A_cyan, cmb_st -> n_bl_Cache, A_norm);

  opt = cmb_st -> COMBI_options;
  cw_opt = cmb_st -> CW_opt;

  if(opt & OPT_WR_ON) {       /* only if write caching is on */
    printf("dirty blocks: %s%d%s, errors: %s%d%s\n",
            A_cyan, cmb_st ->  n_dirty, A_norm,
            A_cyan, cmb_st -> n_errors, A_norm);
  }

  printf("\n\t%sOptions in order:%s\n", A_bold, A_norm);

  if(!(opt & OPT_OFF)) {
    printf("cache is %son%s (%s),\n", A_cyan, A_norm,
           (opt & OPT_FREEZE) ? "frozen" : "not frozen");
    printf("read ahead is %s%s%s,\n", A_cyan,
           (opt & OPT_RA_ON) ? "on"      : "off", A_norm);
    printf("write caching is %s%s%s,\n", A_cyan,
           (opt & OPT_WR_ON)  ? "on"     : "off", A_norm);
    if(opt & OPT_WR_ON) {
      printf("delayed write is %s%s%s,\n", A_cyan,
           (opt & OPT_DW_OFF) ? "off"    : "on",  A_norm);
    printf("update strategy: %s%s%s,\n", A_cyan,
           (cw_opt == OPT_WR_FIFO) ? "FIFO"      :((cw_opt == OPT_WR_SORT) ?
            "Sorted" : "LRU"), A_norm);
    }
  } else {
    printf("cache is %soff%s,\n", A_cyan, A_norm);
  }

  printf("memory allocation is %s%s%s.\n", A_cyan,
         (opt & OPT_MEM)    ? "fixed"  : "not fixed", A_norm);

  printf("\n\t%sStatistics:%s\n", A_bold, A_norm);

  printf("RAM disk reads: %s% 6u/%-6u%s\n",
          A_cyan, cmb_st -> read_RD_num, cmb_st -> read_RD_sect, A_norm);

  printf("RAM disk writes:%s% 6u/%-6u%s\n", A_cyan,
          cmb_st -> write_RD_num, cmb_st -> write_RD_sect, A_norm);

  if(!(opt & OPT_OFF)){
    printf("Hard disk reads - requested: %s% 6u/%-6u%s, passed to BIOS:%s% 6u/%-6u%s\n",
            A_cyan, cmb_st -> read_rq,    cmb_st -> read_sect, A_norm,
            A_cyan, cmb_st -> read_rq_a,  cmb_st -> read_sect_a, A_norm);
    printf("Hard disk writes - requested:%s% 6u/%-6u%s, passed to BIOS:%s% 6u/%-6u%s\n",
            A_cyan, cmb_st -> write_rq,   cmb_st -> write_sect, A_norm,
            A_cyan, cmb_st -> write_rq_a, cmb_st -> write_sect_a, A_norm);
  }

	if(cmb_st -> XMS_err_num){
		printf("XMS errors number: %s%d%s, last XMS error: %s%02Xh%s.\n",
						A_red, cmb_st -> XMS_err_num, A_norm,
						A_cyan, (cmb_st -> XMS_err) & 0xff, A_norm);
	}
}

static int
change_mem(char drive, int memch)
{
  union REGS r;

  if(memch > 0) {
    ctl.command = CMD_EXPAND;    /* command code for COMBI-disk - expand memory */
    *(int *)&ctl.params = memch; /* the amount of Kbytes to change memory size  */
  } else {
    ctl.command = CMD_SHRINK;    /* command code for COMBI-disk - shrink memory */
    *(int *)&ctl.params = - memch;
  }

  r.x.ax = 0x4405;          /* DOS function: 'write IOCtl to block device' */
  r.h.bl = drive;           /* device number  */
  r.x.dx = (unsigned)&ctl;  /* address of IOCtl packet */
  r.x.cx = 11;              /* IOCtl packet length  */

  intdos(&r,&r);

  if(r.x.cflag & 01)        /* test for error return code */
      return -1;
  else
      return 0;
}

static int
change_opt(char drive, char opt)
{
  union REGS r;

  ctl.command = CMD_CH_OPT; /* command to COMBI-disk - change options */
  ctl.params[0] = opt;      /* the byte with new options follows the command */

  r.x.ax = 0x4405;
  r.h.bl = drive;
  r.x.dx = (unsigned)&ctl;
  r.x.cx = 10;

  intdos(&r,&r);

  if(r.x.cflag & 01)
      return -1;
  else
      return 0;
}

static int
change_w_opt(char drive, char opt)
{
  union REGS r;

  ctl.command = CMD_SET_CW_OPT; /* command to COMBI-disk - change write options */
  ctl.params[0] = opt;          /* the byte with new options follows the command */

  r.x.ax = 0x4405;
  r.h.bl = drive;
  r.x.dx = (unsigned)&ctl;
  r.x.cx = 10;

  intdos(&r,&r);

  if(r.x.cflag & 01)
      return -1;
  else
      return 0;
}

static int
reset_counters(char drive)
{
  union REGS r;

  ctl.command = CMD_RESET_C;    /* command to COMBI-disk - reset counters */
  ctl.params[0] = 7;            /* reset all three types of counters */

  r.x.ax = 0x4405;
  r.h.bl = drive;
  r.x.dx = (unsigned)&ctl;
  r.x.cx = 10;

  intdos(&r,&r);

  if(r.x.cflag & 01)
      return -1;
  else
      return 0;
}

static int
flush_cache(char drive)
{
  union REGS r;

  ctl.command = CMD_FLUSH;      /* command to COMBI-disk - flush cache */

  r.x.ax = 0x4405;
  r.h.bl = drive;
  r.x.dx = (unsigned)&ctl;
  r.x.cx = 9;

  intdos(&r,&r);

  if(r.x.cflag & 01)
      return -1;
  else
      return 0;
}

static int
set_hd_parm(char drive, char bl_size, int hdrive, int hcyls,
            int hheads, int hsects)
{
  struct hdparm *hdp = (struct hdparm *) &ctl.params[1];
  union REGS r;

  ctl.command = CMD_SET_HD_PARM;

  if(hdrive > 7)
      return -1;

  ctl.params[0] = (char) hdrive | 0x80;

  if (--hcyls > 4095)
      return -1;

  hdp -> cyl_max = hcyls;

  if(--hheads > 15)
      return -1;

  hdp -> head_max = hheads;

  if(hsects > 63)
      return -1;

  hdp -> sect_max = hsects;
  hdp -> sect_per_cyl = (unsigned)hsects * (hcyls + 1);
  hdp -> bl_per_cyl = hdp -> sect_per_cyl/bl_size;
  hdp -> sec_cyl_rem = hdp -> sect_per_cyl%bl_size;

  r.x.ax = 0x4405;
  r.h.bl = drive;
  r.x.dx = (unsigned)&ctl;
  r.x.cx = 10 + sizeof(struct hdparm);

  intdos(&r,&r);

  if(r.x.cflag & 01)
      return -1;
  else
      return 0;

}

static void
usage(void)
{
  printf("\t%sCOMBI-disk (C) control program.%s\n"
         "Copyright 1992-1994 by Vadim V. Vlasov, Moscow, Russia.\n",
         A_yell, A_norm);

  printf("Usage: %sCOMBI [[+|-]<number>] [<option>[+|-]]...%s\n"
         "where:\n", A_bold, A_norm);
  printf("    %s<number>%s - expand or shrink memory used by COMBI-disk,\n",
         A_bold, A_norm);
  printf("    %sn%s  - don't return 'sector not found' error,\n",
         A_bold, A_norm);
  printf("    %so%s  - turn cache on ('%so-%s' - turn off),\n",
         A_bold, A_norm, A_bold, A_norm);
  printf("    %sa%s  - turn read ahead on ('%sa-%s' - turn off),\n",
         A_bold, A_norm, A_bold, A_norm);
  printf("    %sb%s  - turn write caching on ('%sb-%s' - off),\n",
         A_bold, A_norm, A_bold, A_norm);
  printf("    %si%s  - start writing immediately ('%si-%s' - delayed writing),\n",
         A_bold, A_norm, A_bold, A_norm);
  printf("    %sf%s  - fix memory settings ('%sf-%s' - release),\n",
         A_bold, A_norm, A_bold, A_norm);
  printf("    %sz%s  - freeze cache ('%sz-%s' - un-freese),\n",
         A_bold, A_norm, A_bold, A_norm);
  printf("    %sr%s  - reset all counters,\n",
         A_bold, A_norm);
  printf("    %su%s  - update (write to disk) all dirty sectors.\n",
         A_bold, A_norm);
  printf("    %swf%s - write sectors in FIFO order\n",
         A_bold, A_norm);
  printf("    %sws%s - write sectors in sorted order\n",
         A_bold, A_norm);
  printf("    %swl%s - write least recently used sectors first.\n",
         A_bold, A_norm);
  exit(2);
}

static void
chk_ansi(void)
{
  union REGS r;

  r.x.ax = 0x4400;  /* DOS IOCtl - get handle info */
  r.x.bx = 0x0001;  /* stdout handle */

  intdos(&r,&r);

  if((r.x.dx & 0x82) == 0x82) { /* if stdout is a standard output
                                 char device */
    r.x.ax = 0x1a00;
    int86(0x2f, &r, &r);
    if(r.h.al == 0xff)          /* if ANSI is installed */
      return;
  }

  /* Else output is redirected into file or ANSI is not installed.
   * So, we should turn off escape sequences. */

  A_bold = A_yell = A_red = A_cyan = A_norm = "";
}


main(int argc, char *argv[])
{
  struct combi_status cmb_st;
  union  REGS  r;
  int i,j;
  int exitstat = 0;             /* exit status */
  char   *chpt;                 /* pointer to next options char and */
  char   optchar;               /* options char being analyzed */
  char   oldopt, opt;           /* old and new general options */
  char   old_w_opt, w_opt;      /* same for write options */
  int    hdrive, hcyl, hhead, hsect;  /* new hard disk parameters */

  chk_ansi(); /* Determine whether ANSI driver is installed and whether
               * the output is redirected to file */
/*
 * Scan all drives and try to read IOCtl packet of length sizeof(cmb_st).
 * If successful then probably that drive is COMBI's RAM disk. To ensure
 * this we check that first 6 bytes of returned packet are "COMBI\0".
 * Note! Actual size of the structure is odd, but compilers usually roundate
 * 'sizeof' to word boundary. So, we use defined symbol CMB_ST_SIZE instead.
 */

  do {
    drive++;
    r.x.ax = 0x4404;
    r.h.bl = drive;
    r.x.cx = CMB_ST_SIZE;
    r.x.dx = (unsigned)&cmb_st;
  } while (((intdos(&r,&r)!=CMB_ST_SIZE)|| /* first check: read IOCtl must
                                              return correct number of bytes
                                              read */
            (r.x.cflag & 0x1)||
            strcmp(cmb_st.ID_name,"COMBI"))&& /* second check: "COMBI" string
                                                 in returned data */
           (drive < 25)); /* give up if drive is Z: */

  if(drive >= 25) {
    printf("%sError: COMBI-disk not installed!(?)%s\n", A_red, A_norm);
    exit(3);
  }

  if(argc > 1) {                /* any parameters?  */
    optchar = '\0';
    oldopt = opt = cmb_st.COMBI_options;
    old_w_opt = w_opt = cmb_st.CW_opt;

    for(i=1; i < argc; i++) {   /* Scan all command line arguments. */

      strlwr(argv[i]);
      if(isdigit(*argv[i]) || *argv[i]=='-' || *argv[i]=='+') {

        /* If an argument is a number then try to reallocate COMBI's
         * memory size.*/

        j = atoi(argv[i]);
        if(change_mem(drive, j)) {
          printf("%sCan't reallocate memory!%s\a\n", A_red, A_norm);
          exitstat = 1;
        } else {
          j = cmb_st.curr_size;
          r.x.ax = 0x4404;
          r.h.bl = drive;
          r.x.cx = CMB_ST_SIZE;
          r.x.dx = (unsigned)&cmb_st;
          intdos(&r,&r);        /* check the resulted buffer size */
          if(j == cmb_st.curr_size)
              printf("Buffer size unchanged. "); /* it is not necessary an
                                                    error */
          else
              printf("Memory reallocated by %+dK. ", cmb_st.curr_size - j);
        }
      } else {
        optchar = *argv[i];     /* get the char */
        chpt = argv[i]+1;       /* and advance pointer to next one */
        while(optchar) {        /* while there are any chars in argument */
          switch(optchar) {
            case 'n':
            /* Option 'n' - ignore (or not ignore if 'n-')
             * 'sector not found error' */
              if(*chpt == '-') {
                opt &= ~OPT_NO_SNF;   /* Reset the option bit */
                chpt++;               /* skip the '-' */
              } else {
                opt |= OPT_NO_SNF;    /* Set the option bit   */
                if(*chpt == '+')
                    chpt++;           /* skip the '+' */
              }
              optchar = *chpt++;      /* get next char and advance pointer */
              break;
            case 'o':
            /* Option 'o' - turn cache on (or off)  */
              if(*chpt == '-') {
                opt |= OPT_OFF;
                chpt++;
              } else {
                opt &= ~OPT_OFF;
                if(*chpt == '+')
                    chpt++;
              }
              optchar = *chpt++;
              break;
            case 'a':
            /* Option 'a' - read ahead on (or off) */
              if(*chpt == '-') {
                opt &= ~OPT_RA_ON;
                chpt++;
              } else {
                opt |= OPT_RA_ON;
                if(*chpt == '+')
                    chpt++;
              }
              optchar = *chpt++;
              break;
            case 'b':
            /* Option 'b' - turn background writing
             * (write caching) on (or off)  */
              if(*chpt == '-') {
                opt &= ~OPT_WR_ON;
                chpt++;
              } else {
                opt |= OPT_WR_ON;
                if(*chpt == '+')
                    chpt++;
              }
              optchar = *chpt++;
              break;
            case 'i':
            /* Option 'i' - write immediately
             * (or enable delayed writing if 'i-') in effect only if
             * write caching (background writing) is on  */
              if(*chpt == '-') {
                opt &= ~OPT_DW_OFF;
                chpt++;
              } else {
                opt |= OPT_DW_OFF;
                if(*chpt == '+')
                    chpt++;
              }
              optchar = *chpt++;
              break;
            case 'z':
            /* Option 'z' - freeze cache - no new sectors are read into
             * cache after freezing and only that are in cache may be written
             * in background.   */
              if(*chpt == '-') {
                opt &= ~OPT_FREEZE;
                chpt++;
              } else {
                opt |= OPT_FREEZE;
                if(*chpt == '+')
                    chpt++;
              }
              optchar = *chpt++;
              break;
            case 'f':
            /* Option 'f' - fix memory setting - disables automatic reallocation
             * of XMS memory when it is requested by other programs.  */
              if(*chpt == '-') {
                opt &= ~OPT_MEM;
                chpt++;
              } else {
                opt |= OPT_MEM;
                if(*chpt == '+')
                    chpt++;
              }
              optchar = *chpt++;
              break;
            case 'r':
            /* reset all statistics counters */
              if(reset_counters(drive)) { /* hardly possible */
                  printf("%sCan't reset counters!%s\a\n", A_red, A_norm);
                  exitstat = 1;
              } else
                  printf("Counters reset. ");
              optchar = *chpt++;
              break;
            case 'u':
            /* update all */
              if(flush_cache(drive)) {
                printf("%sErrors flushing cache!%s\a\n", A_red, A_norm);
                exitstat = 1;
              } else {
                printf("Cache flushed. ");
              }
              optchar = *chpt++;
              break;
            case 'w':
            /* set write options */
              optchar = *chpt++;    /* get next char after 'w' */
              switch(optchar) {
                case 'f':
                  w_opt = OPT_WR_FIFO;
                  break;
                case 's':
                  w_opt = OPT_WR_SORT;
                  break;
                case 'l':
                  w_opt = OPT_WR_LRU;
                  break;
                default:
                  usage();
              }
              optchar = *chpt++;
              break;
            case 'h':
            /* set hard disk parameters */
              if(sscanf(chpt,"%1d:%4d:%2d:%2d", &hdrive,&hcyl,&hhead,&hsect) != 4) {
                usage();
              } else {
                if(set_hd_parm(drive,cmb_st.bl_sect,hdrive,hcyl,hhead,hsect))
                    printf("%sBad parameters for hard disk %d !%s\n", A_red, A_norm,
                           hdrive);
                else
                    printf("Hard disk %d parameters set up. ", hdrive);
                optchar = *chpt = 0;
              }
              break;
            default:
              printf("%sInvalid option: %s'%c'%s\n", A_red, A_bold,
                     optchar, A_norm);
            case '?':
              usage();
          }
        }
      }
    }
    if((oldopt != opt) || (old_w_opt != w_opt)) {
      if((oldopt != opt) && change_opt(drive, opt) ||
         (old_w_opt != w_opt) && change_w_opt(drive, w_opt)) {
        printf("%sCan't change options!%s\a\n", A_red, A_norm);
        exitstat = 1;
      } else
        printf("Options changed.");
    } else
        printf("Options unchanged.");
    printf("\n");
  } else {                      /* no arguments */
    show_status(&cmb_st);       /* Show status returned by COMBI. */
  }

  return exitstat;
}

