Unit TMXMS;

{***************************************************************}
{**                                                           **}
{**             TMXMS - A XMS unit developed by:              **}
{**                 Telemachos^Peroxide 1998                  **}
{**                                                           **}
{**            Implements ALL functions for XMS ver2.0        **}
{***************************************************************}

INTERFACE

USES
 crt;

CONST
 unitversion : string = 'ver 1.0';

TYPE
 XMSMoveBlockT = Record
                   length : longint;
                   SourceHandle : word;
                   SourceOffset : longint;
                   DestHandle   : word;
                   DestOffset   : longint;
                 end;


VAR
 XMSFUNC      : LONGINT;           {Stores the address of the XMS function}
 XMSMOVEBLOCK : XMSMOVEBLOCKT;     {This is used to move XMS memory       }
 VERSION      : WORD;              {XMS version XX.YY. Hi-byte = XX       }
                                   {                   Lo-byte = YY       }
 XMS_FLAG     : BOOLEAN;           {Flag to see if an XMM is present      }



{***********************************************************************}
{**                      FUNCTION HEADERS                             **}
{**           Most functions returns a status byte :                  **}
{**                  $01 - call was sucessful                         **}
{**               Others - Appliable to ErrorMsg                      **}
{***********************************************************************}



{***********************************************************************}
{**  ErrorMsg - Displays an error message, then exits to DOS          **}
{***********************************************************************}
PROCEDURE ErrorMsg(nr : byte; proc : string);

{***********************************************************************}
{**  DisplayXMSInfo - Displays version number, HMA and A20 info       **}
{***********************************************************************}
PROCEDURE DisplayXMSInfo;

{***********************************************************************}
{**  HMAExist - checks for the presence of HMA                        **}
{***********************************************************************}
FUNCTION  HMAExist : boolean;

{***********************************************************************}
{**  RequestHMA - tries to reserve the 64K-16bytes of HMA for caller  **}
{***********************************************************************}
FUNCTION  RequestHMA : byte;

{***********************************************************************}
{**  ReleaseHMA - tries to release the HMA, if reserved               **}
{***********************************************************************}
FUNCTION  ReleaseHMA : byte;

{************************************************************************}
{** GlobalEnableA20 - Enables the A20 line. Only use if calling program**}
{**                   has control of the HMA                           **}
{************************************************************************}
FUNCTION  GlobalEnableA20 : byte;

{***********************************************************************}
{** GlobalDisableA20 - Globally disables the A20 line. Use before ter-**}
{**                    minating program, if enabled via function 03h  **}
{***********************************************************************}
FUNCTION  GlobalDisableA20 : byte;

{***********************************************************************}
{** LocalEnableA20 - Enabled the A20 line. Only use if program needs  **}
{**                  direct access to extended memory                 **}
{***********************************************************************}
FUNCTION  LocalEnableA20 : byte;

{***********************************************************************}
{** LocalDisableA20 - Cancel effect of function 05h (LocalEnableA20)  **}
{***********************************************************************}
FUNCTION  LocalDisableA20 : byte;

{***********************************************************************}
{** QueryA20 - Checks if A20 is physically enabled                    **}
{***********************************************************************}
FUNCTION  QueryA20 : byte;

{***********************************************************************}
{** QueryFreeXMS - returns : total_free   = total free XMS in KB      **}
{**                          largest_free = largest free block of XMS **}
{**                          in KB                                    **}
{***********************************************************************}
FUNCTION  QueryFreeXMS(var total_free : word; var largest_free : word) : byte;

{***********************************************************************}
{** AllocateXMS - Allocates a XMS memoryblock of size (in KB) and     **}
{**               assigns a handle to that block through which all    **}
{**               further handling of the memory is performed         **}
{***********************************************************************}
FUNCTION  AllocateXMS(size : word;var handle : word) : byte;

{***********************************************************************}
{** DeallocateXMS - Deallocates the XMS memory block assigned to the  **}
{**                 handle passed to the function and releases the    **}
{**                 memory to the free memory pool.                   **}
{***********************************************************************}
FUNCTION  DeallocateXMS(handle : word) : byte;

{***********************************************************************}
{** MoveXMS - Moves a block of memory from XMS to XMS, from XMS to    **}
{**           conventional memory or even from conv. memory to conv.  **}
{**           memory.                                                 **}
{**         - Length must be even.                                    **}
{**         - If a handle is set to 0 the according offset is seen as **}
{**           segment:offset in conv. memory. Thus to access a memory **}
{**           buffer at seg:ofs one would call this routine with      **}
{**           handle = 0 and offset = longint(ptr(seg,ofs))           **}
{***********************************************************************}
FUNCTION  MoveXMS(length : longint; SHandle : word; SOffs : longint;
                  DHandle : word; DOffs : longint) : byte;



{***********************************************************************}
{** LockXMSBlock - Locks the XMS block assigned to handle and returns **}
{**                the 32bit linear address of the block              **}
{***********************************************************************}
FUNCTION LockXMSBlock(handle : word;var address : longint) : byte;



{***********************************************************************}
{** UnlockXMSBlock - Unlocks a XMS assigned to handle                 **}
{***********************************************************************}
FUNCTION UnlockXMSBlock(handle : word) : byte;



{***********************************************************************}
{** GetXMSHandleInfo - Returns information on the passed XMS-handle   **}
{**                    Number of free handles in system, size of the  **}
{**                    handle and the handles lock-count.             **}
{***********************************************************************}
FUNCTION GetXMSHandleInfo(handle : word; var count, NrFreeHandles : byte;
                          var BlockSize : word) : byte;



{***********************************************************************}
{** ReAllocateXMS - Tries to reallocate a XMS block to a new size     **}
{***********************************************************************}
FUNCTION ReAllocateXMS(handle : word; newSize : Word) : byte;


{***********************************************************************}
{** RequestUMB - Tries to allocate an UMB of 'size' paragraphs. Actual**}
{**              size is returned + segment value of the block. If    **}
{**              request fails, Actual_Size equals the largest free   **}
{**              UMB available.                                       **}
{***********************************************************************}
FUNCTION RequestUMB(size : word; var Actual_Size : word;
                                 var segment : word) : byte;



{***********************************************************************}
{** GetLargestUMB - Returns the size of the largest free UMB in       **}
{**                 paragraphs.                                       **}
{***********************************************************************}
FUNCTION GetLargestUMB : word;


{***********************************************************************}
{** ReleaseUMB - Release an allocated UMB with segment value 'segment'**}
{***********************************************************************}
FUNCTION ReleaseUMB(segment : word) : byte;



IMPLEMENTATION


PROCEDURE ErrorMsg(nr : byte; proc : string);
BEGIN
Asm
 mov ax,03h
 int 10h
end;
Writeln;
Writeln('******************************************************************');
Writeln('*                                                                *');
Writeln('*                   TMXMS ',unitversion,' by Telemachos                  *');
Writeln('*                                                                *');
Writeln('******************************************************************');
Writeln;
Writeln('TMXMS has encountered an error!');
Writeln('The function ''',proc,''' returned the following error message :');
Writeln;

case nr of
   {TMXMS error messages}
   $02 : Writeln('Error - No XMS available!');
   {the XMS API error messages}
   $80 : Writeln('Error - Function not implemented!');
   $81 : Writeln('Error - VDISK device detected!');
   $82 : Writeln('Error - An A20 error has occured!');
   $8E : Writeln('Error - A general driver error has occured!');
   $8F : Writeln('Error - An unrecoverable driver error has occured!');
   $90 : Writeln('Error - The HMA does not exist!');
   $91 : Writeln('Error - The HMA is already in use!');
   $92 : Writeln('Error - Requested HMA size is smaller than defined minimum!');
   $93 : Writeln('Error - The HMA is not allocated!');
   $94 : Writeln('Error - The A20 line is still enabled!');
   $A0 : Writeln('Error - All extended memory is allocated!');
   $A1 : Writeln('Error - All available extended memory handles are in use!');
   $A2 : Writeln('Error - The handle is invalid!');
   $A3 : Writeln('Error - The Source Handle is invalid!');
   $A4 : Writeln('Error - The Source Offset is invalid!');
   $A5 : Writeln('Error - The Destination Handle is invalid!');
   $A6 : Writeln('Error - The Destination Offset is invalid!');
   $A7 : Writeln('Error - The Length is invalid! (must be a power of 2)');
   $A8 : Writeln('Error - The XMS move has an invalid overlap!');
   $A9 : Writeln('Error - A parity error has occured!');
   $AA : Writeln('Error - The block is not locked!');
   $AB : Writeln('Error - The block is locked!');
   $AC : Writeln('Error - The block''s lock count has overflowed!');
   $AD : Writeln('Error - Unable to lock XMS block!');
   $B0 : Writeln('Error - A smaller UMB is available!');
   $B1 : Writeln('Error - No UMBs are available!');
   $B2 : Writeln('Error - The UMB segment number is invalid!');
end; {case}
Writeln;
Writeln('Aborting upon keypress.....');
readkey;
HALT(1);
end;




PROCEDURE DisplayXMSInfo;
VAR
 status : byte;
BEGIN
Writeln('XMS version : ',version DIV 256,'.',version MOD 256);
if HMAExist then writeln('HMA is present!') else
                 writeln('HMA NOT present!');
status := QueryA20;
if status = $01 then Writeln('A20 line is physically enabled!') else
                     Writeln('A20 line is NOT physically enabled!');

END;



Function XMSAreYouThere : boolean;
Assembler;
asm
 mov ax,4300h
 int 2fh
 cmp al,80h
 jne @NoXMS
 mov ax,1
 jmp @out
@NoXMS:
 mov ax,0
@out:
end;

PROCEDURE GetXMSFunc;
Assembler;
asm
 mov ax,4310h
 int 2Fh
 mov word ptr [XMSfunc],bx
 mov word ptr [XMSfunc+2],es
end;


FUNCTION GetXMSVersion : word;
Assembler;
asm
 mov ah,00h
 call [XMSfunc]
end;

FUNCTION HMAExist : boolean;
Assembler;
asm
 mov ah,00h
 call [XMSfunc]
 mov ax,dx
end;


FUNCTION RequestHMA : byte;
Assembler;
asm
 mov ah,01h
 mov dx,$FFFF
 call [XMSfunc]
 cmp ax,0001h
 je @Out
 mov al,bl
@Out:
end;


FUNCTION ReleaseHMA : byte;
Assembler;
asm
 mov ah,02h
 call [XMSfunc]
 cmp ax,0001h
 je @Out
 mov al,bl
@Out:
end;


FUNCTION GlobalEnableA20 : byte;
Assembler;
asm
 mov ah,03h
 call [XMSfunc]
 cmp ax,0001h
 je @Out
 mov al,bl
@Out:
end;


FUNCTION GlobalDisableA20 : byte;
Assembler;
asm
 mov ah,04h
 call [XMSfunc]
 cmp ax,0001h
 je @Out
 mov al,bl
@Out:
end;


FUNCTION LocalEnableA20 : byte;
Assembler;
asm
 mov ah,05h
 call [XMSfunc]
 cmp ax,0001h
 je @Out
 mov al,bl
@Out:
end;


FUNCTION LocalDisableA20 : byte;
Assembler;
asm
 mov ah,06h
 call [XMSfunc]
 cmp ax,0001h
 je @Out
 mov al,bl
@Out:
end;


FUNCTION QueryA20 : byte;
Assembler;
asm
 mov ah,07h
 call [XMSfunc]
 cmp ax,0001h
 je @Out
 mov al,bl
@Out:
end;


FUNCTION QueryFreeXMS(var total_free : word; var largest_free : word) : byte;
Assembler;
asm
 mov ah,08h
 call [XMSfunc]
 les di, [largest_free]
 mov es:[di], ax
 les di, [total_free]
 mov es:[di], dx
 mov al,bl
 cmp al,$80
 je @Out
 cmp al,$81
 je @Out
 cmp al,$A0 {handle all possible error codes here because bl is NOT 1 on
             sucess!}
 je @Out
 mov al,1
@Out:
end;


FUNCTION AllocateXMS(size : word;var handle : word) : byte;
Assembler;
asm
 mov ah,09h
 mov dx,[size]
 call [XMSfunc]
 les di,[handle]
 mov es:[di], dx
 cmp al,1
 je @Out
 mov al,bl
@Out:
end;



FUNCTION DeallocateXMS(handle : word) : byte;
Assembler;
asm
 mov ah,0Ah
 mov dx,[handle]
 call [XMSfunc]
 cmp ax,0001h
 je @Out
 mov al,bl
@Out:
end;


FUNCTION MoveXMSBlock(params : XMSMoveBlockT) : byte;
Assembler;
asm
 push ds
 mov ah,0Bh
 lds si, params
 call [XMSfunc]
 pop ds
 cmp ax,0001h
 je @Out
 mov al,bl
@Out:
end;


FUNCTION MoveXMS(length : longint; SHandle : word; SOffs : longint;
                 DHandle : word; DOffs : longint) : byte;
VAR
 status : byte;
BEGIN
 XMSMoveBlock.length := length;
 XMSMoveBlock.SourceHandle := SHandle;
 XMSMoveBlock.SourceOffset := SOffs;
 XMSMoveBlock.DestHandle   := DHandle;
 XMSMoveBlock.DestOffset   := DOffs;

 Status := MoveXMSBlock(XMSMoveBlock);
 MoveXMS := status;
END;


FUNCTION LockXMSBlock(handle : word;var address : longint) : byte;
Assembler;
asm
 mov ah, 0Ch
 mov dx, handle
 call [XMSfunc]
 les di, [address]
 mov es:[di], bx
 mov es:[di+2], dx
 cmp ax, 1
 je @Out
 mov al,bl
@Out:
end;


FUNCTION UnlockXMSBlock(handle : word) : byte;
Assembler;
asm
 mov ah,0Dh
 mov dx,handle
 call [XMSfunc]
 cmp ax,1
 je @Out
 mov al,bl
@Out:
end;



FUNCTION GetXMSHandleInfo(handle : word; var count, NrFreeHandles : byte;
                                         var BlockSize : word) : byte;
Assembler;
asm
 mov ah,0Eh
 mov dx,handle
 call [XMSfunc]
 les di, [count]
 mov es:[di], bh
 les di, [NrFreeHandles]
 mov es:[di], bl
 les di, [BlockSize]
 mov es:[di], dx
 cmp al, $01
 je @Out
 mov al,bl
@Out:
end;




FUNCTION ReAllocateXMS(handle : word; newSize : Word) : byte;
Assembler;
asm
 mov ah, 0Fh
 mov dx, handle
 mov bx, NewSize
 call [XMSFunc]
 cmp ax,1
 je @Out
 mov al,bl
@Out:
end;



FUNCTION RequestUMB(size : word; var Actual_Size : word;
                                 var segment : word) : byte;
Assembler;
asm
 mov ah, 10h
 mov dx, size
 call [XMSfunc]
 les di,[Actual_Size]
 mov es:[di], dx
 les di, [segment]
 mov es:[di], bx
 cmp ax,1
 je @Out
 mov al,bl
@Out:
end;


FUNCTION GetLargestUMB : word;
Assembler;
asm
 mov ah,10h
 mov dx, $FFFF
 call [XMSfunc]
 mov ax, dx
end;


FUNCTION ReleaseUMB(segment : word) : byte;
Assembler;
asm
 mov ah,11h
 mov dx, segment
 call [XMSfunc]
 cmp ax,1
 je @Out
 mov al,bl
@Out:
end;



{******************************************************************}
{**  BEGINNING OF UNIT - SOME INITIALIZATION CODE IS PERFORMED   **}
{******************************************************************}
BEGIN
 XMS_FLAG := XMSAreYouThere;
 If not(XMS_FLAG) then ErrorMsg($02,'TMXMS Setup');
 GetXMSFunc;
 VERSION := GetXMSVersion;
END.



