-------------------------------------------------
--                DBF.e                         -
--        by Daniel Berstein Z.                 -
--       e-mail: dberst@cmet.net                -
--   November 1996, Santiago de Chile           -
-------------------------------------------------
-- Description: Reads, writes and               -
-- creates Dbase III .dbf files.                -
-------------------------------------------------
-- Instrucions: In the file dbf.txt             -
-------------------------------------------------
-- Functions:                                   -
--  s1 = dbf_open(file name)                    -
--  s3 = dbf_read(s1, record number)            -
--  s3 = dbf_readall(s1)                        -
-- Procedures:                                  -
--  dbf_close(s1)                               -
--  dbf_create(file name, s2)                   -
--  dbf_write(s1, s, record number)             -
-- Not Ready Yet:                               -
--  dbf_index(file name, s1, index key)         -
-------------------------------------------------

--------------------------------------------------------------------------------

include file.e
include get.e

-- Declare variables
integer offset
object aux1, aux2, aux3, aux4
sequence dbf_record
sequence dbf_return

-- Define constants for DBF_HEADER[]
global constant DBF_FN = 1
global constant DBF_CURRENT_RECORD = 2
global constant DBF_RECORDS = 3
global constant DBF_FIELD_NAME = 6
global constant DBF_FIELD_TYPE = 7
global constant DBF_FIELD_LENGTH = 8
global constant DBF_FIELD_DECIMAL = 9
global constant DBF_LAST_UPDATE = 10

--------------------------------------------------------------------------------

global function dbf_open(sequence dbf_file)
    -- Declare variables
    integer file_number
    sequence last_update
    integer number_of_records
    integer recno
    integer length_of_header
    integer length_of_record
    sequence field_name
    sequence field_type
    sequence field_length
    sequence field_decimal
    -- Open .dbf
    file_number = open(dbf_file, "ub")
    aux1 = seek(file_number, 0)
    -- Check if it's a good .dbf, first byte = 131 with .dbt or 3 without
    aux1 = getc(file_number)
    if aux1 != 3 and aux1 != 131 then
	return -1
    end if
    -- Read last update date (YYMMDD)
    offset = 1
    aux1 = seek(file_number, offset)
    last_update = getc(file_number) & getc(file_number) & getc(file_number)
    -- Read number of records (32-bit)
    number_of_records = getc(file_number) + (getc(file_number) * 256) + 
      (getc(file_number) * 65536) + (getc(file_number) * 16777216)
    -- Read length of header (16-bit) 
    length_of_header = getc(file_number) + (getc(file_number) * 256)
    -- Read length of each record (16-bit)
    length_of_record = getc(file_number) + (getc(file_number) * 256)
    -- Read field descriptions
    offset = 32
    field_name = {}
    field_type = {}
    field_length = {}
    field_decimal = {}
    aux2 = {}
    aux1 = seek(file_number, offset)
    while 1 do
	aux2 = {}
	for loop = 0 to 31 do
	    aux2 = append(aux2, getc(file_number))
	    -- Exit if reached the end of the header(0Dh)
	    if aux2[loop + 1] = #0D then
		aux2 = -1
		exit
	    end if
	end for
	-- Asign values to variables
	if sequence(aux2) then
	    field_name = append(field_name, aux2[1..11])
	    field_type = append(field_type, aux2[12])
	    -- If field is charcter, memo, logical or date, it has no decimals
	    if aux2[12] = 'D' or aux2[12] = 'C' or aux2[12] = 'M' or 
	      aux2[12] = 'L' then
		field_decimal = append(field_decimal, 0)
	    else
		field_decimal = append(field_decimal, aux2[18])
	    end if
	    field_length = append(field_length, aux2[17])
	else
	    -- If aux2 = -1 then finish asigning values
	    exit
	end if
    end while 
    if number_of_records <= 0 then
	recno = 0
    else
	recno = 1
    end if
    dbf_return = {file_number, recno, number_of_records, length_of_header,
      length_of_record, field_name, field_type, field_length, field_decimal,
      last_update}
    return dbf_return 
end function

global procedure dbf_close(sequence dbf_header)
    close(dbf_header[1])
end procedure

function dbf_check_num(sequence number, integer decimal)
    for loop = 1 to length(number) do
	
    end for
    aux1 = 0
    aux2 = 0
    for loop = 1 to length(number) do
	-- Valid: -, ., 0, 1, 2, 3, 4, 5, 6, 7, 8, 9 y 0, empty = space(32)
	if number[loop] >= 48 or number[loop] <= 57 or number[loop] = 46 or
	  number[loop] = 45 or number[loop] = 32 then
	else
	    aux1 = -1
	    exit
	end if
	if number[loop] = 46 then
	    if decimal = 0 then
		aux1 = -1
	    elsif loop != length(number) - decimal then
		aux1 = -1
	    end if
	end if
	if loop > 1  then
	    if number[loop] = 45 and number[loop-1] != 32 then
		aux1 = -1
	    end if
	end if
    end for
    if aux1 = -1 then
	number = repeat(0, length(number))
    end if
    return number
end function

function dbf_check_date(sequence dbf_date)
    -- Valid format: YYYYMMDD
    aux1 = 0
    for loop = 1 to length(dbf_date) do
	if dbf_date[loop] < 48 or dbf_date[loop] >57 then
	    aux1 = -1
	end if
    end for
    -- Valid dates between 01/01/0100 and 31/12/2999 (Clipper 5.2 style)
    aux4 = value(dbf_date[1..4])
    aux4 = aux4[2]
    if aux4 < 100 or aux4 > 2999 then
	aux1 = -1
    end if
    -- Check if valid month
    aux2 = value(dbf_date[5..6])
    aux2 = aux2[2]
    if aux2 < 1 or aux2 > 12 then
	aux1 = -1       
    end if
    -- Check if valid day
    aux3 = value(dbf_date[7..8])
    aux3 = aux3[2]
    if aux3 < 1 then
	aux1 = -1
    end if
    -- Check if it's a leap-year
    if integer((aux4 - (floor(aux4 / 100) * 100)) / 4) then
	-- If leap-year, february has max. 29 days
	if aux2 = 2 and aux3 > 29 then
	    aux1 = -1
	end if
    else
	-- If not leap-year, february has max. 28 days
	if aux2 = 2 and aux3 > 28 then
	    aux1 = -1
	end if
    end if
    -- Years '00 aren't leap-year, except: 400,800,1200,1600,200,2400 and 2800
    if integer(aux4 / 100) then
	if integer((aux4 / 100) / 4) then
	    if aux2 = 2 and aux3 > 29 then
		aux1 = -1
	    end if
	else
	    if aux2 = 2 and aux3 > 28 then
		aux1 = -1
	    end if
	end if
    end if
    -- April, June, September and November have 30 days
    if aux3 > 30 and (aux2=4 or aux2=6 or aux2=9 or aux2=11) then
	aux1 = -1
    -- Other months have 31 days
    elsif aux3 > 31 then
	aux1 = -1
    end if
    -- if date wasn't valid, then blank date
    if aux1 = -1 then
	dbf_date = repeat(32,8)
    end if
    return dbf_date
end function

function dbf_check_logical(sequence logical)
    aux2 = {"?","Y","y","N","n","T","t","F","f"}
    if find(logical, aux2) then
	return logical
    else
	return "?"
    end if
end function

function dbf_check_fields(sequence dbf_record, sequence dbf_header)
    -- Check number of fields 
    if length(dbf_record) > length(dbf_header[DBF_FIELD_NAME]) then
	dbf_record = dbf_record[1..length(dbf_header[DBF_FIELD_NAME])]
    end if
    -- Check length and type of fields
    for loop = 1 to length(dbf_record)  do
	if length(dbf_record[loop]) > dbf_header[DBF_FIELD_LENGTH][loop] then
	    dbf_record[loop] = dbf_record[loop][1..dbf_header[DBF_FIELD_LENGTH]
	      [loop]]
	else
	    if dbf_header[DBF_FIELD_TYPE][loop] = 'C' then
		dbf_record[loop] = dbf_record[loop] & repeat(32,
		  dbf_header[DBF_FIELD_LENGTH][loop]-length(dbf_record[loop]))
	    elsif dbf_header[DBF_FIELD_TYPE][loop] = 'N' then
		dbf_record[loop] = repeat(32, dbf_header[DBF_FIELD_LENGTH]
		  [loop]-length(dbf_record[loop])) & dbf_record[loop]
	    end if
	end if
	if dbf_header[DBF_FIELD_TYPE][loop] = 'N' then
	    dbf_record[loop] = dbf_check_num(dbf_record[loop],
	      dbf_header[DBF_FIELD_DECIMAL][loop])
	elsif dbf_header[DBF_FIELD_TYPE][loop] = 'D' then
	    dbf_record[loop] = dbf_check_date(dbf_record[loop])
	elsif dbf_header[DBF_FIELD_TYPE][loop] = 'L' then
	    dbf_record[loop] = dbf_check_logical(dbf_record[loop])
	elsif dbf_header[DBF_FIELD_TYPE][loop] = 'M' then
	elsif dbf_header[DBF_FIELD_TYPE][loop] = 'C' then
	end if
    end for
    return dbf_record
end function

global function dbf_read(sequence dbf_header, integer recno)
    -- Read records
    if dbf_header[DBF_RECORDS] < 1 then
	return -1
    end if
    if recno < 1 then
	recno = 1
    elsif recno > dbf_header[DBF_RECORDS] then
	recno = dbf_header[DBF_RECORDS]
    end if
    dbf_header[DBF_CURRENT_RECORD] = recno
    offset = (dbf_header[4] + 1) + ((recno - 1) * dbf_header[5])
    aux3 = {}
    dbf_record = repeat({}, dbf_header[DBF_RECORDS])
    aux1 = seek(dbf_header[1], offset)
    aux3 = {}
    for loop = 1 to length(dbf_header[DBF_FIELD_NAME]) do
	aux2 = {}
	for loop2 = 1 to dbf_header[DBF_FIELD_LENGTH][loop] do
	    aux2 = append(aux2, getc(dbf_header[1]))
	end for
	aux3 = append(aux3, aux2)
    end for
    -- Verify the field contents
    dbf_record = dbf_check_fields(aux3, dbf_header)
    return dbf_record
end function

global function dbf_readall(sequence dbf_header)
    --Read records
    if dbf_header[DBF_RECORDS] > 0 then
	offset = dbf_header[4] + 1
	aux3 = {}
	dbf_record = repeat({}, dbf_header[DBF_RECORDS])
	for loop = 1 to dbf_header[DBF_RECORDS] do
	    aux1 = seek(dbf_header[1], offset + (dbf_header[5] * (loop - 1)))
	    aux3 = {}
	    for loop2 = 1 to length(dbf_header[DBF_FIELD_NAME]) do
		aux2 = {}
		for loop3 = 1 to dbf_header[DBF_FIELD_LENGTH][loop2] do
		    aux2 = append(aux2, getc(dbf_header[1]))
		end for
		aux3 = append(aux3, aux2)
	    end for
	    -- Verify field contents
	    dbf_record[loop] = dbf_check_fields(aux3, dbf_header)
	end for
	return dbf_record
    else
	return -1
    end if
end function

global procedure dbf_write(sequence dbf_header, sequence dbf_record,
  integer recno)
    -- Verify field contents
    dbf_record = dbf_check_fields(dbf_record, dbf_header)
    -- if recno < 0 or > than number of records, do nothing
    if recno >= 0 or recno <= dbf_header[DBF_RECORDS] then
	-- If recno = 0, append new record
	if recno = 0  then
	    dbf_header[DBF_RECORDS] = dbf_header[DBF_RECORDS] + 1
	    recno = dbf_header[DBF_RECORDS]
	    -- Modify last update date in file header
	    aux1 = seek(dbf_header[DBF_FN], 1)  
	    aux2 = date()
	    aux2 = aux2[1..3]
	    puts(dbf_header[DBF_FN], aux2[1])
	    puts(dbf_header[DBF_FN], aux2[2])
	    puts(dbf_header[DBF_FN], aux2[3])
	    -- Modify number of records in file header
	    aux2 = {0,0,0,0}
	    aux2[4] = floor(recno/16777216)
	    aux2[3] = floor((recno-(aux2[4]*16777216))/65536)
	    aux2[2] = floor((recno-((aux2[4]*16777216)+(aux2[3]*65536)))/256)
	    aux2[1] = recno - ((aux2[4]*16777216)+(aux2[3]*65536)+(aux2[2]*256))
	    for loop = 1 to 4 do
		puts(dbf_header[DBF_FN], aux2[loop])
	    end for
	end if
	-- Position the pointer
	aux1 = seek(dbf_header[DBF_FN], (dbf_header[4] + (dbf_header[5] *
	  (recno - 1))))
	-- Write record to file
	puts(dbf_header[DBF_FN], #20)
	for loop = 1 to length(dbf_record) do
	    for loop2 = 1 to length(dbf_record[loop]) do
		puts(dbf_header[DBF_FN], dbf_record[loop][loop2])
	    end for
	end for
    end if
end procedure

global procedure dbf_create(sequence file_name, sequence structure)
    -- Check header
    for loop = 1 to length(structure) do
	-- Length of field name
	if length(structure[loop][1]) > 10 then
	    structure[loop][1] = structure[loop][1][1..10]
	end if
	-- Type of field, if invalid, defaults to 'C'haracter
	aux1 = {'C','N','L','D','M'}
	if not find(structure[loop][2], aux1) then
	    structure[loop][2] = 'C'
	end if
	-- Length of field
	if structure[loop][3] < 1 then
	    structure[loop][3] = 1
	end if
	-- Logic field, length 1
	if structure[loop][2] = 'L' then
	    structure[loop][3] = 1
	-- Date field, length 8
	elsif structure[loop][2] = 'D' then
	    structure[loop][3] = 8
	-- Memo field, length 10
	elsif structure[loop][2] = 'M' then
	    structure[loop][3] = 10
	-- Numeric field, max. length 19, max. decimals 15
	elsif structure[loop][2] = 'N' then
	    if structure[loop][3] > 19 then
		structure[loop][3] = 19
	    elsif structure[loop][4] > 15 then
		structure[loop][4] = 15
	    end if
	-- Character field, max. length 255
	elsif structure[loop][2] = 'C' and structure[loop][3] > 255 then
	    structure[loop][3] = 255
	end if
	-- Number of decimals
	if structure[loop][2] != 'N' then
	    structure[loop][4] = 0
	end if
    end for
    -- Write header information
    aux1 = open(file_name, "wb")
    aux2 = (length(structure) * 32) + 33
    aux3 = date()
    aux3 = aux3[1..3]
    -- dBASE III file without .dbt
    puts(aux1, #03)
    -- Last update date
    puts(aux1, aux3[1])
    puts(aux1, aux3[2])
    puts(aux1, aux3[3])
    -- Number of records (32-bit)
    for loop = 1 to 4 do
	puts(aux1, 0)
    end for
    -- Length of header (16-bit)
    puts(aux1, aux2 - (floor(aux2 / 256) * 256))
    puts(aux1, floor(aux2 / 256))
    -- Length of each record (16-bit)
    aux2 = 0
    for loop = 1 to length(structure) do
	aux2 = aux2 + structure[loop][3]
    end for
    aux2 = aux2 + 1
    puts(aux1, aux2 - (floor(aux2 / 256) * 256))
    puts(aux1, floor(aux2 / 256))
    -- 20 bytes reserved (version 1.00)
    for loop = 1 to 20 do
	puts(aux1, 0)
    end for
    -- Field descriptions
    for loop = 1 to length(structure) do
	-- Field name in ASCII (11 bytes)
	puts(aux1, structure[loop][1] & repeat(0, 11 - 
	  length(structure[loop][1])))
	-- Field type in ASCII (C, N, L, D, M)
	puts(aux1, structure[loop][2])
	-- Field data address (I dont't know what this is, so lets zero-fill it)
	for loop2 = 1 to 4 do
	    puts(aux1, 0)
	end for
	-- Field length
	puts(aux1, structure[loop][3])
	-- Field decimal count
	puts(aux1, structure[loop][4])
	-- 14 bytes reserved (version 1.00)
	for loop2 = 1 to 14 do
	    puts(aux1, 0)
	end for
    end for
    -- End of field descriptions
    puts(aux1, #0D)
    close(aux1)
end procedure

--------------------------------------------------------------------------------
