/*
  Multiple Corner Object
  Is a flat object with a user-defined number of corners where each corner can be
  rounded.

  (c) copyright Roman Komary Nov.1997

  POVRAY 3.0
*/


/*
  It will be lying in the x-y-plane.
  Its total center will be the origin of the coordinate space.
  The first corner will be in direction of the vector <0,1,0>

  Input Variables:
    multcorner_ccorners           // number of corners ( must be at least 3 )
    multcorner_major_radius       // main radius till the middle point of a (rounded) corner
    multcorner_corner_radius      // radius to round the corners ( 0 = no rounded corners )
                                  // these are the roundings of the corner-lines which are
                                  // parallel to the y-direction.
    multcorner_height             // height (thickness) of the flat object ( parallel z-axes )
                                  // in this height value, the thickness of the roundings (with
                                  // a radius of 'multcorner_round_radius') are included as well !
                                  // so the height value should be larger than '2*multcorner_round_radius'.
    multcorner_round_radius       // should be less or equal to 'multcorner_corner_radius'
                                  // these are the roundings for the top and bottom plane of
                                  // the box (these planes are parallel to the x-y-plane).
                                  // note: settings this to 0 will not work. set it to a very
                                  // low value (compared to the other ones) instead if you want
                                  // no roundings.
    multcorner_scale_corners      // default=1
    multcorner_scale_roundings    // scales the cylinders. default=1

  Output Object:
    MultCornerObject
*/


// First a rounded corner object
#declare MultCornerObject_RoundedCorner = intersection
{
  #declare roundcylinder_radius = multcorner_corner_radius
  #declare roundcylinder_height = multcorner_height
  #declare roundcylinder_round_radius = multcorner_round_radius
  #declare roundcylinder_scale_roundings = multcorner_scale_corners
  #include "roundcyl.pov"
  object
  {
    RoundCylinder
  }
  plane
  {
    <0,-1,0>,0
    rotate <0,0,360*0.5/multcorner_ccorners-90>
  }
  plane
  {
    <0,-1,0>,0
    rotate <0,0,-360*0.5/multcorner_ccorners+90>
  }
}


#declare MultCornerObject_RoundedBox = union
{
  intersection
  {
    #declare multcorner_edge_length = sqrt( 2*multcorner_major_radius*multcorner_major_radius -
                                            2*multcorner_major_radius*multcorner_major_radius*cos(2*pi/multcorner_ccorners) )

    #declare roundbox_cx = multcorner_edge_length+2*multcorner_round_radius
    #declare roundbox_cy = 2*multcorner_round_radius
    #declare roundbox_cz = multcorner_height
    #declare roundbox_round_radius = multcorner_round_radius
    #declare roundbox_scale_corners = multcorner_scale_roundings
    #declare roundbox_scale_roundings = multcorner_scale_roundings
    #include "roundbox.pov"
    object
    {
      RoundBox
    }
    plane
    {
      <-1,0,0>,
      multcorner_edge_length/2
    }
    plane
    {
      <1,0,0>,
      multcorner_edge_length/2
    }
    plane
    {
      <0,-1,0>,0
    }
  }

  // And add cylinders for the roundings (which is necessary if the roundings
  // are scaled larger than factor 1) cause the RoundedBox is cut by a plane
  // and the roundings also are cut.
  cylinder
  {
    <-multcorner_edge_length/2,0,multcorner_height/2-multcorner_round_radius>,
    <multcorner_edge_length/2,0,multcorner_height/2-multcorner_round_radius>,
    multcorner_round_radius*multcorner_scale_roundings
  }
  cylinder
  {
    <-multcorner_edge_length/2,0,-multcorner_height/2+multcorner_round_radius>,
    <multcorner_edge_length/2,0,-multcorner_height/2+multcorner_round_radius>,
    multcorner_round_radius*multcorner_scale_roundings
  }

  translate <multcorner_edge_length/2,multcorner_corner_radius-multcorner_round_radius,0>
  rotate <0,0,-360*0.5/multcorner_ccorners>
}


#declare MultCornerObject_NotRounded = difference
{
  box
  {
    < -multcorner_major_radius-multcorner_corner_radius,
      -multcorner_major_radius-multcorner_corner_radius,
      -multcorner_height/2 >,
    < multcorner_major_radius+multcorner_corner_radius,
      multcorner_major_radius+multcorner_corner_radius,
      multcorner_height/2 >
  }
  #declare multcorner_i=0
  #while( multcorner_i<multcorner_ccorners )
    plane
    {
      <0,-1,0>,0
      translate <0,multcorner_corner_radius-multcorner_round_radius,0>
      rotate <0,0,360*0.5/multcorner_ccorners>
      translate <0,multcorner_major_radius,0>
      rotate <0,0,360*multcorner_i/multcorner_ccorners>
    }
    intersection
    {
      plane
      {
        <0,-1,0>,0
        rotate <0,0,360*0.5/multcorner_ccorners-90>
      }
      plane
      {
        <0,-1,0>,0
        rotate <0,0,-360*0.5/multcorner_ccorners+90>
      }
      translate <0,multcorner_major_radius,0>
      rotate <0,0,360*multcorner_i/multcorner_ccorners>
    }
    #declare multcorner_i=multcorner_i+1
  #end
}


#declare MultCornerObject = merge
{
  // Not rounded MultCornerObject
  object
  {
    MultCornerObject_NotRounded
  }

  // The rounded corners
  #declare multcorner_i=0
  #while( multcorner_i<multcorner_ccorners )
    object
    {
      MultCornerObject_RoundedCorner
      translate <0,multcorner_major_radius,0>
      rotate <0,0,360*multcorner_i/multcorner_ccorners>
    }
    #declare multcorner_i=multcorner_i+1
  #end

  // The lines between the rounded corners
  #declare multcorner_i=0
  #while( multcorner_i<multcorner_ccorners )
    object
    {
      object
      {
        MultCornerObject_RoundedBox
      }
      translate <0,multcorner_major_radius,0>
      rotate <0,0,360*multcorner_i/multcorner_ccorners>
    }
    #declare multcorner_i=multcorner_i+1
  #end
}
