// c_wav_inline.h: implementation of the c_wav class.
//
//////////////////////////////////////////////////////////////////////
/*
PLAY_ITW.EXE v0.03a : Player for Impulse Tracker modules files
Copyright (C) 1998  Olivier AUMAGE
E-mail : Olivier.Aumage@ens-lyon.fr
Web : http://www.ens-lyon.fr/~oaumage/

  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; either version 2 of the License, or
  any later version.
  
	This program is distributed in the hope that it will be useful,
	but WITHOUT ANY WARRANTY; without even the implied warranty of
	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
	GNU General Public License for more details.
	
	  You should have received a copy of the GNU General Public License
	  along with this program; if not, write to the Free Software
	  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
*/

#if !defined(AFX_C_WAV_INLINE_H__2691C106_1FD0_11D1_B35E_DCE971BF2962__INCLUDED_)
#define AFX_C_WAV_INLINE_H__2691C106_1FD0_11D1_B35E_DCE971BF2962__INCLUDED_

//////////////////////////////////////////////////////////////////////
// Construction/Destruction
//////////////////////////////////////////////////////////////////////

c_wav::c_wav(char *wav_file_name, signed long sampling_rate)
{
	m_name = new char[strlen(wav_file_name) + 1];
	strcpy(m_name, wav_file_name);

#if defined __USE_STD_CPP_LIB__
	m_buffer = new char[0x80000];
	m_ifs = new ifstream(m_name, ios::in | ios::binary) ;
	m_ifs->rdbuf()->pubsetbuf(m_buffer, 0x80000); /* using a 1M buffer
												     seems to slow 
													 down the
													 std C++ lib 
													 ifstream class
													 */
#else // __USE_STD_CPP_LIB__
	m_buffer = new char[0x100000];
#ifdef __BORLANDC__
	m_ifs = new ifstream(m_name, ios::in | ios::nocreate | ios::binary, filebuf::openprot) ;
#else // __BORLANDC__
	m_ifs = new ifstream(m_name, ios::in | ios::nocreate | ios::binary, filebuf::sh_read) ;
#endif // __BORLANDC__
	m_ifs->rdbuf()->setbuf(m_buffer, 0x100000);
#endif // __USE_STD_CPP_LIB__

	load (*m_ifs) ;

	m_volume_left = 1.0 ;
	m_volume_right = 1.0 ;
	m_position = 0 ;
	m_frac_position = 0 ;
	m_step_length = (m_sampling_frequency * 256) / sampling_rate ;
	m_step_factor = 1 ;

	if (m_stereo)
	{
		m_step_factor *= 2 ;
	}
	if (m_16bit)
	{
		m_step_factor *=2 ;
	}

	m_length /= m_step_factor ;

	if (m_step_length != 256)
	{
		fill_values() ;
		fill_values() ;
	}
}

void c_wav::load(FILE *wav_file)
{
	// buffers
	unsigned short us ;
	
	{		
		/* Verification of the WAV marker 'RIFF' */
		
		char buffer [5] ;
		
		fread (buffer, (size_t) 1, (size_t) 4, wav_file) ;
		
		if (strncmp ("RIFF", buffer, 4) != 0)
		{
			//	error ("Bad wav marker") ;
		}
	}
	
	/* jumping over the 4 bytes of file size */
	(void) fseek (wav_file, 4L, SEEK_CUR) ;
	
	{		
		/* Verification of the WAV marker 'WAVE' */
		
		char buffer [5] ;
		
		fread (buffer, (size_t) 1, (size_t) 4, wav_file) ;
		
		if (strncmp ("WAVE", buffer, 4) != 0)
		{
			//	error ("Bad wav marker") ;
		}
	}

	{		
		/* Verification of the format chunk marker 'fmt ' */
		
		char buffer [5] ;
		
		fread (buffer, (size_t) 1, (size_t) 4, wav_file) ;
		
		if (strncmp ("fmt ", buffer, 4) != 0)
		{
			//	error ("Bad wav marker") ;
		}
	}

	unsigned long format_chunk_size ;
	fread (&format_chunk_size, (size_t) 1, (size_t) 4, wav_file) ;
	fread (&m_format, (size_t) 1, (size_t) 2, wav_file) ;
	if (m_format != 1)
	{
		//  error ("Unknown format");
	}
	fread (&us, (size_t) 1, (size_t) 2, wav_file) ;
	if (us == 1)
	{
		m_stereo = false ;
	}
	else if (us == 2)
	{
		m_stereo = true ;
	}
	else
	{
		// error ("unknown format") ;
	}

	fread (&m_sampling_frequency, (size_t) 1, (size_t) 4, wav_file) ;
	(void) fseek (wav_file, 4L, SEEK_CUR) ;
	fread (&us, (size_t) 1, (size_t) 2, wav_file) ;
	if (!m_stereo)
	{
		if (format_chunk_size != 16)
		{
			// error ("Unknown format") ;
		}
		if (us == 1)
		{
			m_16bit = false ;
		}
		else if (us == 2)
		{
			m_16bit = true ;
		}
		else
		{
			// error ("Unknown format") ;
		}
	}
	else
	{
		if (format_chunk_size != 18)
		{
			// error ("Unknown format") ;
		}
		fread (&us, (size_t) 1, (size_t) 2, wav_file) ;
		if (us == 8)
		{
			m_16bit = false ;
		}
		else if (us == 16)
		{
			m_16bit = true ;
		}
		else
		{
			// error ("Unknown format");
		}
	}

	{		
		/* Verification of the format chunk marker 'data' */
		
		char buffer [5] ;
		
		fread (buffer, (size_t) 1, (size_t) 4, wav_file) ;
		
		if (strncmp ("data", buffer, 4) != 0)
		{
			//	error ("Bad wav marker") ;
		}
	}

	fread (&m_length, (size_t) 1, (size_t) 4, wav_file) ;
	m_beginning = ftell(wav_file);
}

void c_wav::load(istream &is)
{
	// buffers
	unsigned short us ;
	
	{		
		/* Verification of the WAV marker 'RIFF' */
		
		char buffer [5] ;
		
		is.read(buffer, 4) ;
		
		if (strncmp ("RIFF", buffer, 4) != 0)
		{
			//	error ("Bad wav marker") ;
		}
	}
	
	/* jumping over the 4 bytes of file size */
	is.seekg((streamoff) 4, ios::cur) ;

	{		
		/* Verification of the WAV marker 'WAVE' */
		
		char buffer [5] ;
		
		is.read(buffer, 4) ;
		
		if (strncmp ("WAVE", buffer, 4) != 0)
		{
			//	error ("Bad wav marker") ;
		}
	}

	{		
		/* Verification of the format chunk marker 'fmt ' */
		
		char buffer [5] ;
		
		is.read(buffer, 4) ;
		
		if (strncmp ("fmt ", buffer, 4) != 0)
		{
			//	error ("Bad wav marker") ;
		}
	}

	unsigned long format_chunk_size ;
	is.read((char*)&format_chunk_size, 4) ;
	is.read((char*)&m_format, 2) ;
	if (m_format != 1)
	{
		//  error ("Unknown format");
	}
	is.read((char*)&us, 2) ;
	if (us == 1)
	{
		m_stereo = false ;
	}
	else if (us == 2)
	{
		m_stereo = true ;
	}
	else
	{
		// error ("unknown format") ;
	}

	is.read((char*)&m_sampling_frequency, 4) ;
	is.seekg((streamoff) 4, ios::cur) ;
	is.read((char*)&us, 2) ;
	if (!m_stereo)
	{
		if (format_chunk_size != 16)
		{
			// error ("Unknown format") ;
		}
		if (us == 1)
		{
			m_16bit = false ;
		}
		else if (us == 2)
		{
			m_16bit = true ;
		}
		else
		{
			// error ("Unknown format") ;
		}
	}
	else
	{
		if (format_chunk_size != 18)
		{
			// error ("Unknown format") ;
		}
		is.read((char*)&us, 2) ;
		if (us == 8)
		{
			m_16bit = false ;
		}
		else if (us == 16)
		{
			m_16bit = true ;
		}
		else
		{
			// error ("Unknown format");
		}
	}

	{		
		/* Verification of the format chunk marker 'data' */
		
		char buffer [5] ;
		
		is.read(buffer, 4) ;
		
		if (strncmp ("data", buffer, 4) != 0)
		{
			//	error ("Bad wav marker") ;
		}
	}

	is.read((char*)&m_length, 4) ;
	m_beginning = is.tellg() ;
}

c_wav::~c_wav()
{
	delete (m_ifs);
	delete (m_name);
	if (m_buffer != NULL)
	{
		delete m_buffer ;
	}
}

void c_wav::fill_16_stereo(double *buffer, const double *buffer_end)
{	
	const double volume_left = m_volume_left ;
	const double volume_right = m_volume_right ;

	while (buffer < buffer_end)
	{
		signed short value[2] = {0, 0};
		m_ifs->read((char *)&value, 4) ;

		buffer += 2 ;
		(*(buffer - 2)) += value[0] * volume_left ;
		(*(buffer - 1)) += value[1] * volume_right ;
	}
}

void c_wav::fill_16_mono(double *buffer, const double *buffer_end)
{	
	const double volume_left = m_volume_left ;
	const double volume_right = m_volume_right ;

	while (buffer < buffer_end)
	{
		signed short value = 0 ;
		m_ifs->read((char *)&value, 2) ;

		buffer += 2 ;
		(*(buffer - 2)) += value * volume_left ;
		(*(buffer - 1)) += value * volume_right ;
	}
}

void c_wav::fill_8_stereo(double *buffer, const double *buffer_end)
{	
	const double volume_left = m_volume_left * 256.0 ;
	const double volume_right = m_volume_right * 256.0 ;

	while (buffer < buffer_end)
	{
		unsigned char value[2] = {0, 0};
		m_ifs->read((char *)&value, 2) ;

		buffer += 2 ;
		(*(buffer - 2)) += (value[0] - 127.0) * volume_left ;
		(*(buffer - 1)) += (value[1] - 127.0) * volume_right ;
	}
}

void c_wav::fill_8_mono(double *buffer, const double *buffer_end)
{	
	const double volume_left = m_volume_left * 256.0 ;
	const double volume_right = m_volume_right * 256.0 ;

	while (buffer < buffer_end)
	{
		signed short value = 0 ;
		m_ifs->read((char *)&value, 1) ;

		value -= 127 ;

		buffer += 2 ;
		(*(buffer - 2)) += value * volume_left ;
		(*(buffer - 1)) += value * volume_right ;
	}
}

bool c_wav::fill_buffer(double *buffer, signed long buffer_length, unsigned long &position)
{
	/* samples that have a sampling frequency > m_sampling_rate are not
	   supported for now */
	double *buffer_end = buffer + (buffer_length * 2) ;
	if (m_step_length == 256)
	{
		if (m_16bit)
		{
			if (m_stereo)
			{
				fill_16_stereo(buffer, buffer_end) ;
			}
			else
			{
				fill_16_mono(buffer, buffer_end) ;
			}
		}
		else
		{
			if (m_stereo)
			{
				fill_8_stereo(buffer, buffer_end) ;
			}
			else
			{
				fill_8_mono(buffer, buffer_end) ;
			}
		}
	}
	else
	{
		double volume_left = m_volume_left / 256.0 ;
		double volume_right = m_volume_left / 256.0 ;

		while (buffer < buffer_end)
		{
			double interpolation = m_left_value * 256.0 + (m_next_left_value - m_left_value) * (double)m_frac_position ;
			(*buffer++) += interpolation * volume_left ;
			interpolation = m_right_value * 256.0 + (m_next_right_value - m_right_value) * (double)m_frac_position ;
			(*buffer++) += interpolation * volume_right ;
			
			(m_frac_position += m_step_length) ;
			if (m_frac_position > 255)
			{
				fill_values();
				m_frac_position &= 0xFF ;
			}
		}
	}
	if (m_ifs->eof())
	{
		m_ifs->clear() ;
		m_ifs->seekg((streamoff) m_beginning, ios::beg) ;
	}
	position = ((signed long)m_ifs->tellg()) - m_beginning ;
	return false ;
}

void c_wav::fill_values()
{
	m_left_value = m_next_left_value ;
	m_right_value = m_next_right_value ;

	m_next_left_value = get_value();
	m_next_right_value = (m_stereo)?get_value():m_next_left_value;

}

double c_wav::get_value()
{
/*	if (m_ifs->eof())
	{
		m_ifs->clear() ;
		m_ifs->seekg((streamoff) m_beginning, ios::beg) ;
	}*/

	if (m_16bit)
	{
		signed short value ;
		m_ifs->read((char *)&value, 2) ;
		return (double) value ;
	}
	else
	{
		unsigned char value ;
		m_ifs->read((char *)&value, 1);

		signed short value_2 = value - (signed short)127 ;
		
		return (double) value_2 * 256.0;
	}
}

void c_wav::set_mixing_volume(double mixing_volume_left, double mixing_volume_right, double panning)
{
	m_volume_left = mixing_volume_left ;
	m_volume_right = mixing_volume_right ;
}

char *c_wav::get_name()
{
	return m_name ;
}
void c_wav::rewind()
{
	m_ifs->seekg((streamoff) m_beginning, ios::beg) ;
	m_position = 0 ;
	m_frac_position = 0 ;
	fill_values() ;
	fill_values() ;
}
#endif // AFX_C_WAV_INLINE_H__2691C106_1FD0_11D1_B35E_DCE971BF2962__INCLUDED_
