// c_mixer.cpp: implementation of the c_mixer class.
//
//////////////////////////////////////////////////////////////////////
/*
PLAY_ITW.EXE v0.03a : Player for Impulse Tracker modules files
Copyright (C) 1998  Olivier AUMAGE
E-mail : Olivier.Aumage@ens-lyon.fr
Web : http://www.ens-lyon.fr/~oaumage/

  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; either version 2 of the License, or
  any later version.
  
	This program is distributed in the hope that it will be useful,
	but WITHOUT ANY WARRANTY; without even the implied warranty of
	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
	GNU General Public License for more details.
	
	  You should have received a copy of the GNU General Public License
	  along with this program; if not, write to the Free Software
	  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
*/

#if !defined(AFX_C_MIXER_INLINE_H__F7DA69C2_206F_11D1_B35E_DCE971BF2962__INCLUDED_)
#define AFX_C_MIXER_INLINE_H__F7DA69C2_206F_11D1_B35E_DCE971BF2962__INCLUDED_

//////////////////////////////////////////////////////////////////////
// Construction/Destruction
//////////////////////////////////////////////////////////////////////

c_mixer::c_mixer(char *module_file_name, 
				 signed long sampling_rate, 
				 double mixing_volume,
				 bool loop_allowed,
				 bool nnas,
				 signed long ramp_length,
				 bool linear_interpolation)
{
#if defined __USE_STD_CPP_LIB__
	ifstream ifs(module_file_name, ios::in | ios::binary) ;
#else // __USE_STD_CPP_LIB__
#if defined __BORLANDC__
	ifstream ifs(module_file_name, ios::in | ios::nocreate | ios::binary, filebuf::openprot) ;
#else // __BORLANDC__
	ifstream ifs(module_file_name, ios::in | ios::nocreate | ios::binary, filebuf::sh_read) ;
#endif // __BORLANDC__
#endif // __USE_STD_CPP_LIB__
	m_module = new c_module(ifs, 1, linear_interpolation) ; /* 1 -> IT_module 
															   2 -> RTM_modules (not ready 
																				 for now)
																							*/

	for (signed long channel_counter = 0 ; channel_counter < 64 ; channel_counter ++)
	{
		m_real_channels[channel_counter] = new c_real_channel(this,
			m_module, 
			channel_counter,
			m_module->get_channel_volume(channel_counter),
			m_module->get_channel_panning(channel_counter)) ;
	}

	for (channel_counter = 0 ; channel_counter < 256 ; channel_counter ++)
	{
		m_virtual_channels[channel_counter] = new c_virtual_channel(this, m_module) ;
	}
	
	m_init = false ;
	set_mixing_volume(mixing_volume, mixing_volume, 0.0) ;
	set_sampling_rate(sampling_rate) ;
	set_loop_allowed (loop_allowed) ;
	set_new_note_actions(nnas) ;
	set_ramp_length(ramp_length) ;
}

c_mixer::~c_mixer()
{
	for (signed long channel_counter = 0 ; channel_counter < 64 ; channel_counter ++)
	{
		delete m_real_channels[channel_counter] ;
	}

	for (channel_counter = 0 ; channel_counter < 256 ; channel_counter ++)
	{
		delete m_virtual_channels[channel_counter] ;
	}

	delete m_module ;
}

void c_mixer::new_line()
{
	m_global_volume_slide_value = 0 ;
	clear_active_channels() ;

	for (signed long local_line_counter = 0 ; local_line_counter < m_jump_to_line ; local_line_counter++)
	{
		signed long channel_variable ;

		while ((channel_variable = get_pattern_data()) != 0)  /* 0 = end of row */
		{
			m_real_channels[(channel_variable - 1) & 63]->update_line((channel_variable & 128) > 0, true) ;
		} 
		m_current_line++ ;	
	} 

	m_line_played = m_current_line ;
	m_jump_to_line = 0 ;
	m_pattern_jump = false ;
	m_pattern_break = false ;
	m_loop = false ;
	m_loop_to_line = 0 ;
	signed long channel_variable ;

	while ((channel_variable = get_pattern_data()) != 0)  /* 0 = end of row */
	{

		if (m_real_channels[(channel_variable - 1) & 63]->update_line((channel_variable & 128) > 0, false))
		{
			add_active_channel(m_real_channels[(channel_variable - 1) & 63]) ;
		}
	}	
	m_current_line++ ;	

	if (m_pattern_jump || m_pattern_break || (m_current_line >= m_pattern->get_number_of_rows()))
	{
		if (m_last_order)
		{
			m_last_line = true ;
		}
		else
		{
			m_current_line = 0 ;
			new_pattern (m_jump_to_line) ;
		}
		m_pattern_jump = false ;
		m_pattern_break = false ;
		m_jump_to_line = 0 ;
		m_loop = false ;
		m_loop_to_line = 0 ;
	}
	else if (m_loop)
	{
		m_jump_to_line = m_loop_to_line ;
		m_current_line = 0 ;
		m_pattern_position = 0 ;
		m_loop = false ;
		m_loop_to_line = 0 ;
	}

}

void c_mixer::new_frame()
{

	for (signed long channel_counter = 0 ; channel_counter < m_virtual_channels_counter ; channel_counter++ )
	{
		m_virtual_channels[channel_counter]->reset_second_volume() ;
		m_virtual_channels[channel_counter]->reset_second_panning() ;
		m_virtual_channels[channel_counter]->reset_second_pitch() ;
	}

	bool frame_0 = (m_current_frame == 0) ;
	if (frame_0)
	{
		new_line() ;
	}

	/* effects update */
	
	if (!(frame_0) && (m_global_volume_slide_value != 0))
	{
		slide_global_volume(m_global_volume_slide_value) ;
	}
	
	for (channel_counter = 0 ; channel_counter < m_active_channels_counter ; channel_counter++)
	{
		// update effects
		m_active_channels[channel_counter]->update_frame(frame_0) ;
	}
	
	for (channel_counter = 0 ; channel_counter < m_virtual_channels_counter ; channel_counter++ )
	{
		// update envelopes and fadeout
		if (m_virtual_channels[channel_counter]->update_frame(frame_0))
		{
			drop_virtual_channel(channel_counter) ;
			// Channel dropped
			channel_counter-- ;
		}
	}

	for (channel_counter = 0 ; channel_counter < m_virtual_channels_counter ; channel_counter++ )
	{
		m_virtual_channels[channel_counter]->update_volume() ;
	}

	m_current_frame++ ;
	
	if (m_current_frame >= m_frames_per_line)
	{
		m_current_frame = 0 ;
	}

}

void c_mixer::new_pattern(signed long line)
{
	m_order_played = m_current_order ;
	m_pattern = m_module->get_order_pattern(m_current_order) ;
	m_pattern_position = 0 ;
	m_current_order ++ ;
	
	while ((m_current_order < m_module->get_number_of_orders())
		&& (m_module->get_order(m_current_order) == 254))
	{
		m_current_order ++ ;
	}
	
	if ((m_current_order >= m_module->get_number_of_orders()) 
		|| (m_module->get_order(m_current_order) == 255) /* end of song */
		|| (m_module->get_order(m_current_order) >= m_module->get_number_of_patterns()))
	{
		if (m_loop_allowed)
		{
			m_current_order = 0 ;
		}
		else
		{
			m_last_order = true ;
		}
	}	
	
	if (line < m_pattern->get_number_of_rows())
	{
		m_jump_to_line = line ;
	}
	else
	{
		m_jump_to_line = 0 ;
	}

	/* channel 'new pattern' init*/
	for (signed long i = 0 ; i < 64 ; i++)
	{
		m_real_channels[i]->new_pattern() ;
	}

}
signed long c_mixer::get_current_line()
{
	return m_current_line ;
}

p_virtual_channel c_mixer::get_virtual_channel(signed long virtual_channel)
{
	return m_virtual_channels[virtual_channel] ;
}

signed long c_mixer::get_pattern_data()
{
	return m_pattern->get_data(m_pattern_position++) ;
}

void c_mixer::drop_virtual_channel(signed long virtual_channel)
{
	m_virtual_channels_counter -- ;

	p_virtual_channel pvc_temp = m_virtual_channels[m_virtual_channels_counter] ;
	p_virtual_channel pvc = m_virtual_channels[virtual_channel] ;

	m_virtual_channels[m_virtual_channels_counter] = pvc ;
	m_virtual_channels[virtual_channel] = pvc_temp ;
	
	pvc->clear_playing() ;
	if (!pvc_temp->is_in_background())
	{
		m_real_channels[pvc_temp->get_real_channel()]->set_virtual_channel(virtual_channel) ;	
	}
	if (!pvc->is_in_background())
	{
		m_real_channels[pvc->get_real_channel()]->clear_playing() ;
	}
}

signed long c_mixer::allocate_virtual_channel(signed long real_channel)
{
	if (m_virtual_channels_counter < 255)
	{
		m_virtual_channels[m_virtual_channels_counter]->init(real_channel) ;
		return m_virtual_channels_counter ++ ;
	}

	return -1 ;
}

bool c_mixer::are_new_note_actions_activated()
{
	return m_new_note_actions_activated ;
}

void c_mixer::set_speed(signed long speed)
{
	if (speed > 0)
	{
		m_frames_per_line = speed ;
	}
}

void c_mixer::jump_to_order(signed long order)
{
	m_pattern_break = false ;
	m_jump_to_line = 0 ;
	m_pattern_jump = true ;
	m_current_order = order ;
	
	while ((m_current_order < m_module->get_number_of_orders())
		&& (m_module->get_order(m_current_order) == 254))
	{
		m_current_order ++ ;
	}
	
	if ((m_current_order >= m_module->get_number_of_orders()) 
		|| (m_module->get_order(m_current_order) == 255) /* end of song */
		|| (m_module->get_order(m_current_order) >= m_module->get_number_of_patterns()))
	{
		if (m_loop_allowed)
		{
			m_current_order = 0 ;
		}
		else
		{
			m_last_order = true ;
		}
	}	
}

void c_mixer::pattern_break(signed long line)
{
	m_pattern_break = true ;
	m_jump_to_line = line ;
}

void c_mixer::loop_to_line(signed long line)
{
	if (line >= 0)
	{
		m_loop = true ;
		m_loop_to_line = line ;
	}
	else
	{
		m_loop = false ;
		m_loop_to_line = 0 ;
	}
}

void c_mixer::set_tempo(signed long tempo)
{
	if (tempo != 0)
	{
		if (tempo < 0x10)
		{
			m_beats_per_minute -= tempo ;
			if (m_beats_per_minute < 0x20)
			{
				m_beats_per_minute = 0x20 ;
			}
		}
		else if (tempo < 0x20)
		{
			tempo -= 0x10 ;
			m_beats_per_minute += tempo ;
			if (m_beats_per_minute > 0xFF)
			{
				m_beats_per_minute = 0xFF ;
			}
		}
		else
		{
			m_beats_per_minute = tempo ;
		}
		unsigned long m_frames_per_minute = m_frames_per_beat * m_beats_per_minute ;
		m_samples_per_frame = m_samples_per_minute / m_frames_per_minute ;
	}
}

void c_mixer::set_global_volume(signed long global_volume)
{
	m_global_volume = global_volume ;

	if (m_global_volume < 0)
	{
		m_global_volume = 0 ;
	}
	else if (m_global_volume > 128)
	{
		m_global_volume = 128 ;
	}
}

void c_mixer::global_volume_slide(signed long command_value)
{
	if (command_value == 0)
	{
		command_value = m_global_volume_slide  ;
	}
	else
	{
		m_global_volume_slide = command_value ;
	}

	if ((command_value & 0x0F) == 0x00) /* volume slide up */
	{
		set_global_volume_slide_value ((command_value & 0xF0) / 16) ;
	}
	else if ((command_value & 0xF0) == 0x00) /* volume slide down */
	{
		set_global_volume_slide_value (-(command_value & 0x0F)) ;
	}
	else if ((command_value & 0x0F) == 0x0F) /* fine volume slide up */
	{
		slide_global_volume((command_value & 0xF0) / 16) ;
	}
	else if ((command_value & 0xF0) == 0xF0) /* fine volume slide down */
	{
		slide_global_volume(-(command_value & 0x0F)) ;
	}
}

void c_mixer::past_note_cut(signed long channel)
{
	for (signed long channel_counter = 0 ; channel_counter < m_virtual_channels_counter ; channel_counter++)
	{
		if (channel == m_virtual_channels[channel_counter]->get_real_channel()
			&& m_virtual_channels[channel_counter]->is_in_background())
		{
			m_virtual_channels[channel_counter]->note_cut() ;
		}
	}
}

void c_mixer::past_note_off(signed long channel)
{
	for (signed long channel_counter = 0 ; channel_counter < m_virtual_channels_counter ; channel_counter++)
	{
		if (channel == m_virtual_channels[channel_counter]->get_real_channel()
			&& m_virtual_channels[channel_counter]->is_in_background())
		{
			m_virtual_channels[channel_counter]->note_off() ;
		}
	}
}

void c_mixer::past_note_fade(signed long channel)
{
	for (signed long channel_counter = 0 ; channel_counter < m_virtual_channels_counter ; channel_counter++)
	{
		if (channel == m_virtual_channels[channel_counter]->get_real_channel()
			&& m_virtual_channels[channel_counter]->is_in_background())
		{
			m_virtual_channels[channel_counter]->note_fade() ;
		}
	}
}

void c_mixer::clear_active_channels()
{
	m_active_channels_counter = 0 ;
}

void c_mixer::add_active_channel(p_real_channel active_channel)
{
#ifdef _DEBUG
assert(active_channel != (p_real_channel)NULL);
#endif
	m_active_channels[m_active_channels_counter] = active_channel ;
	m_active_channels_counter ++ ;
#ifdef _DEBUG
assert(m_active_channels_counter <= 256);
#endif
}

void c_mixer::slide_global_volume(signed long slide_value)
{
#ifdef _DEBUG
assert((m_global_volume >= 0) && (m_global_volume <= 128)) ;
#endif
	m_global_volume += slide_value ;

	if (m_global_volume < 0)
	{
		m_global_volume = 0 ;
	}
	else if (m_global_volume > 128)
	{
		m_global_volume = 128 ;
	}
}

void c_mixer::set_global_volume_slide_value(signed long slide_value)
{
	m_global_volume_slide_value = slide_value ;
}

void c_mixer::drop_active_channel(p_real_channel active_channel)
{
	for (signed long channel_counter = 0 ; channel_counter < m_active_channels_counter ; channel_counter++)
	{
		if (active_channel == m_active_channels[channel_counter])
		{
			m_active_channels_counter -- ;
			
			p_real_channel pac_temp = m_active_channels[m_active_channels_counter] ;
			p_real_channel pac = m_active_channels[m_active_channels_counter] = m_active_channels[channel_counter] ;
			m_active_channels[channel_counter] = pac_temp ;
		}
	}
}


void c_mixer::set_order(signed long order, signed long line)
{
	for (signed long channel_counter = 0 ; channel_counter < 64 ; channel_counter ++)
	{
		m_real_channels[channel_counter]->init() ;
	}
	m_samples_per_minute = ((unsigned long)m_sampling_rate) * 60UL ;
	set_speed (m_module->get_initial_speed()) ;
	set_tempo (m_module->get_initial_tempo()) ;
	m_global_volume = m_module->get_global_volume() ;
	m_global_volume_slide = 0 ;
	m_global_volume_slide_value = 0 ;

	m_pattern_jump = false ;
	m_pattern_break = false ;
	m_loop = false ;
	m_loop_to_line = 0 ;
	m_line_played = m_jump_to_line = line ;
	m_active_channels_counter = 0 ;
	m_virtual_channels_counter = 0 ;
	m_current_sample = 0 ;
	m_current_frame = 0 ;
	m_current_line = 0 ;
	m_order_played = m_current_order = order ;
	m_last_frame = false ;
	m_last_line = false ;
	m_last_order = false ;
	new_pattern(line) ;
//	new_line() ;
	new_frame() ;
}

void c_mixer::set_sampling_rate(signed long sampling_rate)
{
	m_sampling_rate = sampling_rate;
	for (signed long sample_counter = 1 ; sample_counter <= m_module->get_number_of_samples() ; sample_counter++)
	{
		p_sample psmp = m_module->get_sample(sample_counter) ;

		if (psmp)
		{
			double C5_speed = psmp->get_C5_speed() ;
			double output_C5_speed = C5_speed / m_sampling_rate ;
			m_module->get_sample(sample_counter)->set_output_C5_speed(output_C5_speed) ;
		}
	}
}

void c_mixer::set_mixing_volume(double mixing_volume_left, double mixing_volume_right, double panning)
{
	m_mixing_volume_left = mixing_volume_left * m_module->get_mixing_volume() ;
	m_mixing_volume_right = mixing_volume_right * m_module->get_mixing_volume() ;
	m_mixing_panning = panning ;
}

void c_mixer::set_loop_allowed(bool loop_allowed)
{
	m_loop_allowed = loop_allowed ;
}

void c_mixer::set_new_note_actions(bool nnas)
{
	m_new_note_actions_activated = nnas ;
}

bool c_mixer::fill_buffer(double *buffer, 
						  unsigned long buffer_length,
						  signed long &order,
						  signed long &line)
{
	if (!m_init)
	{
		set_order(0, 0) ;
		m_init = true ;
	}

	order = m_order_played ;
	line = m_line_played ;

	unsigned long beginning = 0 ;
	unsigned long end = 0 ;
	bool end_of_frame = false ;	
	
	do
	{
		beginning = end ;
		
		if ((m_samples_per_frame - m_current_sample) <= (buffer_length - beginning))
		{
			end = beginning + (m_samples_per_frame - m_current_sample) ;
			m_current_sample = m_samples_per_frame ;
			end_of_frame = true ;
		}
		else
		{
			end = buffer_length ;
			m_current_sample += buffer_length - beginning ;
			end_of_frame = false ;
		}
		
		/* channels loop */
		double *buffer_end = buffer + 2 * end ;

		do
		{
			signed long channel_counter = 0 ;
			double *buf_beg = buffer + (2 * beginning) ;
			double *buf_end ;
			if ((buffer_end - buf_beg) < 256)
			{
				buf_end = buffer_end ;
				beginning = end ;
			}
			else
			{
				buf_end = buf_beg + 256 ;
				beginning += 128 ;
			}
			
			while (channel_counter < m_virtual_channels_counter)
			{
				if (m_virtual_channels[channel_counter] -> fill_buffer(buf_beg, buf_end))
				{
					drop_virtual_channel(channel_counter) ;
				}
				else
				{
					channel_counter ++ ;
				}
			} 
		}
		while (beginning < end) ;
		
		if (end_of_frame)
		{
			if (m_last_frame)
			{
				return true ;
			}
			else
			{
				m_current_sample = 0 ;
				new_frame () ;
			}

			end_of_frame = false  ;
		}
	}
	while (end < buffer_length) ;

	return false ;
}

double c_mixer::get_constant_volume_left()
{
	return m_global_volume * m_mixing_volume_left / m_volume_divisor ;
}

double c_mixer::get_constant_volume_right()
{
	return m_global_volume * m_mixing_volume_right / m_volume_divisor ;
}

double c_mixer::get_constant_panning()
{
	return m_mixing_panning ;
}

signed long c_mixer::get_ramp_length()
{
	return m_ramp_length ;
}

void c_mixer::set_ramp_length(signed long ramp_length)
{
	m_ramp_length = ramp_length ;
}

p_module c_mixer::get_module()
{
	return m_module ;
}

const signed long c_mixer::m_frames_per_beat = 24L ;

const double c_mixer::m_volume_divisor = 
	128.0 * // mixing volume
	128.0 * // global volume
	64.0 * // volume of note
	64.0 * // volume of sample
	64.0 * // volume envelope
	64.0 * // volume of channel
	1024.0 // fadeout
	* 32.0 // panning
	* 65536.0 ; // interpolation factor
											
#endif AFX_C_MIXER_INLINE_H__F7DA69C2_206F_11D1_B35E_DCE971BF2962__INCLUDED_
