// c_driver.cpp: implementation of the c_driver class.
//
//////////////////////////////////////////////////////////////////////
/*
PLAY_ITW.EXE v0.03a : Player for Impulse Tracker modules files
Copyright (C) 1998  Olivier AUMAGE
E-mail : Olivier.Aumage@ens-lyon.fr
Web : http://www.ens-lyon.fr/~oaumage/

  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; either version 2 of the License, or
  any later version.
  
	This program is distributed in the hope that it will be useful,
	but WITHOUT ANY WARRANTY; without even the implied warranty of
	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
	GNU General Public License for more details.
	
	  You should have received a copy of the GNU General Public License
	  along with this program; if not, write to the Free Software
	  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
*/

#include <windows.h>
#include <winbase.h>

#include "interface.h"
#include "headers.h"
#include "inline.h"

/////////////////////////////////////////////////////////////////////////////
// Play It mixing engine

c_driver driver;

/////////////////////////////////////////////////////////////////////////////
// Play It error reporting engine

c_engine_error engine_error ;

/////////////////////////////////////////////////////////////////////////////
// Play It interface

void CLASS_DECLSPEC __stdcall pi_config(bool linear_interpolation, 
					  signed long sampling_rate, 
					  signed long latency,
					  bool loop_allowed,
					  bool nnas,
					  signed long ramp_length,
					  signed long priority_scheme,
					  unsigned long feedback_delay,
					  double feedback_volume, 
					  bool use_reset)
{
	driver.config(linear_interpolation, 
		sampling_rate, 
		latency,
		loop_allowed,
		nnas,
		ramp_length,
		priority_scheme,
		feedback_delay,
		feedback_volume, 
		use_reset) ;
}

void CLASS_DECLSPEC __stdcall pi_set_latency(signed long latency)
{
	driver.set_latency(latency) ;
}

void CLASS_DECLSPEC __stdcall pi_get_module_title(signed long mixer, char *buffer, signed long buffer_length)
{
	strncpy (buffer, driver.get_module_title(mixer), buffer_length - 1) ;
	buffer[buffer_length - 1] = '\0' ;
}

void CLASS_DECLSPEC __stdcall pi_unload(signed long mixer)
{
	driver.unload(mixer);
}

void CLASS_DECLSPEC __stdcall pi_stop_all()
{
	driver.stop();
}

void CLASS_DECLSPEC __stdcall pi_stop(signed long mixer)
{
	driver.stop(mixer);
}

void CLASS_DECLSPEC __stdcall pi_play_all()
{
	driver.play();
}

void CLASS_DECLSPEC __stdcall pi_play(signed long mixer)
{
	driver.play(mixer);
}

signed long CLASS_DECLSPEC __stdcall pi_load(char *module)
{
	return driver.load(module);
}

void CLASS_DECLSPEC __stdcall pi_set_mixing_volume(signed long mixer, double mixing_volume_left, double mixing_volume_right, double panning)
{
	driver.set_mixing_volume(mixer, mixing_volume_left, mixing_volume_right, panning) ;
}

void CLASS_DECLSPEC __stdcall pi_rewind_all()
{
	driver.rewind() ;
}

void CLASS_DECLSPEC __stdcall pi_rewind(signed long mixer)
{
	driver.rewind(mixer);
}

void CLASS_DECLSPEC __stdcall pi_set_feedback_delay(unsigned long feedback_delay)
{
	driver.set_feedback_delay (feedback_delay) ;
}

void CLASS_DECLSPEC __stdcall pi_set_feedback_volume(double feedback_volume)
{
	driver.set_feedback_volume (feedback_volume) ;
}

/////////////////////////////////////////////////////////////////////////////
// Initialization

bool c_driver::m_linear_interpolation = false ;
signed long c_driver::m_latency = 3 ;
signed long c_driver::m_sampling_rate = 44100 ;
bool c_driver::m_mixing_thread_is_running = false ;
volatile signed long c_driver::m_buffer_to_fill = -1 ;
volatile signed long c_driver::m_current_buffer = -1 ;
signed long c_driver::m_ramp_length = -1 ;
bool c_driver::m_nnas = false ;
bool c_driver::m_loop_allowed = false ;
bool c_driver::m_is_first_instance = true ;
bool c_driver::m_is_configured = false ;
double *c_driver::m_input_buffer = (double *)NULL ;
WAVEHDR *c_driver::m_output_buffers = (WAVEHDR *)NULL; 
signed long c_driver::m_buffer_length = -1 ; 
signed long c_driver::m_number_of_buffers = -1 ;
unsigned long c_driver::m_feedback_delay = -1 ;
double c_driver::m_feedback_volume = 0.1 ;
unsigned long c_driver::m_max_feedback_delay = -1 ;
DWORD c_driver::m_mixing_thread_id = -1 ;
HANDLE c_driver::m_mixing_thread_handle = NULL ;
SECURITY_ATTRIBUTES c_driver::m_mixing_thread_security_struct ;
HWAVEOUT c_driver::m_output_device_handler = NULL ;
c_driver *c_driver::m_the_driver = (p_driver)NULL ; 
CRITICAL_SECTION c_driver::m_mixing_critical_section ;
CRITICAL_SECTION c_driver::m_config_critical_section ;
void *c_driver::m_mixers[NB_MIXERS] ;
signed long *(c_driver::m_orders[NB_MIXERS]) ;
signed long *(c_driver::m_lines[NB_MIXERS]) ;
signed long c_driver::m_formats[NB_MIXERS] ;
bool *(c_driver::m_playings[NB_MIXERS]) ;
bool c_driver::m_playing[NB_MIXERS] ;
volatile bool c_driver::m_closed = false ;
#ifdef _DEBUG
FILE *c_driver::f = (FILE*)NULL ;
#endif

//////////////////////////////////////////////////////////////////////
// Construction/Destruction
//////////////////////////////////////////////////////////////////////

c_driver::c_driver()
{
#ifdef _DEBUG
	f = fopen("debug.log", "w");
#endif
	m_is_first_instance = false ;
	m_the_driver = this ;
}

c_driver::~c_driver()
{
	if (!m_is_configured)
	{
		return ;
	}

	stop() ;
	terminate_mixing_thread() ;

	DeleteCriticalSection(&m_mixing_critical_section) ;
	delete_output_buffers() ;
	delete_input_buffer() ;
	
	for (signed long i = 0 ; i < NB_MIXERS ; i ++)
	{
		if (m_mixers[i] != NULL)
		{
			delete[] (m_playings[i]) ;
			
			if (m_formats[i] == 1)
			{
				delete ((p_mixer)m_mixers[i]) ;
				delete[] (m_orders[i]) ;
				delete[] (m_lines[i]) ;
			}
			else if (m_formats[i] == 2)
			{
				delete ((p_wav)m_mixers[i]) ;
			}
		}
	}
}

signed long c_driver::load(char *module)
{
	if (!m_is_configured)
	{
		return -1 ;
	}

	bool mixing_thread_state = m_mixing_thread_is_running ;
	
	signed long return_value = -1 ;
	
	for (signed long counter = 0 ; counter < NB_MIXERS ; counter ++)
	{
		if (m_mixers[counter] == NULL)
		{
			break ;
		}
	}
	
	if (counter < NB_MIXERS)
	{
	/*		if (m_use_reset)
	{
	if (mixing_thread_state)
	{
				stop() ;
				}
				}
				else
				{
				EnterCriticalSection(&m_config_critical_section) ;
	}*/
		
		set_priority_scheme(2) ;
		m_formats[counter] = -1 ;
		
		{
#if defined __USE_STD_CPP_LIB__
			ifstream ifs(module, ios::in | ios::binary) ;
#else // __USE_STD_CPP_LIB__
#if defined __BORLANDC__
			ifstream ifs(module, ios::in | ios::nocreate | ios::binary, filebuf::openprot) ;
#else // __BORLANDC__
			ifstream ifs(module, ios::in | ios::nocreate | ios::binary, filebuf::sh_read) ;
#endif // __BORLANDC__
#endif // __USE_STD_CPP_LIB__
			
			char buffer[5];
			
			ifs.read(buffer, 4) ;
			if (!(strncmp ("IMPM", buffer, 4)))
			{
				m_formats[counter] = 1 ;
			}
			else if (!(strncmp ("RIFF", buffer, 4)))
			{
				m_formats[counter] = 2 ;
			}
			else
			{
				return -1 ;
			}
		}
		if (m_formats[counter] == 1)
		{
			m_playings[counter] = new bool [m_number_of_buffers] ;
			(m_playings[counter])[m_buffer_to_fill] = false ;
			
			m_playing[counter] = false ;
			m_mixers[counter] = (void *)new c_mixer(module, 
				m_sampling_rate, 
				1.0, 
				m_loop_allowed,
				m_nnas,
				m_ramp_length,
				m_linear_interpolation) ;
			m_orders[counter] = new signed long [m_number_of_buffers] ;
			(m_orders[counter])[m_buffer_to_fill] = 0 ;
			
			m_lines[counter] = new signed long [m_number_of_buffers] ;
			(m_lines[counter])[m_buffer_to_fill] = 0 ;
		}
		else if (m_formats[counter] == 2)
		{
			m_playings[counter] = new bool [m_number_of_buffers] ;
			(m_playings[counter])[m_buffer_to_fill] = false ;
			
			m_playing[counter] = false ;
			m_mixers[counter] = new c_wav(module, m_sampling_rate);
		}
		else
		{
			return -1 ;
		}
		
		
		set_priority_scheme(1) ;
		if ((bool)engine_error)
		{
			delete m_mixers[counter] ;
			delete m_playings[counter] ;
			m_orders[counter] = (signed long *)NULL ;
			m_lines[counter] = (signed long *)NULL ;
			m_playings[counter] = (bool *)NULL ;
			
			return -1 ;
		}
		
		//		m_mixing_volume[counter] = 1.0 ;
		
		
		/*		if (m_use_reset)
		{
		if (mixing_thread_state)
		{
		play() ;
		}
		}
		else
		{
		LeaveCriticalSection(&m_config_critical_section) ;
	}*/
		
		return_value = counter ;
	}
	else
	{
		return_value = -1 ;
	}	
	
	return return_value ;
}

void c_driver::play()
{
	if (!m_is_configured)
	{
		return ;
	}

	bool test = false ;
	
	for (signed long counter = 0 ; counter < NB_MIXERS ; counter++)
	{
		test = test || ((m_mixers[counter] != NULL) && m_playing[counter]) ;
	}

	if (!m_mixing_thread_is_running && test)
	{
		open_device() ;
		start_mixing_thread() ;
	}
}

void c_driver::stop()
{
	if (!m_is_configured)
	{
		return ;
	}

	if (m_mixing_thread_is_running)
	{
		EnterCriticalSection(&m_config_critical_section) ;
		stop_mixing_thread() ;
		LeaveCriticalSection(&m_config_critical_section) ;	
		close_device() ;
	}
}

void c_driver::unload(signed long mixer)
{	
	if (!m_is_configured)
	{
		return ;
	}

	if ((mixer >= 0) && (mixer < NB_MIXERS) && (m_mixers[mixer] != NULL))
	{
		bool mixing_thread_state = m_mixing_thread_is_running ;
		
		if (m_use_reset)
		{
			if (mixing_thread_state)
			{
				stop() ;
			}
		}
		else
		{
			EnterCriticalSection(&m_config_critical_section) ;
		}

		void *pointer = m_mixers[mixer] ;
		m_mixers[mixer] = NULL ;
		m_playing[mixer] = false ;

		if (m_use_reset)
		{
			if (mixing_thread_state)
			{
				play() ;
			}
		}
		else
		{
			LeaveCriticalSection(&m_config_critical_section) ;
		}
		
		if (m_formats[mixer] == 1)
		{
			delete ((p_mixer)pointer) ;
		}
		else if (m_formats[mixer] == 2)
		{
			delete ((p_wav)pointer) ;
		}
		delete[] m_orders[mixer] ;
		delete[] m_lines[mixer] ;
		delete[] m_playings[mixer] ;
		
		m_orders[mixer] = (signed long *)NULL ;
		m_lines[mixer] = (signed long *)NULL ;
		m_playings[mixer] = (bool *)NULL ;
	}	
}

void c_driver::allocate_input_buffer()
{
	m_input_buffer = new double[2 * (m_buffer_length + m_max_feedback_delay)] ;
}

void c_driver::allocate_output_buffers()
{
	m_output_buffers = new WAVEHDR[m_number_of_buffers] ;
#if defined _DEBUG
assert(m_output_buffers != NULL) ;
#endif

/*	if (m_output_buffers == (WAVEHDR *)NULL)
	{
		error("Not enough memory to allocate buffers") ;
	}*/
	
	for (signed long buffer_counter = 0 ; buffer_counter < m_number_of_buffers ; buffer_counter++)
	{
		m_output_buffers[buffer_counter].lpData = (char *) new signed short[m_buffer_length * 2] ;
#if defined _DEBUG
assert(m_output_buffers[buffer_counter].lpData != NULL) ;
#endif
		
/*		if (m_output_buffers[buffer_counter].lpData == 0)
		{
			error("Not enough memory to allocate buffers") ;
		}*/
	}
}


// Mixing thread control

void c_driver::init_mixing_thread()
{
	m_mixing_thread_security_struct.nLength = sizeof (SECURITY_ATTRIBUTES) ;
	m_mixing_thread_security_struct.lpSecurityDescriptor = NULL ;
	m_mixing_thread_security_struct.bInheritHandle = TRUE ;
	
	m_mixing_thread_handle = CreateThread (&m_mixing_thread_security_struct,
		0,
		mixing_thread_routine,
		0,
		CREATE_SUSPENDED,
		&m_mixing_thread_id
		);

	m_mixing_thread_is_running = false ;
	SetThreadPriority(m_mixing_thread_handle, THREAD_PRIORITY_ABOVE_NORMAL) ;
	
}

void c_driver::start_mixing_thread()
{
	EnterCriticalSection(&m_config_critical_section) ;
	m_mixing_thread_is_running = true ;
	ResumeThread(m_mixing_thread_handle) ;
	LeaveCriticalSection(&m_config_critical_section) ;
}

void c_driver::stop_mixing_thread()
{
	EnterCriticalSection(&m_config_critical_section) ;
	SuspendThread(m_mixing_thread_handle) ;
	m_mixing_thread_is_running = false ;
	LeaveCriticalSection(&m_config_critical_section) ;
}

void c_driver::terminate_mixing_thread()
{
	EnterCriticalSection(&m_config_critical_section) ;
    TerminateThread (m_mixing_thread_handle, 0) ;
	m_mixing_thread_is_running = false ;
	LeaveCriticalSection(&m_config_critical_section) ;
}

// Priority control

void c_driver::set_priority_scheme(signed long priority_scheme)
{
	if (priority_scheme == 3)
	{
	    SetPriorityClass (GetCurrentProcess(), REALTIME_PRIORITY_CLASS) ;
	}
	if (priority_scheme == 2)
	{
	    SetPriorityClass (GetCurrentProcess(), HIGH_PRIORITY_CLASS) ;
	}
	else if (priority_scheme == 1)
	{
		SetPriorityClass (GetCurrentProcess(), NORMAL_PRIORITY_CLASS) ;
	}
	else if (priority_scheme == 0)
	{
		SetPriorityClass (GetCurrentProcess(), IDLE_PRIORITY_CLASS) ;
	}
	
}

void c_driver::set_idle_priority()
{
	SetThreadPriority(m_mixing_thread_handle, THREAD_PRIORITY_NORMAL) ;
}

void c_driver::set_normal_priority()
{
	SetThreadPriority(m_mixing_thread_handle, THREAD_PRIORITY_ABOVE_NORMAL) ;
}

void c_driver::set_calculation_priority()
{
//	SetThreadPriority(m_mixing_thread_handle, THREAD_PRIORITY_TIME_CRITICAL) ;
	SetThreadPriority(m_mixing_thread_handle, THREAD_PRIORITY_HIGHEST) ;
}

// device control

void c_driver::open_device()
{
	WAVEFORMATEX wave_format_extended_structure ;

	wave_format_extended_structure.wFormatTag = WAVE_FORMAT_PCM ;
	wave_format_extended_structure.nChannels = 2 ;
	wave_format_extended_structure.nSamplesPerSec = (unsigned long) m_sampling_rate ;
	wave_format_extended_structure.nAvgBytesPerSec = 4UL * m_sampling_rate ;
	wave_format_extended_structure.nBlockAlign = 4 ;
	wave_format_extended_structure.wBitsPerSample = 16 ;
	wave_format_extended_structure.cbSize = 0 ;
	
	waveOutOpen (&m_output_device_handler,
		WAVE_MAPPER,
		&wave_format_extended_structure,
		(unsigned long)WaveProc,
		0,
		CALLBACK_FUNCTION
		);	
}

void c_driver::close_device()
{
	reset() ;
	waveOutClose (m_output_device_handler) ;
	while (!m_closed)
	{
	}
}

void CALLBACK c_driver::WaveProc (HWAVE hWave,
		UINT uMsg,
		DWORD dwInstance,
		DWORD dwParam1,
		DWORD dwParam2
		)
{
	EnterCriticalSection(&m_mixing_critical_section) ;

	if (uMsg == WOM_CLOSE)
	{
		m_closed = true ;
		LeaveCriticalSection(&m_mixing_critical_section) ;
		return ;
	}

	if (uMsg != WOM_DONE)
	{
		LeaveCriticalSection(&m_mixing_critical_section) ;
		return ;
	}

	waveOutUnprepareHeader (m_output_device_handler, &(m_output_buffers[m_current_buffer]), sizeof (WAVEHDR)) ;
	
	m_current_buffer ++ ;
	m_current_buffer %= m_number_of_buffers ;
	
	LeaveCriticalSection(&m_mixing_critical_section) ;
}
	
DWORD WINAPI c_driver::mixing_thread_routine (LPVOID)
{
	for (;;)
	{
		EnterCriticalSection(&m_config_critical_section) ;
		EnterCriticalSection(&m_mixing_critical_section) ;
		

		/* === BEGINNING of critical section === */
		if ((((m_buffer_to_fill + m_number_of_buffers - m_current_buffer) % m_number_of_buffers) < m_latency)
			&& (((m_buffer_to_fill + 1) % m_number_of_buffers) != m_current_buffer))
		{
			prepare_input_buffer();

//			set_calculation_priority() ;

			double *buffer = m_input_buffer + (2 * m_feedback_delay) ;

			for (signed long counter = 0 ; counter < NB_MIXERS ; counter++)
			{
				if (m_mixers[counter] != NULL)
				{
					(m_playings[counter])[m_buffer_to_fill] = m_playing[counter] ;

					if  (m_playing[counter])
					{
						if (m_formats[counter] == 1)
						{
							if (((p_mixer)m_mixers[counter])->fill_buffer(buffer,
								m_buffer_length,
								(m_orders[counter])[m_buffer_to_fill],
								(m_lines[counter])[m_buffer_to_fill]
								))
							{
								if (m_loop_allowed)
								{
									((p_mixer)m_mixers[counter])->set_order(0, 0) ;
								}
								else
								{
									m_playing[counter] = false ;
								}
							}
#ifdef _DEBUG
							fprintf(f, "%mix %d : line %d, order %d\n", counter,
								(m_orders[counter])[m_buffer_to_fill],
								(m_lines[counter])[m_buffer_to_fill]
								);

#endif
						}
						else if (m_formats[counter] == 2)
						{
							unsigned long position;
							((p_wav)m_mixers[counter])->fill_buffer(buffer,
								m_buffer_length, 
								position);
						}
					}
				}
			}

			input_to_output() ;

//			set_normal_priority() ;

			m_output_buffers[m_buffer_to_fill].dwBufferLength = m_buffer_length << 2 ;
			m_output_buffers[m_buffer_to_fill].dwFlags = 0 ;
			m_output_buffers[m_buffer_to_fill].dwLoops = 0 ;
			waveOutPrepareHeader (m_output_device_handler, &(m_output_buffers[m_buffer_to_fill]), sizeof (WAVEHDR)) ;
			m_output_buffers[m_buffer_to_fill].dwLoops = 0 ;
			waveOutWrite (m_output_device_handler, &(m_output_buffers[m_buffer_to_fill]), sizeof (WAVEHDR)) ;

			m_buffer_to_fill ++ ;
			m_buffer_to_fill %= m_number_of_buffers ;

		}
		/* === END of critical section === */

		
		LeaveCriticalSection(&m_mixing_critical_section) ;
		LeaveCriticalSection(&m_config_critical_section) ;

		Sleep(0) ;
//		SwitchToThread() ;
	}
	
	return (DWORD) 0 ; /* dummy return */
}

void c_driver::reset()
{
	for (signed long counter = 0 ; counter < NB_MIXERS ; counter++)
	{
		if ((m_mixers[counter] != NULL) && ((m_playings[counter])[m_current_buffer]))
		{
			m_playing[counter] = true ;
			if (m_formats[counter] == 1)
			{
				((p_mixer)m_mixers[counter])->set_order((m_orders[counter])[m_current_buffer], (m_lines[counter])[m_current_buffer]) ;
			}
		}
	}
	clear_input_buffer() ;
	waveOutReset (m_output_device_handler) ;
}


void c_driver::clear_input_buffer()
{
	double *buffer = m_input_buffer ;
	double *end = buffer + 2 * (m_buffer_length + m_max_feedback_delay) ;

	while (buffer < end)
	{
		*(buffer ++) = 0.0 ;
	}
}

void c_driver::clear_feedback_buffer()
{
	double *buffer = m_input_buffer ;
	double *end = buffer + 2 * (m_feedback_delay) ;

	while (buffer < end)
	{
		*(buffer ++) = 0.0 ;
	}
}

void c_driver::prepare_input_buffer()
{
	double *buffer = m_input_buffer ;
	double *end = buffer + (2 * m_feedback_delay) ;

	while (buffer < end)
	{
		*buffer = *(buffer + 2 * m_buffer_length) ;
		buffer++ ;
	}

	end = buffer + 2 * m_buffer_length ;
	while (buffer < end)
	{
		*(buffer ++) = 0.0 ;
	}
}

void c_driver::input_to_output()
{
	if (m_feedback_volume > 0.1)
	{
		/* left */
		double *input_buffer = m_input_buffer + 1 ;
		signed short *output_buffer = (signed short*) m_output_buffers[m_buffer_to_fill].lpData ;
		double *end = input_buffer + 2 * m_buffer_length ;
		unsigned long offset = 2 * m_feedback_delay - 1 ;
		
		while (input_buffer < end)
		{
			double value = *input_buffer * m_feedback_volume + *(input_buffer + offset) ;
			
			if (value < -32767.0)
			{
				*output_buffer = -32767 ;
			}
			else if (value > 32767.0)
			{
				*output_buffer = 32767 ;
			}
			else
			{
				*output_buffer = (signed short)value ;
			}
			
			input_buffer += 2 ;
			output_buffer += 2 ;
		}
		
		/* right */
		input_buffer = m_input_buffer ;
		output_buffer = (signed short*) m_output_buffers[m_buffer_to_fill].lpData + 1;	
		offset = 2 * m_feedback_delay + 1 ;
		
		while (input_buffer < end)
		{
			double value = *input_buffer * m_feedback_volume + *(input_buffer + offset) ;
			
			if (value < -32767.0)
			{
				*output_buffer = -32767 ;
			}
			else if (value > 32767.0)
			{
				*output_buffer = 32767 ;
			}
			else
			{
				*output_buffer = (signed short)value ;
			}
			
			input_buffer += 2 ;
			output_buffer += 2 ;
		}
	}
	else
	{
		double *input_buffer = m_input_buffer + 2 * m_feedback_delay ;
		signed short *output_buffer = (signed short*) m_output_buffers[m_buffer_to_fill].lpData ;
		double *end = input_buffer + 2 * m_buffer_length ;
#ifdef _DEBUG
		bool i = false ;
#endif		
		while (input_buffer < end)
		{
			double value = *(input_buffer) ;
			
			if (value < -32767.0)
			{
				*output_buffer = -32767 ;
			}
			else if (value > 32767.0)
			{
				*output_buffer = 32767 ;
			}
			else
			{
				*output_buffer = (signed short)value ;
			}
			
#ifdef _DEBUG
			if (i)
			{
				fprintf(f, "%d\n", *output_buffer) ;
			}
			else
			{
				fprintf(f, "%d ", *output_buffer) ;
			}
			i = !i ;
#endif
			input_buffer ++ ;
			output_buffer ++ ;
		}
		
	}
}


void c_driver::delete_input_buffer()
{
	delete[] m_input_buffer ;
}

void c_driver::delete_output_buffers()
{
	for (signed long buffer_counter = 0 ; buffer_counter < m_number_of_buffers ; buffer_counter++)
	{
		delete[] (signed short *)m_output_buffers[buffer_counter].lpData ;
	}

	delete[] m_output_buffers ;
}

// interactive commands

void c_driver::set_sampling_rate(signed long sampling_rate)
{
	if (!m_is_configured)
	{
		return ;
	}

	bool mixing_thread_state = m_mixing_thread_is_running ;
	if (m_use_reset)
	{
		if (mixing_thread_state)
		{
			stop() ;
		}
	}
	else
	{
		EnterCriticalSection(&m_config_critical_section) ;
	}


	m_sampling_rate = sampling_rate ;

	for (signed long counter = 0 ; counter < NB_MIXERS ; counter++)
	{
		if (m_mixers[counter] != NULL)
		{			
			if (m_formats[counter] == 1)
			{
				((p_mixer)m_mixers[counter])->set_sampling_rate(sampling_rate) ;
			}
			else if (m_formats[counter] == 2)
			{
//				((p_wav)m_mixers[counter])->set_sampling_rate(sampling_rate) ;
			}
		}
	}

	if (m_use_reset)
	{
		if (mixing_thread_state)
		{
			play() ;
		}
	}
	else
	{
		LeaveCriticalSection(&m_config_critical_section) ;
	}
}

void c_driver::set_ramp_length(signed long ramp_length)
{
	if (!m_is_configured)
	{
		return ;
	}

	bool mixing_thread_state = m_mixing_thread_is_running ;
	if (m_use_reset)
	{
		if (mixing_thread_state)
		{
			stop() ;
		}
	}
	else
	{
		EnterCriticalSection(&m_config_critical_section) ;
	}

	m_ramp_length = ramp_length ;

	for (signed long counter = 0 ; counter < NB_MIXERS ; counter++)
	{
		if (m_mixers[counter] != NULL)
		{			
			if (m_formats[counter] == 1)
			{
				((p_mixer)m_mixers[counter])->set_ramp_length(ramp_length) ;
			}
		}
	}

	if (m_use_reset)
	{
		if (mixing_thread_state)
		{
			play() ;
		}
	}
	else
	{
		LeaveCriticalSection(&m_config_critical_section) ;
	}
}

void c_driver::set_nnas(bool nnas)
{
	if (!m_is_configured)
	{
		return ;
	}

	bool mixing_thread_state = m_mixing_thread_is_running ;
	if (m_use_reset)
	{
		if (mixing_thread_state)
		{
			stop() ;
		}
	}
	else
	{
		EnterCriticalSection(&m_config_critical_section) ;
	}

	m_nnas = nnas ;

	for (signed long counter = 0 ; counter < NB_MIXERS ; counter++)
	{
		if (m_mixers[counter] != NULL)
		{			
			if (m_formats[counter] == 1)
			{
				((p_mixer)m_mixers[counter])->set_new_note_actions(nnas) ;
			}
		}
	}

	if (m_use_reset)
	{
		if (mixing_thread_state)
		{
			play() ;
		}
	}
	else
	{
		LeaveCriticalSection(&m_config_critical_section) ;
	}
}

void c_driver::set_loop_allowed(bool loop_allowed)
{
	if (!m_is_configured)
	{
		return ;
	}

	bool mixing_thread_state = m_mixing_thread_is_running ;
	if (m_use_reset)
	{
		if (mixing_thread_state)
		{
			stop() ;
		}
	}
	else
	{
		EnterCriticalSection(&m_config_critical_section) ;
	}

	m_loop_allowed = loop_allowed ;
	
	for (signed long counter = 0 ; counter < NB_MIXERS ; counter++)
	{
		if (m_mixers[counter] != NULL)
		{			
			if (m_formats[counter] == 1)
			{
				((p_mixer)m_mixers[counter])->set_loop_allowed(loop_allowed) ;
			}
		}
	}

	if (m_use_reset)
	{
		if (mixing_thread_state)
		{
			play() ;
		}
	}
	else
	{
		LeaveCriticalSection(&m_config_critical_section) ;
	}
}

void c_driver::set_feedback_delay(unsigned long feedback_delay)
{
	if (!m_is_configured)
	{
		return ;
	}

	bool mixing_thread_state = m_mixing_thread_is_running ;
	if (m_use_reset)
	{
		if (mixing_thread_state)
		{
			stop() ;
		}
	}
	else
	{
		EnterCriticalSection(&m_config_critical_section) ;
	}


	m_feedback_delay = (feedback_delay < m_max_feedback_delay)?feedback_delay:m_max_feedback_delay;
	clear_feedback_buffer() ;


	if (m_use_reset)
	{
		if (mixing_thread_state)
		{
			play() ;
		}
	}
	else
	{
		LeaveCriticalSection(&m_config_critical_section) ;
	}
}

void c_driver::set_feedback_volume(double feedback_volume)
{
	if (!m_is_configured)
	{
		return ;
	}

	bool mixing_thread_state = m_mixing_thread_is_running ;

	if (m_use_reset)
	{
		if (mixing_thread_state)
		{
			stop() ;
		}
	}
	else
	{
		EnterCriticalSection(&m_config_critical_section) ;
	}

	m_feedback_volume = feedback_volume ;

	if (m_use_reset)
	{
		if (mixing_thread_state)
		{
			play() ;
		}
	}
	else
	{
		LeaveCriticalSection(&m_config_critical_section) ;
	}
}

void c_driver::play(signed long mixer)
{
	if (!m_is_configured)
	{
		return ;
	}

	if (m_mixers[mixer] != NULL)
	{
		bool mixing_thread_state = m_mixing_thread_is_running ;
		
		if (m_use_reset)
		{
			if (mixing_thread_state)
			{
				stop() ;
			}
		}
		else
		{
			EnterCriticalSection(&m_config_critical_section) ;
		}
		
		(m_playings[mixer])[m_buffer_to_fill] = m_playing[mixer] = true ;
		
		if (!m_use_reset)
		{
			LeaveCriticalSection(&m_config_critical_section) ;
		}

		play() ;
	}
}

void c_driver::stop(signed long mixer)
{
	if (!m_is_configured)
	{
		return ;
	}

	if (m_mixers[mixer] != NULL)
	{
		bool mixing_thread_state = m_mixing_thread_is_running ;
		
		if (m_use_reset)
		{
			if (mixing_thread_state)
			{
				stop() ;
			}
		}
		else
		{
			EnterCriticalSection(&m_config_critical_section) ;
		}
		
		if (m_use_reset)
		{
			(m_playings[mixer])[m_current_buffer] = m_playing[mixer] = false ;
		}
		else
		{
			(m_playings[mixer])[m_buffer_to_fill] = m_playing[mixer] = false ;
		}

		bool test = false ;

		for (signed long counter = 0 ; counter < NB_MIXERS ; counter++)
		{
			test = test || ((m_mixers[counter] != NULL) && m_playing[counter]) ;
		}
		
		if (m_use_reset)
		{
			if (mixing_thread_state && test)
			{
				play() ;
			}
		}
		else
		{
			LeaveCriticalSection(&m_config_critical_section) ;
			if (mixing_thread_state && !test)
			{
				(m_playings[mixer])[m_current_buffer] = false ;
				stop() ;
			}
		}
	}
}

void c_driver::set_mixing_volume(signed long mixer, double mixing_volume_left, double mixing_volume_right, double panning)
{
	if (!m_is_configured)
	{
		return ;
	}

	if (m_mixers[mixer] != NULL)
	{
		bool mixing_thread_state = m_mixing_thread_is_running ;
		
		if (m_use_reset)
		{
			if (mixing_thread_state)
			{
				stop() ;
			}
		}
		else
		{
			EnterCriticalSection(&m_config_critical_section) ;
		}
		
/*		m_mixing_volume_left[mixer] = mixing_volume_left ;
		m_mixing_volume_right[mixer] = mixing_volume_right ;*/
		if (m_formats[mixer] == 1)
		{
			((p_mixer)m_mixers[mixer])->set_mixing_volume(mixing_volume_left, mixing_volume_right, panning) ;
		}
		else if (m_formats[mixer] == 2)
		{
			((p_wav)m_mixers[mixer])->set_mixing_volume(mixing_volume_left, mixing_volume_right, panning) ;
		}

		bool test = false ;

		
		if (m_use_reset)
		{
			if (mixing_thread_state && test)
			{
				play() ;
			}
		}
		else
		{
			LeaveCriticalSection(&m_config_critical_section) ;
		}
	}
}


char* c_driver::get_module_title(signed long mixer)
{
	if (!m_is_configured)
	{
		return (char*)NULL;
	}

	if (m_mixers[mixer] != NULL)
	{
		if (m_formats[mixer] == 1)
		{
			return (char *)((p_mixer)m_mixers[mixer])->get_module()->get_name() ;
		}
		else if (m_formats[mixer] == 2)
		{
			return (char *)((p_wav)m_mixers[mixer])->get_name() ;
		}
	} 
	return (char *)NULL ;
}

void c_driver::rewind()
{
	if (!m_is_configured)
	{
		return ;
	}

	bool mixing_thread_state = m_mixing_thread_is_running ;
	
	if (m_use_reset)
	{
		if (mixing_thread_state)
		{
			stop() ;
		}
	}
	else
	{
		EnterCriticalSection(&m_config_critical_section) ;
	}

	for (signed long counter = 0 ; counter < NB_MIXERS ; counter++)
	{
		if (m_mixers[counter] != NULL)
		{
			if (m_use_reset)
			{
				if (m_formats[counter] == 1)
				{
					((p_mixer)m_mixers[counter])->set_order(0, 0) ;
					(m_orders[counter])[m_current_buffer] = 0 ;
					(m_lines[counter])[m_current_buffer] = 0 ;
				}
				else if (m_formats[counter] == 2)
				{
					((p_wav)m_mixers[counter])->rewind() ;
				}
			}
			else
			{
				if (m_formats[counter] == 1)
				{
					((p_mixer)m_mixers[counter])->set_order(0, 0) ;
					(m_orders[counter])[m_buffer_to_fill] = 0 ;
					(m_lines[counter])[m_buffer_to_fill] = 0 ;
				}
				else if (m_formats[counter] == 2)
				{
					((p_wav)m_mixers[counter])->rewind() ;
				}
			}
		}
	}
	
	if (m_use_reset)
	{
		if (mixing_thread_state)
		{
			play() ;
		}
	}
	else
	{
		LeaveCriticalSection(&m_config_critical_section) ;
	}
}

void c_driver::rewind(signed long mixer)
{
	if (!m_is_configured)
	{
		return ;
	}

	bool mixing_thread_state = m_mixing_thread_is_running ;
	
	if (m_use_reset)
	{
		if (mixing_thread_state)
		{
			stop() ;
		}
	}
	else
	{
		EnterCriticalSection(&m_config_critical_section) ;
	}

	if (m_mixers[mixer] != NULL)
	{
		if (m_use_reset)
		{
			if (m_formats[mixer] == 1)
			{
				((p_mixer)m_mixers[mixer])->set_order(0, 0) ;
				(m_orders[mixer])[m_current_buffer] = 0 ;
				(m_lines[mixer])[m_current_buffer] = 0 ;
			}
			else if (m_formats[mixer] == 2)
			{
				((p_wav)m_mixers[mixer])->rewind() ;
			}
		}
		else
		{
			if (m_formats[mixer] == 1)
			{
				((p_mixer)m_mixers[mixer])->set_order(0, 0) ;
				(m_orders[mixer])[m_buffer_to_fill] = 0 ;
				(m_lines[mixer])[m_buffer_to_fill] = 0 ;
			}
			else if (m_formats[mixer] == 2)
			{
				((p_wav)m_mixers[mixer])->rewind() ;
			}
		}
	}
	
	if (m_use_reset)
	{
		if (mixing_thread_state)
		{
			play() ;
		}
	}
	else
	{
		LeaveCriticalSection(&m_config_critical_section) ;
	}
}

void c_driver::set_latency(signed long latency)
{
	bool mixing_thread_state = m_mixing_thread_is_running ;
	if (m_use_reset)
	{
		if (mixing_thread_state)
		{
			stop() ;
		}
	}
	else
	{
		EnterCriticalSection(&m_config_critical_section) ;
	}


	if (latency < 3)
	{
		latency = 3 ;
	}

	m_latency = (latency < m_number_of_buffers)?latency:m_number_of_buffers;

	if (m_use_reset)
	{
		if (mixing_thread_state)
		{
			play() ;
		}
	}
	else
	{
		LeaveCriticalSection(&m_config_critical_section) ;
	}
}

void c_driver::config(bool linear_interpolation, 
					  signed long sampling_rate, 
					  signed long latency,
					  bool loop_allowed,
					  bool nnas,
					  signed long ramp_length,
					  signed long priority_scheme,
					  unsigned long feedback_delay,
					  double feedback_volume, 
					  bool use_reset)
{
	if (!m_is_configured)
	{
		m_closed = false ;
		m_use_reset = use_reset ;

		for (signed long i = 0 ; i < NB_MIXERS ; i ++)
		{
			m_mixers[i] = NULL ;
			m_orders[i] = (signed long *)NULL ;
			m_lines[i] = (signed long *)NULL ;
			m_playings[i] = (bool *)NULL ;
		}
		
		m_linear_interpolation = linear_interpolation ;
		m_is_configured = true ;
		m_sampling_rate = sampling_rate ;
		m_number_of_buffers = 64 ;
		m_buffer_length = 0x1000;
		m_max_feedback_delay = m_buffer_length * 16 ;
		m_feedback_delay = (feedback_delay < m_max_feedback_delay)?feedback_delay:m_max_feedback_delay;
		m_feedback_volume = feedback_volume ;
		
		m_input_buffer = (double *)NULL;
		allocate_input_buffer() ;
		
		m_output_buffers = (WAVEHDR *)NULL ;
		allocate_output_buffers() ;
		
		InitializeCriticalSection(&m_mixing_critical_section) ;
		InitializeCriticalSection(&m_config_critical_section) ;
		
		set_priority_scheme(priority_scheme) ;
		init_mixing_thread() ;
		
		m_loop_allowed = loop_allowed ;
		m_nnas = nnas ;
		m_ramp_length = ramp_length ;
		m_current_buffer = 0 ;
		m_buffer_to_fill = 0 ;
	}
}
